/***************************************************************************//**
 * @file
 * @brief WSTK display driver implementation for ESL Tag example.
 *******************************************************************************
 * # License
 * <b>Copyright 2021 Silicon Laboratories Inc. www.silabs.com</b>
 *******************************************************************************
 *
 * SPDX-License-Identifier: Zlib
 *
 * The licensor of this software is Silicon Laboratories Inc.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 ******************************************************************************/
#include <stdint.h>
#include <stdarg.h>
#include "dmd.h"
#include "esl_tag_log.h"
#include "esl_tag_display.h"
#include "sl_board_control.h"
#include "sl_common.h"
#include "esl_tag_image_core.h"
#include "esl_tag_wstk_lcd_driver.h"

// Embedded image metadata
#define PIC_WIDTH                 (128u)
#define PIC_HEIGHT                (128u)
#define BITS_PER_PIXEL            (1u)

// Calculate the pixel count for a given amount of data bytes
#define GetPixelCount(bytes)      ((8u * bytes) / BITS_PER_PIXEL)

// An arbitrary size of temporary image buffer - must fit on the stack, though!
#define IMAGE_CHUNK_BUFFER_SIZE   (257u) // in bytes

// QR placeholder information block
PACKSTRUCT(struct qr_ph_s {
  const uint8_t magic[16];          /// Magic memory pattern
  const uint16_t size;              /// Size of the pic_data placeholder
  const uint8_t* volatile qr_data;  /// Pointer to pic_data[]
});

// Embedded image
static const uint8_t pic_data[] = {
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xC7, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFD, 0xBB, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFB, 0xAB,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x1F, 0xFF, 0xFF, 0x7F, 0x00, 0x00, 0xFE,
  0xFF, 0xFF, 0xE7, 0xBB, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xE3, 0xFF, 0xFF,
  0x07, 0x00, 0x00, 0xE0, 0xFF, 0xFF, 0x8F, 0xC7, 0xFF, 0xFF, 0xFF, 0xFF,
  0x3F, 0xFC, 0xFF, 0x7F, 0x00, 0x00, 0x00, 0x80, 0xFF, 0xFF, 0x0F, 0xFE,
  0xFF, 0xFF, 0xFF, 0xFF, 0x87, 0xFF, 0xFF, 0x1F, 0x00, 0x00, 0x00, 0x00,
  0xFF, 0xFF, 0x0F, 0xFC, 0xFF, 0xFF, 0xFF, 0xFF, 0xE0, 0xFF, 0xFF, 0x03,
  0x00, 0x00, 0x00, 0x00, 0xFE, 0xFF, 0x1F, 0xF8, 0xFF, 0xFF, 0xFF, 0x1F,
  0xFC, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFC, 0xFF, 0x0F, 0xE0,
  0xFF, 0xFF, 0xFF, 0x07, 0xFF, 0xFF, 0x3F, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xFC, 0xFF, 0x0F, 0xE0, 0xFF, 0xFF, 0xFF, 0xC1, 0xFF, 0xFF, 0x0F, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xFE, 0xFF, 0x07, 0xC0, 0xFF, 0xFF, 0x7F, 0xE0,
  0xFF, 0xFF, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x03, 0xC0,
  0xFF, 0xFF, 0x1F, 0xF8, 0xFF, 0xFF, 0x03, 0x00, 0x00, 0x00, 0x00, 0x80,
  0xFF, 0xFF, 0x01, 0x80, 0xFF, 0xFF, 0x07, 0xFC, 0xFF, 0xFF, 0x01, 0x00,
  0x00, 0x00, 0x00, 0xC0, 0xFF, 0x7F, 0x00, 0x80, 0xFF, 0xFF, 0x00, 0xFE,
  0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x03, 0x00, 0x80,
  0xFF, 0x3F, 0x00, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x80, 0xFF, 0x1F, 0x80, 0xFF, 0xFF, 0x7F, 0x00, 0x00,
  0xE0, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xC0, 0xFF, 0x0F, 0x80, 0xFF,
  0xFF, 0x7F, 0x00, 0xF0, 0xFF, 0xFF, 0xFF, 0x7F, 0x00, 0x00, 0x00, 0xC0,
  0xFF, 0x03, 0x80, 0xFF, 0xFF, 0xFF, 0xC0, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0x0F, 0x00, 0x00, 0xC0, 0xFF, 0x00, 0x80, 0xFF, 0xFF, 0xFF, 0xE1, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0xE0, 0x7F, 0x00, 0x80, 0xFF,
  0xFF, 0xFF, 0x03, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x01, 0x00, 0xF0,
  0x3F, 0x00, 0x80, 0xFF, 0xFF, 0xFF, 0x07, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0x03, 0x00, 0xF8, 0x1F, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0x3F, 0x00,
  0x00, 0xF8, 0xFF, 0xFF, 0xFF, 0x07, 0x00, 0xFC, 0x0F, 0x00, 0x00, 0xFE,
  0xFF, 0xFF, 0xFF, 0x07, 0x00, 0x80, 0xFF, 0xFF, 0xFF, 0x0F, 0x00, 0xFC,
  0x0F, 0x00, 0x00, 0xFC, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x03, 0xFE, 0xFF,
  0xFF, 0x0F, 0x00, 0xFF, 0x07, 0x00, 0x00, 0xE0, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0x3F, 0xF8, 0xFF, 0xFF, 0x0F, 0x80, 0xFF, 0x03, 0x00, 0x00, 0x00,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x07, 0xF8, 0xFF, 0xFF, 0x0F, 0xE0, 0xFF,
  0x03, 0x00, 0x00, 0x00, 0xE0, 0xFF, 0xFF, 0xFF, 0x0F, 0x00, 0xF0, 0xFF,
  0xFF, 0x0F, 0xF0, 0xFF, 0x01, 0x00, 0x00, 0x00, 0x00, 0x80, 0x7F, 0x00,
  0x00, 0x00, 0xF0, 0xFF, 0xFF, 0x07, 0xF8, 0xFF, 0x01, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF0, 0xFF, 0xFF, 0x07, 0xFE, 0xFF,
  0x01, 0x00, 0x00, 0x1F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF8, 0xFF,
  0xFF, 0x83, 0xFF, 0xFF, 0x01, 0x00, 0xC0, 0xFF, 0x01, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xF8, 0xFF, 0xFF, 0xE0, 0xFF, 0xFF, 0x01, 0x00, 0xF0, 0xFF,
  0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFE, 0xFF, 0x7F, 0xFC, 0xFF, 0xFF,
  0x01, 0x00, 0xF8, 0xFF, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF,
  0x1F, 0xFE, 0xFF, 0xFF, 0x03, 0x00, 0xFE, 0xFF, 0x03, 0x00, 0x00, 0x00,
  0x00, 0x80, 0xFF, 0xFF, 0xCF, 0xFF, 0xFF, 0xFF, 0x07, 0x00, 0xFF, 0xFF,
  0x01, 0x00, 0x00, 0x00, 0x00, 0xE0, 0xFF, 0xFF, 0xF1, 0xFF, 0xFF, 0xFF,
  0x07, 0x80, 0xFF, 0xFF, 0x03, 0x00, 0x00, 0x00, 0x00, 0xF8, 0xFF, 0x7F,
  0xFE, 0xFF, 0xFF, 0xFF, 0x1F, 0x80, 0xFF, 0xFF, 0x07, 0x00, 0x00, 0x00,
  0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x3F, 0x80, 0xFF, 0xFF,
  0x1F, 0x00, 0x00, 0x00, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0x7F, 0x80, 0xFF, 0xFF, 0x7F, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x81, 0xFF, 0xFF, 0xFF, 0x0F, 0x00, 0xFC,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x07, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0x7F, 0xFC, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xE3, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0x03, 0x3F, 0x9F, 0x7F, 0x7E, 0xE0, 0x07, 0x3E, 0xCF, 0xFF, 0x9F, 0xFF,
  0xF7, 0x0F, 0xF8, 0xC0, 0x01, 0x3E, 0x9F, 0x7F, 0x3E, 0xC0, 0x03, 0x3C,
  0xCE, 0xFF, 0x9F, 0xFF, 0xE3, 0x0F, 0x70, 0x80, 0x79, 0x3E, 0x9F, 0x7F,
  0x3E, 0xCF, 0xF3, 0x3C, 0xCE, 0xFF, 0x9F, 0xFF, 0xE3, 0xCF, 0x73, 0x9E,
  0xF9, 0x3F, 0x9F, 0x7F, 0x3E, 0xFF, 0xF3, 0x3C, 0xCC, 0xFF, 0x9F, 0xFF,
  0xC9, 0xCF, 0x73, 0xFE, 0xF9, 0x3F, 0x9F, 0x7F, 0x3E, 0xFF, 0xF3, 0x3C,
  0xCC, 0xFF, 0x9F, 0xFF, 0xC9, 0xCF, 0x73, 0xFE, 0xE1, 0x3F, 0x9F, 0x7F,
  0x3E, 0xFF, 0xF3, 0x3C, 0xCC, 0xFF, 0x9F, 0xFF, 0xC9, 0xCF, 0x7B, 0xF8,
  0x07, 0x3F, 0x9F, 0x7F, 0x3E, 0xFF, 0xF3, 0x3C, 0xC9, 0xFF, 0x9F, 0xFF,
  0x9C, 0x0F, 0xFC, 0xC1, 0x3F, 0x3E, 0x9F, 0x7F, 0x3E, 0xFF, 0xF3, 0x3C,
  0xC9, 0xFF, 0x9F, 0xFF, 0x9C, 0x0F, 0xF8, 0x8F, 0x7F, 0x3E, 0x9F, 0x7F,
  0x3E, 0xFF, 0xF3, 0x3C, 0xC3, 0xFF, 0x9F, 0xFF, 0x9C, 0xCF, 0xF3, 0x9F,
  0x7F, 0x3E, 0x9F, 0x7F, 0x3E, 0xFF, 0xF3, 0x3C, 0xC3, 0xFF, 0x9F, 0xFF,
  0x80, 0xCF, 0xF3, 0x9F, 0x79, 0x3E, 0x9F, 0x7F, 0x3E, 0xCF, 0xF3, 0x3C,
  0xC7, 0xFF, 0x9F, 0x7F, 0x00, 0xCF, 0x73, 0x9E, 0x79, 0x3E, 0x9F, 0x7F,
  0x3E, 0xCF, 0xF3, 0x3C, 0xC7, 0xFF, 0x9F, 0x7F, 0x3E, 0xCF, 0x73, 0x9E,
  0x01, 0x3E, 0x1F, 0x70, 0x3E, 0xC0, 0x03, 0x3C, 0xC7, 0xFF, 0x1F, 0x70,
  0x3E, 0x0F, 0x78, 0x80, 0x03, 0x3F, 0x1F, 0x70, 0x7E, 0xE0, 0x07, 0x3E,
  0xCF, 0xFF, 0x1F, 0x70, 0x3E, 0x0F, 0xFC, 0xC0, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xDF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xC1, 0xFE, 0xFF, 0xFF, 0xFF, 0xDF, 0xFF,
  0xE3, 0xFE, 0xDF, 0x7D, 0xFF, 0xEF, 0xFF, 0xFD, 0xFF, 0xFD, 0xFE, 0x7F,
  0xFF, 0xFF, 0xFF, 0xFF, 0xDD, 0xFE, 0xDF, 0x7E, 0xFF, 0xEF, 0xFF, 0xFD,
  0xFF, 0xFD, 0xCE, 0x33, 0xD6, 0x38, 0xDC, 0xF9, 0xDD, 0xF0, 0x5C, 0x7C,
  0x7F, 0x0E, 0xCF, 0xFD, 0xFF, 0xFD, 0xB6, 0x6D, 0x67, 0xB7, 0xDB, 0xF6,
  0xF9, 0x6E, 0xDB, 0x7E, 0xBF, 0x6D, 0xB7, 0xFD, 0xFF, 0xC1, 0xB6, 0x7D,
  0x77, 0xB7, 0xDB, 0xFE, 0xC3, 0x6E, 0xDB, 0x7E, 0xFF, 0xED, 0xB6, 0xFD,
  0xFF, 0xFD, 0x86, 0x7D, 0x77, 0xB7, 0xDB, 0xFE, 0xDF, 0x6E, 0xD8, 0x7E,
  0x7F, 0xEC, 0x86, 0xFD, 0xFF, 0xFD, 0xF6, 0x7D, 0x77, 0xB7, 0xDB, 0xFE,
  0xDD, 0x6E, 0xDF, 0x7E, 0xBF, 0xED, 0xF6, 0xFD, 0xFF, 0xFD, 0xB6, 0x6D,
  0x77, 0xB7, 0xDB, 0xF6, 0xDD, 0x6E, 0xDB, 0x7E, 0xBF, 0x6D, 0xB7, 0xFD,
  0xFF, 0xC1, 0xCE, 0xF3, 0xF6, 0xB8, 0xDB, 0xF9, 0xE3, 0xEE, 0xDC, 0x7E,
  0x30, 0x0C, 0xCF, 0xFD, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

#define MAGIC_DATA_OFFSET 1164

// data slice from pic_data at @MAGIC_DATA_OFFSET
static const uint8_t original_magic[] = {
  0x80, 0xCF, 0xF3, 0x9F, 0x79, 0x3E, 0x9F, 0x7F, 0x3E, 0xCF, 0xF3, 0x3C
};

// QR placeholder information block
const struct qr_ph_s qr_info = {
  .magic = {
    '.', 'q', 'r', 'c',
    '_', 'p', 'h', '_',
    'p', 'i', 'c', '_',
    'd', 'a', 't', 'a'
  },
  .size = sizeof(pic_data),
  .qr_data = pic_data
};

bool esl_wstk_lcd_is_logo()
{
  if (memcmp(&original_magic,
             &(qr_info.qr_data[MAGIC_DATA_OFFSET]),
             sizeof(original_magic)) == 0) {
    return true;
  } else {
    return false;
  }
}

sl_status_t esl_wstk_lcd_write(int param_count, ...)
{
  // Declaring pointer to the argument list
  va_list               ptr;
  EMSTATUS              status;
  sl_status_t           result = SL_STATUS_FAIL;
  DMD_DisplayGeometry*  pgeometry;
  uint8_t               image_index;

  // LCD write_func: Invalid parameters!
  sl_bt_esl_assert(param_count == ESL_DISPLAY_WRITE_FUNC_PARAMETERS_COUNT);

  // initializing argument to the list pointer
  va_start(ptr, param_count);

  // accessing variables (after each call to va_arg our ptr points to next one)
  (void)va_arg(ptr, int); // this simple driver just ignores the display index!
  image_index = (uint8_t)va_arg(ptr, int);

  // end argument list traversal
  va_end(ptr);

  // get display geometry data
  status = DMD_getDisplayGeometry(&pgeometry);

  if (status == DMD_OK) {
    uint8_t        data[IMAGE_CHUNK_BUFFER_SIZE];
    uint16_t       offset     = 0;
    uint16_t       read_count = 1;

    result = SL_STATUS_OK;

    while (result == SL_STATUS_OK && read_count) {
      uint16_t x_offset = GetPixelCount(offset) % pgeometry->clipWidth;
      uint16_t y_offset = GetPixelCount(offset) / pgeometry->clipHeight;

      read_count = offset;  // temporary storage
      result = esl_image_get_data(image_index, &offset, sizeof(data), data);

      read_count = offset - read_count;

      status = DMD_ERROR_TEST_FAILED;
      if (result == SL_STATUS_OK) {
        status = DMD_writeData(x_offset, y_offset, data,
                               GetPixelCount(read_count));
        if (status == DMD_OK) {
          status = DMD_updateDisplay();
        }
      }

      if (status != DMD_OK) {
        result = SL_STATUS_FAIL;
      }
    }
  }

  return result;
}

SL_WEAK sl_status_t esl_tag_wstk_lcd_run_qrcode()
{
  return SL_STATUS_OK;
}

sl_status_t esl_wstk_lcd_init(int param_count, ...)
{
  sl_status_t result = SL_STATUS_FAIL;
  EMSTATUS    status;

  // this simple driver ignores all the parameters
  (void)param_count;

  // enable the DMD HW module for the DISPLAY device driver.
  sl_board_enable_display();

  status = DMD_init(0);
  // Failed to init display driver
  sl_bt_esl_assert(DMD_OK == status);

  // write initial bitmap to display
  status = DMD_writeData(0, 0, qr_info.qr_data, 8 * qr_info.size);

  if (status == DMD_OK) {
    // refresh visible content of the display
    status = DMD_updateDisplay();
    status = esl_tag_wstk_lcd_run_qrcode();
  }
  // put the LCD display into sleep mode - greatly reduces power consumption by
  // disabling auto-refresh, while still allowing static images to be displayed
  (void)DMD_sleep();

  return result;
}

void esl_wstk_lcd_bt_on_event(sl_bt_msg_t *evt)
{
  sl_status_t sc;
  (void)sc; // suppress the compiler warning if sl_bt_esl_assert disabled

  // This event shall precede the esl_core_bt_on_event ahdnler, and hence the
  // esl_core_boot handler invocation. Only then we can assure that the WST LCD
  // will be registered as the very first one in the list of available displays.
  switch (SL_BT_MSG_ID(evt->header)) {
    case sl_bt_evt_system_boot_id:
      // here we're checking that we precede the esl_core_bt_on_event invocation
      if (esl_core_get_status() == esl_state_boot) {
        esl_display_info_p info;
        // create the display info characteristic data
        sc = esl_display_create(PIC_WIDTH,
                                PIC_HEIGHT,
                                ESL_DISPLAY_TYPE_BLACK_WHITE,
                                &info);
        sl_bt_esl_assert(sc == SL_STATUS_OK);
        // register display we just created temporary - this makes it permanent
        sc = esl_display_add(info, esl_wstk_lcd_init, esl_wstk_lcd_write);
        sl_bt_esl_assert(sc == SL_STATUS_OK);
      }
      break;

    default:
      break;
  }
}
