/***************************************************************************//**
 * @file
 * @brief Host communication application module (windows).
 *******************************************************************************
 * # License
 * <b>Copyright 2023 Silicon Laboratories Inc. www.silabs.com</b>
 *******************************************************************************
 *
 * SPDX-License-Identifier: Zlib
 *
 * The licensor of this software is Silicon Laboratories Inc.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 ******************************************************************************/

#include <stdint.h>
#include <string.h>
#include <stdbool.h>
#include <stdlib.h>
#include <pthread.h>
#include <windows.h>
#include "uart.h"
#include "tcp.h"
#include "app_assert.h"
#include "app_log.h"
#include "host_comm.h"
#include "app_sleep.h"
#include "host_comm_config.h"

// Default parameter values.
#define DEFAULT_UART_PORT             ""
#define DEFAULT_UART_BAUD_RATE        115200
#define DEFAULT_UART_FLOW_CONTROL     1
#define DEFAULT_UART_TIMEOUT          100
#define DEFAULT_TCP_ADDRESS           ""
#define DEFAULT_TCP_PORT              "4901"
#define MAX_OPT_LEN                   255
#define DEFAULT_CPC_INST_NAME         "cpcd_0"

#define IS_EMPTY_STRING(s)            ((s)[0] == '\0')
#define HANDLE_VALUE_MIN              0

// Define global HOST_COMM_API_DEFINE library.
HOST_COMM_API_DEFINE();

typedef struct {
  uint16_t len;
  uint8_t buf[DEFAULT_HOST_BUFLEN];
} buf_t;

static buf_t buf_tmp = { 0 };
static buf_t buf_in = { 0 };

// end the receiving loop if signal is received.
static volatile bool run = true;

static bool comm_channel_selected = false;

// UART serial port options.
static char uart_port[MAX_OPT_LEN] = DEFAULT_UART_PORT;
static uint32_t uart_baud_rate = DEFAULT_UART_BAUD_RATE;
static uint32_t uart_flow_control = DEFAULT_UART_FLOW_CONTROL;

// TCP/IP address.
static char tcp_address[MAX_OPT_LEN] = DEFAULT_TCP_ADDRESS;

HANDLE serial_handle;
SOCKET socket_handle;
void *handle_ptr;

// Static receive function
void *msg_recv_func(void *ptr);

pthread_t thread_rx;

pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;

/**************************************************************************//**
 * Initialize low level connection.
 *****************************************************************************/
sl_status_t host_comm_init(void)
{
  int32_t status;
  int iret;

  if (!comm_channel_selected) {
    app_log_error("No communication channel provided, "
                  "but exactly one is expected!" APP_LOG_NL);
    return SL_STATUS_INVALID_PARAMETER;
  }

  if (!IS_EMPTY_STRING(uart_port)) {
    // Initialise UART serial connection.
    handle_ptr = &serial_handle;
    HOST_COMM_API_INITIALIZE_NONBLOCK(uartTx, uartRx, uartRxPeek);
    status = uartOpen(handle_ptr, (int8_t *)uart_port, uart_baud_rate,
                      uart_flow_control, DEFAULT_UART_TIMEOUT);
    if (status < HANDLE_VALUE_MIN) {
      app_log_error("Failed to open serial connection (%d)"
                    APP_LOG_NL, status);
      exit(EXIT_FAILURE);
    }
    uartFlush(handle_ptr);
  } else if (!IS_EMPTY_STRING(tcp_address)) {
    // Initialise TCP/IP connection.
    handle_ptr = &socket_handle;
    HOST_COMM_API_INITIALIZE_NONBLOCK(tcp_tx, tcp_rx, tcp_rx_peek);
    status = tcp_open(handle_ptr, tcp_address, DEFAULT_TCP_PORT);
    app_assert(status == HANDLE_VALUE_MIN,
               "[E: %d] Failed to open TCP/IP connection" APP_LOG_NL,
               status);
  } else {
    return SL_STATUS_INVALID_PARAMETER;
  }

  iret = pthread_create(&thread_rx, NULL, msg_recv_func, NULL);
  if (iret) {
    app_log_error("Error - pthread_create() return code: %d" APP_LOG_NL, iret);
    return SL_STATUS_FAIL;
  }
  app_log_info("Opened port on windows." APP_LOG_NL);
  return SL_STATUS_OK;
}

/**************************************************************************//**
 * Set low level host communication connection options.
 *****************************************************************************/
sl_status_t host_comm_set_option(char option, char *value)
{
  sl_status_t sc = SL_STATUS_OK;

  switch (option) {
    // TCP/IP address.
    case 't':
      if (!comm_channel_selected) {
        strncpy(tcp_address, value, MAX_OPT_LEN);
        comm_channel_selected = true;
      } else {
        app_log_error("More than one communication channel "
                      "provided, but exactly one is expected!" APP_LOG_NL);
        sc = SL_STATUS_INVALID_PARAMETER;
      }
      break;
    // UART serial port.
    case 'u':
      if (!comm_channel_selected) {
        strncpy(uart_port, value, MAX_OPT_LEN);
        comm_channel_selected = true;
      } else {
        app_log_error("More than one communication channel "
                      "provided, but exactly one is expected!" APP_LOG_NL);
        sc = SL_STATUS_INVALID_PARAMETER;
      }
      break;
    // UART baud rate.
    case 'b':
      uart_baud_rate = atol(value);
      break;
    // UART flow control disable.
    case 'f':
      uart_flow_control = 0;
      break;
    // Unknown option.
    default:
      sc = SL_STATUS_NOT_FOUND;
      break;
  }
  return sc;
}

/**************************************************************************//**
 * Deinitialize low level connection.
 *****************************************************************************/
void host_comm_deinit(void)
{
  run = false;
  pthread_cancel(thread_rx);

  if (!IS_EMPTY_STRING(uart_port)) {
    uartClose(handle_ptr);
  } else if (!IS_EMPTY_STRING(tcp_address)) {
    tcp_close(handle_ptr);
  }
}

/**************************************************************************//**
 * Write data to NCP through low level drivers.
 *****************************************************************************/
int32_t host_comm_tx(uint32_t len, uint8_t* data)
{
  return host_comm_output(handle_ptr, len, data);
}

/**************************************************************************//**
 * Read data from NCP.
 *****************************************************************************/
int32_t host_comm_rx(uint32_t len, uint8_t* data)
{
  int32_t ret = -1;

  pthread_mutex_lock(&mutex);
  if (buf_in.len >= len) {
    memcpy(data, &buf_in.buf[0], len);
    buf_in.len -= len;
    memmove(buf_in.buf, &buf_in.buf[len], buf_in.len);
    ret = len;
  }
  pthread_mutex_unlock(&mutex);

  return ret;
}

/**************************************************************************//**
 * Peek if readable data exists.
 *****************************************************************************/
int32_t host_comm_peek(void)
{
  int32_t len = 0;
  pthread_mutex_lock(&mutex);
  len = buf_in.len;
  pthread_mutex_unlock(&mutex);
  return len;
}

/**************************************************************************//**
 * Read data from low level drivers.
 *****************************************************************************/
void *msg_recv_func(void *ptr)
{
  int32_t ret;

  // unused variable
  (void)ptr;

  while (run) {
    int32_t len;
    len = host_comm_pk(handle_ptr);

    if (len < 0) {
      // Peek is not supported, read data one by one
      len = 1;
    } else if ((size_t)len > sizeof(buf_tmp.buf)) {
      // If readable data exceeds the buffer size then read it one by one to avoid overflow
      len = 1;
      app_log_warning("Input buffer size too low, please increase it." APP_LOG_NL);
    }

    if (len > 0) {
      ret = host_comm_input(handle_ptr, len, buf_tmp.buf);
      pthread_mutex_lock(&mutex);

      if (ret <= (sizeof(buf_in.buf) - buf_in.len)) {
        memcpy(&buf_in.buf[buf_in.len], &buf_tmp.buf[0], ret);
        buf_in.len += ret;
        pthread_mutex_unlock(&mutex);
      } else {
        pthread_mutex_unlock(&mutex);
        app_log_error("Received data lost." APP_LOG_NL);
      }
    } else {
      // No data available
      app_sleep_us(RECV_FUNC_US_SLEEP);
    }
  }

  return 0;
}
