/***************************************************************************/ /**
 * @file
 * @brief BT Mesh Configurator Component - Job
 *******************************************************************************
 * # License
 * <b>Copyright 2021 Silicon Laboratories Inc. www.silabs.com</b>
 *******************************************************************************
 *
 * SPDX-License-Identifier: Zlib
 *
 * The licensor of this software is Silicon Laboratories Inc.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 ******************************************************************************/

#ifndef BTMESH_CONF_JOB_H
#define BTMESH_CONF_JOB_H

#include "sl_slist.h"
#include "btmesh_conf_types.h"
#include "btmesh_conf_task.h"

#ifdef __cplusplus
extern "C"
{
#endif

/***************************************************************************//**
 * @addtogroup btmesh_conf BT Mesh Configurator Component
 * @{
 ******************************************************************************/

/***************************************************************************//**
 * @addtogroup btmesh_conf_job BT Mesh Configuration Job
 * @brief BT Mesh Configuration Job aggregates group of configuration tasks which
 *   targets the same configuration server model on a remote node.
 * @{
 ******************************************************************************/

/// Result of BT Mesh Configuration job
typedef enum btmesh_conf_job_result_t {
  /// Every executed task of configuration job is successful
  BTMESH_CONF_JOB_RESULT_SUCCESS,
  /// At least one executed task is failed during configuration job execution
  BTMESH_CONF_JOB_RESULT_FAIL,
  /// Internal error occurred during configuration job execution
  BTMESH_CONF_JOB_RESULT_CRITICAL_ERROR,
  /// The configuration job is canceled because the configurator is deinitialized
  BTMESH_CONF_JOB_RESULT_CANCEL,
  /// The result of configuration job is not known yet
  BTMESH_CONF_JOB_RESULT_UNKNOWN
} btmesh_conf_job_result_t;

/// Type definition of @ref struct btmesh_conf_job_t
typedef struct btmesh_conf_job_t btmesh_conf_job_t;

/***************************************************************************//**
 * Type of BT Mesh Configuration Job status notification callback
 *
 * @param[in] job Configuration job with changed status
 *
 * If task execution path ends in the configuration job then the job ends as
 * well and job status notification callback is called with the result.
 ******************************************************************************/
typedef void (*btmesh_conf_on_job_notification_t)(const btmesh_conf_job_t *job);

/***************************************************************************//**
 * @brief BT Mesh Configuration Job aggregates group of configuration tasks which
 * targets the same configuration server model on a remote node.
 ******************************************************************************/
struct btmesh_conf_job_t {
  /// List element (node) to support placement of jobs in singly-linked list
  sl_slist_node_t list_elem;
  /// Unique job identifier generated by @ref btmesh_conf_job_id_generator
  uint32_t job_id;
  /// Configuration tasks which shall be executed in the specified order
  btmesh_conf_task_t *task_tree;
  /// Active configuration task
  btmesh_conf_task_t *current_task;
  /// Network key used to encrypt the config requests on the network layer
  uint16_t enc_netkey_index;
  /// Destination node primary element address
  uint16_t server_address;
  /// Result of configuration job
  btmesh_conf_job_result_t result;
  /// @brief Configuration job status notification callback which is called when
  /// the status of the job changes at the end of the job
  void (*on_job_notification)(const btmesh_conf_job_t *job);
  /// @brief Job status param can be set by the application to differentiate
  /// config jobs if the same callback function is used in multiple jobs
  btmesh_conf_varg_t job_status_param;
  /// @brief Auto destroy deallocates the configuration job and its tasks
  /// automatically after the job status notification callback returns.
  /// If @ref btmesh_conf_submit_job operation fails and the definition
  /// SL_BTMESH_CONF_JOB_AUTO_DESTROY_ON_SUBMIT_FAILURE_CFG_VAL is turned on
  /// then the job is deallocated automatically on submit failure.
  bool auto_destroy;
};

/***************************************************************************//**
 * Create BT Mesh Configuration Job with default parameters
 *
 * @param[in] enc_netkey_index Network key used to encrypt the config requests
 * @param[in] server_address Destination node primary element address
 * @param[in] task_tree Config tasks which shall be executed in specified order
 * @param[in] on_job_notification Configuration job status notification callback
 *   which is called when the status of the job changes at the end of the job
 * @returns Created configuration job.
 * @retval NULL If the configuration job creation fails.
 *
 * This function calls @ref btmesh_conf_job_create with the following parameters:
 *   - @a job_status_param: @ref BTMESH_CONF_VARG_NULL
 *   - @a auto_destroy: @ref SL_BTMESH_CONF_JOB_AUTO_DESTROY_CFG_VAL
 *   - @a job_id: NULL
 ******************************************************************************/
btmesh_conf_job_t *btmesh_conf_job_create_default(uint16_t enc_netkey_index,
                                                  uint16_t server_address,
                                                  btmesh_conf_task_t *task_tree,
                                                  btmesh_conf_on_job_notification_t on_job_notification);

/***************************************************************************//**
 * Create BT Mesh Configuration Job with given tasks
 *
 * @param[in] enc_netkey_index Network key used to encrypt the config requests
 * @param[in] server_address Destination node primary element address
 * @param[in] task_tree Config tasks which shall be executed in specified order
 * @param[in] on_job_notification Configuration job status notification callback
 *   which is called when the status of the job changes at the end of the job
 * @param[in] job_status_param Job status param can be set by the application to
 *    differentiate config jobs if the same jobs status notification callback
 *    function is used in multiple jobs
 * @param[in] auto_destroy Auto destroy deallocates the configuration job and
 *   its tasks automatically after the job status notification callback returns.
 *   If @ref btmesh_conf_submit_job operation fails and the definition
 *   @ref SL_BTMESH_CONF_JOB_AUTO_DESTROY_ON_SUBMIT_FAILURE_CFG_VAL is turned on then the
 *   job is deallocated automatically on submit failure.
 * @param[out] job_id Unique job ID generated by @ref btmesh_conf_job_id_generator
 * @returns Created configuration job.
 * @retval NULL If the configuration job creation fails.
 *
 * Configuration job can be created if the tasks are created and linked first
 * in order to provide the @p task_tree parameter. The created configuration
 * job can be submitted for execution by calling @ref btmesh_conf_submit_job
 * function.
 ******************************************************************************/
btmesh_conf_job_t *btmesh_conf_job_create(uint16_t enc_netkey_index,
                                          uint16_t server_address,
                                          btmesh_conf_task_t *task_tree,
                                          btmesh_conf_on_job_notification_t on_job_notification,
                                          btmesh_conf_varg_t job_status_param,
                                          bool auto_destroy,
                                          uint32_t *const job_id);

/***************************************************************************//**
 * Deallocates the BT Mesh Configuration Job instance and all of its tasks
 *
 * @param[in] self Pointer to the configuration job which shall be destroyed
 ******************************************************************************/
void btmesh_conf_job_destroy(btmesh_conf_job_t *const self);

/***************************************************************************//**
 * Return generated unique BT Mesh Configuration Job identifier.
 *
 * @returns Unique configuration job identifier
 *
 * Default implementation is provided in btmesh_conf_job.c and it generates job
 * identifiers by incrementing the last one. If the deault implementation is not
 * sufficient for the application then the function can be overridden in the
 * application by function definition with strong symbol.
 *
 * @note Default implementation is provided with weak symbol.
 ******************************************************************************/
uint32_t btmesh_conf_job_id_generator(void);

/***************************************************************************//**
 * Set next BT Mesh configuration task based on the current active task of a job
 *
 * @param[in] self Pointer to the configuration job instance
 * @return Status of the next task setup
 * @retval SL_STATUS_OK If next task is set properly.
 * @retval SL_STATUS_NOT_FOUND If next is not set because this was the last task.
 * @retval SL_STATUS_FAIL If fatal error occurred.
 ******************************************************************************/
sl_status_t btmesh_conf_job_set_next_task(btmesh_conf_job_t *const self);

/** @} (end addtogroup btmesh_conf_job) */
/** @} (end addtogroup btmesh_conf) */

#ifdef __cplusplus
};
#endif

#endif /* BTMESH_CONF_JOB_H */
