# Copyright 2022 Silicon Laboratories Inc. www.silabs.com
#
# SPDX-License-Identifier: Zlib
#
# The licensor of this software is Silicon Laboratories Inc.
#
# This software is provided 'as-is', without any express or implied
# warranty. In no event will the authors be held liable for any damages
# arising from the use of this software.
#
# Permission is granted to anyone to use this software for any purpose,
# including commercial applications, and to alter it and redistribute it
# freely, subject to the following restrictions:
#
# 1. The origin of this software must not be misrepresented; you must not
#    claim that you wrote the original software. If you use this software
#    in a product, an acknowledgment in the product documentation would be
#    appreciated but is not required.
# 2. Altered source versions must be plainly marked as such, and must not be
#    misrepresented as being the original software.
# 3. This notice may not be removed or altered from any source distribution.

import argparse
import os
import re
import sys
from string import Template

DEFAULT_SL_STATUS_H_PATH = os.path.abspath(
    os.path.join(
        os.path.dirname(__file__),
        "../../../../../platform/common/inc/sl_status.h",
    )
)

DEFAULT_OUT_PATH = os.path.join(os.path.dirname(__file__), "../bgapix/slstatus.py")

TEMPLATE_STR = """# Copyright 2022 Silicon Laboratories Inc. www.silabs.com
#
# SPDX-License-Identifier: Zlib
#
# The licensor of this software is Silicon Laboratories Inc.
#
# This software is provided 'as-is', without any express or implied
# warranty. In no event will the authors be held liable for any damages
# arising from the use of this software.
#
# Permission is granted to anyone to use this software for any purpose,
# including commercial applications, and to alter it and redistribute it
# freely, subject to the following restrictions:
#
# 1. The origin of this software must not be misrepresented; you must not
#    claim that you wrote the original software. If you use this software
#    in a product, an acknowledgment in the product documentation would be
#    appreciated but is not required.
# 2. Altered source versions must be plainly marked as such, and must not be
#    misrepresented as being the original software.
# 3. This notice may not be removed or altered from any source distribution.

#                __        __               _             _                    #
#                \ \      / /_ _ _ __ _ __ (_)_ __   __ _| |                   #
#                 \ \ /\ / / _` | '__| '_ \| | '_ \ / _` | |                   #
#                  \ V  V / (_| | |  | | | | | | | | (_| |_|                   #
#                   \_/\_/ \__,_|_|  |_| |_|_|_| |_|\__, (_)                   #
#                                                   |___/                      #
#                                                                              #
#    This file is generated by btmesh_host_dfu/gen/slstatus_gen.py script.     #

import enum


@enum.unique
class SlStatus(enum.IntEnum):
    @classmethod
    def get_name(cls, value):
        try:
            return cls(value).name
        except ValueError:
            return "UNKNOWN_VALUE"

    @classmethod
    def from_int(cls, value):
        try:
            return cls(value)
        except ValueError:
            return cls.UNKNOWN_VALUE

${enumvals}
    UNKNOWN_VALUE = 0xCAFEBABE
"""

if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description="Creates python enum from sl_status.h file."
    )
    parser.add_argument(
        "-p", "--path", default=DEFAULT_SL_STATUS_H_PATH, help="Path of sl_status.h"
    )
    parser.add_argument(
        "-o", "--out", default=DEFAULT_OUT_PATH, help="Path of generated out file."
    )
    parser.add_argument(
        "-i",
        "--indent",
        type=int,
        default=4,
        help="Indentation space count in generated file.",
    )

    args = parser.parse_args()
    if not os.path.exists(args.path):
        sys.exit(f"Specified {args.path} file does not exist.")
    elif os.path.basename(args.path) != "sl_status.h":
        sys.exit(f"Specified {args.path} is not sl_status.h.")

    status_pairs = []
    with open(args.path) as sl_status_h:
        def_regex = r"^\s*#define\s+SL_STATUS_(\w+)\s*.*(0x\w+)"
        for line in sl_status_h.readlines():
            match = re.match(def_regex, line)
            if match:
                status_name = match[1]
                status_code = match[2]
                if not status_name.endswith(("_SPACE", "SPACE_MASK")):
                    status_pairs.append((status_name, status_code))
    if status_pairs:
        with open(args.out, mode="w") as out_file:
            template = Template(TEMPLATE_STR)
            enum_lines = []
            for status_name, status_code in status_pairs:
                line = "    " + status_name + " = " + status_code
                enum_lines.append(line)
            text = template.substitute(enumvals="\n".join(enum_lines))
            text = text.replace("    ", " " * args.indent)
            out_file.write(text)
