/***************************************************************************//**
 * @file
 * @brief Core interrupt handling API (Generic)
 *******************************************************************************
 * # License
 * <b>Copyright 2023 Silicon Laboratories Inc. www.silabs.com</b>
 *******************************************************************************
 *
 * SPDX-License-Identifier: Zlib
 *
 * The licensor of this software is Silicon Laboratories Inc.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 ******************************************************************************/
#ifndef EM_CORE_GENERIC_H
#define EM_CORE_GENERIC_H

#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

/*******************************************************************************
 ************************   MACRO API   ***************************************
 ******************************************************************************/

//
//  CRITICAL section macro API.
//

/* Allocate storage for PRIMASK or BASEPRI value for use by
 * CORE_ENTER/EXIT_ATOMIC() and CORE_ENTER/EXIT_CRITICAL() macros. */
#define CORE_DECLARE_IRQ_STATE

/* CRITICAL style interrupt disable. */
#define CORE_CRITICAL_IRQ_DISABLE()

/* CRITICAL style interrupt enable. */
#define CORE_CRITICAL_IRQ_ENABLE()

/* Convenience macro for implementing a CRITICAL section. */
#define CORE_CRITICAL_SECTION(yourcode) \
  {                                     \
    CORE_DECLARE_IRQ_STATE;             \
    CORE_ENTER_CRITICAL();              \
    {                                   \
      yourcode                          \
    }                                   \
    CORE_EXIT_CRITICAL();               \
  }

/* Enter CRITICAL section. Assumes that a @ref CORE_DECLARE_IRQ_STATE exist in
 * scope. */
#define CORE_ENTER_CRITICAL()

/* Exit CRITICAL section. Assumes that a @ref CORE_DECLARE_IRQ_STATE exist in
 * scope. */
#define CORE_EXIT_CRITICAL()

/* CRITICAL style yield. */
#define CORE_YIELD_CRITICAL()

//
//  ATOMIC section macro API.
//

/* ATOMIC style interrupt disable. */
#define CORE_ATOMIC_IRQ_DISABLE()

/* ATOMIC style interrupt enable. */
#define CORE_ATOMIC_IRQ_ENABLE()

/* Convenience macro for implementing an ATOMIC section. */
#define CORE_ATOMIC_SECTION(yourcode) \
  {                                   \
    CORE_DECLARE_IRQ_STATE;           \
    CORE_ENTER_ATOMIC();              \
    {                                 \
      yourcode                        \
    }                                 \
    CORE_EXIT_ATOMIC();               \
  }

/* Enter ATOMIC section. Assumes that a @ref CORE_DECLARE_IRQ_STATE exist in
 * scope. */
#define CORE_ENTER_ATOMIC()

/* Exit ATOMIC section. Assumes that a @ref CORE_DECLARE_IRQ_STATE exist in
 * scope. */
#define CORE_EXIT_ATOMIC()

/* ATOMIC style yield. */
#define CORE_YIELD_ATOMIC()

/*******************************************************************************
 *************************   TYPEDEFS   ****************************************
 ******************************************************************************/

/* Storage for PRIMASK or BASEPRI value. */
typedef uint32_t CORE_irqState_t;

#ifdef __cplusplus
}
#endif

#endif /* EM_CORE_GENERIC_H */
