/*******************************************************************************
 * @file
 * @brief  OpenThread wrapper functions for OpenThread Trel APIs
 *   allowing access to the thread stack in a multi-threaded environment.
 *******************************************************************************
 * # License
 * <b>Copyright 2024 Silicon Laboratories Inc. www.silabs.com</b>
 *******************************************************************************
 *
 * SPDX-License-Identifier: Zlib
 *
 * The licensor of this software is Silicon Laboratories Inc.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 ******************************************************************************/

#include "sl_ot_rtos_adaptation.h"
#include <openthread/trel.h>

#if defined(__GNUC__)
#define REAL_NAME(function) __real_##function
#define WRAPPER_NAME(function) __wrap_##function
#define OT_API_REAL_NAME(function) REAL_NAME(function)
#define OT_API_WRAPPER_NAME(function) WRAPPER_NAME(function)
// #elif defined(__IAR_SYSTEMS_ICC__)
//     #define REAL_NAME(function)             $Super$$##function
//     #define WRAPPER_NAME(function)          $Sub$$##function
//     #define OT_API_REAL_NAME(function)      $Super$$__iar_dl##function
//     #define OT_API_WRAPPER_NAME(function)   $Sub$$__iar_dl##function
#else
#error Unsupported compiler
#endif

extern bool                  OT_API_REAL_NAME(otTrelIsEnabled)(otInstance *aInstance);
extern bool                  OT_API_REAL_NAME(otTrelIsFilterEnabled)(otInstance *aInstance);
extern const otTrelCounters *OT_API_REAL_NAME(otTrelGetCounters)(otInstance *aInstance);
extern const otTrelPeer     *OT_API_REAL_NAME(otTrelGetNextPeer)(otInstance *aInstance, otTrelPeerIterator *aIterator);
extern uint16_t              OT_API_REAL_NAME(otTrelGetNumberOfPeers)(otInstance *aInstance);
extern void OT_API_REAL_NAME(otTrelInitPeerIterator)(otInstance *aInstance, otTrelPeerIterator *aIterator);
extern void OT_API_REAL_NAME(otTrelResetCounters)(otInstance *aInstance);
extern void OT_API_REAL_NAME(otTrelSetEnabled)(otInstance *aInstance, bool aEnable);
extern void OT_API_REAL_NAME(otTrelSetFilterEnabled)(otInstance *aInstance, bool aEnable);

bool OT_API_WRAPPER_NAME(otTrelIsEnabled)(otInstance *aInstance)
{
    sl_ot_rtos_acquire_stack_mutex();
    bool ret = OT_API_REAL_NAME(otTrelIsEnabled)(aInstance);
    sl_ot_rtos_release_stack_mutex();
    return ret;
}

bool OT_API_WRAPPER_NAME(otTrelIsFilterEnabled)(otInstance *aInstance)
{
    sl_ot_rtos_acquire_stack_mutex();
    bool ret = OT_API_REAL_NAME(otTrelIsFilterEnabled)(aInstance);
    sl_ot_rtos_release_stack_mutex();
    return ret;
}

const otTrelCounters *OT_API_WRAPPER_NAME(otTrelGetCounters)(otInstance *aInstance)
{
    sl_ot_rtos_acquire_stack_mutex();
    const otTrelCounters *ret = OT_API_REAL_NAME(otTrelGetCounters)(aInstance);
    sl_ot_rtos_release_stack_mutex();
    return ret;
}

const otTrelPeer *OT_API_WRAPPER_NAME(otTrelGetNextPeer)(otInstance *aInstance, otTrelPeerIterator *aIterator)
{
    sl_ot_rtos_acquire_stack_mutex();
    const otTrelPeer *ret = OT_API_REAL_NAME(otTrelGetNextPeer)(aInstance, aIterator);
    sl_ot_rtos_release_stack_mutex();
    return ret;
}

uint16_t OT_API_WRAPPER_NAME(otTrelGetNumberOfPeers)(otInstance *aInstance)
{
    sl_ot_rtos_acquire_stack_mutex();
    uint16_t ret = OT_API_REAL_NAME(otTrelGetNumberOfPeers)(aInstance);
    sl_ot_rtos_release_stack_mutex();
    return ret;
}

void OT_API_WRAPPER_NAME(otTrelInitPeerIterator)(otInstance *aInstance, otTrelPeerIterator *aIterator)
{
    sl_ot_rtos_acquire_stack_mutex();
    OT_API_REAL_NAME(otTrelInitPeerIterator)(aInstance, aIterator);
    sl_ot_rtos_release_stack_mutex();
}

void OT_API_WRAPPER_NAME(otTrelResetCounters)(otInstance *aInstance)
{
    sl_ot_rtos_acquire_stack_mutex();
    OT_API_REAL_NAME(otTrelResetCounters)(aInstance);
    sl_ot_rtos_release_stack_mutex();
}

void OT_API_WRAPPER_NAME(otTrelSetEnabled)(otInstance *aInstance, bool aEnable)
{
    sl_ot_rtos_acquire_stack_mutex();
    OT_API_REAL_NAME(otTrelSetEnabled)(aInstance, aEnable);
    sl_ot_rtos_release_stack_mutex();
}

void OT_API_WRAPPER_NAME(otTrelSetFilterEnabled)(otInstance *aInstance, bool aEnable)
{
    sl_ot_rtos_acquire_stack_mutex();
    OT_API_REAL_NAME(otTrelSetFilterEnabled)(aInstance, aEnable);
    sl_ot_rtos_release_stack_mutex();
}
