/***************************************************************************//**
 * @brief Zigbee Stack sleep code.
 *******************************************************************************
 * # License
 * <b>Copyright 2024 Silicon Laboratories Inc. www.silabs.com</b>
 *******************************************************************************
 *
 * The licensor of this software is Silicon Laboratories Inc. Your use of this
 * software is governed by the terms of Silicon Labs Master Software License
 * Agreement (MSLA) available at
 * www.silabs.com/about-us/legal/master-software-license-agreement. This
 * software is distributed to you in Source Code format and is governed by the
 * sections of the MSLA applicable to Source Code.
 *
 ******************************************************************************/

#define CURRENT_MODULE_NAME "zigbee_stack_sleep"
#include PLATFORM_HEADER

#ifdef SL_COMPONENT_CATALOG_PRESENT
#include "sl_component_catalog.h"
#endif //SL_COMPONENT_CATALOG_PRESENT

#if defined(SL_CATALOG_POWER_MANAGER_PRESENT) && (!defined(SL_CATALOG_ZIGBEE_PRO_COMPLIANCE_PRESENT) || defined(SL_CATALOG_KERNEL_PRESENT) )

#ifdef EZSP_SPI
uint32_t get_next_sleep_tick_ms(void);
#endif //EZSP_SPI

#ifdef SL_ZIGBEE_AF_NCP
bool ncp_ok_to_sleep (void);
#endif //SL_ZIGBEE_AF_NCP

#include "child.h"
#include "sl_sleeptimer.h"
#include "sl_power_manager.h"
#include "stack-info.h"
#include "stack/core/sl_zigbee_multi_network.h"
#include "hal.h"
#define STACK_POWER_MANAGER_EVENTS_OF_INTEREST      \
  (SL_POWER_MANAGER_EVENT_TRANSITION_ENTERING_EM0   \
   | SL_POWER_MANAGER_EVENT_TRANSITION_LEAVING_EM0  \
   | SL_POWER_MANAGER_EVENT_TRANSITION_ENTERING_EM2 \
   | SL_POWER_MANAGER_EVENT_TRANSITION_LEAVING_EM2)

#ifdef SL_CATALOG_ZIGBEE_FORCE_SLEEP_AND_WAKEUP_PRESENT
#include "force-sleep-wakeup.h"
#endif // SL_CATALOG_ZIGBEE_FORCE_SLEEP_AND_WAKEUP_PRESENT

//------------------------------------------------------------------------------
// Static and externs
// CAUTION : This variable may appear useless but it serves a purpose
// Power manager does not care where EM rewuirements are added or removed from
// Therefore it is mandatory that we check that we added the requirement before removing it
// Also note that indiscriminately adding EM1 requirement will cause overflow and make the
// microcontroller go into EM2 if EFM_ASSERT is not enabled. This results in a multitude of
// problems that may manifest as "BLE connection dies, Rx turns off etc"!!!
static bool zigbee_stack_em1_requirement_set = false;
static sl_power_manager_em_transition_event_handle_t stack_pm_handle;
static void stack_energy_mode_transition_callback(sl_power_manager_em_t from, sl_power_manager_em_t to);
static sl_power_manager_em_transition_event_info_t stack_pm_event_info =
{ STACK_POWER_MANAGER_EVENTS_OF_INTEREST,
  stack_energy_mode_transition_callback };

#if !defined(SL_CATALOG_KERNEL_PRESENT)
static void zigbee_stack_wakeup_timer_callback(sl_sleeptimer_timer_handle_t* timer_id, void *user);
static sl_sleeptimer_timer_handle_t zigbee_stack_wakeup_timer_id;
#endif //!defined(SL_CATALOG_KERNEL_PRESENT)
uint32_t sli_zigbee_stack_get_ms_to_next_wakeup(void);
void sli_zigbee_stack_sleep_init(void)
{
  zigbee_stack_em1_requirement_set = true;

  sl_power_manager_add_em_requirement(SL_POWER_MANAGER_EM1);
  sl_power_manager_add_em_requirement(SL_POWER_MANAGER_EM2);
  sl_power_manager_subscribe_em_transition_event(&stack_pm_handle, &stack_pm_event_info);
}

static void stack_energy_mode_transition_callback(sl_power_manager_em_t from,
                                                  sl_power_manager_em_t to)
{
  (void)to;
  static bool wdog_enabled = true;

  if (to == SL_POWER_MANAGER_EM1) { //Entering EM1 - check and disable watchdog
    wdog_enabled = halInternalWatchDogEnabled();
    if (wdog_enabled) {
      halInternalDisableWatchDog(0);
    }
  } else if (to == SL_POWER_MANAGER_EM0) { //Entering EM0 - check and reenable watchdog
    if (wdog_enabled) {
      halInternalEnableWatchDog();
    }
  }
}

#ifdef SL_CATALOG_ZIGBEE_FORCE_SLEEP_AND_WAKEUP_PRESENT
bool sli_zigbee_stack_is_force_sleep(void)
{
  if (sli_zigbee_app_framework_get_force_sleep_flag()) {
    // We NEED to go to EM2. Remove EM1 requirement if set and allow power manager to sleep
    if (zigbee_stack_em1_requirement_set) {
      sl_power_manager_remove_em_requirement(SL_POWER_MANAGER_EM1);
      zigbee_stack_em1_requirement_set = false;
    }
    return true;
  }
  return false;
}
#endif //#ifdef SL_CATALOG_ZIGBEE_FORCE_SLEEP_AND_WAKEUP_PRESENT

#if !defined(SL_CATALOG_KERNEL_PRESENT)
// Only for bare-metal
// This function is called in the sl_power_manager_is_ok_to_sleep autogenerated code
// This function is the zigbee check to ensure it is okay to idle or sleep
// The return value of this function indicates that it is okay to go to EM1
// If this function removes all EM1 requirements, then power manager will allow going into EM2
bool sli_zigbee_stack_is_ok_to_sleep(void)
{
  uint32_t duration_ms = 0;

#ifdef SL_CATALOG_ZIGBEE_FORCE_SLEEP_AND_WAKEUP_PRESENT
  if (sli_zigbee_stack_is_force_sleep() == true) {
    return true;
  }
#endif //#ifdef SL_CATALOG_ZIGBEE_FORCE_SLEEP_AND_WAKEUP_PRESENT
#if defined(EZSP_SPI)
  duration_ms = get_next_sleep_tick_ms();
#else //!EZSP_SPI
  duration_ms = sli_zigbee_stack_get_ms_to_next_wakeup();
#endif //EZSP_SPI

  // Limit the value of sleep duration to what the sleep timer allows
  uint32_t max_millisecond_allowed_by_sleeptimer = sl_sleeptimer_get_max_ms32_conversion();
  duration_ms = SL_MIN(duration_ms, max_millisecond_allowed_by_sleeptimer);

  // Set up sleep timer to wake us up in time for the next event
  // Note that this duration is always over SL_ZIGBEE_APP_FRAMEWORK_MINIMUM_SLEEP_DURATION_MS
  // for EM2, but may not adhere to this minimum for EM1
  if ( duration_ms > 0 ) {
    assert(sl_sleeptimer_restart_timer_ms(&zigbee_stack_wakeup_timer_id,
                                          duration_ms,
                                          zigbee_stack_wakeup_timer_callback,
                                          NULL,
                                          0u,
                                          0u) == SL_STATUS_OK);
    return true;
  }

  // Not ok to go to EM1 or EM2
  return false;
}
static void zigbee_stack_wakeup_timer_callback(sl_sleeptimer_timer_handle_t* timer_id,
                                               void *user)
{
  (void)timer_id;
  (void)user;
}
bool sli_zigbee_is_stack_task_or_isr_current_context(void)
{
  return true;
}
sl_power_manager_on_isr_exit_t sli_zigbee_stack_sleep_on_isr_exit(void)
{
  if (!sli_zigbee_stack_is_ok_to_sleep()) {
    return SL_POWER_MANAGER_WAKEUP;
  }
  return SL_POWER_MANAGER_IGNORE;
}
#endif //!DSL_CATALOG_KERNEL_PRESENT

// This function is called from both baremetal and RTOS applications
// Its primary purpose is to check all application idle/sleep override flags and
// the time-remaining to the next stack and application events to answer two questions:
// a) Is it okay to sleep ? If so, it removes the EM1 requirement
// b) How long can we idle/sleep for ? the function returns the duration in ms that sleep / idle is allowed for
// The baremetal and RTOS applications then use the returned duration to either start a sleep-timer OR
// yield the zigbee task.
uint32_t sli_zigbee_stack_get_ms_to_next_wakeup(void)
{
  uint32_t duration_ms = 0;

  // Check if sleep is permitted
  // Note that for about 5 seconds after initialization and before radio is on,
  // this function returns that it is okay to sleep
  // to keep CLI functional, we have added a check for device type as well
  bool stack_is_ok_to_nap = sl_zigbee_ok_to_nap() && (sli_zigbee_node_type >= SL_ZIGBEE_SLEEPY_END_DEVICE);

#ifdef SL_ZIGBEE_AF_NCP
  stack_is_ok_to_nap = stack_is_ok_to_nap && ncp_ok_to_sleep();
#endif //SL_ZIGBEE_AF_NCP

  if (stack_is_ok_to_nap) {
    // If the stack says we can hiberate, it means we can sleep as long as we
    // want.  Otherwise, we cannot sleep longer than the duration to its next
    // event.  In either case, we will never sleep longer than the duration
    // to our next event.
    duration_ms = (sl_zigbee_ok_to_hibernate() && (sli_zigbee_node_type >= SL_ZIGBEE_SLEEPY_END_DEVICE)
                   ? MAX_INT32U_VALUE
                   : sl_zigbee_ms_to_next_stack_event());
  }

  // If durations_ms is 0 it means we could not enter em2, so we see how long we
  // can enter em1 instead. Ensure application permits EM1
  if (duration_ms == 0) {
    if (!zigbee_stack_em1_requirement_set) {
      zigbee_stack_em1_requirement_set = true;
      sl_power_manager_add_em_requirement(SL_POWER_MANAGER_EM1);
    }
    duration_ms = sl_zigbee_ms_to_next_stack_event();
  } else {
    if (zigbee_stack_em1_requirement_set) {
      zigbee_stack_em1_requirement_set = false;
      sl_power_manager_remove_em_requirement(SL_POWER_MANAGER_EM1);
    }
  }

  return (duration_ms);
}

#if defined(SL_ZIGBEE_AF_NCP)

#ifdef EZSP_CPC
extern bool zigbee_cpc_is_tx_queue_empty(void);
#endif //EZSP_CPC

#include "app/util/ezsp/ezsp-protocol.h"
#include "serial-interface.h"
#include "stack/core/sl_zigbee_multi_network.h"

extern uint16_t sl_zigbee_get_callback_queue_length(void);

#ifdef EZSP_SPI
static uint32_t next_sleep_ms = 0;
uint32_t get_next_sleep_tick_ms(void)
{
  return next_sleep_ms;
}

void set_next_sleep_tick_ms(uint32_t ms_to_next_event)
{
  next_sleep_ms = ms_to_next_event;
}

extern uint8_t sleepMode;
extern void halNcpClearWakeFlag(void);
extern void halHostSerialPowerdown(void);
extern void halHostSerialPowerup(void);

static void sleep_tick(uint32_t *ms_to_next_event, bool *enter_em2)
{
  uint16_t stack_tasks;
  *ms_to_next_event = 0;
  *enter_em2 = false;

  // If sleepMode is reserved value 0x03, don't idle or sleep. Useful for debug.
  // Don't sleep or idle if:
  //    * ASH or SPI is busy.
  //    * There are messages waiting to be sent.
  //    * Incomming messages are being processed.
  //    * The node is a sleepy device and the radio is on.
  stack_tasks = sl_zigbee_current_stack_tasks();
  if ((sleepMode == EZSP_FRAME_CONTROL_RESERVED_SLEEP)
      || !serialOkToSleep()
      || (stack_tasks & (SL_ZIGBEE_OUTGOING_MESSAGES | SL_ZIGBEE_INCOMING_MESSAGES))
      || ((sli_zigbee_node_type == SL_ZIGBEE_SLEEPY_END_DEVICE) & (stack_tasks & SL_ZIGBEE_RADIO_IS_ON))) {
    return;
  }

  // Sleep or idle until it is time for the next event.
  *ms_to_next_event = sl_zigbee_ms_to_next_stack_event();

  if (sleepMode == EZSP_FRAME_CONTROL_POWER_DOWN \
      && serialOkToSleep()
      && !(stack_tasks & SL_ZIGBEE_OUTGOING_MESSAGES)) {
    // Sleep until an external interrupt wakes us up
    *ms_to_next_event = MAX_INT32U_VALUE;
  }

  if ((sleepMode == EZSP_FRAME_CONTROL_DEEP_SLEEP)
      || (sleepMode == EZSP_FRAME_CONTROL_POWER_DOWN)) {
    if (zigbee_stack_em1_requirement_set) {
      // Allow the system to enter em2
      sl_power_manager_remove_em_requirement(SL_POWER_MANAGER_EM1);
      zigbee_stack_em1_requirement_set = false;
    }
    *enter_em2 = true;
  } else {
    // The NCP always idles its processor whenever possible
    if (!zigbee_stack_em1_requirement_set) {
      // Prevent the system from entering em2
      sl_power_manager_add_em_requirement(SL_POWER_MANAGER_EM1);
      zigbee_stack_em1_requirement_set = true;
    }
  }

  return;
}

void sl_zigbee_ncp_spi_sleep_tick()
{
  uint32_t ms_to_next_event = 0;
  bool enter_em2 = false;
  INTERRUPTS_OFF();
  // get next event ms and whether NCP can go into EM2
  sleep_tick(&ms_to_next_event, &enter_em2);
  set_next_sleep_tick_ms(ms_to_next_event);
  if (enter_em2) {
    sl_zigbee_stack_power_down();
    serialPowerDown();
  }
  // enter sleep with PRIMASK set and BASEPRI cleared
  CORE_CriticalDisableIrq();
  INTERRUPTS_ON();
  // go to sleep
  sl_power_manager_sleep();
  CORE_CriticalEnableIrq();
  // if we are going back from EM2
  if (enter_em2) {
    sl_zigbee_stack_power_up();
    serialPowerUp();
    if (serialHostIsActive()) {
      // If the Host woke us up, we switch back to idle until the next command
      // arrives with new sleep mode instructions.
      sleepMode = EZSP_FRAME_CONTROL_IDLE;

      // If we had a Wake transaction pending, we're done with it now, so clear
      // the state flag. (We had to wait until the wakeup process completed to do this.)
      halNcpClearWakeFlag();
    }
  }
}
#endif //EZSP_SPI

bool ncp_ok_to_sleep(void)
{
  if (sl_zigbee_get_callback_queue_length()) {
    return false;
  }
  #ifdef EZSP_CPC
  if (!zigbee_cpc_is_tx_queue_empty()) {
    return false;
  }
  #endif //EZSP_CPC
  return true;
}
#endif //SL_ZIGBEE_AF_NCP
#else
void sli_zigbee_stack_sleep_init(void)
{
}
bool sli_zigbee_stack_is_ok_to_sleep(void)
{
  return false;
}

bool sli_zigbee_is_stack_task_or_isr_current_context(void)
{
  return true;
}
uint8_t sli_zigbee_stack_sleep_on_isr_exit(void)
{
  return (1UL << 0UL);
}
#endif //SL_CATALOG_POWER_MANAGER_PRESENT && !defined(SL_CATALOG_ZIGBEE_PRO_COMPLIANCE_PRESENT)
