/*
 * FreeRTOS V1.4.7
 * Copyright (C) 2020 Amazon.com, Inc. or its affiliates.  All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
 * the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * http://aws.amazon.com/freertos
 * http://www.FreeRTOS.org
 */

/* FreeRTOS includes. */
#include "FreeRTOS.h"
#include "task.h"
#include "FreeRTOS_IP.h"

#include "sl_iostream.h"
#include "sl_iostream_handles.h"
#include "sl_power_manager.h"

/* Demo includes */
#include "aws_demo.h"

/* AWS library includes. */
#include "iot_system_init.h"
#include "iot_logging_task.h"
#include "aws_clientcredential.h"
#include "aws_application_version.h"
#include "aws_dev_mode_key_provisioning.h"



/* Logging Task Defines. */
#define mainLOGGING_MESSAGE_QUEUE_LENGTH    (15)
#define mainLOGGING_TASK_STACK_SIZE         (configMINIMAL_STACK_SIZE * 8)

/* The task delay for allowing the lower priority logging task to print out Wi-Fi
 * failure status before blocking indefinitely. */
#define mainLOGGING_WIFI_STATUS_DELAY       pdMS_TO_TICKS(1000)

/* Unit test defines. */
#define mainTEST_RUNNER_TASK_STACK_SIZE     (configMINIMAL_STACK_SIZE * 16)

/* The name of the devices for xApplicationDNSQueryHook. */
#define mainDEVICE_NICK_NAME        "SiLabsGATTServer"

/**
 * @brief Application task startup hook for applications using Wi-Fi. If you are not
 * using Wi-Fi, then start network dependent applications in the vApplicationIPNetorkEventHook
 * function. If you are not using Wi-Fi, this hook can be disabled by setting
 * configUSE_DAEMON_TASK_STARTUP_HOOK to 0.
 */
void vApplicationDaemonTaskStartupHook(void);

/**
 * @brief Application IP network event hook called by the FreeRTOS+TCP stack for
 * applications using Ethernet. If you are not using Ethernet and the FreeRTOS+TCP stack,
 * start network dependent applications in vApplicationDaemonTaskStartupHook after the
 * network status is up.
 */
void vApplicationIPNetworkEventHook(eIPCallbackEvent_t eNetworkEvent);

/**
 * @brief Connects to Wi-Fi.
 */
static void prvWifiConnect(void);

/**
 * @brief Initializes the board.
 */
static void prvMiscInitialization(void);

/*-----------------------------------------------------------*/

/**
 * @brief Application runtime entry point.
 */
int main(void)
{
  /* Perform any hardware initialization that does not require the RTOS to be
   * running.  */
  prvMiscInitialization();

  /* Create tasks that are not dependent on the Wi-Fi being initialized. */
  xLoggingTaskInitialize(mainLOGGING_TASK_STACK_SIZE,
                         tskIDLE_PRIORITY,
                         mainLOGGING_MESSAGE_QUEUE_LENGTH);

  /* FIX ME: If you are using Ethernet network connections and the FreeRTOS+TCP stack,
   * uncomment the initialization function, FreeRTOS_IPInit(), below. */
  /*FreeRTOS_IPInit( ucIPAddress,
   *                 ucNetMask,
   *                 ucGatewayAddress,
   *                 ucDNSServerAddress,
   *                 ucMACAddress );
   */

  /* Start the scheduler.  Initialization that requires the OS to be running,
   * including the Wi-Fi initialization, is performed in the RTOS daemon task
   * startup hook. */
  vTaskStartScheduler();

  return 0;
}
/*-----------------------------------------------------------*/

static void prvMiscInitialization(void)
{
  /* FIX ME: Perform any hardware initializations, that don't require the RTOS to be
   * running, here.
   */

  /* initialize the SDK */
  sl_system_init();
}
/*-----------------------------------------------------------*/

void vApplicationDaemonTaskStartupHook(void)
{
  /* FIX ME: Perform any hardware initialization, that require the RTOS to be
   * running, here. */

  /* FIX ME: If your MCU is using Wi-Fi, delete surrounding compiler directives to
   * enable the unit tests and after MQTT, Bufferpool, and Secure Sockets libraries
   * have been imported into the project. If you are not using Wi-Fi, see the
   * vApplicationIPNetworkEventHook function. */

  if ( SYSTEM_Init() == pdPASS ) {
    /* Connect to the Wi-Fi before running the tests. */
    prvWifiConnect();

    /* Provision the device with AWS certificate and private key. */
    vDevModeKeyProvisioning();

    /* Start the demo tasks. */
    DEMO_RUNNER_RunDemos();
  }
}
/*-----------------------------------------------------------*/

void vApplicationIPNetworkEventHook(eIPCallbackEvent_t eNetworkEvent)
{
  /* FIX ME: If your application is using Ethernet network connections and the
   * FreeRTOS+TCP stack, delete the surrounding compiler directives to enable the
   * unit tests and after MQTT, Bufferpool, and Secure Sockets libraries have been
   * imported into the project. If you are not using Ethernet see the
   * vApplicationDaemonTaskStartupHook function. */
    #if 0
  static BaseType_t xTasksAlreadyCreated = pdFALSE;

  /* If the network has just come up...*/
  if ( eNetworkEvent == eNetworkUp ) {
    if ( (xTasksAlreadyCreated == pdFALSE) && (SYSTEM_Init() == pdPASS) ) {
      DEMO_RUNNER_RunDemos();
      xTasksAlreadyCreated = pdTRUE;
    }
  }
    #endif /* if 0 */
}
/*-----------------------------------------------------------*/

void prvWifiConnect(void)
{
  /* FIX ME: Delete surrounding compiler directives when the Wi-Fi library is ported. */
    #if 0
  WIFINetworkParams_t xNetworkParams;
  WIFIReturnCode_t xWifiStatus;
  uint8_t ucTempIp[4] = { 0 };

  xWifiStatus = WIFI_On();

  if ( xWifiStatus == eWiFiSuccess ) {
    configPRINTF( ("Wi-Fi module initialized. Connecting to AP...\r\n") );
  } else {
    configPRINTF( ("Wi-Fi module failed to initialize.\r\n") );

    /* Delay to allow the lower priority logging task to print the above status.
     * The while loop below will block the above printing. */
    TaskDelay(mainLOGGING_WIFI_STATUS_DELAY);

    while ( 1 ) {
    }
  }

  /* Setup parameters. */
  xNetworkParams.pcSSID = clientcredentialWIFI_SSID;
  xNetworkParams.ucSSIDLength = sizeof(clientcredentialWIFI_SSID);
  xNetworkParams.pcPassword = clientcredentialWIFI_PASSWORD;
  xNetworkParams.ucPasswordLength = sizeof(clientcredentialWIFI_PASSWORD);
  xNetworkParams.xSecurity = clientcredentialWIFI_SECURITY;
  xNetworkParams.cChannel = 0;

  xWifiStatus = WIFI_ConnectAP(&(xNetworkParams) );

  if ( xWifiStatus == eWiFiSuccess ) {
    configPRINTF( ("Wi-Fi Connected to AP. Creating tasks which use network...\r\n") );

    xWifiStatus = WIFI_GetIP(ucTempIp);
    if ( eWiFiSuccess == xWifiStatus ) {
      configPRINTF( ("IP Address acquired %d.%d.%d.%d\r\n",
                     ucTempIp[0], ucTempIp[1], ucTempIp[2], ucTempIp[3]) );
    }
  } else {
    /* Connection failed, configure SoftAP. */
    configPRINTF( ("Wi-Fi failed to connect to AP %s.\r\n", xNetworkParams.pcSSID) );

    xNetworkParams.pcSSID = wificonfigACCESS_POINT_SSID_PREFIX;
    xNetworkParams.pcPassword = wificonfigACCESS_POINT_PASSKEY;
    xNetworkParams.xSecurity = wificonfigACCESS_POINT_SECURITY;
    xNetworkParams.cChannel = wificonfigACCESS_POINT_CHANNEL;

    configPRINTF( ("Connect to SoftAP %s using password %s. \r\n",
                   xNetworkParams.pcSSID, xNetworkParams.pcPassword) );

    while ( WIFI_ConfigureAP(&xNetworkParams) != eWiFiSuccess ) {
      configPRINTF( ("Connect to SoftAP %s using password %s and configure Wi-Fi. \r\n",
                     xNetworkParams.pcSSID, xNetworkParams.pcPassword) );
    }

    configPRINTF( ("Wi-Fi configuration successful. \r\n") );
  }
    #endif /* if 0 */
}
/*-----------------------------------------------------------*/

/**
 * @brief User defined Idle task function.
 *
 * @note Do not make any blocking operations in this function.
 */
void vApplicationIdleHook(void)
{
  /* FIX ME. If necessary, update to application idle periodic actions. */

  static TickType_t xLastPrint = 0;
  TickType_t xTimeNow;
  const TickType_t xPrintFrequency = pdMS_TO_TICKS(5000);

  xTimeNow = xTaskGetTickCount();

  if ( (xTimeNow - xLastPrint) > xPrintFrequency ) {
    configPRINT(".");
    xLastPrint = xTimeNow;
  }
}
/*-----------------------------------------------------------*/

/**
 * @brief User defined application hook to process names returned by the DNS server.
 */
#if (ipconfigUSE_LLMNR != 0) || (ipconfigUSE_NBNS != 0)
BaseType_t xApplicationDNSQueryHook(const char * pcName)
{
  /* FIX ME. If necessary, update to applicable DNS name lookup actions. */

  BaseType_t xReturn;

  /* Determine if a name lookup is for this node.  Two names are given
   * to this node: that returned by pcApplicationHostnameHook() and that set
   * by mainDEVICE_NICK_NAME. */
  if ( strcmp(pcName, pcApplicationHostnameHook() ) == 0 ) {
    xReturn = pdPASS;
  } else if ( strcmp(pcName, mainDEVICE_NICK_NAME) == 0 ) {
    xReturn = pdPASS;
  } else {
    xReturn = pdFAIL;
  }

  return xReturn;
}

#endif /* if ( ipconfigUSE_LLMNR != 0 ) || ( ipconfigUSE_NBNS != 0 ) */
/*-----------------------------------------------------------*/

/**
 * @brief User defined assertion call. This function is plugged into configASSERT.
 * See FreeRTOSConfig.h to define configASSERT to something different.
 */
void vAssertCalled(const char * pcFile,
                   uint32_t ulLine)
{
  /* FIX ME. If necessary, update to applicable assertion routine actions. */

  const uint32_t ulLongSleep = 1000UL;
  volatile uint32_t ulBlockVariable = 0UL;
  volatile char * pcFileName = (volatile char *)pcFile;
  volatile uint32_t ulLineNumber = ulLine;

  (void)pcFileName;
  (void)ulLineNumber;

  printf("vAssertCalled %s, %ld\n", pcFile, (long)ulLine);

  /* Setting ulBlockVariable to a non-zero value in the debugger will allow
   * this function to be exited. */
  taskDISABLE_INTERRUPTS();
  {
    while (ulBlockVariable == 0UL) {
      vTaskDelay(pdMS_TO_TICKS(ulLongSleep) );
    }
  }
  taskENABLE_INTERRUPTS();
}

/*-----------------------------------------------------------*/

void vUartWriteBuf(const char * const buffer, uint32_t length)
{
  sl_iostream_write(SL_IOSTREAM_STDOUT, buffer, length);
}

int32_t xPortGetUserInput( uint8_t * pMessage,
                           uint32_t messageLength,
                           TickType_t timeoutTicks )
{
  (void)timeoutTicks;
  size_t bytes_read;
  int32_t ret;
  sl_status_t sc;
  sl_power_manager_add_em_requirement(SL_POWER_MANAGER_EM1);
  sc = sl_iostream_read(SL_IOSTREAM_STDIN, pMessage, messageLength, &bytes_read);
  sl_power_manager_remove_em_requirement(SL_POWER_MANAGER_EM1);
  if (sc != SL_STATUS_OK) {
      ret = -1;
  } else {
      ret = (int32_t)bytes_read;
  }
  return ret;
}
