/***************************************************************************//**
 * @file
 * @brief TZ Non-Secure Execution Start-up
 *******************************************************************************
 * # License
 * <b>Copyright 2024 Silicon Laboratories Inc. www.silabs.com</b>
 *******************************************************************************
 *
 * SPDX-License-Identifier: Zlib
 *
 * The licensor of this software is Silicon Laboratories Inc.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 ******************************************************************************/

#include "em_device.h"
#include <stdint.h>

#if defined(SL_APP_PROPERTIES)
#include "api/application_properties.h"
#endif // SL_APP_PROPERTIES

/*----------------------------------------------------------------------------
 * Defines.
 *----------------------------------------------------------------------------*/

#define TOTAL_INTERNAL_INTERRUPTS         (16)

#define SECURITY_LEVEL_BIT                  0x10000000
#define THUMB_INSTRUCTION_BIT               0x00000001
#define VECTOR_TABLE_ENTRY_OFFSET(x)        ((uint32_t)(x) + (uint32_t)(SECURITY_LEVEL_BIT | THUMB_INSTRUCTION_BIT))

#if defined(__GNUC__)
#define __ATTRIBUTE_SECURE_VECTORS           __attribute__((used, section(".secure_vectors")))
#define __ATTRIBUTE_SECURE_CONFIG_DATA       __attribute__((used, section(".secure_config_data")))
#define __ATTRIBUTE_SECURE_RESET_HANDLER     __attribute__((used, section(".secure_reset_handler")))
#define __ATTRIBUTE_SECURE_FAULT_HANDLER     __attribute__((used, section(".secure_fault_handler")))
#define __ATTRIBUTE_SECURE_VECTORS_COPY      __attribute__((used, section(".secure_vectors_copy")))
#elif defined(__ICCARM__)
#define __ATTRIBUTE_SECURE_VECTORS           _Pragma("location =\"secure_vectors\"") __root
#define __ATTRIBUTE_SECURE_CONFIG_DATA       _Pragma("location =\"secure_config_data\"") __root
#define __ATTRIBUTE_SECURE_RESET_HANDLER     _Pragma("location =\"secure_reset_handler\"") __root
#define __ATTRIBUTE_SECURE_FAULT_HANDLER     _Pragma("location =\"secure_fault_handler\"") __root
#define __ATTRIBUTE_SECURE_VECTORS_COPY      _Pragma("location =\"secure_vectors_copy\"") __root
#endif

/*----------------------------------------------------------------------------
 * Structures.
 *----------------------------------------------------------------------------*/

typedef struct secure_config_data {
  uint32_t *secure_vector_table;
  uint32_t *non_secure_vector_table;
  uint32_t mspu_region_size;
} secure_config_data_t;

/*----------------------------------------------------------------------------
 * Link-time injected symbols
 *----------------------------------------------------------------------------*/
#if defined(__GNUC__)

extern uint32_t __Secure_Vectors;
extern uint32_t linker_vectors_begin;
extern uint32_t __mspu_region_size__;

__ATTRIBUTE_SECURE_CONFIG_DATA const secure_config_data_t sl_tz_secure_config_data = {
  .secure_vector_table = (uint32_t*) (((uint32_t)&__Secure_Vectors) + 0x10000000),
  .non_secure_vector_table = (uint32_t*) &linker_vectors_begin,
  .mspu_region_size = (uint32_t)&__mspu_region_size__
};

#elif defined(__ICCARM__)

#pragma language=save
#pragma language=extended
#pragma section="secure_vectors"
extern uint32_t __vector_table;
extern uint32_t __mspu_region_size__;

__ATTRIBUTE_SECURE_CONFIG_DATA const secure_config_data_t sl_tz_secure_config_data = {
  .secure_vector_table = (uint32_t*) ((uint32_t)__section_begin("secure_vectors") + 0x10000000),
  .non_secure_vector_table = (uint32_t*) &__vector_table,
  .mspu_region_size = (uint32_t)&__mspu_region_size__
};

#pragma language=restore
#endif

extern uint32_t __INITIAL_SP;

#if defined(SL_APP_PROPERTIES)
extern ApplicationProperties_t sl_app_properties;
#define APP_PROPERTIES_ADDR (void(*)(void)) & sl_app_properties
#else
#define APP_PROPERTIES_ADDR (void(*)(void))(VECTOR_TABLE_ENTRY_OFFSET(sli_tz_secure_fault_handler))
#endif

/*----------------------------------------------------------------------------
 * Trustzone Non-Secure execution pre-compiled binary.
 *----------------------------------------------------------------------------*/

// This Start-up is pre-compiled as it uses the Secure aliases to do all the
// necessary configurations for the application to execute in Non-Secure mode.

// The source from which these pre-compiled arrays are compiled can be found in
// "platform/common/src/sl_tz_non_secure_execution_source.c".

__ATTRIBUTE_SECURE_RESET_HANDLER const uint8_t sli_tz_secure_reset_handler[] = {
  0x66, 0x48, 0x5b, 0x4b, 0x78, 0x44, 0xd0, 0xf8, 0x00, 0xc0, 0x46, 0x68,
  0xc3, 0xf8, 0x08, 0xc0, 0x32, 0x68, 0x82, 0xf3, 0x88, 0x88, 0x57, 0x4a,
  0x57, 0x4c, 0x96, 0x60, 0x5a, 0x6a, 0x42, 0xf4, 0x00, 0x22, 0x5a, 0x62,
  0x55, 0x4a, 0xda, 0x60, 0xd3, 0xf8, 0x8c, 0x20, 0x42, 0xf4, 0x40, 0x62,
  0x42, 0xf0, 0x01, 0x02, 0xc3, 0xf8, 0x8c, 0x20, 0x00, 0x23, 0x4f, 0xf0,
  0xff, 0x32, 0x21, 0x46, 0x41, 0xf8, 0x23, 0x20, 0x01, 0x33, 0x10, 0x2b,
  0xfa, 0xd1, 0x4f, 0xf4, 0x50, 0x42, 0x4c, 0x4b, 0xff, 0x21, 0x9a, 0x66,
  0x01, 0x22, 0x4b, 0x4b, 0x4b, 0x4c, 0xc3, 0xf8, 0x90, 0x11, 0xc3, 0xf8,
  0x30, 0x21, 0xc4, 0xf8, 0x70, 0x11, 0x80, 0x68, 0xb0, 0xf5, 0x00, 0x3f,
  0x7b, 0xd0, 0x07, 0xd8, 0xb0, 0xf5, 0x00, 0x4f, 0x0d, 0xd0, 0xb0, 0xf5,
  0x80, 0x3f, 0x09, 0xd0, 0x00, 0x22, 0x08, 0xe0, 0xb0, 0xf5, 0x80, 0x2f,
  0x71, 0xd0, 0xb0, 0xf5, 0x00, 0x2f, 0xf7, 0xd1, 0x05, 0x22, 0x00, 0xe0,
  0x02, 0x22, 0x4f, 0xf0, 0xff, 0x37, 0xc3, 0xf8, 0x64, 0x22, 0xc3, 0xf8,
  0x70, 0x72, 0x33, 0x46, 0x5c, 0x68, 0x01, 0x21, 0x4f, 0xf0, 0x10, 0x09,
  0x38, 0x4e, 0xdf, 0xf8, 0xf4, 0xe0, 0xdf, 0xf8, 0xf4, 0x80, 0x0a, 0x00,
  0x33, 0x46, 0x1a, 0x60, 0x0a, 0x00, 0x73, 0x46, 0x1a, 0x60, 0x1a, 0x68,
  0x02, 0xea, 0x01, 0x02, 0x00, 0x2a, 0xfa, 0xd1, 0x4f, 0xf4, 0x10, 0x46,
  0x30, 0x4d, 0x2b, 0x46, 0x32, 0x46, 0x1a, 0x60, 0x06, 0xf1, 0xa2, 0x46,
  0x6f, 0xf4, 0x80, 0x18, 0xdf, 0xf8, 0xc8, 0xe0, 0x06, 0xf5, 0x83, 0x56,
  0x33, 0x46, 0x3a, 0x46, 0x1a, 0x60, 0x73, 0x46, 0x42, 0x46, 0x1a, 0x60,
  0xa6, 0xf5, 0x4f, 0x66, 0x33, 0x46, 0x0a, 0x00, 0x1a, 0x60, 0x26, 0x4d,
  0xf0, 0x3e, 0x29, 0x46, 0x2c, 0xf0, 0x7f, 0x43, 0xb3, 0xfb, 0xf0, 0xf3,
  0x20, 0x2b, 0x05, 0xd9, 0xa3, 0xf1, 0x20, 0x03, 0x4f, 0xf0, 0xff, 0x32,
  0x0a, 0x60, 0x08, 0xe0, 0x4f, 0xf0, 0x01, 0x02, 0x9a, 0x40, 0x6f, 0xea,
  0x02, 0x02, 0x0a, 0x60, 0x4f, 0xf0, 0xff, 0x33, 0xff, 0xe7, 0x01, 0xf1,
  0x04, 0x01, 0xb1, 0x42, 0xea, 0xdd, 0x19, 0x49, 0x64, 0x08, 0x64, 0x00,
  0x02, 0x22, 0x0b, 0x46, 0x1a, 0x60, 0x00, 0x20, 0x00, 0x21, 0x00, 0x22,
  0x00, 0x23, 0x00, 0x25, 0x00, 0x26, 0x00, 0x27, 0x5f, 0xf0, 0x00, 0x08,
  0x5f, 0xf0, 0x00, 0x09, 0x5f, 0xf0, 0x00, 0x0a, 0x5f, 0xf0, 0x00, 0x0b,
  0x5f, 0xf0, 0x00, 0x0c, 0xbf, 0xf3, 0x4f, 0x8f, 0x24, 0x47, 0x03, 0x22,
  0x93, 0xe7, 0x04, 0x22, 0x91, 0xe7, 0x00, 0xbf, 0x00, 0xed, 0x00, 0xe0,
  0x00, 0xed, 0x02, 0xe0, 0x80, 0xe3, 0x00, 0xe0, 0x00, 0x20, 0xfa, 0x05,
  0x00, 0x50, 0x80, 0x40, 0x00, 0x90, 0x00, 0x51, 0x00, 0xa0, 0x00, 0x51,
  0x18, 0x50, 0x83, 0x40, 0x68, 0x60, 0x80, 0x40, 0x74, 0x92, 0x00, 0x51,
  0xd0, 0xed, 0x00, 0xe0, 0xec, 0xff, 0xff, 0xff, 0x18, 0xd0, 0x85, 0x40,
  0x14, 0xc0, 0x85, 0x40, 0x64, 0xa0, 0x00, 0x51,
};

__ATTRIBUTE_SECURE_FAULT_HANDLER const uint8_t sli_tz_secure_fault_handler[] = {
  0xfe, 0xe7
};

#define SECURE_VECTOR_TABLE {                                                                                          \
    { .topOfStack = &__INITIAL_SP },                                             /*      Initial Stack Pointer      */ \
    { (void(*)(void))(VECTOR_TABLE_ENTRY_OFFSET(sli_tz_secure_reset_handler)) }, /*      Secure_Reset_Handler       */ \
    { (void(*)(void))(VECTOR_TABLE_ENTRY_OFFSET(sli_tz_secure_fault_handler)) }, /*      Secure_Default_Handler     */ \
    { (void(*)(void))(VECTOR_TABLE_ENTRY_OFFSET(sli_tz_secure_fault_handler)) }, /*      Secure_Default_Handler     */ \
    { (void(*)(void))(VECTOR_TABLE_ENTRY_OFFSET(sli_tz_secure_fault_handler)) }, /*      Secure_Default_Handler     */ \
    { (void(*)(void))(VECTOR_TABLE_ENTRY_OFFSET(sli_tz_secure_fault_handler)) }, /*      Secure_Default_Handler     */ \
    { (void(*)(void))(VECTOR_TABLE_ENTRY_OFFSET(sli_tz_secure_fault_handler)) }, /*      Secure_Default_Handler     */ \
    { (void(*)(void))(VECTOR_TABLE_ENTRY_OFFSET(sli_tz_secure_fault_handler)) }, /*      Secure_Default_Handler     */ \
    { (void(*)(void))(VECTOR_TABLE_ENTRY_OFFSET(sli_tz_secure_fault_handler)) }, /*      Secure_Default_Handler     */ \
    { (void(*)(void))(VECTOR_TABLE_ENTRY_OFFSET(sli_tz_secure_fault_handler)) }, /*      Secure_Default_Handler     */ \
    { (void(*)(void))(VECTOR_TABLE_ENTRY_OFFSET(sli_tz_secure_fault_handler)) }, /*      Secure_Default_Handler     */ \
    { (void(*)(void))(VECTOR_TABLE_ENTRY_OFFSET(sli_tz_secure_fault_handler)) }, /*      Secure_Default_Handler     */ \
    { (void(*)(void))(VECTOR_TABLE_ENTRY_OFFSET(sli_tz_secure_fault_handler)) }, /*      Secure_Default_Handler     */ \
    { APP_PROPERTIES_ADDR },                                                     /*      Application properties     */ \
    { (void(*)(void))(VECTOR_TABLE_ENTRY_OFFSET(sli_tz_secure_fault_handler)) }, /*      Secure_Default_Handler     */ \
    { (void(*)(void))(VECTOR_TABLE_ENTRY_OFFSET(sli_tz_secure_fault_handler)) }, /*      Secure_Default_Handler     */ \
}

// Secure Vector Table.
__ATTRIBUTE_SECURE_VECTORS const tVectorEntry sli_tz_secure_vectors[TOTAL_INTERNAL_INTERRUPTS] = SECURE_VECTOR_TABLE;

// When a bootloader is present, the Non-Secure Execution Start-up might need to be pushed further in memory to
// make sure the start address is aligned with a MSPU region. In that case, another instance of the Secure Vector Table is
// flashed after the bootloader where it would be expected to be.
__ATTRIBUTE_SECURE_VECTORS_COPY const tVectorEntry sli_tz_secure_vectors_copy[TOTAL_INTERNAL_INTERRUPTS] = SECURE_VECTOR_TABLE;
