/***************************************************************************//**
 * @file
 * @brief KEYSCAN HAL implementation.
 *******************************************************************************
 * # License
 * <b>Copyright 2018 Silicon Laboratories Inc. www.silabs.com</b>
 *******************************************************************************
 *
 * SPDX-License-Identifier: Zlib
 *
 * The licensor of this software is Silicon Laboratories Inc.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 ******************************************************************************/

#include <stdint.h>
#include <stdbool.h>

#include "em_core.h"
#include "em_device.h"
#include "sl_assert.h"
#include "sl_clock_manager.h"
#include "sl_gpio.h"
#include "sl_hal_keyscan.h"
#include "keyscan_driver_hal.h"
#include "keyscan_driver_config.h"
#include "sl_device_peripheral.h"

/*******************************************************************************
 *******************************   DEFINES   ***********************************
 ******************************************************************************/

// A period of 2 ms must be obtained by dividing the Peripheral Clock with the
// internal keyscan clock divider.
#define KEYSCAN_CLOCK_PERIOD_MS  2U
#define KEYSCAN_CLOCK_FREQUENCY  (1000U / KEYSCAN_CLOCK_PERIOD_MS)

/***************************************************************************//**
 * Initializes the Keyscan peripheral.
 ******************************************************************************/
void sli_keyscan_driver_hal_init(void)
{
  sl_hal_keyscan_config_t hw_config;
  uint32_t clock_freq;
  uint32_t clock_div;
  sl_clock_branch_t clock_branch;

  // Set hardware configuration
  hw_config.auto_start_enable = false;
  hw_config.column_number = SL_KEYSCAN_DRIVER_COLUMN_NUMBER;
  hw_config.row_number = SL_KEYSCAN_DRIVER_ROW_NUMBER;
  hw_config.scan_delay = SL_KEYSCAN_DRIVER_SCAN_DELAY_MS;
  hw_config.debounce_delay = SL_KEYSCAN_DRIVER_DEBOUNCE_DELAY_MS;
  hw_config.stable_delay = SL_KEYSCAN_DRIVER_STABLE_DELAY_MS;
  hw_config.single_press_enable = SL_KEYSCAN_DRIVER_SINGLEPRESS;

  // Clock initialization
  sl_clock_manager_enable_bus_clock(SL_BUS_CLOCK_KEYSCAN);

  clock_branch = sl_device_peripheral_get_clock_branch(SL_PERIPHERAL_KEYSCAN);
  sl_clock_manager_get_clock_branch_frequency(clock_branch, &clock_freq);

  clock_div = clock_freq / KEYSCAN_CLOCK_FREQUENCY - 1U;
  hw_config.clock_divider = clock_div;

  // Hardware initialization
  sl_hal_keyscan_init(&hw_config);

  // Clear interrupt flags
  sl_hal_keyscan_disable_interrupts(_KEYSCAN_IEN_MASK);
  sl_hal_keyscan_clear_interrupts(_KEYSCAN_IEN_MASK);

  // Enable interrupts
  NVIC_ClearPendingIRQ(KEYSCAN_IRQn);
  NVIC_EnableIRQ(KEYSCAN_IRQn);
}

/***************************************************************************//**
 * Initializes the Keyscan peripheral GPIO.
 ******************************************************************************/
void sli_keyscan_driver_hal_init_gpio(void)
{
  sl_gpio_t gpio;

  // GPIO setup
  for (uint8_t i = 0; i < SL_KEYSCAN_DRIVER_COLUMN_NUMBER; i++) {
    GPIO->KEYSCANROUTE.ROUTEEN |= (1 << i);
  }

#if (SL_KEYSCAN_DRIVER_COLUMN_NUMBER >= 1u)
  GPIO->KEYSCANROUTE.COLOUT0ROUTE = (SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_0_PORT << _GPIO_KEYSCAN_COLOUT0ROUTE_PORT_SHIFT)
                                    | (SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_0_PIN << _GPIO_KEYSCAN_COLOUT0ROUTE_PIN_SHIFT);
  gpio.port = SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_0_PORT;
  gpio.pin = SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_0_PIN;
  sl_gpio_set_pin_mode(&gpio, SL_GPIO_MODE_WIRED_AND, 1);

#if (SL_KEYSCAN_DRIVER_COLUMN_NUMBER >= 2u)
  GPIO->KEYSCANROUTE.COLOUT1ROUTE = (SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_1_PORT << _GPIO_KEYSCAN_COLOUT1ROUTE_PORT_SHIFT)
                                    | (SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_1_PIN << _GPIO_KEYSCAN_COLOUT1ROUTE_PIN_SHIFT);
  gpio.port = SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_1_PORT;
  gpio.pin = SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_1_PIN;
  sl_gpio_set_pin_mode(&gpio, SL_GPIO_MODE_WIRED_AND, 1);

#if (SL_KEYSCAN_DRIVER_COLUMN_NUMBER >= 3u)
  GPIO->KEYSCANROUTE.COLOUT2ROUTE = (SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_2_PORT << _GPIO_KEYSCAN_COLOUT2ROUTE_PORT_SHIFT)
                                    | (SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_2_PIN << _GPIO_KEYSCAN_COLOUT2ROUTE_PIN_SHIFT);
  gpio.port = SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_2_PORT;
  gpio.pin = SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_2_PIN;
  sl_gpio_set_pin_mode(&gpio, SL_GPIO_MODE_WIRED_AND, 1);

#if (SL_KEYSCAN_DRIVER_COLUMN_NUMBER >= 4u)
  GPIO->KEYSCANROUTE.COLOUT3ROUTE = (SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_3_PORT << _GPIO_KEYSCAN_COLOUT3ROUTE_PORT_SHIFT)
                                    | (SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_3_PIN << _GPIO_KEYSCAN_COLOUT3ROUTE_PIN_SHIFT);
  gpio.port = SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_3_PORT;
  gpio.pin = SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_3_PIN;
  sl_gpio_set_pin_mode(&gpio, SL_GPIO_MODE_WIRED_AND, 1);

#if (SL_KEYSCAN_DRIVER_COLUMN_NUMBER >= 5u)
  GPIO->KEYSCANROUTE.COLOUT4ROUTE = (SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_4_PORT << _GPIO_KEYSCAN_COLOUT4ROUTE_PORT_SHIFT)
                                    | (SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_4_PIN << _GPIO_KEYSCAN_COLOUT4ROUTE_PIN_SHIFT);
  gpio.port = SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_4_PORT;
  gpio.pin = SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_4_PIN;
  sl_gpio_set_pin_mode(&gpio, SL_GPIO_MODE_WIRED_AND, 1);

#if (SL_KEYSCAN_DRIVER_COLUMN_NUMBER >= 6u)
  GPIO->KEYSCANROUTE.COLOUT5ROUTE = (SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_5_PORT << _GPIO_KEYSCAN_COLOUT5ROUTE_PORT_SHIFT)
                                    | (SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_5_PIN << _GPIO_KEYSCAN_COLOUT5ROUTE_PIN_SHIFT);
  gpio.port = SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_5_PORT;
  gpio.pin = SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_5_PIN;
  sl_gpio_set_pin_mode(&gpio, SL_GPIO_MODE_WIRED_AND, 1);

#if (SL_KEYSCAN_DRIVER_COLUMN_NUMBER >= 7u)
  GPIO->KEYSCANROUTE.COLOUT6ROUTE = (SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_6_PORT << _GPIO_KEYSCAN_COLOUT6ROUTE_PORT_SHIFT)
                                    | (SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_6_PIN << _GPIO_KEYSCAN_COLOUT6ROUTE_PIN_SHIFT);
  gpio.port = SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_6_PORT;
  gpio.pin = SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_6_PIN;
  sl_gpio_set_pin_mode(&gpio, SL_GPIO_MODE_WIRED_AND, 1);

#if (SL_KEYSCAN_DRIVER_COLUMN_NUMBER >= 8u)
  GPIO->KEYSCANROUTE.COLOUT7ROUTE = (SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_7_PORT << _GPIO_KEYSCAN_COLOUT7ROUTE_PORT_SHIFT)
                                    | (SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_7_PIN << _GPIO_KEYSCAN_COLOUT7ROUTE_PIN_SHIFT);
  gpio.port = SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_7_PORT;
  gpio.pin = SL_KEYSCAN_DRIVER_KEYSCAN_COL_OUT_7_PIN;
  sl_gpio_set_pin_mode(&gpio, SL_GPIO_MODE_WIRED_AND, 1);

#endif
#endif
#endif
#endif
#endif
#endif
#endif
#endif

#if (SL_KEYSCAN_DRIVER_ROW_NUMBER >= 3u)
  GPIO->KEYSCANROUTE.ROWSENSE0ROUTE = (SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_0_PORT << _GPIO_KEYSCAN_ROWSENSE0ROUTE_PORT_SHIFT)
                                      | (SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_0_PIN << _GPIO_KEYSCAN_ROWSENSE0ROUTE_PIN_SHIFT);
  gpio.port = SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_0_PORT;
  gpio.pin = SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_0_PIN;
  sl_gpio_set_pin_mode(&gpio, SL_GPIO_MODE_INPUT_PULL, 1);

  GPIO->KEYSCANROUTE.ROWSENSE1ROUTE = (SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_1_PORT << _GPIO_KEYSCAN_ROWSENSE1ROUTE_PORT_SHIFT)
                                      | (SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_1_PIN << _GPIO_KEYSCAN_ROWSENSE1ROUTE_PIN_SHIFT);
  gpio.port = SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_1_PORT;
  gpio.pin = SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_1_PIN;
  sl_gpio_set_pin_mode(&gpio, SL_GPIO_MODE_INPUT_PULL, 1);

  GPIO->KEYSCANROUTE.ROWSENSE2ROUTE = (SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_2_PORT << _GPIO_KEYSCAN_ROWSENSE2ROUTE_PORT_SHIFT)
                                      | (SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_2_PIN << _GPIO_KEYSCAN_ROWSENSE2ROUTE_PIN_SHIFT);
  gpio.port = SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_2_PORT;
  gpio.pin = SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_2_PIN;
  sl_gpio_set_pin_mode(&gpio, SL_GPIO_MODE_INPUT_PULL, 1);

  // KEYSCAN_E301 - Connect unused rows 3, 4, and 5 to row 2, a single used row
  GPIO->KEYSCANROUTE.ROWSENSE3ROUTE = SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_2_PORT << _GPIO_KEYSCAN_ROWSENSE3ROUTE_PORT_SHIFT
                                      | SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_2_PIN << _GPIO_KEYSCAN_ROWSENSE3ROUTE_PIN_SHIFT;
  GPIO->KEYSCANROUTE.ROWSENSE4ROUTE = SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_2_PORT << _GPIO_KEYSCAN_ROWSENSE4ROUTE_PORT_SHIFT
                                      | SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_2_PIN << _GPIO_KEYSCAN_ROWSENSE4ROUTE_PIN_SHIFT;
  GPIO->KEYSCANROUTE.ROWSENSE5ROUTE = SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_2_PORT << _GPIO_KEYSCAN_ROWSENSE5ROUTE_PORT_SHIFT
                                      | SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_2_PIN << _GPIO_KEYSCAN_ROWSENSE5ROUTE_PIN_SHIFT;

#if (SL_KEYSCAN_DRIVER_ROW_NUMBER >= 4u)
  GPIO->KEYSCANROUTE.ROWSENSE3ROUTE = (SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_3_PORT << _GPIO_KEYSCAN_ROWSENSE3ROUTE_PORT_SHIFT)
                                      | (SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_3_PIN << _GPIO_KEYSCAN_ROWSENSE3ROUTE_PIN_SHIFT);
  gpio.port = SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_3_PORT;
  gpio.pin = SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_3_PIN;
  sl_gpio_set_pin_mode(&gpio, SL_GPIO_MODE_INPUT_PULL, 1);

  // KEYSCAN_E301 - Connect unused rows 4, and 5 to row 3, a single used row
  GPIO->KEYSCANROUTE.ROWSENSE4ROUTE = SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_3_PORT << _GPIO_KEYSCAN_ROWSENSE4ROUTE_PORT_SHIFT
                                      | SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_3_PIN << _GPIO_KEYSCAN_ROWSENSE4ROUTE_PIN_SHIFT;
  GPIO->KEYSCANROUTE.ROWSENSE5ROUTE = SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_3_PORT << _GPIO_KEYSCAN_ROWSENSE5ROUTE_PORT_SHIFT
                                      | SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_3_PIN << _GPIO_KEYSCAN_ROWSENSE5ROUTE_PIN_SHIFT;

#if (SL_KEYSCAN_DRIVER_ROW_NUMBER >= 5u)
  GPIO->KEYSCANROUTE.ROWSENSE4ROUTE = (SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_4_PORT << _GPIO_KEYSCAN_ROWSENSE4ROUTE_PORT_SHIFT)
                                      | (SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_4_PIN << _GPIO_KEYSCAN_ROWSENSE4ROUTE_PIN_SHIFT);
  gpio.port = SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_4_PORT;
  gpio.pin = SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_4_PIN;
  sl_gpio_set_pin_mode(&gpio, SL_GPIO_MODE_INPUT_PULL, 1);

  // KEYSCAN_E301 - Connect unused row 5 to row 4, a single used row
  GPIO->KEYSCANROUTE.ROWSENSE5ROUTE = SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_4_PORT << _GPIO_KEYSCAN_ROWSENSE5ROUTE_PORT_SHIFT
                                      | SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_4_PIN << _GPIO_KEYSCAN_ROWSENSE5ROUTE_PIN_SHIFT;

#if (SL_KEYSCAN_DRIVER_ROW_NUMBER >= 6u)
  GPIO->KEYSCANROUTE.ROWSENSE5ROUTE = (SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_5_PORT << _GPIO_KEYSCAN_ROWSENSE5ROUTE_PORT_SHIFT)
                                      | (SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_5_PIN << _GPIO_KEYSCAN_ROWSENSE5ROUTE_PIN_SHIFT);
  gpio.port = SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_5_PORT;
  gpio.pin = SL_KEYSCAN_DRIVER_KEYSCAN_ROW_SENSE_5_PIN;
  sl_gpio_set_pin_mode(&gpio, SL_GPIO_MODE_INPUT_PULL, 1);
#endif
#endif
#endif
#endif
}

/***************************************************************************//**
 * Enables Keyscan interrupts by passing a set of flags.
 ******************************************************************************/
void sli_keyscan_driver_hal_enable_interrupts(uint8_t local_flags)
{
  uint32_t keyscan_ien = 0u;

  if (local_flags & KEYSCAN_DRIVER_EVENT_WAKEUP) {
    keyscan_ien |= KEYSCAN_IF_WAKEUP;
  }

  if (local_flags & KEYSCAN_DRIVER_EVENT_KEY) {
    keyscan_ien |= KEYSCAN_IF_KEY;
  }

  if (local_flags & KEYSCAN_DRIVER_EVENT_NOKEY) {
    keyscan_ien |= KEYSCAN_IF_NOKEY;
  }

  if (local_flags & KEYSCAN_DRIVER_EVENT_SCANNED) {
    keyscan_ien |= KEYSCAN_IF_SCANNED;
  }

  sl_hal_keyscan_enable_interrupts(keyscan_ien);
}

/***************************************************************************//**
 * Gets the column and row information from the peripheral when a key is pressed.
 ******************************************************************************/
void sli_keyscan_driver_hal_get_column_row(uint8_t *p_column,
                                           uint8_t *p_row)
{
  uint32_t status;
  uint32_t mask = 0UL;

  status = sl_hal_keyscan_get_status();

  /* KEYSCAN_E301: The unused row bits in the KEYSCAN_STATUS field should be masked
     so that unused row bits are set to 1, indicating a key is not pressed. */
  mask = ((1 <<  SL_KEYSCAN_DRIVER_ROW_NUMBER) - 1) << _KEYSCAN_STATUS_ROW_SHIFT;
  status |= (mask ^ _KEYSCAN_STATUS_ROW_MASK);

  *p_column = (status & _KEYSCAN_STATUS_COL_MASK) >> _KEYSCAN_STATUS_COL_SHIFT;
  *p_row = (status & _KEYSCAN_STATUS_ROW_MASK) >> _KEYSCAN_STATUS_ROW_SHIFT;
  /* Invert Row Detection Status (0: pull up for no key press, 1: key press). */
  *p_row = ~*p_row;
  *p_row &= _KEYSCAN_STATUS_ROW_MASK;
}

/***************************************************************************//**
 * Return if keyscan scan is currently running or not.
 ******************************************************************************/
bool sli_keyscan_driver_hal_is_scan_running(void)
{
  uint32_t status;

  status = sl_hal_keyscan_get_status();

  return ((status & _KEYSCAN_STATUS_RUNNING_MASK) >> _KEYSCAN_STATUS_RUNNING_SHIFT ? true : false);
}

/***************************************************************************//**
 * Starts the keyscan scan.
 ******************************************************************************/
void sli_keyscan_driver_hal_start_scan(bool enable)
{
  if (enable) {
    sl_hal_keyscan_enable();
  }

  sl_hal_keyscan_start_scan();

  sl_hal_keyscan_wait_sync();
}

/***************************************************************************//**
 * Stops the keyscan scan.
 ******************************************************************************/
void sli_keyscan_driver_hal_stop_scan(void)
{
  sl_hal_keyscan_stop_scan();

  sl_hal_keyscan_wait_sync();

  sl_hal_keyscan_disable();
}

/***************************************************************************//**
 * KEYSCAN interrupt handler.
 ******************************************************************************/
void KEYSCAN_IRQHandler(void)
{
  uint32_t irq_flags;
  uint8_t local_flags = 0;

  // Retrieve Keyscan interrupt flags
  irq_flags = sl_hal_keyscan_get_interrupts();

  // Interrupt "wake-up from sleep" handling.
  if (irq_flags & KEYSCAN_IF_WAKEUP) {
    local_flags |= KEYSCAN_DRIVER_EVENT_WAKEUP;
  }

  // Interrupt "key pressed detected" handling.
  if (irq_flags & KEYSCAN_IF_KEY) {
    local_flags |= KEYSCAN_DRIVER_EVENT_KEY;
  }

  // Interrupt "all keys pressed now released" handling.
  if (irq_flags & KEYSCAN_IF_NOKEY) {
    local_flags |= KEYSCAN_DRIVER_EVENT_NOKEY;
  }

  // Interrupt "scanned complete with no key pressed" handling.
  if (irq_flags & KEYSCAN_IF_SCANNED) {
    local_flags |= KEYSCAN_DRIVER_EVENT_SCANNED;
  }

  // Clear all interrupt flags
  sl_hal_keyscan_clear_interrupts(irq_flags);

  // Process interrupts
  sli_keyscan_process_irq(local_flags);
}
