/***************************************************************************//**
 * @file
 * @brief Direct memory access (LDMA) API
 *******************************************************************************
 * # License
 * <b>Copyright 2023 Silicon Laboratories Inc. www.silabs.com</b>
 *******************************************************************************
 *
 * SPDX-License-Identifier: Zlib
 *
 * The licensor of this software is Silicon Laboratories Inc.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 ******************************************************************************/

#ifndef SL_HAL_LDMA_H
#define SL_HAL_LDMA_H

#include "em_device.h"

#if defined(LDMA_PRESENT)

#ifdef __cplusplus
extern "C" {
#endif

#include <stdbool.h>

#include "sl_assert.h"
#include "sl_enum.h"
#include "sl_code_classification.h"

/***************************************************************************//**
 * @addtogroup ldma LDMA - Linked DMA
 * @brief Linked Direct Memory Access (LDMA) Peripheral API
 * @{
 ******************************************************************************/

/*******************************************************************************
 ********************************   ENUMS   ************************************
 ******************************************************************************/

/// Controls the number of unit data transfers per arbitration
/// cycle, providing a means to balance LDMA channels' load on the controller.
SL_ENUM(sl_hal_ldma_ctrl_block_size_t) {
  SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_1    = _LDMA_CH_CTRL_BLOCKSIZE_UNIT1,    ///< One transfer per arbitration.
  SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_2    = _LDMA_CH_CTRL_BLOCKSIZE_UNIT2,    ///< Two transfers per arbitration.
  SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_3    = _LDMA_CH_CTRL_BLOCKSIZE_UNIT3,    ///< Three transfers per arbitration.
  SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_4    = _LDMA_CH_CTRL_BLOCKSIZE_UNIT4,    ///< Four transfers per arbitration.
  SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_6    = _LDMA_CH_CTRL_BLOCKSIZE_UNIT6,    ///< Six transfers per arbitration.
  SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_8    = _LDMA_CH_CTRL_BLOCKSIZE_UNIT8,    ///< Eight transfers per arbitration.
  SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_16   = _LDMA_CH_CTRL_BLOCKSIZE_UNIT16,   ///< 16 transfers per arbitration.
  SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_32   = _LDMA_CH_CTRL_BLOCKSIZE_UNIT32,   ///< 32 transfers per arbitration.
  SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_64   = _LDMA_CH_CTRL_BLOCKSIZE_UNIT64,   ///< 64 transfers per arbitration.
  SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_128  = _LDMA_CH_CTRL_BLOCKSIZE_UNIT128,  ///< 128 transfers per arbitration.
  SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_256  = _LDMA_CH_CTRL_BLOCKSIZE_UNIT256,  ///< 256 transfers per arbitration.
  SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_512  = _LDMA_CH_CTRL_BLOCKSIZE_UNIT512,  ///< 512 transfers per arbitration.
  SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_1024 = _LDMA_CH_CTRL_BLOCKSIZE_UNIT1024, ///< 1024 transfers per arbitration.
  SL_HAL_LDMA_CTRL_BLOCK_SIZE_ALL       = _LDMA_CH_CTRL_BLOCKSIZE_ALL       ///< Lock arbitration during transfer.
};

/// LDMA structure type.
SL_ENUM(sl_hal_ldma_ctrl_struct_type_t) {
  SL_HAL_LDMA_CTRL_STRUCT_TYPE_XFER  = _LDMA_CH_CTRL_STRUCTTYPE_TRANSFER,    ///< TRANSFER transfer type.
  SL_HAL_LDMA_CTRL_STRUCT_TYPE_SYNC  = _LDMA_CH_CTRL_STRUCTTYPE_SYNCHRONIZE, ///< SYNCHRONIZE transfer type.
  SL_HAL_LDMA_CTRL_STRUCT_TYPE_WRITE = _LDMA_CH_CTRL_STRUCTTYPE_WRITE        ///< WRITE transfer type.
};

/// LDMA transfer block or cycle selector.
SL_ENUM(sl_hal_ldma_ctrl_req_mode_t) {
  SL_HAL_LDMA_CTRL_REQ_MODE_BLOCK = _LDMA_CH_CTRL_REQMODE_BLOCK, ///< Each LDMA request trigger transfer of one block.
  SL_HAL_LDMA_CTRL_REQ_MODE_ALL   = _LDMA_CH_CTRL_REQMODE_ALL    ///< A LDMA request trigger transfer of a complete cycle.
};

/// Source address increment unit size.
SL_ENUM(sl_hal_ldma_ctrl_src_inc_t) {
  SL_HAL_LDMA_CTRL_SRC_INC_ONE  = _LDMA_CH_CTRL_SRCINC_ONE,  ///< Increment source address by one unit data size.
  SL_HAL_LDMA_CTRL_SRC_INC_TWO  = _LDMA_CH_CTRL_SRCINC_TWO,  ///< Increment source address by two unit data sizes.
  SL_HAL_LDMA_CTRL_SRC_INC_FOUR = _LDMA_CH_CTRL_SRCINC_FOUR, ///< Increment source address by four unit data sizes.
  SL_HAL_LDMA_CTRL_SRC_INC_NONE = _LDMA_CH_CTRL_SRCINC_NONE  ///< Do not increment source address.
};

/// LDMA transfer unit size.
SL_ENUM(sl_hal_ldma_ctrl_size_t) {
  SL_HAL_LDMA_CTRL_SIZE_BYTE = _LDMA_CH_CTRL_SIZE_BYTE,     ///< Each unit transfer is a byte.
  SL_HAL_LDMA_CTRL_SIZE_HALF = _LDMA_CH_CTRL_SIZE_HALFWORD, ///< Each unit transfer is a half-word.
  SL_HAL_LDMA_CTRL_SIZE_WORD = _LDMA_CH_CTRL_SIZE_WORD      ///< Each unit transfer is a word.
};

/// Destination address increment unit size.
SL_ENUM(sl_hal_ldma_ctrl_dst_inc_t) {
  SL_HAL_LDMA_CTRL_DST_INC_ONE  = _LDMA_CH_CTRL_DSTINC_ONE,  ///< Increment destination address by one unit data size.
  SL_HAL_LDMA_CTRL_DST_INC_TWO  = _LDMA_CH_CTRL_DSTINC_TWO,  ///< Increment destination address by two unit data sizes.
  SL_HAL_LDMA_CTRL_DST_INC_FOUR = _LDMA_CH_CTRL_DSTINC_FOUR, ///< Increment destination address by four unit data sizes.
  SL_HAL_LDMA_CTRL_DST_INC_NONE = _LDMA_CH_CTRL_DSTINC_NONE  ///< Do not increment destination address.
};

/// Source addressing mode.
SL_ENUM(sl_hal_ldma_ctrl_src_addr_mode_t) {
  SL_HAL_LDMA_CTRL_SRC_ADDR_MODE_ABS = _LDMA_CH_CTRL_SRCMODE_ABSOLUTE, ///< Address fetched from a linked structure is absolute.
  SL_HAL_LDMA_CTRL_SRC_ADDR_MODE_REL = _LDMA_CH_CTRL_SRCMODE_RELATIVE  ///< Address fetched from a linked structure is relative.
};

/// Destination addressing mode.
SL_ENUM(sl_hal_ldma_ctrl_dst_addr_mode_t) {
  SL_HAL_LDMA_CTRL_DST_ADDR_MODE_ABS = _LDMA_CH_CTRL_DSTMODE_ABSOLUTE, ///< Address fetched from a linked structure is absolute.
  SL_HAL_LDMA_CTRL_DST_ADDR_MODE_REL = _LDMA_CH_CTRL_DSTMODE_RELATIVE  ///< Address fetched from a linked structure is relative.
};

/// LDMA link load address mode.
SL_ENUM(sl_hal_ldma_link_mode_t) {
  SL_HAL_LDMA_LINK_MODE_ABS = _LDMA_CH_LINK_LINKMODE_ABSOLUTE, ///< Link address is an absolute address value.
  SL_HAL_LDMA_LINK_MODE_REL = _LDMA_CH_LINK_LINKMODE_RELATIVE  ///< Link address is a two's complement relative address.
};

/// Insert extra arbitration slots to increase channel arbitration priority.
SL_ENUM(sl_hal_ldma_cfg_arb_slots_t) {
  SL_HAL_LDMA_CFG_ARBSLOTS_ONE   = _LDMA_CH_CFG_ARBSLOTS_ONE,  ///< One arbitration slot selected.
  SL_HAL_LDMA_CFG_ARBSLOTS_TWO   = _LDMA_CH_CFG_ARBSLOTS_TWO,  ///< Two arbitration slots selected.
  SL_HAL_LDMA_CFG_ARBSLOTS_FOUR  = _LDMA_CH_CFG_ARBSLOTS_FOUR, ///< Four arbitration slots selected.
  SL_HAL_LDMA_CFG_ARBSLOTS_EIGHT = _LDMA_CH_CFG_ARBSLOTS_EIGHT ///< Eight arbitration slots selected.
};

/// Source address increment sign.
SL_ENUM(sl_hal_ldma_cfg_src_inc_sign_t) {
  SL_HAL_LDMA_CFG_SRC_INC_SIGN_POS = _LDMA_CH_CFG_SRCINCSIGN_POSITIVE, ///< Increment source address.
  SL_HAL_LDMA_CFG_SRC_INC_SIGN_NEG = _LDMA_CH_CFG_SRCINCSIGN_NEGATIVE  ///< Decrement source address.
};

/// Destination address increment sign.
SL_ENUM(sl_hal_ldma_cfg_dst_inc_sign_t) {
  SL_HAL_LDMA_CFG_DST_INC_SIGN_POS = _LDMA_CH_CFG_DSTINCSIGN_POSITIVE, ///< Increment destination address.
  SL_HAL_LDMA_CFG_DST_INC_SIGN_NEG = _LDMA_CH_CFG_DSTINCSIGN_NEGATIVE  ///< Decrement destination address.
};

#if defined(_LDMA_CH_CFG_STRUCTBUSPORT_MASK)
/// Structure fetch operation bus port.
SL_ENUM(sl_hal_ldma_cfg_struct_bus_port_t) {
  SL_HAL_LDMA_CFG_STRUCT_BUS_PORT_0 = _LDMA_CH_CFG_STRUCTBUSPORT_AHBM0, ///< AHB Master 0 port.
  SL_HAL_LDMA_CFG_STRUCT_BUS_PORT_1 = _LDMA_CH_CFG_STRUCTBUSPORT_AHBM1  ///< AHB Master 1 port.
};
#endif

#if defined(_LDMA_CH_CFG_SRCBUSPORT_MASK)
/// Source operation bus port.
SL_ENUM(sl_hal_ldma_cfg_src_bus_port_t) {
  SL_HAL_LDMA_CFG_SRC_BUS_PORT_0 = _LDMA_CH_CFG_SRCBUSPORT_AHBM0, ///< AHB Master 0 port.
  SL_HAL_LDMA_CFG_SRC_BUS_PORT_1 = _LDMA_CH_CFG_SRCBUSPORT_AHBM1  ///< AHB Master 1 port.
};
#endif

#if defined(_LDMA_CH_CFG_DSTBUSPORT_MASK)
/// Destination operation bus port.
SL_ENUM(sl_hal_ldma_cfg_dst_bus_port_t) {
  SL_HAL_LDMA_CFG_DST_BUS_PORT_0 = _LDMA_CH_CFG_DSTBUSPORT_AHBM0, ///< AHB Master 0 port.
  SL_HAL_LDMA_CFG_DST_BUS_PORT_1 = _LDMA_CH_CFG_DSTBUSPORT_AHBM1  ///< AHB Master 1 port.
};
#endif

#if defined(_LDMA_CH_XCTRL_ILMODE_MASK)
/// Rules table mode (interleaving destination).
SL_ENUM(sl_hal_ldma_rules_table_mode_t) {
  SL_HAL_LDMA_MODE_ABSOLUTE    = _LDMA_CH_XCTRL_ILMODE_ABSOLUTE,    ///< Address by value in rules. Size of WORD
  SL_HAL_LDMA_MODE_RELATIVE_16 = _LDMA_CH_XCTRL_ILMODE_RELATIVE16,  ///< Address by adding rules to DST. Size of HALFWORD
  SL_HAL_LDMA_MODE_RELATIVE_8  = _LDMA_CH_XCTRL_ILMODE_RELATIVE8,   ///< Address by adding rules to DST. Size of BYTE
#if defined(_LDMA_CH_XCTRL_ILMODE_BITMASK)
  SL_HAL_LDMA_MODE_BITMASK     = _LDMA_CH_XCTRL_ILMODE_BITMASK      ///< Address offset by bit enable mask in rules. Size of WORD
#endif
};
#endif

#if defined(_LDMA_CH_XCTRL_SRCMEMTYPE_MASK)
/// Type of memory access.
SL_ENUM(sl_hal_ldma_mem_type_t) {
  SL_HAL_LDMA_MEM_TYPE_NON_CACHEABLE = _LDMA_CH_XCTRL_SRCMEMTYPE_DEVNB, ///< Device Non-Bufferable
  SL_HAL_LDMA_MEM_TYPE_CACHEABLE     = _LDMA_CH_XCTRL_SRCMEMTYPE_WBRW   ///< Write-Back Read and Write-Allocate
};
#endif

/// Peripherals that can trigger LDMA transfers.
SL_ENUM_GENERIC(sl_hal_ldma_peripheral_signal_t, uint32_t) {
#if defined(_SILICON_LABS_32B_SERIES_3)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_NONE = LDMAXBAR0_CH_REQSEL_SOURCESEL_NONE,                                                                ///< No peripheral selected for LDMA triggering.
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_LDMAXBAR0PRSREQ0)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_LDMAXBAR0_PRSREQ0 = LDMAXBAR0_CH_REQSEL_SIGSEL_LDMAXBAR0PRSREQ0 | LDMAXBAR0_CH_REQSEL_SOURCESEL_LDMAXBAR0,    ///< Trigger on PRS REQ0.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_LDMAXBAR0PRSREQ1)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_LDMAXBAR0_PRSREQ1 = LDMAXBAR0_CH_REQSEL_SIGSEL_LDMAXBAR0PRSREQ1 | LDMAXBAR0_CH_REQSEL_SOURCESEL_LDMAXBAR0,    ///< Trigger on PRS REQ1.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER0CC0)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER0_CC0 = LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER0CC0 | LDMAXBAR0_CH_REQSEL_SOURCESEL_TIMER0,                  ///< Trigger on TIMER0_CC0.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER0CC1)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER0_CC1 = LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER0CC1 | LDMAXBAR0_CH_REQSEL_SOURCESEL_TIMER0,                  ///< Trigger on TIMER0_CC1.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER0CC2)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER0_CC2 = LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER0CC2 | LDMAXBAR0_CH_REQSEL_SOURCESEL_TIMER0,                  ///< Trigger on TIMER0_CC2.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER0UFOF)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER0_UFOF = LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER0UFOF | LDMAXBAR0_CH_REQSEL_SOURCESEL_TIMER0,                ///< Trigger on TIMER0_UFOF.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER1CC0)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER1_CC0 = LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER1CC0 | LDMAXBAR0_CH_REQSEL_SOURCESEL_TIMER1,                  ///< Trigger on TIMER1_CC0.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER1CC1)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER1_CC1 = LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER1CC1 | LDMAXBAR0_CH_REQSEL_SOURCESEL_TIMER1,                  ///< Trigger on TIMER1_CC1.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER1CC2)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER1_CC2 = LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER1CC2 | LDMAXBAR0_CH_REQSEL_SOURCESEL_TIMER1,                  ///< Trigger on TIMER1_CC2.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER1UFOF)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER1_UFOF = LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER1UFOF | LDMAXBAR0_CH_REQSEL_SOURCESEL_TIMER1,                ///< Trigger on TIMER1_UFOF.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER2CC0)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER2_CC0 = LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER2CC0 | LDMAXBAR0_CH_REQSEL_SOURCESEL_TIMER2,                  ///< Trigger on TIMER2_CC0.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER2CC1)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER2_CC1 = LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER2CC1 | LDMAXBAR0_CH_REQSEL_SOURCESEL_TIMER2,                  ///< Trigger on TIMER2_CC1.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER2CC2)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER2_CC2 = LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER2CC2 | LDMAXBAR0_CH_REQSEL_SOURCESEL_TIMER2,                  ///< Trigger on TIMER2_CC2.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER2UFOF)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER2_UFOF = LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER2UFOF | LDMAXBAR0_CH_REQSEL_SOURCESEL_TIMER2,                ///< Trigger on TIMER2_UFOF.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER2CC3)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER2_CC3 = LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER2CC3 | LDMAXBAR0_CH_REQSEL_SOURCESEL_TIMER2,                  ///< Trigger on TIMER2_CC3.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER2CC4)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER2_CC4 = LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER2CC4 | LDMAXBAR0_CH_REQSEL_SOURCESEL_TIMER2,                  ///< Trigger on TIMER2_CC4.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER2CC5)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER2_CC5 = LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER2CC5 | LDMAXBAR0_CH_REQSEL_SOURCESEL_TIMER2,                  ///< Trigger on TIMER2_CC5.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER2CC6)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER2_CC6 = LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER2CC6 | LDMAXBAR0_CH_REQSEL_SOURCESEL_TIMER2,                  ///< Trigger on TIMER2_CC6.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER3CC0)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER3_CC0 = LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER3CC0 | LDMAXBAR0_CH_REQSEL_SOURCESEL_TIMER3,                  ///< Trigger on TIMER3_CC0.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER3CC1)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER3_CC1 = LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER3CC1 | LDMAXBAR0_CH_REQSEL_SOURCESEL_TIMER3,                  ///< Trigger on TIMER3_CC1.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER3CC2)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER3_CC2 = LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER3CC2 | LDMAXBAR0_CH_REQSEL_SOURCESEL_TIMER3,                  ///< Trigger on TIMER3_CC2.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER3UFOF)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER3_UFOF = LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER3UFOF | LDMAXBAR0_CH_REQSEL_SOURCESEL_TIMER3,                ///< Trigger on TIMER3_UFOF.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER3CC3)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER3_CC3 = LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER3CC3 | LDMAXBAR0_CH_REQSEL_SOURCESEL_TIMER3,                  ///< Trigger on TIMER3_CC3.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER3CC4)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER3_CC4 = LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER3CC4 | LDMAXBAR0_CH_REQSEL_SOURCESEL_TIMER3,                  ///< Trigger on TIMER3_CC4.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER3CC5)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER3_CC5 = LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER3CC5 | LDMAXBAR0_CH_REQSEL_SOURCESEL_TIMER3,                  ///< Trigger on TIMER3_CC5.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER3CC6)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER3_CC6 = LDMAXBAR0_CH_REQSEL_SIGSEL_TIMER3CC6 | LDMAXBAR0_CH_REQSEL_SOURCESEL_TIMER3,                  ///< Trigger on TIMER3_CC6.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_I2C0RXDATAV)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_I2C0_RXDATAV = LDMAXBAR0_CH_REQSEL_SIGSEL_I2C0RXDATAV | LDMAXBAR0_CH_REQSEL_SOURCESEL_I2C0,                ///< Trigger on I2C0_RXDATAV.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_I2C0TXBL)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_I2C0_TXBL = LDMAXBAR0_CH_REQSEL_SIGSEL_I2C0TXBL | LDMAXBAR0_CH_REQSEL_SOURCESEL_I2C0,                      ///< Trigger on I2C0_TXBL.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_I2C1RXDATAV)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_I2C1_RXDATAV = LDMAXBAR0_CH_REQSEL_SIGSEL_I2C1RXDATAV | LDMAXBAR0_CH_REQSEL_SOURCESEL_I2C1,                ///< Trigger on I2C1_RXDATAV.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_I2C1TXBL)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_I2C1_TXBL = LDMAXBAR0_CH_REQSEL_SIGSEL_I2C1TXBL | LDMAXBAR0_CH_REQSEL_SOURCESEL_I2C1,                      ///< Trigger on I2C1_TXBL.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_I2C2RXDATAV)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_I2C2_RXDATAV = LDMAXBAR0_CH_REQSEL_SIGSEL_I2C2RXDATAV | LDMAXBAR0_CH_REQSEL_SOURCESEL_I2C2,                ///< Trigger on I2C2_RXDATAV.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_I2C2TXBL)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_I2C2_TXBL = LDMAXBAR0_CH_REQSEL_SIGSEL_I2C2TXBL | LDMAXBAR0_CH_REQSEL_SOURCESEL_I2C2,                      ///< Trigger on I2C2_TXBL.
  #endif
  #if defined LDMAXBAR0_CH_REQSEL_SIGSEL_AGCRSSI
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_AGC_RSSI = LDMAXBAR0_CH_REQSEL_SIGSEL_AGCRSSI | LDMAXBAR0_CH_REQSEL_SOURCESEL_AGC,
  #endif
  #if defined LDMAXBAR0_CH_REQSEL_SIGSEL_PROTIMERBOF
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_PROTIMER_BOF = LDMAXBAR0_CH_REQSEL_SIGSEL_PROTIMERBOF | LDMAXBAR0_CH_REQSEL_SOURCESEL_PROTIMER,
  #endif
  #if defined LDMAXBAR0_CH_REQSEL_SIGSEL_PROTIMERCC0
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_PROTIMER_CC0 = LDMAXBAR0_CH_REQSEL_SIGSEL_PROTIMERCC0 | LDMAXBAR0_CH_REQSEL_SOURCESEL_PROTIMER,
  #endif
  #if defined LDMAXBAR0_CH_REQSEL_SIGSEL_PROTIMERCC1
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_PROTIMER_CC1 = LDMAXBAR0_CH_REQSEL_SIGSEL_PROTIMERCC1 | LDMAXBAR0_CH_REQSEL_SOURCESEL_PROTIMER,
  #endif
  #if defined LDMAXBAR0_CH_REQSEL_SIGSEL_PROTIMERCC2
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_PROTIMER_CC2 = LDMAXBAR0_CH_REQSEL_SIGSEL_PROTIMERCC2 | LDMAXBAR0_CH_REQSEL_SOURCESEL_PROTIMER,
  #endif
  #if defined LDMAXBAR0_CH_REQSEL_SIGSEL_PROTIMERCC3
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_PROTIMER_CC3 = LDMAXBAR0_CH_REQSEL_SIGSEL_PROTIMERCC3 | LDMAXBAR0_CH_REQSEL_SOURCESEL_PROTIMER,
  #endif
  #if defined LDMAXBAR0_CH_REQSEL_SIGSEL_PROTIMERCC4
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_PROTIMER_CC4 = LDMAXBAR0_CH_REQSEL_SIGSEL_PROTIMERCC4 | LDMAXBAR0_CH_REQSEL_SOURCESEL_PROTIMER,
  #endif
  #if defined LDMAXBAR0_CH_REQSEL_SIGSEL_PROTIMERPOF
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_PROTIMER_POF = LDMAXBAR0_CH_REQSEL_SIGSEL_PROTIMERPOF | LDMAXBAR0_CH_REQSEL_SOURCESEL_PROTIMER,
  #endif
  #if defined LDMAXBAR0_CH_REQSEL_SIGSEL_PROTIMERWOF
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_PROTIMER_WOF = LDMAXBAR0_CH_REQSEL_SIGSEL_PROTIMERWOF | LDMAXBAR0_CH_REQSEL_SOURCESEL_PROTIMER,
  #endif
  #if defined LDMAXBAR0_CH_REQSEL_SIGSEL_MODEMDEBUG
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_MODEM_DEBUG = LDMAXBAR0_CH_REQSEL_SIGSEL_MODEMDEBUG | LDMAXBAR0_CH_REQSEL_SOURCESEL_MODEM,
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_ADC0SCAN)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_ADC0_SCAN = LDMAXBAR0_CH_REQSEL_SIGSEL_ADC0SCAN | LDMAXBAR0_CH_REQSEL_SOURCESEL_ADC0,                      ///< Trigger on ADC0_SCAN.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_EUSART0RXFL)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_EUSART0_RXFL = LDMAXBAR0_CH_REQSEL_SIGSEL_EUSART0RXFL | LDMAXBAR0_CH_REQSEL_SOURCESEL_EUSART0,             ///< Trigger on EUSART0_RXFL.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_EUSART0TXFL)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_EUSART0_TXFL = LDMAXBAR0_CH_REQSEL_SIGSEL_EUSART0TXFL | LDMAXBAR0_CH_REQSEL_SOURCESEL_EUSART0,             ///< Trigger on EUSART0_TXFL.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_EUSART1RXFL)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_EUSART1_RXFL = LDMAXBAR0_CH_REQSEL_SIGSEL_EUSART1RXFL | LDMAXBAR0_CH_REQSEL_SOURCESEL_EUSART1,             ///< Trigger on EUSART1_RXFL.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_EUSART1TXFL)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_EUSART1_TXFL = LDMAXBAR0_CH_REQSEL_SIGSEL_EUSART1TXFL | LDMAXBAR0_CH_REQSEL_SOURCESEL_EUSART1,             ///< Trigger on EUSART1_TXFL.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_EUSART2RXFL)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_EUSART2_RXFL = LDMAXBAR0_CH_REQSEL_SIGSEL_EUSART2RXFL | LDMAXBAR0_CH_REQSEL_SOURCESEL_EUSART2,             ///< Trigger on EUSART2_RXFL.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_EUSART2TXFL)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_EUSART2_TXFL = LDMAXBAR0_CH_REQSEL_SIGSEL_EUSART2TXFL | LDMAXBAR0_CH_REQSEL_SOURCESEL_EUSART2,             ///< Trigger on EUSART2_TXFL.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_PIXELRZ0REQ_TXF)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_PIXELRZ0REQ_TXF = LDMAXBAR0_CH_REQSEL_SIGSEL_PIXELRZ0REQ_TXF | LDMAXBAR0_CH_REQSEL_SOURCESEL_PIXELRZ0,     ///< Trigger on PIXELRZ0REQ_TXF.
  #endif
  #if defined(LDMAXBAR0_CH_REQSEL_SIGSEL_PIXELRZ1REQ_TXF)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_PIXELRZ1REQ_TXF = LDMAXBAR0_CH_REQSEL_SIGSEL_PIXELRZ1REQ_TXF | LDMAXBAR0_CH_REQSEL_SOURCESEL_PIXELRZ1,     ///< Trigger on PIXELRZ1REQ_TXF.
  #endif
#else
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_NONE = LDMAXBAR_CH_REQSEL_SOURCESEL_NONE,                                                                ///< No peripheral selected for LDMA triggering.
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_LDMAXBARPRSREQ0)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_LDMAXBAR_PRSREQ0 = LDMAXBAR_CH_REQSEL_SIGSEL_LDMAXBARPRSREQ0 | LDMAXBAR_CH_REQSEL_SOURCESEL_LDMAXBAR,    ///< Trigger on PRS REQ0.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_LDMAXBARPRSREQ1)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_LDMAXBAR_PRSREQ1 = LDMAXBAR_CH_REQSEL_SIGSEL_LDMAXBARPRSREQ1 | LDMAXBAR_CH_REQSEL_SOURCESEL_LDMAXBAR,    ///< Trigger on PRS REQ1.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_TIMER0CC0)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER0_CC0 = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER0CC0 | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER0,                  ///< Trigger on TIMER0_CC0.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_TIMER0CC1)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER0_CC1 = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER0CC1 | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER0,                  ///< Trigger on TIMER0_CC1.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_TIMER0CC2)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER0_CC2 = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER0CC2 | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER0,                  ///< Trigger on TIMER0_CC2.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_TIMER0UFOF)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER0_UFOF = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER0UFOF | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER0,                ///< Trigger on TIMER0_UFOF.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_TIMER1CC0)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER1_CC0 = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER1CC0 | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER1,                  ///< Trigger on TIMER1_CC0.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_TIMER1CC1)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER1_CC1 = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER1CC1 | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER1,                  ///< Trigger on TIMER1_CC1.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_TIMER1CC2)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER1_CC2 = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER1CC2 | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER1,                  ///< Trigger on TIMER1_CC2.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_TIMER1UFOF)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER1_UFOF = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER1UFOF | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER1,                ///< Trigger on TIMER1_UFOF.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_USART0RXDATAV
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_USART0_RXDATAV = LDMAXBAR_CH_REQSEL_SIGSEL_USART0RXDATAV | LDMAXBAR_CH_REQSEL_SOURCESEL_USART0,          ///< Trigger on USART0_RXDATAV.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_USART0RXDATAVRIGHT
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_USART0_RXDATAVRIGHT = LDMAXBAR_CH_REQSEL_SIGSEL_USART0RXDATAVRIGHT | LDMAXBAR_CH_REQSEL_SOURCESEL_USART0,///< Trigger on USART0_RXDATAVRIGHT.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_USART0TXBL
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_USART0_TXBL = LDMAXBAR_CH_REQSEL_SIGSEL_USART0TXBL | LDMAXBAR_CH_REQSEL_SOURCESEL_USART0,                ///< Trigger on USART0_TXBL.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_USART0TXBLRIGHT
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_USART0_TXBLRIGHT = LDMAXBAR_CH_REQSEL_SIGSEL_USART0TXBLRIGHT | LDMAXBAR_CH_REQSEL_SOURCESEL_USART0,      ///< Trigger on USART0_TXBLRIGHT.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_USART0TXEMPTY
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_USART0_TXEMPTY = LDMAXBAR_CH_REQSEL_SIGSEL_USART0TXEMPTY | LDMAXBAR_CH_REQSEL_SOURCESEL_USART0,          ///< Trigger on USART0_TXEMPTY.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_USART1RXDATAV
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_USART1_RXDATAV = LDMAXBAR_CH_REQSEL_SIGSEL_USART1RXDATAV | LDMAXBAR_CH_REQSEL_SOURCESEL_USART1,          ///< Trigger on USART1_RXDATAV.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_USART1RXDATAVRIGHT
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_USART1_RXDATAVRIGHT = LDMAXBAR_CH_REQSEL_SIGSEL_USART1RXDATAVRIGHT | LDMAXBAR_CH_REQSEL_SOURCESEL_USART1,///< Trigger on USART1_RXDATAVRIGHT.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_USART1TXBL
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_USART1_TXBL = LDMAXBAR_CH_REQSEL_SIGSEL_USART1TXBL | LDMAXBAR_CH_REQSEL_SOURCESEL_USART1,                ///< Trigger on USART1_TXBL.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_USART1TXBLRIGHT
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_USART1_TXBLRIGHT = LDMAXBAR_CH_REQSEL_SIGSEL_USART1TXBLRIGHT | LDMAXBAR_CH_REQSEL_SOURCESEL_USART1,      ///< Trigger on USART1_TXBLRIGHT.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_USART1TXEMPTY
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_USART1_TXEMPTY = LDMAXBAR_CH_REQSEL_SIGSEL_USART1TXEMPTY | LDMAXBAR_CH_REQSEL_SOURCESEL_USART1,          ///< Trigger on USART1_TXEMPTY.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_USART2RXDATAV
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_USART2_RXDATAV = LDMAXBAR_CH_REQSEL_SIGSEL_USART2RXDATAV | LDMAXBAR_CH_REQSEL_SOURCESEL_USART2,          ///< Trigger on USART2_RXDATAV.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_USART2RXDATAVRIGHT
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_USART2_RXDATAVRIGHT = LDMAXBAR_CH_REQSEL_SIGSEL_USART2RXDATAVRIGHT | LDMAXBAR_CH_REQSEL_SOURCESEL_USART2,///< Trigger on USART2_RXDATAVRIGHT.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_USART2TXBL
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_USART2_TXBL = LDMAXBAR_CH_REQSEL_SIGSEL_USART2TXBL | LDMAXBAR_CH_REQSEL_SOURCESEL_USART2,                ///< Trigger on USART2_TXBL.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_USART2TXBLRIGHT
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_USART2_TXBLRIGHT = LDMAXBAR_CH_REQSEL_SIGSEL_USART2TXBLRIGHT | LDMAXBAR_CH_REQSEL_SOURCESEL_USART2,      ///< Trigger on USART2_TXBLRIGHT.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_USART2TXEMPTY
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_USART2_TXEMPTY = LDMAXBAR_CH_REQSEL_SIGSEL_USART2TXEMPTY | LDMAXBAR_CH_REQSEL_SOURCESEL_USART2,          ///< Trigger on USART2_TXEMPTY.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_TIMER2CC0)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER2_CC0 = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER2CC0 | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER2,                  ///< Trigger on TIMER2_CC0.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_TIMER2CC1)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER2_CC1 = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER2CC1 | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER2,                  ///< Trigger on TIMER2_CC1.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_TIMER2CC2)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER2_CC2 = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER2CC2 | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER2,                  ///< Trigger on TIMER2_CC2.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_TIMER2UFOF)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER2_UFOF = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER2UFOF | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER2,                ///< Trigger on TIMER2_UFOF.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_TIMER3CC0)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER3_CC0 = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER3CC0 | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER3,                  ///< Trigger on TIMER3_CC0.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_TIMER3CC1)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER3_CC1 = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER3CC1 | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER3,                  ///< Trigger on TIMER3_CC1.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_TIMER3CC2)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER3_CC2 = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER3CC2 | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER3,                  ///< Trigger on TIMER3_CC2.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_TIMER3UFOF)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER3_UFOF = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER3UFOF | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER3,                ///< Trigger on TIMER3_UFOF.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_TIMER5CC0
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER5_CC0 = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER5CC0 | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER5,                  ///< Trigger on TIMER5_CC0.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_TIMER5CC1
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER5_CC1 = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER5CC1 | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER5,                  ///< Trigger on TIMER5_CC1.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_TIMER5CC2
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER5_CC2 = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER5CC2 | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER5,                  ///< Trigger on TIMER5_CC2.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_TIMER5UFOF
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER5_UFOF = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER5UFOF | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER5,                ///< Trigger on TIMER5_UFOF.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_TIMER6CC0
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER6_CC0 = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER6CC0 | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER6,                  ///< Trigger on TIMER6_CC0.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_TIMER6CC1
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER6_CC1 = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER6CC1 | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER6,                  ///< Trigger on TIMER6_CC1.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_TIMER6CC2
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER6_CC2 = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER6CC2 | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER6,                  ///< Trigger on TIMER6_CC2.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_TIMER6UFOF
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER6_UFOF = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER6UFOF | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER6,                ///< Trigger on TIMER6_UFOF.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_TIMER7CC0
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER7_CC0 = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER7CC0 | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER7,                  ///< Trigger on TIMER7_CC0.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_TIMER7CC1
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER7_CC1 = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER7CC1 | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER7,                  ///< Trigger on TIMER7_CC1.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_TIMER7CC2
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER7_CC2 = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER7CC2 | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER7,                  ///< Trigger on TIMER7_CC2.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_TIMER7UFOF
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER7_UFOF = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER7UFOF | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER7,                ///< Trigger on TIMER7_UFOF.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_LCD
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_LCD = LDMAXBAR_CH_REQSEL_SIGSEL_LCD | LDMAXBAR_CH_REQSEL_SOURCESEL_LCD,
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_TIMER4CC0
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER4_CC0 = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER4CC0 | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER4,                  ///< Trigger on TIMER4_CC0.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_TIMER4CC1
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER4_CC1 = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER4CC1 | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER4,                  ///< Trigger on TIMER4_CC1.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_TIMER4CC2
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER4_CC2 = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER4CC2 | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER4,                  ///< Trigger on TIMER4_CC2.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_TIMER4UFOF
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_TIMER4_UFOF = LDMAXBAR_CH_REQSEL_SIGSEL_TIMER4UFOF | LDMAXBAR_CH_REQSEL_SOURCESEL_TIMER4,                ///< Trigger on TIMER4_UFOF.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_VDAC0CH0_REQ)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_VDAC0CH0REQ = LDMAXBAR_CH_REQSEL_SIGSEL_VDAC0CH0_REQ | LDMAXBAR_CH_REQSEL_SOURCESEL_VDAC0,               ///< Trigger on VDAC0_CH0REQ.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_VDAC0CH1_REQ)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_VDAC0CH1REQ = LDMAXBAR_CH_REQSEL_SIGSEL_VDAC0CH1_REQ | LDMAXBAR_CH_REQSEL_SOURCESEL_VDAC0,               ///< Trigger on VDAC0_CH1REQ.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_VDAC1CH0_REQ)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_VDAC1CH0REQ = LDMAXBAR_CH_REQSEL_SIGSEL_VDAC1CH0_REQ | LDMAXBAR_CH_REQSEL_SOURCESEL_VDAC1,               ///< Trigger on VDAC1_CH0REQ.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_VDAC1CH1_REQ)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_VDAC1CH1REQ = LDMAXBAR_CH_REQSEL_SIGSEL_VDAC1CH1_REQ | LDMAXBAR_CH_REQSEL_SOURCESEL_VDAC1,               ///< Trigger on VDAC1_CH1REQ.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_EUART0RXFL
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_EUART0_RXFL = LDMAXBAR_CH_REQSEL_SIGSEL_EUART0RXFL | LDMAXBAR_CH_REQSEL_SOURCESEL_EUART0,                ///< Trigger on EUART0_RXFL.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_EUART0TXFL
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_EUART0_TXFL = LDMAXBAR_CH_REQSEL_SIGSEL_EUART0TXFL | LDMAXBAR_CH_REQSEL_SOURCESEL_EUART0,                ///< Trigger on EUART0_TXFL.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_I2C0RXDATAV)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_I2C0_RXDATAV = LDMAXBAR_CH_REQSEL_SIGSEL_I2C0RXDATAV | LDMAXBAR_CH_REQSEL_SOURCESEL_I2C0,                ///< Trigger on I2C0_RXDATAV.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_I2C0TXBL)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_I2C0_TXBL = LDMAXBAR_CH_REQSEL_SIGSEL_I2C0TXBL | LDMAXBAR_CH_REQSEL_SOURCESEL_I2C0,                      ///< Trigger on I2C0_TXBL.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_I2C1RXDATAV)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_I2C1_RXDATAV = LDMAXBAR_CH_REQSEL_SIGSEL_I2C1RXDATAV | LDMAXBAR_CH_REQSEL_SOURCESEL_I2C1,                ///< Trigger on I2C1_RXDATAV.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_I2C1TXBL)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_I2C1_TXBL = LDMAXBAR_CH_REQSEL_SIGSEL_I2C1TXBL | LDMAXBAR_CH_REQSEL_SOURCESEL_I2C1,                      ///< Trigger on I2C1_TXBL.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_I2C2RXDATAV)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_I2C2_RXDATAV = LDMAXBAR_CH_REQSEL_SIGSEL_I2C2RXDATAV | LDMAXBAR_CH_REQSEL_SOURCESEL_I2C2,                ///< Trigger on I2C2_RXDATAV.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_I2C2TXBL)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_I2C2_TXBL = LDMAXBAR_CH_REQSEL_SIGSEL_I2C2TXBL | LDMAXBAR_CH_REQSEL_SOURCESEL_I2C2,                      ///< Trigger on I2C2_TXBL.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_AGCRSSI
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_AGC_RSSI = LDMAXBAR_CH_REQSEL_SIGSEL_AGCRSSI | LDMAXBAR_CH_REQSEL_SOURCESEL_AGC,
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_PDMRXDATAV)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_PDM_RXDATAV = LDMAXBAR_CH_REQSEL_SIGSEL_PDMRXDATAV | LDMAXBAR_CH_REQSEL_SOURCESEL_PDM,                   ///< Trigger on PDM_RXDATAV.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_PROTIMERBOF
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_PROTIMER_BOF = LDMAXBAR_CH_REQSEL_SIGSEL_PROTIMERBOF | LDMAXBAR_CH_REQSEL_SOURCESEL_PROTIMER,
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_PROTIMERCC0
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_PROTIMER_CC0 = LDMAXBAR_CH_REQSEL_SIGSEL_PROTIMERCC0 | LDMAXBAR_CH_REQSEL_SOURCESEL_PROTIMER,
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_PROTIMERCC1
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_PROTIMER_CC1 = LDMAXBAR_CH_REQSEL_SIGSEL_PROTIMERCC1 | LDMAXBAR_CH_REQSEL_SOURCESEL_PROTIMER,
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_PROTIMERCC2
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_PROTIMER_CC2 = LDMAXBAR_CH_REQSEL_SIGSEL_PROTIMERCC2 | LDMAXBAR_CH_REQSEL_SOURCESEL_PROTIMER,
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_PROTIMERCC3
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_PROTIMER_CC3 = LDMAXBAR_CH_REQSEL_SIGSEL_PROTIMERCC3 | LDMAXBAR_CH_REQSEL_SOURCESEL_PROTIMER,
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_PROTIMERCC4
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_PROTIMER_CC4 = LDMAXBAR_CH_REQSEL_SIGSEL_PROTIMERCC4 | LDMAXBAR_CH_REQSEL_SOURCESEL_PROTIMER,
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_PROTIMERPOF
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_PROTIMER_POF = LDMAXBAR_CH_REQSEL_SIGSEL_PROTIMERPOF | LDMAXBAR_CH_REQSEL_SOURCESEL_PROTIMER,
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_PROTIMERWOF
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_PROTIMER_WOF = LDMAXBAR_CH_REQSEL_SIGSEL_PROTIMERWOF | LDMAXBAR_CH_REQSEL_SOURCESEL_PROTIMER,
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_MODEMDEBUG
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_MODEM_DEBUG = LDMAXBAR_CH_REQSEL_SIGSEL_MODEMDEBUG | LDMAXBAR_CH_REQSEL_SOURCESEL_MODEM,
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_IADC0IADC_SCAN
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_IADC0_IADC_SCAN = LDMAXBAR_CH_REQSEL_SIGSEL_IADC0IADC_SCAN | LDMAXBAR_CH_REQSEL_SOURCESEL_IADC0,         ///< Trigger on IADC0_IADC_SCAN.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_IADC0IADC_SINGLE
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_IADC0_IADC_SINGLE = LDMAXBAR_CH_REQSEL_SIGSEL_IADC0IADC_SINGLE | LDMAXBAR_CH_REQSEL_SOURCESEL_IADC0,     ///< Trigger on IADC0_IADC_SINGLE.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_MSCWDATA
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_MSC_WDATA = LDMAXBAR_CH_REQSEL_SIGSEL_MSCWDATA | LDMAXBAR_CH_REQSEL_SOURCESEL_MSC,                       ///< Trigger on MSC_WDATA.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_EUSART0RXFL)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_EUSART0_RXFL = LDMAXBAR_CH_REQSEL_SIGSEL_EUSART0RXFL | LDMAXBAR_CH_REQSEL_SOURCESEL_EUSART0,             ///< Trigger on EUSART0_RXFL.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_EUSART0TXFL)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_EUSART0_TXFL = LDMAXBAR_CH_REQSEL_SIGSEL_EUSART0TXFL | LDMAXBAR_CH_REQSEL_SOURCESEL_EUSART0,             ///< Trigger on EUSART0_TXFL.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_EUSART1RXFL)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_EUSART1_RXFL = LDMAXBAR_CH_REQSEL_SIGSEL_EUSART1RXFL | LDMAXBAR_CH_REQSEL_SOURCESEL_EUSART1,             ///< Trigger on EUSART1_RXFL.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_EUSART1TXFL)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_EUSART1_TXFL = LDMAXBAR_CH_REQSEL_SIGSEL_EUSART1TXFL | LDMAXBAR_CH_REQSEL_SOURCESEL_EUSART1,             ///< Trigger on EUSART1_TXFL.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_EUSART2RXFL
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_EUSART2_RXFL = LDMAXBAR_CH_REQSEL_SIGSEL_EUSART2RXFL | LDMAXBAR_CH_REQSEL_SOURCESEL_EUSART2,             ///< Trigger on EUSART2_RXFL.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_EUSART2TXFL
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_EUSART2_TXFL = LDMAXBAR_CH_REQSEL_SIGSEL_EUSART2TXFL | LDMAXBAR_CH_REQSEL_SOURCESEL_EUSART2,             ///< Trigger on EUSART2_TXFL.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_EUSART3RXFL
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_EUSART3_RXFL = LDMAXBAR_CH_REQSEL_SIGSEL_EUSART3RXFL | LDMAXBAR_CH_REQSEL_SOURCESEL_EUSART3,             ///< Trigger on EUSART3_RXFL.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_EUSART3TXFL
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_EUSART3_TXFL = LDMAXBAR_CH_REQSEL_SIGSEL_EUSART3TXFL | LDMAXBAR_CH_REQSEL_SOURCESEL_EUSART3,             ///< Trigger on EUSART3_TXFL.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_EUSART4RXFL
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_EUSART4_RXFL = LDMAXBAR_CH_REQSEL_SIGSEL_EUSART4RXFL | LDMAXBAR_CH_REQSEL_SOURCESEL_EUSART4,             ///< Trigger on EUSART4_RXFL.
  #endif
  #if defined LDMAXBAR_CH_REQSEL_SIGSEL_EUSART4TXFL
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_EUSART4_TXFL = LDMAXBAR_CH_REQSEL_SIGSEL_EUSART4TXFL | LDMAXBAR_CH_REQSEL_SOURCESEL_EUSART4,             ///< Trigger on EUSART4_TXFL.
  #endif
  #if defined(LDMAXBAR_CH_REQSEL_SIGSEL_LESENSEFIFO)
  SL_HAL_LDMA_PERIPHERAL_SIGNAL_LESENSE_BUFDATAV = LDMAXBAR_CH_REQSEL_SIGSEL_LESENSEFIFO | LDMAXBAR_CH_REQSEL_SOURCESEL_LESENSE,        ///< Trigger on LESENSEFIFO.
  #endif
#endif
};

/*******************************************************************************
 *******************************   STRUCTS   ***********************************
 ******************************************************************************/

/***************************************************************************//**
 * @brief
 *   LDMA descriptor.
 *
 * @details
 *   The LDMA LDMA controller supports three different LDMA descriptors. Each
 *   consists of four WORDs which map directly onto HW control registers for a
 *   given LDMA channel. The three descriptor types are XFER, SYNC and WRI.
 *   Refer to the reference manual for further information.
 ******************************************************************************/
typedef union {
  /// TRANSFER LDMA descriptor, this is the only descriptor type which can be
  /// used to start a LDMA transfer.
  struct {
    uint32_t  struct_type       : 2;   ///< Set to 0 to select XFER descriptor type.
    uint32_t  reserved_0        : 1;   ///< Reserved.
    uint32_t  struct_req        : 1;   ///< LDMA transfer trigger during LINKLOAD.
    uint32_t  xfer_count        : 11;  ///< Transfer count minus one.
    uint32_t  byte_swap         : 1;   ///< Enable byte swapping transfers.
    uint32_t  block_size        : 4;   ///< Number of unit transfers per arbitration cycle.
    uint32_t  done_ifs          : 1;   ///< Generate interrupt when done.
    uint32_t  req_mode          : 1;   ///< Block or cycle transfer selector.
    uint32_t  dec_loop_count    : 1;   ///< Enable looped transfers.
    uint32_t  ignore_single_req : 1;   ///< Ignore single requests.
    uint32_t  src_inc           : 2;   ///< Source address increment unit size.
    uint32_t  size              : 2;   ///< LDMA transfer unit size.
    uint32_t  dst_inc           : 2;   ///< Destination address increment unit size.
    uint32_t  src_addr_mode     : 1;   ///< Source addressing mode.
    uint32_t  dst_addr_mode     : 1;   ///< Destination addressing mode.

    uint32_t  src_addr;                ///< LDMA source address.
    uint32_t  dst_addr;                ///< LDMA destination address.

    uint32_t  link_mode         : 1;   ///< Select absolute or relative link address.
    uint32_t  link              : 1;   ///< Enable LINKLOAD when transfer is done.
    int32_t   link_addr         : 30;  ///< Address of next (linked) descriptor.
  } xfer;

  /// SYNCHRONIZE LDMA descriptor, used for intra channel transfer synchronization.
  struct {
    uint32_t  struct_type       : 2;   ///< Set to 1 to select SYNC descriptor type.
    uint32_t  reserved_0        : 1;   ///< Reserved.
    uint32_t  struct_req        : 1;   ///< LDMA transfer trigger during LINKLOAD.
    uint32_t  xfer_count        : 11;  ///< Transfer count minus one.
    uint32_t  byte_swap         : 1;   ///< Enable byte swapping transfers.
    uint32_t  block_size        : 4;   ///< Number of unit transfers per arbitration cycle.
    uint32_t  done_ifs          : 1;   ///< Generate interrupt when done.
    uint32_t  req_mode          : 1;   ///< Block or cycle transfer selector.
    uint32_t  dec_loop_count    : 1;   ///< Enable looped transfers.
    uint32_t  ignore_single_req : 1;   ///< Ignore single requests.
    uint32_t  src_inc           : 2;   ///< Source address increment unit size.
    uint32_t  size              : 2;   ///< LDMA transfer unit size.
    uint32_t  dst_inc           : 2;   ///< Destination address increment unit size.
    uint32_t  src_addr_mode     : 1;   ///< Source addressing mode.
    uint32_t  dst_addr_mode     : 1;   ///< Destination addressing mode.

    uint32_t  sync_set          : 8;   ///< Set bits in LDMA_CTRL.SYNCTRIG register.
    uint32_t  sync_clr          : 8;   ///< Clear bits in LDMA_CTRL.SYNCTRIG register.
    uint32_t  reserved_1        : 16;  ///< Reserved.
#if defined(_LDMA_CH_CTRL_EXTEND_MASK)
    uint32_t  match_en          : 8;   ///< Sync trigger match enable.
    uint32_t  match_val         : 8;   ///< Sync trigger match value.
#else
    uint32_t  match_val         : 8;   ///< Sync trigger match value.
    uint32_t  match_en          : 8;   ///< Sync trigger match enable.
#endif
    uint32_t  reserved_2        : 16;  ///< Reserved.

    uint32_t  link_mode         : 1;   ///< Select absolute or relative link address.
    uint32_t  link              : 1;   ///< Enable LINKLOAD when transfer is done.
    int32_t   link_addr         : 30;  ///< Address of next (linked) descriptor.
  } sync;

  /// WRITE LDMA descriptor, used for write immediate operations.
  struct {
    uint32_t  struct_type       : 2;   ///< Set to 2 to select WRITE descriptor type.
    uint32_t  reserved_0        : 1;   ///< Reserved.
    uint32_t  struct_req        : 1;   ///< LDMA transfer trigger during LINKLOAD.
    uint32_t  xfer_count        : 11;  ///< Transfer count minus one.
    uint32_t  byte_swap         : 1;   ///< Enable byte swapping transfers.
    uint32_t  block_size        : 4;   ///< Number of unit transfers per arbitration cycle.
    uint32_t  done_ifs          : 1;   ///< Generate interrupt when done.
    uint32_t  req_mode          : 1;   ///< Block or cycle transfer selector.
    uint32_t  dec_loop_count    : 1;   ///< Enable looped transfers.
    uint32_t  ignore_single_req : 1;   ///< Ignore single requests.
    uint32_t  src_inc           : 2;   ///< Source address increment unit size.
    uint32_t  size              : 2;   ///< LDMA transfer unit size.
    uint32_t  dst_inc           : 2;   ///< Destination address increment unit size.
    uint32_t  src_addr_mode     : 1;   ///< Source addressing mode.
    uint32_t  dst_addr_mode     : 1;   ///< Destination addressing mode.

    uint32_t  imm_val;                 ///< Data to be written at dstAddr.
    uint32_t  dst_addr;                ///< LDMA write destination address.

    uint32_t  link_mode         : 1;   ///< Select absolute or relative link address.
    uint32_t  link              : 1;   ///< Enable LINKLOAD when transfer is done.
    int32_t   link_addr         : 30;  ///< Address of next (linked) descriptor.
  } wri;
} sl_hal_ldma_descriptor_t;

#if defined(_LDMA_CH_CTRL_EXTEND_MASK)
/***************************************************************************//**
 * @brief
 *   LDMA extended descriptor.
 *
 * @details
 *   The extended descriptor adds additional fields for the extended features
 *   available on the MMLDMA peripheral: destination interleaving and bufferable.
 *   The same three different LDMA descriptors supported by the LDMA LDMA controller
 *   are available. Each consists of seven WORDs (instead of four in non-extended
 *   descriptors) which map directly onto HW control registers for a given DMA
 *   channel. The three descriptor types are XFER, SYNC and WRI. But the
 *   extended fields are true only for XFER. The extended fields are the following:
 *
 *             +-          +- CTRL
 *             | Original  |  SRC
 *             | Structure |  DST
 *   Extended  |           +- LINK
 *   Structure |              XCTRL
 *             |              DUALDST
 *             +-             ILSRC
 *
 *   Refer to the reference manual for further information.
 ******************************************************************************/
typedef struct {
  uint32_t  struct_type       : 2;  ///< Set to 0 to select XFER descriptor type.
  uint32_t  extend            : 1;  ///< Extend data structure.
  uint32_t  struct_req        : 1;  ///< LDMA transfer trigger during LINKLOAD.
  uint32_t  xfer_count        : 11; ///< Transfer count minus one.
  uint32_t  byte_swap         : 1;  ///< Enable byte swapping transfers.
  uint32_t  block_size        : 4;  ///< Number of unit transfers per arbitration cycle.
  uint32_t  done_ifs          : 1;  ///< Generate interrupt when done.
  uint32_t  req_mode          : 1;  ///< Block or cycle transfer selector.
  uint32_t  dec_loop_count    : 1;  ///< Enable looped transfers.
  uint32_t  ignore_single_req : 1;  ///< Ignore single requests.
  uint32_t  src_inc           : 2;  ///< Source address increment unit size.
  uint32_t  size              : 2;  ///< LDMA transfer unit size.
  uint32_t  dst_inc           : 2;  ///< Destination address increment unit size.
  uint32_t  src_addr_mode     : 1;  ///< Source addressing mode.
  uint32_t  dst_addr_mode     : 1;  ///< Destination addressing mode.

  uint32_t  src_addr;               ///< LDMA source address.
  uint32_t  dst_addr;               ///< LDMA destination address.

  uint32_t  link_mode         : 1;  ///< Select absolute or relative link address.
  uint32_t  link              : 1;  ///< Enable LINKLOAD when transfer is done.
  int32_t   link_addr         : 30; ///< Address of next (linked) descriptor.

#if defined(_LDMA_CH_XCTRL_BUFFERABLE_MASK)
  uint32_t  reserved1         : 4;  ///< Reserved
  uint32_t  dst_il_en         : 1;  ///< Destination interleave.
  uint32_t  il_mode           : 2;  ///< Interleave mode.
  uint32_t  bufferable        : 1;  ///< Allow AHB buffering.
  uint32_t  reserved2         : 24; ///< Reserved
  uint32_t  reserved3;              ///< Reserved
#else
  uint32_t  dual_dst_en       : 1;  ///< Dual destination.
  uint32_t  dual_dst_inc      : 2;  ///< Dual destination address increment unit size.
  uint32_t  reserved1         : 1;  ///< Reserved
  uint32_t  dst_il_en         : 1;  ///< Destination interleave.
  uint32_t  il_mode           : 2;  ///< Interleave mode.
  uint32_t  reserved2         : 4;  ///< Reserved
  uint32_t  xfer_count_high   : 5;  ///< High bits of the transfer count minus one.
  uint32_t  src_mem_type      : 4;  ///< Type of memory access for the source.
  uint32_t  dst_mem_type      : 4;  ///< Type of memory access for the destination.
  uint32_t  reserved3         : 8;  ///< Reserved
  uint32_t  dual_dst_addr;          ///< LDMA dual destination address.
#endif

  uint32_t  il_src_addr;            ///< LDMA rules table base address in memory.
} sl_hal_ldma_descriptor_extend_t;
#endif

/// LDMA initialization configuration structure.
typedef struct {
  uint8_t               num_fixed_priority; ///< Number of fixed priority channels.
  uint8_t               sync_prs_clr_en;    ///< PRS Synctrig clear enable.
  uint8_t               sync_prs_set_en;    ///< PRS Synctrig set enable.
} sl_hal_ldma_init_t;

/** @cond DO_NOT_INCLUDE_WITH_DOXYGEN */
// Typedef for configuration structure used for backward compatibility purposes.
typedef sl_hal_ldma_init_t sl_hal_ldma_config_t;
/** @endcond */

/***************************************************************************//**
 * @brief
 *   LDMA transfer configuration structure.
 * @details
 *   This structure configures all aspects of a LDMA transfer.
 ******************************************************************************/
typedef struct {
  uint32_t                          request_sel;                     ///< Selects LDMA trigger source.
  uint8_t                           ldma_control_sync_prs_clear_off; ///< PRS Synctrig clear enables to clear.
  uint8_t                           ldma_control_sync_prs_clear_on;  ///< PRS Synctrig clear enables to set.
  uint8_t                           ldma_control_sync_prs_set_off;   ///< PRS Synctrig set enables to clear.
  uint8_t                           ldma_control_sync_prs_set_on;    ///< PRS Synctrig set enables to set.
  bool                              request_dis;                     ///< Mask the PRS trigger input.
  bool                              debug_halt_en;                   ///< Dis. LDMA trig when CPU is halted.
  sl_hal_ldma_cfg_arb_slots_t       arb_slots;                       ///< Arbitration slot number.
  sl_hal_ldma_cfg_src_inc_sign_t    src_inc_sign;                    ///< Source address increment sign.
  sl_hal_ldma_cfg_dst_inc_sign_t    dst_inc_sign;                    ///< Destination address increment sign.
  uint8_t                           loop_count;                      ///< Counter for looped transfers.
#if defined(_LDMA_CH_CFG_STRUCTBUSPORT_MASK)
  sl_hal_ldma_cfg_struct_bus_port_t struct_bus_port;                 ///< Structure fetch operation bus port.
  sl_hal_ldma_cfg_src_bus_port_t    src_bus_port;                    ///< Source operation bus port.
  sl_hal_ldma_cfg_dst_bus_port_t    dst_bus_port;                    ///< Destination operation bus port.
#endif
} sl_hal_ldma_transfer_init_t;

/** @cond DO_NOT_INCLUDE_WITH_DOXYGEN */
// Typedef for configuration structure used for backward compatibility purposes.
typedef sl_hal_ldma_transfer_init_t sl_hal_ldma_transfer_config_t;
/** @endcond */

/*******************************************************************************
 ********************************   DEFINES   **********************************
 ******************************************************************************/

/** @cond DO_NOT_INCLUDE_WITH_DOXYGEN */
// This define aligns series 2 LDMA IP name with series 3.
#ifndef LDMA0
#define LDMA0 LDMA
#endif
/** @endcond */

/// Size in words of a non-extended LDMA descriptor.
#define SL_HAL_LDMA_DESCRIPTOR_NON_EXTEND_SIZE_WORD  4

/// Size in words of an extended LDMA descriptor.
#define SL_HAL_LDMA_DESCRIPTOR_EXTEND_SIZE_WORD  7

/// Maximum transfer size possible per descriptor.
#define SL_HAL_LDMA_DESCRIPTOR_MAX_XFER_SIZE ((_LDMA_CH_CTRL_XFERCNT_MASK >> _LDMA_CH_CTRL_XFERCNT_SHIFT) + 1)

#if defined(_LDMA_CH_CTRL_EXTEND_MASK)
/// Maximum transfer size possible per extended descriptor.
#if defined(_LDMA_CH_XCTRL_XFERCNT_MASK)
#define SL_HAL_LDMA_DESCRIPTOR_EXTEND_MAX_XFER_SIZE   (SL_HAL_LDMA_DESCRIPTOR_MAX_XFER_SIZE + _LDMA_CH_XCTRL_XFERCNT_MASK)
#else
#define SL_HAL_LDMA_DESCRIPTOR_EXTEND_MAX_XFER_SIZE   SL_HAL_LDMA_DESCRIPTOR_MAX_XFER_SIZE
#endif
#endif

/// Converts a sl_hal_ldma_descriptor_t pointer to the value suitable
/// to write to the link_addr field of a sl_hal_ldma_descriptor_t.
#define SL_HAL_LDMA_DESCRIPTOR_LINKABS_ADDR_TO_LINKADDR(addr) (int32_t) ((((uintptr_t)addr) & _LDMA_CH_LINK_LINKADDR_MASK) >> _LDMA_CH_LINK_LINKADDR_SHIFT)

/// Converts a sl_hal_ldma_descriptor_t link_addr field value back
/// to a sl_hal_ldma_descriptor_t pointer.
#define SL_HAL_LDMA_DESCRIPTOR_LINKABS_LINKADDR_TO_ADDR(link_addr) (sl_hal_ldma_descriptor_t *) (link_addr << _LDMA_CH_LINK_LINKADDR_SHIFT)

/// Default value for BUS PORT for sl_hal_ldma_transfer_init_t struct initializer.
#if defined(_LDMA_CH_CFG_STRUCTBUSPORT_MASK)
#define SL_HAL_LDMA_CFG_BUS_PORT_DEFAULT \
  SL_HAL_LDMA_CFG_STRUCT_BUS_PORT_0,     \
  SL_HAL_LDMA_CFG_SRC_BUS_PORT_0,        \
  SL_HAL_LDMA_CFG_DST_BUS_PORT_0
#else
#define SL_HAL_LDMA_CFG_BUS_PORT_DEFAULT
#endif

#if defined(_LDMA_CH_CTRL_EXTEND_MASK)
/// XFER_COUNT suitable for sl_hal_ldma_descriptor_extend_t.
#if defined(_LDMA_CH_XCTRL_XFERCNT_MASK)
#define SL_HAL_LDMA_EXTEND_XFER_COUNT(count)      (((count) - 1) & (_LDMA_CH_CTRL_XFERCNT_MASK >> _LDMA_CH_CTRL_XFERCNT_SHIFT))
#define SL_HAL_LDMA_EXTEND_XFER_COUNT_HIGH(count) (((count) - 1) >> _LDMA_CH_XCTRL_XFERCNT_SHIFT)
#else
#define SL_HAL_LDMA_EXTEND_XFER_COUNT(count)      ((count) - 1)
#endif

/// Default value for extended descriptor for sl_hal_ldma_descriptor_extend_t struct initializer.
#if defined(_LDMA_CH_XCTRL_BUFFERABLE_MASK)
#define SL_HAL_LDMA_EXTEND_DEFAULT(count) \
  .dst_il_en   = 0,                       \
  .il_mode     = 0,                       \
  .bufferable  = 0,                       \
  .il_src_addr = 0
#else
#define SL_HAL_LDMA_EXTEND_DEFAULT(count)                        \
  .dual_dst_en      = 0,                                         \
  .dual_dst_inc     = SL_HAL_LDMA_CTRL_DST_INC_NONE,             \
  .dst_il_en        = 0,                                         \
  .il_mode          = 0,                                         \
  .xfer_count_high  = SL_HAL_LDMA_EXTEND_XFER_COUNT_HIGH(count), \
  .src_mem_type     = SL_HAL_LDMA_MEM_TYPE_CACHEABLE,            \
  .dst_mem_type     = SL_HAL_LDMA_MEM_TYPE_CACHEABLE,            \
  .dual_dst_addr    = 0,                                         \
  .il_src_addr      = 0
#endif
#endif

/*******************************************************************************
 **************************   STRUCT INITIALIZERS   ****************************
 ******************************************************************************/

/// Default LDMA initialization structure.
#define SL_HAL_LDMA_INIT_DEFAULT                                                            \
  {                                                                                         \
    .num_fixed_priority     = _LDMA_CTRL_NUMFIXED_DEFAULT,/* Fixed priority arbitration.*/  \
    .sync_prs_clr_en        = 0,                          /* No PRS Synctrig clear enable*/ \
    .sync_prs_set_en        = 0,                          /* No PRS Synctrig set enable. */ \
  }

/// Generic LDMA transfer configuration for memory to memory transfers.
#define SL_HAL_LDMA_TRANSFER_CFG_MEMORY() \
  {                                       \
    0, 0, 0, 0, 0,                        \
    false,                                \
    false,                                \
    SL_HAL_LDMA_CFG_ARBSLOTS_ONE,         \
    SL_HAL_LDMA_CFG_SRC_INC_SIGN_POS,     \
    SL_HAL_LDMA_CFG_DST_INC_SIGN_POS,     \
    0,                                    \
    SL_HAL_LDMA_CFG_BUS_PORT_DEFAULT      \
  }

/// Generic LDMA transfer configuration for looped memory to memory transfers.
#define SL_HAL_LDMA_TRANSFER_CFG_MEMORY_LOOP(loop_count) \
  {                                                      \
    0, 0, 0, 0, 0,                                       \
    false,                                               \
    false,                                               \
    SL_HAL_LDMA_CFG_ARBSLOTS_ONE,                        \
    SL_HAL_LDMA_CFG_SRC_INC_SIGN_POS,                    \
    SL_HAL_LDMA_CFG_DST_INC_SIGN_POS,                    \
    loop_count,                                          \
    SL_HAL_LDMA_CFG_BUS_PORT_DEFAULT                     \
  }

/// Generic LDMA transfer configuration for memory to/from peripheral transfers.
#define SL_HAL_LDMA_TRANSFER_CFG_PERIPHERAL(signal) \
  {                                                 \
    signal, 0, 0, 0, 0,                             \
    false,                                          \
    false,                                          \
    SL_HAL_LDMA_CFG_ARBSLOTS_ONE,                   \
    SL_HAL_LDMA_CFG_SRC_INC_SIGN_POS,               \
    SL_HAL_LDMA_CFG_DST_INC_SIGN_POS,               \
    0,                                              \
    SL_HAL_LDMA_CFG_BUS_PORT_DEFAULT                \
  }

/// Generic LDMA transfer configuration for looped memory to/from peripheral transfers.
#define SL_HAL_LDMA_TRANSFER_CFG_PERIPHERAL_LOOP(signal, loop_count) \
  {                                                                  \
    signal, 0, 0, 0, 0,                                              \
    false,                                                           \
    false,                                                           \
    SL_HAL_LDMA_CFG_ARBSLOTS_ONE,                                    \
    SL_HAL_LDMA_CFG_SRC_INC_SIGN_POS,                                \
    SL_HAL_LDMA_CFG_DST_INC_SIGN_POS,                                \
    loop_count,                                                      \
    SL_HAL_LDMA_CFG_BUS_PORT_DEFAULT                                 \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for single memory to memory transfer.
 *
 * @param[in] transfer_size
 *   Transfer size
 *       SL_HAL_LDMA_CTRL_SIZE_WORD,
 *       SL_HAL_LDMA_CTRL_SIZE_HALF,
 *       SL_HAL_LDMA_CTRL_SIZE_BYTE.
 *
 * @param[in] src
 *   Source data address.
 *
 * @param[in] dest
 *   Destination data address.
 *
 * @param[in] count
 *   Number of words to transfer.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_SINGLE_M2M(transfer_size, src, dest, count) \
  {                                                                        \
    .xfer =                                                                \
    {                                                                      \
      .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_XFER,              \
      .struct_req        = 1,                                              \
      .xfer_count        = (count) - 1,                                    \
      .byte_swap         = 0,                                              \
      .block_size        = SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_1,             \
      .done_ifs          = 1,                                              \
      .req_mode          = SL_HAL_LDMA_CTRL_REQ_MODE_ALL,                  \
      .dec_loop_count    = 0,                                              \
      .ignore_single_req = 0,                                              \
      .src_inc           = SL_HAL_LDMA_CTRL_SRC_INC_ONE,                   \
      .size              = (transfer_size),                                \
      .dst_inc           = SL_HAL_LDMA_CTRL_DST_INC_ONE,                   \
      .src_addr_mode     = SL_HAL_LDMA_CTRL_SRC_ADDR_MODE_ABS,             \
      .dst_addr_mode     = SL_HAL_LDMA_CTRL_DST_ADDR_MODE_ABS,             \
      .src_addr          = (uint32_t)(src),                                \
      .dst_addr          = (uint32_t)(dest),                               \
      .link_mode         = 0,                                              \
      .link              = 0,                                              \
      .link_addr         = 0                                               \
    }                                                                      \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for single peripheral to memory transfer.
 *
 * @param[in] transfer_size
 *   Transfer size
 *       SL_HAL_LDMA_CTRL_SIZE_WORD,
 *       SL_HAL_LDMA_CTRL_SIZE_HALF,
 *       SL_HAL_LDMA_CTRL_SIZE_BYTE.
 *
 * @param[in] src
 *   Peripheral data source register address.
 *
 * @param[in] dest
 *   Destination data address.
 *
 * @param[in] count
 *   Number of bytes to transfer.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_SINGLE_P2M(transfer_size, src, dest, count) \
  {                                                                        \
    .xfer =                                                                \
    {                                                                      \
      .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_XFER,              \
      .struct_req        = 0,                                              \
      .xfer_count        = (count) - 1,                                    \
      .byte_swap         = 0,                                              \
      .block_size        = SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_1,             \
      .done_ifs          = 1,                                              \
      .req_mode          = SL_HAL_LDMA_CTRL_REQ_MODE_BLOCK,                \
      .dec_loop_count    = 0,                                              \
      .ignore_single_req = 0,                                              \
      .src_inc           = SL_HAL_LDMA_CTRL_SRC_INC_NONE,                  \
      .size              = (transfer_size),                                \
      .dst_inc           = SL_HAL_LDMA_CTRL_DST_INC_ONE,                   \
      .src_addr_mode     = SL_HAL_LDMA_CTRL_SRC_ADDR_MODE_ABS,             \
      .dst_addr_mode     = SL_HAL_LDMA_CTRL_DST_ADDR_MODE_ABS,             \
      .src_addr          = (uint32_t)(src),                                \
      .dst_addr          = (uint32_t)(dest),                               \
      .link_mode         = 0,                                              \
      .link              = 0,                                              \
      .link_addr         = 0                                               \
    }                                                                      \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for single peripheral to peripheral transfer.
 *
 * @param[in] transfer_size
 *   Transfer size
 *       SL_HAL_LDMA_CTRL_SIZE_WORD,
 *       SL_HAL_LDMA_CTRL_SIZE_HALF,
 *       SL_HAL_LDMA_CTRL_SIZE_BYTE.
 *
 * @param[in] src
 *   Peripheral data source register address.
 *
 * @param[in] dest
 *   Peripheral data destination register address.
 *
 * @param[in] count
 *   Number of bytes to transfer.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_SINGLE_P2P(transfer_size, src, dest, count) \
  {                                                                        \
    .xfer =                                                                \
    {                                                                      \
      .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_XFER,              \
      .struct_req        = 0,                                              \
      .xfer_count        = (count) - 1,                                    \
      .byte_swap         = 0,                                              \
      .block_size        = SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_1,             \
      .done_ifs          = 1,                                              \
      .req_mode          = SL_HAL_LDMA_CTRL_REQ_MODE_BLOCK,                \
      .dec_loop_count    = 0,                                              \
      .ignore_single_req = 0,                                              \
      .src_inc           = SL_HAL_LDMA_CTRL_SRC_INC_NONE,                  \
      .size              = (transfer_size),                                \
      .dst_inc           = SL_HAL_LDMA_CTRL_DST_INC_NONE,                  \
      .src_addr_mode     = SL_HAL_LDMA_CTRL_SRC_ADDR_MODE_ABS,             \
      .dst_addr_mode     = SL_HAL_LDMA_CTRL_DST_ADDR_MODE_ABS,             \
      .src_addr          = (uint32_t)(src),                                \
      .dst_addr          = (uint32_t)(dest),                               \
      .link_mode         = 0,                                              \
      .link              = 0,                                              \
      .link_addr         = 0                                               \
    }                                                                      \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for single memory to peripheral transfer.
 *
 * @param[in] transfer_size
 *   Transfer size
 *                                     SL_HAL_LDMA_CTRL_SIZE_WORD,
 *                                     SL_HAL_LDMA_CTRL_SIZE_HALF,
 *                                     SL_HAL_LDMA_CTRL_SIZE_BYTE.
 *
 * @param[in] src
 *   Source data address.
 *
 * @param[in] dest
 *   Peripheral data register destination address.
 *
 * @param[in] count
 *   Number of bytes to transfer.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_SINGLE_M2P(transfer_size, src, dest, count) \
  {                                                                        \
    .xfer =                                                                \
    {                                                                      \
      .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_XFER,              \
      .struct_req        = 0,                                              \
      .xfer_count        = (count) - 1,                                    \
      .byte_swap         = 0,                                              \
      .block_size        = SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_1,             \
      .done_ifs          = 1,                                              \
      .req_mode          = SL_HAL_LDMA_CTRL_REQ_MODE_BLOCK,                \
      .dec_loop_count    = 0,                                              \
      .ignore_single_req = 0,                                              \
      .src_inc           = SL_HAL_LDMA_CTRL_SRC_INC_ONE,                   \
      .size              = (transfer_size),                                \
      .dst_inc           = SL_HAL_LDMA_CTRL_DST_INC_NONE,                  \
      .src_addr_mode     = SL_HAL_LDMA_CTRL_SRC_ADDR_MODE_ABS,             \
      .dst_addr_mode     = SL_HAL_LDMA_CTRL_DST_ADDR_MODE_ABS,             \
      .src_addr          = (uint32_t)(src),                                \
      .dst_addr          = (uint32_t)(dest),                               \
      .link_mode         = 0,                                              \
      .link              = 0,                                              \
      .link_addr         = 0                                               \
    }                                                                      \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for linked memory to memory transfer.
 *   Link address must be an absolute address.
 *
 * @note
 *   The link_addr member of the transfer descriptor is not initialized.
 *   link_addr must be initialized by using the proper bits right-shift
 *   to get the correct bits from the absolute address.
 *   _LDMA_CH_LINK_LINKADDR_SHIFT should be used for that operation:
 *
 * @code
 * desc.link_addr = ((int32_t)&next_desc) >> _LDMA_CH_LINK_LINKADDR_SHIFT;
 * @endcode
 * The opposite bit shift (left) must be done if link_addr is read.
 *
 * @param[in] transfer_size
 *   Transfer size
 *                                     SL_HAL_LDMA_CTRL_SIZE_WORD,
 *                                     SL_HAL_LDMA_CTRL_SIZE_HALF,
 *                                     SL_HAL_LDMA_CTRL_SIZE_BYTE.
 *
 * @param[in] src
 *   Source data address.
 *
 * @param[in] dest
 *   Destination data address.
 *
 * @param[in] count
 *   Number of words to transfer.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_LINKABS_M2M(transfer_size, src, dest, count) \
  {                                                                         \
    .xfer =                                                                 \
    {                                                                       \
      .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_XFER,               \
      .struct_req        = 1,                                               \
      .xfer_count        = (count) - 1,                                     \
      .byte_swap         = 0,                                               \
      .block_size        = SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_1,              \
      .done_ifs          = 0,                                               \
      .req_mode          = SL_HAL_LDMA_CTRL_REQ_MODE_ALL,                   \
      .dec_loop_count    = 0,                                               \
      .ignore_single_req = 0,                                               \
      .src_inc           = SL_HAL_LDMA_CTRL_SRC_INC_ONE,                    \
      .size              = (transfer_size),                                 \
      .dst_inc           = SL_HAL_LDMA_CTRL_DST_INC_ONE,                    \
      .src_addr_mode     = SL_HAL_LDMA_CTRL_SRC_ADDR_MODE_ABS,              \
      .dst_addr_mode     = SL_HAL_LDMA_CTRL_DST_ADDR_MODE_ABS,              \
      .src_addr          = (uint32_t)(src),                                 \
      .dst_addr          = (uint32_t)(dest),                                \
      .link_mode         = SL_HAL_LDMA_LINK_MODE_ABS,                       \
      .link              = 1,                                               \
      .link_addr         = 0 /* Must be set runtime ! */                    \
    }                                                                       \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for linked peripheral to memory transfer.
 *   Link address must be an absolute address.
 *
 * @note
 *   The link_addr member of the transfer descriptor is not initialized.
 *   link_addr must be initialized by using the proper bits right-shift
 *   to get the correct bits from the absolute address.
 *   _LDMA_CH_LINK_LINKADDR_SHIFT should be used for that operation:
 *
 * @code
 * desc.link_addr = ((int32_t)&next_desc) >> _LDMA_CH_LINK_LINKADDR_SHIFT;
 * @endcode
 * The opposite bit shift (left) must be done if link_addr is read.
 *
 * @param[in] transfer_size
 *   Transfer size
 *                                     SL_HAL_LDMA_CTRL_SIZE_WORD,
 *                                     SL_HAL_LDMA_CTRL_SIZE_HALF,
 *                                     SL_HAL_LDMA_CTRL_SIZE_BYTE.
 *
 * @param[in] src
 *   Source data address.
 *
 * @param[in] dest
 *   Destination data address.
 *
 * @param[in] count
 *   Number of words to transfer.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_LINKABS_P2M(transfer_size, src, dest, count) \
  {                                                                         \
    .xfer =                                                                 \
    {                                                                       \
      .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_XFER,               \
      .struct_req        = 0,                                               \
      .xfer_count        = (count) - 1,                                     \
      .byte_swap         = 0,                                               \
      .block_size        = SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_1,              \
      .done_ifs          = 0,                                               \
      .req_mode          = SL_HAL_LDMA_CTRL_REQ_MODE_BLOCK,                 \
      .dec_loop_count    = 0,                                               \
      .ignore_single_req = 0,                                               \
      .src_inc           = SL_HAL_LDMA_CTRL_SRC_INC_NONE,                   \
      .size              = (transfer_size),                                 \
      .dst_inc           = SL_HAL_LDMA_CTRL_DST_INC_ONE,                    \
      .src_addr_mode     = SL_HAL_LDMA_CTRL_SRC_ADDR_MODE_ABS,              \
      .dst_addr_mode     = SL_HAL_LDMA_CTRL_DST_ADDR_MODE_ABS,              \
      .src_addr          = (uint32_t)(src),                                 \
      .dst_addr          = (uint32_t)(dest),                                \
      .link_mode         = SL_HAL_LDMA_LINK_MODE_ABS,                       \
      .link              = 1,                                               \
      .link_addr         = 0 /* Must be set runtime ! */                    \
    }                                                                       \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for linked memory to peripheral word transfer.
 *   Link address must be an absolute address.
 *
 * @note
 *   The link_addr member of the transfer descriptor is not initialized.
 *   link_addr must be initialized by using the proper bits right-shift
 *   to get the correct bits from the absolute address.
 *   _LDMA_CH_LINK_LINKADDR_SHIFT should be used for that operation:
 *
 * @code
 * desc.link_addr = ((int32_t)&next_desc) >> _LDMA_CH_LINK_LINKADDR_SHIFT;
 * @endcode
 * The opposite bit shift (left) must be done if link_addr is read.
 *
 * @param[in] transfer_size
 *   Transfer size
 *       SL_HAL_LDMA_CTRL_SIZE_WORD,
 *       SL_HAL_LDMA_CTRL_SIZE_HALF,
 *       SL_HAL_LDMA_CTRL_SIZE_BYTE.
 *
 * @param[in] src
 *   Source data address.
 *
 * @param[in] dest
 *   Destination data address.
 *
 * @param[in] count
 *   Number of words to transfer.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_LINKABS_M2P(transfer_size, src, dest, count) \
  {                                                                         \
    .xfer =                                                                 \
    {                                                                       \
      .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_XFER,               \
      .struct_req        = 0,                                               \
      .xfer_count        = (count) - 1,                                     \
      .byte_swap         = 0,                                               \
      .block_size        = SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_1,              \
      .done_ifs          = 0,                                               \
      .req_mode          = SL_HAL_LDMA_CTRL_REQ_MODE_BLOCK,                 \
      .dec_loop_count    = 0,                                               \
      .ignore_single_req = 0,                                               \
      .src_inc           = SL_HAL_LDMA_CTRL_SRC_INC_ONE,                    \
      .size              = (transfer_size),                                 \
      .dst_inc           = SL_HAL_LDMA_CTRL_DST_INC_NONE,                   \
      .src_addr_mode     = SL_HAL_LDMA_CTRL_SRC_ADDR_MODE_ABS,              \
      .dst_addr_mode     = SL_HAL_LDMA_CTRL_DST_ADDR_MODE_ABS,              \
      .src_addr          = (uint32_t)(src),                                 \
      .dst_addr          = (uint32_t)(dest),                                \
      .link_mode         = SL_HAL_LDMA_LINK_MODE_ABS,                       \
      .link              = 1,                                               \
      .link_addr         = 0 /* Must be set runtime ! */                    \
    }                                                                       \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for linked peripheral to peripheral word transfer.
 *   Link address must be an absolute address.
 *
 * @note
 *   The link_addr member of the transfer descriptor is not initialized.
 *   link_addr must be initialized by using the proper bits right-shift
 *   to get the correct bits from the absolute address.
 *   _LDMA_CH_LINK_LINKADDR_SHIFT should be used for that operation:
 *
 * @code
 * desc.link_addr = ((int32_t)&next_desc) >> _LDMA_CH_LINK_LINKADDR_SHIFT;
 * @endcode
 * The opposite bit shift (left) must be done if link_addr is read.
 *
 * @param[in] transfer_size
 *   Transfer size
 *       SL_HAL_LDMA_CTRL_SIZE_WORD,
 *       SL_HAL_LDMA_CTRL_SIZE_HALF,
 *       SL_HAL_LDMA_CTRL_SIZE_BYTE.
 *
 * @param[in] src
 *   Source data address.
 *
 * @param[in] dest
 *   Destination data address.
 *
 * @param[in] count
 *   Number of words to transfer.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_LINKABS_P2P(transfer_size, src, dest, count) \
  {                                                                         \
    .xfer =                                                                 \
    {                                                                       \
      .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_XFER,               \
      .struct_req        = 0,                                               \
      .xfer_count        = (count) - 1,                                     \
      .byte_swap         = 0,                                               \
      .block_size        = SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_1,              \
      .done_ifs          = 0,                                               \
      .req_mode          = SL_HAL_LDMA_CTRL_REQ_MODE_BLOCK,                 \
      .dec_loop_count    = 0,                                               \
      .ignore_single_req = 0,                                               \
      .src_inc           = SL_HAL_LDMA_CTRL_SRC_INC_NONE,                   \
      .size              = (transfer_size),                                 \
      .dst_inc           = SL_HAL_LDMA_CTRL_DST_INC_NONE,                   \
      .src_addr_mode     = SL_HAL_LDMA_CTRL_SRC_ADDR_MODE_ABS,              \
      .dst_addr_mode     = SL_HAL_LDMA_CTRL_DST_ADDR_MODE_ABS,              \
      .src_addr          = (uint32_t)(src),                                 \
      .dst_addr          = (uint32_t)(dest),                                \
      .link_mode         = SL_HAL_LDMA_LINK_MODE_ABS,                       \
      .link              = 1,                                               \
      .link_addr         = 0 /* Must be set runtime ! */                    \
    }                                                                       \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for linked peripheral to peripheral word transfer.
 *   Link address must be an absolute address.
 *
 * @note
 *   The link_addr member of the transfer descriptor is initialized to 4
 *   (regular descriptor) or 7 (extended descriptor),  assuming that
 *   the next descriptor immediately follows this descriptor (in memory).
 *
 * @param[in] transfer_size
 *   Transfer size
 *       SL_HAL_LDMA_CTRL_SIZE_WORD,
 *       SL_HAL_LDMA_CTRL_SIZE_HALF,
 *       SL_HAL_LDMA_CTRL_SIZE_BYTE.
 * @param[in] src
 *   Source data address.
 *
 * @param[in] dest
 *   Destination data address.
 *
 * @param[in] count
 *   Number of words to transfer.
 *
 * @param[in] link_jump
 *   Address of descriptor to link to, expressed as a
 *       signed number of descriptors from "here".
 *       1=one descriptor forward in memory,
 *       0=this descriptor,
 *       -1=one descriptor back in memory.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_LINKREL_M2M(transfer_size, src, dest, count, link_jump) \
  {                                                                                    \
    .xfer =                                                                            \
    {                                                                                  \
      .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_XFER,                          \
      .struct_req        = 1,                                                          \
      .xfer_count        = (count) - 1,                                                \
      .byte_swap         = 0,                                                          \
      .block_size        = SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_1,                         \
      .done_ifs          = 0,                                                          \
      .req_mode          = SL_HAL_LDMA_CTRL_REQ_MODE_ALL,                              \
      .dec_loop_count    = 0,                                                          \
      .ignore_single_req = 0,                                                          \
      .src_inc           = SL_HAL_LDMA_CTRL_SRC_INC_ONE,                               \
      .size              = (transfer_size),                                            \
      .dst_inc           = SL_HAL_LDMA_CTRL_DST_INC_ONE,                               \
      .src_addr_mode     = SL_HAL_LDMA_CTRL_SRC_ADDR_MODE_ABS,                         \
      .dst_addr_mode     = SL_HAL_LDMA_CTRL_DST_ADDR_MODE_ABS,                         \
      .src_addr          = (uint32_t)(src),                                            \
      .dst_addr          = (uint32_t)(dest),                                           \
      .link_mode         = SL_HAL_LDMA_LINK_MODE_REL,                                  \
      .link              = 1,                                                          \
      .link_addr         = (link_jump) * SL_HAL_LDMA_DESCRIPTOR_NON_EXTEND_SIZE_WORD   \
    }                                                                                  \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for linked peripheral to memory transfer.
 *
 * @param[in] transfer_size
 *   Transfer size
 *       SL_HAL_LDMA_CTRL_SIZE_WORD,
 *       SL_HAL_LDMA_CTRL_SIZE_HALF,
 *       SL_HAL_LDMA_CTRL_SIZE_BYTE.
 *
 * @param[in] src
 *   Peripheral data source register address.
 *
 * @param[in] dest
 *   Destination data address.
 *
 * @param[in] count
 *   Number of bytes to transfer.
 *
 * @param[in] link_jump
 *   Address of descriptor to link to, expressed as a
 *       signed number of descriptors from "here".
 *       1=one descriptor forward in memory,
 *       0=this descriptor,
 *       -1=one descriptor back in memory.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_LINKREL_P2M(transfer_size, src, dest, count, link_jump) \
  {                                                                                    \
    .xfer =                                                                            \
    {                                                                                  \
      .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_XFER,                          \
      .struct_req        = 0,                                                          \
      .xfer_count        = (count) - 1,                                                \
      .byte_swap         = 0,                                                          \
      .block_size        = SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_1,                         \
      .done_ifs          = 0,                                                          \
      .req_mode          = SL_HAL_LDMA_CTRL_REQ_MODE_BLOCK,                            \
      .dec_loop_count    = 0,                                                          \
      .ignore_single_req = 0,                                                          \
      .src_inc           = SL_HAL_LDMA_CTRL_SRC_INC_NONE,                              \
      .size              = (transfer_size),                                            \
      .dst_inc           = SL_HAL_LDMA_CTRL_DST_INC_ONE,                               \
      .src_addr_mode     = SL_HAL_LDMA_CTRL_SRC_ADDR_MODE_ABS,                         \
      .dst_addr_mode     = SL_HAL_LDMA_CTRL_DST_ADDR_MODE_ABS,                         \
      .src_addr          = (uint32_t)(src),                                            \
      .dst_addr          = (uint32_t)(dest),                                           \
      .link_mode         = SL_HAL_LDMA_LINK_MODE_REL,                                  \
      .link              = 1,                                                          \
      .link_addr         = (link_jump) * SL_HAL_LDMA_DESCRIPTOR_NON_EXTEND_SIZE_WORD   \
    }                                                                                  \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for linked memory to peripheral transfer.
 *
 * @param[in] transfer_size
 *   Transfer size
 *       SL_HAL_LDMA_CTRL_SIZE_WORD,
 *       SL_HAL_LDMA_CTRL_SIZE_HALF,
 *       SL_HAL_LDMA_CTRL_SIZE_BYTE.
 *
 * @param[in] src
 *   Peripheral data source register address.
 *
 * @param[in] dest
 *   Destination data address.
 *
 * @param[in] count
 *   Number of bytes to transfer.
 *
 * @param[in] link_jump
 *   Address of descriptor to link to, expressed as a
 *       signed number of descriptors from "here".
 *       1=one descriptor forward in memory,
 *       0=this descriptor,
 *       -1=one descriptor back in memory.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_LINKREL_M2P(transfer_size, src, dest, count, link_jump) \
  {                                                                                    \
    .xfer =                                                                            \
    {                                                                                  \
      .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_XFER,                          \
      .struct_req        = 0,                                                          \
      .xfer_count        = (count) - 1,                                                \
      .byte_swap         = 0,                                                          \
      .block_size        = SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_1,                         \
      .done_ifs          = 0,                                                          \
      .req_mode          = SL_HAL_LDMA_CTRL_REQ_MODE_BLOCK,                            \
      .dec_loop_count    = 0,                                                          \
      .ignore_single_req = 0,                                                          \
      .src_inc           = SL_HAL_LDMA_CTRL_SRC_INC_ONE,                               \
      .size              = (transfer_size),                                            \
      .dst_inc           = SL_HAL_LDMA_CTRL_DST_INC_NONE,                              \
      .src_addr_mode     = SL_HAL_LDMA_CTRL_SRC_ADDR_MODE_ABS,                         \
      .dst_addr_mode     = SL_HAL_LDMA_CTRL_DST_ADDR_MODE_ABS,                         \
      .src_addr          = (uint32_t)(src),                                            \
      .dst_addr          = (uint32_t)(dest),                                           \
      .link_mode         = SL_HAL_LDMA_LINK_MODE_REL,                                  \
      .link              = 1,                                                          \
      .link_addr         = (link_jump) * SL_HAL_LDMA_DESCRIPTOR_NON_EXTEND_SIZE_WORD   \
    }                                                                                  \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for linked peripheral to peripheral transfer.
 *
 * @param[in] transfer_size
 *   Transfer size
 *       SL_HAL_LDMA_CTRL_SIZE_WORD,
 *       SL_HAL_LDMA_CTRL_SIZE_HALF,
 *       SL_HAL_LDMA_CTRL_SIZE_BYTE.
 *
 * @param[in] src
 *   Peripheral data source register address.
 *
 * @param[in] dest
 *   Destination data address.
 *
 * @param[in] count
 *   Number of bytes to transfer.
 *
 * @param[in] link_jump
 *   Address of descriptor to link to, expressed as a
 *       signed number of descriptors from "here".
 *       1=one descriptor forward in memory,
 *       0=this descriptor,
 *       -1=one descriptor back in memory.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_LINKREL_P2P(transfer_size, src, dest, count, link_jump) \
  {                                                                                    \
    .xfer =                                                                            \
    {                                                                                  \
      .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_XFER,                          \
      .struct_req        = 0,                                                          \
      .xfer_count        = (count) - 1,                                                \
      .byte_swap         = 0,                                                          \
      .block_size        = SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_1,                         \
      .done_ifs          = 0,                                                          \
      .req_mode          = SL_HAL_LDMA_CTRL_REQ_MODE_BLOCK,                            \
      .dec_loop_count    = 0,                                                          \
      .ignore_single_req = 0,                                                          \
      .src_inc           = SL_HAL_LDMA_CTRL_SRC_INC_NONE,                              \
      .size              = (transfer_size),                                            \
      .dst_inc           = SL_HAL_LDMA_CTRL_DST_INC_NONE,                              \
      .src_addr_mode     = SL_HAL_LDMA_CTRL_SRC_ADDR_MODE_ABS,                         \
      .dst_addr_mode     = SL_HAL_LDMA_CTRL_DST_ADDR_MODE_ABS,                         \
      .src_addr          = (uint32_t)(src),                                            \
      .dst_addr          = (uint32_t)(dest),                                           \
      .link_mode         = SL_HAL_LDMA_LINK_MODE_REL,                                  \
      .link              = 1,                                                          \
      .link_addr         = (link_jump) * SL_HAL_LDMA_DESCRIPTOR_NON_EXTEND_SIZE_WORD   \
    }                                                                                  \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for Immediate WRITE transfer
 *
 * @param[in] value
 *   Immediate value to write.
 *
 * @param[in] address
 *   Write address.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_SINGLE_WRITE(value, address)    \
  {                                                            \
    .wri =                                                     \
    {                                                          \
      .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_WRITE, \
      .struct_req        = 1,                                  \
      .xfer_count        = 0,                                  \
      .byte_swap         = 0,                                  \
      .block_size        = 0,                                  \
      .done_ifs          = 1,                                  \
      .req_mode          = 0,                                  \
      .dec_loop_count    = 0,                                  \
      .ignore_single_req = 0,                                  \
      .src_inc           = 0,                                  \
      .size              = 0,                                  \
      .dst_inc           = 0,                                  \
      .src_addr_mode     = 0,                                  \
      .dst_addr_mode     = 0,                                  \
      .imm_val           = (value),                            \
      .dst_addr          = (uint32_t)(address),                \
      .link_mode         = 0,                                  \
      .link              = 0,                                  \
      .link_addr         = 0                                   \
    }                                                          \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for Immediate WRITE transfer.
 *   Link address must be an absolute address.
 *
 *   Link address must be an absolute address.
 *
 * @note
 *   The link_addr member of the transfer descriptor is not initialized.
 *   link_addr must be initialized by using the proper bits right-shift
 *   to get the correct bits from the absolute address.
 *   _LDMA_CH_LINK_LINKADDR_SHIFT should be used for that operation:
 *
 * @code
     desc.link_addr = ((int32_t)&next_desc) >> _LDMA_CH_LINK_LINKADDR_SHIFT;@endcode
 *   The opposite bit shift (left) must be done if link_addr is read.
 *
 * @param[in] value
 *   Immediate value to write.
 *
 * @param[in] address
 *   Write address.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_LINKABS_WRITE(value, address)   \
  {                                                            \
    .wri =                                                     \
    {                                                          \
      .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_WRITE, \
      .struct_req        = 1,                                  \
      .xfer_count        = 0,                                  \
      .byte_swap         = 0,                                  \
      .block_size        = 0,                                  \
      .done_ifs          = 0,                                  \
      .req_mode          = 0,                                  \
      .dec_loop_count    = 0,                                  \
      .ignore_single_req = 0,                                  \
      .src_inc           = 0,                                  \
      .size              = 0,                                  \
      .dst_inc           = 0,                                  \
      .src_addr_mode     = 0,                                  \
      .dst_addr_mode     = 0,                                  \
      .imm_val           = (value),                            \
      .dst_addr          = (uint32_t)(address),                \
      .link_mode         = SL_HAL_LDMA_LINK_MODE_ABS,          \
      .link              = 1,                                  \
      .link_addr         = 0                                   \
    }                                                          \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for Immediate WRITE transfer.
 *
 * @param[in] value
 *   Immediate value to write.
 *
 * @param[in] address
 *   Write address.
 *
 * @param[in] link_jump
 *   Address of descriptor to link to, expressed as a
 *     signed number of descriptors from "here".
 *     1=one descriptor forward in memory,
 *     0=this descriptor,
 *     -1=one descriptor back in memory.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_LINKREL_WRITE(value, address, link_jump)              \
  {                                                                                  \
    .wri =                                                                           \
    {                                                                                \
      .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_WRITE,                       \
      .struct_req        = 1,                                                        \
      .xfer_count        = 0,                                                        \
      .byte_swap         = 0,                                                        \
      .block_size        = 0,                                                        \
      .done_ifs          = 0,                                                        \
      .req_mode          = 0,                                                        \
      .dec_loop_count    = 0,                                                        \
      .ignore_single_req = 0,                                                        \
      .src_inc           = 0,                                                        \
      .size              = 0,                                                        \
      .dst_inc           = 0,                                                        \
      .src_addr_mode     = 0,                                                        \
      .dst_addr_mode     = 0,                                                        \
      .imm_val           = (value),                                                  \
      .dst_addr          = (uint32_t)(address),                                      \
      .link_mode         = SL_HAL_LDMA_LINK_MODE_REL,                                \
      .link              = 1,                                                        \
      .link_addr         = (link_jump) * SL_HAL_LDMA_DESCRIPTOR_NON_EXTEND_SIZE_WORD \
    }                                                                                \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for SYNC transfer.
 *
 * @param[in] set
 *   Sync pattern bits to set.
 *
 * @param[in] clr
 *   Sync pattern bits to clear.
 *
 * @param[in] match_value
 *   Sync pattern to match.
 *
 * @param[in] match_enable
 *   Sync pattern bits to enable for match.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_SINGLE_SYNC(set, clr, match_value, match_enable) \
  {                                                                             \
    .sync =                                                                     \
    {                                                                           \
      .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_SYNC,                   \
      .struct_req        = 1,                                                   \
      .xfer_count        = 0,                                                   \
      .byte_swap         = 0,                                                   \
      .block_size        = 0,                                                   \
      .done_ifs          = 1,                                                   \
      .req_mode          = 0,                                                   \
      .dec_loop_count    = 0,                                                   \
      .ignore_single_req = 0,                                                   \
      .src_inc           = 0,                                                   \
      .size              = 0,                                                   \
      .dst_inc           = 0,                                                   \
      .src_addr_mode     = 0,                                                   \
      .dst_addr_mode     = 0,                                                   \
      .sync_set          = (set),                                               \
      .sync_clr          = (clr),                                               \
      .match_val         = (match_value),                                       \
      .match_en          = (match_enable),                                      \
      .link_mode         = 0,                                                   \
      .link              = 0,                                                   \
      .link_addr         = 0                                                    \
    }                                                                           \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for SYNC transfer.
 *   Link address must be an absolute address.
 *
 * @note
 *   The link_addr member of the transfer descriptor is not initialized.
 *   link_addr must be initialized by using the proper bits right-shift
 *   to get the correct bits from the absolute address.
 *   _LDMA_CH_LINK_LINKADDR_SHIFT should be used for that operation:
 *
 * @code
 *   desc.link_addr = ((int32_t)&next_desc) >> _LDMA_CH_LINK_LINKADDR_SHIFT;@endcode
 *   The opposite bit shift (left) must be done if link_addr is read.
 *
 * @param[in] set
 *   Sync pattern bits to set.
 *
 * @param[in] clr
 *   Sync pattern bits to clear.
 *
 * @param[in] match_value
 *   Sync pattern to match.
 *
 * @param[in] match_enable
 *   Sync pattern bits to enable for match.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_LINKABS_SYNC(set, clr, match_value, match_enable) \
  {                                                                              \
    .sync =                                                                      \
    {                                                                            \
      .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_SYNC,                    \
      .struct_req        = 1,                                                    \
      .xfer_count        = 0,                                                    \
      .byte_swap         = 0,                                                    \
      .block_size        = 0,                                                    \
      .done_ifs          = 0,                                                    \
      .req_mode          = 0,                                                    \
      .dec_loop_count    = 0,                                                    \
      .ignore_single_req = 0,                                                    \
      .src_inc           = 0,                                                    \
      .size              = 0,                                                    \
      .dst_inc           = 0,                                                    \
      .src_addr_mode     = 0,                                                    \
      .dst_addr_mode     = 0,                                                    \
      .sync_set          = (set),                                                \
      .sync_clr          = (clr),                                                \
      .match_val         = (match_value),                                        \
      .match_en          = (match_enable),                                       \
      .link_mode         = SL_HAL_LDMA_LINK_MODE_ABS,                            \
      .link              = 1,                                                    \
      .link_addr         = 0                                                     \
    }                                                                            \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for SYNC transfer.
 *
 * @param[in] set
 *   Sync pattern bits to set.
 *
 * @param[in] clr
 *   Sync pattern bits to clear.
 *
 * @param[in] match_value
 *   Sync pattern to match.
 *
 * @param[in] match_enable
 *   Sync pattern bits to enable for match.
 *
 * @param[in] link_jump
 *   Address of descriptor to link to, expressed as a
 *       signed number of descriptors from "here".
 *       1=one descriptor forward in memory,
 *       0=this descriptor,
 *       -1=one descriptor back in memory.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_LINKREL_SYNC(set, clr, match_value, match_enable, link_jump) \
  {                                                                                         \
    .sync =                                                                                 \
    {                                                                                       \
      .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_SYNC,                               \
      .struct_req        = 1,                                                               \
      .xfer_count        = 0,                                                               \
      .byte_swap         = 0,                                                               \
      .block_size        = 0,                                                               \
      .done_ifs          = 0,                                                               \
      .req_mode          = 0,                                                               \
      .dec_loop_count    = 0,                                                               \
      .ignore_single_req = 0,                                                               \
      .src_inc           = 0,                                                               \
      .size              = 0,                                                               \
      .dst_inc           = 0,                                                               \
      .src_addr_mode     = 0,                                                               \
      .dst_addr_mode     = 0,                                                               \
      .sync_set          = (set),                                                           \
      .sync_clr          = (clr),                                                           \
      .match_val         = (match_value),                                                   \
      .match_en          = (match_enable),                                                  \
      .link_mode         = SL_HAL_LDMA_LINK_MODE_REL,                                       \
      .link              = 1,                                                               \
      .link_addr         = (link_jump) * SL_HAL_LDMA_DESCRIPTOR_NON_EXTEND_SIZE_WORD        \
    }                                                                                       \
  }

#if defined(_LDMA_CH_CTRL_EXTEND_MASK)
/***************************************************************************//**
 * @brief
 *   Initializer for the destination interleaving portion of the LDMA extended descriptor.
 *
 * @param[in] desc
 *   Transfer-type descriptor.
 *
 * @param[in] il_mode_value
 *   Rules table addressing mode for interleaved data.
 *
 * @param[in] il_src_addr_value
 *   Base address for rules table in memory.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_EXTEND_DST_IL_CFG(desc, il_mode_value, il_src_addr_value) \
  {                                                                                      \
    (desc).dst_inc     = SL_HAL_LDMA_CTRL_DST_INC_NONE;                                  \
    (desc).dst_il_en   = true;                                                           \
    (desc).il_mode     = (il_mode_value);                                                \
    (desc).il_src_addr = (uint32_t)(il_src_addr_value);                                  \
  }

#if defined(_LDMA_CH_XCTRL_SRCMEMTYPE_MASK)
/***************************************************************************//**
 * @brief
 *   Initializer for the type of memory access for the source and for the destination.
 *
 * @param[in] desc
 *   Transfer-type descriptor.
 *
 * @param[in] src_mem_type_value
 *   Type of memory access for the source.
 *
 * @param[in] dst_mem_type_value
 *   Type of memory access for the destination.
 *       SL_HAL_LDMA_MEM_TYPE_NON_CACHEABLE,
 *       SL_HAL_LDMA_MEM_TYPE_CACHEABLE.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_EXTEND_MEM_TYPE(desc, src_mem_type_value, dst_mem_type_value) \
  {                                                                                          \
    (desc).src_mem_type = src_mem_type_value;                                                \
    (desc).dst_mem_type = dst_mem_type_value;                                                \
  }
#endif

#if defined(_LDMA_CH_XCTRL_DUALDSTEN_MASK)
/***************************************************************************//**
 * @brief
 *   Initializer for the dual destination portion of the LDMA extended descriptor.
 *
 * @param[in] desc
 *   Transfer-type descriptor.
 *
 * @param[in] dual_dst_addr_value
 *   Base address for the second destination.
 *
 * @param[in] dual_dst_inc_value
 *   Increment unit size for the second destination.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_EXTEND_DUAL_DST_CFG(desc, dual_dst_addr_value, dual_dst_inc_value) \
  {                                                                                               \
    (desc).dual_dst_en   = true;                                                                  \
    (desc).dual_dst_inc  = dual_dst_inc_value;                                                    \
    (desc).dual_dst_addr = (uint32_t)(dual_dst_addr_value);                                       \
  }
#endif

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for single memory to memory transfer.
 *
 * @param[in] transfer_size
 *   Transfer size
 *                                     SL_HAL_LDMA_CTRL_SIZE_WORD,
 *                                     SL_HAL_LDMA_CTRL_SIZE_HALF,
 *                                     SL_HAL_LDMA_CTRL_SIZE_BYTE.
 *
 * @param[in] src
 *   Source data address.
 *
 * @param[in] dest
 *   Destination data address.
 *
 * @param[in] count
 *   Number of words to transfer.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_SINGLE_M2M_EXTEND(transfer_size, src, dest, count) \
  {                                                                               \
    .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_XFER,                       \
    .extend            = 1,                                                       \
    .struct_req        = 1,                                                       \
    .xfer_count        = SL_HAL_LDMA_EXTEND_XFER_COUNT(count),                    \
    .byte_swap         = 0,                                                       \
    .block_size        = SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_1,                      \
    .done_ifs          = 1,                                                       \
    .req_mode          = SL_HAL_LDMA_CTRL_REQ_MODE_ALL,                           \
    .dec_loop_count    = 0,                                                       \
    .ignore_single_req = 0,                                                       \
    .src_inc           = SL_HAL_LDMA_CTRL_SRC_INC_ONE,                            \
    .size              = (transfer_size),                                         \
    .dst_inc           = SL_HAL_LDMA_CTRL_DST_INC_ONE,                            \
    .src_addr_mode     = SL_HAL_LDMA_CTRL_SRC_ADDR_MODE_ABS,                      \
    .dst_addr_mode     = SL_HAL_LDMA_CTRL_DST_ADDR_MODE_ABS,                      \
    .src_addr          = (uint32_t)(src),                                         \
    .dst_addr          = (uint32_t)(dest),                                        \
    .link_mode         = 0,                                                       \
    .link              = 0,                                                       \
    .link_addr         = 0,                                                       \
    SL_HAL_LDMA_EXTEND_DEFAULT(count)                                             \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for single peripheral to memory transfer.
 *
 * @param[in] transfer_size
 *   Transfer size
 *       SL_HAL_LDMA_CTRL_SIZE_WORD,
 *       SL_HAL_LDMA_CTRL_SIZE_HALF,
 *       SL_HAL_LDMA_CTRL_SIZE_BYTE.
 *
 * @param[in] src
 *   Peripheral data source register address.
 *
 * @param[in] dest
 *   Destination data address.
 *
 * @param[in] count
 *   Number of bytes to transfer.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_SINGLE_P2M_EXTEND(transfer_size, src, dest, count) \
  {                                                                               \
    .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_XFER,                       \
    .extend            = 1,                                                       \
    .struct_req        = 0,                                                       \
    .xfer_count        = SL_HAL_LDMA_EXTEND_XFER_COUNT(count),                    \
    .byte_swap         = 0,                                                       \
    .block_size        = SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_1,                      \
    .done_ifs          = 1,                                                       \
    .req_mode          = SL_HAL_LDMA_CTRL_REQ_MODE_BLOCK,                         \
    .dec_loop_count    = 0,                                                       \
    .ignore_single_req = 0,                                                       \
    .src_inc           = SL_HAL_LDMA_CTRL_DST_INC_NONE,                           \
    .size              = (transfer_size),                                         \
    .dst_inc           = SL_HAL_LDMA_CTRL_DST_INC_ONE,                            \
    .src_addr_mode     = SL_HAL_LDMA_CTRL_SRC_ADDR_MODE_ABS,                      \
    .dst_addr_mode     = SL_HAL_LDMA_CTRL_DST_ADDR_MODE_ABS,                      \
    .src_addr          = (uint32_t)(src),                                         \
    .dst_addr          = (uint32_t)(dest),                                        \
    .link_mode         = 0,                                                       \
    .link              = 0,                                                       \
    .link_addr         = 0,                                                       \
    SL_HAL_LDMA_EXTEND_DEFAULT(count)                                             \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for single memory to peripheral transfer.
 *
 * @param[in] transfer_size
 *   Transfer size
 *                                     SL_HAL_LDMA_CTRL_SIZE_WORD,
 *                                     SL_HAL_LDMA_CTRL_SIZE_HALF,
 *                                     SL_HAL_LDMA_CTRL_SIZE_BYTE.
 *
 * @param[in] src
 *   Source data address.
 *
 * @param[in] dest
 *   Peripheral data register destination address.
 *
 * @param[in] count
 *   Number of bytes to transfer.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_SINGLE_M2P_EXTEND(transfer_size, src, dest, count) \
  {                                                                               \
    .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_XFER,                       \
    .extend            = 1,                                                       \
    .struct_req        = 0,                                                       \
    .xfer_count        = SL_HAL_LDMA_EXTEND_XFER_COUNT(count),                    \
    .byte_swap         = 0,                                                       \
    .block_size        = SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_1,                      \
    .done_ifs          = 1,                                                       \
    .req_mode          = SL_HAL_LDMA_CTRL_REQ_MODE_BLOCK,                         \
    .dec_loop_count    = 0,                                                       \
    .ignore_single_req = 0,                                                       \
    .src_inc           = SL_HAL_LDMA_CTRL_DST_INC_ONE,                            \
    .size              = (transfer_size),                                         \
    .dst_inc           = SL_HAL_LDMA_CTRL_DST_INC_NONE,                           \
    .src_addr_mode     = SL_HAL_LDMA_CTRL_SRC_ADDR_MODE_ABS,                      \
    .dst_addr_mode     = SL_HAL_LDMA_CTRL_DST_ADDR_MODE_ABS,                      \
    .src_addr          = (uint32_t)(src),                                         \
    .dst_addr          = (uint32_t)(dest),                                        \
    .link_mode         = 0,                                                       \
    .link              = 0,                                                       \
    .link_addr         = 0,                                                       \
    SL_HAL_LDMA_EXTEND_DEFAULT(count)                                             \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for linked memory to memory transfer.
 *   Link address must be an absolute address.
 *
 * @note
 *   The link_addr member of the transfer descriptor is not initialized.
 *   link_addr must be initialized by using the proper bits right-shift
 *   to get the correct bits from the absolute address.
 *   _LDMA_CH_LINK_LINKADDR_SHIFT should be used for that operation:
 *
 * @code
 * desc.link_addr = ((int32_t)&next_desc) >> _LDMA_CH_LINK_LINKADDR_SHIFT;
 * @endcode
 * The opposite bit shift (left) must be done if link_addr is read.
 *
 * @param[in] transfer_size
 *   Transfer size
 *       SL_HAL_LDMA_CTRL_SIZE_WORD,
 *       SL_HAL_LDMA_CTRL_SIZE_HALF,
 *       SL_HAL_LDMA_CTRL_SIZE_BYTE.
 *
 * @param[in] src
 *   Source data address.
 *
 * @param[in] dest
 *   Destination data address.
 *
 * @param[in] count
 *   Number of words to transfer.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_LINKABS_M2M_EXTEND(transfer_size, src, dest, count) \
  {                                                                                \
    .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_XFER,                        \
    .extend            = 1,                                                        \
    .struct_req        = 1,                                                        \
    .xfer_count        = SL_HAL_LDMA_EXTEND_XFER_COUNT(count),                     \
    .byte_swap         = 0,                                                        \
    .block_size        = SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_1,                       \
    .done_ifs          = 0,                                                        \
    .req_mode          = SL_HAL_LDMA_CTRL_REQ_MODE_ALL,                            \
    .dec_loop_count    = 0,                                                        \
    .ignore_single_req = 0,                                                        \
    .src_inc           = SL_HAL_LDMA_CTRL_SRC_INC_ONE,                             \
    .size              = (transfer_size),                                          \
    .dst_inc           = SL_HAL_LDMA_CTRL_DST_INC_ONE,                             \
    .src_addr_mode     = SL_HAL_LDMA_CTRL_SRC_ADDR_MODE_ABS,                       \
    .dst_addr_mode     = SL_HAL_LDMA_CTRL_DST_ADDR_MODE_ABS,                       \
    .src_addr          = (uint32_t)(src),                                          \
    .dst_addr          = (uint32_t)(dest),                                         \
    .link_mode         = SL_HAL_LDMA_LINK_MODE_ABS,                                \
    .link              = 1,                                                        \
    .link_addr         = 0, /* Must be set runtime ! */                            \
    SL_HAL_LDMA_EXTEND_DEFAULT(count)                                              \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for linked peripheral to memory transfer.
 *   Link address must be an absolute address.
 *
 * @note
 *   The link_addr member of the transfer descriptor is not initialized.
 *   link_addr must be initialized by using the proper bits right-shift
 *   to get the correct bits from the absolute address.
 *   _LDMA_CH_LINK_LINKADDR_SHIFT should be used for that operation:
 *
 * @code
 * desc.link_addr = ((int32_t)&next_desc) >> _LDMA_CH_LINK_LINKADDR_SHIFT;
 * @endcode
 * The opposite bit shift (left) must be done if link_addr is read.
 *
 * @param[in] transfer_size
 *   Transfer size
 *       SL_HAL_LDMA_CTRL_SIZE_WORD,
 *       SL_HAL_LDMA_CTRL_SIZE_HALF,
 *       SL_HAL_LDMA_CTRL_SIZE_BYTE.
 *
 * @param[in] src
 *   Source data address.
 *
 * @param[in] dest
 *   Destination data address.
 *
 * @param[in] count
 *   Number of words to transfer.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_LINKABS_P2M_EXTEND(transfer_size, src, dest, count) \
  {                                                                                \
    .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_XFER,                        \
    .extend            = 1,                                                        \
    .struct_req        = 0,                                                        \
    .xfer_count        = SL_HAL_LDMA_EXTEND_XFER_COUNT(count),                     \
    .byte_swap         = 0,                                                        \
    .block_size        = SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_1,                       \
    .done_ifs          = 0,                                                        \
    .req_mode          = SL_HAL_LDMA_CTRL_REQ_MODE_BLOCK,                          \
    .dec_loop_count    = 0,                                                        \
    .ignore_single_req = 0,                                                        \
    .src_inc           = SL_HAL_LDMA_CTRL_SRC_INC_NONE,                            \
    .size              = (transfer_size),                                          \
    .dst_inc           = SL_HAL_LDMA_CTRL_DST_INC_ONE,                             \
    .src_addr_mode     = SL_HAL_LDMA_CTRL_SRC_ADDR_MODE_ABS,                       \
    .dst_addr_mode     = SL_HAL_LDMA_CTRL_DST_ADDR_MODE_ABS,                       \
    .src_addr          = (uint32_t)(src),                                          \
    .dst_addr          = (uint32_t)(dest),                                         \
    .link_mode         = SL_HAL_LDMA_LINK_MODE_ABS,                                \
    .link              = 1,                                                        \
    .link_addr         = 0, /* Must be set runtime ! */                            \
    SL_HAL_LDMA_EXTEND_DEFAULT(count)                                              \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for linked memory to peripheral word transfer.
 *   Link address must be an absolute address.
 *
 * @note
 *   The link_addr member of the transfer descriptor is not initialized.
 *   link_addr must be initialized by using the proper bits right-shift
 *   to get the correct bits from the absolute address.
 *   _LDMA_CH_LINK_LINKADDR_SHIFT should be used for that operation:
 *
 * @code
 * desc.link_addr = ((int32_t)&next_desc) >> _LDMA_CH_LINK_LINKADDR_SHIFT;
 * @endcode
 * The opposite bit shift (left) must be done if link_addr is read.
 *
 * @param[in] transfer_size
 *   Transfer size
 *       SL_HAL_LDMA_CTRL_SIZE_WORD,
 *       SL_HAL_LDMA_CTRL_SIZE_HALF,
 *       SL_HAL_LDMA_CTRL_SIZE_BYTE.
 *
 * @param[in] src
 *   Source data address.
 *
 * @param[in] dest
 *   Destination data address.
 *
 * @param[in] count
 *   Number of words to transfer.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_LINKABS_M2P_EXTEND(transfer_size, src, dest, count) \
  {                                                                                \
    .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_XFER,                        \
    .extend            = 1,                                                        \
    .struct_req        = 0,                                                        \
    .xfer_count        = SL_HAL_LDMA_EXTEND_XFER_COUNT(count),                     \
    .byte_swap         = 0,                                                        \
    .block_size        = SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_1,                       \
    .done_ifs          = 0,                                                        \
    .req_mode          = SL_HAL_LDMA_CTRL_REQ_MODE_BLOCK,                          \
    .dec_loop_count    = 0,                                                        \
    .ignore_single_req = 0,                                                        \
    .src_inc           = SL_HAL_LDMA_CTRL_SRC_INC_ONE,                             \
    .size              = (transfer_size),                                          \
    .dst_inc           = SL_HAL_LDMA_CTRL_DST_INC_NONE,                            \
    .src_addr_mode     = SL_HAL_LDMA_CTRL_SRC_ADDR_MODE_ABS,                       \
    .dst_addr_mode     = SL_HAL_LDMA_CTRL_DST_ADDR_MODE_ABS,                       \
    .src_addr          = (uint32_t)(src),                                          \
    .dst_addr          = (uint32_t)(dest),                                         \
    .link_mode         = SL_HAL_LDMA_LINK_MODE_ABS,                                \
    .link              = 1,                                                        \
    .link_addr         = 0, /* Must be set runtime ! */                            \
    SL_HAL_LDMA_EXTEND_DEFAULT(count)                                              \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for linked memory to memory transfer.
 *   Link address is a relative address.
 *
 * @note
 *   The link_addr member of the transfer descriptor is initialized to 4
 *   (regular descriptor) or 7 (extended descriptor),  assuming that
 *   the next descriptor immediately follows this descriptor (in memory).
 *
 * @param[in] transfer_size
 *   Transfer size
 *       SL_HAL_LDMA_CTRL_SIZE_WORD,
 *       SL_HAL_LDMA_CTRL_SIZE_HALF,
 *       SL_HAL_LDMA_CTRL_SIZE_BYTE.
 *
 * @param[in] src
 *   Source data address.
 *
 * @param[in] dest
 *   Destination data address.
 *
 * @param[in] count
 *   Number of words to transfer.
 *
 * @param[in] link_jump
 *   Address of descriptor to link to, expressed as a
 *       signed number of descriptors from "here".
 *       1=one descriptor forward in memory,
 *       0=this descriptor,
 *       -1=one descriptor back in memory.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_LINKREL_M2M_EXTEND(transfer_size, src, dest, count, link_jump) \
  {                                                                                           \
    .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_XFER,                                   \
    .extend            = 1,                                                                   \
    .struct_req        = 1,                                                                   \
    .xfer_count        = SL_HAL_LDMA_EXTEND_XFER_COUNT(count),                                \
    .byte_swap         = 0,                                                                   \
    .block_size        = SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_1,                                  \
    .done_ifs          = 0,                                                                   \
    .req_mode          = SL_HAL_LDMA_CTRL_REQ_MODE_ALL,                                       \
    .dec_loop_count    = 0,                                                                   \
    .ignore_single_req = 0,                                                                   \
    .src_inc           = SL_HAL_LDMA_CTRL_SRC_INC_ONE,                                        \
    .size              = (transfer_size),                                                     \
    .dst_inc           = SL_HAL_LDMA_CTRL_DST_INC_ONE,                                        \
    .src_addr_mode     = SL_HAL_LDMA_CTRL_SRC_ADDR_MODE_ABS,                                  \
    .dst_addr_mode     = SL_HAL_LDMA_CTRL_DST_ADDR_MODE_ABS,                                  \
    .src_addr          = (uint32_t)(src),                                                     \
    .dst_addr          = (uint32_t)(dest),                                                    \
    .link_mode         = SL_HAL_LDMA_LINK_MODE_REL,                                           \
    .link              = 1,                                                                   \
    .link_addr         = (link_jump) * SL_HAL_LDMA_DESCRIPTOR_EXTEND_SIZE_WORD,               \
    SL_HAL_LDMA_EXTEND_DEFAULT(count)                                                         \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for linked peripheral to memory transfer.
 *
 * @param[in] transfer_size
 *   Transfer size
 *       SL_HAL_LDMA_CTRL_SIZE_WORD,
 *       SL_HAL_LDMA_CTRL_SIZE_HALF,
 *       SL_HAL_LDMA_CTRL_SIZE_BYTE.
 *
 * @param[in] src
 *   Peripheral data source register address.
 *
 * @param[in] dest
 *   Destination data address.
 *
 * @param[in] count
 *   Number of bytes to transfer.
 *
 * @param[in] link_jump
 *   Address of descriptor to link to, expressed as a
 *       signed number of descriptors from "here".
 *       1=one descriptor forward in memory,
 *       0=this descriptor,
 *       -1=one descriptor back in memory.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_LINKREL_P2M_EXTEND(transfer_size, src, dest, count, link_jump) \
  {                                                                                           \
    .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_XFER,                                   \
    .extend            = 1,                                                                   \
    .struct_req        = 0,                                                                   \
    .xfer_count        = SL_HAL_LDMA_EXTEND_XFER_COUNT(count),                                \
    .byte_swap         = 0,                                                                   \
    .block_size        = SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_1,                                  \
    .done_ifs          = 0,                                                                   \
    .req_mode          = SL_HAL_LDMA_CTRL_REQ_MODE_BLOCK,                                     \
    .dec_loop_count    = 0,                                                                   \
    .ignore_single_req = 0,                                                                   \
    .src_inc           = SL_HAL_LDMA_CTRL_SRC_INC_NONE,                                       \
    .size              = (transfer_size),                                                     \
    .dst_inc           = SL_HAL_LDMA_CTRL_DST_INC_ONE,                                        \
    .src_addr_mode     = SL_HAL_LDMA_CTRL_SRC_ADDR_MODE_ABS,                                  \
    .dst_addr_mode     = SL_HAL_LDMA_CTRL_DST_ADDR_MODE_ABS,                                  \
    .src_addr          = (uint32_t)(src),                                                     \
    .dst_addr          = (uint32_t)(dest),                                                    \
    .link_mode         = SL_HAL_LDMA_LINK_MODE_REL,                                           \
    .link              = 1,                                                                   \
    .link_addr         = (link_jump) * SL_HAL_LDMA_DESCRIPTOR_EXTEND_SIZE_WORD,               \
    SL_HAL_LDMA_EXTEND_DEFAULT(count)                                                         \
  }

/***************************************************************************//**
 * @brief
 *   LDMA descriptor initializer for linked memory to peripheral transfer.
 *
 * @param[in] transfer_size
 *   Transfer size
 *       SL_HAL_LDMA_CTRL_SIZE_WORD,
 *       SL_HAL_LDMA_CTRL_SIZE_HALF,
 *       SL_HAL_LDMA_CTRL_SIZE_BYTE.
 * @param[in] src
 *   Peripheral data source register address.
 *
 * @param[in] dest
 *   Destination data address.
 *
 * @param[in] count
 *   Number of bytes to transfer.
 *
 * @param[in] link_jump
 *   Address of descriptor to link to, expressed as a
 *       signed number of descriptors from "here".
 *       1=one descriptor forward in memory,
 *       0=this descriptor,
 *       -1=one descriptor back in memory.
 ******************************************************************************/
#define SL_HAL_LDMA_DESCRIPTOR_LINKREL_M2P_EXTEND(transfer_size, src, dest, count, link_jump) \
  {                                                                                           \
    .struct_type       = SL_HAL_LDMA_CTRL_STRUCT_TYPE_XFER,                                   \
    .extend            = 1,                                                                   \
    .struct_req        = 0,                                                                   \
    .xfer_count        = SL_HAL_LDMA_EXTEND_XFER_COUNT(count),                                \
    .byte_swap         = 0,                                                                   \
    .block_size        = SL_HAL_LDMA_CTRL_BLOCK_SIZE_UNIT_1,                                  \
    .done_ifs          = 0,                                                                   \
    .req_mode          = SL_HAL_LDMA_CTRL_REQ_MODE_BLOCK,                                     \
    .dec_loop_count    = 0,                                                                   \
    .ignore_single_req = 0,                                                                   \
    .src_inc           = SL_HAL_LDMA_CTRL_SRC_INC_ONE,                                        \
    .size              = (transfer_size),                                                     \
    .dst_inc           = SL_HAL_LDMA_CTRL_DST_INC_NONE,                                       \
    .src_addr_mode     = SL_HAL_LDMA_CTRL_SRC_ADDR_MODE_ABS,                                  \
    .dst_addr_mode     = SL_HAL_LDMA_CTRL_DST_ADDR_MODE_ABS,                                  \
    .src_addr          = (uint32_t)(src),                                                     \
    .dst_addr          = (uint32_t)(dest),                                                    \
    .link_mode         = SL_HAL_LDMA_LINK_MODE_REL,                                           \
    .link              = 1,                                                                   \
    .link_addr         = (link_jump) * SL_HAL_LDMA_DESCRIPTOR_EXTEND_SIZE_WORD,               \
    SL_HAL_LDMA_EXTEND_DEFAULT(count)                                                         \
  }
#endif

/*******************************************************************************
 *****************************   PROTOTYPES   **********************************
 ******************************************************************************/

/***************************************************************************//**
 * @brief
 *   Initialize an extended LDMA transfer.
 *
 * @details
 *   This function will disable all the LDMA channels.
 *   This function will also set the LDMA IRQ priority to a user-configurable
 *   priority. The LDMA interrupt priority is configured using the
 *   @ref sl_hal_ldma__t structure.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @param[in] config
 *   A pointer to the initialization structure used to configure the LDMA.
 ******************************************************************************/
void sl_hal_ldma_init(LDMA_TypeDef *ldma,
                      const sl_hal_ldma_init_t *init);

/***************************************************************************//**
 * @brief
 *   Initialize a LDMA transfer.
 *
 * @note
 *   If the descriptor is allocated on the stack, its content may change
 *   unpredictably when the function exits. To prevent this, declare the
 *   descriptor as a static or global variable, or allocate memory for it
 *   using `malloc`. Ensure that the memory location of the descriptor
 *   remains stable until the LDMA transfer is complete, as the LDMA controller
 *   may load the descriptor at any time.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @param[in] channel
 *   A LDMA channel to stop.
 *
 * @param[in] transfer_init
 *   The initialization structure used to configure the transfer.
 *
 * @param[in] descriptor
 *   The transfer descriptor, which can be an array of descriptors linked together.
 *   Each descriptor's fields stored in RAM will be loaded into the certain
 *   hardware registers at the proper time to perform the LDMA transfer.
 ******************************************************************************/
void sl_hal_ldma_init_transfer(LDMA_TypeDef *ldma,
                               uint32_t channel,
                               const sl_hal_ldma_transfer_init_t *transfer_init,
                               const sl_hal_ldma_descriptor_t *descriptor);

#if defined(_LDMA_CH_CTRL_EXTEND_MASK)
/***************************************************************************//**
 * @brief
 *   Initialize a LDMA transfer.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @param[in] channel
 *   A LDMA channel to stop.
 *
 * @param[in] transfer_init
 *   The initialization structure used to configure the transfer.
 *
 * @param[in] descriptor_extend
 *   The transfer extended descriptor, which can be an array of descriptors
 *   linked together. Each descriptor's fields stored in RAM will be loaded
 *   into the certain hardware registers at the proper time to perform the
 *   LDMA transfer.
 ******************************************************************************/
void sl_hal_ldma_init_transfer_extend(LDMA_TypeDef *ldma,
                                      uint32_t channel,
                                      const sl_hal_ldma_transfer_init_t *transfer_init,
                                      const sl_hal_ldma_descriptor_extend_t *descriptor_extend);
#endif

/***************************************************************************//**
 * @brief
 *   Start a LDMA transfer.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @param[in] channel
 *   A DMA channel.
 ******************************************************************************/
void sl_hal_ldma_start_transfer(LDMA_TypeDef *ldma,
                                uint32_t channel);

/***************************************************************************//**
 * @brief
 *   Stop a LDMA transfer.
 *
 * @note
 *   The LDMA will complete the current AHB burst transfer before stopping.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @param[in] channel
 *   A LDMA channel to stop.
 ******************************************************************************/
SL_CODE_CLASSIFY(SL_CODE_COMPONENT_HAL_LDMA, SL_CODE_CLASS_TIME_CRITICAL)
void sl_hal_ldma_stop_transfer(LDMA_TypeDef *ldma,
                               uint32_t channel);

/***************************************************************************//**
 * @brief
 *   Check if a LDMA transfer has completed.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @param[in] channel
 *   A LDMA channel to check.
 *
 * @return
 *   True if transfer has completed, false if not.
 ******************************************************************************/
SL_CODE_CLASSIFY(SL_CODE_COMPONENT_HAL_LDMA, SL_CODE_CLASS_TIME_CRITICAL)
bool sl_hal_ldma_transfer_is_done(LDMA_TypeDef *ldma,
                                  uint32_t channel);

/***************************************************************************//**
 * @brief
 *  Get the number of items remaining in a transfer.
 *
 * @note
 *  This function does not take into account that a LDMA transfer with
 *  a chain of linked transfers might be ongoing. It will only check the
 *  count for the current transfer.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @param[in] channel
 *  The channel number of the transfer to check.
 *
 * @return
 *  A number of items remaining in the transfer.
 ******************************************************************************/
SL_CODE_CLASSIFY(SL_CODE_COMPONENT_HAL_LDMA, SL_CODE_CLASS_TIME_CRITICAL)
uint32_t sl_hal_ldma_transfer_remaining_count(LDMA_TypeDef *ldma,
                                              uint32_t channel);

#if defined(_LDMA_SWRST_MASK) && defined(_LDMA_EN_DISABLING_MASK)
/***************************************************************************//**
 * @brief
 *   Wait for the LDMA to complete disabling procedure.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 ******************************************************************************/
__INLINE void sl_hal_ldma_wait_ready(LDMA_TypeDef *ldma)
{
  while (ldma->SWRST & _LDMA_SWRST_RESETTING_MASK) {
    // Wait for reset to complete.
  }

  while (ldma->EN & _LDMA_EN_DISABLING_MASK) {
    // Wait for disabling to finish.
  }
}
#endif

/***************************************************************************//**
 * @brief
 *   Enable LDMA.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 ******************************************************************************/
__INLINE void sl_hal_ldma_enable(LDMA_TypeDef *ldma)
{
#if defined(_LDMA_SWRST_MASK) && defined(_LDMA_EN_DISABLING_MASK)
  // Wait for the LDMA to complete disabling procedure.
  sl_hal_ldma_wait_ready(ldma);
#endif

  // Enable LDMA.
  ldma->EN_SET = LDMA_EN_EN;
}

/***************************************************************************//**
 * @brief
 *   Enable LDMA channel.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @param[in] channel
 *   A LDMA channel.
 ******************************************************************************/
__INLINE void sl_hal_ldma_enable_channel(LDMA_TypeDef *ldma,
                                         uint32_t channel)
{
  uint32_t ch_mask = 1UL << channel;
  EFM_ASSERT(channel < DMA_CHAN_COUNT);

  // Enable LDMA channel
  ldma->CHEN_SET = ch_mask;
}

/***************************************************************************//**
 * @brief
 *   Disable LDMA.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 ******************************************************************************/
__INLINE void sl_hal_ldma_disable(LDMA_TypeDef *ldma)
{
  if (ldma->EN != LDMA_EN_EN) {
    return;
  }

  // Disable LDMA
  ldma->EN_CLR = LDMA_EN_EN;
}

/***************************************************************************//**
 * @brief
 *   Disable LDMA channel.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @param[in] channel
 *   A LDMA channel.
 ******************************************************************************/
SL_CODE_CLASSIFY(SL_CODE_COMPONENT_HAL_LDMA, SL_CODE_CLASS_TIME_CRITICAL)
__INLINE void sl_hal_ldma_disable_channel(LDMA_TypeDef *ldma,
                                          uint32_t channel)
{
  uint32_t ch_mask = 1UL << channel;
  EFM_ASSERT(channel < DMA_CHAN_COUNT);

  // Disable LDMA channel
  ldma->CHDIS_SET = ch_mask;
}

/***************************************************************************//**
 * @brief
 *   Gets LDMA STATUS register value.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @return  Current STATUS register value.
 ******************************************************************************/
__INLINE uint32_t sl_hal_ldma_get_status(LDMA_TypeDef *ldma)
{
  return ldma->STATUS;
}

#if defined(_LDMA_SWRST_MASK)
/***************************************************************************//**
 * @brief
 *   Reset the LDMA.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 ******************************************************************************/
__INLINE void sl_hal_ldma_reset(LDMA_TypeDef *ldma)
{
  ldma->SWRST_SET = LDMA_SWRST_SWRST;
}
#endif

/***************************************************************************//**
 * @brief
 *   Check if a certain channel is enabled.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @param[in] channel
 *   LDMA channel to check.
 *
 * @return
 *   return true if the LDMA channel is enabled and false if the channel is not
 *   enabled.
 ******************************************************************************/
SL_CODE_CLASSIFY(SL_CODE_COMPONENT_HAL_LDMA, SL_CODE_CLASS_TIME_CRITICAL)
__INLINE bool sl_hal_ldma_channel_is_enabled(LDMA_TypeDef *ldma,
                                             uint32_t channel)
{
  EFM_ASSERT(channel < DMA_CHAN_COUNT);
  return ldma->CHSTATUS & (1 << channel);
}

/***************************************************************************//**
 * @brief
 *   Enable an LDMA channel request.
 *
 * @details
 *   Use this function to enable an LDMA channel request. This will
 *   prevent the LDMA from proceeding after its current transaction is disabled.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @param[in] channel
 *   LDMA channel to enable requests.
 *
 ******************************************************************************/
__INLINE void sl_hal_ldma_enable_channel_request(LDMA_TypeDef *ldma,
                                                 uint32_t channel)
{
  uint32_t ch_mask = 1UL << channel;
  EFM_ASSERT(channel < DMA_CHAN_COUNT);

  // Enable LDMA channel request
  ldma->REQDIS_CLR = ch_mask;
}

/***************************************************************************//**
 * @brief
 *   Disable an LDMA channel request.
 *
 * @details
 *   Use this function to disable an LDMA channel request. This will
 *   prevent the LDMA from proceeding after its current transaction is disabled.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @param[in] channel
 *   LDMA channel to disable requests.
 *
 ******************************************************************************/
__INLINE void sl_hal_ldma_disable_channel_request(LDMA_TypeDef *ldma,
                                                  uint32_t channel)
{
  uint32_t ch_mask = 1UL << channel;
  EFM_ASSERT(channel < DMA_CHAN_COUNT);

  // Disable LDMA channel request
  ldma->REQDIS_SET = ch_mask;
}

/***************************************************************************//**
 * @brief
 *   Clear one or more pending LDMA interrupts for LDMA channels 0 to 15.
 *
 * @note
 *   For series 2 devices, there is only one LDMA_IF_ERROR shared across all
 *   channels. Use the appropriate flags to clear these interrupts.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @param[in] flags
 *   Pending LDMA interrupt sources to clear. Use one or more valid
 *   interrupt flags for the LDMA peripheral module. The flags are one LDMA_IF_ERRORx flag
 *   for each channel (x = a supported channel number) and one LDMA_IF_DONEx flag
 *   for each channel (x = a supported channel number). For series 2 devices,
 *   use LDMA_IF_ERROR and LDMA_IF_DONEx.
 ******************************************************************************/
__INLINE void sl_hal_ldma_clear_interrupts(LDMA_TypeDef *ldma,
                                           uint32_t flags)
{
  ldma->IF_CLR = flags;
}

/***************************************************************************//**
 * @brief
 *   Disable one or more LDMA interrupts for LDMA channels 0 to 15.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @param[in] flags
 *   LDMA interrupt sources to disable. Use one or more valid
 *   interrupt flags for LDMA peripheral module. The flags are one LDMA_IEN_ERROR flag for
 *   all the channels and one LDMA_IEN_DONEx flag for each channel
 *   (x = a supported channel number).
 ******************************************************************************/
__INLINE void sl_hal_ldma_disable_interrupts(LDMA_TypeDef *ldma,
                                             uint32_t flags)
{
  ldma->IEN_CLR = flags;
}

/***************************************************************************//**
 * @brief
 *   Enable one or more LDMA interrupts for LDMA channels 0 to 15.
 *
 * @note
 *   Depending on the use, a pending interrupt may already be set before to
 *   enabling the interrupt. To ignore a pending interrupt, consider using
 *   sl_hal_ldma_clear_interrupts() prior to enabling the interrupt.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @param[in] flags
 *   LDMA interrupt sources to enable. Use one or more valid
 *   interrupt flags for LDMA peripheral module. The flags are one LDMA_IEN_ERROR flag for
 *   all channels and one LDMA_IEN_DONEx flag for each channel
 *   (x = a supported channel number).
 ******************************************************************************/
__INLINE void sl_hal_ldma_enable_interrupts(LDMA_TypeDef *ldma,
                                            uint32_t flags)
{
  ldma->IEN_SET = flags;
}

/***************************************************************************//**
 * @brief
 *   Get pending LDMA interrupt flags for LDMA channels 0 to 15.
 *
 * @note
 *   Event bits are not cleared by the use of this function.
 *   For series 2 devices, there is only one LDMA_IF_ERROR shared across all
 *   channels. Use the appropriate flags to clear these interrupts.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @return
 *   Pending LDMA interrupt sources. Returns one or more valid interrupt
 *   flags for the LDMA module. The flags are one LDMA_IF_ERRORx flag for
 *   each channel (x = a supported channel number) and one LDMA_IF_DONEx flag
 *   for each channel (x = a supported channel number). For series 2 devices,
 *   use LDMA_IF_ERROR and LDMA_IF_DONEx.
 ******************************************************************************/
__INLINE uint32_t sl_hal_ldma_get_pending_interrupts(LDMA_TypeDef *ldma)
{
  return ldma->IF;
}

/***************************************************************************//**
 * @brief
 *   Set one or more pending LDMA interrupts for LDMA channels 0 to 15.
 *
 * @note
 *   For series 2 devices, there is only one LDMA_IF_ERROR shared across all
 *   channels. Use the appropriate flags to clear these interrupts.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @param[in] flags
 *   LDMA interrupt sources to set to pending. Use one or more valid
 *   interrupt flags for LDMA peripheral module. The flags are one LDMA_IF_ERRORx flag for
 *   each channel (x = a supported channel number) and one LDMA_IF_DONEx flag
 *   for each channel (x = a supported channel number). For series 2 devices,
 *   use LDMA_IF_ERROR and LDMA_IF_DONEx.
 ******************************************************************************/
__INLINE void sl_hal_ldma_set_interrupts(LDMA_TypeDef *ldma,
                                         uint32_t flags)
{
  ldma->IF_SET = flags;
}

/***************************************************************************//**
 * @brief
 *   Get enabled and pending LDMA interrupt flags for LDMA channels 0 to 15.
 *   Useful for handling more interrupt sources in the same interrupt handler.
 *
 * @note
 *   Interrupt flags are not cleared by the use of this function.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @return
 *   Pending and enabled LDMA interrupt sources
 *   Return value is the bitwise AND of
 *   - the enabled interrupt sources in LDMA_IEN and
 *   - the pending interrupt flags LDMA_IF
 ******************************************************************************/
__INLINE uint32_t sl_hal_ldma_get_enabled_pending_interrupts(LDMA_TypeDef *ldma)
{
  uint32_t ien;

  ien = ldma->IEN;
  return ldma->IF & ien;
}

#if defined(_LDMA_IFH_MASK)
/***************************************************************************//**
 * @brief
 *   Clear one or more pending LDMA interrupts for the LDMA channels 16 to 31.
 *
 * @note
 *   For series 2 devices, there is only one LDMA_IF_ERROR shared across all
 *   channels. Use the appropriate flags to clear these interrupts.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @param[in] flags
 *   Pending LDMA interrupt sources to clear. Use one or more valid
 *   interrupt flags for the LDMA peripheral module. The flags are one LDMA_IF_ERRORx flag
 *   for each channel (x = a supported channel number) and one LDMA_IF_DONEx flag
 *   for each channel (x = a supported channel number).  For series 2 devices,
 *   use LDMA_IF_ERROR and LDMA_IF_DONEx.
 ******************************************************************************/
__INLINE void sl_hal_ldma_clear_high_interrupts(LDMA_TypeDef *ldma,
                                                uint32_t flags)
{
  ldma->IFH_CLR = flags;
}

/***************************************************************************//**
 * @brief
 *   Disable one or more LDMA interrupts for the LDMA channels 16 to 31.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @param[in] flags
 *   LDMA interrupt sources to disable. Use one or more valid
 *   interrupt flags for LDMA peripheral module. The flags are one LDMA_IEN_ERROR flag for
 *   all channels and one LDMA_IEN_DONEx flag for each channel
 *   (x = a supported channel number).
 ******************************************************************************/
__INLINE void sl_hal_ldma_disable_high_interrupts(LDMA_TypeDef *ldma,
                                                  uint32_t flags)
{
  ldma->IENH_CLR = flags;
}

/***************************************************************************//**
 * @brief
 *   Enable one or more LDMA interrupts for the LDMA channels 16 to 31.
 *
 * @note
 *   Depending on the use, a pending interrupt may already be set before to
 *   enabling the interrupt. To ignore a pending interrupt, consider using
 *   sl_hal_ldma_clear_interrupts() prior to enabling the interrupt.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @param[in] flags
 *   LDMA interrupt sources to enable. Use one or more valid
 *   interrupt flags for LDMA peripheral module. The flags are one LDMA_IEN_ERROR flag for
 *   all channels and one LDMA_IEN_DONEx flag for each channel
 *   (x = a supported channel number).
 ******************************************************************************/
__INLINE void sl_hal_ldma_enable_high_interrupts(LDMA_TypeDef *ldma,
                                                 uint32_t flags)
{
  ldma->IENH_SET = flags;
}

/***************************************************************************//**
 * @brief
 *   Get pending LDMA interrupt flags for the LDMA channels 16 to 31.
 *
 * @note
 *   Event bits are not cleared by the use of this function.
 *   For series 2 devices, there is only one LDMA_IF_ERROR shared across all
 *   channels. Use the appropriate flags to clear these interrupts.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @return
 *   Pending LDMA interrupt sources. Returns one or more valid interrupt
 *   flags for the LDMA module. The flags are one LDMA_IF_ERRORx flag for
 *   each channel (x = a supported channel number) and one LDMA_IF_DONEx flag
 *   for each channel (x = a supported channel number). For series 2 devices,
 *   use LDMA_IF_ERROR and LDMA_IF_DONEx.
 ******************************************************************************/
__INLINE uint32_t sl_hal_ldma_get_pending_high_interrupts(LDMA_TypeDef *ldma)
{
  return ldma->IFH;
}

/***************************************************************************//**
 * @brief
 *   Set one or more pending LDMA interrupts for the LDMA channels 16 to 31.
 *
 * @note
 *   For series 2 devices, there is only one LDMA_IF_ERROR shared across all
 *   channels. Use the appropriate flags to clear these interrupts.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @param[in] flags
 *   LDMA interrupt sources to set to pending. Use one or more valid
 *   interrupt flags for LDMA peripheral module. The flags are one LDMA_IF_ERRORx flag for
 *   each channel (x = a supported channel number) and one LDMA_IF_DONEx flag
 *   for each channel (x = a supported channel number). For series 2 devices,
 *   use LDMA_IF_ERROR and LDMA_IF_DONEx.
 ******************************************************************************/
__INLINE void sl_hal_ldma_set_high_interrupts(LDMA_TypeDef *ldma,
                                              uint32_t flags)
{
  ldma->IFH_SET = flags;
}

/***************************************************************************//**
 * @brief
 *   Get enabled and pending LDMA interrupt flags for the LDMA channels 16 to 31.
 *   Useful for handling more interrupt sources in the same interrupt handler.
 *
 * @note
 *   Interrupt flags are not cleared by the use of this function.
 *
 * @param[in] ldma
 *   A LDMA peripheral module.
 *
 * @return
 *   Pending and enabled LDMA interrupt sources
 *   Return value is the bitwise AND of
 *   - the enabled interrupt sources in LDMA_IEN and
 *   - the pending interrupt flags LDMA_IF
 ******************************************************************************/
__INLINE uint32_t sl_hal_ldma_get_enabled_pending_high_interrupts(LDMA_TypeDef *ldma)
{
  uint32_t ienh;

  ienh = ldma->IENH;
  return ldma->IFH & ienh;
}
#endif

/** @} (end addtogroup ldma) */

#ifdef __cplusplus
}
#endif

/****************************************************************************//**
 * @addtogroup ldma LDMA - Linked DMA
 * @{
 *
 * @li @ref ldma_intro
 * @li @ref ldma_example
 *
 *@n @section ldma_intro Introduction
 * LDMA API functions provide full support for the LDMA peripheral.
 *
 * LDMA supports these LDMA transfer types:
 *
 * @li Memory to memory.
 * @li Memory to peripheral.
 * @li Peripheral to memory.
 * @li Peripheral to peripheral.
 * @li Constant value to memory.
 *
 * LDMA supports linked lists of LDMA descriptors allowing:
 *
 * @li Circular and ping-pong buffer transfers.
 * @li Scatter-gather transfers.
 * @li Looped transfers.
 *
 * LDMA has some advanced features:
 *
 * @li Intra-channel synchronization (SYNC), allowing hardware events to
 *     pause and restart a LDMA sequence.
 * @li Immediate-write (WRI), allowing LDMA to write a constant anywhere
 *     in the memory map.
 * @li Complex flow control allowing if-else constructs.
 *
 * Basic understanding of LDMA controller is assumed. Please refer to
 * the reference manual for further details. The LDMA examples described
 * in the reference manual are particularly helpful in understanding LDMA
 * operations.
 *
 * In order to use the LDMA controller, the initialization function @ref
 * sl_hal_ldma_init() must have been executed once (normally during system initialization).
 *
 * LDMA transfers are initialized by a call to @ref sl_hal_ldma_init_transfer(),
 * transfer properties are controlled by the contents of @ref sl_hal_ldma_transfer_init_t
 * and @ref sl_hal_ldma_descriptor_t structure parameters. The LDMA transfers can then be initiated
 * by calling @ref sl_hal_ldma_start_transfer(). The @ref sl_hal_ldma_descriptor_t
 * structure parameter may be a pointer to an array of descriptors, descriptors in array should
 * be linked together as needed.
 *
 * Transfer and descriptor initialization macros are provided for the most common
 * transfer types. Due to the flexibility of LDMA peripheral, only a small
 * subset of all possible initializer macros are provided, users should create
 * new ones when needed.
 *
 *@n @section ldma_example Example
 *
 *  A simple memory to memory transfer:
 *  @code{.c}
 *  {
 *    // Enable bus clock.
 *    sl_clock_manager_enable_bus_clock(SL_BUS_CLOCK_LDMA0);
 *
 *    // Two buffers used in the memory-to-memory transfers.
 *    static volatile uint32_t src[4];
 *    static volatile uint32_t dst[4];
 *
 *    static const sl_hal_ldma_transfer_init_t transferCfg = SL_HAL_LDMA_TRANSFER_CFG_MEMORY();
 *    static const sl_hal_ldma_descriptor_t desc = SL_HAL_LDMA_DESCRIPTOR_SINGLE_M2M(SL_HAL_LDMA_CTRL_SIZE_WORD, src, dst, 4);
 *
 *    // Initialize the LDMA with default values.
 *    sl_hal_ldma_init_t initCfg = SL_HAL_LDMA_INIT_DEFAULT;
 *    sl_hal_ldma_init(LDMA0, &initCfg);
 *
 *    // Start the memory transfer.
 *    uint8_t channel = 0;
 *    sl_hal_ldma_init_transfer(LDMA0, channel, &transferCfg, &desc);
 *    sl_hal_ldma_start_transfer(LDMA0, channel);
 *
 *     while (!sl_hal_ldma_transfer_is_done(LDMA0, channel)) {
 *       // Wait until the transfer is over.
 *     }
 *  }
 *
 *  @endcode
 *
 *  A linked list of three memory-to-memory transfers:
 *  @code{.c}
 *  {
 *    // Enable bus clock.
 *    sl_clock_manager_enable_bus_clock(SL_BUS_CLOCK_LDMA0);
 *
 *    // Two buffers used in the memory-to-memory transfers.
 *    static volatile uint32_t src[12];
 *    static volatile uint32_t dst[12];
 *
 *    // A transfer consisting of 3 descriptors linked together and each descriptor
 *    // transfers 4 words from the source to the destination.
 *    static const sl_hal_ldma_transfer_init_t transferCfg = SL_HAL_LDMA_TRANSFER_CFG_MEMORY();
 *    static const sl_hal_ldma_descriptor_t descList[] =
 *    {
 *      SL_HAL_LDMA_DESCRIPTOR_LINKREL_M2M(SL_HAL_LDMA_CTRL_SIZE_WORD, src, dst, 4, 1),
 *      SL_HAL_LDMA_DESCRIPTOR_LINKREL_M2M(SL_HAL_LDMA_CTRL_SIZE_WORD, src + 4, dst + 4, 4, 1),
 *      SL_HAL_LDMA_DESCRIPTOR_SINGLE_M2M(SL_HAL_LDMA_CTRL_SIZE_WORD, src + 8, dst + 8, 4)
 *    };
 *
 *    // Initialize the LDMA with default values.
 *    sl_hal_ldma_init_t initCfg = SL_HAL_LDMA_INIT_DEFAULT;
 *    sl_hal_ldma_init(LDMA0, &initCfg);
 *
 *    // Start the linked memory transfer.
 *    uint8_t channel = 0;
 *    sl_hal_ldma_init_transfer(LDMA0, channel, &transferCfg, descList);
 *    sl_hal_ldma_start_transfer(LDMA0, channel);
 *
 *     while (!sl_hal_ldma_transfer_is_done(LDMA0, channel)) {
 *       // Wait until the transfer is over.
 *     }
 *  }
 *
 *  @endcode
 *
 *  LDMA from serial port peripheral to memory:
 *  @code{.c}
 *  {
 *    // Enable bus clock.
 *    sl_clock_manager_enable_bus_clock(SL_BUS_CLOCK_LDMA0);
 *
 *    // Two buffer used in the transfer from EUSART0.
 *    static volatile uint8_t dst[4];
 *
 *    // The LDMA transfer should be triggered by the EUSART0 RX data available signal.
 *    static const sl_hal_ldma_transfer_init_t eusart0RxTransfer =
 *    SL_HAL_LDMA_TRANSFER_CFG_PERIPHERAL(SL_HAL_LDMA_PERIPHERAL_SIGNAL_EUSART0_RXFL);
 *
 *    // Transfer 4 bytes from the EUSART0 RX FIFO to memory.
 *    static const sl_hal_ldma_descriptor_t eusart0RxDesc =
 *    SL_HAL_LDMA_DESCRIPTOR_SINGLE_P2M(SL_HAL_LDMA_CTRL_SIZE_BYTE,  // Transfer size
 *                                    &EUSART0->RXDATA,              // Peripheral address
 *                                    dst,                           // Destination (SRAM)
 *                                    4);                            // Number of byte transfers
 *
 *    // Initialize the LDMA with default values.
 *    sl_hal_ldma_init_t initCfg = SL_HAL_LDMA_INIT_DEFAULT;
 *    sl_hal_ldma_init(LDMA0, &initCfg);
 *
 *    // Start the peripheral transfer which is triggered by the EUSART0
 *    // peripheral RXFL signal.
 *    uint8_t channel = 0;
 *    sl_hal_ldma_init_transfer(LDMA0, channel, &eusart0RxTransfer, &eusart0RxDesc);
 *    sl_hal_ldma_start_transfer(LDMA0, channel);
 *
 *     while (!sl_hal_ldma_transfer_is_done(LDMA0, channel)) {
 *       // Wait until the transfer is over.
 *     }
 *  }
 *
 *  @endcode
 *
 *  Ping-pong LDMA from serial port peripheral to memory:
 *  @code{.c}
 *  {
 *    // Enable bus clock.
 *    sl_clock_manager_enable_bus_clock(SL_BUS_CLOCK_LDMA0);
 *
 *    // Two buffers used in the ping-pong transfer from EUSART0.
 *    static volatile uint8_t buffer0[5];
 *    static volatile uint8_t buffer1[5];
 *
 *    // The LDMA transfer should be triggered by the EUSART0 RX data available signal.
 *    static const sl_hal_ldma_transfer_init_t eusart0RxTransfer =
 *    SL_HAL_LDMA_TRANSFER_CFG_PERIPHERAL(SL_HAL_LDMA_PERIPHERAL_SIGNAL_EUSART0_RXFL);
 *
 *    // Both descriptors transfer 5 bytes from the EUSART0 RX data register into
 *    // one of the buffers. Note that the first descriptor uses a relative address
 *    // of 1 to link to the next descriptor, while the last descriptor uses a
 *    // relative address of -1 to link to the first descriptor.
 *    static const sl_hal_ldma_descriptor_t rxLoopDesc[] =
 *    {
 *      SL_HAL_LDMA_DESCRIPTOR_LINKREL_P2M(SL_HAL_LDMA_CTRL_SIZE_BYTE, &EUSART0->RXDATA, buffer0, 5, 1),
 *      SL_HAL_LDMA_DESCRIPTOR_LINKREL_P2M(SL_HAL_LDMA_CTRL_SIZE_BYTE, &EUSART0->RXDATA, buffer1, 5, -1)
 *    };
 *
 *    // Initialize the LDMA with default values.
 *    sl_hal_ldma_init_t initCfg = SL_HAL_LDMA_INIT_DEFAULT;
 *    sl_hal_ldma_init(LDMA0, &initCfg);
 *
 *    // Start the peripheral transfer which is triggered by the EUSART0
 *    // peripheral RXFL signal.
 *    uint8_t channel = 0;
 *    sl_hal_ldma_init_transfer(LDMA0, channel, &eusart0RxTransfer, rxLoopDesc);
 *    sl_hal_ldma_start_transfer(LDMA0, channel);
 *  }
 *
 *  @endcode
 *
 *  @note LDMA peripheral module does not implement LDMA interrupt handler. A
 *  template for an LDMA IRQ handler is included here as an example.
 *  For series 2, a single IRQ handler is shared by all channels.
 *  @code{.c}
 *  void LDMA_IRQHandler(void)
 *  {
 *    //Get all pending and enabled interrupts.
 *    uint32_t pending = sl_hal_ldma_get_enabled_pending_interrupts(LDMA);
 *
 *    while (pending & LDMA_IF_ERROR) {
 *      // Loop here on an LDMA error to enable debugging.
 *    }
 *
 *    // Iterate over all LDMA channels.
 *    for (uint32_t channel = 0; channel < DMA_CHAN_COUNT; channel++) {
 *      uint32_t mask = 0x1 << channel;
 *      if (pending & mask) {
 *        // Clear interrupt flag.
 *        sl_hal_ldma_clear_interrupts(LDMA, mask);
 *
 *        // Do more stuff here, execute callbacks, etc.
 *      }
 *    }
 *  }
 *
 *  @endcode
 *
 *  @note For series 3, every channel has its own IRQ handler.
 *  Here is an example handler for channel 0.
 *  @code{.c}
 *  void LDMA0_CHNL0_IRQHandler(void)
 *  {
 *    // Get all pending and enabled interrupts.
 *    uint32_t pending = sl_hal_ldma_get_enabled_pending_interrupts(LDMA0);
 *
 *    while (pending & LDMA_IF_ERROR0) {
 *      // Loop here on an LDMA error to enable debugging.
 *    }
 *
 *    if (pending & LDMA_IF_DONE0) {
 *      // Clear interrupt flag.
 *      sl_hal_ldma_clear_interrupts(LDMA0, LDMA_IF_DONE0);
 *
 *      // Do more stuff here, execute callbacks, etc.
 *    }
 *  }
 *
 *  @endcode
 *
 * @} (end addtogroup ldma)
 ******************************************************************************/
/* *INDENT-ON* */

#endif // defined( LDMA_PRESENT ) && ( LDMA_COUNT == 1 )
#endif // SL_HAL_LDMA_H
