from pyradioconfig.parts.nixi.calculators.calc_global import CALC_Global_nixi
from pycalcmodel.core.variable import ModelVariableFormat
from py_2_and_3_compatibility import *


class CALC_Global_panther(CALC_Global_nixi):

    def buildVariables(self, model):

        # Build variables from the Nixi calculations
        super().buildVariables(model)

        # Add additional variables for Panther

        # These were extracted from Panther register model. Only writeable registers that are new (do not exist in EFR90nm) are listed here.
        # Must have a corresponding ModelVariableContainer entry in pyradioconfig/parts/common/profiles/panther_regs.py build_modem_regs_panther.
        # MCUW_RADIO_CFG-734 Panther: PHY/MAC Design requests more registers added (all MODEM, AGC, etc.) with re-writes of reset values

        # AGC peripheral has 133 writeable fields in total

        # All Panther PHY cfg files show 133 AGC field writes

        # Except _EN_EN and _IEN_

        # Note that 39 Panther AGC fields were explicitly omitted as they exist on EFR90nm.
        # HOWEVER, those fields may not have been added to the Radio Configurator in the EFR90nm era.
        # Thus they will need to be manually added here:


        # An error such as
        # AttributeError: 'ModelVariableContainer' object has no attribute 'AGC_CTRL2_DMASEL'
        # indicates a line below needs to be uncommented (enabled)


        # AGC.CTRL0.PWRTARGET
        # AGC.CTRL0.MODE
        # AGC.CTRL0.RSSISHIFT
        # AGC.CTRL0.DISCFLOOPADJ
        # AGC.CTRL0.AGCCLKUNDIVREQ
        # AGC.CTRL0.DISRESETCHPWR
        # AGC.CTRL1.RSSIPERIOD
        # AGC.CTRL1.SUBPERIOD
        # AGC.CTRL1.SUBNUM
        # AGC.CTRL1.SUBDEN
        # AGC.CTRL1.SUBINT
        # AGC.CTRL2.DMASEL
        # AGC.RSSISTEPTHR.POSSTEPTHR
        # AGC.RSSISTEPTHR.NEGSTEPTHR
        # AGC.MANGAIN.MANGAINIFPGA
        # AGC.MANGAIN.MANGAINEN
        # AGC.IF.RSSIVALID
        # AGC.IF.CCA
        # AGC.IF.RSSIPOSSTEP
        # AGC.IF.RSSINEGSTEP
        # AGC.IF.RSSIDONE
        # AGC.IEN.RSSIVALID
        # AGC.IEN.CCA
        # AGC.IEN.RSSIPOSSTEP
        # AGC.IEN.RSSINEGSTEP
        # AGC.IEN.RSSIDONE
        # AGC.CMD.RSSISTART
        # AGC.GAINSTEPLIM.CFLOOPSTEPMAX
        # AGC.CTRL0.ENRSSIRESET
        # AGC.RSSISTEPTHR.STEPPER
        # AGC.RSSISTEPTHR.DEMODRESTARTPER
        # AGC.RSSISTEPTHR.DEMODRESTARTTHR
        # AGC.LBT.CCARSSIPERIOD
        # AGC.LBT.ENCCARSSIPERIOD
        # AGC.LBT.ENCCAGAINREDUCED
        # AGC.LBT.ENCCARSSIMAX
        # AGC.IF.SHORTRSSIPOSSTEP
        # AGC.IEN.SHORTRSSIPOSSTEP
        # AGC.MIRRORIF.IFMIRRORCLEAR

        #Add each register using a separate method
        #This is best practice moving forward so that we can handle inheritance granularly
        self._add_RAC_PGACTRL(model)
        self._add_SYNTH_DSMCTRL_regs(model)
        self._add_SYNTH_LPFCTRL_regs(model)
        self._add_SYNTH_DENOMINIT_regs(model)
        self._add_RAC_SYTRIM0_SYCHPREPLICACURRADJ_regs(model)
        self._add_RAC_SYNTHCTRL_MMDPOWERBALANCEDISABLE_regs(model)
        self._add_SYTRIM_regs(model)
        self._add_RAC_CLKMULTEN0_en(model)
        self._add_lpfincap_regs(model)


    def _add_common_regs(self, model):
        super()._add_common_regs(model)
        self._addModelRegister(model, 'AGC.STATUS0.ADCINDEX', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'AGC.CTRL0.PWRTARGET', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'AGC.CTRL0.MODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL0.DISCFLOOPADJ', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL0.AGCCLKUNDIVREQ', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL0.DISRESETCHPWR', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL0.ADCATTENMODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL0.ADCATTENCODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL0.ENRSSIRESET', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL0.DSADISCFLOOP', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL0.DISPNGAINUP', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL0.DISPNDWNCOMP', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL0.AGCRST', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'AGC.CTRL1.RSSIPERIOD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL1.PWRPERIOD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'AGC.CTRL1.SUBPERIOD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'AGC.CTRL1.SUBNUM', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'AGC.CTRL1.SUBDEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'AGC.CTRL1.SUBINT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL2.DMASEL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL2.SAFEMODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL2.SAFEMODETHD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL2.REHICNTTHD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL2.RELOTHD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL2.RELBYCHPWR', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL2.RELTARGETPWR', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL2.PRSDEBUGEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL2.DISRFPKD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL3.IFPKDDEB', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL3.IFPKDDEBTHD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL3.IFPKDDEBPRD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL3.IFPKDDEBRST', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL3.RFPKDDEB', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL3.RFPKDDEBTHD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL3.RFPKDDEBPRD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.CTRL3.RFPKDDEBRST', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'AGC.RSSISTEPTHR.POSSTEPTHR', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'AGC.RSSISTEPTHR.NEGSTEPTHR', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'AGC.RSSISTEPTHR.STEPPER', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'AGC.RSSISTEPTHR.DEMODRESTARTPER', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'AGC.RSSISTEPTHR.DEMODRESTARTTHR', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.RSSISTEPTHR.RSSIFAST', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.MANGAIN.MANGAINEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.MANGAIN.MANGAINIFPGA', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.MANGAIN.MANGAINLNA', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.MANGAIN.MANGAINPN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.MANGAIN.MANRFLATRST', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.MANGAIN.MANIFLOLATRST', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.MANGAIN.MANIFHILATRST', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'AGC.CMD.RSSISTART', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.GAINRANGE.LNAINDEXBORDER', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.GAINRANGE.PGAINDEXBORDER', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.GAINRANGE.GAININCSTEP', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.GAINRANGE.PNGAINSTEP', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.GAINRANGE.LATCHEDHISTEP', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.GAINRANGE.HIPWRTHD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.GAINRANGE.BOOSTLNA', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.GAINRANGE.LNABWADJ', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.AGCPERIOD.PERIODHI', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.AGCPERIOD.PERIODLO', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.AGCPERIOD.MAXHICNTTHD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.AGCPERIOD.SETTLETIMEIF', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.AGCPERIOD.SETTLETIMERF', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.HICNTREGION.HICNTREGION0', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.HICNTREGION.HICNTREGION1', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.HICNTREGION.HICNTREGION2', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.HICNTREGION.HICNTREGION3', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.HICNTREGION.HICNTREGION4', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.STEPDWN.STEPDWN0', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.STEPDWN.STEPDWN1', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.STEPDWN.STEPDWN2', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.STEPDWN.STEPDWN3', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.STEPDWN.STEPDWN4', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.STEPDWN.STEPDWN5', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'AGC.GAINSTEPLIM.CFLOOPSTEPMAX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.GAINSTEPLIM.CFLOOPDEL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.GAINSTEPLIM.HYST', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.GAINSTEPLIM.MAXPWRVAR', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.GAINSTEPLIM.TRANRSTAGC', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PNRFATT0.LNAMIXRFATT1', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PNRFATT0.LNAMIXRFATT2', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PNRFATT0.LNAMIXRFATT3', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PNRFATT0.LNAMIXRFATT4', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PNRFATT0.LNAMIXRFATT5', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PNRFATT1.LNAMIXRFATT6', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PNRFATT1.LNAMIXRFATT7', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PNRFATT1.LNAMIXRFATT8', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PNRFATT1.LNAMIXRFATT9', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PNRFATT1.LNAMIXRFATT10', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PNRFATT2.LNAMIXRFATT11', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PNRFATT2.LNAMIXRFATT12', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PNRFATT2.LNAMIXRFATT13', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PNRFATT2.LNAMIXRFATT14', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.LNAMIXCODE0.LNAMIXSLICE1', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.LNAMIXCODE0.LNAMIXSLICE2', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.LNAMIXCODE0.LNAMIXSLICE3', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.LNAMIXCODE0.LNAMIXSLICE4', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.LNAMIXCODE0.LNAMIXSLICE5', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.LNAMIXCODE1.LNAMIXSLICE6', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.LNAMIXCODE1.LNAMIXSLICE7', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.LNAMIXCODE1.LNAMIXSLICE8', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.LNAMIXCODE1.LNAMIXSLICE9', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.LNAMIXCODE1.LNAMIXSLICE10', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PGACODE0.PGAGAIN1', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PGACODE0.PGAGAIN2', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PGACODE0.PGAGAIN3', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PGACODE0.PGAGAIN4', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PGACODE0.PGAGAIN5', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PGACODE0.PGAGAIN6', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PGACODE0.PGAGAIN7', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PGACODE0.PGAGAIN8', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PGACODE1.PGAGAIN9', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PGACODE1.PGAGAIN10', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.PGACODE1.PGAGAIN11', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.LBT.CCARSSIPERIOD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.LBT.ENCCARSSIPERIOD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.LBT.ENCCAGAINREDUCED', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'AGC.LBT.ENCCARSSIMAX', int, ModelVariableFormat.HEX)





        # Now defiend in pyradioconfig/parts/panther/calculators/calc_crc.py
        # self._addModelRegister(model, 'RFCRC.CTRL.BITREVERSE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RFCRC.CTRL.BITSPERWORD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RFCRC.CTRL.BYTEREVERSE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RFCRC.CTRL.CRCWIDTH', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RFCRC.CTRL.INPUTBITORDER', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RFCRC.CTRL.INPUTINV', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RFCRC.CTRL.OUTPUTINV', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RFCRC.CTRL.PADCRCINPUT', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RFCRC.EN.EN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RFCRC.INIT.INIT', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RFCRC.POLY.POLY', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.ADDRFILTCTRL.ADDRESS', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.ADDRFILTCTRL.BRDCST00EN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.ADDRFILTCTRL.BRDCSTFFEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.ADDRFILTCTRL.EN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.AUTOCG.AUTOCGEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.BLOCKRAMADDR.BLOCKRAMADDR', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.BUFFERMODE.RXBUFFERMODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.BUFFERMODE.RXFRCBUFMUX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.BUFFERMODE.TXBUFFERMODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.CGCLKSTOP.FORCEOFF', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.CONVGENERATOR.GENERATOR0', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.CONVGENERATOR.GENERATOR1', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.CONVGENERATOR.NONSYSTEMATIC', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.CONVGENERATOR.RECURSIVE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.CONVRAMADDR.CONVRAMADDR', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.CTRL.BITORDER', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.CTRL.BITSPERWORD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.CTRL.PRBSTEST', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.CTRL.RANDOMTX', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.CTRL.RATESELECT', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.CTRL.RXFCDMODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.CTRL.SEQHANDSHAKE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.CTRL.TXFCDMODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.CTRL.TXPREFETCH', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.CTRL.UARTMODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.DATABUFFER.DATABUFFER', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.DFLCTRL.DFLBITORDER', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.DFLCTRL.DFLBITS', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.DFLCTRL.DFLINCLUDECRC', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.DFLCTRL.DFLMODE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.DFLCTRL.DFLOFFSET', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.DFLCTRL.DFLSHIFT', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.DFLCTRL.MINLENGTH', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.EN.EN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD0.ADDTRAILTXDATA', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD0.BUFFER', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD0.CALCCRC', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD0.EXCLUDESUBFRAMEWCNT', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD0.INCLUDECRC', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD0.SKIPCRC', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD0.SKIPWHITE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD0.WORDS', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD1.ADDTRAILTXDATA', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD1.BUFFER', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD1.CALCCRC', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD1.EXCLUDESUBFRAMEWCNT', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD1.INCLUDECRC', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD1.SKIPCRC', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD1.SKIPWHITE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD1.WORDS', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD2.ADDTRAILTXDATA', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD2.BUFFER', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD2.CALCCRC', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD2.EXCLUDESUBFRAMEWCNT', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD2.INCLUDECRC', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD2.SKIPCRC', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD2.SKIPWHITE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD2.WORDS', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD3.ADDTRAILTXDATA', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD3.BUFFER', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD3.CALCCRC', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD3.EXCLUDESUBFRAMEWCNT', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD3.INCLUDECRC', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD3.SKIPCRC', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD3.SKIPWHITE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FCD3.WORDS', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FECCTRL.BLOCKWHITEMODE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FECCTRL.CONVBUSLOCK', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FECCTRL.CONVDECODEMODE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FECCTRL.CONVHARDERROR', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FECCTRL.CONVINV', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FECCTRL.CONVMODE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FECCTRL.CONVSUBFRAMETERMINATE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FECCTRL.CONVTRACEBACKDISABLE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FECCTRL.FORCE2FSK', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FECCTRL.INTERLEAVEFIRSTINDEX', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FECCTRL.INTERLEAVEMODE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FECCTRL.INTERLEAVEWIDTH', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.FECCTRL.SINGLEBLOCK', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.INTREADPOINT.INTREADPOINT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.INTWRITEPOINT.INTWRITEPOINT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.MAXLENGTH.INILENGTH', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.MAXLENGTH.MAXLENGTH', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.PAUSECTRL.CONVPAUSECNT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.PAUSECTRL.FRAMEDETPAUSEEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.PAUSECTRL.INTERLEAVEREADPAUSECNT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.PAUSECTRL.INTERLEAVEREADPAUSEEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.PAUSECTRL.INTERLEAVEWRITEPAUSECNT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.PAUSECTRL.RXINTERLEAVEWRITEPAUSEEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.PAUSECTRL.TXINTERLEAVEWRITEPAUSEEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.PAUSECTRL.TXSUBFRAMEPAUSEEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.PUNCTCTRL.PUNCT0', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.PUNCTCTRL.PUNCT1', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.RAMBASEADDR.RAMBASEADDR', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.RAWCTRL.DEMODRAWDATAMUX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.RAWCTRL.RXRAWMODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.RAWCTRL.RXRAWRANDOM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.RAWCTRL.RXRAWTRIGGER', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.RAWCTRL.TXRAWMODE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.RXCTRL.ACCEPTBLOCKERRORS', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.RXCTRL.ACCEPTCRCERRORS', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.RXCTRL.BUFCLEAR', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.RXCTRL.BUFRESTOREFRAMEERROR', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.RXCTRL.BUFRESTORERXABORTED', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.RXCTRL.RXFRAMEENDAHEADBYTES', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.RXCTRL.STORECRC', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.RXCTRL.TRACKABFRAME', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.SNIFFCTRL.SNIFFAUXDATA', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.SNIFFCTRL.SNIFFBITS', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.SNIFFCTRL.SNIFFBR', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.SNIFFCTRL.SNIFFMODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.SNIFFCTRL.SNIFFRSSI', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.SNIFFCTRL.SNIFFRXDATA', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.SNIFFCTRL.SNIFFSLEEPCTRL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.SNIFFCTRL.SNIFFSTATE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.SNIFFCTRL.SNIFFSYNCWORD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.SNIFFCTRL.SNIFFTXDATA', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.TRAILRXDATA.CRCOK', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.TRAILRXDATA.PROTIMERCC0BASE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.TRAILRXDATA.PROTIMERCC0WRAPH', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.TRAILRXDATA.PROTIMERCC0WRAPL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.TRAILRXDATA.RSSI', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.TRAILRXDATA.RTCSTAMP', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.TRAILTXDATACTRL.TRAILTXDATA', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.TRAILTXDATACTRL.TRAILTXDATACNT', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.TRAILTXDATACTRL.TRAILTXDATAFORCE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.WCNTCMP0.FRAMELENGTH', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.WCNTCMP1.LENGTHFIELDLOC', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.WCNTCMP2.ADDRFIELDLOC', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'FRC.WHITECTRL.BLOCKERRORCORRECT', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.WHITECTRL.FEEDBACKSEL', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.WHITECTRL.SHROUTPUTSEL', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.WHITECTRL.XORFEEDBACK', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.WHITEINIT.WHITEINIT', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'FRC.WHITEPOLY.POLY', int, ModelVariableFormat.HEX)


        # MODEM peripheral has 430 writeable fields in total

        # Previously, all Panther PHY cfg files showed 187 MODEM field writes
        # Now all Panther PHY cfg files show 430 MODEM field writes

        # Note that _addModelRegister CANNOT be called more than once for the same register.
        # So many fields are commented out below because they are already added to _addModelRegister
        # by Jumbo, Dumbo, Nerio, or Nixi

        # This can be found by grepping *.py for
        # .*addModelRegister\(model,.*MODEM
        # (first delete all allModelRegister that are Panther-specific in this file (and elsewhere)
        # Then filter that list OUT of the master Panther list....

        # @ Windows command prompt with Cygwin grep:

        # C:\validation\host_py_radio_config\Package\pyradioconfig > cd
        # C:\validation\host_py_radio_config\Package\pyradioconfig
        #
        # C:\validation\host_py_radio_config\Package\pyradioconfig>C:\cygwin64\bin\grep.exe  -rnw C:\validation\host_py_radio_config\Package\pyradioconfig -e .*addModelRegister\(model,.*MODEM > C:\temp\existing_add_model_MODEM.txt


        self._addModelRegister(model, 'MODEM.EN.EN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.MIXCTRL.ANAMIXMODE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.MIXCTRL.DIGIQSWAPEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL0.FDM0DIFFDIS', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL0.MAPFSK', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL0.CODING', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL0.MODFORMAT', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL0.DUALCORROPTDIS', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.CTRL0.OOKASYNCPIN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL0.DSSSLEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL0.DSSSSHIFTS', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL0.DSSSDOUBLE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.CTRL0.DETDIS', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL0.DIFFENCMODE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL0.SHAPING', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.CTRL0.DEMODRAWDATASEL', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL0.FRAMEDETDEL', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL1.SYNCBITS', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL1.SYNCERRORS', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL1.DUALSYNC', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.CTRL1.TXSYNC', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL1.SYNCDATA', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL1.SYNC1INV', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL1.COMPMODE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL1.RESYNCPER', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL1.PHASEDEMOD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL1.FREQOFFESTPER', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL1.FREQOFFESTLIM', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL2.SQITHRESH', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.CTRL2.RXFRCDIS', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL2.RXPINMODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.CTRL2.TXPINMODE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL2.DATAFILTER', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL2.BRDIVA', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL2.BRDIVB', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.CTRL2.DEVMULA', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.CTRL2.DEVMULB', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL2.RATESELMODE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL2.DEVWEIGHTDIS', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.CTRL2.DMASEL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.CTRL3.PRSDINEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.CTRL3.RAMTESTEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL3.TSAMPMODE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL3.TSAMPDEL', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL3.TSAMPLIM', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL4.ISICOMP', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL4.DEVOFFCOMP', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL4.PREDISTGAIN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL4.PREDISTDEB', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL4.PREDISTAVG', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL4.PREDISTRST', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.CTRL4.PHASECLICKFILT', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL4.SOFTDSSSMODE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL4.ADCSATLEVEL', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL4.ADCSATDENS', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL4.OFFSETPHASEMASKING', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.CTRL4.OFFSETPHASESCALING', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.CTRL4.CLKUNDIVREQ', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL5.BRCALEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL5.BRCALMODE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL5.BRCALAVG', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL5.DETDEL', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL5.TDEDGE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL5.TREDGE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL5.DSSSCTD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL5.BBSS', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL5.POEPER', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.CTRL5.DEMODRAWDATASEL2', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL5.FOEPREAVG', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL5.LINCORR', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL5.PRSDEBUG', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL5.RESYNCBAUDTRANS', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL5.RESYNCLIMIT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.CTRL5.XINGDEBUG', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL6.TDREW', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL6.PREBASES', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL6.PSTIMABORT0', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL6.PSTIMABORT1', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL6.PSTIMABORT2', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL6.PSTIMABORT3', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL6.ARW', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL6.TIMTHRESHGAIN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL6.CPLXCORREN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL6.DSSS3SYMBOLSYNCEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL6.CODINGB', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.CTRL6.RXRESTARTUPONRSSI', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.CTRL6.RXRESTARTUPONSHORTRSSI', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL6.RXBRCALCDIS', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CTRL6.DEMODRESTARTALL', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.TXBR.TXBRNUM', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.TXBR.TXBRDEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.RXBR.RXBRNUM', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.RXBR.RXBRDEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.RXBR.RXBRINT', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CF.DEC0', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CF.DEC1', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CF.DEC2', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CF.CFOSR', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CF.DEC1GAIN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.CF.RESYNCRESETTIMING', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.CF.RESYNCBYPASS', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.CF.ADCBITORDERI', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.CF.ADCBITORDERQ', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.PRE.BASE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.PRE.BASEBITS', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.PRE.PRESYMB4FSK', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.PRE.PREERRORS', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.PRE.DSSSPRE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.PRE.SYNCSYMB4FSK', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.PRE.TXBASES', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SYNC0.SYNC0', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SYNC1.SYNC1', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.TIMING.TIMTHRESH', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.TIMING.TIMINGBASES', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.TIMING.ADDTIMSEQ', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.TIMING.TIMSEQINVEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.TIMING.TIMSEQSYNC', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.TIMING.FDM0THRESH', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.TIMING.OFFSUBNUM', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.TIMING.OFFSUBDEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.TIMING.TSAGCDEL', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.TIMING.FASTRESYNC', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSSS0.DSSS0', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.MODINDEX.MODINDEXM', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.MODINDEX.MODINDEXE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.MODINDEX.FREQGAINE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.MODINDEX.FREQGAINM', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.AFC.AFCSCALEM', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.AFC.AFCSCALEE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.AFC.AFCRXMODE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.AFC.AFCTXMODE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.AFC.AFCRXCLR', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.AFC.AFCDEL', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.AFC.AFCAVGPER', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.AFC.AFCLIMRESET', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.AFC.AFCONESHOT', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.AFC.AFCENINTCOMP', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.AFC.AFCDSAFREQOFFEST', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.AFC.AFCDELDET', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.AFCADJLIM.AFCADJLIM', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING0.COEFF0', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING0.COEFF1', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING0.COEFF2', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING0.COEFF3', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING1.COEFF4', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING1.COEFF5', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING1.COEFF6', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING1.COEFF7', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING2.COEFF8', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING2.COEFF9', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING2.COEFF10', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING2.COEFF11', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING3.COEFF12', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING3.COEFF13', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING3.COEFF14', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING3.COEFF15', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING4.COEFF16', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING4.COEFF17', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING4.COEFF18', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING4.COEFF19', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING4.COEFF20', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING4.COEFF21', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING5.COEFF22', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING5.COEFF23', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING5.COEFF24', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING5.COEFF25', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING5.COEFF26', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING5.COEFF27', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING5.COEFF28', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING5.COEFF29', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING6.COEFF30', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING6.COEFF31', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING6.COEFF32', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING6.COEFF33', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING6.COEFF34', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING6.COEFF35', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING6.COEFF36', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING6.COEFF37', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING6.COEFF38', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SHAPING6.COEFF39', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DCCOMP.DCESTIEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DCCOMP.DCCOMPEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DCCOMP.DCRSTEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DCCOMP.DCCOMPFREEZE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DCCOMP.DCCOMPGEAR', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DCCOMP.DCLIMIT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DCCOMPFILTINIT.DCCOMPINITVALI', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DCCOMPFILTINIT.DCCOMPINITVALQ', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DCCOMPFILTINIT.DCCOMPINIT', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SRCCHF.SRCRATIO1', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SRCCHF.SRCENABLE1', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SRCCHF.SRCRATIO2', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SRCCHF.SRCENABLE2', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SRCCHF.BWSEL', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.SRCCHF.INTOSR', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.INTAFC.FOEPREAVG0', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.INTAFC.FOEPREAVG1', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.INTAFC.FOEPREAVG2', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.INTAFC.FOEPREAVG3', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.INTAFC.FOEPREAVG4', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.INTAFC.FOEPREAVG5', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.INTAFC.FOEPREAVG6', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.INTAFC.FOEPREAVG7', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSATHD0.SPIKETHD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSATHD0.UNMODTHD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSATHD0.FDEVMINTHD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSATHD0.FDEVMAXTHD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSATHD1.POWABSTHD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSATHD1.POWRELTHD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSATHD1.DSARSTCNT', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSATHD1.RSSIJMPTHD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSATHD1.FREQLATDLY', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSATHD1.PWRFLTBYP', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSATHD1.AMPFLTBYP', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSATHD1.PWRDETDIS', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSATHD1.FREQSCALE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSACTRL.DSAMODE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSACTRL.ARRTHD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSACTRL.ARRTOLERTHD0', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSACTRL.ARRTOLERTHD1', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSACTRL.SCHPRD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSACTRL.FREQAVGSYM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DSACTRL.TRANRSTDSA', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSACTRL.DSARSTON', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSACTRL.GAINREDUCDLY', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSACTRL.LOWDUTY', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSACTRL.RESTORE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSACTRL.AGCBAUDEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSACTRL.AMPJUPTHD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.VITERBIDEMOD.VTDEMODEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.VITERBIDEMOD.HARDDECISION', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.VITERBIDEMOD.VITERBIKSI1', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.VITERBIDEMOD.VITERBIKSI2', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.VITERBIDEMOD.VITERBIKSI3', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.VITERBIDEMOD.SYNTHAFC', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.VITERBIDEMOD.CORRCYCLE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.VITERBIDEMOD.CORRSTPSIZE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.VITERBIDEMOD.DISDEMODOF', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.VTCORRCFG0.EXPECTPATT', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DIGMIXCTRL.DIGMIXFREQ', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DIGMIXCTRL.DIGMIXMODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DIGMIXCTRL.MIXERCONJ', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.VTCORRCFG1.CORRSHFTLEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.VTCORRCFG1.VTFRQLIM', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.VTCORRCFG1.EXPSYNCLEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.VTCORRCFG1.BUFFHEAD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.VTCORRCFG1.EXPECTHT', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.VTTRACK.FREQTRACKMODE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.VTTRACK.TIMTRACKTHD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.VTTRACK.TIMEACQUTHD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.VTTRACK.TIMCHK', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.VTTRACK.TIMEOUTMODE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.VTTRACK.TIMGEAR', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.VTTRACK.FREQBIAS', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.VTTRACK.HIPWRTHD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.AUTOCG.AUTOCGEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CGCLKSTOP.FORCEOFF', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSATHD2.POWABSTHDLOG', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSATHD2.JUMPDETEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSATHD2.FDADJTHD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSATHD2.PMDETPASSTHD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSATHD2.FREQESTTHD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSATHD2.INTERFERDET', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.DSATHD2.PMDETFORCE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DIRECTMODE.DMENABLE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DIRECTMODE.SYNCASYNC', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DIRECTMODE.SYNCPREAM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DIRECTMODE.CLKWIDTH', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE.LRCORRTHD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE.LRCORRSCHWIN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE.LRBLE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE.LRTIMCORRTHD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE.LRBLEDSA', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE.LRDEC', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.LONGRANGE.LRCORRTHDDYNEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE1.LRSS', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE1.LRTIMEOUTTHD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE1.CHPWRACCUDEL', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE1.HYSVAL', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE1.AVGWIN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE1.LRSPIKETHADD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.LONGRANGE1.LOGICBASEDPUGATE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.LONGRANGE1.LOGICBASEDLRDEMODGATE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE2.LRCHPWRTH1', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE2.LRCHPWRTH2', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE2.LRCHPWRTH3', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE2.LRCHPWRTH4', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE3.LRCHPWRTH5', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE3.LRCHPWRTH6', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE3.LRCHPWRTH7', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE3.LRCHPWRTH8', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE4.LRCHPWRTH9', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE4.LRCHPWRTH10', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE4.LRCHPWRSH1', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE4.LRCHPWRSH2', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE4.LRCHPWRSH3', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE4.LRCHPWRSH4', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE5.LRCHPWRSH5', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE5.LRCHPWRSH6', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE5.LRCHPWRSH7', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE5.LRCHPWRSH8', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE5.LRCHPWRSH9', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE5.LRCHPWRSH10', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE5.LRCHPWRSH11', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE6.LRCHPWRSPIKETH', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LONGRANGE6.LRSPIKETHD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.LONGRANGE6.LRCHPWRTH11', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.LONGRANGE6.LRCHPWRSH12', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LRFRC.CI500', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.LRFRC.FRCACKTIMETHD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH0.COHDYNAMICBBSSEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH0.COHDYNAMICSYNCTHRESH', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.COH0.COHDYNAMICPRETHRESH', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.COH0.COHDYNAMICPRETHRESHSEL', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH0.COHCHPWRTH0', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH0.COHCHPWRTH1', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH0.COHCHPWRTH2', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH1.SYNCTHRESH0', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH1.SYNCTHRESH1', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH1.SYNCTHRESH2', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH1.SYNCTHRESH3', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH2.SYNCTHRESHDELTA0', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH2.SYNCTHRESHDELTA1', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH2.SYNCTHRESHDELTA2', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH2.SYNCTHRESHDELTA3', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH2.DSAPEAKCHPWRTH', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH2.FIXEDCDTHFORIIR', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH3.COHDSAEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH3.COHDSAADDWNDSIZE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH3.CDSS', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH3.DSAPEAKCHKEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH3.DSAPEAKINDLEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH3.DSAPEAKCHPWREN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH3.LOGICBASEDCOHDEMODGATE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH3.DYNIIRCOEFOPTION', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH3.ONEPEAKQUALEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH3.PEAKCHKTIMOUT', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.COH3.COHDSADETDIS', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DSATHD3.SPIKETHDLO', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DSATHD3.UNMODTHDLO', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DSATHD3.FDEVMINTHDLO', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DSATHD3.FDEVMAXTHDLO', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DSATHD4.POWABSTHDLO', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DSATHD4.ARRTOLERTHD0LO', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DSATHD4.ARRTOLERTHD1LO', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DSATHD4.SWTHD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.VTBLETIMING.VTBLETIMINGSEL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.VTBLETIMING.TIMINGDELAY', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.VTBLETIMING.FLENOFF', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.IEN.TXFRAMESENT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.IEN.TXSYNCSENT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.IEN.TXPRESENT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.IEN.RXTIMDET', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.IEN.RXPREDET', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.IEN.RXFRAMEDET0', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.IEN.RXFRAMEDET1', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.IEN.RXTIMLOST', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.IEN.RXPRELOST', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.IEN.RXFRAMEDETOF', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.IEN.RXTIMNF', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.IEN.FRCTIMOUT', int, ModelVariableFormat.HEX)
        # MSW: this is write-only... do not touch it!
        # self._addModelRegister(model, 'MODEM.CMD.PRESTOP', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CMD.AFCTXLOCK', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CMD.AFCTXCLEAR', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'MODEM.CMD.AFCRXCLEAR', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.BLEIQDSA.BLEIQDSAEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.BLEIQDSA.BLEIQDSATH', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.BLEIQDSA.BLEIQDSAIIRCOEFPWR', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.BLEIQDSA.BLEIQDSADIFFTH1', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.BLEIQDSAEXT1.FREQSCALEIQDSA', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.BLEIQDSAEXT1.CHPWRFIRAVGEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.BLEIQDSAEXT1.CHPWRFIRAVGVAL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.BLEIQDSAEXT1.CORRIIRAVGMULFACT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.BLEIQDSAEXT1.BLEIQDSAADDRBIAS', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.BLEIQDSAEXT1.BLEIQDSATHCOMB', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.BLEIQDSAEXT1.MAXCORRCNTIQDSA', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.BLEIQDSAEXT1.IIRRST', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.SYNCPROPERTIES.SYNCCORRCLR', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.SYNCPROPERTIES.SYNCSECPEAKTH', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DIGIGAINCTRL.DIGIGAINEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DIGIGAINCTRL.DIGIGAINSEL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DIGIGAINCTRL.DIGIGAINDOUBLE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DIGIGAINCTRL.DIGIGAINHALF', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.DIGIGAINCTRL.DEC0GAIN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.PADEBUG.MANPAAMPCTRL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.PADEBUG.ENMANPAAMPCTRL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.PADEBUG.MANPACLKAMPCTRL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.PADEBUG.ENMANPACLKAMPCTRL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.PADEBUG.ENMANPAPOWER', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.PADEBUG.ENMANPASELSLICE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'MODEM.PADEBUG.ENMANPATRIMPASLICE0DBM', int, ModelVariableFormat.HEX)






        # Panther_modem_field_names = [

        # 
        # ]
        # 
        # for modem_field_name in Panther_modem_field_names:
        #     try:
        #         self._addModelRegister(model, modem_field_name, int, ModelVariableFormat.HEX)
        #     except:
        #         pass




        self._addModelRegister(model, 'RAC.ANTDIV.EN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.APC.AMPCONTROLLIMITSW', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.APC.ENAPCSW', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCCTRL0.CLRCOUNTER', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCCTRL0.CLRFILTER', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCCTRL0.CYCLES', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCCTRL0.INPUTRESSEL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCCTRL0.MUXSEL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCCTRL1.AUXADCINPUTRESSEL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCCTRL1.AUXADCINPUTSELECT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCCTRL1.AUXADCPMONSELECT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCCTRL1.AUXADCRESET', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCCTRL1.AUXADCTSENSESELCURR', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCCTRL1.AUXADCTSENSESELVBE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCEN.AUXADCENAUXADC', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCEN.AUXADCENINPUTBUFFER', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCEN.AUXADCENLDO', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCEN.AUXADCENOUTPUTDRV', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCEN.AUXADCENPMON', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCEN.AUXADCENRESONDIAGA', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCEN.AUXADCENTSENSE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCEN.AUXADCENTSENSECAL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCEN.AUXADCINPUTBUFFERBYPASS', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCTRIM.AUXADCCLKINVERT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCTRIM.AUXADCLDOVREFTRIM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCTRIM.AUXADCOUTPUTINVERT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCTRIM.AUXADCRCTUNE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCTRIM.AUXADCTRIMADCINPUTRES', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCTRIM.AUXADCTRIMCURRINPUTBUF', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCTRIM.AUXADCTRIMCURROPA1', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCTRIM.AUXADCTRIMCURROPA2', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCTRIM.AUXADCTRIMCURRREFBUF', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCTRIM.AUXADCTRIMCURRTSENSE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCTRIM.AUXADCTRIMCURRVCMBUF', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCTRIM.AUXADCTRIMLDOHIGHCURRENT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCTRIM.AUXADCTRIMREFP', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCTRIM.AUXADCTRIMVREFVCM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.AUXADCTRIM.AUXADCTSENSETRIMVBE2', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CHPCTRL.CHPOUTPUTTRISTATEMODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTCTRL.CLKMULTDIVN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTCTRL.CLKMULTDIVR', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTCTRL.CLKMULTDIVX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTCTRL.CLKMULTENRESYNC', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTCTRL.CLKMULTVALID', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN0.CLKMULTBWCAL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN0.CLKMULTENDITHER', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN0.CLKMULTENDRVDIFF', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN0.CLKMULTENDRVRX2P4G', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN0.CLKMULTENDRVRXSUBG', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN0.CLKMULTENDRVTXDUALB', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN0.CLKMULTFREQCAL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN0.CLKMULTREG1ADJV', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN0.CLKMULTREG2ADJI', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN0.CLKMULTREG2ADJV', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN1.CLKMULTINNIBBLE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN1.CLKMULTLDCNIB', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN1.CLKMULTLDFNIB', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN1.CLKMULTLDMNIB', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN1.CLKMULTRDNIBBLE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CTRL.ACTIVEPOL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CTRL.FORCEDISABLE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CTRL.LNAENPOL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CTRL.PAENPOL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CTRL.PRSCLR', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CTRL.PRSFORCETX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CTRL.PRSMODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CTRL.PRSRXDIS', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CTRL.PRSTXEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CTRL.TXAFTERRX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CTRL.TXPOSTPONE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.DIGCLKRETIMECTRL.DIGCLKRETIMEDISRETIME', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.DIGCLKRETIMECTRL.DIGCLKRETIMEENRETIME', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.DIGCLKRETIMECTRL.DIGCLKRETIMELIMITH', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.DIGCLKRETIMECTRL.DIGCLKRETIMELIMITL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.DIGCLKRETIMECTRL.DIGCLKRETIMERESETN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.EN.EN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.FORCESTATE.FORCESTATE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFADCCAL.IFADCENRCCAL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFADCCAL.IFADCTUNERC', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFADCCAL.IFADCTUNERCCALMODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFADCDBG.IFADCENADC', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFADCDBG.IFADCENCLK', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFADCDBG.IFADCENREFBUF', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFADCDBG.IFADCENSIDETONE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFADCDBG.IFADCINVERTCLK', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFADCDBG.IFADCSHORTINPUT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFADCTRIM.IFADCCLKSEL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFADCTRIM.IFADCENHALFMODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFADCTRIM.IFADCLDOSERIESAMPLVL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFADCTRIM.IFADCLDOSHUNTAMPLVL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFADCTRIM.IFADCLDOSHUNTCURLP', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFADCTRIM.IFADCLDOSHUNTCURLVL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFADCTRIM.IFADCOTAST1CURRENT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFADCTRIM.IFADCOTAST2CURRENT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFADCTRIM.IFADCREFBUFAMPLVL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFADCTRIM.IFADCREFBUFCURLVL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFADCTRIM.IFADCSIDETONEAMP', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFADCTRIM.IFADCSIDETONEFREQ', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFADCTRIM.IFADCTUNEZERO', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFADCTRIM.IFADCVCMLVL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFPGACTRL.DCCALDCGEAR', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFPGACTRL.DCCALDEC0', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFPGACTRL.DCCALON', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFPGACTRL.DCESTIEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.IFPGACTRL.DCRSTEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXCAL.LNAMIXCALEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXCAL.LNAMIXCALVMODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXCAL.LNAMIXENIRCAL1', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXCAL.LNAMIXENIRCAL2', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXCAL.LNAMIXIRCAL1AMP', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXCAL.LNAMIXIRCAL2AMP', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXDEBUG.LNAMIXDISMXR1', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXDEBUG.LNAMIXDISMXR2', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXEN.LNAMIXENLDO', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXTRIM0.LNAMIXCURCTRL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXTRIM0.LNAMIXHIGHCUR', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXTRIM0.LNAMIXLOWCUR', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXTRIM0.LNAMIXRFPKDBWSEL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXTRIM0.LNAMIXRFPKDCALCM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXTRIM0.LNAMIXRFPKDCALDM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXTRIM0.LNAMIXTRIMVREG', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXTRIM1.LNAMIXIBIAS1ADJ', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXTRIM1.LNAMIXLNA1CAPSEL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXTRIM1.LNAMIXMXRBIAS1', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXTRIM1.LNAMIXNCAS1ADJ', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXTRIM1.LNAMIXPCAS1ADJ', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXTRIM1.LNAMIXVOUT1ADJ', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXTRIM2.LNAMIXLNA2CAPSEL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXTRIM2.LNAMIXMXRBIAS2', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXTRIM2.LNAMIXNCAS2ADJ', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXTRIM2.LNAMIXPCAS2ADJ', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.LNAMIXTRIM2.LNAMIXVOUT2ADJ', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.PACTRL.PAEN10DBMVMID', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.PACTRL.PAEN20DBMVMID', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.PACTRL.PAENCAPATT', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.PACTRL.PAENLATCHBYPASS', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.PACTRL.PAENPOWERRAMPINGCLK', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.PACTRL.PAPOWER', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.PACTRL.PAPULLDOWNVDDPA', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.PACTRL.PAREGBYPASSPDRVLDO', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.PACTRL.PAREGBYPASSPREREG', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.PACTRL.PASELLDOVDDPA', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.PACTRL.PASELLDOVDDRF', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.PACTRL.PASELSLICE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.PACTRL.PASLICERST', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PAENCTRL.PARAMP', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM0.PATRIMDRVREGIBCORE0DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM0.PATRIMDRVREGIBNDIO0DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM0.PATRIMDRVREGPSR0DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM0.PATRIMDRVSLICE0DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM0.PATRIMFB0DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM0.PATRIMPABIASN0DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM0.PATRIMPABIASP0DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM0.PATRIMPASLICE0DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM0.PATRIMVREF0DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM1.PATRIM10DBMDUTYCYN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM1.PATRIM10DBMDUTYCYP', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM1.PATRIM20DBMPREDRV', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM1.PATRIMANTSWBIAS', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM1.PATRIMBLEEDAUTOPREREG', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM1.PATRIMCAPPAOUTM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM1.PATRIMCAPPAOUTP', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM1.PATRIMCMGAIN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM1.PATRIMDLY0', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM1.PATRIMDLY1', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM1.PATRIMFBKATTPDRVLDO', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM1.PATRIMIBIASMASTER', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM2.PATRIMLDOFBHVPDRVLDO', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM2.PATRIMLDOFBHVPREREG', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM2.PATRIMLDOHVPDRVLDO', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM2.PATRIMLDOHVPREREG', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM2.PATRIMLDOPSRPDRVLDO', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM2.PATRIMLDOPSRPREREG', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM2.PATRIMLDOSLICESPDRVLDO', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM2.PATRIMLDOSLICESPREREG', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM2.PATRIMNBIAS', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM2.PATRIMNCASC', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM2.PATRIMPADACGLITCH', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM2.PATRIMPBIAS', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PATRIM2.PATRIMPCASC', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGACAL.PGAOFFNCALI', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGACAL.PGAOFFNCALQ', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGACAL.PGAOFFPCALI', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGACAL.PGAOFFPCALQ', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGADEBUG.PGACCTRIM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGATRIM.PGACTUNE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGATRIM.PGADISANTILOCK', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGATRIM.PGAVCMOUTTRIM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGATRIM.PGAVLDOTRIM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PRECTRL.PREBYPFORCE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PRECTRL.PREREGTRIM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PRECTRL.PREVREFTRIM', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RADIOEN.PREEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RADIOEN.PRESTB100UDIS', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RADIOEN.RFBIASEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFBIASCAL.RFBIASCALBIAS', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFBIASCAL.RFBIASCALTC', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFBIASCAL.RFBIASCALVREF', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFBIASCAL.RFBIASCALVREFSTARTUP', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFBIASCTRL.RFBIASDISABLEBOOTSTRAP', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFBIASCTRL.RFBIASLDOHIGHCURRENT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFBIASCTRL.RFBIASLDOVREFTRIM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFBIASCTRL.RFBIASNONFLASHMODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFBIASCTRL.RFBIASSTARTUPCORE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFBIASCTRL.RFBIASSTARTUPSUPPLY', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFLOCK0.FRCCONVMODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFLOCK0.FRCPAUSING', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFLOCK0.MODEMANTDIVMODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFLOCK0.MODEMDEC1', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFLOCK0.MODEMDSSS', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFLOCK0.MODEMDUALSYNC', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFLOCK0.MODEMMODFORMAT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFLOCK0.RACIFPGAEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFLOCK0.RACPAEN10DBMM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFLOCK0.RACPAEN10DBMP', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFLOCK0.RACPAEN20DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFLOCK0.RACSGPAEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFLOCK0.SYNTHLODIVFREQCTRL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFLOCK1.RACDRVSLICE0DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFLOCK1.RACPASLICE0DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFLOCK1.RACPASLICE10DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFLOCK1.RACPASLICE20DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFLOCK1.RACPASTRIPE10DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RFLOCK1.RACPASTRIPE20DBM', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RFPATHEN1.LNAMIXEN0DBMPA1', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RFPATHEN1.LNAMIXEN1', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RFPATHEN1.LNAMIXRFATTDCEN1', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RFPATHEN1.LNAMIXRFPKDENRF1', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RFPATHEN1.LNAMIXTRSW1', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RFPATHEN1.PAENANT1', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RFPATHEN1.PAENPA10DBM', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RFPATHEN1.PAENPAPREDRV10DBM', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RFPATHEN2.LNAMIXEN0DBMPA2', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RFPATHEN2.LNAMIXEN2', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RFPATHEN2.LNAMIXRFATTDCEN2', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RFPATHEN2.LNAMIXRFPKDENRF2', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RFPATHEN2.LNAMIXTRSW2', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RFPATHEN2.PAENANT2', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RFPATHEN2.PAENPA20DBM', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RFPATHEN2.PAENPAPREDRV20DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RX.IFADCCAPRESET', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RX.IFADCENLDOSERIES', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.RX.IFADCENLDOSHUNT', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RX.LNAMIXENRFPKD', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RX.LNAMIXLDOLOWCUR', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RX.LNAMIXREGLOADEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RX.PGAENLDO', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RX.SYCHPBIASTRIMBUF', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RX.SYCHPQNC3EN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RX.SYMMDMODE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RX.SYPFDCHPLPEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RX.SYPFDFPWEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RXENSRCEN.CHANNELBUSYEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RXENSRCEN.DEMODRXREQEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RXENSRCEN.FRAMEDETEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RXENSRCEN.PREDETEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RXENSRCEN.PRSRXEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RXENSRCEN.SWRXEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.RXENSRCEN.TIMDETEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SPARE.AUXADCSPARE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SPARE.CLKMULTSPARE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SPARE.IFADCSPARE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SPARE.PATRIMSPARE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SPARE.PGASPACER0', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SPARE.SYSPARE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYCAL.SYHILOADCHPREG', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYCAL.SYVCOMODEPKD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYCAL.SYVCOMORECURRENT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYCAL.SYVCOSLOWNOISEFILTER', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYCAL.SYVCOVCAPVCM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYDEBUG.SYCHPSWITCHEDBIASEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYDEBUG.SYENMMDDREGBIASSTB', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYDEBUG.SYLODIVAUXLODIS', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYDEBUG.SYLODIVCLKSYNCSGEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYDEBUG.SYLODIVLDOPSRTRIM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYDEBUG.SYPFDLOCKWINDOWN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYDEBUG.SYPFDTRADMODEEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYDEBUG.SYVCOREGBYPASS', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYEN.SYCHPEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYEN.SYCHPLPEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYEN.SYENCHPREG', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYEN.SYENCHPREPLICA', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYEN.SYENMMDREG', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYEN.SYENMMDREPLICA1', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYEN.SYENMMDREPLICA2', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYEN.SYENVCOBIAS', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYEN.SYENVCOPFET', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYEN.SYENVCOREG', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYEN.SYLODIVEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYEN.SYLODIVLDOBIASEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYEN.SYLODIVLDOEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYEN.SYSTARTCHPREG', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYEN.SYSTARTMMDREG', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.SYLOEN.SYLODIVRLO12G4EN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.SYLOEN.SYLODIVRLO1SGEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.SYLOEN.SYLODIVRLO22G4EN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.SYLOEN.SYLODIVRLO2SGEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.SYLOEN.SYLODIVRLOADCCLK2G4EN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.SYLOEN.SYLODIVTLO0DBM2G4AUXEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.SYLOEN.SYLODIVTLO0DBM2G4EN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.SYLOEN.SYLODIVTLO0DBMSGAUXEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.SYLOEN.SYLODIVTLO0DBMSGEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.SYLOEN.SYLODIVTLO20DBM2G4AUXEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.SYLOEN.SYLODIVTLO20DBM2G4EN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.SYLOEN.SYLODIVTLO20DBMSGAUXEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.SYLOEN.SYLODIVTLO20DBMSGEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYMMDCTRL.SYMMDDIVRSDIG', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYMMDCTRL.SYMMDENRSDIG', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYNTHENCTRL.LPFBWSEL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYNTHENCTRL.SYNTHSTARTREQ', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYNTHENCTRL.VCBUFEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYNTHENCTRL.VCOSTARTUP', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYNTHREGCTRL.CHPLDOVREFTRIM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYNTHREGCTRL.MMDLDOVREFTRIM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TESTCTRL.AUX2RFSENSE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TESTCTRL.DEMODEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TESTCTRL.LOOPBACK2LNAINPUT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TESTCTRL.LOOPBACK2LNAOUTPUT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TESTCTRL.MODEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TX.ENPAPOWER', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TX.ENPASELSLICE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TX.ENPATRIMPASLICE0DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TX.ENXOSQBUFFILT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TX.PABLEEDDRVREG0DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TX.PABLEEDREG0DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TX.PAEN10DBMM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TX.PAEN10DBMP', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TX.PAEN10DBMPDRV', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TX.PAEN20DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TX.PAEN20DBMPDRV', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TX.PAENBIAS0DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TX.PAENBLEEDPDRVLDO', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TX.PAENBLEEDPREREG', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TX.PAENDRVREG0DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TX.PAENDRVREGBIAS0DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TX.PAENLDOHVPDRVLDO', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TX.PAENLDOHVPREREG', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TX.PAENLO0DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TX.PAENPAOUT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TX.PAENREG0DBM', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.TX.PAENTAPCAP0DBM', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.VCOCTRL.VCOAMPLITUDE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'RAC.VCOCTRL.VCODETAMPLITUDE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.XORETIMECTRL.XORETIMEDISRETIME', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.XORETIMECTRL.XORETIMEENRETIME', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.XORETIMECTRL.XORETIMELIMITH', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.XORETIMECTRL.XORETIMELIMITL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.XORETIMECTRL.XORETIMERESETN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.XOSQBUFFILT.XOSQBUFFILT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.CALCTRL.CAPCALCYCLEWAIT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.CALCTRL.CAPCALENDMODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.CALCTRL.CAPCALSIDE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.CALCTRL.CAPCALSINGLESTEP', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.CALCTRL.CAPCALWAITMODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.CALCTRL.CLKLOADDENOMVAL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.CALCTRL.COUNTERRESETVAL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.CALCTRL.NUMCYCLES', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.CALCTRL.PRESCALERRESETVAL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.CALCTRL.STARTUPTIMING', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.CALCTRL.SYNTHOVERRIDEEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'SYNTH.CALOFFSET.CALOFFSET', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'SYNTH.CHCTRL.CHNO', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.CHPDACCTRL.CHPDACVAL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.CHPDACCTRL.OVERRIDE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.CHPDACINIT.DACINIT', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'SYNTH.CHSP.CHSP', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.CTRL.INTEGERMODE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'SYNTH.CTRL.LOCKTHRESHOLD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.CTRL.MMDMANRSTN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.CTRL.MMDRSTNOVERRIDEEN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'SYNTH.CTRL.PRSMUX0', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'SYNTH.CTRL.PRSMUX1', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.CTRL.TRISTATEPOSTPONE', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'SYNTH.DIVCTRL.LODIVFREQCTRL', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'SYNTH.EN.EN', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'SYNTH.FREQ.FREQ', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'SYNTH.IFFREQ.IFFREQ', int, ModelVariableFormat.HEX)
        # self._addModelRegister(model, 'SYNTH.IFFREQ.LOSIDE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LOCNTCTRL.ENABLE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LOCNTCTRL.LOCNTMANCLEAR', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LOCNTCTRL.LOCNTMANRUN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LOCNTCTRL.LOCNTOVERRIDEEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LOCNTCTRL.NUMCYCLE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LOCNTCTRL.READ', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.VCDACCTRL.LPFEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.VCDACCTRL.LPFQSEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.VCDACCTRL.VCDACEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.VCDACCTRL.VCDACVAL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.VCOFRCCAL.TIMEOUTCOUNTMAX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.VCOFRCCAL.TIMEOUTCOUNTRESTART', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.VCOGAIN.VCOKVCOARSE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.VCOGAIN.VCOKVFINE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.VCORANGE.CMPENABLE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.VCORANGE.CMPRESET', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.VCORANGE.DATA', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.VCORANGE.HIGHTHRESH', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.VCORANGE.INVCOMPDIR', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.VCORANGE.LOAD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.VCORANGE.LOWTHRESH', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.VCORANGE.MODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.VCORANGE.OVERRIDEEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.VCORANGE.RAMPEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.VCORANGE.RAMPRATE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.VCORANGE.SWITCHMODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.VCORANGE.THROUGH', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.VCOTUNING.VCAPSEL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.VCOTUNING.VCOTUNING', int, ModelVariableFormat.HEX)

        self._add_channel_definition(model)

    def _add_SYNTH_DSMCTRL_regs(self, model):
        self._addModelRegister(model, 'SYNTH.DSMCTRLRX.DEMMODERX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.DSMCTRLRX.DITHERDACRX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.DSMCTRLRX.DITHERDSMINPUTRX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.DSMCTRLRX.DITHERDSMOUTPUTRX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.DSMCTRLRX.DSMMODERX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.DSMCTRLRX.LSBFORCERX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.DSMCTRLRX.MASHORDERRX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.DSMCTRLRX.REQORDERRX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.DSMCTRLTX.DEMMODETX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.DSMCTRLTX.DITHERDACTX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.DSMCTRLTX.DITHERDSMINPUTTX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.DSMCTRLTX.DITHERDSMOUTPUTTX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.DSMCTRLTX.DSMMODETX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.DSMCTRLTX.LSBFORCETX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.DSMCTRLTX.MASHORDERTX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.DSMCTRLTX.REQORDERTX', int, ModelVariableFormat.HEX)

    def _add_SYNTH_LPFCTRL_regs(self, model):
        self._addModelRegister(model, 'SYNTH.LPFCTRL1CAL.OP1BWCAL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL1CAL.OP1COMPCAL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL1CAL.RFBVALCAL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL1CAL.RPVALCAL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL1CAL.RZVALCAL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL1RX.OP1BWRX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL1RX.OP1COMPRX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL1RX.RFBVALRX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL1RX.RPVALRX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL1RX.RZVALRX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL1TX.OP1BWRX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL1TX.OP1BWTX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL1TX.OP1COMPTX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL1TX.RFBVALTX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL1TX.RPVALTX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL1TX.RZVALTX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL2RX.CASELRX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL2RX.CAVALRX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL2RX.CFBSELRX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL2RX.CZSELRX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL2RX.CZVALRX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL2RX.LPFGNDSWENRX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL2RX.LPFSWENRX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL2RX.MODESELRX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL2RX.VCMLVLRX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL2TX.CASELTX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL2TX.CAVALTX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL2TX.CFBSELTX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL2TX.CZSELTX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL2TX.CZVALTX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL2TX.LPFGNDSWENTX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL2TX.LPFSWENTX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL2TX.MODESELTX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL2TX.VCMLVLTX', int, ModelVariableFormat.HEX)

    def _add_lpfincap_regs(self, model):
        self._addModelRegister(model, 'SYNTH.LPFCTRL2RX.LPFINCAPRX', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.LPFCTRL2TX.LPFINCAPTX', int, ModelVariableFormat.HEX)

    def _add_SYNTH_DENOMINIT_regs(self, model):
        self._addModelRegister(model, 'SYNTH.MMDDENOMINIT.DENOMINIT0', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.MMDDENOMINIT.DENOMINIT1', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'SYNTH.MMDDENOMINIT.DENOMINIT2', int, ModelVariableFormat.HEX)



    def _add_RAC_SYTRIM0_SYCHPREPLICACURRADJ_regs(self, model):
        self._addModelRegister(model, 'RAC.SYTRIM0.SYCHPREPLICACURRADJ', int, ModelVariableFormat.HEX)

    def _add_RAC_SYNTHCTRL_MMDPOWERBALANCEDISABLE_regs(self, model):
        self._addModelRegister(model, 'RAC.SYNTHCTRL.MMDPOWERBALANCEDISABLE', int, ModelVariableFormat.HEX)

    def _add_RAC_PGACTRL(self,model):
        self._addModelRegister(model, 'RAC.PGACTRL.LNAMIXRFPKDTHRESHSEL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGACTRL.PGABWMODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGACTRL.PGAENBIAS', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGACTRL.PGAENGHZ', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGACTRL.PGAENHYST', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGACTRL.PGAENLATCHI', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGACTRL.PGAENLATCHQ', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGACTRL.PGAENLDOLOAD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGACTRL.PGAENOFFD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGACTRL.PGAENPGAI', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGACTRL.PGAENPGAQ', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGACTRL.PGAENPKD', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGACTRL.PGAENRCMOUT', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGACTRL.PGAENSUBG', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGACTRL.PGAPOWERMODE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGACTRL.PGATHRPKDHISEL', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.PGACTRL.PGATHRPKDLOSEL', int, ModelVariableFormat.HEX)

    def _add_SYTRIM_regs(self, model):
        self._addModelRegister(model, 'RAC.SYTRIM0.SYCHPBIAS', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYTRIM0.SYCHPCURR', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYTRIM0.SYCHPLEVNSRC', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYTRIM0.SYCHPLEVPSRC', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYTRIM0.SYCHPSRCEN', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYTRIM0.SYTRIMCHPREGAMPBIAS', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYTRIM0.SYTRIMCHPREGAMPBW', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYTRIM1.SYLODIVLDOTRIMCORE', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYTRIM1.SYLODIVLDOTRIMNDIO', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYTRIM1.SYMMDREPLICA1CURRADJ', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYTRIM1.SYMMDREPLICA2CURRADJ', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYTRIM1.SYTRIMMMDREGAMPBIAS', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.SYTRIM1.SYTRIMMMDREGAMPBW', int, ModelVariableFormat.HEX)

    def _add_RAC_CLKMULTEN0_en(self, model):
        self._addModelRegister(model, 'RAC.CLKMULTEN0.CLKMULTENDRVADC', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN0.CLKMULTDISICO', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN0.CLKMULTENBBDET', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN0.CLKMULTENBBXLDET', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN0.CLKMULTENBBXMDET', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN0.CLKMULTENCFDET', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN0.CLKMULTENFBDIV', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN0.CLKMULTENREFDIV', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN0.CLKMULTENREG1', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN0.CLKMULTENREG2', int, ModelVariableFormat.HEX)
        self._addModelRegister(model, 'RAC.CLKMULTEN0.CLKMULTENROTDET', int, ModelVariableFormat.HEX)

    def _add_channel_definition(self, model):
        self._addModelVariable(model, 'chcfg_base_frequency_hz', long, ModelVariableFormat.DECIMAL, units='Hz', desc='Channel Config RF frequency of channel 0.')
        self._addModelVariable(model, 'chcfg_channel_spacing_hz', int, ModelVariableFormat.DECIMAL, units='Hz', desc='Channel Config channel raster used for relative frequency configuration')
        self._addModelVariable(model, 'chcfg_physical_channel_offset', int, ModelVariableFormat.DECIMAL, desc='Channel Config Physical channel offset')
        self._addModelVariable(model, 'rail_tx_power_max', int, ModelVariableFormat.DECIMAL, is_array=True)


