/***************************************************************************//**
 * @file
 * @brief Thermistor conversion functions provided to the customer as source for
 *   highest level of customization.
 * @details This file contains the logic that converts thermistor impedance
 * to temperature in Celsius.
 *******************************************************************************
 * # License
 * <b>Copyright 2022 Silicon Laboratories Inc. www.silabs.com</b>
 *******************************************************************************
 *
 * SPDX-License-Identifier: Zlib
 *
 * The licensor of this software is Silicon Laboratories Inc.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 ******************************************************************************/

#include "sl_rail.h"
#include "rail.h"

#if (RAIL_SUPPORTS_EXTERNAL_THERMISTOR || SL_RAIL_SUPPORTS_EXTERNAL_THERMISTOR)

#include "sl_rail_util_thermistor.h"
#include <math.h>

#ifndef SLI_LIBRAIL_BUILD
// Do not put an sl_rail_util_thermistor_init() implementation into librail.
void sl_rail_util_thermistor_init(void)
{
#if (defined(GPIO_THMSW_EN_PORT) && defined(GPIO_THMSW_EN_PIN))
  sl_rail_hfxo_thermistor_config_t hfxo_thermistor_config = {
    .port = GPIO_THMSW_EN_PORT,
    .pin = GPIO_THMSW_EN_PIN
  };
  (void)sl_rail_config_hfxo_thermistor(SL_RAIL_EFR32_HANDLE, &hfxo_thermistor_config);
#endif// (defined(GPIO_THMSW_EN_PORT) && defined(GPIO_THMSW_EN_PIN))
}
#endif//SLI_LIBRAIL_BUILD

/* This macro is defined when Silicon Labs builds this into the library as WEAK
   to ensure it can be overriden by customer versions of these functions. The macro
   should *not* be defined in a customer build. */
SLI_LIBRAIL_WEAK
RAIL_Status_t RAIL_ConvertThermistorImpedance(RAIL_Handle_t railHandle,
                                              uint32_t thermistorImpedance,
                                              int16_t *thermistorTemperatureC)
{
  (void) railHandle;
  // T = 4200 / (log(Rtherm/100000) + 4200/(273.15+25)) - 273.15
  double logParam = (double)(thermistorImpedance) / 100000.0;
  if (logParam <= 0.0) {
    return RAIL_STATUS_INVALID_PARAMETER;
  }
  double impedanceLn = log(logParam);
  // Multiply by 8 to convert to eighth of Celsius degrees
  *thermistorTemperatureC = (int16_t) round((4200.0 / (impedanceLn + 4200.0 / (273.15 + 25.0)) - 273.15) * 8.0);
  return RAIL_STATUS_NO_ERROR;
}

SLI_LIBRAIL_WEAK
RAIL_Status_t RAIL_ComputeHFXOPPMError(RAIL_Handle_t railHandle,
                                       int16_t crystalTemperatureC,
                                       int8_t *crystalPPMError)
{
  (void) railHandle;
  // Equation is:
  // f(T) = 1.05*10^-4(T-T0)^3 + 1.0*10^-4(T-T0)^2 - 0.74(T-T0) - 0.35, Reference to 30C
  // Cache T-T0
  double deltaRefTempC = (double)crystalTemperatureC - 30.0;
  *crystalPPMError = (int8_t) round(1.05 * 0.0001 * pow(deltaRefTempC, 3.0)
                                    + 0.0001 * pow(deltaRefTempC, 2.0)
                                    - 0.74 * deltaRefTempC
                                    - 0.35);
  return RAIL_STATUS_NO_ERROR;
}

// Below needed for RAIL 3

SLI_LIBRAIL_WEAK
sl_rail_status_t sl_railcb_convert_thermistor_impedance(sl_rail_handle_t rail_handle,
                                                        uint32_t thermistor_impedance_ohms,
                                                        int16_t *p_thermistor_temperature_8c)
{
  return (sl_rail_status_t)RAIL_ConvertThermistorImpedance((RAIL_Handle_t)rail_handle,
                                                           thermistor_impedance_ohms,
                                                           p_thermistor_temperature_8c);
}

SLI_LIBRAIL_WEAK
sl_rail_status_t sl_railcb_compute_hfxo_error_ppm(sl_rail_handle_t rail_handle,
                                                  int16_t crystal_temperature_c,
                                                  int8_t *p_crystal_error_ppm)
{
  return (sl_rail_status_t)RAIL_ComputeHFXOPPMError((RAIL_Handle_t)rail_handle,
                                                    crystal_temperature_c,
                                                    p_crystal_error_ppm);
}

#endif // (RAIL_SUPPORTS_EXTERNAL_THERMISTOR || SL_RAIL_SUPPORTS_EXTERNAL_THERMISTOR)
