/***************************************************************************//**
 * @file sl_rail_util_mp_transition_time.h
 * @brief
 *******************************************************************************
 * # License
 * <b>Copyright 2025 Silicon Laboratories Inc. www.silabs.com</b>
 *******************************************************************************
 *
 * SPDX-License-Identifier: Zlib
 *
 * The licensor of this software is Silicon Laboratories Inc.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 ******************************************************************************/
#ifndef SL_RAIL_UTIL_MP_TRANSITION_TIME_H
#define SL_RAIL_UTIL_MP_TRANSITION_TIME_H

#include "sl_rail_types.h"

/**
 * @brief Represents an entry in the multi-protocol transition time table.
 *
 * This structure maps a minimum system clock frequency to the corresponding
 * transition time required for multi-protocol radio operations.
 *
 * Each entry specifies a `transition_time_us` that applies to all system clock
 * frequencies greater than or equal to the `minimum_frequency_hz`. The table
 * must be sorted in descending order of frequency to ensure correct lookups.
 * If the table is not sorted, the lookup process may yield incorrect results.
 *
 * The transition time table is autogenerated during project generation. For
 * more details, refer to the `sl_rail_util_mp_transition_time.md` file.
 */
typedef struct {
  uint32_t minimum_frequency_hz; // Minimum frequency in Hz for this entry
  sl_rail_time_t transition_time_us; // Transition time in microseconds
} sl_rail_util_mp_transition_time_entry_t;

/**
 * @brief Retrieves the transition time for the current system clock frequency.
 *
 * Searches the provided transition time table for the entry corresponding to
 * the current system clock frequency. The table must be sorted in descending
 * order of frequency to ensure correct lookups. If the table is not sorted,
 * the lookup process may yield incorrect results.
 *
 * The transition time table is autogenerated during project generation. For
 * more details, refer to the `sl_rail_util_mp_transition_time.md` file.
 *
 * @param[in] p_table A non-NULL pointer to the transition time table.
 * @param[in] table_size Number of entries in the transition time table.
 * @param[out] p_transition_time A non-NULL pointer to store the resulting transition time (in microseconds).
 *
 * @return SL_STATUS_OK if a matching entry is found and the transition time is retrieved.
 * @return SL_STATUS_INVALID_PARAMETER if the input parameters are invalid.
 * @return SL_STATUS_NOT_FOUND if no matching entry is found for the current system clock frequency.
 */
sl_status_t sl_rail_util_mp_transition_time_get(const sl_rail_util_mp_transition_time_entry_t *p_table,
                                                const size_t table_size,
                                                sl_rail_time_t *p_transition_time);

/**
 * @brief Initializes the multi-protocol transition time utility.
 *
 * Determines the appropriate transition time for the current system clock
 * frequency and configures the RAIL library with the calculated transition
 * time. It uses the transition time table defined in the configuration.
 *
 * The transition time table is autogenerated during project generation. For
 * more details, refer to the `sl_rail_util_mp_transition_time.md` file.
 *
 * If the transition time cannot be determined, the function does not configure
 * the RAIL library.
 */
void sl_rail_util_mp_transition_time_init(void);

#endif
