/***************************************************************************/ /**
 * @file sli_cpc_debug.h
 * @brief CPC SystemView header file
 *******************************************************************************
 * # License
 * <b>Copyright 2020 Silicon Laboratories Inc. www.silabs.com</b>
 *******************************************************************************
 *
 * SPDX-License-Identifier: Zlib
 *
 * The licensor of this software is Silicon Laboratories Inc.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 ******************************************************************************/

#ifndef SLI_CPC_DEBUG_H
#define SLI_CPC_DEBUG_H

#include "string.h"

#include "sl_cpc_config.h"
#include "sli_cpc_instance.h"

#if defined(SL_COMPONENT_CATALOG_PRESENT)
#include "sl_component_catalog.h"
#endif

#if (SL_CPC_DEBUG_SYSTEM_VIEW_LOG_CORE_EVENT == 1) \
  || (SL_CPC_DEBUG_SYSTEM_VIEW_LOG_ENDPOINT_EVENT == 1)
#include "SEGGER_SYSVIEW.h"
#endif

#if ((SL_CPC_DEBUG_SYSTEM_VIEW_LOG_CORE_EVENT == 1)     \
  || (SL_CPC_DEBUG_SYSTEM_VIEW_LOG_ENDPOINT_EVENT == 1) \
  || (SL_CPC_DEBUG_CORE_EVENT_COUNTERS == 1)            \
  || (SL_CPC_DEBUG_ENDPOINT_EVENT_COUNTERS == 1))
#define CPC_DEBUG_TRACE
#endif

// Unifed TRACING MACROs
#define SLI_CPC_DEBUG_TRACE_CORE_OPEN_ENDPOINT(instance)                { SLI_CPC_DEBUG_CORE_COUNTER_INC(instance, endpoint_opened); \
                                                                          SLI_CPC_SYSVIEW_MARK_EVENT_ON_CORE(OPEN_ENDPOINT); }

#define SLI_CPC_DEBUG_TRACE_CORE_CLOSE_ENDPOINT(instance)               { SLI_CPC_DEBUG_CORE_COUNTER_INC(instance, endpoint_closed); \
                                                                          SLI_CPC_SYSVIEW_MARK_EVENT_ON_CORE(CLOSE_ENDPOINT); }

#define SLI_CPC_DEBUG_TRACE_CORE_RXD_FRAME(instance)                    { SLI_CPC_DEBUG_CORE_COUNTER_INC(instance, rxd_frame); \
                                                                          SLI_CPC_SYSVIEW_MARK_EVENT_ON_CORE(RX_FRAME); }

#define SLI_CPC_DEBUG_TRACE_CORE_RXD_VALID_IFRAME(instance)             { SLI_CPC_DEBUG_CORE_COUNTER_INC(instance, rxd_valid_iframe); \
                                                                          SLI_CPC_SYSVIEW_MARK_EVENT_ON_CORE(RXD_VALID_IFRAME); }

#define SLI_CPC_DEBUG_TRACE_CORE_RXD_VALID_UFRAME(instance)             { SLI_CPC_DEBUG_CORE_COUNTER_INC(instance, rxd_valid_uframe); \
                                                                          SLI_CPC_SYSVIEW_MARK_EVENT_ON_CORE(RXD_VALID_UFRAME); }

#define SLI_CPC_DEBUG_TRACE_CORE_RXD_VALID_SFRAME(instance)             { SLI_CPC_DEBUG_CORE_COUNTER_INC(instance, rxd_valid_sframe); \
                                                                          SLI_CPC_SYSVIEW_MARK_EVENT_ON_CORE(RXD_VALID_SFRAME); }

#define SLI_CPC_DEBUG_TRACE_CORE_RXD_DATA_FRAME_DROPPED(instance)       { SLI_CPC_DEBUG_CORE_COUNTER_INC(instance, rxd_data_frame_dropped); \
                                                                          SLI_CPC_SYSVIEW_MARK_EVENT_ON_CORE(RXD_DATA_FRAME_DROPPED); }

#define SLI_CPC_DEBUG_TRACE_CORE_TXD_REJECT_DESTINATION_UNREACHABLE(instance)   { SLI_CPC_DEBUG_CORE_COUNTER_INC(instance, txd_reject_destination_unreachable); \
                                                                                  SLI_CPC_SYSVIEW_MARK_EVENT_ON_CORE(TXD_REJECT_DESTINATION_UNREACHABLE); }

#define SLI_CPC_DEBUG_TRACE_CORE_TXD_REJECT_ERROR_FAULT(instance)       { SLI_CPC_DEBUG_CORE_COUNTER_INC(instance, txd_reject_error_fault); \
                                                                          SLI_CPC_SYSVIEW_MARK_EVENT_ON_CORE(TXD_REJECT_ERROR_FAULT); }

#define SLI_CPC_DEBUG_TRACE_CORE_DRIVER_ERROR(instance)                 { SLI_CPC_DEBUG_CORE_COUNTER_INC(instance, driver_error); \
                                                                          SLI_CPC_SYSVIEW_MARK_EVENT_ON_CORE(DRIVER_READ_ERROR); }

#define SLI_CPC_DEBUG_TRACE_CORE_DRIVER_PACKET_DROPPED(instance)        { SLI_CPC_DEBUG_CORE_COUNTER_INC(instance, driver_packet_dropped); \
                                                                          SLI_CPC_SYSVIEW_MARK_EVENT_ON_CORE(DRIVER_PACKET_DROPPED); }

#define SLI_CPC_DEBUG_TRACE_CORE_INVALID_HEADER_CHECKSUM(instance)      { SLI_CPC_DEBUG_CORE_COUNTER_INC(instance, invalid_header_checksum); \
                                                                          SLI_CPC_SYSVIEW_MARK_EVENT_ON_CORE(INVALID_HEADER_CHECKSUM); }

#define SLI_CPC_DEBUG_TRACE_CORE_INVALID_PAYLOAD_CHECKSUM(instance)     { SLI_CPC_DEBUG_CORE_COUNTER_INC(instance, invalid_payload_checksum); \
                                                                          SLI_CPC_SYSVIEW_MARK_EVENT_ON_CORE(INVALID_PAYLOAD_CHECKSUM); }

#define SLI_CPC_DEBUG_TRACE_CORE_TXD_TRANSMIT_COMPLETED(instance)       { SLI_CPC_DEBUG_CORE_COUNTER_INC(instance, txd_completed); \
                                                                          SLI_CPC_SYSVIEW_MARK_EVENT_ON_CORE(TXD_TRANSMIT_COMPLETED); }

#define SLI_CPC_DEBUG_TRACE_CORE_RE_TRANSMIT_FRAME(instance)            { SLI_CPC_DEBUG_CORE_COUNTER_INC(instance, retxd_data_frame); \
                                                                          SLI_CPC_SYSVIEW_MARK_EVENT_ON_CORE(RETRANSMIT); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_RXD_FRAME(endpoint)                { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, rxd_packet); \
                                                                          SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(RXD_FRAME, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_RXD_DATA_FRAME(endpoint)           { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, rxd_data_frame); \
                                                                          SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(RXD_DATA_FRAME, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_RXD_DATA_FRAME_QUEUED(endpoint) { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, rxd_data_frame_queued); \
                                                                       SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(RXD_DATA_FRAME_QUEUED, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_RXD_DATA_FRAME_DROPPED(endpoint) { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, rxd_data_frame_dropped); \
                                                                        SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(RXD_DATA_FRAME_DROPPED, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_RXD_SUPERVISORY_FRAME(endpoint) { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, rxd_supervisory_frame); \
                                                                       SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(RXD_SUPERVISORY_FRAME, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_RXD_SUPERVISORY_PROCESSED(endpoint) { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, rxd_supervisory_frame_processed); \
                                                                           SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(RXD_SUPERVISORY_PROCESSED, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_RXD_SUPERVISORY_DROPPED(endpoint) { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, rxd_supervisory_frame_dropped); \
                                                                         SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(RXD_SUPERVISORY_DROPPED, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_RXD_UNNUMBERED_FRAME(endpoint) { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, rxd_unnumbered_frame); \
                                                                      SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(RXD_SUPERVISORY_FRAME, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_RXD_UNNUMBERED_PROCESSED(endpoint) { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, rxd_unnumbered_frame_processed); \
                                                                          SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(RXD_SUPERVISORY_PROCESSED, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_RXD_UNNUMBERED_DROPPED(endpoint) { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, rxd_unnumbered_frame_dropped); \
                                                                        SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(RXD_SUPERVISORY_DROPPED, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_RXD_DUPLICATE_DATA_FRAME(endpoint)  { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, rxd_duplicate_data_frame); \
                                                                           SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(RXD_DUPLICATE_DATA_FRAME, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_RXD_ACK(endpoint)  { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, rxd_ack); \
                                                          SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(RXD_ACK, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_RXD_REJECT_DESTINATION_UNREACHABLE(endpoint)  { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, rxd_reject_destination_unreachable); \
                                                                                     SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(RXD_REJECT_DESTINATION_UNREACHABLE, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_RXD_REJECT_SEQ_MISMATCH(endpoint)  { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, rxd_reject_seq_mismatch); \
                                                                          SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(RXD_REJECT_SEQ_MISMATCH, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_RXD_REJECT_CHECSUM_MISMATCH(endpoint)  { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, rxd_reject_checksum_mismatch); \
                                                                              SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(RXD_REJECT_CHECSUM_MISMATCH, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_RXD_REJECT_SECURITY_ISSUE(endpoint)  { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, rxd_reject_security_issue); \
                                                                            SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(RXD_REJECT_SECURITY_ISSUE, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_RXD_REJECT_OUT_OF_MEMORY(endpoint)  { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, rxd_reject_out_of_memory); \
                                                                           SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(RXD_REJECT_OUT_OF_MEMORY, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_RXD_REJECT_FAULT(endpoint)  { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, rxd_reject_fault); \
                                                                   SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(RXD_REJECT_FAULT, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_TXD_DATA_FRAME(endpoint)  { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, txd_data_frame); \
                                                                 SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(TXD_DATA_FRAME, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_TXD_ACK(endpoint)  { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, txd_ack); \
                                                          SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(TXD_ACK, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_TXD_REJECT_DESTINATION_UNREACHABLE(endpoint)  { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, txd_reject_destination_unreachable); \
                                                                                     SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(TXD_REJECT_DESTINATION_UNREACHABLE, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_TXD_REJECT_SEQ_MISMATCH(endpoint)  { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, txd_reject_seq_mismatch); \
                                                                          SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(TXD_REJECT_SEQ_MISMATCH, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_TXD_REJECT_CHECKSUM_MISMATCH(endpoint)  { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, txd_reject_checksum_mismatch); \
                                                                               SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(TXD_REJECT_CHECSUM_MISMATCH, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_TXD_REJECT_SECURITY_ISSUE(endpoint)  { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, txd_reject_security_issue); \
                                                                            SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(TXD_REJECT_SECURITY_ISSUE, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_TXD_REJECT_OUT_OF_MEMORY(endpoint)  { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, txd_reject_out_of_memory); \
                                                                           SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(TXD_REJECT_OUT_OF_MEMORY, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_TXD_REJECT_FAULT(endpoint)  { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, txd_reject_fault); \
                                                                   SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(TXD_REJECT_FAULT, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_RETXD_DATA_FRAME(endpoint)  { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, retxd_data_frame); \
                                                                   SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(RETXD_DATA_FRAME, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_FRAME_TRANSMIT_SUBMITTED(endpoint)  { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, frame_transmit_submitted); \
                                                                           SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(FRAME_TRANSMIT_SUBMITTED, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_FRAME_TRANSMIT_COMPLETED(endpoint)  { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, frame_transmit_completed); \
                                                                           SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(FRAME_TRANSMIT_COMPLETED, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_DATA_FRAME_TRANSMIT_COMPLETED(endpoint)  { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, data_frame_transmit_completed); \
                                                                                SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(FRAME_TRANSMIT_COMPLETED, endpoint->id); }

#define SLI_CPC_DEBUG_TRACE_ENDPOINT_SUPERVISORY_FRAME_TRANSMIT_COMPLETED(endpoint)  { SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, supervisory_frame_transmit_completed); \
                                                                                       SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(FRAME_TRANSMIT_COMPLETED, endpoint->id); }

// DEBUG EVENTS COUNTERS ON CORE
#if (SL_CPC_DEBUG_CORE_EVENT_COUNTERS == 1)
#define SLI_CPC_DEBUG_CORE_INIT(instance)                 (memset(&(instance)->core_debug.core_counters, 0, sizeof((instance)->core_debug.core_counters)))
#define SLI_CPC_DEBUG_CORE_COUNTER_INC(instance, counter)   (((instance)->core_debug.core_counters.counter)++)
#else // SL_CPC_DEBUG_CORE_EVENT_COUNTERS
#define SLI_CPC_DEBUG_CORE_INIT(instance)
#define SLI_CPC_DEBUG_CORE_COUNTER_INC(instance, counter)
#endif  // SL_CPC_DEBUG_CORE_EVENT_COUNTERS

// DEBUG EVENTS COUNTERS ON ENDPOINT
#if (SL_CPC_DEBUG_ENDPOINT_EVENT_COUNTERS == 1)
#define SLI_CPC_DEBUG_ENDPOINT_INIT(endpoint)                   (memset(&endpoint->debug_counters, 0, sizeof(endpoint->debug_counters)))
#define SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, counter)   ((endpoint->debug_counters.counter)++)
#else // SL_CPC_DEBUG_ENDPOINT_EVENT_COUNTERS
#define SLI_CPC_DEBUG_ENDPOINT_INIT(endpoint)
#define SLI_CPC_DEBUG_ENDPOINT_COUNTER_INC(endpoint, counter)
#endif  // SL_CPC_DEBUG_ENDPOINT_EVENT_COUNTERS

// CORE EVENT ID
#if !defined(SLI_CPC_SYSVIEW_CORE_EVENT_ID)
#define SLI_CPC_SYSVIEW_CORE_EVENT_ID 9999
#endif

// ENDPOINT ID and MULTIPLICATOR
#if !defined(CPC_CORE_EVENT_ID_MULTIPLIER)
#define CPC_ENDPOINT_EVENT_ID_OFFSET 10000
#endif
#if !defined(CPC_CORE_EVENT_ID_MULTIPLIER)
#define CPC_ENDPOINT_EVENT_ID_MULTIPLIER 1000
#endif

// SYSTEMVIEW TRACE MACRO
#define SLI_CPC_SYSVIEW_MARK_EVENT_ON_CORE(event)
#define SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(event, endpoint)
#if (SL_CPC_DEBUG_SYSTEM_VIEW_LOG_CORE_EVENT == 1)
#undef SLI_CPC_SYSVIEW_MARK_EVENT_ON_CORE
#define SLI_CPC_SYSVIEW_MARK_EVENT_ON_CORE(event) SEGGER_SYSVIEW_NameMarker(SLI_CPC_SYSVIEW_CORE_EVENT_ID, SLI_CPC_SYSVIEW_ON_CORE_ ## event ## _MESSAGE)
#endif

#if (SL_CPC_DEBUG_SYSTEM_VIEW_LOG_ENDPOINT_EVENT == 1)
#undef SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT
#define SLI_CPC_SYSVIEW_MARK_EVENT_ON_ENDPOINT(event, endpoint) SEGGER_SYSVIEW_NameMarker(SLI_CPC_SYSVIEW_ON_ENDPOINT_ ## event ## _EVENT_ID + endpoint, SLI_CPC_SYSVIEW_ON_ENDPOINT_ ## event ## _EVENT_MESSAGE)
#endif

#define STR_EXPAND(tok) #tok
#define STR(tok) STR_EXPAND(tok)

#define ON_ENDPOINT_STR_MESSAGE(EVENT, NRB, MULTIPLER)   "CPC: " EVENT " [EP ID = MarkerID - (" STR(NRB) " * " STR(MULTIPLER) ")]"

// SYSTEMVIEW; CORE MESSAGE
#define SLI_CPC_SYSVIEW_ON_CORE_OPEN_ENDPOINT_MESSAGE "CPC: Open endpoint"
#define SLI_CPC_SYSVIEW_ON_CORE_CLOSE_ENDPOINT_MESSAGE "CPC: Close endpoint"
#define SLI_CPC_SYSVIEW_ON_CORE_RX_FRAME_MESSAGE "CPC: Receive Frame"
#define SLI_CPC_SYSVIEW_ON_CORE_RXD_VALID_FRAME_MESSAGE "CPC: Received valid frame"
#define SLI_CPC_SYSVIEW_ON_CORE_RXD_DATA_FRAME_DROPPED_MESSAGE "CPC: Frame dropped"
#define SLI_CPC_SYSVIEW_ON_CORE_TXD_REJECT_DESTINATION_UNREACHABLE_MESSAGE "CPC: Transmit Destination unreachable"
#define SLI_CPC_SYSVIEW_ON_CORE_TXD_REJECT_ERROR_FAULT_MESSAGE "CPC: Transmit Error fault"
#define SLI_CPC_SYSVIEW_ON_CORE_TXD_TRANSMIT_COMPLETED_MESSAGE "CPC: Transmit completed"
#define SLI_CPC_SYSVIEW_ON_CORE_INVALID_HEADER_CHECKSUM_MESSAGE "CPC: Invalid checksum in protocol header"
#define SLI_CPC_SYSVIEW_ON_CORE_DRIVER_READ_ERROR_MESSAGE "CPC: Driver reported an error"
#define SLI_CPC_SYSVIEW_ON_CORE_DRIVER_PACKET_DROPPED_MESSAGE "CPC: Driver dropped a packet"
#define SLI_CPC_SYSVIEW_ON_CORE_RXD_VALID_IFRAME_MESSAGE "CPC: Valid iframe received"
#define SLI_CPC_SYSVIEW_ON_CORE_RXD_VALID_SFRAME_MESSAGE "CPC: Valid sframe received"
#define SLI_CPC_SYSVIEW_ON_CORE_RXD_VALID_UFRAME_MESSAGE "CPC: Valid uframe received"
#define SLI_CPC_SYSVIEW_ON_CORE_INVALID_PAYLOAD_CHECKSUM_MESSAGE "CPC: Invalid checksum on payload"
#define SLI_CPC_SYSVIEW_ON_CORE_RETRANSMIT_MESSAGE "CPC: Frame re-transmitted"

// SYSTEMVIEW; ENDPOINT EVENT IDs and MESSAGES
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_FRAME_EVENT_NBR                         1
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_FRAME_EVENT_MARKER                      (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_FRAME_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_FRAME_EVENT_ID                          (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_FRAME_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_FRAME_EVENT_MESSAGE                     ON_ENDPOINT_STR_MESSAGE("Received frame", SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_FRAME_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_DATA_RX_EVENT_NBR                           2
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_DATA_RX_EVENT_MARKER                        (SLI_CPC_SYSVIEW_ON_ENDPOINT_DATA_RX_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_DATA_RX_EVENT_ID                            (SLI_CPC_SYSVIEW_ON_ENDPOINT_DATA_RX_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_DATA_RX_EVENT_MESSAGE                       ON_ENDPOINT_STR_MESSAGE("Received Data", SLI_CPC_SYSVIEW_ON_ENDPOINT_DATA_RX_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_DATA_TX_EVENT_NBR                           3
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_DATA_TX_EVENT_MARKER                        (SLI_CPC_SYSVIEW_ON_ENDPOINT_DATA_TX_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_DATA_TX_EVENT_ID                            (SLI_CPC_SYSVIEW_ON_ENDPOINT_DATA_TX_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_DATA_TX_EVENT_MESSAGE                       ON_ENDPOINT_STR_MESSAGE("Packet sent", SLI_CPC_SYSVIEW_ON_ENDPOINT_DATA_TX_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_ACK_TX_EVENT_NBR                            4
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_ACK_TX_EVENT_MARKER                         (SLI_CPC_SYSVIEW_ON_ENDPOINT_ACK_TX_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_ACK_TX_EVENT_ID                             (SLI_CPC_SYSVIEW_ON_ENDPOINT_ACK_TX_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_ACK_TX_EVENT_MESSAGE                        ON_ENDPOINT_STR_MESSAGE("ACK sent", SLI_CPC_SYSVIEW_ON_ENDPOINT_ACK_TX_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_ACK_RX_EVENT_NBR                            5
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_ACK_RX_EVENT_MARKER                         (SLI_CPC_SYSVIEW_ON_ENDPOINT_ACK_RX_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_ACK_RX_EVENT_ID                             (SLI_CPC_SYSVIEW_ON_ENDPOINT_ACK_RX_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_ACK_RX_EVENT_MESSAGE                        ON_ENDPOINT_STR_MESSAGE("Received ACK", SLI_CPC_SYSVIEW_ON_ENDPOINT_ACK_RX_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RETRANSMIT_EVENT_NBR                        6
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RETRANSMIT_EVENT_MARKER                     (SLI_CPC_SYSVIEW_ON_ENDPOINT_RETRANSMIT_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RETRANSMIT_EVENT_ID                         (SLI_CPC_SYSVIEW_ON_ENDPOINT_RETRANSMIT_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RETRANSMIT_EVENT_MESSAGE                    ON_ENDPOINT_STR_MESSAGE("Retransmitted frame", SLI_CPC_SYSVIEW_ON_ENDPOINT_RETRANSMIT_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_PACKET_DROPPED_EVENT_NBR                    7
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_PACKET_DROPPED_EVENT_MARKER                 (SLI_CPC_SYSVIEW_ON_ENDPOINT_PACKET_DROPPED_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_PACKET_DROPPED_EVENT_ID                     (SLI_CPC_SYSVIEW_ON_ENDPOINT_PACKET_DROPPED_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_PACKET_DROPPED_EVENT_MESSAGE                ON_ENDPOINT_STR_MESSAGE("Dropped packet", SLI_CPC_SYSVIEW_ON_ENDPOINT_PACKET_DROPPED_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_CRC_ERROR_EVENT_NBR                         8
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_CRC_ERROR_EVENT_MARKER                      (SLI_CPC_SYSVIEW_ON_ENDPOINT_CRC_ERROR_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_CRC_ERROR_EVENT_ID                          (SLI_CPC_SYSVIEW_ON_ENDPOINT_CRC_ERROR_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_CRC_ERROR_EVENT_MESSAGE                     ON_ENDPOINT_STR_MESSAGE("CRC Error", SLI_CPC_SYSVIEW_ON_ENDPOINT_CRC_ERROR_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_INVALID_FCS_EVENT_NBR                       9
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_INVALID_FCS_EVENT_MARKER                    (SLI_CPC_SYSVIEW_ON_ENDPOINT_INVALID_FCS_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_INVALID_FCS_EVENT_ID                        (SLI_CPC_SYSVIEW_ON_ENDPOINT_INVALID_FCS_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_INVALID_FCS_EVENT_MESSAGE                   ON_ENDPOINT_STR_MESSAGE("Invalid FCS", SLI_CPC_SYSVIEW_ON_ENDPOINT_INVALID_FCS_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_INVALID_SEQ_EVENT_NBR                       10
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_INVALID_SEQ_EVENT_MARKER                    (SLI_CPC_SYSVIEW_ON_ENDPOINT_INVALID_SEQ_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_INVALID_SEQ_EVENT_ID                        (SLI_CPC_SYSVIEW_ON_ENDPOINT_INVALID_SEQ_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_INVALID_SEQ_EVENT_MESSAGE                   ON_ENDPOINT_STR_MESSAGE("Invalid sequence number", SLI_CPC_SYSVIEW_ON_ENDPOINT_INVALID_SEQ_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_INVALID_OUT_OF_MEMORY_EVENT_NBR             11
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_INVALID_OUT_OF_MEMORY_EVENT_MARKER          (SLI_CPC_SYSVIEW_ON_ENDPOINT_INVALID_OUT_OF_MEMORY_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_INVALID_OUT_OF_MEMORY_EVENT_ID              (SLI_CPC_SYSVIEW_ON_ENDPOINT_INVALID_OUT_OF_MEMORY_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_INVALID_OUT_OF_MEMORY_EVENT_MESSAGE         ON_ENDPOINT_STR_MESSAGE("Ran out of memory", SLI_CPC_SYSVIEW_ON_ENDPOINT_INVALID_OUT_OF_MEMORY_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_DESTINATION_UNREACHABLE_EVENT_NBR      12
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_DESTINATION_UNREACHABLE_EVENT_MARKER   (SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_DESTINATION_UNREACHABLE_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_DESTINATION_UNREACHABLE_EVENT_ID       (SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_DESTINATION_UNREACHABLE_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_DESTINATION_UNREACHABLE_EVENT_MESSAGE  ON_ENDPOINT_STR_MESSAGE("Transmit destination unreachable", SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_DESTINATION_UNREACHABLE_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_FRAME_TRANSMIT_COMPLETED_NBR                13
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_FRAME_TRANSMIT_COMPLETED_EVENT_MARKER       (SLI_CPC_SYSVIEW_ON_ENDPOINT_FRAME_TRANSMIT_COMPLETED_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_FRAME_TRANSMIT_COMPLETED_EVENT_ID           (SLI_CPC_SYSVIEW_ON_ENDPOINT_FRAME_TRANSMIT_COMPLETED_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_FRAME_TRANSMIT_COMPLETED_EVENT_MESSAGE      ON_ENDPOINT_STR_MESSAGE("Transmit completed", SLI_CPC_SYSVIEW_ON_ENDPOINT_FRAME_TRANSMIT_COMPLETED_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DATA_FRAME_EVENT_NBR                    14
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DATA_FRAME_EVENT_MARKER                 (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DATA_FRAME_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DATA_FRAME_EVENT_ID                     (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DATA_FRAME_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DATA_FRAME_EVENT_MESSAGE                ON_ENDPOINT_STR_MESSAGE("Received Data frame", SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DATA_FRAME_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DUPLICATE_DATA_FRAME_EVENT_NBR          15
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DUPLICATE_DATA_FRAME_EVENT_MARKER       (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DUPLICATE_DATA_FRAME_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DUPLICATE_DATA_FRAME_EVENT_ID           (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DUPLICATE_DATA_FRAME_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DUPLICATE_DATA_FRAME_EVENT_MESSAGE      ON_ENDPOINT_STR_MESSAGE("Received duplicated data", SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DUPLICATE_DATA_FRAME_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_SUPERVISORY_FRAME_EVENT_NBR             16
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_SUPERVISORY_FRAME_EVENT_MARKER          (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_SUPERVISORY_FRAME_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_SUPERVISORY_FRAME_EVENT_ID              (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_SUPERVISORY_FRAME_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_SUPERVISORY_FRAME_EVENT_MESSAGE         ON_ENDPOINT_STR_MESSAGE("Supervisory frame received", SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_SUPERVISORY_FRAME_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_SUPERVISORY_PROCESSED_EVENT_NBR         17
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_SUPERVISORY_PROCESSED_EVENT_MARKER      (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_SUPERVISORY_PROCESSED_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_SUPERVISORY_PROCESSED_EVENT_ID          (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_SUPERVISORY_PROCESSED_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_SUPERVISORY_PROCESSED_EVENT_MESSAGE     ON_ENDPOINT_STR_MESSAGE("Supervisory frame processed", SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_SUPERVISORY_PROCESSED_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_SEQ_MISMATCH_EVENT_NBR           18
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_SEQ_MISMATCH_EVENT_MARKER        (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_SEQ_MISMATCH_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_SEQ_MISMATCH_EVENT_ID            (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_SEQ_MISMATCH_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_SEQ_MISMATCH_EVENT_MESSAGE       ON_ENDPOINT_STR_MESSAGE("Received reject with sequence mismatch", SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_SEQ_MISMATCH_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_CHECSUM_MISMATCH_EVENT_NBR       19
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_CHECSUM_MISMATCH_EVENT_MARKER    (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_CHECSUM_MISMATCH_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_CHECSUM_MISMATCH_EVENT_ID        (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_CHECSUM_MISMATCH_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_CHECSUM_MISMATCH_EVENT_MESSAGE   ON_ENDPOINT_STR_MESSAGE("Received reject with checksum mismatch", SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_CHECSUM_MISMATCH_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_OUT_OF_MEMORY_EVENT_NBR          20
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_OUT_OF_MEMORY_EVENT_MARKER       (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_OUT_OF_MEMORY_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_OUT_OF_MEMORY_EVENT_ID           (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_OUT_OF_MEMORY_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_OUT_OF_MEMORY_EVENT_MESSAGE      ON_ENDPOINT_STR_MESSAGE("Received reject with out of memory", SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_OUT_OF_MEMORY_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_SECURITY_ISSUE_EVENT_NBR         21
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_SECURITY_ISSUE_EVENT_MARKER      (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_SECURITY_ISSUE_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_SECURITY_ISSUE_EVENT_ID          (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_SECURITY_ISSUE_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_SECURITY_ISSUE_EVENT_MESSAGE     ON_ENDPOINT_STR_MESSAGE("Received reject with security issue", SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_SECURITY_ISSUE_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_DESTINATION_UNREACHABLE_EVENT_NBR        22
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_DESTINATION_UNREACHABLE_EVENT_MARKER     (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_DESTINATION_UNREACHABLE_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_DESTINATION_UNREACHABLE_EVENT_ID         (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_DESTINATION_UNREACHABLE_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_DESTINATION_UNREACHABLE_EVENT_MESSAGE    ON_ENDPOINT_STR_MESSAGE("Received reject with destination unreachable", SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_DESTINATION_UNREACHABLE_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_FAULT_EVENT_NBR                  23
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_FAULT_EVENT_MARKER               (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_FAULT_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_FAULT_EVENT_ID                   (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_FAULT_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_FAULT_EVENT_MESSAGE              ON_ENDPOINT_STR_MESSAGE("Received reject with fault", SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_REJECT_FAULT_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_SUPERVISORY_DROPPED_EVENT_NBR           24
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_SUPERVISORY_DROPPED_EVENT_MARKER        (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_SUPERVISORY_DROPPED_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_SUPERVISORY_DROPPED_EVENT_ID            (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_SUPERVISORY_DROPPED_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_SUPERVISORY_DROPPED_EVENT_MESSAGE       ON_ENDPOINT_STR_MESSAGE("Dropped supervisory frame", SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_SUPERVISORY_DROPPED_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_ACK_EVENT_NBR                           26
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_ACK_EVENT_MARKER                        (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_ACK_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_ACK_EVENT_ID                            (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_ACK_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_ACK_EVENT_MESSAGE                       ON_ENDPOINT_STR_MESSAGE("Received ack", SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_ACK_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_ACK_EVENT_NBR                           28
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_ACK_EVENT_MARKER                        (SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_ACK_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_ACK_EVENT_ID                            (SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_ACK_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_ACK_EVENT_MESSAGE                       ON_ENDPOINT_STR_MESSAGE("Transmitted ack", SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_ACK_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RETXD_DATA_FRAME_EVENT_NBR                  29
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RETXD_DATA_FRAME_EVENT_MARKER               (SLI_CPC_SYSVIEW_ON_ENDPOINT_RETXD_DATA_FRAME_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RETXD_DATA_FRAME_EVENT_ID                   (SLI_CPC_SYSVIEW_ON_ENDPOINT_RETXD_DATA_FRAME_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RETXD_DATA_FRAME_EVENT_MESSAGE              ON_ENDPOINT_STR_MESSAGE("Retransmitted data frame", SLI_CPC_SYSVIEW_ON_ENDPOINT_RETXD_DATA_FRAME_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_CHECSUM_MISMATCH_EVENT_NBR       30
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_CHECSUM_MISMATCH_EVENT_MARKER    (SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_CHECSUM_MISMATCH_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_CHECSUM_MISMATCH_EVENT_ID        (SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_CHECSUM_MISMATCH_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_CHECSUM_MISMATCH_EVENT_MESSAGE   ON_ENDPOINT_STR_MESSAGE("Transmitted reject with checksum mismatch", SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_CHECSUM_MISMATCH_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_SEQ_MISMATCH_EVENT_NBR           31
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_SEQ_MISMATCH_EVENT_MARKER        (SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_SEQ_MISMATCH_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_SEQ_MISMATCH_EVENT_ID            (SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_SEQ_MISMATCH_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_SEQ_MISMATCH_EVENT_MESSAGE       ON_ENDPOINT_STR_MESSAGE("Transmitted reject with sequence mismatch", SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_SEQ_MISMATCH_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_OUT_OF_MEMORY_EVENT_NBR          32
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_OUT_OF_MEMORY_EVENT_MARKER       (SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_OUT_OF_MEMORY_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_OUT_OF_MEMORY_EVENT_ID           (SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_OUT_OF_MEMORY_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_OUT_OF_MEMORY_EVENT_MESSAGE      ON_ENDPOINT_STR_MESSAGE("Transmitted reject with out of memory", SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_OUT_OF_MEMORY_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_SECURITY_ISSUE_EVENT_NBR         33
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_SECURITY_ISSUE_EVENT_MARKER      (SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_SECURITY_ISSUE_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_SECURITY_ISSUE_EVENT_ID          (SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_SECURITY_ISSUE_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_SECURITY_ISSUE_EVENT_MESSAGE     ON_ENDPOINT_STR_MESSAGE("Transmitted reject with security issue", SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_SECURITY_ISSUE_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_FAULT_EVENT_NBR                  34
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_FAULT_EVENT_MARKER               (SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_FAULT_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_FAULT_EVENT_ID                   (SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_FAULT_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_FAULT_EVENT_MESSAGE              ON_ENDPOINT_STR_MESSAGE("Transmitted reject with fault", SLI_CPC_SYSVIEW_ON_ENDPOINT_TXD_REJECT_FAULT_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DATA_FRAME_QUEUED_EVENT_NBR             35
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DATA_FRAME_QUEUED_EVENT_MARKER          (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DATA_FRAME_QUEUED_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DATA_FRAME_QUEUED_EVENT_ID              (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DATA_FRAME_QUEUED_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DATA_FRAME_QUEUED_EVENT_MESSAGE         ON_ENDPOINT_STR_MESSAGE("Received data has been push to endpoint's receive queue", SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DATA_FRAME_QUEUED_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_FRAME_TRANSMIT_SUBMITTED_NBR                36
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_FRAME_TRANSMIT_SUBMITTED_EVENT_MARKER       (SLI_CPC_SYSVIEW_ON_ENDPOINT_FRAME_TRANSMIT_SUBMITTED_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_FRAME_TRANSMIT_SUBMITTED_EVENT_ID           (SLI_CPC_SYSVIEW_ON_ENDPOINT_FRAME_TRANSMIT_SUBMITTED_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_FRAME_TRANSMIT_SUBMITTED_EVENT_MESSAGE      ON_ENDPOINT_STR_MESSAGE("Transmit completed", SLI_CPC_SYSVIEW_ON_ENDPOINT_FRAME_TRANSMIT_SUBMITTED_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DATA_FRAME_DROPPED_EVENT_NBR             37
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DATA_FRAME_DROPPED_EVENT_MARKER          (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DATA_FRAME_DROPPED_EVENT_NBR * CPC_ENDPOINT_EVENT_ID_MULTIPLIER)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DATA_FRAME_DROPPED_EVENT_ID              (SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DATA_FRAME_DROPPED_EVENT_MARKER + CPC_ENDPOINT_EVENT_ID_OFFSET)
#define SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DATA_FRAME_DROPPED_EVENT_MESSAGE         ON_ENDPOINT_STR_MESSAGE("Received data has been dropped", SLI_CPC_SYSVIEW_ON_ENDPOINT_RXD_DATA_FRAME_DROPPED_EVENT_NBR, CPC_ENDPOINT_EVENT_ID_MULTIPLIER)

#endif // SLI_CPC_DEBUG_H
