/***************************************************************************//**
 * @brief SL_BTMESH_API command declarations
 *******************************************************************************
 * # License
 * <b>Copyright 2019 Silicon Laboratories Inc. www.silabs.com</b>
 *******************************************************************************
 *
 * The licensor of this software is Silicon Laboratories Inc. Your use of this
 * software is governed by the terms of Silicon Labs Master Software License
 * Agreement (MSLA) available at
 * www.silabs.com/about-us/legal/master-software-license-agreement. This
 * software is distributed to you in Source Code format and is governed by the
 * sections of the MSLA applicable to Source Code.
 *
 ******************************************************************************/


#ifndef SL_BTMESH_API_H
#define SL_BTMESH_API_H

#ifdef __cplusplus
extern "C" {
#endif

#include <string.h>
#include <stdint.h>
#include <stddef.h>
#include <stdbool.h>
#include "sl_status.h"
#include "sl_bgapi.h"

/**
 * @addtogroup sl_btmesh_event_masks BT Event System Event Masks
 * @{
 *
 * @brief Event Mask values used with Event System
 *
 * When the component `bluetooth_event_system_ipc` is included in the
 * application, the Bluetooth mesh stack events are published using the Event
 * System provided by the `event_system` component. The constants in this group
 * define the event masks used for Bluetooth mesh stack events.
 */

/**
 * @brief Event mask bit set in all public Bluetooth mesh events
 */
#define SL_BTMESH_EVENT_MASK_PUBLIC ((uint32_t) 0x01)

/** @} */ // end addtogroup sl_btmesh_event_masks




/**
 * @addtogroup sl_btmesh_node Mesh Node
 * @{
 *
 * @brief Mesh Node
 *
 * Bluetooth mesh stack API for unprovisioned devices and provisioned nodes.
 *
 * <b>Initialization:</b>
 *   - @ref sl_btmesh_node_init : Initialize node
 *   - @ref sl_btmesh_node_init_oob : Initialize node with out-of-band
 *     authentication
 *   - @ref sl_btmesh_evt_node_initialized : Node initialized
 *
 * <b>Provisioning a node:</b>
 *   - @ref sl_btmesh_node_get_uuid : Get device UUID
 *   - @ref sl_btmesh_node_start_unprov_beaconing : Start unprovisioned device
 *     beaconing
 *   - @ref sl_btmesh_node_stop_unprov_beaconing : Stop unprovisioned device
 *     beaconing
 *   - @ref sl_btmesh_evt_node_provisioning_started : Provisioning process has
 *     started
 *   - @ref sl_btmesh_evt_node_input_oob_request : Request to input out-of-band
 *     authentication data
 *   - @ref sl_btmesh_node_send_input_oob_request_response : Respond to input
 *     out-of-band authentication request
 *   - @ref sl_btmesh_evt_node_display_output_oob : Request to display output
 *     out-of-band authentication data
 *   - @ref sl_btmesh_evt_node_static_oob_request : Request for static
 *     out-of-band authentication data
 *   - @ref sl_btmesh_node_send_static_oob_request_response : Respond to static
 *     out-of-band authentication request
 *   - @ref sl_btmesh_evt_node_provisioned : Node has been provisioned
 *   - @ref sl_btmesh_evt_node_provisioning_failed : Provisioning process has
 *     failed
 *   - @ref sl_btmesh_node_set_provisioning_data : Pre-provision a device
 *
 * <b>Node Configuration:</b>
 *   - @ref sl_btmesh_evt_node_key_added : A cryptographic key has been added to
 *     the node
 *   - @ref sl_btmesh_evt_node_config_get : Node-wide configuration has been
 *     queried
 *   - @ref sl_btmesh_evt_node_config_set : Node-wide configuration has been
 *     modified
 *   - @ref sl_btmesh_evt_node_model_config_changed : Model configuration has
 *     been modified
 *   - @ref sl_btmesh_node_reset : Factory reset mesh node
 *
 * <b>Note on Bluetooth mesh addresses</b>
 *
 * Bluetooth mesh address space is divided into sections containing ranges of
 * addresses of various types. Different address types are used in different
 * contexts. Some requests accept only certain address types.
 *
 * The address types are as follows:
 *   - <b>0x0000 Unassigned address:</b> represents an address that has not been
 *     set
 *   - <b>0x0001..0x7fff Unicast addresses</b> are allocated by the Provisioner
 *     to provisioned nodes. Each element of a node has its own unicast address.
 *   - <b>0x8000..0xbfff Virtual addresses</b> are 16-bit shorthand for 128-bit
 *     label UUIDs which are pre-allocated to specific purposes in relevant
 *     Bluetooth SIG specifications. Virtual addresses can typically be used in
 *     the same context as group addresses. Some commands require specifying the
 *     full label UUID instead of the virtual address shorthand.
 *   - <b>0xc000..0xffef Group addresses</b> are allocated by the Provisioner
 *     for multicast communication.
 *   - <b>0xfff0..0xffff Fixed group addresses</b> are allocated in the Mesh
 *     specification for multicast communication in a particular context. They
 *     can be used in the same context as regular group addresses. The following
 *     addresses are currently defined:
 *       - 0xfffc All-proxies broadcast address
 *       - 0xfffd All-friends broadcast address
 *       - 0xfffe All-relays broadcast address
 *       - 0xffff All-nodes broadcast address
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_node_init_id                                       0x00140028
#define sl_btmesh_cmd_node_set_exportable_keys_id                        0x24140028
#define sl_btmesh_cmd_node_start_unprov_beaconing_id                     0x01140028
#define sl_btmesh_cmd_node_stop_unprov_beaconing_id                      0x16140028
#define sl_btmesh_cmd_node_get_rssi_id                                   0x17140028
#define sl_btmesh_cmd_node_send_input_oob_request_response_id            0x02140028
#define sl_btmesh_cmd_node_get_uuid_id                                   0x03140028
#define sl_btmesh_cmd_node_set_provisioning_data_id                      0x04140028
#define sl_btmesh_cmd_node_init_oob_id                                   0x05140028
#define sl_btmesh_cmd_node_set_ivrecovery_mode_id                        0x06140028
#define sl_btmesh_cmd_node_get_ivrecovery_mode_id                        0x07140028
#define sl_btmesh_cmd_node_set_net_relay_delay_id                        0x0b140028
#define sl_btmesh_cmd_node_get_net_relay_delay_id                        0x0c140028
#define sl_btmesh_cmd_node_get_ivupdate_state_id                         0x0d140028
#define sl_btmesh_cmd_node_request_ivupdate_id                           0x0e140028
#define sl_btmesh_cmd_node_get_seq_remaining_id                          0x0f140028
#define sl_btmesh_cmd_node_save_replay_protection_list_id                0x10140028
#define sl_btmesh_cmd_node_set_uuid_id                                   0x11140028
#define sl_btmesh_cmd_node_get_replay_protection_list_status_id          0x14140028
#define sl_btmesh_cmd_node_get_element_address_id                        0x12140028
#define sl_btmesh_cmd_node_send_static_oob_request_response_id           0x13140028
#define sl_btmesh_cmd_node_reset_id                                      0x15140028
#define sl_btmesh_cmd_node_set_beacon_reporting_id                       0x18140028
#define sl_btmesh_cmd_node_set_iv_update_age_id                          0x19140028
#define sl_btmesh_cmd_node_get_key_count_id                              0x1a140028
#define sl_btmesh_cmd_node_get_key_id                                    0x1b140028
#define sl_btmesh_cmd_node_get_networks_id                               0x1c140028
#define sl_btmesh_cmd_node_get_element_seqnum_id                         0x1d140028
#define sl_btmesh_cmd_node_set_model_option_id                           0x1e140028
#define sl_btmesh_cmd_node_get_local_dcd_id                              0x1f140028
#define sl_btmesh_cmd_node_power_off_id                                  0x21140028
#define sl_btmesh_cmd_node_set_adv_phy_id                                0x22140028
#define sl_btmesh_cmd_node_get_adv_phy_id                                0x23140028
#define sl_btmesh_cmd_node_set_provisioning_algorithms_id                0x25140028
#define sl_btmesh_cmd_node_init_provisioning_records_id                  0x26140028
#define sl_btmesh_cmd_node_set_unprov_beaconing_adv_interval_id          0x28140028
#define sl_btmesh_cmd_node_set_proxy_service_adv_interval_id             0x29140028
#define sl_btmesh_cmd_node_set_provisioning_service_adv_interval_id      0x30140028
#define sl_btmesh_cmd_node_get_local_model_metadata_page_id              0x20140028
#define sl_btmesh_cmd_node_save_proxy_solicitation_rpl_id                0x31140028
#define sl_btmesh_cmd_node_get_proxy_solicitation_rpl_status_id          0x32140028
#define sl_btmesh_cmd_node_set_oob_uri_id                                0x33140028
#define sl_btmesh_cmd_node_get_oob_uri_id                                0x34140028
#define sl_btmesh_cmd_node_set_proxy_service_uuid_id                     0x35140028
#define sl_btmesh_cmd_node_test_identity_id                              0x36140028
#define sl_btmesh_cmd_node_set_proxy_service_scan_response_id            0x37140028
#define sl_btmesh_cmd_node_clear_proxy_service_scan_response_id          0x38140028
#define sl_btmesh_cmd_node_set_provisioning_service_scan_response_id     0x39140028
#define sl_btmesh_cmd_node_clear_provisioning_service_scan_response_id   0x3a140028
#define sl_btmesh_cmd_node_compare_dcd_id                                0x3b140028
#define sl_btmesh_rsp_node_init_id                                       0x00140028
#define sl_btmesh_rsp_node_set_exportable_keys_id                        0x24140028
#define sl_btmesh_rsp_node_start_unprov_beaconing_id                     0x01140028
#define sl_btmesh_rsp_node_stop_unprov_beaconing_id                      0x16140028
#define sl_btmesh_rsp_node_get_rssi_id                                   0x17140028
#define sl_btmesh_rsp_node_send_input_oob_request_response_id            0x02140028
#define sl_btmesh_rsp_node_get_uuid_id                                   0x03140028
#define sl_btmesh_rsp_node_set_provisioning_data_id                      0x04140028
#define sl_btmesh_rsp_node_init_oob_id                                   0x05140028
#define sl_btmesh_rsp_node_set_ivrecovery_mode_id                        0x06140028
#define sl_btmesh_rsp_node_get_ivrecovery_mode_id                        0x07140028
#define sl_btmesh_rsp_node_set_net_relay_delay_id                        0x0b140028
#define sl_btmesh_rsp_node_get_net_relay_delay_id                        0x0c140028
#define sl_btmesh_rsp_node_get_ivupdate_state_id                         0x0d140028
#define sl_btmesh_rsp_node_request_ivupdate_id                           0x0e140028
#define sl_btmesh_rsp_node_get_seq_remaining_id                          0x0f140028
#define sl_btmesh_rsp_node_save_replay_protection_list_id                0x10140028
#define sl_btmesh_rsp_node_set_uuid_id                                   0x11140028
#define sl_btmesh_rsp_node_get_replay_protection_list_status_id          0x14140028
#define sl_btmesh_rsp_node_get_element_address_id                        0x12140028
#define sl_btmesh_rsp_node_send_static_oob_request_response_id           0x13140028
#define sl_btmesh_rsp_node_reset_id                                      0x15140028
#define sl_btmesh_rsp_node_set_beacon_reporting_id                       0x18140028
#define sl_btmesh_rsp_node_set_iv_update_age_id                          0x19140028
#define sl_btmesh_rsp_node_get_key_count_id                              0x1a140028
#define sl_btmesh_rsp_node_get_key_id                                    0x1b140028
#define sl_btmesh_rsp_node_get_networks_id                               0x1c140028
#define sl_btmesh_rsp_node_get_element_seqnum_id                         0x1d140028
#define sl_btmesh_rsp_node_set_model_option_id                           0x1e140028
#define sl_btmesh_rsp_node_get_local_dcd_id                              0x1f140028
#define sl_btmesh_rsp_node_power_off_id                                  0x21140028
#define sl_btmesh_rsp_node_set_adv_phy_id                                0x22140028
#define sl_btmesh_rsp_node_get_adv_phy_id                                0x23140028
#define sl_btmesh_rsp_node_set_provisioning_algorithms_id                0x25140028
#define sl_btmesh_rsp_node_init_provisioning_records_id                  0x26140028
#define sl_btmesh_rsp_node_set_unprov_beaconing_adv_interval_id          0x28140028
#define sl_btmesh_rsp_node_set_proxy_service_adv_interval_id             0x29140028
#define sl_btmesh_rsp_node_set_provisioning_service_adv_interval_id      0x30140028
#define sl_btmesh_rsp_node_get_local_model_metadata_page_id              0x20140028
#define sl_btmesh_rsp_node_save_proxy_solicitation_rpl_id                0x31140028
#define sl_btmesh_rsp_node_get_proxy_solicitation_rpl_status_id          0x32140028
#define sl_btmesh_rsp_node_set_oob_uri_id                                0x33140028
#define sl_btmesh_rsp_node_get_oob_uri_id                                0x34140028
#define sl_btmesh_rsp_node_set_proxy_service_uuid_id                     0x35140028
#define sl_btmesh_rsp_node_test_identity_id                              0x36140028
#define sl_btmesh_rsp_node_set_proxy_service_scan_response_id            0x37140028
#define sl_btmesh_rsp_node_clear_proxy_service_scan_response_id          0x38140028
#define sl_btmesh_rsp_node_set_provisioning_service_scan_response_id     0x39140028
#define sl_btmesh_rsp_node_clear_provisioning_service_scan_response_id   0x3a140028
#define sl_btmesh_rsp_node_compare_dcd_id                                0x3b140028

/**
 * @brief Flags for allowed provisioning algorithms during provisioning, which
 * use a bitmap so that multiple algorithms can be supported.
 */
typedef enum
{
  sl_btmesh_node_algorithm_flag_ecdh_p256_cmac_aes128_aes_ccm = 0x1, /**< (0x1)
                                                                          ECDH
                                                                          based
                                                                          provisioning,
                                                                          authenticated
                                                                          using
                                                                          AES-CMAC,
                                                                          and
                                                                          protected
                                                                          with
                                                                          AES-CCM */
  sl_btmesh_node_algorithm_flag_ecdh_p256_hmac_sha256_aes_ccm = 0x2  /**< (0x2)
                                                                          ECDH
                                                                          based
                                                                          provisioning,
                                                                          authenticated
                                                                          using
                                                                          HMAC-SHA-256,
                                                                          and
                                                                          protected
                                                                          with
                                                                          AES-CCM */
} sl_btmesh_node_algorithm_flag_t;

/**
 * @brief Flags for supported OOB authentication methods during provisioning,
 * which use a bitmap so that multiple methods can be supported.
 */
typedef enum
{
  sl_btmesh_node_auth_method_flag_none   = 0x1, /**< (0x1) Authentication
                                                     without OOB is supported */
  sl_btmesh_node_auth_method_flag_static = 0x2, /**< (0x2) Static OOB data
                                                     authentication is supported */
  sl_btmesh_node_auth_method_flag_output = 0x4, /**< (0x4) Output OOB
                                                     authentication is supported */
  sl_btmesh_node_auth_method_flag_input  = 0x8  /**< (0x8) Input OOB
                                                     authentication is supported */
} sl_btmesh_node_auth_method_flag_t;

/**
 * @brief Flags for supported input OOB actions during provisioning, which use a
 * bitmap so that multiple actions can be supported.
 */
typedef enum
{
  sl_btmesh_node_oob_input_action_flag_push    = 0x1, /**< (0x1) Push a button
                                                           on the device. */
  sl_btmesh_node_oob_input_action_flag_twist   = 0x2, /**< (0x2) Twist a dial on
                                                           the device. */
  sl_btmesh_node_oob_input_action_flag_numeric = 0x4, /**< (0x4) Input a numeric
                                                           authentication code. */
  sl_btmesh_node_oob_input_action_flag_alpha   = 0x8  /**< (0x8) Input an
                                                           alphanumeric
                                                           authentication code. */
} sl_btmesh_node_oob_input_action_flag_t;

/**
 * @brief Indicate the input OOB action selected by the Provisioner during
 * provisioning of the device.
 */
typedef enum
{
  sl_btmesh_node_oob_input_action_push    = 0x0, /**< (0x0) Push a button on the
                                                      device. */
  sl_btmesh_node_oob_input_action_twist   = 0x1, /**< (0x1) Twist a dial on the
                                                      device. */
  sl_btmesh_node_oob_input_action_numeric = 0x2, /**< (0x2) Input a numeric
                                                      authentication code. */
  sl_btmesh_node_oob_input_action_alpha   = 0x3  /**< (0x3) Input an
                                                      alphanumeric
                                                      authentication code. */
} sl_btmesh_node_oob_input_action_t;

/**
 * @brief Flags for supported output OOB actions during provisioning, which use
 * a bitmap so that multiple actions can be supported.
 */
typedef enum
{
  sl_btmesh_node_oob_output_action_flag_blink   = 0x1,  /**< (0x1) Blink a
                                                             light. */
  sl_btmesh_node_oob_output_action_flag_beep    = 0x2,  /**< (0x2) Emit a sound. */
  sl_btmesh_node_oob_output_action_flag_vibrate = 0x4,  /**< (0x4) Vibrate the
                                                             device. */
  sl_btmesh_node_oob_output_action_flag_numeric = 0x8,  /**< (0x8) Output a
                                                             numeric
                                                             authentication
                                                             code. */
  sl_btmesh_node_oob_output_action_flag_alpha   = 0x10  /**< (0x10) Output an
                                                             alphanumeric
                                                             authentication
                                                             code. */
} sl_btmesh_node_oob_output_action_flag_t;

/**
 * @brief Indicate the output OOB action selected by the Provisioner during
 * provisioning of the device.
 */
typedef enum
{
  sl_btmesh_node_oob_output_action_blink   = 0x0, /**< (0x0) Blink a light. */
  sl_btmesh_node_oob_output_action_beep    = 0x1, /**< (0x1) Emit a sound. */
  sl_btmesh_node_oob_output_action_vibrate = 0x2, /**< (0x2) Vibrate the device. */
  sl_btmesh_node_oob_output_action_numeric = 0x3, /**< (0x3) Output a numeric
                                                       authentication code. */
  sl_btmesh_node_oob_output_action_alpha   = 0x4  /**< (0x4) Output an
                                                       alphanumeric
                                                       authentication code. */
} sl_btmesh_node_oob_output_action_t;

/**
 * @brief Specify the type of a key in key manipulation commands.
 */
typedef enum
{
  sl_btmesh_node_key_type_net = 0x0, /**< (0x0) Network key */
  sl_btmesh_node_key_type_app = 0x1  /**< (0x1) Application key */
} sl_btmesh_node_key_type_t;

/**
 * @brief Specify the state to which a Configuration Client/Server command/event
 * applies.
 */
typedef enum
{
  sl_btmesh_node_dcd         = 0x8008, /**< (0x8008) Device Composition Data */
  sl_btmesh_node_beacon      = 0x8009, /**< (0x8009) Status of broadcasting
                                            Secure Network Beacons */
  sl_btmesh_node_default_ttl = 0x800c, /**< (0x800c) Default Time-To-Live for
                                            messages */
  sl_btmesh_node_friendship  = 0x800f, /**< (0x800f) Friend status */
  sl_btmesh_node_gatt_proxy  = 0x8012, /**< (0x8012) GATT proxy status */
  sl_btmesh_node_key_refresh = 0x8015, /**< (0x8015) Key refresh status */
  sl_btmesh_node_relay       = 0x8023, /**< (0x8023) Relay status */
  sl_btmesh_node_identity    = 0x8042, /**< (0x8042) Identity status */
  sl_btmesh_node_nettx       = 0x8024  /**< (0x8024) Network transmit status */
} sl_btmesh_node_config_state_t;

/**
 * @brief Indicates the type of differences found between a given composition
 * data page content and the current one. Available details are page-specific
 */
typedef enum
{
  sl_btmesh_node_composition_changed  = 0x1,    /**< (0x1) Page content differs
                                                     in any way. */
  sl_btmesh_node_cid_changed          = 0x2,    /**< (0x2) Company Identifier
                                                     changed */
  sl_btmesh_node_pid_changed          = 0x4,    /**< (0x4) Product Identifier
                                                     changed */
  sl_btmesh_node_vid_changed          = 0x8,    /**< (0x8) Version Identifier
                                                     changed */
  sl_btmesh_node_crpl_changed         = 0x10,   /**< (0x10) Replay Protection
                                                     List size changed */
  sl_btmesh_node_features_changed     = 0x20,   /**< (0x20) Features changed
                                                     (Relay, Proxy, Friend or
                                                     Low Power */
  sl_btmesh_node_element_loc_changed  = 0x40,   /**< (0x40) Element location
                                                     changed for some element */
  sl_btmesh_node_sig_model_added      = 0x80,   /**< (0x80) SIG model entry
                                                     added */
  sl_btmesh_node_sig_model_removed    = 0x100,  /**< (0x100) SIG model entry
                                                     removed */
  sl_btmesh_node_vendor_model_added   = 0x200,  /**< (0x200) Vendor model added */
  sl_btmesh_node_vendor_model_removed = 0x400,  /**< (0x400) Vendor model
                                                     removed */
  sl_btmesh_node_element_added        = 0x800,  /**< (0x800) Element added */
  sl_btmesh_node_element_removed      = 0x1000  /**< (0x1000) Element removed */
} sl_btmesh_node_dcd_comparison_t;

/**
 * @addtogroup sl_btmesh_node_message_flags Message Flags
 * @{
 *

 */

/** Message does not have flags */
#define SL_BTMESH_NODE_MESSAGE_FLAG_NONE            0x0       

/** Message is an access layer reply to a unicast request */
#define SL_BTMESH_NODE_MESSAGE_FLAG_REPLY           0x1       

/** Message is an access layer reply to a multicast request */
#define SL_BTMESH_NODE_MESSAGE_FLAG_MULTICAST_REPLY 0x2       

/** Message is either received with TTL=0 or is a response to such a message and
 * should be sent out with TTL=0 */
#define SL_BTMESH_NODE_MESSAGE_FLAG_TTL0            0x4       

/** Message is segmented in transport layer, can be used to force the use of
 * transport layer segmentation */
#define SL_BTMESH_NODE_MESSAGE_FLAG_SEGTRANS        0x10      

/** Use friendship credentials instead of normal credentials when sending data;
 * used when publishing with friendship credential flag on in model publish
 * settings */
#define SL_BTMESH_NODE_MESSAGE_FLAG_FRIENDCRED      0x20      

/** Use extended packet size instead of standard */
#define SL_BTMESH_NODE_MESSAGE_FLAG_EXTENDED        0x40      

/** long delay needed in sending two messages in reply that need to be sent in
 * order */
#define SL_BTMESH_NODE_MESSAGE_FLAG_LONG_DELAY      0x80      

/** @} */ // end Message Flags

/**
 * @addtogroup sl_btmesh_evt_node_initialized sl_btmesh_evt_node_initialized
 * @{
 * @brief Node is initialized and operational.
 */

/** @brief Identifier of the initialized event */
#define sl_btmesh_evt_node_initialized_id                                0x001400a8

/***************************************************************************//**
 * @brief Data structure of the initialized event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_node_initialized_s
{
  uint8_t  provisioned; /**< 1 if node is provisioned into a network, 0 if
                             unprovisioned. */
  uint16_t address;     /**< Unicast address of the primary element of the node.
                             Ignored if unprovisioned. Secondary elements have
                             been assigned sequential unicast addresses
                             following the primary element address. */
  uint32_t iv_index;    /**< IV index for the first network of the node, ignore
                             if unprovisioned. */
});

typedef struct sl_btmesh_evt_node_initialized_s sl_btmesh_evt_node_initialized_t;

/** @} */ // end addtogroup sl_btmesh_evt_node_initialized

/**
 * @addtogroup sl_btmesh_evt_node_provisioned sl_btmesh_evt_node_provisioned
 * @{
 * @brief The node has received provisioning data (address allocation and a
 * network key) from the Provisioner
 *
 * A @ref sl_btmesh_evt_node_key_added event will follow for the network key.
 *
 * The node is now ready for further configuration by the Provisioner but is not
 * yet ready for communication with other nodes in the network (it does not have
 * any application keys and its models have not been set up).
 */

/** @brief Identifier of the provisioned event */
#define sl_btmesh_evt_node_provisioned_id                                0x011400a8

/***************************************************************************//**
 * @brief Data structure of the provisioned event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_node_provisioned_s
{
  uint16_t address;  /**< The unicast address that the Provisioner allocated for
                          the primary element of the node. Secondary elements
                          have been assigned sequentially following unicast
                          addresses. */
  uint32_t iv_index; /**< Current IV index of the provisioned network */
});

typedef struct sl_btmesh_evt_node_provisioned_s sl_btmesh_evt_node_provisioned_t;

/** @} */ // end addtogroup sl_btmesh_evt_node_provisioned

/**
 * @addtogroup sl_btmesh_evt_node_config_get sl_btmesh_evt_node_config_get
 * @{
 * @brief Informative; Configuration Client requested the current value of a
 * State in the Configuration Server Model.
 */

/** @brief Identifier of the config_get event */
#define sl_btmesh_evt_node_config_get_id                                 0x021400a8

/***************************************************************************//**
 * @brief Data structure of the config_get event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_node_config_get_s
{
  uint16_t netkey_index; /**< The network key index of the network to which the
                              command applies. 0xffff for node-wide states. */
  uint16_t id;           /**< Enum @ref sl_btmesh_node_config_state_t. Specifies
                              to which State the command applies */
});

typedef struct sl_btmesh_evt_node_config_get_s sl_btmesh_evt_node_config_get_t;

/** @} */ // end addtogroup sl_btmesh_evt_node_config_get

/**
 * @addtogroup sl_btmesh_evt_node_config_set sl_btmesh_evt_node_config_set
 * @{
 * @brief Informative; Configuration Client changes the State in the
 * Configuration Server Model.
 */

/** @brief Identifier of the config_set event */
#define sl_btmesh_evt_node_config_set_id                                 0x031400a8

/***************************************************************************//**
 * @brief Data structure of the config_set event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_node_config_set_s
{
  uint16_t   netkey_index; /**< The network key index of the network to which
                                the command applies. 0xffff for node-wide
                                states. */
  uint16_t   id;           /**< Enum @ref sl_btmesh_node_config_state_t.
                                Specifies to which state the command applies */
  uint8array value;        /**< The new value */
});

typedef struct sl_btmesh_evt_node_config_set_s sl_btmesh_evt_node_config_set_t;

/** @} */ // end addtogroup sl_btmesh_evt_node_config_set

/**
 * @addtogroup sl_btmesh_evt_node_display_output_oob sl_btmesh_evt_node_display_output_oob
 * @{
 * @brief Display output OOB data so Provisioner can input it.
 */

/** @brief Identifier of the display_output_oob event */
#define sl_btmesh_evt_node_display_output_oob_id                         0x041400a8

/***************************************************************************//**
 * @brief Data structure of the display_output_oob event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_node_display_output_oob_s
{
  uint8_t    output_action; /**< Enum @ref sl_btmesh_node_oob_output_action_t.
                                 Selected output action */
  uint8_t    output_size;   /**< Size of data to output in characters. */
  uint8array data;          /**< Raw 16-byte or 32-byte array containing the
                                 output authentication data. The length of the
                                 data depends on the chosen provisioning
                                 algorithm; see the @ref
                                 sl_btmesh_evt_node_start_received event for
                                 details on how the stack indicates the
                                 Provisioner's choice of provisioning algorithm
                                 to the application.

                                 Note that the raw data is encoded in the manner
                                 described in the Mesh Protocol 1.1 specifcation
                                 Section 5.4.2.4, and must be converted to a
                                 human-readable format for displaying. Whether
                                 the data should be treated as Numeric or
                                 Alphanumeric depends on the selected output
                                 action. */
});

typedef struct sl_btmesh_evt_node_display_output_oob_s sl_btmesh_evt_node_display_output_oob_t;

/** @} */ // end addtogroup sl_btmesh_evt_node_display_output_oob

/**
 * @addtogroup sl_btmesh_evt_node_input_oob_request sl_btmesh_evt_node_input_oob_request
 * @{
 * @brief The Provisioner is displaying an out of band authentication value
 *
 * The application on the node should provide the value to the Bluetooth mesh
 * stack using the @ref sl_btmesh_node_send_input_oob_request_response command.
 */

/** @brief Identifier of the input_oob_request event */
#define sl_btmesh_evt_node_input_oob_request_id                          0x051400a8

/***************************************************************************//**
 * @brief Data structure of the input_oob_request event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_node_input_oob_request_s
{
  uint8_t input_action; /**< Enum @ref sl_btmesh_node_oob_input_action_t.
                             Selected input action */
  uint8_t input_size;   /**< Size of data in the input in characters. */
});

typedef struct sl_btmesh_evt_node_input_oob_request_s sl_btmesh_evt_node_input_oob_request_t;

/** @} */ // end addtogroup sl_btmesh_evt_node_input_oob_request

/**
 * @addtogroup sl_btmesh_evt_node_provisioning_started sl_btmesh_evt_node_provisioning_started
 * @{
 * @brief Provisioner has started provisioning this node.
 */

/** @brief Identifier of the provisioning_started event */
#define sl_btmesh_evt_node_provisioning_started_id                       0x061400a8

/***************************************************************************//**
 * @brief Data structure of the provisioning_started event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_node_provisioning_started_s
{
  uint16_t result; /**< SL_STATUS_OK if successful. Error code otherwise. */
});

typedef struct sl_btmesh_evt_node_provisioning_started_s sl_btmesh_evt_node_provisioning_started_t;

/** @} */ // end addtogroup sl_btmesh_evt_node_provisioning_started

/**
 * @addtogroup sl_btmesh_evt_node_provisioning_failed sl_btmesh_evt_node_provisioning_failed
 * @{
 * @brief Provisioning the node has failed.
 */

/** @brief Identifier of the provisioning_failed event */
#define sl_btmesh_evt_node_provisioning_failed_id                        0x071400a8

/***************************************************************************//**
 * @brief Data structure of the provisioning_failed event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_node_provisioning_failed_s
{
  uint16_t result; /**< SL_STATUS_OK if successful. Error code otherwise. */
});

typedef struct sl_btmesh_evt_node_provisioning_failed_s sl_btmesh_evt_node_provisioning_failed_t;

/** @} */ // end addtogroup sl_btmesh_evt_node_provisioning_failed

/**
 * @addtogroup sl_btmesh_evt_node_key_added sl_btmesh_evt_node_key_added
 * @{
 * @brief Received when a Configuration Client has deployed a new network or
 * application key to the node.
 */

/** @brief Identifier of the key_added event */
#define sl_btmesh_evt_node_key_added_id                                  0x081400a8

/***************************************************************************//**
 * @brief Data structure of the key_added event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_node_key_added_s
{
  uint8_t  type;         /**< Type of the new key. Values are as follows:
                                - <b>0x00:</b> Network key
                                - <b>0x01:</b> Application key */
  uint16_t index;        /**< Key index of the new key */
  uint16_t netkey_index; /**< Network key index to which the application key is
                              bound, which is ignored for network keys */
});

typedef struct sl_btmesh_evt_node_key_added_s sl_btmesh_evt_node_key_added_t;

/** @} */ // end addtogroup sl_btmesh_evt_node_key_added

/**
 * @addtogroup sl_btmesh_evt_node_model_config_changed sl_btmesh_evt_node_model_config_changed
 * @{
 * @brief Informative
 *
 * This event notifies that a remote Configuration Client has changed the
 * configuration of a local model.
 */

/** @brief Identifier of the model_config_changed event */
#define sl_btmesh_evt_node_model_config_changed_id                       0x091400a8

/***************************************************************************//**
 * @brief Data structure of the model_config_changed event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_node_model_config_changed_s
{
  uint8_t  node_config_state; /**< The configuration state which has changed.
                                   Values are as follows:
                                     - <b>0x00:</b> Model application key
                                       bindings
                                     - <b>0x01:</b> Model publication parameters
                                     - <b>0x02:</b> Model subscription list */
  uint16_t element_address;   /**< Address of the element which contains the
                                   model */
  uint16_t vendor_id;         /**< Vendor ID of the model; value 0xffff is used
                                   for Bluetooth SIG models. */
  uint16_t model_id;          /**< Model ID of the model */
});

typedef struct sl_btmesh_evt_node_model_config_changed_s sl_btmesh_evt_node_model_config_changed_t;

/** @} */ // end addtogroup sl_btmesh_evt_node_model_config_changed

/**
 * @addtogroup sl_btmesh_evt_node_reset sl_btmesh_evt_node_reset
 * @{
 * @brief Provisioner has instructed the node to reset
 *
 * This event is generated when the Provisioner has ordered the node to be
 * reset. Stack data has already been reset. This event is generated to inform
 * the application that it should do its own cleanup duties and reset the
 * hardware.
 */

/** @brief Identifier of the reset event */
#define sl_btmesh_evt_node_reset_id                                      0x0a1400a8

/** @} */ // end addtogroup sl_btmesh_evt_node_reset

/**
 * @addtogroup sl_btmesh_evt_node_ivrecovery_needed sl_btmesh_evt_node_ivrecovery_needed
 * @{
 * @brief Network IV index recovery needed
 *
 * This event is generated when the node detects the network IV index is too far
 * in the future to be automatically updated. See @ref
 * sl_btmesh_node_set_ivrecovery_mode command.
 */

/** @brief Identifier of the ivrecovery_needed event */
#define sl_btmesh_evt_node_ivrecovery_needed_id                          0x0b1400a8

/***************************************************************************//**
 * @brief Data structure of the ivrecovery_needed event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_node_ivrecovery_needed_s
{
  uint32_t node_iv_index;    /**< Current IV index of the node */
  uint32_t network_iv_index; /**< Received network IV index */
});

typedef struct sl_btmesh_evt_node_ivrecovery_needed_s sl_btmesh_evt_node_ivrecovery_needed_t;

/** @} */ // end addtogroup sl_btmesh_evt_node_ivrecovery_needed

/**
 * @addtogroup sl_btmesh_evt_node_changed_ivupdate_state sl_btmesh_evt_node_changed_ivupdate_state
 * @{
 * @brief Network IV index update state has changed.
 */

/** @brief Identifier of the changed_ivupdate_state event */
#define sl_btmesh_evt_node_changed_ivupdate_state_id                     0x0c1400a8

/***************************************************************************//**
 * @brief Data structure of the changed_ivupdate_state event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_node_changed_ivupdate_state_s
{
  uint32_t iv_index; /**< Current IV index */
  uint8_t  state;    /**< Indicates whether the IV index update is ongoing (1)
                          or not (0). */
});

typedef struct sl_btmesh_evt_node_changed_ivupdate_state_s sl_btmesh_evt_node_changed_ivupdate_state_t;

/** @} */ // end addtogroup sl_btmesh_evt_node_changed_ivupdate_state

/**
 * @addtogroup sl_btmesh_evt_node_static_oob_request sl_btmesh_evt_node_static_oob_request
 * @{
 * @brief Static out of band authentication data is needed in the provisioning
 *
 * The application on the node should provide the value to the Bluetooth mesh
 * stack using the @ref sl_btmesh_node_send_static_oob_request_response command.
 */

/** @brief Identifier of the static_oob_request event */
#define sl_btmesh_evt_node_static_oob_request_id                         0x0d1400a8

/** @} */ // end addtogroup sl_btmesh_evt_node_static_oob_request

/**
 * @addtogroup sl_btmesh_evt_node_key_removed sl_btmesh_evt_node_key_removed
 * @{
 * @brief Received when a Configuration Client removes a network or application
 * key from the node.
 */

/** @brief Identifier of the key_removed event */
#define sl_btmesh_evt_node_key_removed_id                                0x0e1400a8

/***************************************************************************//**
 * @brief Data structure of the key_removed event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_node_key_removed_s
{
  uint8_t  type;         /**< Type of the removed key. Values are as follows:
                                - <b>0x00:</b> Network key
                                - <b>0x01:</b> Application key */
  uint16_t index;        /**< Key index of the removed key */
  uint16_t netkey_index; /**< Network key index to which the application key is
                              bound, which is ignored for network keys */
});

typedef struct sl_btmesh_evt_node_key_removed_s sl_btmesh_evt_node_key_removed_t;

/** @} */ // end addtogroup sl_btmesh_evt_node_key_removed

/**
 * @addtogroup sl_btmesh_evt_node_key_updated sl_btmesh_evt_node_key_updated
 * @{
 * @brief Received when a Configuration Client updates a network or application
 * key of the node.
 */

/** @brief Identifier of the key_updated event */
#define sl_btmesh_evt_node_key_updated_id                                0x0f1400a8

/***************************************************************************//**
 * @brief Data structure of the key_updated event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_node_key_updated_s
{
  uint8_t  type;         /**< Type of the updated key. Values are as follows:
                                - <b>0x00:</b> Network key
                                - <b>0x01:</b> Application key */
  uint16_t index;        /**< Key index of the updated key */
  uint16_t netkey_index; /**< Network key index to which the application key is
                              bound. Ignore this value if the event is for
                              network key updates. */
});

typedef struct sl_btmesh_evt_node_key_updated_s sl_btmesh_evt_node_key_updated_t;

/** @} */ // end addtogroup sl_btmesh_evt_node_key_updated

/**
 * @addtogroup sl_btmesh_evt_node_heartbeat sl_btmesh_evt_node_heartbeat
 * @{
 * @brief Indicates reception of heartbeat message.
 */

/** @brief Identifier of the heartbeat event */
#define sl_btmesh_evt_node_heartbeat_id                                  0x101400a8

/***************************************************************************//**
 * @brief Data structure of the heartbeat event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_node_heartbeat_s
{
  uint16_t src_addr; /**< Source address for the heartbeat message */
  uint16_t dst_addr; /**< Destination address for the heartbeat message */
  uint8_t  hops;     /**< Hops traveled by the heartbeat message */
});

typedef struct sl_btmesh_evt_node_heartbeat_s sl_btmesh_evt_node_heartbeat_t;

/** @} */ // end addtogroup sl_btmesh_evt_node_heartbeat

/**
 * @addtogroup sl_btmesh_evt_node_heartbeat_start sl_btmesh_evt_node_heartbeat_start
 * @{
 * @brief Indicates start of heartbeat reception.
 */

/** @brief Identifier of the heartbeat_start event */
#define sl_btmesh_evt_node_heartbeat_start_id                            0x111400a8

/***************************************************************************//**
 * @brief Data structure of the heartbeat_start event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_node_heartbeat_start_s
{
  uint16_t src_addr;   /**< Source address for the heartbeat message */
  uint16_t dst_addr;   /**< Destination address for the heartbeat message */
  uint32_t period_sec; /**< Heartbeat subscription period in seconds. */
});

typedef struct sl_btmesh_evt_node_heartbeat_start_s sl_btmesh_evt_node_heartbeat_start_t;

/** @} */ // end addtogroup sl_btmesh_evt_node_heartbeat_start

/**
 * @addtogroup sl_btmesh_evt_node_heartbeat_stop sl_btmesh_evt_node_heartbeat_stop
 * @{
 * @brief Indicates end of heartbeat reception.
 */

/** @brief Identifier of the heartbeat_stop event */
#define sl_btmesh_evt_node_heartbeat_stop_id                             0x121400a8

/***************************************************************************//**
 * @brief Data structure of the heartbeat_stop event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_node_heartbeat_stop_s
{
  uint16_t src_addr; /**< Source address for the heartbeat message */
  uint16_t dst_addr; /**< Destination address for the heartbeat message */
});

typedef struct sl_btmesh_evt_node_heartbeat_stop_s sl_btmesh_evt_node_heartbeat_stop_t;

/** @} */ // end addtogroup sl_btmesh_evt_node_heartbeat_stop

/**
 * @addtogroup sl_btmesh_evt_node_beacon_received sl_btmesh_evt_node_beacon_received
 * @{
 * @brief Indicates reception of the secure network beacon.
 */

/** @brief Identifier of the beacon_received event */
#define sl_btmesh_evt_node_beacon_received_id                            0x131400a8

/***************************************************************************//**
 * @brief Data structure of the beacon_received event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_node_beacon_received_s
{
  uint16_t netkey_index; /**< Index of the network key used to encrypt the
                              beacon */
  uint8_t  key_refresh;  /**< Indicates whether there is an ongoing key refresh. */
  uint8_t  iv_update;    /**< Indicates whether there is an ongoing IV update. */
  uint32_t iv_index;     /**< IV index contained in the beacon. */
});

typedef struct sl_btmesh_evt_node_beacon_received_s sl_btmesh_evt_node_beacon_received_t;

/** @} */ // end addtogroup sl_btmesh_evt_node_beacon_received

/**
 * @addtogroup sl_btmesh_evt_node_local_dcd_data sl_btmesh_evt_node_local_dcd_data
 * @{
 * @brief Event reporting queried composition data page contents
 *
 * The contents are requested using the @ref sl_btmesh_node_get_local_dcd
 * command. More than one event may be generated. Page contents are terminated
 * by a @ref sl_btmesh_evt_node_local_dcd_data_end event. Note that the
 * interpretation of the received data is page-specific. Page 0 contains the
 * element and model layout of the node.
 */

/** @brief Identifier of the local_dcd_data event */
#define sl_btmesh_evt_node_local_dcd_data_id                             0x141400a8

/***************************************************************************//**
 * @brief Data structure of the local_dcd_data event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_node_local_dcd_data_s
{
  uint8_t    page; /**< Composition data page containing data */
  uint8array data; /**< Composition data page contents */
});

typedef struct sl_btmesh_evt_node_local_dcd_data_s sl_btmesh_evt_node_local_dcd_data_t;

/** @} */ // end addtogroup sl_btmesh_evt_node_local_dcd_data

/**
 * @addtogroup sl_btmesh_evt_node_local_dcd_data_end sl_btmesh_evt_node_local_dcd_data_end
 * @{
 * @brief Terminating event for node composition data
 */

/** @brief Identifier of the local_dcd_data_end event */
#define sl_btmesh_evt_node_local_dcd_data_end_id                         0x151400a8

/***************************************************************************//**
 * @brief Data structure of the local_dcd_data_end event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_node_local_dcd_data_end_s
{
  uint16_t result; /**< SL_STATUS_OK if successful. Error code otherwise. */
});

typedef struct sl_btmesh_evt_node_local_dcd_data_end_s sl_btmesh_evt_node_local_dcd_data_end_t;

/** @} */ // end addtogroup sl_btmesh_evt_node_local_dcd_data_end

/**
 * @addtogroup sl_btmesh_evt_node_local_model_metadata_page sl_btmesh_evt_node_local_model_metadata_page
 * @{
 * @brief Event reporting queried model metadata page contents
 *
 * The contents are requested using the @ref
 * sl_btmesh_node_get_local_model_metadata_page command. More than one event may
 * be generated. Page contents are terminated by a @ref
 * sl_btmesh_evt_node_local_model_metadata_page_end event. Note that the
 * interpretation of the received data is page-specific. Page 0 contains the
 * optional model metadata for mesh models, if applicable.
 */

/** @brief Identifier of the local_model_metadata_page event */
#define sl_btmesh_evt_node_local_model_metadata_page_id                  0x191400a8

/***************************************************************************//**
 * @brief Data structure of the local_model_metadata_page event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_node_local_model_metadata_page_s
{
  uint8_t    page; /**< Model metadata page number */
  uint8array data; /**< Model metadata page contents */
});

typedef struct sl_btmesh_evt_node_local_model_metadata_page_s sl_btmesh_evt_node_local_model_metadata_page_t;

/** @} */ // end addtogroup sl_btmesh_evt_node_local_model_metadata_page

/**
 * @addtogroup sl_btmesh_evt_node_local_model_metadata_page_end sl_btmesh_evt_node_local_model_metadata_page_end
 * @{
 * @brief Terminating event for Mesh Model Metadata page
 */

/** @brief Identifier of the local_model_metadata_page_end event */
#define sl_btmesh_evt_node_local_model_metadata_page_end_id              0x1a1400a8

/***************************************************************************//**
 * @brief Data structure of the local_model_metadata_page_end event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_node_local_model_metadata_page_end_s
{
  uint16_t result; /**< SL_STATUS_OK if successful. Error code otherwise. */
});

typedef struct sl_btmesh_evt_node_local_model_metadata_page_end_s sl_btmesh_evt_node_local_model_metadata_page_end_t;

/** @} */ // end addtogroup sl_btmesh_evt_node_local_model_metadata_page_end

/**
 * @addtogroup sl_btmesh_evt_node_start_received sl_btmesh_evt_node_start_received
 * @{
 * @brief Provisioning Start PDU received
 *
 * This diagnostic event shows the algorithm, public key, and authentication
 * choices Provisioner made and communicated to the unprovisioned device.
 */

/** @brief Identifier of the start_received event */
#define sl_btmesh_evt_node_start_received_id                             0x161400a8

/***************************************************************************//**
 * @brief Data structure of the start_received event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_node_start_received_s
{
  uint8_t algorithm;             /**< Selected provisioning algorithm */
  uint8_t public_key;            /**< Selected OOB public key */
  uint8_t authentication_method; /**< Selected authentication method */
  uint8_t authentication_action; /**< Selected authentication action */
  uint8_t authentication_size;   /**< Selected authentication size */
});

typedef struct sl_btmesh_evt_node_start_received_s sl_btmesh_evt_node_start_received_t;

/** @} */ // end addtogroup sl_btmesh_evt_node_start_received

/**
 * @addtogroup sl_btmesh_evt_node_address_updated sl_btmesh_evt_node_address_updated
 * @{
 * @brief Address of the device has changed due to a NPPI procedure.
 */

/** @brief Identifier of the address_updated event */
#define sl_btmesh_evt_node_address_updated_id                            0x171400a8

/***************************************************************************//**
 * @brief Data structure of the address_updated event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_node_address_updated_s
{
  uint16_t new_address; /**< New address taken into use. */
});

typedef struct sl_btmesh_evt_node_address_updated_s sl_btmesh_evt_node_address_updated_t;

/** @} */ // end addtogroup sl_btmesh_evt_node_address_updated

/**
 * @addtogroup sl_btmesh_evt_node_dcd_updated sl_btmesh_evt_node_dcd_updated
 * @{
 * @brief DCD page 128 has been taken in use as Page 0 due to a NPPI procedure
 *
 * Node needs to be restarted.
 */

/** @brief Identifier of the dcd_updated event */
#define sl_btmesh_evt_node_dcd_updated_id                                0x181400a8

/** @} */ // end addtogroup sl_btmesh_evt_node_dcd_updated

/***************************************************************************//**
 *
 * Initialize the Bluetooth mesh stack in Node role. When initialization is
 * complete, a @ref sl_btmesh_evt_node_initialized event will be generated.
 *
 * This command must be issued before any other Bluetooth Mesh commands, except
 * for @ref sl_btmesh_node_set_uuid command.
 *
 * Note that you may initialize a device either in the Provisioner or the Node
 * role, but not both.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_node_initialized
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_init();

/***************************************************************************//**
 *
 * Set the node in mode that allows exporting encryption keys. By default on
 * devices with PSA/ITS support the keys on normal node cannot be exported. This
 * command must called before either @ref sl_btmesh_node_init or @ref
 * sl_btmesh_node_init_oob. <b>On everey boot device boots to node role unless
 * this command is issued before node initialization</b>
 *
 * This command is to be used if device is originally in a node role and at
 * later stage switches to a provisioner role.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 * 
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_set_exportable_keys();

/***************************************************************************//**
 *
 * Start sending Unprovisioned Device Beacons.
 *
 * This command makes an unprovisioned device available for provisioning. The
 * device will start sending periodic unprovisioned device beacons containing
 * device UUID. It will also start listening for incoming Provisioner connection
 * attempts on the specified bearers (PB-ADV, PB-GATT, or both). For PB-GATT,
 * the device will also begin advertising its provisioning GATT service.
 *
 * At the beginning of a provisioning process, a @ref
 * sl_btmesh_evt_node_provisioning_started event will be generated. When the
 * device receives provisioning data from the Provisioner, a @ref
 * sl_btmesh_evt_node_provisioned event will be generated. If provisioning fails
 * with an error, a @ref sl_btmesh_evt_node_provisioning_failed event will be
 * generated.
 *
 * After it is provisioned, addresses are allocated for the node elements and a
 * network key is deployed to the node, making the node ready for further
 * configuration by the Provisioner. Note that, at this point, the node is not
 * yet fully ready to communicate with other nodes on the network.
 *
 * @param[in] bearer @parblock
 *   Bit mask for which bearer to use. Values are as follows:
 *     - <b>1 (bit 0):</b> PB-ADV
 *     - <b>2 (bit 1):</b> PB-GATT
 *
 *   Other bits are reserved and must not be used.
 *   @endparblock
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_node_provisioning_started
 *   - @ref sl_btmesh_evt_node_provisioned
 *   - @ref sl_btmesh_evt_node_provisioning_failed
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_start_unprov_beaconing(uint8_t bearer);

/***************************************************************************//**
 *
 * Stop sending Unprovisioned Device Beacons.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_stop_unprov_beaconing();

/***************************************************************************//**
 *
 * Get the latest RSSI value of a provisioned Bluetooth device.
 *
 * The value indicates the best signal strength received from any node within
 * the network. The value is cleared after calling this function meaning the
 * next call will fail if no new RSSI value is received.
 *
 * @param[out] rssi Latest RSSI value. Units: dBm. Ignore this parameter if the
 *   command fails.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_get_rssi(int8_t *rssi);

/***************************************************************************//**
 *
 * Provide the stack with the input out-of-band authentication data which the
 * Provisioner is displaying.
 *
 * @param[in] data_len Length of data in @p data
 * @param[in] data @parblock
 *   Raw 16-byte or 32-byte array containing the input authentication data. The
 *   length of the data depends on the chosen provisioning algorithm; see the
 *   @ref sl_btmesh_evt_node_start_received event for details on how the stack
 *   indicates the Provisioner's choice of provisioning algorithm to the
 *   application.
 *
 *   Note that the raw data given to the stack must be encoded in the manner
 *   described in the Mesh Protocol 1.1 specifcation Section 5.4.2.4. Whether
 *   the data should be treated as Numeric or Alphanumeric depends on the
 *   selected input action.
 *   @endparblock
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_send_input_oob_request_response(size_t data_len,
                                                           const uint8_t* data);

/***************************************************************************//**
 *
 * Get the device UUID.
 *
 * Every mesh device has a 128-bit UUID identifying the device. It is used
 * primarily during provisioning, because it is broadcast in Unprovisioned
 * Device Beacons to indicate that the device is ready to be provisioned.
 *
 * This command is used for debugging purposes. During provisioning, the stack
 * automatically uses the UUID of the device, which does not need to be
 * explicitly specified when @ref sl_btmesh_node_start_unprov_beaconing is
 * started.
 *
 * If get uuid is used before @ref sl_btmesh_node_init, the uuid will be read
 * from the nvme if it was manually set by @ref sl_btmesh_node_set_uuid. If get
 * uuid is used without a prior @ref sl_btmesh_node_init or @ref
 * sl_btmesh_node_set_uuid, SL_STATUS_BT_MESH_DOES_NOT_EXIST will be returned.
 *
 * @param[out] uuid The 16-byte UUID of the device
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_get_uuid(uuid_128 *uuid);

/***************************************************************************//**
 *
 * Provision devices completely out-of-band. Populate the Provisioner's device
 * database with the corresponding values to make the device reachable and
 * configurable in the Provisioner's network.
 *
 * See also the Provisioner command for @ref sl_btmesh_prov_add_ddb_entry to
 * Provisioner's device database.
 *
 * <b>NOTE</b> : The device must be reset after this command has been issued.
 *
 * @param[in] device_key Device Key for this Device, shared by the Provisioner
 * @param[in] network_key Network key that the Provisioner has selected for this
 *   device
 * @param[in] netkey_index Index of the Network Key the Provisioner has selected
 *   for this device
 * @param[in] iv_index Current IV Index used in the network
 * @param[in] address Address the Provisioner has allocated for this device's
 *   primary element
 * @param[in] kr_in_progress Set to 1 if key refresh is currently in progress,
 *   otherwise 0.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_set_provisioning_data(aes_key_128 device_key,
                                                 aes_key_128 network_key,
                                                 uint16_t netkey_index,
                                                 uint32_t iv_index,
                                                 uint16_t address,
                                                 uint8_t kr_in_progress);

/***************************************************************************//**
 *
 * Initialize the Bluetooth mesh stack in the Node role. When initialization is
 * complete, a @ref sl_btmesh_evt_node_initialized event is generated.
 *
 * This command is the same as the @ref sl_btmesh_node_init command except for
 * parameters defining whether OOB authentication data stored on the device can
 * be used during provisioning.
 *
 * This command must be issued before any other Bluetooth mesh commands, except
 * for @ref sl_btmesh_node_set_uuid command.
 *
 * Note that you may initialize a device either in the Provisioner or the Node
 * role, but not both.
 *
 * @param[in] public_key If non-zero, use the ECC key stored in the persistent
 *   store during provisioning instead of an ephemeral key.
 * @param[in] auth_methods Enum @ref sl_btmesh_node_auth_method_flag_t. Allowed
 *   OOB authentication methods. The value is a bitmap so that multiple methods
 *   can be supported.
 * @param[in] output_actions Enum @ref sl_btmesh_node_oob_output_action_flag_t.
 *   Allowed OOB Output Action types
 * @param[in] output_size Maximum Output OOB size Valid values range from 0
 *   (feature not supported) to 8.
 * @param[in] input_actions Enum @ref sl_btmesh_node_oob_input_action_flag_t.
 *   Allowed OOB Input Action types
 * @param[in] input_size Maximum Input OOB size. Valid values range from 0
 *   (feature not supported) to 8.
 * @param[in] oob_location Defines the OOB data location bitmask. See @ref
 *   sl_btmesh_prov_oob_capabilities.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_node_initialized
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_init_oob(uint8_t public_key,
                                    uint8_t auth_methods,
                                    uint16_t output_actions,
                                    uint8_t output_size,
                                    uint16_t input_actions,
                                    uint8_t input_size,
                                    uint16_t oob_location);

/***************************************************************************//**
 *
 * Enable/disable the IV index recovery mode.
 *
 * If the node has not been in communication with the network for a long time
 * (e.g., because it was turned off), it may have missed IV index updates and
 * isn't able to communicate with other nodes. In this case, enable the IV index
 * recovery mode.
 *
 * @param[in] mode Zero to disable; non-zero to enable
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_set_ivrecovery_mode(uint8_t mode);

/***************************************************************************//**
 *
 * Get the current IV index recovery mode state. See @ref
 * sl_btmesh_node_set_ivrecovery_mode for details.
 *
 * @param[out] mode If non-zero, IV recovery is enabled.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_get_ivrecovery_mode(uint8_t *mode);

/***************************************************************************//**
 *
 * Set the network relay delay interval.
 *
 * This parameter determines the time a relay waits until it relays a network
 * PDU. The value used is a random number within the specified interval.
 *
 * Note that this value affects the first instance of the relayed network PDU.
 * If relay retransmissions are enabled, the interval between retransmissions is
 * defined by the relay state, set by the Provisioner of the network or by @ref
 * sl_btmesh_test_set_relay test command.
 *
 * @param[in] min_ms Minimum interval, in milliseconds
 * @param[in] max_ms Maximum interval, in milliseconds, which must be equal to
 *   or greater than the minimum.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_set_net_relay_delay(uint8_t min_ms, uint8_t max_ms);

/***************************************************************************//**
 *
 * Get network relay delay interval. See @ref sl_btmesh_node_set_net_relay_delay
 * command for details.
 *
 * @param[out] min_ms Minimum interval, in milliseconds
 * @param[out] max_ms Maximum interval, in milliseconds
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_get_net_relay_delay(uint8_t *min_ms,
                                               uint8_t *max_ms);

/***************************************************************************//**
 *
 * Get the current IV index update state in the network.
 *
 * @param[out] iv_index Current IV index
 * @param[out] state Indicates whether the IV index update is ongoing (1) or not
 *   (0).
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_get_ivupdate_state(uint32_t *iv_index,
                                              uint8_t *state);

/***************************************************************************//**
 *
 * Attempt to request an IV index update in the network.
 *
 * Each network layer PDU that a node sends has a 24-bit sequence number
 * attached to it. Each node element keeps a sequence number counter, which is
 * incremented for every PDU sent out to the network. Repeating sequence numbers
 * for a given IV index value is not allowed. As a result, if a node determines
 * it is about to exhaust the available sequence numbers in one of its elements,
 * it needs to request an IV index update by issuing this command.
 *
 * Determining when a node may run out of sequence numbers has to be done at the
 * application level because the stack can't determine how often the application
 * plans to transmit to the network, i.e., how long the remaining sequence
 * numbers might last.
 *
 * See also the @ref sl_btmesh_node_get_seq_remaining command.
 *
 * Note that the call may fail for various reasons, for example if an IV index
 * update is already ongoing, or if not enough time has passed since the
 * previous IV index update.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_request_ivupdate();

/***************************************************************************//**
 *
 * Get the number of sequence numbers remaining on an element (before sequence
 * numbers are exhausted). Note that every node element keeps a separate
 * sequence number counter.
 *
 * @param[in] elem_index The index of queried element
 * @param[out] count Remaining sequence number count
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_get_seq_remaining(uint16_t elem_index,
                                             uint32_t *count);

/***************************************************************************//**
 *
 * Save the current replay protection list to the persistent store.
 *
 * The replay protection list keeps track of the packet sequence numbers from
 * different sources received by the node. The node will not process messages
 * associated with already used sequence numbers and is therefore protected from
 * replay attacks using previously recorded messages.
 *
 * The replay protection list is kept in RAM during runtime. It needs to be
 * saved to the persistent store periodically and always before the device
 * powers off. Because the stack is not aware when this will happen, the
 * application has to call this method while the node is getting ready to power
 * down but is still running.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_save_replay_protection_list();

/***************************************************************************//**
 *
 * Write device UUID into the persistent store. This command must be called
 * before initializing the Bluetooth mesh stack (before @ref sl_btmesh_node_init
 * or @ref sl_btmesh_node_init_oob), otherwise the change will not take effect
 * before a reboot.
 *
 * Ensure that the UUID conforms to the format defined in <a
 * href="https://tools.ietf.org/html/rfc4122">RFC 4122</a>
 *
 * Note that UUID must not be changed when the device is provisioned to a
 * network.
 *
 * Furthermore, ensure that the UUID remains constant if a device has received a
 * firmware update, which requires reprovisioning of the device after the update
 * has been applied (e.g., new elements are added by the update).
 *
 * @param[in] uuid UUID to set
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_set_uuid(uuid_128 uuid);

/***************************************************************************//**
 *
 * Get the status of the replay protection list.
 *
 * @param[out] total_entries Total number of entries in use
 * @param[out] unsaved_entries Number of unsaved entries
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_get_replay_protection_list_status(uint16_t *total_entries,
                                                             uint16_t *unsaved_entries);

/***************************************************************************//**
 *
 * Get the unicast address configured to an element.
 *
 * @param[in] elem_index The index of the target element, 0 is the primary
 *   element
 * @param[out] address The address of the element. Returns 0x0000 if the address
 *   is not configured or if an error occurs.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_get_element_address(uint16_t elem_index,
                                               uint16_t *address);

/***************************************************************************//**
 *
 * Provide the stack with static out-of-band authentication data, which the
 * stack requested.
 *
 * @param[in] data_len Length of data in @p data
 * @param[in] data @parblock
 *   Raw 16-byte or 32-byte array containing the static authentication data. The
 *   length of the data depends on the chosen provisioning algorithm; see the
 *   @ref sl_btmesh_evt_node_start_received event for details on how the stack
 *   indicates the Provisioner's choice of provisioning algorithm to the
 *   application.
 *
 *   Note that the raw data given to the stack must be encoded in the manner
 *   described in the Mesh Protocol 1.1 specifcation Section 5.4.2.4. The data
 *   type for static OOB authentication data is Binary.
 *   @endparblock
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_send_static_oob_request_response(size_t data_len,
                                                            const uint8_t* data);

/***************************************************************************//**
 *
 * Factory reset of the mesh node.
 *
 * To complete procedure, the application should do its own cleanup duties and
 * reset the hardware.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_reset();

/***************************************************************************//**
 *
 * Set secure network beaconing on or off. When on, every received secure
 * network beacon will generate a @ref sl_btmesh_evt_node_beacon_received event.
 *
 * @param[in] report Turn reporting on (nonzero) or off (zero).
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_set_beacon_reporting(uint8_t report);

/***************************************************************************//**
 *
 * Set the time since the last IV update. After the reboot, the node doesn't
 * know the time since the last IV update and assumes that it happened at the
 * time of the reboot.
 *
 * @param[in] age_sec Seconds since last IV update. Values from 0 to 345600
 *   (96h)
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_set_iv_update_age(uint32_t age_sec);

/***************************************************************************//**
 *
 * Get the total number of keys in node.
 *
 * @param[in] type Enum @ref sl_btmesh_node_key_type_t. 0 for network key, 1 for
 *   application key
 * @param[out] count Number of keys of the given type on the device
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_get_key_count(uint8_t type, uint32_t *count);

/***************************************************************************//**
 *
 * Get a key by position. Only current key data exits in normal mode. Old key
 * data can be queried only during the key refresh.
 *
 * @param[in] type Enum @ref sl_btmesh_node_key_type_t. 0 for network key, 1 for
 *   application key
 * @param[in] index Key position, ranging from zero to key count minus one
 * @param[in] current 1: Current key, 0: Old key
 * @param[out] id Mesh key index of the key
 * @param[out] netkey_index For application keys, the network key index of the
 *   network key this key is bound to. Ignore for other key types.
 * @param[out] key Key data, 16 bytes
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_get_key(uint8_t type,
                                   uint32_t index,
                                   uint8_t current,
                                   uint16_t *id,
                                   uint16_t *netkey_index,
                                   aes_key_128 *key);

/***************************************************************************//**
 *
 * Get a list of networks supported by the node. This is a deprecated function.
 * Please use @ref sl_btmesh_node_get_key_count and @ref sl_btmesh_node_get_key
 * instead.
 *
 * @param[in] max_networks_size Size of output buffer passed in @p networks
 * @param[out] networks_len On return, set to the length of output data written
 *   to @p networks
 * @param[out] networks List of network IDs. Each ID is two bytes in
 *   little-endian format
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
SL_BGAPI_DEPRECATED sl_status_t sl_btmesh_node_get_networks(size_t max_networks_size,
                                        size_t *networks_len,
                                        uint8_t *networks);

/***************************************************************************//**
 *
 * Get the current sequence number of an element.
 *
 * @param[in] elem_index The index of the target element, 0 is the primary
 *   element
 * @param[out] seqnum Current sequence number of the element Ignore the value if
 *   the result code indicates an error (for example, when the element index is
 *   out of bounds).
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_get_element_seqnum(uint16_t elem_index,
                                              uint32_t *seqnum);

/***************************************************************************//**
 *
 * Set a model-specific option.
 *
 * @param[in] elem_index The index of the target element, 0 is the primary
 *   element
 * @param[in] vendor_id Vendor ID for vendor-specific models. Use 0xffff for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID
 * @param[in] option Option to set. The following options are defined:
 *     - <b>0x80</b> Generic level delta behavior. Used only with generic level
 *       models.
 *     - <b>0xa0</b> Light models range status behavior. Used only with
 *       Lightness, Ctl and Hsl models. Note that state codes are different for
 *       states with status
 * @param[in] value @parblock
 *   Value for the option.
 *
 *   The following values are defined for generic level delta behavior option:
 *     - <b>0x0</b> Generic level delta behavior: pass raw delta request data to
 *       application
 *     - <b>0x1</b> Generic level delta behavior: pass processed delta request
 *       data to application (default)
 *
 *   The following values are defined for for Lightness, Ctl and Hsl range
 *   status message behavior option:
 *     - <b>0x0</b> Range Status behavior: pass status messages without status
 *       field(default)
 *     - <b>0x1</b> Range Status behavior: pass status messagages with status
 *       field
 *   @endparblock
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_set_model_option(uint16_t elem_index,
                                            uint16_t vendor_id,
                                            uint16_t model_id,
                                            uint8_t option,
                                            uint32_t value);

/***************************************************************************//**
 *
 * Get the local Device Composition Data.
 *
 * @param[in] page Composition data page to query
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_node_local_dcd_data
 *   - @ref sl_btmesh_evt_node_local_dcd_data_end
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_get_local_dcd(uint8_t page);

/***************************************************************************//**
 *
 * Power off the mesh node. To be called prior power off to allow the device to
 * save its own state.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_power_off();

/***************************************************************************//**
 * @cond RESTRICTED
 *
 * Restricted/experimental API. Contact Silicon Labs sales for more information.
 *
 * Set PHY used for ADV traffic
 *
 * @param[in] phy   - 1: Primary and secondary channels are set to 1M
 *     - 2: Primary channels are 1M, secondary channels are 2M
 *     - 4: Primary and secondary channels are set to 125K Coded Phy (S=8)
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @endcond
 ******************************************************************************/
sl_status_t sl_btmesh_node_set_adv_phy(uint8_t phy);

/***************************************************************************//**
 * @cond RESTRICTED
 *
 * Restricted/experimental API. Contact Silicon Labs sales for more information.
 *
 * Get PHY used for ADV traffic
 *
 * @param[out] phy   - 1: Primary and secondary channels are set to 1M
 *     - 2: Primary channels are 1M, secondary channels are 2M
 *     - 4: Primary and secondary channels are set to 125K Coded Phy (S=8)
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @endcond
 ******************************************************************************/
sl_status_t sl_btmesh_node_get_adv_phy(uint8_t *phy);

/***************************************************************************//**
 *
 * Set the set of allowed provisioning algorithms. The application can use this
 * command to limit the algorithms used in provisioning, even if the stack would
 * contain support for multiple algorithms. This command has to be issued before
 * initializing the stack with @ref sl_btmesh_node_init or @ref
 * sl_btmesh_node_init_oob.
 *
 * @param[in] algorithms Enum @ref sl_btmesh_node_algorithm_flag_t. Allowed
 *   provisioning algorithms bitmask; zero represents a special value to allow
 *   any available algorithm.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_set_provisioning_algorithms(uint16_t algorithms);

/***************************************************************************//**
 *
 * Initialize provisioning record retrieval functionality on the device. This
 * must be called before device is provisioned, if provisioning records are to
 * be used.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_init_provisioning_records();

/***************************************************************************//**
 *
 * Set Unprovisioned Device Beaconing advertisement interval timing parameters
 * both with and without URI.
 *
 * This command sets the timing parameters of Unprovisioned Device Beaconing
 * advertisement interval with and without URI. This setting will take effect
 * next time the Unprovisioned Device Beaconing advertisement is started.
 *
 * @param[in] adv_interval_min Minimum advertisement interval. Value is in units
 *   of 0.625 ms. Range: 0x20 to 0xFFFF. Time range: 20 ms to 40.96 s
 * @param[in] adv_interval_max Maximum advertisement interval. Value is in units
 *   of 0.625 ms. Must be equal to or greater than the minimum interval. Range:
 *   0x20 to 0xFFFF. Time range: 20 ms to 40.96 s
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_set_unprov_beaconing_adv_interval(uint16_t adv_interval_min,
                                                             uint16_t adv_interval_max);

/***************************************************************************//**
 *
 * Set Mesh Proxy Service advertisement interval timing parameters.
 *
 * This command sets the timing parameters of Mesh Proxy Service advertisement
 * interval. This setting will take effect next time the Mesh Proxy Service
 * advertisement is started.
 *
 * @param[in] adv_interval_min Minimum advertisement interval. Value is in units
 *   of 0.625 ms. Range: 0x20 to 0xFFFF. Time range: 20 ms to 40.96 s
 * @param[in] adv_interval_max Maximum advertisement interval. Value is in units
 *   of 0.625 ms. Must be equal to or greater than the minimum interval. Range:
 *   0x20 to 0xFFFF. Time range: 20 ms to 40.96 s
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_set_proxy_service_adv_interval(uint16_t adv_interval_min,
                                                          uint16_t adv_interval_max);

/***************************************************************************//**
 *
 * Set Mesh Provisioning Service advertisement interval timing parameters.
 *
 * This command sets the timing parameters of Mesh Provisioning Service
 * advertisement interval. This setting will take effect next time the Mesh
 * Provisioning Service advertisement is started.
 *
 * @param[in] adv_interval_min Minimum advertisement interval. Value is in units
 *   of 0.625 ms. Range: 0x20 to 0xFFFF. Time range: 20 ms to 40.96 s
 * @param[in] adv_interval_max Maximum advertisement interval. Value is in units
 *   of 0.625 ms. Must be equal to or greater than the minimum interval. Range:
 *   0x20 to 0xFFFF. Time range: 20 ms to 40.96 s
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_set_provisioning_service_adv_interval(uint16_t adv_interval_min,
                                                                 uint16_t adv_interval_max);

/***************************************************************************//**
 *
 * Get the local Mesh Model Metadata.
 *
 * @param[in] page Model metadata page to query
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_node_local_model_metadata_page
 *   - @ref sl_btmesh_evt_node_local_model_metadata_page_end
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_get_local_model_metadata_page(uint8_t page);

/***************************************************************************//**
 *
 * Save the current Private GATT Proxy Solicitation Replay Protection List to
 * persistent storage.
 *
 * Solicitation PDU RPL is in use only if the node supports On Demand Private
 * Proxy Server.
 *
 * The replay protection list keeps track of the Solicitation PDU sequence
 * numbers from different sources received by the node. The node will not
 * process messages associtated with already used sequence numbers and is
 * therefore protected from replay attacks using previously recorded messages.
 *
 * The replay protection list is kept is RAM during runtime. It needs to be
 * saved to persistent storage periodally and always before the device powers
 * off. Because the stack is not aware when this will happen, the application
 * has to call this method while the node is getting ready to power down but is
 * still running.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_save_proxy_solicitation_rpl();

/***************************************************************************//**
 *
 * Get the status of the Solicitation Replay Protection List.
 *
 * @param[out] total_entries Total number of entries in use
 * @param[out] unsaved_entries Number of unsaved entries
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_get_proxy_solicitation_rpl_status(uint16_t *total_entries,
                                                             uint16_t *unsaved_entries);

/***************************************************************************//**
 *
 * Set OOB URI.
 *
 * @param[in] uri_len Length of data in @p uri
 * @param[in] uri Base URI for OOB public key retrieval
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_set_oob_uri(size_t uri_len, const uint8_t* uri);

/***************************************************************************//**
 *
 * Get the OOB URI.
 *
 * @param[in] max_uri_size Size of output buffer passed in @p uri
 * @param[out] uri_len On return, set to the length of output data written to @p
 *   uri
 * @param[out] uri Base URI for OOB public key retrieval
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_get_oob_uri(size_t max_uri_size,
                                       size_t *uri_len,
                                       uint8_t *uri);

/***************************************************************************//**
 * @cond RESTRICTED
 *
 * Restricted/experimental API. Contact Silicon Labs sales for more information.
 *
 * Set Mesh Proxy Service advertisement UUID.
 *
 * This command sets the UUID Mesh Proxy Service advertisement and service. This
 * setting will take effect next time the Mesh Proxy Service advertisement is
 * started.
 *
 * @param[in] uuid A 16bit uuid for gatt proxy service advertisement.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @endcond
 ******************************************************************************/
sl_status_t sl_btmesh_node_set_proxy_service_uuid(uint16_t uuid);

/***************************************************************************//**
 *
 * Tets if receiver network identity beacon matches mesh network.
 *
 * @param[in] address Mesh address of the node, if this is 0 then testing for
 *   network identity. If valid a node address is give then testing for for node
 *   identity.
 * @param[in] netkey_index Network key index of the node.
 * @param[in] data_len Length of data in @p data
 * @param[in] data Contents of the network identity beacon.
 * @param[out] match   - 0: Network/Node Identity record did not match
 *     - 1: Network/Node Identity record did match
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_test_identity(uint16_t address,
                                         uint16_t netkey_index,
                                         size_t data_len,
                                         const uint8_t* data,
                                         uint8_t *match);

/***************************************************************************//**
 *
 * Set Mesh Proxy Service scan response data.
 *
 * This command sets the scan response data for a Mesh Proxy Service
 * advertisement. The Mesh Proxy Service advertisement is a connectable and
 * scannable advertisement, meaning that a client can issue a scan request and
 * receive a scan response that carries additional data relevant to the service.
 * This setting will take effect next time the Mesh Proxy Service advertisement
 * is started.
 *
 * @param[in] netkey_index Index of the network key associated with the proxy
 *   service advertisement. The caller may set scan response data separately for
 *   each network key, so that data encrypted and/or authenticated with the
 *   network key can be used as a scan response, if desired.
 * @param[in] scan_response_data_len Length of data in @p scan_response_data
 * @param[in] scan_response_data Binary scan response data encoded as AD types
 *   as defined in the Core specification. Data must fit into a SCAN_RSP PDU. It
 *   is up to the caller to ensure that valid data is given.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_set_proxy_service_scan_response(uint16_t netkey_index,
                                                           size_t scan_response_data_len,
                                                           const uint8_t* scan_response_data);

/***************************************************************************//**
 *
 * Clear Mesh Proxy Service scan response data.
 *
 * This command clears the scan response data for a Mesh Proxy Service
 * advertisement. This setting will take effect next time the Mesh Proxy Service
 * advertisement is started.
 *
 * @param[in] netkey_index Index of the network key associated with the proxy
 *   service advertisement. The caller may set scan response data separately for
 *   each network key, so that data encrypted and/or authenticated with the
 *   network key can be used as a scan response, if desired.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_clear_proxy_service_scan_response(uint16_t netkey_index);

/***************************************************************************//**
 *
 * Set Mesh Provisioning Service scan response data.
 *
 * This command sets the scan response data for a Mesh Provisioning Service
 * advertisement. The Mesh Provisioning Service advertisement is a connectable
 * and scannable advertisement, meaning that a client can issue a scan request
 * and receive a scan response that carries additional data relevant to the
 * service. This setting will take effect next time the Mesh Provisioning
 * Service advertisement is started.
 *
 * @param[in] scan_response_data_len Length of data in @p scan_response_data
 * @param[in] scan_response_data Binary scan response data encoded as AD types
 *   as defined in the Core specification. Data must fit into a SCAN_RSP PDU. It
 *   is up to the caller to ensure that valid data is given.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_set_provisioning_service_scan_response(size_t scan_response_data_len,
                                                                  const uint8_t* scan_response_data);

/***************************************************************************//**
 *
 * Clear Mesh Provisioning Service scan response data.
 *
 * This command clears the scan response data for a Mesh Provisioning Service
 * advertisement. This setting will take effect next time the Mesh Provisioning
 * Service advertisement is started.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_clear_provisioning_service_scan_response();

/***************************************************************************//**
 *
 * Compare a composition data page to the currently active one.
 *
 * This command compares scans the given data and the currently active
 * compositon data page for differences.
 *
 * @param[in] page_number Composition Data page number. Valid range: 0-2.
 * @param[in] page_data_len Length of data in @p page_data
 * @param[in] page_data Composition Data page content to be compared
 * @param[out] diff Enum @ref sl_btmesh_node_dcd_comparison_t. Bit field
 *   indication the types of differences found.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_node_compare_dcd(uint8_t page_number,
                                       size_t page_data_len,
                                       const uint8_t* page_data,
                                       uint32_t *diff);

/** @} */ // end addtogroup sl_btmesh_node

/**
 * @addtogroup sl_btmesh_prov Bluetooth Mesh Stack Provisioner
 * @{
 *
 * @brief Bluetooth Mesh Stack Provisioner
 *
 * Bluetooth mesh stack API for the embedded Provisioner
 *
 * Commands in this class provision nodes in the mesh network and generate
 * security keys for the network.
 *
 * <b>Initialization:</b>
 *   - @ref sl_btmesh_prov_init
 *   - @ref sl_btmesh_evt_prov_initialized
 *   - @ref sl_btmesh_evt_prov_initialization_failed
 *
 * <b>Provisioning a node:</b>
 *   - @ref sl_btmesh_prov_scan_unprov_beacons : Scan for unprovisioned device
 *     beacons
 *   - @ref sl_btmesh_prov_stop_scan_unprov_beacons : Stop scanning for
 *     unprovisioned device beacons
 *   - @ref sl_btmesh_evt_prov_unprov_beacon : Unprovisioned device beacon seen
 *   - @ref sl_btmesh_evt_prov_uri : URI advertisement seen
 *   - @ref sl_btmesh_prov_create_provisioning_session : Create provisioning
 *     session
 *   - @ref sl_btmesh_prov_provision_adv_device : Provision a device over PB-ADV
 *   - @ref sl_btmesh_prov_provision_gatt_device : Provision a device over
 *     PB-GATT
 *   - @ref sl_btmesh_evt_prov_oob_display_input : Request to display input
 *     out-of-band data to the user to input on the node
 *   - @ref sl_btmesh_evt_prov_oob_pkey_request : Request for out-of-band public
 *     key of a node
 *   - @ref sl_btmesh_prov_send_oob_pkey_response : Provide stack with
 *     out-of-band public key of a node
 *   - @ref sl_btmesh_evt_prov_oob_auth_request : Request for out-of-band
 *     authentication data of a node
 *   - @ref sl_btmesh_prov_send_oob_auth_response : Provide stack with
 *     out-of-band authentication data of a node
 *   - @ref sl_btmesh_evt_prov_device_provisioned : Device Provisioned
 *   - @ref sl_btmesh_evt_prov_provisioning_failed : Provisioning a device
 *     failed
 *
 * <b>Key Management</b>
 *   - @ref sl_btmesh_prov_create_network : Create a new network key on the
 *     Provisioner
 *   - @ref sl_btmesh_prov_create_appkey : Create a new application key on the
 *     Provisioner
 *   - @ref sl_btmesh_prov_start_key_refresh : Start a key refresh procedure
 *   - @ref sl_btmesh_prov_suspend_key_refresh : Suspend an ongoing key refresh
 *     procedure
 *   - @ref sl_btmesh_prov_resume_key_refresh : Resume a suspended key refresh
 *     procedure
 *   - @ref sl_btmesh_prov_get_key_refresh_exclusion : Get node key refresh
 *     exclusion status
 *   - @ref sl_btmesh_prov_set_key_refresh_exclusion : Set node key refresh
 *     exclusion status
 *   - @ref sl_btmesh_prov_get_key_refresh_phase : Get node key refresh phase
 *   - @ref sl_btmesh_prov_start_key_refresh_from_phase : Start a key refresh
 *     procedure from the given phase
 *
 * <b>Device Database</b>
 *   - @ref sl_btmesh_prov_add_ddb_entry : Add a node to the device database
 *   - @ref sl_btmesh_prov_delete_ddb_entry : Remove a node from the device
 *     database
 *   - @ref sl_btmesh_prov_get_ddb_entry : Fetch node data from the device
 *     database
 *   - @ref sl_btmesh_prov_list_ddb_entries : Request a list of nodes in the
 *     device database
 *   - @ref sl_btmesh_evt_prov_ddb_list : Device database list result
 *   - @ref sl_btmesh_prov_update_device_netkey_index : Update default network
 *     key index for a device database entry
 *
 * These commands are available only if the Provisioner functionality is
 * compiled in the device. Otherwise, a "feature not implemented" error code
 * will be returned for all functions in this class.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_prov_init_id                                       0x00150028
#define sl_btmesh_cmd_prov_scan_unprov_beacons_id                        0x01150028
#define sl_btmesh_cmd_prov_create_provisioning_session_id                0x41150028
#define sl_btmesh_cmd_prov_set_provisioning_suspend_event_id             0x42150028
#define sl_btmesh_cmd_prov_provision_adv_device_id                       0x43150028
#define sl_btmesh_cmd_prov_provision_gatt_device_id                      0x44150028
#define sl_btmesh_cmd_prov_provision_remote_device_id                    0x7f150028
#define sl_btmesh_cmd_prov_continue_provisioning_id                      0x45150028
#define sl_btmesh_cmd_prov_abort_provisioning_id                         0x47150028
#define sl_btmesh_cmd_prov_set_device_address_id                         0x46150028
#define sl_btmesh_cmd_prov_create_network_id                             0x03150028
#define sl_btmesh_cmd_prov_create_appkey_id                              0x07150028
#define sl_btmesh_cmd_prov_send_oob_pkey_response_id                     0x08150028
#define sl_btmesh_cmd_prov_send_oob_auth_response_id                     0x09150028
#define sl_btmesh_cmd_prov_set_oob_requirements_id                       0x0a150028
#define sl_btmesh_cmd_prov_start_key_refresh_id                          0x0b150028
#define sl_btmesh_cmd_prov_get_key_refresh_exclusion_id                  0x0c150028
#define sl_btmesh_cmd_prov_set_key_refresh_exclusion_id                  0x0d150028
#define sl_btmesh_cmd_prov_set_key_refresh_failure_id                    0x12150028
#define sl_btmesh_cmd_prov_phase_timeout_get_id                          0x10150028
#define sl_btmesh_cmd_prov_phase_timeout_set_id                          0x11150028
#define sl_btmesh_cmd_prov_get_ddb_entry_id                              0x16150028
#define sl_btmesh_cmd_prov_delete_ddb_entry_id                           0x17150028
#define sl_btmesh_cmd_prov_add_ddb_entry_id                              0x18150028
#define sl_btmesh_cmd_prov_list_ddb_entries_id                           0x19150028
#define sl_btmesh_cmd_prov_initialize_network_id                         0x37150028
#define sl_btmesh_cmd_prov_get_key_refresh_appkey_exclusion_id           0x38150028
#define sl_btmesh_cmd_prov_set_key_refresh_appkey_exclusion_id           0x39150028
#define sl_btmesh_cmd_prov_stop_scan_unprov_beacons_id                   0x3a150028
#define sl_btmesh_cmd_prov_update_device_netkey_index_id                 0x3b150028
#define sl_btmesh_cmd_prov_suspend_key_refresh_id                        0x3c150028
#define sl_btmesh_cmd_prov_resume_key_refresh_id                         0x3d150028
#define sl_btmesh_cmd_prov_get_key_refresh_phase_id                      0x3e150028
#define sl_btmesh_cmd_prov_start_key_refresh_from_phase_id               0x3f150028
#define sl_btmesh_cmd_prov_flush_key_refresh_state_id                    0x40150028
#define sl_btmesh_cmd_prov_get_provisioning_records_list_id              0x1c150028
#define sl_btmesh_cmd_prov_get_provisioning_record_data_id               0x1d150028
#define sl_btmesh_cmd_prov_init_provisioning_records_id                  0x1e150028
#define sl_btmesh_rsp_prov_init_id                                       0x00150028
#define sl_btmesh_rsp_prov_scan_unprov_beacons_id                        0x01150028
#define sl_btmesh_rsp_prov_create_provisioning_session_id                0x41150028
#define sl_btmesh_rsp_prov_set_provisioning_suspend_event_id             0x42150028
#define sl_btmesh_rsp_prov_provision_adv_device_id                       0x43150028
#define sl_btmesh_rsp_prov_provision_gatt_device_id                      0x44150028
#define sl_btmesh_rsp_prov_provision_remote_device_id                    0x7f150028
#define sl_btmesh_rsp_prov_continue_provisioning_id                      0x45150028
#define sl_btmesh_rsp_prov_abort_provisioning_id                         0x47150028
#define sl_btmesh_rsp_prov_set_device_address_id                         0x46150028
#define sl_btmesh_rsp_prov_create_network_id                             0x03150028
#define sl_btmesh_rsp_prov_create_appkey_id                              0x07150028
#define sl_btmesh_rsp_prov_send_oob_pkey_response_id                     0x08150028
#define sl_btmesh_rsp_prov_send_oob_auth_response_id                     0x09150028
#define sl_btmesh_rsp_prov_set_oob_requirements_id                       0x0a150028
#define sl_btmesh_rsp_prov_start_key_refresh_id                          0x0b150028
#define sl_btmesh_rsp_prov_get_key_refresh_exclusion_id                  0x0c150028
#define sl_btmesh_rsp_prov_set_key_refresh_exclusion_id                  0x0d150028
#define sl_btmesh_rsp_prov_set_key_refresh_failure_id                    0x12150028
#define sl_btmesh_rsp_prov_phase_timeout_get_id                          0x10150028
#define sl_btmesh_rsp_prov_phase_timeout_set_id                          0x11150028
#define sl_btmesh_rsp_prov_get_ddb_entry_id                              0x16150028
#define sl_btmesh_rsp_prov_delete_ddb_entry_id                           0x17150028
#define sl_btmesh_rsp_prov_add_ddb_entry_id                              0x18150028
#define sl_btmesh_rsp_prov_list_ddb_entries_id                           0x19150028
#define sl_btmesh_rsp_prov_initialize_network_id                         0x37150028
#define sl_btmesh_rsp_prov_get_key_refresh_appkey_exclusion_id           0x38150028
#define sl_btmesh_rsp_prov_set_key_refresh_appkey_exclusion_id           0x39150028
#define sl_btmesh_rsp_prov_stop_scan_unprov_beacons_id                   0x3a150028
#define sl_btmesh_rsp_prov_update_device_netkey_index_id                 0x3b150028
#define sl_btmesh_rsp_prov_suspend_key_refresh_id                        0x3c150028
#define sl_btmesh_rsp_prov_resume_key_refresh_id                         0x3d150028
#define sl_btmesh_rsp_prov_get_key_refresh_phase_id                      0x3e150028
#define sl_btmesh_rsp_prov_start_key_refresh_from_phase_id               0x3f150028
#define sl_btmesh_rsp_prov_flush_key_refresh_state_id                    0x40150028
#define sl_btmesh_rsp_prov_get_provisioning_records_list_id              0x1c150028
#define sl_btmesh_rsp_prov_get_provisioning_record_data_id               0x1d150028
#define sl_btmesh_rsp_prov_init_provisioning_records_id                  0x1e150028

/**
 * @addtogroup sl_btmesh_prov_oob_capabilities OOB Capabilities
 * @{
 *
 * OOB capability bitmask constants
 */

/** Uncategorized */
#define SL_BTMESH_PROV_OOB_OTHER      0x1       

/** URI or other electronic */
#define SL_BTMESH_PROV_OOB_URI        0x2       

/** 2D machine-readable code */
#define SL_BTMESH_PROV_OOB_2D_MR_CODE 0x4       

/** Barcode */
#define SL_BTMESH_PROV_OOB_BAR_CODE   0x8       

/** NFC */
#define SL_BTMESH_PROV_OOB_NFC        0x10      

/** Number */
#define SL_BTMESH_PROV_OOB_NUMBER     0x20      

/** String */
#define SL_BTMESH_PROV_OOB_STRING     0x40      

/** Reserved */
#define SL_BTMESH_PROV_OOB_RFU_7      0x80      

/** Reserved */
#define SL_BTMESH_PROV_OOB_RFU_8      0x100     

/** Reserved */
#define SL_BTMESH_PROV_OOB_RFU_9      0x200     

/** Reserved */
#define SL_BTMESH_PROV_OOB_RFU_A      0x400     

/** On the box */
#define SL_BTMESH_PROV_OOB_LOC_ON_BOX 0x800     

/** Inside the box */
#define SL_BTMESH_PROV_OOB_LOC_IN_BOX 0x1000    

/** On a piece of paper */
#define SL_BTMESH_PROV_OOB_LOC_PAPER  0x2000    

/** In the device manual */
#define SL_BTMESH_PROV_OOB_LOC_MANUAL 0x4000    

/** On the device */
#define SL_BTMESH_PROV_OOB_LOC_DEVICE 0x8000    

/** Mask of reserved bits */
#define SL_BTMESH_PROV_OOB_RFU_MASK   0x780     

/** @} */ // end OOB Capabilities

/**
 * @addtogroup sl_btmesh_evt_prov_initialized sl_btmesh_evt_prov_initialized
 * @{
 * @brief Provisioner is initialized and operational.
 */

/** @brief Identifier of the initialized event */
#define sl_btmesh_evt_prov_initialized_id                                0x001500a8

/***************************************************************************//**
 * @brief Data structure of the initialized event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_prov_initialized_s
{
  uint8_t  networks; /**< Number of network keys that the Provisioner has */
  uint16_t address;  /**< Unicast address of the primary element of the
                          Provisioner */
  uint32_t iv_index; /**< IVI for network primary network (index 0) */
});

typedef struct sl_btmesh_evt_prov_initialized_s sl_btmesh_evt_prov_initialized_t;

/** @} */ // end addtogroup sl_btmesh_evt_prov_initialized

/**
 * @addtogroup sl_btmesh_evt_prov_provisioning_suspended sl_btmesh_evt_prov_provisioning_suspended
 * @{
 * @brief Provisioning suspended.
 */

/** @brief Identifier of the provisioning_suspended event */
#define sl_btmesh_evt_prov_provisioning_suspended_id                     0x181500a8

/***************************************************************************//**
 * @brief Data structure of the provisioning_suspended event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_prov_provisioning_suspended_s
{
  uuid_128 uuid;   /**< UUID of the device */
  uint8_t  reason; /**< Reason for suspension. Values are as follows:
                          - <b>0:</b> Capabilities Event received
                          - <b>1:</b> Provisioning link opened */
});

typedef struct sl_btmesh_evt_prov_provisioning_suspended_s sl_btmesh_evt_prov_provisioning_suspended_t;

/** @} */ // end addtogroup sl_btmesh_evt_prov_provisioning_suspended

/**
 * @addtogroup sl_btmesh_evt_prov_capabilities sl_btmesh_evt_prov_capabilities
 * @{
 * @brief Provisioning capabilities message
 */

/** @brief Identifier of the capabilities event */
#define sl_btmesh_evt_prov_capabilities_id                               0x171500a8

/***************************************************************************//**
 * @brief Data structure of the capabilities event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_prov_capabilities_s
{
  uuid_128 uuid;              /**< UUID of the device being provisioned */
  uint8_t  elements;          /**< Number of elements supported by the device */
  uint16_t algorithms;        /**< Supported algorithms and other capabilities. */
  uint8_t  pkey_type;         /**< Supported public key types */
  uint8_t  static_oob_type;   /**< Supported static OOB Types */
  uint8_t  ouput_oob_size;    /**< Maximum size of Output OOB supported */
  uint16_t output_oob_action; /**< Supported Output OOB Actions */
  uint8_t  input_oob_size;    /**< Maximum size of input OOB supported */
  uint16_t intput_oob_action; /**< Supported input OOB Actions */
});

typedef struct sl_btmesh_evt_prov_capabilities_s sl_btmesh_evt_prov_capabilities_t;

/** @} */ // end addtogroup sl_btmesh_evt_prov_capabilities

/**
 * @addtogroup sl_btmesh_evt_prov_provisioning_failed sl_btmesh_evt_prov_provisioning_failed
 * @{
 * @brief Provisioning a device failed.
 */

/** @brief Identifier of the provisioning_failed event */
#define sl_btmesh_evt_prov_provisioning_failed_id                        0x011500a8

/***************************************************************************//**
 * @brief Data structure of the provisioning_failed event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_prov_provisioning_failed_s
{
  uint8_t  reason; /**< Reason for failure. Values are as follows:
                          - <b>0:</b> Link closed
                          - <b>1:</b> Invalid PDU
                          - <b>2:</b> Invalid PDU format
                          - <b>3:</b> Unexpected PDU
                          - <b>4:</b> Confirmation failed
                          - <b>5:</b> Out of resources
                          - <b>6:</b> Decryption failed
                          - <b>7:</b> Unexpected error
                          - <b>8:</b> Unable to assign address */
  uuid_128 uuid;   /**< UUID of the device */
});

typedef struct sl_btmesh_evt_prov_provisioning_failed_s sl_btmesh_evt_prov_provisioning_failed_t;

/** @} */ // end addtogroup sl_btmesh_evt_prov_provisioning_failed

/**
 * @addtogroup sl_btmesh_evt_prov_device_provisioned sl_btmesh_evt_prov_device_provisioned
 * @{
 * @brief Device provisioned successfully.
 */

/** @brief Identifier of the device_provisioned event */
#define sl_btmesh_evt_prov_device_provisioned_id                         0x021500a8

/***************************************************************************//**
 * @brief Data structure of the device_provisioned event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_prov_device_provisioned_s
{
  uint16_t address; /**< Address assigned to the node's primary element. If the
                         node has multiple elements, they have been assigned an
                         address in a consecutive sequence following the primary
                         element address. */
  uuid_128 uuid;    /**< UUID of the device */
});

typedef struct sl_btmesh_evt_prov_device_provisioned_s sl_btmesh_evt_prov_device_provisioned_t;

/** @} */ // end addtogroup sl_btmesh_evt_prov_device_provisioned

/**
 * @addtogroup sl_btmesh_evt_prov_unprov_beacon sl_btmesh_evt_prov_unprov_beacon
 * @{
 * @brief Unprovisioned beacon seen.
 */

/** @brief Identifier of the unprov_beacon event */
#define sl_btmesh_evt_prov_unprov_beacon_id                              0x031500a8

/***************************************************************************//**
 * @brief Data structure of the unprov_beacon event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_prov_unprov_beacon_s
{
  uint16_t oob_capabilities; /**< OOB capabilities bitfield, which indicates the
                                  means by which out-of-band provisioning data
                                  may be retrieved. See @ref
                                  sl_btmesh_prov_oob_capabilities. */
  uint32_t uri_hash;         /**< Hash of the out-of-band URI, which is received
                                  in @ref sl_btmesh_evt_prov_uri. If the URI bit
                                  (bit 1) is not set in the OOB capabilities
                                  bitfield, this field is ignored. */
  uint8_t  bearer;           /**< Bearer on which the beacon was seen. Values
                                  are as follows:
                                    - <b>0:</b> PB-ADV
                                    - <b>1:</b> PB-GATT */
  bd_addr  address;          /**< Address of the device beaconing */
  uint8_t  address_type;     /**< Beaconing device address type. Values:
                                    - <b>0:</b> Public address
                                    - <b>1:</b> Random address */
  uuid_128 uuid;             /**< 16-byte UUID of the beaconing device. */
  int8_t   rssi;             /**< RSSI value of the beacon seen. Units: dBm. */
});

typedef struct sl_btmesh_evt_prov_unprov_beacon_s sl_btmesh_evt_prov_unprov_beacon_t;

/** @} */ // end addtogroup sl_btmesh_evt_prov_unprov_beacon

/**
 * @addtogroup sl_btmesh_evt_prov_oob_pkey_request sl_btmesh_evt_prov_oob_pkey_request
 * @{
 * @brief The Provisioner needs the OOB public key of the Device with given UUID
 *
 * Input the key using prov_oob_pkey_rsp.
 */

/** @brief Identifier of the oob_pkey_request event */
#define sl_btmesh_evt_prov_oob_pkey_request_id                           0x061500a8

/***************************************************************************//**
 * @brief Data structure of the oob_pkey_request event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_prov_oob_pkey_request_s
{
  uuid_128 uuid; /**< UUID of the Device */
});

typedef struct sl_btmesh_evt_prov_oob_pkey_request_s sl_btmesh_evt_prov_oob_pkey_request_t;

/** @} */ // end addtogroup sl_btmesh_evt_prov_oob_pkey_request

/**
 * @addtogroup sl_btmesh_evt_prov_oob_auth_request sl_btmesh_evt_prov_oob_auth_request
 * @{
 * @brief Provide the Provisioner with the device's output or static data using
 * the @ref sl_btmesh_prov_send_oob_auth_response command.
 */

/** @brief Identifier of the oob_auth_request event */
#define sl_btmesh_evt_prov_oob_auth_request_id                           0x071500a8

/***************************************************************************//**
 * @brief Data structure of the oob_auth_request event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_prov_oob_auth_request_s
{
  uint8_t  output;        /**< Zero for static data, non-zero for output */
  uint8_t  output_action; /**< Enum @ref sl_btmesh_node_oob_output_action_t.
                               Output action type. Ignored for Static. */
  uint8_t  output_size;   /**< Size of output data. Ignored for Static. */
  uuid_128 uuid;          /**< UUID of the device */
});

typedef struct sl_btmesh_evt_prov_oob_auth_request_s sl_btmesh_evt_prov_oob_auth_request_t;

/** @} */ // end addtogroup sl_btmesh_evt_prov_oob_auth_request

/**
 * @addtogroup sl_btmesh_evt_prov_oob_display_input sl_btmesh_evt_prov_oob_display_input
 * @{
 * @brief Random OOB input data was generated and should be displayed to and
 * input with the device.
 */

/** @brief Identifier of the oob_display_input event */
#define sl_btmesh_evt_prov_oob_display_input_id                          0x081500a8

/***************************************************************************//**
 * @brief Data structure of the oob_display_input event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_prov_oob_display_input_s
{
  uuid_128   uuid;         /**< UUID of the device */
  uint8_t    input_action; /**< Enum @ref sl_btmesh_node_oob_input_action_t.
                                Input action type */
  uint8_t    input_size;   /**< Number of digits or characters */
  uint8array data;         /**< Raw 16-byte or 32-byte array containing the
                                input authentication data. The length of the
                                data depends on the chosen provisioning
                                algorithm; see @ref
                                sl_btmesh_evt_prov_start_sent event for details
                                on how to determine which provisioning algorithm
                                was chosen.

                                Note that the raw data is encoded in the manner
                                described in the Mesh Protocol 1.1 specifcation
                                Section 5.4.2.4, and must be converted to a
                                human-readable format for displaying. Whether
                                the data should be treated as Numeric or
                                Alphanumeric depends on the selected input
                                action. */
});

typedef struct sl_btmesh_evt_prov_oob_display_input_s sl_btmesh_evt_prov_oob_display_input_t;

/** @} */ // end addtogroup sl_btmesh_evt_prov_oob_display_input

/**
 * @addtogroup sl_btmesh_evt_prov_ddb_list sl_btmesh_evt_prov_ddb_list
 * @{
 * @brief Provisioner's device database list entry
 */

/** @brief Identifier of the ddb_list event */
#define sl_btmesh_evt_prov_ddb_list_id                                   0x091500a8

/***************************************************************************//**
 * @brief Data structure of the ddb_list event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_prov_ddb_list_s
{
  uuid_128 uuid;     /**< UUID of the Device */
  uint16_t address;  /**< Unicast address of the primary element of the node */
  uint8_t  elements; /**< Number of elements the device has */
});

typedef struct sl_btmesh_evt_prov_ddb_list_s sl_btmesh_evt_prov_ddb_list_t;

/** @} */ // end addtogroup sl_btmesh_evt_prov_ddb_list

/**
 * @addtogroup sl_btmesh_evt_prov_uri sl_btmesh_evt_prov_uri
 * @{
 * @brief URI advertisement received from a nearby device.
 */

/** @brief Identifier of the uri event */
#define sl_btmesh_evt_prov_uri_id                                        0x0d1500a8

/***************************************************************************//**
 * @brief Data structure of the uri event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_prov_uri_s
{
  uint32_t   hash; /**< URI hash. If a Provisioner is provisioning a device
                        which supports out-of-band provisioning and has supplied
                        a URI hash value in its Unprovisioned Device beacon, the
                        Provisioner should check whether the hash matches this
                        value. */
  uint8array data; /**< Raw URI data, formatted as specified in Bluetooth Core
                        System Supplement v6. */
});

typedef struct sl_btmesh_evt_prov_uri_s sl_btmesh_evt_prov_uri_t;

/** @} */ // end addtogroup sl_btmesh_evt_prov_uri

/**
 * @addtogroup sl_btmesh_evt_prov_key_refresh_phase_update sl_btmesh_evt_prov_key_refresh_phase_update
 * @{
 * @brief Key refresh phase change for a network key has occurred
 *
 * This event is generated when all nodes participating in a key refresh
 * procedure have been moved to a new state (or have timed out, dropping them
 * from the key refresh procedure).
 */

/** @brief Identifier of the key_refresh_phase_update event */
#define sl_btmesh_evt_prov_key_refresh_phase_update_id                   0x141500a8

/***************************************************************************//**
 * @brief Data structure of the key_refresh_phase_update event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_prov_key_refresh_phase_update_s
{
  uint16_t key;   /**< Network key index */
  uint8_t  phase; /**< Phase moved into */
});

typedef struct sl_btmesh_evt_prov_key_refresh_phase_update_s sl_btmesh_evt_prov_key_refresh_phase_update_t;

/** @} */ // end addtogroup sl_btmesh_evt_prov_key_refresh_phase_update

/**
 * @addtogroup sl_btmesh_evt_prov_key_refresh_node_update sl_btmesh_evt_prov_key_refresh_node_update
 * @{
 * @brief Key refresh phase change for a node has occurred
 *
 * This event is generated when a particular node has moved to a new key refresh
 * phase.
 */

/** @brief Identifier of the key_refresh_node_update event */
#define sl_btmesh_evt_prov_key_refresh_node_update_id                    0x151500a8

/***************************************************************************//**
 * @brief Data structure of the key_refresh_node_update event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_prov_key_refresh_node_update_s
{
  uint16_t key;     /**< Network key index */
  uint8_t  phase;   /**< Phase moved into */
  uuid_128 uuid;    /**< 16-byte UUID of the node. */
  uint8_t  failure; /**< Shows if the node failed the key refresh phase */
});

typedef struct sl_btmesh_evt_prov_key_refresh_node_update_s sl_btmesh_evt_prov_key_refresh_node_update_t;

/** @} */ // end addtogroup sl_btmesh_evt_prov_key_refresh_node_update

/**
 * @addtogroup sl_btmesh_evt_prov_key_refresh_complete sl_btmesh_evt_prov_key_refresh_complete
 * @{
 * @brief Key refresh for a network key has completed
 */

/** @brief Identifier of the key_refresh_complete event */
#define sl_btmesh_evt_prov_key_refresh_complete_id                       0x161500a8

/***************************************************************************//**
 * @brief Data structure of the key_refresh_complete event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_prov_key_refresh_complete_s
{
  uint16_t key;    /**< Network key index */
  uint16_t result; /**< SL_STATUS_OK if successful. Error code otherwise. */
});

typedef struct sl_btmesh_evt_prov_key_refresh_complete_s sl_btmesh_evt_prov_key_refresh_complete_t;

/** @} */ // end addtogroup sl_btmesh_evt_prov_key_refresh_complete

/**
 * @addtogroup sl_btmesh_evt_prov_add_ddb_entry_complete sl_btmesh_evt_prov_add_ddb_entry_complete
 * @{
 * @brief Adding a DDB entry has been completed
 *
 * See the result code for operation status.
 */

/** @brief Identifier of the add_ddb_entry_complete event */
#define sl_btmesh_evt_prov_add_ddb_entry_complete_id                     0x191500a8

/***************************************************************************//**
 * @brief Data structure of the add_ddb_entry_complete event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_prov_add_ddb_entry_complete_s
{
  uint16_t result; /**< SL_STATUS_OK if successful. Error code otherwise. */
  uuid_128 uuid;   /**< UUID of the Device */
});

typedef struct sl_btmesh_evt_prov_add_ddb_entry_complete_s sl_btmesh_evt_prov_add_ddb_entry_complete_t;

/** @} */ // end addtogroup sl_btmesh_evt_prov_add_ddb_entry_complete

/**
 * @addtogroup sl_btmesh_evt_prov_delete_ddb_entry_complete sl_btmesh_evt_prov_delete_ddb_entry_complete
 * @{
 * @brief Deleting a DDB entry has been completed
 *
 * See the result code for operation status.
 */

/** @brief Identifier of the delete_ddb_entry_complete event */
#define sl_btmesh_evt_prov_delete_ddb_entry_complete_id                  0x1a1500a8

/***************************************************************************//**
 * @brief Data structure of the delete_ddb_entry_complete event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_prov_delete_ddb_entry_complete_s
{
  uint16_t result; /**< SL_STATUS_OK if successful. Error code otherwise. */
  uuid_128 uuid;   /**< UUID of the Device */
});

typedef struct sl_btmesh_evt_prov_delete_ddb_entry_complete_s sl_btmesh_evt_prov_delete_ddb_entry_complete_t;

/** @} */ // end addtogroup sl_btmesh_evt_prov_delete_ddb_entry_complete

/**
 * @addtogroup sl_btmesh_evt_prov_initialization_failed sl_btmesh_evt_prov_initialization_failed
 * @{
 * @brief Provisioner has not been initialized successfully and is not
 * operational
 *
 * It is not possible to use the device as a Provisioner. See the result code
 * for details.
 */

/** @brief Identifier of the initialization_failed event */
#define sl_btmesh_evt_prov_initialization_failed_id                      0x1b1500a8

/***************************************************************************//**
 * @brief Data structure of the initialization_failed event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_prov_initialization_failed_s
{
  uint16_t result; /**< SL_STATUS_OK if successful. Error code otherwise. */
});

typedef struct sl_btmesh_evt_prov_initialization_failed_s sl_btmesh_evt_prov_initialization_failed_t;

/** @} */ // end addtogroup sl_btmesh_evt_prov_initialization_failed

/**
 * @addtogroup sl_btmesh_evt_prov_provisioning_records_list sl_btmesh_evt_prov_provisioning_records_list
 * @{
 * @brief Provisioning records list received, or retrieval failed.
 */

/** @brief Identifier of the provisioning_records_list event */
#define sl_btmesh_evt_prov_provisioning_records_list_id                  0x1c1500a8

/***************************************************************************//**
 * @brief Data structure of the provisioning_records_list event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_prov_provisioning_records_list_s
{
  uint16_t   result; /**< SL_STATUS_OK if successful. Error code otherwise. */
  uuid_128   uuid;   /**< UUID of the device */
  uint8array list;   /**< Array of 16-bit provisioning record IDs */
});

typedef struct sl_btmesh_evt_prov_provisioning_records_list_s sl_btmesh_evt_prov_provisioning_records_list_t;

/** @} */ // end addtogroup sl_btmesh_evt_prov_provisioning_records_list

/**
 * @addtogroup sl_btmesh_evt_prov_provisioning_record_data sl_btmesh_evt_prov_provisioning_record_data
 * @{
 * @brief Provisioning record data received, or retrieval failed
 *
 * If successful, record data may be delivered fragmented to several events. The
 * events will be delivered in order.
 */

/** @brief Identifier of the provisioning_record_data event */
#define sl_btmesh_evt_prov_provisioning_record_data_id                   0x1d1500a8

/***************************************************************************//**
 * @brief Data structure of the provisioning_record_data event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_prov_provisioning_record_data_s
{
  uint16_t   result; /**< SL_STATUS_OK if successful. Error code otherwise. */
  uuid_128   uuid;   /**< UUID of the device */
  uint16_t   record; /**< Provisioning record ID */
  uint8_t    final;  /**< Indicates the end of record data; if zero, more events
                          will follow; if nonzero, this is the final data
                          fragment */
  uint8array data;   /**< Raw provisioning record data or fragment thereof */
});

typedef struct sl_btmesh_evt_prov_provisioning_record_data_s sl_btmesh_evt_prov_provisioning_record_data_t;

/** @} */ // end addtogroup sl_btmesh_evt_prov_provisioning_record_data

/**
 * @addtogroup sl_btmesh_evt_prov_start_sent sl_btmesh_evt_prov_start_sent
 * @{
 * @brief Provisioning Start PDU sent
 *
 * This diagnostic event shows the algorithm, public key, and authentication
 * choices Provisioner made and communicated to the unprovisioned device.
 */

/** @brief Identifier of the start_sent event */
#define sl_btmesh_evt_prov_start_sent_id                                 0x1e1500a8

/***************************************************************************//**
 * @brief Data structure of the start_sent event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_prov_start_sent_s
{
  uuid_128 uuid;                  /**< UUID of the device */
  uint8_t  algorithm;             /**< Selected provisioning algorithm */
  uint8_t  public_key;            /**< Selected OOB public key */
  uint8_t  authentication_method; /**< Selected authentication method */
  uint8_t  authentication_action; /**< Selected authentication action */
  uint8_t  authentication_size;   /**< Selected authentication size */
});

typedef struct sl_btmesh_evt_prov_start_sent_s sl_btmesh_evt_prov_start_sent_t;

/** @} */ // end addtogroup sl_btmesh_evt_prov_start_sent

/***************************************************************************//**
 *
 * Initialize the Bluetooth mesh stack in the Provisioner role. Note that the
 * result code of this command only indicates that the request to initialize has
 * been accepted for processing. When initialization is completed successfully,
 * a @ref sl_btmesh_evt_prov_initialized will be generated. On failed
 * initialization a @ref sl_btmesh_evt_prov_initialization_failed event will be
 * generated. Note that the application must wait for an event to be generated
 * before executing further BGAPI commands.
 *
 * This command must be issued before any other Bluetooth mesh stack commands.
 * Note that the Bluetooth mesh stack can be initialized either in the
 * Provisioner or the Node role, but not both.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_prov_initialized
 *   - @ref sl_btmesh_evt_prov_initialization_failed
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_init();

/***************************************************************************//**
 *
 * Start scanning for unprovisioned device beacons.
 *
 * Unprovisioned devices send out beacons containing their UUID. An @ref
 * sl_btmesh_evt_prov_unprov_beacon will be generated for each beacon seen. Once
 * the UUID of a device is known, the Provisioner may start provisioning the
 * device with the @ref sl_btmesh_prov_create_provisioning_session command.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_prov_unprov_beacon
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_scan_unprov_beacons();

/***************************************************************************//**
 *
 * Initialize the provisioning session. After this command, it is possible to
 * set the provisioning parameter with the following command:
 *
 *   - @ref sl_btmesh_prov_set_provisioning_suspend_event
 *
 * After all optional parameters have been set, the provisioning process is
 * triggered by one of the following commands:
 *
 *   - @ref sl_btmesh_prov_provision_adv_device
 *   - @ref sl_btmesh_prov_provision_gatt_device
 *
 * @param[in] netkey_index Index of the initial network key, which is sent to
 *   the device during provisioning
 * @param[in] uuid UUID of the device
 * @param[in] attention_timer_sec Attention timer value, in seconds, which
 *   indicates the time that the provisioned device should attract human
 *   attention
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 * 
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_create_provisioning_session(uint16_t netkey_index,
                                                       uuid_128 uuid,
                                                       uint8_t attention_timer_sec);

/***************************************************************************//**
 *
 * Set whether the provisioning can be suspended at a specified point while it
 * is occurring. Currently, this can happen after reception on provisioning
 * capabilities message or when the provisioning link is opened.
 *
 * @param[in] status @parblock
 *   Controls when provisioning is suspended and @ref
 *   sl_btmesh_evt_prov_provisioning_suspended event is created. The value is a
 *   bitmap so multiple possibilities can be set.
 *
 *     - <b>Bit 0 set</b> provisioning will be suspended when Capabilities PDU
 *       is received from the unprovisioned device
 *     - <b>Bit 1 set</b> provisioning will be suspended when the provisioning
 *       session is opened
 *
 *   If no bits are set (value is zero) provisioning will not be suspended. This
 *   is the default.
 *   @endparblock
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_prov_provisioning_suspended
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_set_provisioning_suspend_event(uint8_t status);

/***************************************************************************//**
 *
 * Provision device over PB-ADV
 *
 * @param[in] uuid UUID of the device
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_prov_device_provisioned
 *   - @ref sl_btmesh_evt_prov_provisioning_failed
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_provision_adv_device(uuid_128 uuid);

/***************************************************************************//**
 *
 * Provision device over PB-GATT
 *
 * @param[in] uuid UUID of the device
 * @param[in] connection Connection handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_prov_device_provisioned
 *   - @ref sl_btmesh_evt_prov_provisioning_failed
 *   - @ref sl_btmesh_evt_prov_capabilities
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_provision_gatt_device(uuid_128 uuid,
                                                 uint8_t connection);

/***************************************************************************//**
 *
 * Provision device over remote provisioning server connection
 *
 * @param[in] uuid UUID of the device
 * @param[in] server Address of the remote provisioning server
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_prov_device_provisioned
 *   - @ref sl_btmesh_evt_prov_provisioning_failed
 *   - @ref sl_btmesh_evt_prov_capabilities
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_provision_remote_device(uuid_128 uuid,
                                                   uint16_t server);

/***************************************************************************//**
 *
 * Continue provisioning after the suspension indicated by @ref
 * sl_btmesh_evt_prov_provisioning_suspended.
 *
 * @param[in] uuid UUID of the device being provisioned
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 * 
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_continue_provisioning(uuid_128 uuid);

/***************************************************************************//**
 *
 * Abort provisioning. This command is typically used after the suspension
 * indicated by @ref sl_btmesh_evt_prov_provisioning_suspended.
 *
 * @param[in] uuid UUID of the device being provisioned
 * @param[in] reason Reason for aborting. Values are as follows:
 *     - <b>1:</b> Invalid PDU
 *     - <b>2:</b> Invalid PDU format
 *     - <b>3:</b> Unexpected PDU
 *     - <b>4:</b> Confirmation failed
 *     - <b>5:</b> Out of resources
 *     - <b>6:</b> Decryption failed
 *     - <b>7:</b> Unexpected error
 *     - <b>8:</b> Unable to assign address
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 * 
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_abort_provisioning(uuid_128 uuid, uint8_t reason);

/***************************************************************************//**
 *
 * Set the address for the device after the provisioning has been suspended, as
 * indicated by @ref sl_btmesh_evt_prov_provisioning_suspended event.
 *
 * @param[in] uuid UUID of the device being provisioned
 * @param[in] address Unicast address to be assigned for the primary element of
 *   the node.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 * 
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_set_device_address(uuid_128 uuid, uint16_t address);

/***************************************************************************//**
 *
 * Create a new network key on the Provisioner.
 *
 * The created key can be deployed on a node using the @ref
 * sl_btmesh_config_client_add_netkey command.
 *
 * @param[in] netkey_index Index to use for network key. Allowed values are from
 *   0x000 to 0xfff.
 * @param[in] key_len Length of data in @p key
 * @param[in] key Key value to use. Set to zero-length array to generate a
 *   random key.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_create_network(uint16_t netkey_index,
                                          size_t key_len,
                                          const uint8_t* key);

/***************************************************************************//**
 *
 * Create a new application key on the Provisioner.
 *
 * An application key is always bound to a network key. In other words, the
 * application key is only valid in the context of a particular network key. The
 * selected network key must exist on the Provisioner (see @ref
 * sl_btmesh_prov_create_network command).
 *
 * The created application key can be deployed on a node using the @ref
 * sl_btmesh_config_client_add_appkey command.
 *
 * @param[in] netkey_index Index of the network key to which the application key
 *   will be bound
 * @param[in] appkey_index Index to use for application key. Allowed values are
 *   from 0x000 to 0xfff.
 * @param[in] key_len Length of data in @p key
 * @param[in] key Key value to use; set to zero-length array to generate random
 *   key.
 * @param[in] max_application_key_size Size of output buffer passed in @p
 *   application_key
 * @param[out] application_key_len On return, set to the length of output data
 *   written to @p application_key
 * @param[out] application_key New application key. Ignore it if the result was
 *   non-zero.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_create_appkey(uint16_t netkey_index,
                                         uint16_t appkey_index,
                                         size_t key_len,
                                         const uint8_t* key,
                                         size_t max_application_key_size,
                                         size_t *application_key_len,
                                         uint8_t *application_key);

/***************************************************************************//**
 *
 * Respond to the prov_oob_pkey_request.
 *
 * @param[in] uuid UUID of the Device
 * @param[in] pkey_len Length of data in @p pkey
 * @param[in] pkey Public Key read out-of-band
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_send_oob_pkey_response(uuid_128 uuid,
                                                  size_t pkey_len,
                                                  const uint8_t* pkey);

/***************************************************************************//**
 *
 * Respond to the prov_oob_auth_request.
 *
 * @param[in] uuid UUID of the Device
 * @param[in] data_len Length of data in @p data
 * @param[in] data @parblock
 *   Raw 16-byte or 32-byte array containing the static or output authentication
 *   data. The length of the data depends on the chosen provisioning algorithm;
 *   see @ref sl_btmesh_evt_prov_start_sent event for details on how to
 *   determine which provisioning algorithm was chosen.
 *
 *   Note that the raw data given to the stack must be encoded in the manner
 *   described in the Mesh Protocol 1.1 specifcation Section 5.4.2.4. Whether
 *   the output data should be treated as Numeric or Alphanumeric depends on the
 *   selected output action. The data type for Static OOB authentication data is
 *   Binary.
 *   @endparblock
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_send_oob_auth_response(uuid_128 uuid,
                                                  size_t data_len,
                                                  const uint8_t* data);

/***************************************************************************//**
 *
 * Set the OOB requirements for a device being provisioned. This command can be
 * issued after @ref sl_btmesh_prov_create_provisioning_session.
 *
 * @param[in] uuid UUID of the Device being provisioned
 * @param[in] public_key The public key. Set to zero if the provisioning does
 *   not use OOB public Key.
 * @param[in] auth_methods Enum @ref sl_btmesh_node_auth_method_flag_t. Allowed
 *   OOB authentication methods The value is a bitmap so that multiple methods
 *   can be supported.
 * @param[in] output_actions Enum @ref sl_btmesh_node_oob_output_action_flag_t.
 *   Allowed OOB Output Action types
 * @param[in] input_actions Enum @ref sl_btmesh_node_oob_input_action_flag_t.
 *   Allowed OOB Input Action types
 * @param[in] min_size Minimum input/output OOB size. Values range from 0
 *   (input/output OOB not used) to 8.
 * @param[in] max_size Maximum input/output OOB size. Must be larger than or
 *   equal to the minimum size. Values range from 0 (input/output OOB not used)
 *   to 8.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_set_oob_requirements(uuid_128 uuid,
                                                uint8_t public_key,
                                                uint8_t auth_methods,
                                                uint16_t output_actions,
                                                uint16_t input_actions,
                                                uint8_t min_size,
                                                uint8_t max_size);

/***************************************************************************//**
 *
 * Start a key refresh procedure in the network.
 *
 * A key refresh procedure updates a network key and, optionally, application
 * keys associated with it in all nodes of the network except for excluded
 * nodes. After the refresh procedure is complete, the old keys will be
 * discarded. Therefore, the excluded nodes, which did not receive new keys will
 * be shut out of the network at the completion of the procedure.
 *
 * @param[in] netkey_index Index of the network key to update
 * @param[in] num_appkeys Number of application keys to update; may be zero.
 * @param[in] appkey_indices_len Length of data in @p appkey_indices
 * @param[in] appkey_indices Indices of the application keys to update,
 *   represented as little endian two byte sequences. The array must contain
 *   num_appkeys indices and therefore 2*num_appkeys bytes total.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_prov_key_refresh_node_update
 *   - @ref sl_btmesh_evt_prov_key_refresh_phase_update
 *   - @ref sl_btmesh_evt_prov_key_refresh_complete
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_start_key_refresh(uint16_t netkey_index,
                                             uint8_t num_appkeys,
                                             size_t appkey_indices_len,
                                             const uint8_t* appkey_indices);

/***************************************************************************//**
 *
 * Check the key refresh exclusion status of a node. Excluded nodes do not
 * participate in the key refresh procedure and can therefore be shut out of the
 * network.
 *
 * @param[in] key Network key index
 * @param[in] uuid UUID of the Device
 * @param[out] status Non-zero for excluded node
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_get_key_refresh_exclusion(uint16_t key,
                                                     uuid_128 uuid,
                                                     uint8_t *status);

/***************************************************************************//**
 *
 * Set the key refresh exclusion status of a node. Excluded nodes do not
 * participate in the key refresh procedure and can therefore be shut out of the
 * network.
 *
 * @param[in] key Network key index
 * @param[in] status Non-zero for excluded node
 * @param[in] uuid UUID of the Device
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_set_key_refresh_exclusion(uint16_t key,
                                                     uint8_t status,
                                                     uuid_128 uuid);

/***************************************************************************//**
 *
 * Mark the node as failed for the keyrefresh. This function should be called if
 * the provisioner knows that the node has become unresponsive for some external
 * reason (such as factory reset performed by the end user), and will not
 * respond to key refresh messages even after a long delay. Note that if a node
 * drops out of the key refresh procedure, it will not be possible to
 * communicate with the node using the key that was refreshed during the key
 * refresh procedure.
 *
 * @param[in] key Network key index
 * @param[in] uuid UUID of the Device
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_set_key_refresh_failure(uint16_t key, uuid_128 uuid);

/***************************************************************************//**
 *
 * Get the phase_timeout_s parameter that will be used for the start key refresh
 * command. A node fails the key refresh procedure phase, if it doesnt reply to
 * the provisioner during the timeout.
 *
 * @param[out] phase_timeout_s Value to get in seconds
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_phase_timeout_get(uint32_t *phase_timeout_s);

/***************************************************************************//**
 *
 * Set the phase_timeout_s parameter that will be used for the start key refresh
 * command. A node fails the key refresh procedure phase, if it doesnt reply to
 * the provisioner during the timeout. Maximum allowed value is 345600 sec
 * (4days).
 *
 * @param[in] phase_timeout_s Value to be set in seconds
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_phase_timeout_set(uint32_t phase_timeout_s);

/***************************************************************************//**
 *
 * Get a Provisioner device database entry with a matching UUID.
 *
 * @param[in] uuid UUID of the Device to retrieve
 * @param[out] device_key Device Key
 * @param[out] netkey_index Index of the network key with which the node was
 *   initially provisioned. Used for network-level encryption of Configuration
 *   Client messages.
 * @param[out] address Unicast address of the primary element of the node
 * @param[out] elements Number of elements in the node
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_get_ddb_entry(uuid_128 uuid,
                                         aes_key_128 *device_key,
                                         uint16_t *netkey_index,
                                         uint16_t *address,
                                         uint8_t *elements);

/***************************************************************************//**
 *
 * Delete the node information from the Provisioner database. Note that a
 * successful result from this command only means the command has been accepted
 * for processing. The status of the actual operation will be returned in the
 * following event; application should not make new BGAPI requests until the
 * event is received. Note also that this should be followed by a @ref
 * sl_btmesh_prov_start_key_refresh updating the keys of the remaining nodes to
 * make sure the deleted node is shut off from the network.
 *
 * @param[in] uuid UUID of the node to delete
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_prov_delete_ddb_entry_complete
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_delete_ddb_entry(uuid_128 uuid);

/***************************************************************************//**
 *
 * Add a new node entry to the Provisioner's device database. Note that a
 * successful result from this command only means the command has been accepted
 * for processing. The status of the actual operation will be returned in the
 * following event; application should not make new BGAPI requests until the
 * event is received. Note also that the device key, primary element address,
 * and network key need to be deployed to the node being added to ensure it's
 * configurable. See @ref sl_btmesh_node_set_provisioning_data command.
 *
 * @param[in] uuid UUID of the node to add
 * @param[in] device_key Device key value for the node
 * @param[in] netkey_index Index of the network key the node will be used for
 *   configuration
 * @param[in] address Unicast address to allocate for the node's primary element
 * @param[in] elements Number of elements the device has
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_prov_add_ddb_entry_complete
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_add_ddb_entry(uuid_128 uuid,
                                         aes_key_128 device_key,
                                         uint16_t netkey_index,
                                         uint16_t address,
                                         uint8_t elements);

/***************************************************************************//**
 *
 * List nodes known by this Provisioner. A number of @ref
 * sl_btmesh_evt_prov_ddb_list events will be generated.
 *
 * @param[out] count Number of events that will follow
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_prov_ddb_list
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_list_ddb_entries(uint16_t *count);

/***************************************************************************//**
 *
 * Initialize the mesh network and assign the provisioner address and IV index
 * for the network. If this command is not invoked prior to invoking @ref
 * sl_btmesh_prov_create_network, the network will be initialized with default
 * address and IV index.
 *
 * @param[in] address Address to assign for provisioner.
 * @param[in] ivi IV index of the network.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_initialize_network(uint16_t address, uint32_t ivi);

/***************************************************************************//**
 *
 * Check the application key refresh exclusion status of a node. Nodes which are
 * excluded for a given application key do not receive updates for that
 * particular application key, but do participate in the key refresh procedure
 * as a whole. This enables the Provisioner to set up and update restricted sets
 * of application keys across nodes.
 *
 * @param[in] netkey_index Network key index
 * @param[in] appkey_index Application key index
 * @param[in] uuid UUID of the Device
 * @param[out] status Non-zero for excluded node
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_get_key_refresh_appkey_exclusion(uint16_t netkey_index,
                                                            uint16_t appkey_index,
                                                            uuid_128 uuid,
                                                            uint8_t *status);

/***************************************************************************//**
 *
 * Set the application key refresh exclusion status of a node. Nodes which are
 * excluded for a given application key do not receive updates for that
 * particular application key, but do participate in the key refresh procedure
 * as a whole. This enables the Provisioner to set up and update restricted sets
 * of application keys across nodes.
 *
 * @param[in] netkey_index Network key index
 * @param[in] appkey_index Application key index
 * @param[in] status Non-zero for excluded node
 * @param[in] uuid UUID of the device
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_set_key_refresh_appkey_exclusion(uint16_t netkey_index,
                                                            uint16_t appkey_index,
                                                            uint8_t status,
                                                            uuid_128 uuid);

/***************************************************************************//**
 *
 * Stop scanning for unprovisioned device beacons.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_stop_scan_unprov_beacons();

/***************************************************************************//**
 *
 * Update a node's entry in the Provisioner's device database by setting a new
 * value to the netkey_index field. The netkey_index field is used to determine
 * the network key to use when encrypting and decrypting configuration model
 * messages to and from the node.
 *
 * @param[in] uuid UUID of the node
 * @param[in] netkey_index Index of the network key used in configuring the
 *   node.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_update_device_netkey_index(uuid_128 uuid,
                                                      uint16_t netkey_index);

/***************************************************************************//**
 *
 * Suspend an ongoing key refresh procedure.
 *
 * Suspending a key refresh procedure means no further requests for updating
 * keys or setting key refresh phase will be sent to the network by the
 * Provisioner until the key refresh procedure is resumed.
 *
 * @param[in] netkey_index Index of the network key identifying an ongoing key
 *   refresh procedure
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_suspend_key_refresh(uint16_t netkey_index);

/***************************************************************************//**
 *
 * Resume a suspended key refresh procedure.
 *
 * By resuming a suspended key refresh procedure, the Provisioner will again
 * start to send requests for updating keys or setting key refresh phase to the
 * network.
 *
 * @param[in] netkey_index Index of the network key identifying a suspended key
 *   refresh procedure
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_resume_key_refresh(uint16_t netkey_index);

/***************************************************************************//**
 *
 * Get the key refresh phase of an ongoing key refresh procedure.
 *
 * @param[in] netkey_index Index of the network key identifying an ongoing key
 *   refresh procedure
 * @param[out] phase The current key refresh phase. Values are as follows:
 *     - 0: Normal operation (no ongoing key refresh)
 *     - 1: First phase of key refresh procedure (key deployment)
 *     - 2: Second phase of key refresh procedure (new key activation)
 *     - 3: Third phase of key refresh procedure (old key deletion)
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_get_key_refresh_phase(uint16_t netkey_index,
                                                 uint8_t *phase);

/***************************************************************************//**
 *
 * Start a key refresh procedure from a non-default phase. Before calling this
 * function, keys to be used in the key refresh procedure should have been
 * specified by calling @ref sl_btmesh_test_prov_prepare_key_refresh command.
 *
 * Note that this command should not normally be used. It is intended only for
 * resuming an interrupted key refresh procedure on a backup Provisioner when
 * the original Provisioner, that started the key refresh procedure, is no
 * longer available to complete the procedure.
 *
 * @param[in] phase Current key refresh phase
 * @param[in] netkey_index Index of the network key identifying a key refresh
 *   procedure
 * @param[in] num_appkeys Number of application keys to update; may be zero.
 * @param[in] appkey_indices_len Length of data in @p appkey_indices
 * @param[in] appkey_indices Indices of the application keys to update,
 *   represented as little endian two byte sequences. The array must contain
 *   num_appkeys indices and therefore 2*num_appkeys bytes total.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_prov_key_refresh_node_update
 *   - @ref sl_btmesh_evt_prov_key_refresh_phase_update
 *   - @ref sl_btmesh_evt_prov_key_refresh_complete
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_start_key_refresh_from_phase(uint8_t phase,
                                                        uint16_t netkey_index,
                                                        uint8_t num_appkeys,
                                                        size_t appkey_indices_len,
                                                        const uint8_t* appkey_indices);

/***************************************************************************//**
 *
 * Clear the key refresh state stored in persistent storage.
 *
 * Note that this command should not normally be used. It is intended only for
 * clearing stored key refresh state when a key refresh procedure has been
 * suspended and will not be resumed, either because the network key has been
 * deleted from all nodes or the responsibility for completing the key refresh
 * has been moved to another Provisioner.
 *
 * @param[in] netkey_index Index of the network key identifying a key refresh
 *   procedure
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_flush_key_refresh_state(uint16_t netkey_index);

/***************************************************************************//**
 *
 * Get the list of provisioning records on the device. This command can be
 * issued when provisioning is suspended after the provisioning session has been
 * opened, but no earlier or later. The list of provisioning records on the
 * device will be delivered with the @ref
 * sl_btmesh_evt_prov_provisioning_records_list event.
 *
 * @param[in] uuid UUID of the Device being provisioned
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_prov_provisioning_records_list
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_get_provisioning_records_list(uuid_128 uuid);

/***************************************************************************//**
 *
 * Get provisioning record data from the device. This command can be issued when
 * provisioning is suspended after the provisioning session has been opened, but
 * no earlier or later. The provisioning record data will be delivered with the
 * @ref sl_btmesh_evt_prov_provisioning_record_data event.
 *
 * @param[in] uuid UUID of the Device being provisioned
 * @param[in] record Provisioning record ID
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_prov_provisioning_record_data
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_get_provisioning_record_data(uuid_128 uuid,
                                                        uint16_t record);

/***************************************************************************//**
 *
 * Initialize provisioning record retrieval functionality on the Provisioner.
 * This must be called before trying to retrieve provisioning records list or
 * individual record data.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_prov_init_provisioning_records();

/** @} */ // end addtogroup sl_btmesh_prov

/**
 * @addtogroup sl_btmesh_proxy Bluetooth Mesh Proxy Connections
 * @{
 *
 * @brief Bluetooth Mesh Proxy Connections
 *
 * Bluetooth mesh stack functions for GATT proxy connections
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_proxy_connect_id                                   0x00180028
#define sl_btmesh_cmd_proxy_disconnect_id                                0x01180028
#define sl_btmesh_cmd_proxy_set_filter_type_id                           0x02180028
#define sl_btmesh_cmd_proxy_allow_id                                     0x03180028
#define sl_btmesh_cmd_proxy_deny_id                                      0x04180028
#define sl_btmesh_cmd_proxy_optimisation_toggle_id                       0x05180028
#define sl_btmesh_cmd_proxy_send_solicitation_id                         0x06180028
#define sl_btmesh_rsp_proxy_connect_id                                   0x00180028
#define sl_btmesh_rsp_proxy_disconnect_id                                0x01180028
#define sl_btmesh_rsp_proxy_set_filter_type_id                           0x02180028
#define sl_btmesh_rsp_proxy_allow_id                                     0x03180028
#define sl_btmesh_rsp_proxy_deny_id                                      0x04180028
#define sl_btmesh_rsp_proxy_optimisation_toggle_id                       0x05180028
#define sl_btmesh_rsp_proxy_send_solicitation_id                         0x06180028

/**
 * @addtogroup sl_btmesh_evt_proxy_connected sl_btmesh_evt_proxy_connected
 * @{
 * @brief Indication that a connection has been successfully formed, received as
 * a response to the @ref sl_btmesh_proxy_connect command.
 */

/** @brief Identifier of the connected event */
#define sl_btmesh_evt_proxy_connected_id                                 0x001800a8

/***************************************************************************//**
 * @brief Data structure of the connected event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_proxy_connected_s
{
  uint32_t handle; /**< Proxy handle */
});

typedef struct sl_btmesh_evt_proxy_connected_s sl_btmesh_evt_proxy_connected_t;

/** @} */ // end addtogroup sl_btmesh_evt_proxy_connected

/**
 * @addtogroup sl_btmesh_evt_proxy_disconnected sl_btmesh_evt_proxy_disconnected
 * @{
 * @brief Indication that a connection has been disconnected or a connection
 * attempt failed, received as a response to @ref sl_btmesh_proxy_connect or
 * @ref sl_btmesh_proxy_disconnect command.
 */

/** @brief Identifier of the disconnected event */
#define sl_btmesh_evt_proxy_disconnected_id                              0x011800a8

/***************************************************************************//**
 * @brief Data structure of the disconnected event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_proxy_disconnected_s
{
  uint32_t handle; /**< Proxy handle */
  uint16_t reason; /**< Reason for disconnection */
});

typedef struct sl_btmesh_evt_proxy_disconnected_s sl_btmesh_evt_proxy_disconnected_t;

/** @} */ // end addtogroup sl_btmesh_evt_proxy_disconnected

/**
 * @addtogroup sl_btmesh_evt_proxy_filter_status sl_btmesh_evt_proxy_filter_status
 * @{
 * @brief Proxy status report event
 */

/** @brief Identifier of the filter_status event */
#define sl_btmesh_evt_proxy_filter_status_id                             0x021800a8

/***************************************************************************//**
 * @brief Data structure of the filter_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_proxy_filter_status_s
{
  uint32_t handle; /**< Proxy handle */
  uint8_t  type;   /**< Filter type: 0x00 to allow traffic from the given
                        address, 0x01 to block traffic from the given address */
  uint16_t count;  /**< Current filter list length */
});

typedef struct sl_btmesh_evt_proxy_filter_status_s sl_btmesh_evt_proxy_filter_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_proxy_filter_status

/***************************************************************************//**
 *
 * Start connecting a proxy client to a proxy server. After the connection is
 * complete, a @ref sl_btmesh_evt_proxy_connected event will be generated.
 * LE-connection must be opened prior to opening the proxy connection.
 *
 * @param[in] connection Connection handle
 * @param[out] handle If a connection attempt is successfully initiated, a valid
 *   proxy handle will be returned.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_proxy_connected
 *   - @ref sl_btmesh_evt_proxy_disconnected
 *
 ******************************************************************************/
sl_status_t sl_btmesh_proxy_connect(uint8_t connection, uint32_t *handle);

/***************************************************************************//**
 *
 * Disconnect. This call can be used also for a connection, which is not yet
 * fully formed.
 *
 * @param[in] handle Proxy handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_proxy_disconnected
 *
 ******************************************************************************/
sl_status_t sl_btmesh_proxy_disconnect(uint32_t handle);

/***************************************************************************//**
 *
 * Set up the proxy filtering type. At the proxy server side, this is a local
 * configuration, while on the proxy client a proxy configuration PDU will be
 * sent to the proxy server.
 *
 * @param[in] handle Proxy handle
 * @param[in] netkey_index Network key index used in encrypting the request to
 *   the proxy server
 * @param[in] type Filter type: 0x00 to allow traffic from the given address,
 *   0x01 to block traffic from the given address
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_proxy_set_filter_type(uint32_t handle,
                                            uint16_t netkey_index,
                                            uint8_t type);

/***************************************************************************//**
 *
 * Allow messages meant for the given address to be forwarded over the proxy
 * connection to the proxy client. At the proxy server side, this is a local
 * configuration, while on the proxy client a proxy configuration PDU will be
 * sent to the proxy server.
 *
 * @param[in] handle Proxy handle
 * @param[in] netkey_index Network key index used in encrypting the request to
 *   the proxy server
 * @param[in] address Destination address to allow. The address may be either a
 *   unicast address, a group address, or a virtual address.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_proxy_allow(uint32_t handle,
                                  uint16_t netkey_index,
                                  uint16_t address);

/***************************************************************************//**
 *
 * Block messages meant for the given address from being forwarded over the
 * proxy connection to the proxy client. At the proxy server side, this is a
 * local configuration, while on the proxy client a proxy configuration PDU will
 * be sent to the proxy server.
 *
 * @param[in] handle Proxy handle
 * @param[in] netkey_index Network key index used in encrypting the request to
 *   the proxy server
 * @param[in] address Destination address to block. The address may be either a
 *   unicast address, a group address, or a virtual address.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_proxy_deny(uint32_t handle,
                                 uint16_t netkey_index,
                                 uint16_t address);

/***************************************************************************//**
 *
 * In case of unicast address, if proxy identified the destination, the message
 * will be forwarded only to that node, otherwise to all. This functionality
 * could be enabled or disabled with this function.
 *
 * @param[in] enable Non zero - enable, otherwise disable
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_proxy_optimisation_toggle(uint8_t enable);

/***************************************************************************//**
 *
 * Send Solicitation PDU to activate a Private Proxy on a node that support
 * On-Demand Private Proxy.
 *
 * @param[in] enc_netkey_index Network key index used to encrypt the message
 * @param[in] elem_index Solicitation source element index. Proxy client may use
 *   the primary address of any of the secondary addresses as a solicitation
 *   source
 * @param[in] dst Solicitation destination address. Solicitation does not have a
 *   subscribe mechanism, the address can be the primary or any of the secondary
 *   addresses. In addition, 0xFFFC (All Proxies) is permitted
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_proxy_send_solicitation(uint16_t enc_netkey_index,
                                              uint16_t elem_index,
                                              uint16_t dst);

/** @} */ // end addtogroup sl_btmesh_proxy

/**
 * @addtogroup sl_btmesh_silabs_config_client Bluetooth Mesh Silabs Configuration Client Vendor Model
 * @{
 *
 * @brief Bluetooth Mesh Silabs Configuration Client Vendor Model
 *
 * Silicon Labs Configuration Client vendor model. It is used to configure
 * global and per-model specific settings of nodes such as Tx-PHY for long
 * packets, usage of advertising extensions (AE) on per-model basis.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_silabs_config_client_init_id                       0x00010028
#define sl_btmesh_cmd_silabs_config_client_set_tx_id                     0x01010028
#define sl_btmesh_cmd_silabs_config_client_get_tx_id                     0x02010028
#define sl_btmesh_cmd_silabs_config_client_set_model_enable_id           0x03010028
#define sl_btmesh_cmd_silabs_config_client_get_model_enable_id           0x04010028
#define sl_btmesh_cmd_silabs_config_client_set_network_pdu_id            0x05010028
#define sl_btmesh_cmd_silabs_config_client_get_network_pdu_id            0x06010028
#define sl_btmesh_cmd_silabs_config_client_deinit_id                     0x07010028
#define sl_btmesh_rsp_silabs_config_client_init_id                       0x00010028
#define sl_btmesh_rsp_silabs_config_client_set_tx_id                     0x01010028
#define sl_btmesh_rsp_silabs_config_client_get_tx_id                     0x02010028
#define sl_btmesh_rsp_silabs_config_client_set_model_enable_id           0x03010028
#define sl_btmesh_rsp_silabs_config_client_get_model_enable_id           0x04010028
#define sl_btmesh_rsp_silabs_config_client_set_network_pdu_id            0x05010028
#define sl_btmesh_rsp_silabs_config_client_get_network_pdu_id            0x06010028
#define sl_btmesh_rsp_silabs_config_client_deinit_id                     0x07010028

/**
 * @brief The Silabs Configuration Client Status codes.
 */
typedef enum
{
  sl_btmesh_silabs_config_client_status_success           = 0x0, /**< (0x0) The
                                                                      Client
                                                                      message
                                                                      was
                                                                      successfully
                                                                      processed
                                                                      by the
                                                                      Server. */
  sl_btmesh_silabs_config_client_status_unknown_cmd       = 0x1, /**< (0x1) The
                                                                      command
                                                                      was not
                                                                      understood
                                                                      by the
                                                                      Server. */
  sl_btmesh_silabs_config_client_status_invalid_parameter = 0x2, /**< (0x2) A
                                                                      parameter
                                                                      in the
                                                                      Client
                                                                      message is
                                                                      outside of
                                                                      its valid
                                                                      range. */
  sl_btmesh_silabs_config_client_status_does_not_exist    = 0x3  /**< (0x3)
                                                                      Resource
                                                                      specifed
                                                                      by the
                                                                      Client
                                                                      message
                                                                      does not
                                                                      exist. */
} sl_btmesh_silabs_config_client_status_t;

/**
 * @addtogroup sl_btmesh_evt_silabs_config_client_tx_status sl_btmesh_evt_silabs_config_client_tx_status
 * @{
 * @brief Node specific TX configuration state
 *
 * Received as a response to the @ref sl_btmesh_silabs_config_client_set_tx and
 * @ref sl_btmesh_silabs_config_client_get_tx command.
 */

/** @brief Identifier of the tx_status event */
#define sl_btmesh_evt_silabs_config_client_tx_status_id                  0x010100a8

/***************************************************************************//**
 * @brief Data structure of the tx_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_silabs_config_client_tx_status_s
{
  uint16_t server;            /**< Server address. */
  uint8_t  status;            /**< Enum @ref
                                   sl_btmesh_silabs_config_client_status_t.
                                   Status code from the Server. */
  uint8_t  tx_phy;            /**< TX PHY for long packets (packets that would
                                   be segmented).
                                     - 1: LE 1M PHY
                                     - 2: LE 2M PHY */
  uint32_t tx_options_bitmap; /**< TX options. NOTE: Currently not used:
                                   reserved for future use. */
});

typedef struct sl_btmesh_evt_silabs_config_client_tx_status_s sl_btmesh_evt_silabs_config_client_tx_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_silabs_config_client_tx_status

/**
 * @addtogroup sl_btmesh_evt_silabs_config_client_model_status sl_btmesh_evt_silabs_config_client_model_status
 * @{
 * @brief TX over AE on per-model basis enable/disable state
 *
 * Received as a response to the @ref
 * sl_btmesh_silabs_config_client_set_model_enable and @ref
 * sl_btmesh_silabs_config_client_get_model_enable command.
 */

/** @brief Identifier of the model_status event */
#define sl_btmesh_evt_silabs_config_client_model_status_id               0x020100a8

/***************************************************************************//**
 * @brief Data structure of the model_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_silabs_config_client_model_status_s
{
  uint16_t server;     /**< Server address. */
  uint8_t  status;     /**< Enum @ref sl_btmesh_silabs_config_client_status_t.
                            Status code from the Server. */
  uint8_t  elem_index; /**< Element index of model to configure. */
  uint16_t vendor_id;  /**< Vendor ID of model to configure. */
  uint16_t model_id;   /**< Model ID of model to configure. */
  uint16_t value;      /**< Extended packet size to set, currently 1 to use
                            extended 0 to not */
});

typedef struct sl_btmesh_evt_silabs_config_client_model_status_s sl_btmesh_evt_silabs_config_client_model_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_silabs_config_client_model_status

/**
 * @addtogroup sl_btmesh_evt_silabs_config_client_network_pdu_status sl_btmesh_evt_silabs_config_client_network_pdu_status
 * @{
 * @brief Node specific Network PDU configuration state
 *
 * Received as a response to the @ref
 * sl_btmesh_silabs_config_client_set_network_pdu and @ref
 * sl_btmesh_silabs_config_client_get_network_pdu command.
 */

/** @brief Identifier of the network_pdu_status event */
#define sl_btmesh_evt_silabs_config_client_network_pdu_status_id         0x030100a8

/***************************************************************************//**
 * @brief Data structure of the network_pdu_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_silabs_config_client_network_pdu_status_s
{
  uint16_t server;       /**< Server address. */
  uint8_t  status;       /**< Enum @ref sl_btmesh_silabs_config_client_status_t.
                              Status code from the Server. */
  uint16_t pdu_max_size; /**< Network PDU max size */
});

typedef struct sl_btmesh_evt_silabs_config_client_network_pdu_status_s sl_btmesh_evt_silabs_config_client_network_pdu_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_silabs_config_client_network_pdu_status

/***************************************************************************//**
 *
 * Initialize Silabs Configuration client model.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_silabs_config_client_init();

/***************************************************************************//**
 *
 * Set node specific TX configuration.
 *
 * @param[in] server_address Destination address of the message. It can be a
 *   unicast address, a group address, or a virtual address.
 * @param[in] appkey_index The application key index used
 * @param[in] tx_phy TX PHY for long packets (packets that would be segmented).
 *     - 1: LE 1M PHY
 *     - 2: LE 2M PHY
 * @param[in] tx_options_bitmap TX options. NOTE: Currently not used: reserved
 *   for future use, set as 0.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_silabs_config_client_tx_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_silabs_config_client_set_tx(uint16_t server_address,
                                                  uint16_t appkey_index,
                                                  uint8_t tx_phy,
                                                  uint32_t tx_options_bitmap);

/***************************************************************************//**
 *
 * Get node specific TX configuration.
 *
 * @param[in] server_address Destination address of the message. It can be a
 *   unicast address, a group address, or a virtual address.
 * @param[in] appkey_index The application key index used
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_silabs_config_client_tx_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_silabs_config_client_get_tx(uint16_t server_address,
                                                  uint16_t appkey_index);

/***************************************************************************//**
 *
 * Set TX over AE on per-model basis enable/disable state.
 *
 * @param[in] server_address Destination address of the message. It can be a
 *   unicast address, a group address, or a virtual address.
 * @param[in] appkey_index The application key index used
 * @param[in] elem_index The index of the server target element, 0 is the
 *   primary element.
 * @param[in] vendor_id Vendor ID of the model to configure.
 * @param[in] model_id Model ID of the server target.
 * @param[in] enable Non zero - enable, otherwise disable
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_silabs_config_client_model_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_silabs_config_client_set_model_enable(uint16_t server_address,
                                                            uint16_t appkey_index,
                                                            uint16_t elem_index,
                                                            uint16_t vendor_id,
                                                            uint16_t model_id,
                                                            uint8_t enable);

/***************************************************************************//**
 *
 * Get TX over AE on per-model basis enable/disable state.
 *
 * @param[in] server_address Destination address of the message. It can be a
 *   unicast address, a group address, or a virtual address.
 * @param[in] appkey_index The application key index used
 * @param[in] elem_index The index of the server target element, 0 is the
 *   primary element.
 * @param[in] vendor_id Vendor ID of the model to configure.
 * @param[in] model_id Model ID of the server target.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_silabs_config_client_model_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_silabs_config_client_get_model_enable(uint16_t server_address,
                                                            uint16_t appkey_index,
                                                            uint16_t elem_index,
                                                            uint16_t vendor_id,
                                                            uint16_t model_id);

/***************************************************************************//**
 *
 * Set network PDU state.
 *
 * @param[in] server_address Destination address of the message. It can be a
 *   unicast address, a group address, or a virtual address.
 * @param[in] appkey_index The application key index used
 * @param[in] pdu_max_size Network PDU max size. Valid range: 29 - 398.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_silabs_config_client_network_pdu_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_silabs_config_client_set_network_pdu(uint16_t server_address,
                                                           uint16_t appkey_index,
                                                           uint16_t pdu_max_size);

/***************************************************************************//**
 *
 * Get network PDU state.
 *
 * @param[in] server_address Destination address of the message. It can be a
 *   unicast address, a group address, or a virtual address.
 * @param[in] appkey_index The application key index used
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_silabs_config_client_network_pdu_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_silabs_config_client_get_network_pdu(uint16_t server_address,
                                                           uint16_t appkey_index);

/***************************************************************************//**
 *
 * Deinitialize Silabs Configuration Client model. After this call, the model
 * cannot be used until it is initialized again. See @ref
 * sl_btmesh_silabs_config_client_init.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_silabs_config_client_deinit();

/** @} */ // end addtogroup sl_btmesh_silabs_config_client

/**
 * @addtogroup sl_btmesh_vendor_model Bluetooth Mesh Vendor Model
 * @{
 *
 * @brief Bluetooth Mesh Vendor Model
 *
 * Vendor model API provides functionality to send and receive vendor-specific
 * messages.
 *
 * Throughout the API, the manipulated model is identified by its element
 * address, vendor ID, and model ID.
 *
 * The API has functions for sending, receiving, and publishing messages. The
 * application has to implement additional complex functionality (state machines
 * or other model-specific logic).
 *
 * The stack will handle Mesh transaction layer segmentation and reassembly
 * automatically if the messages sent are long enough to require it.
 *
 * Note that as the application layer overhead for vendor messages is three
 * bytes (vendor ID and opcode) and the access layer MIC is at least four bytes,
 * the longest vendor application payload which can be sent using an unsegmented
 * transport layer PDU is eight bytes. On the other hand, the longest vendor
 * application payload which can be sent using transport layer segmentation is
 * 377 bytes (fragmented into 32 segments).
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_vendor_model_send_id                               0x00190028
#define sl_btmesh_cmd_vendor_model_set_publication_id                    0x01190028
#define sl_btmesh_cmd_vendor_model_clear_publication_id                  0x02190028
#define sl_btmesh_cmd_vendor_model_publish_id                            0x03190028
#define sl_btmesh_cmd_vendor_model_init_id                               0x04190028
#define sl_btmesh_cmd_vendor_model_deinit_id                             0x05190028
#define sl_btmesh_cmd_vendor_model_send_tracked_id                       0x06190028
#define sl_btmesh_cmd_vendor_model_set_publication_tracked_id            0x07190028
#define sl_btmesh_cmd_vendor_model_set_option_id                         0x08190028
#define sl_btmesh_rsp_vendor_model_send_id                               0x00190028
#define sl_btmesh_rsp_vendor_model_set_publication_id                    0x01190028
#define sl_btmesh_rsp_vendor_model_clear_publication_id                  0x02190028
#define sl_btmesh_rsp_vendor_model_publish_id                            0x03190028
#define sl_btmesh_rsp_vendor_model_init_id                               0x04190028
#define sl_btmesh_rsp_vendor_model_deinit_id                             0x05190028
#define sl_btmesh_rsp_vendor_model_send_tracked_id                       0x06190028
#define sl_btmesh_rsp_vendor_model_set_publication_tracked_id            0x07190028
#define sl_btmesh_rsp_vendor_model_set_option_id                         0x08190028

/**
 * @brief Key values to identify vendor model configuration options
 */
typedef enum
{
  sl_btmesh_vendor_model_heap_work_buffer = 0x0  /**< (0x0) If set to 1, each
                                                      vendor model allocates a
                                                      heap work buffer for
                                                      messages being received.
                                                      If set to 0, no heap work
                                                      buffer allocation is made,
                                                      which saves memory but
                                                      risks losing messages in a
                                                      high load environment.
                                                      Range: 0...1 Default: 1 */
} sl_btmesh_vendor_model_options_t;

/**
 * @addtogroup sl_btmesh_evt_vendor_model_receive sl_btmesh_evt_vendor_model_receive
 * @{
 * @brief Vendor model message reception event
 *
 * Stack generates this event when a vendor message with a valid opcode is
 * received.
 *
 * Note that because of the bgapi event length restrictions, the message may be
 * fragmented into several events. If this is the case, the events will be
 * generated by the stack in the correct order and the last event will be marked
 * with the final flag set to a non-zero value. The application has to
 * concatenate the messages into a single buffer if necessary.
 */

/** @brief Identifier of the receive event */
#define sl_btmesh_evt_vendor_model_receive_id                            0x001900a8

/***************************************************************************//**
 * @brief Data structure of the receive event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_vendor_model_receive_s
{
  uint16_t   destination_address; /**< Address the message was sent to. It can
                                       be either the model element's unicast
                                       address or a subscription address of the
                                       model */
  uint16_t   elem_index;          /**< Receiving model element index */
  uint16_t   vendor_id;           /**< Vendor ID of the receiving model */
  uint16_t   model_id;            /**< Model ID of the receiving model */
  uint16_t   source_address;      /**< Unicast address of the model which sent
                                       the message */
  int8_t     va_index;            /**< Index of the destination Label UUID
                                       (valid only is the destination address is
                                       a virtual addres) */
  uint16_t   appkey_index;        /**< The application key index used */
  uint8_t    nonrelayed;          /**< If non-zero, indicates that the received
                                       message was not relayed (TTL was zero),
                                       which means that the devices are within
                                       direct radio range of each other. */
  uint8_t    message_flags;       /**< Unsigned 8-bit integer */
  uint8_t    opcode;              /**< Message opcode */
  uint8_t    final;               /**< Indicates whether this payload chunk is
                                       the final one of the message or whether
                                       more will follow */
  uint8array payload;             /**< Payload data (either complete or partial;
                                       see final parameter) */
});

typedef struct sl_btmesh_evt_vendor_model_receive_s sl_btmesh_evt_vendor_model_receive_t;

/** @} */ // end addtogroup sl_btmesh_evt_vendor_model_receive

/**
 * @addtogroup sl_btmesh_evt_vendor_model_send_complete sl_btmesh_evt_vendor_model_send_complete
 * @{
 * @brief Vendor model message send complete event
 *
 * Stack generates this event when a vendor message has been sent or published,
 * either successfully or unsuccessfully.
 *
 * The time this event is generated depends on the message being sent or
 * published. For an unsegmented message the event is generated immediately
 * after the network PDU is sent out, while for a segmented message the event is
 * generated once the segmented message transmission state machine completes.
 */

/** @brief Identifier of the send_complete event */
#define sl_btmesh_evt_vendor_model_send_complete_id                      0x011900a8

/***************************************************************************//**
 * @brief Data structure of the send_complete event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_vendor_model_send_complete_s
{
  uint16_t result;        /**< SL_STATUS_OK if successful. Error code otherwise. */
  uint16_t handle;        /**< Tracking handle for the message sending attempt
                               as reported by @ref
                               sl_btmesh_vendor_model_send_tracked or @ref
                               sl_btmesh_vendor_model_set_publication_tracked
                               command responses. */
  uint8_t  message_flags; /**< Unsigned 8-bit integer */
});

typedef struct sl_btmesh_evt_vendor_model_send_complete_s sl_btmesh_evt_vendor_model_send_complete_t;

/** @} */ // end addtogroup sl_btmesh_evt_vendor_model_send_complete

/***************************************************************************//**
 *
 * Send vendor-specific data.
 *
 * Note that, because of the bgapi event length restrictions, the message sent
 * may need to be fragmented into several commands. If this is the case, the
 * application must issue the commands in the correct order and mark the command
 * carrying the last message fragment with the final flag set to a non-zero
 * value. The stack will not start sending the message until the complete
 * message is provided by the application. Fragments from multiple messages must
 * not be interleaved.
 *
 * @param[in] destination_address Destination address of the message. It can be
 *   a unicast address, a group address, or a virtual address.
 * @param[in] va_index Index of the destination Label UUID (used only is the
 *   destination address is a virtual address)
 * @param[in] appkey_index The application key index used
 * @param[in] elem_index Sending model element index
 * @param[in] vendor_id Vendor ID of the sending model
 * @param[in] model_id Model ID of the sending model
 * @param[in] nonrelayed If the message is a response to a received message, set
 *   this parameter according to what was received in the receive event.
 *   Otherwise, set to non-zero if the message affects only devices in the
 *   immediate radio neighborhood.
 * @param[in] opcode Message opcode
 * @param[in] final Indicates whether this payload chunk is the final one of the
 *   message or whether more will follow.
 * @param[in] payload_len Length of data in @p payload
 * @param[in] payload Payload data (either complete or partial; see final
 *   parameter).
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_vendor_model_send(uint16_t destination_address,
                                        int8_t va_index,
                                        uint16_t appkey_index,
                                        uint16_t elem_index,
                                        uint16_t vendor_id,
                                        uint16_t model_id,
                                        uint8_t nonrelayed,
                                        uint8_t opcode,
                                        uint8_t final,
                                        size_t payload_len,
                                        const uint8_t* payload);

/***************************************************************************//**
 *
 * Set the vendor model publication message.
 *
 * The model publication message will be sent out when the model publication
 * occurs either periodically (if the model is configured for periodic
 * publishing) or explicitly (see @ref sl_btmesh_vendor_model_publish.
 *
 * Note that, because of bgapi length requirements, the message may need to be
 * fragmented over multiple commands. If this is the case, the application must
 * issue the commands in the correct order and mark the command carrying the
 * last message fragment with the final flag set to a non-zero value. The stack
 * will not assign the message to the model until the complete message is
 * provided by the application.
 *
 * To disable publication, the publication message may be erased using the @ref
 * sl_btmesh_vendor_model_clear_publication command.
 *
 * @param[in] elem_index Publishing model element index
 * @param[in] vendor_id Vendor ID of the model
 * @param[in] model_id Model ID of the model
 * @param[in] opcode Message opcode
 * @param[in] final Indicates whether this payload chunk is the final one of the
 *   message or whether more will follow.
 * @param[in] payload_len Length of data in @p payload
 * @param[in] payload Payload data (either complete or partial; see final
 *   parameter).
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_vendor_model_set_publication(uint16_t elem_index,
                                                   uint16_t vendor_id,
                                                   uint16_t model_id,
                                                   uint8_t opcode,
                                                   uint8_t final,
                                                   size_t payload_len,
                                                   const uint8_t* payload);

/***************************************************************************//**
 *
 * Clear the vendor model publication message.
 *
 * Clearing the model publication message disables model publishing, which can
 * be re-enabled by defining the publication message using the @ref
 * sl_btmesh_vendor_model_set_publication command.
 *
 * @param[in] elem_index Publishing model element index
 * @param[in] vendor_id Vendor ID of the model
 * @param[in] model_id Model ID of the model
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_vendor_model_clear_publication(uint16_t elem_index,
                                                     uint16_t vendor_id,
                                                     uint16_t model_id);

/***************************************************************************//**
 *
 * Publish the vendor model publication message.
 *
 * Sends the stored publication message to the network using the application key
 * and destination address stored in the model publication parameters.
 *
 * @param[in] elem_index Publishing model element index
 * @param[in] vendor_id Vendor ID of the model
 * @param[in] model_id Model ID of the model
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_vendor_model_publish(uint16_t elem_index,
                                           uint16_t vendor_id,
                                           uint16_t model_id);

/***************************************************************************//**
 *
 * Initialize the vendor model. This function has to be called before the model
 * can be used. Note that the model can be deinitialized if it is no longer
 * needed. See @ref sl_btmesh_vendor_model_deinit.
 *
 * Opcodes that the model is able to receive at initialization must be defined.
 * This enables the stack to pass only valid messages up to the model during
 * runtime. Per Mesh specification there are up to 64 opcodes per vendor,
 * ranging from 0 to 63. Specifying opcodes outside of that range will result in
 * an error response. Duplicate opcodes in the array do not result in an error,
 * but will of course be recorded only once.
 *
 * @param[in] elem_index Model element index
 * @param[in] vendor_id Vendor ID of the model
 * @param[in] model_id Model ID of the model
 * @param[in] publish Indicates if the model is a publish model (non-zero) or
 *   not (zero).
 * @param[in] opcodes_len Length of data in @p opcodes
 * @param[in] opcodes Array of opcodes the model can handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_vendor_model_init(uint16_t elem_index,
                                        uint16_t vendor_id,
                                        uint16_t model_id,
                                        uint8_t publish,
                                        size_t opcodes_len,
                                        const uint8_t* opcodes);

/***************************************************************************//**
 *
 * Deinitialize the model. After this call, the model cannot be used until it is
 * initialized again. See @ref sl_btmesh_vendor_model_init.
 *
 * @param[in] elem_index Model element index
 * @param[in] vendor_id Vendor ID of the model
 * @param[in] model_id Model ID of the model
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_vendor_model_deinit(uint16_t elem_index,
                                          uint16_t vendor_id,
                                          uint16_t model_id);

/***************************************************************************//**
 *
 * Send vendor-specific data with a delivery report.
 *
 * This command works otherwise as @ref sl_btmesh_vendor_model_send but also
 * provides a tracking handle if there is no immediate local error. Once the
 * message sending completes a @ref sl_btmesh_evt_vendor_model_send_complete
 * event is generated, and the handle received here can be used to correlate the
 * event to the sent message.
 *
 * @param[in] destination_address Destination address of the message. It can be
 *   a unicast address, a group address, or a virtual address.
 * @param[in] va_index Index of the destination Label UUID (used only is the
 *   destination address is a virtual address)
 * @param[in] appkey_index The application key index used
 * @param[in] elem_index Sending model element index
 * @param[in] vendor_id Vendor ID of the sending model
 * @param[in] model_id Model ID of the sending model
 * @param[in] nonrelayed If the message is a response to a received message, set
 *   this parameter according to what was received in the receive event.
 *   Otherwise, set to non-zero if the message affects only devices in the
 *   immediate radio neighborhood.
 * @param[in] segment If nonzero, instruct the stack to use transport layer
 *   segmentation, even if the data would be short enough to send as an
 *   unsegmented message. If zero, stack selects automatically based on data
 *   length whether to use segmentation or not.
 * @param[in] opcode Message opcode
 * @param[in] final Indicates whether this payload chunk is the final one of the
 *   message or whether more will follow.
 * @param[in] payload_len Length of data in @p payload
 * @param[in] payload Payload data (either complete or partial; see final
 *   parameter).
 * @param[out] handle Tracking handle for the message sending. Note that a valid
 *   handle is returned only when the final payload fragment is received by the
 *   stack (as indicated by the final parameter of this message) and the message
 *   is accepted for delivery (as indicated by the result code). Otherwise the
 *   value of this parameter should be ignored.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_vendor_model_send_tracked(uint16_t destination_address,
                                                int8_t va_index,
                                                uint16_t appkey_index,
                                                uint16_t elem_index,
                                                uint16_t vendor_id,
                                                uint16_t model_id,
                                                uint8_t nonrelayed,
                                                uint8_t segment,
                                                uint8_t opcode,
                                                uint8_t final,
                                                size_t payload_len,
                                                const uint8_t* payload,
                                                uint16_t *handle);

/***************************************************************************//**
 *
 * Set the vendor model publication message with a delivery report.
 *
 * This command works otherwise as @ref sl_btmesh_vendor_model_set_publication
 * but also sets up a tracking handle for monitoring message delivery. When the
 * message publishing completes a @ref sl_btmesh_evt_vendor_model_send_complete
 * event is generated, and the handle received here can be used to correlate the
 * event to the sent message.
 *
 * Note that the same tracking handle will be reported multiple times in case of
 * periodic publication or publication retransmissions.
 *
 * @param[in] elem_index Publishing model element index
 * @param[in] vendor_id Vendor ID of the model
 * @param[in] model_id Model ID of the model
 * @param[in] segment If nonzero, instruct the stack to use transport layer
 *   segmentation, even if the data would be short enough to publish as an
 *   unsegmented message. If zero, stack selects automatically based on data
 *   length whether to use segmentation or not.
 * @param[in] opcode Message opcode
 * @param[in] final Indicates whether this payload chunk is the final one of the
 *   message or whether more will follow.
 * @param[in] payload_len Length of data in @p payload
 * @param[in] payload Payload data (either complete or partial; see final
 *   parameter).
 * @param[out] handle Tracking handle for the message publishing. Note that a
 *   valid handle is returned only when the final payload fragment is received
 *   by the stack (as indicated by the final parameter of this message) and the
 *   message is accepted for publication (as indicated by the result code).
 *   Otherwise the value of this parameter should be ignored. attempt
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_vendor_model_set_publication_tracked(uint16_t elem_index,
                                                           uint16_t vendor_id,
                                                           uint16_t model_id,
                                                           uint8_t segment,
                                                           uint8_t opcode,
                                                           uint8_t final,
                                                           size_t payload_len,
                                                           const uint8_t* payload,
                                                           uint16_t *handle);

/***************************************************************************//**
 *
 * Set global vendor model configuration options.
 *
 * This command sets global configuration options for all vendor models. Options
 * take effect when models are initialized, so this command should be called
 * before any @ref sl_btmesh_vendor_model_init call.
 *
 * @param[in] option Enum @ref sl_btmesh_vendor_model_options_t. Configuration
 *   option.
 * @param[in] value Configuration value
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_vendor_model_set_option(uint8_t option, uint32_t value);

/** @} */ // end addtogroup sl_btmesh_vendor_model

/**
 * @addtogroup sl_btmesh_health_client Bluetooth Mesh Health Client Model
 * @{
 *
 * @brief Bluetooth Mesh Health Client Model
 *
 * This class provides the commands and messages to interface with the Health
 * Client model.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_health_client_init_id                              0x071a0028
#define sl_btmesh_cmd_health_client_deinit_id                            0x081a0028
#define sl_btmesh_cmd_health_client_get_id                               0x001a0028
#define sl_btmesh_cmd_health_client_clear_id                             0x011a0028
#define sl_btmesh_cmd_health_client_test_id                              0x021a0028
#define sl_btmesh_cmd_health_client_get_period_id                        0x031a0028
#define sl_btmesh_cmd_health_client_set_period_id                        0x041a0028
#define sl_btmesh_cmd_health_client_get_attention_id                     0x051a0028
#define sl_btmesh_cmd_health_client_set_attention_id                     0x061a0028
#define sl_btmesh_rsp_health_client_init_id                              0x071a0028
#define sl_btmesh_rsp_health_client_deinit_id                            0x081a0028
#define sl_btmesh_rsp_health_client_get_id                               0x001a0028
#define sl_btmesh_rsp_health_client_clear_id                             0x011a0028
#define sl_btmesh_rsp_health_client_test_id                              0x021a0028
#define sl_btmesh_rsp_health_client_get_period_id                        0x031a0028
#define sl_btmesh_rsp_health_client_set_period_id                        0x041a0028
#define sl_btmesh_rsp_health_client_get_attention_id                     0x051a0028
#define sl_btmesh_rsp_health_client_set_attention_id                     0x061a0028

/**
 * @addtogroup sl_btmesh_evt_health_client_server_status sl_btmesh_evt_health_client_server_status
 * @{
 * @brief Receiving a Health Server fault status message generates this event
 *
 * The Client model may receive a status message because:
 *   - it made a @ref sl_btmesh_health_client_get to which a Server model
 *     responded
 *   - it made a @ref sl_btmesh_health_client_clear to which a Server model
 *     responded
 *   - it made a @ref sl_btmesh_health_client_test to which a Server model
 *     responded
 */

/** @brief Identifier of the server_status event */
#define sl_btmesh_evt_health_client_server_status_id                     0x001a00a8

/***************************************************************************//**
 * @brief Data structure of the server_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_health_client_server_status_s
{
  uint16_t   client_address; /**< Destination address the message was sent to */
  uint16_t   elem_index;     /**< Client model element index. Identifies the
                                  client model which received the status
                                  message. */
  uint16_t   server_address; /**< Address of the Server model which sent the
                                  message */
  uint16_t   result;         /**< Response status. If an error occurs (e.g.,
                                  request timeout), the parameters other than
                                  element index, client address, and server
                                  address are to be ignored. */
  uint8_t    current;        /**< Whether the event lists current fault array or
                                  registered fault array */
  uint8_t    test_id;        /**< Test ID */
  uint16_t   vendor_id;      /**< Bluetooth vendor ID used in the request */
  uint8array faults;         /**< Fault array. See the Bluetooth Mesh Profile
                                  specification for a list of defined fault IDs. */
});

typedef struct sl_btmesh_evt_health_client_server_status_s sl_btmesh_evt_health_client_server_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_health_client_server_status

/**
 * @addtogroup sl_btmesh_evt_health_client_server_status_period sl_btmesh_evt_health_client_server_status_period
 * @{
 * @brief Receiving a Health Server period status message generates this event
 *
 * The Client model may receive a status message because:
 *   - it made a @ref sl_btmesh_health_client_get_period to which a Server model
 *     responded
 *   - it made a @ref sl_btmesh_health_client_set_period to which a Server model
 *     responded
 */

/** @brief Identifier of the server_status_period event */
#define sl_btmesh_evt_health_client_server_status_period_id              0x011a00a8

/***************************************************************************//**
 * @brief Data structure of the server_status_period event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_health_client_server_status_period_s
{
  uint16_t client_address; /**< Destination address the message was sent to */
  uint16_t elem_index;     /**< Client model element index. Identifies the
                                client model, which received the status message. */
  uint16_t server_address; /**< Address of the Server model which sent the
                                message */
  uint16_t result;         /**< Response status. If an error occurs (e.g.,
                                request timeout), ignore the parameters other
                                than element index, client address, and server
                                address. */
  uint8_t  period;         /**< Health period divisor value */
});

typedef struct sl_btmesh_evt_health_client_server_status_period_s sl_btmesh_evt_health_client_server_status_period_t;

/** @} */ // end addtogroup sl_btmesh_evt_health_client_server_status_period

/**
 * @addtogroup sl_btmesh_evt_health_client_server_status_attention sl_btmesh_evt_health_client_server_status_attention
 * @{
 * @brief Receiving a Health Server attention status message generates this
 * event
 *
 * The Client model may receive a status message because:
 *   - it made a @ref sl_btmesh_health_client_get_attention to which a Server
 *     model responded
 *   - it made a @ref sl_btmesh_health_client_set_attention to which a Server
 *     model responded
 */

/** @brief Identifier of the server_status_attention event */
#define sl_btmesh_evt_health_client_server_status_attention_id           0x021a00a8

/***************************************************************************//**
 * @brief Data structure of the server_status_attention event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_health_client_server_status_attention_s
{
  uint16_t client_address; /**< Destination address the message was sent to */
  uint16_t elem_index;     /**< Client model element index. Identifies the
                                client model which received the status message. */
  uint16_t server_address; /**< Address of the Server model which sent the
                                message */
  uint16_t result;         /**< Response status. If an error occurs (e.g.,
                                request timeout), ignore the parameters other
                                than element index, client address, and server
                                address. */
  uint8_t  attention_sec;  /**< Current attention timer value in seconds */
});

typedef struct sl_btmesh_evt_health_client_server_status_attention_s sl_btmesh_evt_health_client_server_status_attention_t;

/** @} */ // end addtogroup sl_btmesh_evt_health_client_server_status_attention

/***************************************************************************//**
 *
 * Initialize health client.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_health_client_init();

/***************************************************************************//**
 *
 * Deinitialize health client.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_health_client_deinit();

/***************************************************************************//**
 *
 * Get the registered fault status of a Health Server model or models in the
 * network.
 *
 * Besides the immediate result code, the response or responses (if the
 * destination server address is a group address) from the network will generate
 * @ref sl_btmesh_evt_health_client_server_status.
 *
 * @param[in] server_address Destination server model address. May be a unicast
 *   address or a group address.
 * @param[in] elem_index Client model element index. Identifies the client model
 *   used for sending the request.
 * @param[in] appkey_index The application key index to use in encrypting the
 *   request
 * @param[in] vendor_id Bluetooth vendor ID used in the request
 *
 * @return If an error occurs locally, (for instance, because of invalid
 *   parameters) an errorcode parameter is returned immediately.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_health_client_server_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_health_client_get(uint16_t server_address,
                                        uint16_t elem_index,
                                        uint16_t appkey_index,
                                        uint16_t vendor_id);

/***************************************************************************//**
 *
 * Clear the fault status of a Health Server model or models in the network.
 *
 * Besides the immediate result code, the response or responses (if the
 * destination server address is a group address) from the network will generate
 * @ref sl_btmesh_evt_health_client_server_status.
 *
 * @param[in] server_address Destination server model address. May be a unicast
 *   address or a group address.
 * @param[in] elem_index Client model element index. Identifies the client model
 *   used for sending the request.
 * @param[in] appkey_index The application key index to use in encrypting the
 *   request
 * @param[in] vendor_id Bluetooth vendor ID used in the request
 * @param[in] reliable If non-zero, a reliable model message is used.
 *
 * @return If an error occurs locally (for instance, because of invalid
 *   parameters), an errorcode parameter is returned immediately.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_health_client_server_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_health_client_clear(uint16_t server_address,
                                          uint16_t elem_index,
                                          uint16_t appkey_index,
                                          uint16_t vendor_id,
                                          uint8_t reliable);

/***************************************************************************//**
 *
 * Execute a self test on a server model or models in the network.
 *
 * @param[in] server_address Destination server model address. May be a unicast
 *   address or a group address.
 * @param[in] elem_index Client model element index. Identifies the client model
 *   used for sending the request.
 * @param[in] appkey_index The application key index to use in encrypting the
 *   request
 * @param[in] test_id Test ID used in the request
 * @param[in] vendor_id Bluetooth vendor ID used in the request
 * @param[in] reliable If non-zero, a reliable model message is used.
 *
 * @return If an error occurs locally (for instance, because of invalid
 *   parameters) an errorcode parameter is returned immediately.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_health_client_server_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_health_client_test(uint16_t server_address,
                                         uint16_t elem_index,
                                         uint16_t appkey_index,
                                         uint8_t test_id,
                                         uint16_t vendor_id,
                                         uint8_t reliable);

/***************************************************************************//**
 *
 * Get the health period log of a Health Server model or models in the network.
 *
 * Except for the immediate result code, the response or responses (if the
 * destination server address is a group address) from the network will generate
 * @ref sl_btmesh_evt_health_client_server_status_period.
 *
 * @param[in] server_address Destination server model address, which may be a
 *   unicast address or a group address
 * @param[in] elem_index Client model element index. Identifies the client model
 *   used for sending the request.
 * @param[in] appkey_index The application key index to use in encrypting the
 *   request
 *
 * @return If an error occurs locally, (for instance, because of invalid
 *   parameters) an errorcode parameter is returned immediately.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_health_client_server_status_period
 *
 ******************************************************************************/
sl_status_t sl_btmesh_health_client_get_period(uint16_t server_address,
                                               uint16_t elem_index,
                                               uint16_t appkey_index);

/***************************************************************************//**
 *
 * Set the health period divisor of a Health Server model or models in the
 * network.
 *
 * Except for the immediate result code, the response or responses (if the
 * destination server address is a group address) from the network will generate
 * @ref sl_btmesh_evt_health_client_server_status_period.
 *
 * @param[in] server_address Destination server model address. May be a unicast
 *   address or a group address.
 * @param[in] elem_index Client model element index, which identifies the client
 *   model used for sending the request.
 * @param[in] appkey_index The application key index to use in encrypting the
 *   request
 * @param[in] period Health period divisor value. Valid range: 0 - 16.
 * @param[in] reliable If non-zero, a reliable model message is used.
 *
 * @return If an error occurs locally (for instance, because of invalid
 *   parameters), an errorcode parameter is returned immediately.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_health_client_server_status_period
 *
 ******************************************************************************/
sl_status_t sl_btmesh_health_client_set_period(uint16_t server_address,
                                               uint16_t elem_index,
                                               uint16_t appkey_index,
                                               uint8_t period,
                                               uint8_t reliable);

/***************************************************************************//**
 *
 * Get the attention timer value of a Health Server model or models in the
 * network.
 *
 * Besides the immediate result code, the response or responses (if the
 * destination server address is a group address) from the network will generate
 * @ref sl_btmesh_evt_health_client_server_status_attention.
 *
 * @param[in] server_address Destination server model address. May be a unicast
 *   address or a group address.
 * @param[in] elem_index Client model element index. Identifies the client model
 *   used for sending the request.
 * @param[in] appkey_index The application key index to use in encrypting the
 *   request
 *
 * @return If an error occurs locally (for instance, because of invalid
 *   parameters), an errorcode parameter is returned immediately.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_health_client_server_status_attention
 *
 ******************************************************************************/
sl_status_t sl_btmesh_health_client_get_attention(uint16_t server_address,
                                                  uint16_t elem_index,
                                                  uint16_t appkey_index);

/***************************************************************************//**
 *
 * Set the attention timer value of a Health Server model or models in the
 * network.
 *
 * Except for the immediate result code, the response or responses (if the
 * destination server address is a group address) from the network will generate
 * @ref sl_btmesh_evt_health_client_server_status_attention.
 *
 * @param[in] server_address Destination server model address. May be a unicast
 *   address or a group address.
 * @param[in] elem_index Client model element index. Identifies the client model
 *   used for sending the request.
 * @param[in] appkey_index The application key index to use in encrypting the
 *   request
 * @param[in] attention_sec Attention timer period in seconds
 * @param[in] reliable If non-zero, a reliable model message is used.
 *
 * @return If an error occurs locally (for instance, because of invalid
 *   parameters), an errorcode parameter is returned immediately.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_health_client_server_status_attention
 *
 ******************************************************************************/
sl_status_t sl_btmesh_health_client_set_attention(uint16_t server_address,
                                                  uint16_t elem_index,
                                                  uint16_t appkey_index,
                                                  uint8_t attention_sec,
                                                  uint8_t reliable);

/** @} */ // end addtogroup sl_btmesh_health_client

/**
 * @addtogroup sl_btmesh_health_server Bluetooth Mesh Health Server Model
 * @{
 *
 * @brief Bluetooth Mesh Health Server Model
 *
 * This class provides the commands and messages to interface with the Health
 * Server model
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_health_server_set_fault_id                         0x001b0028
#define sl_btmesh_cmd_health_server_clear_fault_id                       0x011b0028
#define sl_btmesh_cmd_health_server_send_test_response_id                0x021b0028
#define sl_btmesh_rsp_health_server_set_fault_id                         0x001b0028
#define sl_btmesh_rsp_health_server_clear_fault_id                       0x011b0028
#define sl_btmesh_rsp_health_server_send_test_response_id                0x021b0028

/**
 * @addtogroup sl_btmesh_evt_health_server_attention sl_btmesh_evt_health_server_attention
 * @{
 * @brief
 *
 * The attention timer on an element is set to a given value. This may happen,
 * for instance, during provisioning. The application should use suitable means
 * to get the user's attention, e.g., by vibrating or blinking an LED.
 */

/** @brief Identifier of the attention event */
#define sl_btmesh_evt_health_server_attention_id                         0x001b00a8

/***************************************************************************//**
 * @brief Data structure of the attention event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_health_server_attention_s
{
  uint16_t elem_index; /**< Index of the element for which attention timer was
                            set */
  uint8_t  timer_sec;  /**< Timer value in seconds. If zero, user attention is
                            no longer required. */
});

typedef struct sl_btmesh_evt_health_server_attention_s sl_btmesh_evt_health_server_attention_t;

/** @} */ // end addtogroup sl_btmesh_evt_health_server_attention

/**
 * @addtogroup sl_btmesh_evt_health_server_test_request sl_btmesh_evt_health_server_test_request
 * @{
 * @brief
 *
 * Health client request for a self test generates this event. After the test
 * has been executed, test results may need to be reported.
 */

/** @brief Identifier of the test_request event */
#define sl_btmesh_evt_health_server_test_request_id                      0x011b00a8

/***************************************************************************//**
 * @brief Data structure of the test_request event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_health_server_test_request_s
{
  uint16_t server_address;    /**< Destination address the message was sent to.
                                   It can be either the Server model element's
                                   unicast address, or a subscription address of
                                   the Server model. */
  uint16_t elem_index;        /**< Server model element index. Identifies the
                                   Server model that received the request as
                                   well as the element on which the test is to
                                   be performed. */
  uint16_t client_address;    /**< Address of the client model which sent the
                                   message */
  uint16_t appkey_index;      /**< The application key index to use in
                                   encrypting the request. Any response sent
                                   must be encrypted using the same key. */
  uint8_t  test_id;           /**< Test ID */
  uint16_t vendor_id;         /**< Bluetooth vendor ID used in the request */
  uint8_t  response_required; /**< Non-zero if client expects a response. The
                                   application should issue a @ref
                                   sl_btmesh_health_server_send_test_response
                                   once it has processed the request. */
});

typedef struct sl_btmesh_evt_health_server_test_request_s sl_btmesh_evt_health_server_test_request_t;

/** @} */ // end addtogroup sl_btmesh_evt_health_server_test_request

/***************************************************************************//**
 *
 * Set the fault condition on an element.
 *
 * @param[in] elem_index Index of the element on which the fault is occurring
 * @param[in] id Fault ID. See the Mesh Profile specification for IDs defined by
 *   the Bluetooth SIG.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_health_server_set_fault(uint16_t elem_index, uint8_t id);

/***************************************************************************//**
 *
 * Clear the fault condition on an element.
 *
 * @param[in] elem_index Index of the element on which the fault is no longer
 *   occurring.
 * @param[in] id Fault ID. See the Mesh Profile specification for IDs defined by
 *   the Bluetooth SIG.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_health_server_clear_fault(uint16_t elem_index,
                                                uint8_t id);

/***************************************************************************//**
 *
 * Indicate to the stack that a test request has been completed and that the
 * status may be communicated to the Health Client, which made the test request.
 *
 * @param[in] client_address Address of the client model which sent the message
 * @param[in] elem_index Server model element index. Identifies the Server model
 *   that received the request as well as the element on which the test is to be
 *   performed.
 * @param[in] appkey_index The application key index to use in encrypting the
 *   request.
 * @param[in] vendor_id Bluetooth vendor ID used in the request
 *
 * @return If an error occurs locally (for instance, because of invalid
 *   parameters), an errorcode parameter is returned immediately.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_health_server_send_test_response(uint16_t client_address,
                                                       uint16_t elem_index,
                                                       uint16_t appkey_index,
                                                       uint16_t vendor_id);

/** @} */ // end addtogroup sl_btmesh_health_server

/**
 * @addtogroup sl_btmesh_generic_client Bluetooth Mesh Generic Client Model
 * @{
 *
 * @brief Bluetooth Mesh Generic Client Model
 *
 * Generic client model API provides a functionality to send and receive
 * messages using Bluetooth SIG client models, including generic client models
 * and lighting client models.
 *
 * In the API, the client model that is used is identified by its element
 * address and model ID, while the server model responding to client model
 * requests is identified by its element address and model ID.
 *
 * The API has functions for querying server model states, requesting server
 * model state changes, and publishing messages. The application has to
 * implement more complex functionality (state machines or other model-specific
 * logic).
 *
 * Data for state change requests and server responses is passed as serialized
 * byte arrays through BGAPI. There are functions to convert byte arrays to and
 * from model state structures in the Bluetooth mesh SDK.
 *
 * The stack will handle Mesh transaction layer segmentation and reassembly
 * automatically if the messages sent are long enough to require it.
 *
 * <b>Note on time resolution</b>
 *
 * Because of message formatting, transition time and remaining time resolution
 * units depend on the requested or reported value. For example, until 6.2
 * seconds it is 100 ms; until 62 seconds it is 1 s; until 620 seconds it is 10
 * s; and until 620 minutes it is 10 minutes. The value cannot be longer than
 * 620 minutes. Therefore, it is not possible to request a delay of exactly 7500
 * ms. The resolution unit is 1 s between 6.2 and 62 seconds, so the value would
 * be rounded down to 7 s.
 *
 * Delay resolution is 5 ms and values will be rounded down to the closest 5 ms.
 * The value can't be longer than 1275 ms.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_generic_client_get_id                              0x001e0028
#define sl_btmesh_cmd_generic_client_set_id                              0x011e0028
#define sl_btmesh_cmd_generic_client_publish_id                          0x021e0028
#define sl_btmesh_cmd_generic_client_get_params_id                       0x031e0028
#define sl_btmesh_cmd_generic_client_init_id                             0x041e0028
#define sl_btmesh_cmd_generic_client_init_common_id                      0x051e0028
#define sl_btmesh_cmd_generic_client_init_on_off_id                      0x061e0028
#define sl_btmesh_cmd_generic_client_init_level_id                       0x071e0028
#define sl_btmesh_cmd_generic_client_init_default_transition_time_id     0x081e0028
#define sl_btmesh_cmd_generic_client_init_power_on_off_id                0x091e0028
#define sl_btmesh_cmd_generic_client_init_power_level_id                 0x0a1e0028
#define sl_btmesh_cmd_generic_client_init_battery_id                     0x0b1e0028
#define sl_btmesh_cmd_generic_client_init_location_id                    0x0c1e0028
#define sl_btmesh_cmd_generic_client_init_property_id                    0x0d1e0028
#define sl_btmesh_cmd_generic_client_init_lightness_id                   0x0e1e0028
#define sl_btmesh_cmd_generic_client_init_ctl_id                         0x0f1e0028
#define sl_btmesh_cmd_generic_client_init_hsl_id                         0x101e0028
#define sl_btmesh_rsp_generic_client_get_id                              0x001e0028
#define sl_btmesh_rsp_generic_client_set_id                              0x011e0028
#define sl_btmesh_rsp_generic_client_publish_id                          0x021e0028
#define sl_btmesh_rsp_generic_client_get_params_id                       0x031e0028
#define sl_btmesh_rsp_generic_client_init_id                             0x041e0028
#define sl_btmesh_rsp_generic_client_init_common_id                      0x051e0028
#define sl_btmesh_rsp_generic_client_init_on_off_id                      0x061e0028
#define sl_btmesh_rsp_generic_client_init_level_id                       0x071e0028
#define sl_btmesh_rsp_generic_client_init_default_transition_time_id     0x081e0028
#define sl_btmesh_rsp_generic_client_init_power_on_off_id                0x091e0028
#define sl_btmesh_rsp_generic_client_init_power_level_id                 0x0a1e0028
#define sl_btmesh_rsp_generic_client_init_battery_id                     0x0b1e0028
#define sl_btmesh_rsp_generic_client_init_location_id                    0x0c1e0028
#define sl_btmesh_rsp_generic_client_init_property_id                    0x0d1e0028
#define sl_btmesh_rsp_generic_client_init_lightness_id                   0x0e1e0028
#define sl_btmesh_rsp_generic_client_init_ctl_id                         0x0f1e0028
#define sl_btmesh_rsp_generic_client_init_hsl_id                         0x101e0028

/**
 * @addtogroup sl_btmesh_generic_client_get_state_type Generic Client Get State Types
 * @{
 *
 * Generic client get state type identifies the state which the client retrieves
 * from the remote server model.
 */

/** Generic on/off get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_ON_OFF                    0x0       

/** Generic on power up get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_ON_POWER_UP               0x1       

/** Generic level get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_LEVEL                     0x2       

/** Generic power level get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_POWER_LEVEL               0x3       

/** Generic power level last get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_POWER_LEVEL_LAST          0x4       

/** Generic power level default get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_POWER_LEVEL_DEFAULT       0x5       

/** Generic power level range get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_POWER_LEVEL_RANGE         0x6       

/** Generic transition time get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_TRANSITION_TIME           0x6       

/** Generic battery get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_BATTERY                   0x8       

/** Generic global location get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_LOCATION_GLOBAL           0x9       

/** Generic local location get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_LOCATION_LOCAL            0xa       

/** Generic user property get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_PROPERTY_USER             0xb       

/** Generic admin property get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_PROPERTY_ADMIN            0xc       

/** Generic manufacturer property get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_PROPERTY_MANUF            0xd       

/** Generic user property list get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_PROPERTY_LIST_USER        0xe       

/** Generic admin property list get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_PROPERTY_LIST_ADMIN       0xf       

/** Generic manufacturer property list get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_PROPERTY_LIST_MANUF       0x10      

/** Generic client property list get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_PROPERTY_LIST_CLIENT      0x11      

/** Light actual lightness get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_LIGHTNESS_ACTUAL          0x80      

/** Light linear lightness get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_LIGHTNESS_LINEAR          0x81      

/** Light last lightness get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_LIGHTNESS_LAST            0x82      

/** Light default lightness get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_LIGHTNESS_DEFAULT         0x83      

/** Light lightness range get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_LIGHTNESS_RANGE           0x84      

/** Light lightness, color temperature, and delta UV server state identifier.
 * Not to be used by client get requests. */
#define SL_BTMESH_GENERIC_CLIENT_STATE_CTL                       0x85      

/** Light color temperature and delta UV get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_CTL_TEMPERATURE           0x86      

/** Light lightness, color temperature, and delta UV default get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_CTL_DEFAULT               0x87      

/** Light color temperature range get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_CTL_RANGE                 0x88      

/** Light lightness and color temperature get request. */
#define SL_BTMESH_GENERIC_CLIENT_STATE_CTL_LIGHTNESS_TEMPERATURE 0x89      

/** Light lightness, color hue, and color saturation current value get request.
 * */
#define SL_BTMESH_GENERIC_CLIENT_STATE_HSL                       0x8a      

/** Light color hue get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_HSL_HUE                   0x8b      

/** Light color saturation get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_HSL_SATURATION            0x8c      

/** Light lightness, color hue, and color saturation default get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_HSL_DEFAULT               0x8d      

/** Light color hue and saturation range get request */
#define SL_BTMESH_GENERIC_CLIENT_STATE_HSL_RANGE                 0x8e      

/** Light lightness, color hue, and color saturation target value get request.
 * */
#define SL_BTMESH_GENERIC_CLIENT_STATE_HSL_TARGET                0x8f      

/** @} */ // end Generic Client Get State Types

/**
 * @addtogroup sl_btmesh_generic_client_set_request_type Generic Client Set Request Types
 * @{
 *
 * Generic client set request type identifies the state which the client
 * requests to be set to a new value on the remote server model.
 */

/** Generic on/off set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_ON_OFF              0x0       

/** Generic on power up set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_ON_POWER_UP         0x1       

/** Generic level set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_LEVEL               0x2       

/** Generic level delta set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_LEVEL_DELTA         0x3       

/** Generic level move set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_LEVEL_MOVE          0x4       

/** Generic level halt request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_LEVEL_HALT          0x5       

/** Generic power level set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_POWER_LEVEL         0x6       

/** Generic power level default set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_POWER_LEVEL_DEFAULT 0x7       

/** Generic power level range set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_POWER_LEVEL_RANGE   0x8       

/** Generic transition time set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_TRANSITION_TIME     0x9       

/** Generic global location set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_LOCATION_GLOBAL     0xa       

/** Generic local location set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_LOCATION_LOCAL      0xb       

/** Generic user property set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_PROPERTY_USER       0xc       

/** Generic admin property set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_PROPERTY_ADMIN      0xd       

/** Generic manufacturer property set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_PROPERTY_MANUF      0xe       

/** Light actual lightness set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_LIGHTNESS_ACTUAL    0x80      

/** Light linear lightness set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_LIGHTNESS_LINEAR    0x81      

/** Light default lightness set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_LIGHTNESS_DEFAULT   0x82      

/** Light lightness range set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_LIGHTNESS_RANGE     0x83      

/** Light lightness, color temperature, and delta UV set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_CTL                 0x84      

/** Light color temperature and delta UV set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_CTL_TEMPERATURE     0x85      

/** Light lightness, color temperature, and delta UV default set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_CTL_DEFAULT         0x86      

/** Light color temperature range set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_CTL_RANGE           0x87      

/** Light lightness, color hue, and color saturation set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_HSL                 0x88      

/** Light color hue set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_HSL_HUE             0x89      

/** Light color saturation set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_HSL_SATURATION      0x8a      

/** Light lightness, color hue, and color saturation default set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_HSL_DEFAULT         0x8b      

/** Light color hue and color saturation range set request */
#define SL_BTMESH_GENERIC_CLIENT_REQUEST_HSL_RANGE           0x8c      

/** @} */ // end Generic Client Set Request Types

/**
 * @addtogroup sl_btmesh_evt_generic_client_server_status sl_btmesh_evt_generic_client_server_status
 * @{
 * @brief Status report sent by a server model
 *
 * This may be generated either because of a response to a get or set request
 * was received by the client model or because the client model received a
 * spontaneously generated status indication sent to an address the model was
 * subscribed to.
 */

/** @brief Identifier of the server_status event */
#define sl_btmesh_evt_generic_client_server_status_id                    0x001e00a8

/***************************************************************************//**
 * @brief Data structure of the server_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_generic_client_server_status_s
{
  uint16_t   client_address; /**< Address that the message was sent to, which
                                  can be either the model element's unicast
                                  address or a subscription address of the model */
  uint16_t   elem_index;     /**< Client model element index */
  uint16_t   model_id;       /**< Client model ID */
  uint16_t   server_address; /**< Address of the server model which sent the
                                  message */
  uint32_t   remaining_ms;   /**< Time (in milliseconds) remaining before
                                  transition from the current state to target
                                  state is complete. Set to zero if no
                                  transition is taking place or if transition
                                  time does not apply to the message. */
  uint16_t   flags;          /**< Message flags. It is a bitmask of the
                                  following values:
                                    - <b>Bit 0:</b> Non-relayed. If non-zero,
                                      indicates a response to a non-relayed
                                      request. */
  uint8_t    type;           /**< Model-specific state type, identifying the
                                  kind of state reported in the status event.
                                  See get state types list for details. */
  uint8array parameters;     /**< Message-specific parameters, serialized into a
                                  byte array */
});

typedef struct sl_btmesh_evt_generic_client_server_status_s sl_btmesh_evt_generic_client_server_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_generic_client_server_status

/***************************************************************************//**
 *
 * Get the current state of a server model or models in the network. Besides the
 * immediate result code, the response or responses from the network will
 * generate server state report events for the replies received.
 *
 * The server model responses will be reported in @ref
 * sl_btmesh_evt_generic_client_server_status events.
 *
 * @param[in] server_address Destination server model address
 * @param[in] elem_index Client model element index
 * @param[in] model_id Client model ID
 * @param[in] appkey_index The application key index to use
 * @param[in] type Model-specific state type, identifying the kind of state to
 *   retrieve. See @ref sl_btmesh_generic_client_get_state_type list for
 *   details.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_generic_client_server_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_client_get(uint16_t server_address,
                                         uint16_t elem_index,
                                         uint16_t model_id,
                                         uint16_t appkey_index,
                                         uint8_t type);

/***************************************************************************//**
 *
 * Set the current state of a server model or models in the network. Besides the
 * immediate result code, the response or responses from the network will
 * generate server state report events for the replies received.
 *
 * The server model responses will be reported in @ref
 * sl_btmesh_evt_generic_client_server_status events. Note that for responses to
 * be generated the corresponding flag needs to be set.
 *
 * @param[in] server_address Destination server model address
 * @param[in] elem_index Client model element index
 * @param[in] model_id Client model ID
 * @param[in] appkey_index The application key index to use
 * @param[in] tid Transaction identifier. This applies to those messages the
 *   Mesh Model specification defines as transactional and can be left as zero
 *   for others.
 * @param[in] transition_ms @parblock
 *   Transition time (in milliseconds) for the state change. If both the
 *   transition time and the delay are zero, the transition is immediate.
 *
 *   This applies to messages the Mesh Model specification defines to have
 *   transition and delay times and can be left as zero for others.
 *   @endparblock
 * @param[in] delay_ms @parblock
 *   Delay time (in milliseconds) before starting the state change. If both the
 *   transition time and the delay are zero, the transition is immediate.
 *
 *   This applies to messages the Mesh Model specification defines to have
 *   transition and delay times and can be left as zero for others.
 *   @endparblock
 * @param[in] flags Message flags. Bitmask of the following:
 *     - <b>Bit 0:</b> Response required. If non-zero, the client expects a
 *       response from the server
 *     - <b>Bit 1:</b> Default transition timer. If non-zero, the client
 *       requests that server uses its default transition timer and the supplied
 *       transition and delay values are ignored.
 * @param[in] type Model-specific request type. See set request types list for
 *   details.
 * @param[in] parameters_len Length of data in @p parameters
 * @param[in] parameters Message-specific set request parameters serialized into
 *   a byte array
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_generic_client_server_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_client_set(uint16_t server_address,
                                         uint16_t elem_index,
                                         uint16_t model_id,
                                         uint16_t appkey_index,
                                         uint8_t tid,
                                         uint32_t transition_ms,
                                         uint16_t delay_ms,
                                         uint16_t flags,
                                         uint8_t type,
                                         size_t parameters_len,
                                         const uint8_t* parameters);

/***************************************************************************//**
 *
 * Publish a set request to the network using the publish address and publish
 * application key of the model. The message will be received by the server
 * models which subscribe to the publish address, and there's no need to
 * explicitly specify a destination address or application key.
 *
 * The server model responses will be reported in @ref
 * sl_btmesh_evt_generic_client_server_status events. To generate responses, the
 * corresponding flag needs to be set.
 *
 * @param[in] elem_index Client model element index
 * @param[in] model_id Client model ID
 * @param[in] tid Transaction identifier
 * @param[in] transition_ms @parblock
 *   Transition time (in milliseconds) for the state change. If both the
 *   transition time and the delay are zero, the transition is immediate.
 *
 *   This applies to messages the Mesh Model specification defines to have
 *   transition and delay times and can be left as zero for others.
 *   @endparblock
 * @param[in] delay_ms @parblock
 *   Delay time (in milliseconds) before starting the state change. If both the
 *   transition time and the delay are zero, the transition is immediate.
 *
 *   This applies to messages the Mesh Model specification defines to have
 *   transition and delay times, and can be left as zero for others.
 *   @endparblock
 * @param[in] flags Message flags. Bitmask of the following:
 *     - <b>Bit 0:</b> Response required. If non-zero, the client expects a
 *       response from the server
 *     - <b>Bit 1:</b> Default transition timer. If non-zero client requests
 *       that server uses its default transition timer and the supplied
 *       transition and delay values are ignored.
 * @param[in] type Model-specific request type. See set request types list for
 *   details.
 * @param[in] parameters_len Length of data in @p parameters
 * @param[in] parameters Message-specific set request parameters serialized into
 *   a byte array
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_generic_client_server_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_client_publish(uint16_t elem_index,
                                             uint16_t model_id,
                                             uint8_t tid,
                                             uint32_t transition_ms,
                                             uint16_t delay_ms,
                                             uint16_t flags,
                                             uint8_t type,
                                             size_t parameters_len,
                                             const uint8_t* parameters);

/***************************************************************************//**
 *
 * Get the current state of a server model or models in the network, with
 * additional parameters detailing the request. Besides the immediate result
 * code, the response or responses from the network will generate server state
 * report events for the replies received.
 *
 * The server model responses will be reported in @ref
 * sl_btmesh_evt_generic_client_server_status events.
 *
 * This call is used to query properties, for which the property ID is given as
 * a parameter.
 *
 * @param[in] server_address Destination server model address
 * @param[in] elem_index Client model element index
 * @param[in] model_id Client model ID
 * @param[in] appkey_index The application key index to use.
 * @param[in] type Model-specific state type, identifying the kind of state to
 *   retrieve. See @ref sl_btmesh_generic_client_get_state_type list for
 *   details.
 * @param[in] parameters_len Length of data in @p parameters
 * @param[in] parameters Message-specific get request parameters serialized into
 *   a byte array
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_generic_client_server_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_client_get_params(uint16_t server_address,
                                                uint16_t elem_index,
                                                uint16_t model_id,
                                                uint16_t appkey_index,
                                                uint8_t type,
                                                size_t parameters_len,
                                                const uint8_t* parameters);

/***************************************************************************//**
 *
 * Initialize generic client models. This command initializes all generic client
 * models on the device. Alternatively, only the necessary client models can be
 * initialized using model-specific initialization commands. Using
 * model-specific initialization can result in a smaller firmware image size for
 * SoC projects.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_client_init();

/***************************************************************************//**
 *
 * Initialize the generic client model common functionality. This command should
 * be called after all model-specific initialization calls are done. It does not
 * need to be called if @ref sl_btmesh_generic_client_init is used.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_client_init_common();

/***************************************************************************//**
 *
 * Initialize generic on/off client models.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_client_init_on_off();

/***************************************************************************//**
 *
 * Initialize generic level client models.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_client_init_level();

/***************************************************************************//**
 *
 * Initialize generic default transition time client models.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_client_init_default_transition_time();

/***************************************************************************//**
 *
 * Initialize generic power on/off client models.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_client_init_power_on_off();

/***************************************************************************//**
 *
 * Initialize generic power level client models.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_client_init_power_level();

/***************************************************************************//**
 *
 * Initialize generic battery client models.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_client_init_battery();

/***************************************************************************//**
 *
 * Initialize generic location client models.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_client_init_location();

/***************************************************************************//**
 *
 * Initialize generic property client models.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_client_init_property();

/***************************************************************************//**
 *
 * Initialize light lightness client models.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_client_init_lightness();

/***************************************************************************//**
 *
 * Initialize light CTL client models.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_client_init_ctl();

/***************************************************************************//**
 *
 * Initialize light HSL client models
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_client_init_hsl();

/** @} */ // end addtogroup sl_btmesh_generic_client

/**
 * @addtogroup sl_btmesh_generic_server Bluetooth Mesh Generic Server Model
 * @{
 *
 * @brief Bluetooth Mesh Generic Server Model
 *
 * Generic server model API provides functionality to send and receive messages
 * using Bluetooth SIG server models, including generic server models and
 * lighting server models.
 *
 * Throughout the API the server model being used is identified by its element
 * address and model ID, while the client model generating requests to the
 * server model is identified by its element address and model ID.
 *
 * The generic server model API is designed on the premise that the actual state
 * the model represents resides in and is owned by the application, not by the
 * stack.
 *
 * The model acts as a cache for client queries. In other words, the stack
 * handles state requests from the client automatically. The application does
 * not need to handle those. The cached value is also used for periodic
 * publication.
 *
 * The flip side of caching is that when the state represented by the model
 * changes in the application, it must update the cached value to the stack by
 * issuing a @ref sl_btmesh_generic_server_update command.
 *
 * When a client model requests a state change, the stack will generate a @ref
 * sl_btmesh_evt_generic_server_client_request event which the application must
 * process. Then, if the client needs a response the application has to issue a
 * @ref sl_btmesh_generic_server_respond command corresponding to the request.
 * Otherwise, the application only has to update the state with a @ref
 * sl_btmesh_generic_server_update command, which does not result in sending any
 * messages to the network.
 *
 * Note that, because the Mesh Model specification requires that certain states
 * are bound together and because the stack enforces that, updating one cached
 * state may result in an update of the corresponding bound state, for which the
 * stack generates a @ref sl_btmesh_evt_generic_server_state_changed event. For
 * example, when a dimmable light is switched off, the lightness level bound to
 * the on/off state, is also set to zero because the states are bound.
 *
 * Data for state change requests and server responses is passed as serialized
 * byte arrays through BGAPI. Bluetooth mesh SDK has functions to convert byte
 * arrays to and from model state structures.
 *
 * The stack will handle Mesh transaction layer segmentation and reassembly
 * automatically if the messages sent are long enough to require it.
 *
 * <b>Note on time resolution</b>
 *
 * Because of message formatting, transition time and remaining time resolution
 * units depend on the requested or reported value. For example, until 6.2
 * seconds it is 100 ms; until 62 seconds it is 1 s; until 620 seconds it is 10
 * s; and until 620 minutes it is 10 minutes. The value can't be longer than 620
 * minutes. Therefore, it is not possible to request a delay of exactly 7500 ms.
 * The resolution unit is 1 s between 6.2 and 62 seconds, so the value is
 * rounded down to 7 s.
 *
 * Delay resolution is 5 ms and values will be rounded down to the closest 5 ms.
 * The value can't be longer than 1275 ms.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_generic_server_respond_id                          0x001f0028
#define sl_btmesh_cmd_generic_server_update_id                           0x011f0028
#define sl_btmesh_cmd_generic_server_publish_id                          0x021f0028
#define sl_btmesh_cmd_generic_server_init_id                             0x041f0028
#define sl_btmesh_cmd_generic_server_init_common_id                      0x051f0028
#define sl_btmesh_cmd_generic_server_init_on_off_id                      0x061f0028
#define sl_btmesh_cmd_generic_server_init_level_id                       0x071f0028
#define sl_btmesh_cmd_generic_server_init_default_transition_time_id     0x081f0028
#define sl_btmesh_cmd_generic_server_init_power_on_off_id                0x091f0028
#define sl_btmesh_cmd_generic_server_init_power_level_id                 0x0a1f0028
#define sl_btmesh_cmd_generic_server_init_battery_id                     0x0b1f0028
#define sl_btmesh_cmd_generic_server_init_location_id                    0x0c1f0028
#define sl_btmesh_cmd_generic_server_init_property_id                    0x0d1f0028
#define sl_btmesh_cmd_generic_server_init_lightness_id                   0x0e1f0028
#define sl_btmesh_cmd_generic_server_init_ctl_id                         0x0f1f0028
#define sl_btmesh_cmd_generic_server_init_hsl_id                         0x101f0028
#define sl_btmesh_cmd_generic_server_get_cached_state_id                 0x111f0028
#define sl_btmesh_rsp_generic_server_respond_id                          0x001f0028
#define sl_btmesh_rsp_generic_server_update_id                           0x011f0028
#define sl_btmesh_rsp_generic_server_publish_id                          0x021f0028
#define sl_btmesh_rsp_generic_server_init_id                             0x041f0028
#define sl_btmesh_rsp_generic_server_init_common_id                      0x051f0028
#define sl_btmesh_rsp_generic_server_init_on_off_id                      0x061f0028
#define sl_btmesh_rsp_generic_server_init_level_id                       0x071f0028
#define sl_btmesh_rsp_generic_server_init_default_transition_time_id     0x081f0028
#define sl_btmesh_rsp_generic_server_init_power_on_off_id                0x091f0028
#define sl_btmesh_rsp_generic_server_init_power_level_id                 0x0a1f0028
#define sl_btmesh_rsp_generic_server_init_battery_id                     0x0b1f0028
#define sl_btmesh_rsp_generic_server_init_location_id                    0x0c1f0028
#define sl_btmesh_rsp_generic_server_init_property_id                    0x0d1f0028
#define sl_btmesh_rsp_generic_server_init_lightness_id                   0x0e1f0028
#define sl_btmesh_rsp_generic_server_init_ctl_id                         0x0f1f0028
#define sl_btmesh_rsp_generic_server_init_hsl_id                         0x101f0028
#define sl_btmesh_rsp_generic_server_get_cached_state_id                 0x111f0028

/**
 * @addtogroup sl_btmesh_evt_generic_server_client_request sl_btmesh_evt_generic_server_client_request
 * @{
 * @brief State change request sent by a client model
 *
 * This may be generated either because of a request directly to this model, or
 * a request sent to an address which is subscribed to by the model.
 */

/** @brief Identifier of the client_request event */
#define sl_btmesh_evt_generic_server_client_request_id                   0x001f00a8

/***************************************************************************//**
 * @brief Data structure of the client_request event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_generic_server_client_request_s
{
  uint16_t   server_address; /**< Address the message was sent to, which can be
                                  either the model element's unicast address, or
                                  model's subscription address */
  uint16_t   elem_index;     /**< Server model element index */
  uint16_t   model_id;       /**< Server model ID */
  uint16_t   client_address; /**< Address of the client model which sent the
                                  message */
  uint16_t   appkey_index;   /**< The application key index used in encrypting
                                  the request; Responses need to be encrypted
                                  with the same key. */
  uint32_t   transition_ms;  /**< Requested transition time (in milliseconds)
                                  for the state change. If both the transition
                                  time and the delay are zero, the transition is
                                  immediate.

                                  This applies to messages, which the Mesh Model
                                  specification defines to have transition and
                                  delay times and will be zero for others. */
  uint16_t   delay_ms;       /**< Delay time (in milliseconds) before starting
                                  the state change. If both the transition time
                                  and the delay are zero, the transition is
                                  immediate.

                                  This applies to messages, which the Mesh Model
                                  specification defines to have transition and
                                  delay times and will be zero for others. */
  uint16_t   flags;          /**< Message flags. Bitmask of the following
                                  values:
                                    - <b>Bit 0:</b> Non-relayed. If non-zero,
                                      indicates that the client message was not
                                      relayed (TTL was zero) and that the server
                                      is within direct radio range of the
                                      client.
                                    - <b>Bit 1:</b> Response required. If
                                      non-zero, the client expects a response
                                      from the server. */
  uint8_t    type;           /**< Model-specific request type. See set request
                                  types list for details. */
  uint8array parameters;     /**< Message-specific parameters serialized into a
                                  byte array */
});

typedef struct sl_btmesh_evt_generic_server_client_request_s sl_btmesh_evt_generic_server_client_request_t;

/** @} */ // end addtogroup sl_btmesh_evt_generic_server_client_request

/**
 * @addtogroup sl_btmesh_evt_generic_server_state_changed sl_btmesh_evt_generic_server_state_changed
 * @{
 * @brief Cached model state changed
 *
 * This may happen either as a direct result of model state update by the
 * application, in which case the event can be ignored, or because the update of
 * one model state resulted in an update of a bound model state according to the
 * Mesh model specification. In this case, the application should update its own
 * value accordingly.
 */

/** @brief Identifier of the state_changed event */
#define sl_btmesh_evt_generic_server_state_changed_id                    0x011f00a8

/***************************************************************************//**
 * @brief Data structure of the state_changed event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_generic_server_state_changed_s
{
  uint16_t   elem_index;   /**< Server model element index */
  uint16_t   model_id;     /**< Server model ID */
  uint32_t   remaining_ms; /**< Time (in milliseconds) remaining before
                                transition from current state to target state is
                                complete. Ignored if no transition is taking
                                place. */
  uint8_t    type;         /**< Model-specific state type, identifying the kind
                                of state reported in the state change event. See
                                get state types list for details. */
  uint8array parameters;   /**< Message-specific parameters, serialized into a
                                byte array */
});

typedef struct sl_btmesh_evt_generic_server_state_changed_s sl_btmesh_evt_generic_server_state_changed_t;

/** @} */ // end addtogroup sl_btmesh_evt_generic_server_state_changed

/**
 * @addtogroup sl_btmesh_evt_generic_server_state_recall sl_btmesh_evt_generic_server_state_recall
 * @{
 * @brief Cached model state changed due to scene recall operation
 */

/** @brief Identifier of the state_recall event */
#define sl_btmesh_evt_generic_server_state_recall_id                     0x021f00a8

/***************************************************************************//**
 * @brief Data structure of the state_recall event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_generic_server_state_recall_s
{
  uint16_t   elem_index;         /**< Server model element index */
  uint16_t   model_id;           /**< Server model ID */
  uint32_t   transition_time_ms; /**< Time (in milliseconds) remaining before
                                      transition from current state to target
                                      state should be complete. Ignored if no
                                      transition is taking place. */
  uint8_t    type;               /**< Model-specific state type, identifying the
                                      kind of state reported in the state change
                                      event. See get state types list for
                                      details. */
  uint8array parameters;         /**< Model state - specific parameters,
                                      serialized into a byte array */
});

typedef struct sl_btmesh_evt_generic_server_state_recall_s sl_btmesh_evt_generic_server_state_recall_t;

/** @} */ // end addtogroup sl_btmesh_evt_generic_server_state_recall

/***************************************************************************//**
 *
 * Server response to a client request. This command must be used when an
 * application updates the server model state as a response to a @ref
 * sl_btmesh_evt_generic_server_client_request event which required a response.
 *
 * @param[in] client_address Address of the client model which sent the message
 * @param[in] elem_index Server model element index
 * @param[in] model_id Server model ID
 * @param[in] appkey_index The application key index used
 * @param[in] remaining_ms Time (in milliseconds) remaining before transition
 *   from current state to target state is complete. Set to zero if no
 *   transition is taking place or if the transition time does not apply to the
 *   state change.
 * @param[in] flags Message flags. Bitmask of the following:
 *     - <b>Bit 0:</b> Non-relayed. If non-zero, indicates a response to a
 *       non-relayed request.
 * @param[in] type Model-specific state type, identifying the kind of state to
 *   be updated. See @ref sl_btmesh_generic_client_get_state_type list for
 *   details.
 * @param[in] parameters_len Length of data in @p parameters
 * @param[in] parameters Message-specific parameters serialized into a byte
 *   array
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_server_respond(uint16_t client_address,
                                             uint16_t elem_index,
                                             uint16_t model_id,
                                             uint16_t appkey_index,
                                             uint32_t remaining_ms,
                                             uint16_t flags,
                                             uint8_t type,
                                             size_t parameters_len,
                                             const uint8_t* parameters);

/***************************************************************************//**
 *
 * Server state update. This command must be used when an application updates
 * the server model state as a response to a @ref
 * sl_btmesh_evt_generic_server_client_request event which did not require a
 * response, but also when the application state changes spontaneously or as a
 * result of some external (non-Mesh) event.
 *
 * @param[in] elem_index Server model element index
 * @param[in] model_id Server model ID
 * @param[in] remaining_ms Time (in milliseconds) remaining before transition
 *   from current state to target state is complete. Set to zero if no
 *   transition is taking place or if transition time does not apply to the
 *   state change.
 * @param[in] type Model-specific state type, identifying the kind of state to
 *   be updated. See @ref sl_btmesh_generic_client_get_state_type list for
 *   details.
 * @param[in] parameters_len Length of data in @p parameters
 * @param[in] parameters Message-specific parameters, serialized into a byte
 *   array
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_server_update(uint16_t elem_index,
                                            uint16_t model_id,
                                            uint32_t remaining_ms,
                                            uint8_t type,
                                            size_t parameters_len,
                                            const uint8_t* parameters);

/***************************************************************************//**
 *
 * Publish the server state to the network using the publish parameters
 * configured in the model. The message is constructed using the cached state in
 * the stack.
 *
 * @param[in] elem_index Server model element index
 * @param[in] model_id Server model ID
 * @param[in] type Model-specific state type, identifying the kind of state used
 *   in the published message. See @ref sl_btmesh_generic_client_get_state_type
 *   list for details.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_server_publish(uint16_t elem_index,
                                             uint16_t model_id,
                                             uint8_t type);

/***************************************************************************//**
 *
 * Initialize generic server models. This command initializes all generic server
 * models on the device. Alternatively, only the necessary server models can be
 * initialized using model-specific initialization commands. Using
 * model-specific initialization can result in a smaller firmware image size for
 * SoC projects.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_server_init();

/***************************************************************************//**
 *
 * Initialize the generic server model common functionality. This should be
 * called after all model-specific initialization calls are done, and does not
 * need to be called if @ref sl_btmesh_generic_server_init is used.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_server_init_common();

/***************************************************************************//**
 *
 * Initialize generic on/off server models.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_server_init_on_off();

/***************************************************************************//**
 *
 * Initialize generic level server models.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_server_init_level();

/***************************************************************************//**
 *
 * Initialize generic default transition time server models.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_server_init_default_transition_time();

/***************************************************************************//**
 *
 * Initialize generic power on/off server models, power on/off setup server
 * models, and all models they extend.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_server_init_power_on_off();

/***************************************************************************//**
 *
 * Initialize generic power level server models, power level setup server
 * models, and all models they extend.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_server_init_power_level();

/***************************************************************************//**
 *
 * Initialize generic battery server models.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_server_init_battery();

/***************************************************************************//**
 *
 * Initialize generic location and generic location setup server models.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_server_init_location();

/***************************************************************************//**
 *
 * Initialize generic property server models.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_server_init_property();

/***************************************************************************//**
 *
 * Initialize light lightness server models, light lightness setup server
 * models, and all models they extend.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_server_init_lightness();

/***************************************************************************//**
 *
 * Initialize light CTL server models, light CTL temperature server models,
 * light CTL setup server models, and all models they extend.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_server_init_ctl();

/***************************************************************************//**
 *
 * Initialize light HSL server models, light HSL hue server models, light HSL
 * saturation server models, light HSL setup server models, and all models they
 * extend
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_server_init_hsl();

/***************************************************************************//**
 *
 * Get model cached state. This command can be used to get cached model states
 * after scene recall when using compacted recall events. This command supports
 * only those states that would have been reported by @ref
 * sl_btmesh_evt_generic_server_state_recall events.
 *
 * @param[in] elem_index Server model element index
 * @param[in] model_id Server model ID
 * @param[in] type Model-specific state type, identifying the kind of state
 *   reported in the state change event. See @ref
 *   sl_btmesh_generic_client_get_state_type list for details.
 * @param[out] remaining_ms Time (in milliseconds) remaining before transition
 *   from current state to target state is complete. Ignored if no transition is
 *   taking place.
 * @param[in] max_parameters_size Size of output buffer passed in @p parameters
 * @param[out] parameters_len On return, set to the length of output data
 *   written to @p parameters
 * @param[out] parameters Message-specific parameters, serialized into a byte
 *   array
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_generic_server_get_cached_state(uint16_t elem_index,
                                                      uint16_t model_id,
                                                      uint8_t type,
                                                      uint32_t *remaining_ms,
                                                      size_t max_parameters_size,
                                                      size_t *parameters_len,
                                                      uint8_t *parameters);

/** @} */ // end addtogroup sl_btmesh_generic_server

/**
 * @addtogroup sl_btmesh_test Bluetooth Mesh Test Utilities
 * @{
 *
 * @brief Bluetooth Mesh Test Utilities
 *
 * These commands are meant for development and testing. Do not use in
 * production software.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_test_get_nettx_id                                  0x00220028
#define sl_btmesh_cmd_test_set_nettx_id                                  0x01220028
#define sl_btmesh_cmd_test_get_relay_id                                  0x02220028
#define sl_btmesh_cmd_test_set_relay_id                                  0x03220028
#define sl_btmesh_cmd_test_set_ivupdate_test_mode_id                     0x05220028
#define sl_btmesh_cmd_test_get_ivupdate_test_mode_id                     0x06220028
#define sl_btmesh_cmd_test_set_ivupdate_state_id                         0x08220028
#define sl_btmesh_cmd_test_send_beacons_id                               0x09220028
#define sl_btmesh_cmd_test_bind_local_model_app_id                       0x0a220028
#define sl_btmesh_cmd_test_unbind_local_model_app_id                     0x0b220028
#define sl_btmesh_cmd_test_add_local_model_sub_id                        0x0c220028
#define sl_btmesh_cmd_test_remove_local_model_sub_id                     0x0d220028
#define sl_btmesh_cmd_test_add_local_model_sub_va_id                     0x0e220028
#define sl_btmesh_cmd_test_remove_local_model_sub_va_id                  0x0f220028
#define sl_btmesh_cmd_test_get_local_model_sub_id                        0x10220028
#define sl_btmesh_cmd_test_set_local_model_pub_id                        0x11220028
#define sl_btmesh_cmd_test_set_local_model_pub_va_id                     0x12220028
#define sl_btmesh_cmd_test_get_local_model_pub_id                        0x13220028
#define sl_btmesh_cmd_test_set_local_heartbeat_subscription_id           0x14220028
#define sl_btmesh_cmd_test_get_local_heartbeat_subscription_id           0x15220028
#define sl_btmesh_cmd_test_get_local_heartbeat_publication_id            0x16220028
#define sl_btmesh_cmd_test_set_local_heartbeat_publication_id            0x17220028
#define sl_btmesh_cmd_test_add_local_key_id                              0x1a220028
#define sl_btmesh_cmd_test_remove_local_key_id                           0x1b220028
#define sl_btmesh_cmd_test_update_local_key_id                           0x1c220028
#define sl_btmesh_cmd_test_set_adv_bearer_state_id                       0x1f220028
#define sl_btmesh_cmd_test_prov_get_device_key_id                        0x23220028
#define sl_btmesh_cmd_test_prov_prepare_key_refresh_id                   0x24220028
#define sl_btmesh_cmd_test_cancel_segmented_tx_id                        0x25220028
#define sl_btmesh_cmd_test_set_iv_index_id                               0x26220028
#define sl_btmesh_cmd_test_set_element_seqnum_id                         0x27220028
#define sl_btmesh_cmd_test_set_model_option_id                           0x28220028
#define sl_btmesh_cmd_test_get_local_model_app_bindings_id               0x29220028
#define sl_btmesh_cmd_test_get_replay_protection_list_entry_id           0x2a220028
#define sl_btmesh_cmd_test_clear_replay_protection_list_entry_id         0x2b220028
#define sl_btmesh_cmd_test_set_replay_protection_list_diagnostics_id     0x2c220028
#define sl_btmesh_cmd_test_get_model_option_id                           0x2d220028
#define sl_btmesh_cmd_test_get_default_ttl_id                            0x2e220028
#define sl_btmesh_cmd_test_set_default_ttl_id                            0x2f220028
#define sl_btmesh_cmd_test_get_gatt_proxy_id                             0x30220028
#define sl_btmesh_cmd_test_set_gatt_proxy_id                             0x31220028
#define sl_btmesh_cmd_test_get_identity_id                               0x32220028
#define sl_btmesh_cmd_test_set_identity_id                               0x33220028
#define sl_btmesh_cmd_test_get_friend_id                                 0x34220028
#define sl_btmesh_cmd_test_set_friend_id                                 0x35220028
#define sl_btmesh_cmd_test_get_beacon_id                                 0x36220028
#define sl_btmesh_cmd_test_set_beacon_id                                 0x37220028
#define sl_btmesh_cmd_test_get_private_identity_id                       0x38220028
#define sl_btmesh_cmd_test_set_private_identity_id                       0x39220028
#define sl_btmesh_cmd_test_set_adv_provisioning_bearer_timing_id         0x3a220028
#define sl_btmesh_cmd_test_update_keyrefresh_phase_id                    0x3b220028
#define sl_btmesh_cmd_test_send_private_beacons_id                       0x3c220028
#define sl_btmesh_cmd_test_set_adv_params_id                             0x3d220028
#define sl_btmesh_cmd_test_set_scan_params_id                            0x3e220028
#define sl_btmesh_cmd_test_adv_use_random_address_id                     0x3f220028
#define sl_btmesh_rsp_test_get_nettx_id                                  0x00220028
#define sl_btmesh_rsp_test_set_nettx_id                                  0x01220028
#define sl_btmesh_rsp_test_get_relay_id                                  0x02220028
#define sl_btmesh_rsp_test_set_relay_id                                  0x03220028
#define sl_btmesh_rsp_test_set_ivupdate_test_mode_id                     0x05220028
#define sl_btmesh_rsp_test_get_ivupdate_test_mode_id                     0x06220028
#define sl_btmesh_rsp_test_set_ivupdate_state_id                         0x08220028
#define sl_btmesh_rsp_test_send_beacons_id                               0x09220028
#define sl_btmesh_rsp_test_bind_local_model_app_id                       0x0a220028
#define sl_btmesh_rsp_test_unbind_local_model_app_id                     0x0b220028
#define sl_btmesh_rsp_test_add_local_model_sub_id                        0x0c220028
#define sl_btmesh_rsp_test_remove_local_model_sub_id                     0x0d220028
#define sl_btmesh_rsp_test_add_local_model_sub_va_id                     0x0e220028
#define sl_btmesh_rsp_test_remove_local_model_sub_va_id                  0x0f220028
#define sl_btmesh_rsp_test_get_local_model_sub_id                        0x10220028
#define sl_btmesh_rsp_test_set_local_model_pub_id                        0x11220028
#define sl_btmesh_rsp_test_set_local_model_pub_va_id                     0x12220028
#define sl_btmesh_rsp_test_get_local_model_pub_id                        0x13220028
#define sl_btmesh_rsp_test_set_local_heartbeat_subscription_id           0x14220028
#define sl_btmesh_rsp_test_get_local_heartbeat_subscription_id           0x15220028
#define sl_btmesh_rsp_test_get_local_heartbeat_publication_id            0x16220028
#define sl_btmesh_rsp_test_set_local_heartbeat_publication_id            0x17220028
#define sl_btmesh_rsp_test_add_local_key_id                              0x1a220028
#define sl_btmesh_rsp_test_remove_local_key_id                           0x1b220028
#define sl_btmesh_rsp_test_update_local_key_id                           0x1c220028
#define sl_btmesh_rsp_test_set_adv_bearer_state_id                       0x1f220028
#define sl_btmesh_rsp_test_prov_get_device_key_id                        0x23220028
#define sl_btmesh_rsp_test_prov_prepare_key_refresh_id                   0x24220028
#define sl_btmesh_rsp_test_cancel_segmented_tx_id                        0x25220028
#define sl_btmesh_rsp_test_set_iv_index_id                               0x26220028
#define sl_btmesh_rsp_test_set_element_seqnum_id                         0x27220028
#define sl_btmesh_rsp_test_set_model_option_id                           0x28220028
#define sl_btmesh_rsp_test_get_local_model_app_bindings_id               0x29220028
#define sl_btmesh_rsp_test_get_replay_protection_list_entry_id           0x2a220028
#define sl_btmesh_rsp_test_clear_replay_protection_list_entry_id         0x2b220028
#define sl_btmesh_rsp_test_set_replay_protection_list_diagnostics_id     0x2c220028
#define sl_btmesh_rsp_test_get_model_option_id                           0x2d220028
#define sl_btmesh_rsp_test_get_default_ttl_id                            0x2e220028
#define sl_btmesh_rsp_test_set_default_ttl_id                            0x2f220028
#define sl_btmesh_rsp_test_get_gatt_proxy_id                             0x30220028
#define sl_btmesh_rsp_test_set_gatt_proxy_id                             0x31220028
#define sl_btmesh_rsp_test_get_identity_id                               0x32220028
#define sl_btmesh_rsp_test_set_identity_id                               0x33220028
#define sl_btmesh_rsp_test_get_friend_id                                 0x34220028
#define sl_btmesh_rsp_test_set_friend_id                                 0x35220028
#define sl_btmesh_rsp_test_get_beacon_id                                 0x36220028
#define sl_btmesh_rsp_test_set_beacon_id                                 0x37220028
#define sl_btmesh_rsp_test_get_private_identity_id                       0x38220028
#define sl_btmesh_rsp_test_set_private_identity_id                       0x39220028
#define sl_btmesh_rsp_test_set_adv_provisioning_bearer_timing_id         0x3a220028
#define sl_btmesh_rsp_test_update_keyrefresh_phase_id                    0x3b220028
#define sl_btmesh_rsp_test_send_private_beacons_id                       0x3c220028
#define sl_btmesh_rsp_test_set_adv_params_id                             0x3d220028
#define sl_btmesh_rsp_test_set_scan_params_id                            0x3e220028
#define sl_btmesh_rsp_test_adv_use_random_address_id                     0x3f220028

/**
 * @brief Specify the type of a key in key manipulation commands.
 */
typedef enum
{
  sl_btmesh_test_key_type_net = 0x0, /**< (0x0) Network key */
  sl_btmesh_test_key_type_app = 0x1  /**< (0x1) Application key */
} sl_btmesh_test_key_type_t;

/**
 * @addtogroup sl_btmesh_evt_test_local_heartbeat_subscription_complete sl_btmesh_evt_test_local_heartbeat_subscription_complete
 * @{
 * @brief Indicate that the heartbeat subscription period is over.
 */

/** @brief Identifier of the local_heartbeat_subscription_complete event */
#define sl_btmesh_evt_test_local_heartbeat_subscription_complete_id      0x002200a8

/***************************************************************************//**
 * @brief Data structure of the local_heartbeat_subscription_complete event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_test_local_heartbeat_subscription_complete_s
{
  uint16_t count;   /**< Number of received heartbeat messages */
  uint8_t  hop_min; /**< Minimum observed hop count in heartbeat messages */
  uint8_t  hop_max; /**< Maximum observed hop count in heartbeat messages */
});

typedef struct sl_btmesh_evt_test_local_heartbeat_subscription_complete_s sl_btmesh_evt_test_local_heartbeat_subscription_complete_t;

/** @} */ // end addtogroup sl_btmesh_evt_test_local_heartbeat_subscription_complete

/**
 * @addtogroup sl_btmesh_evt_test_replay_protection_list_entry_set sl_btmesh_evt_test_replay_protection_list_entry_set
 * @{
 * @brief Indication that a replay protection list entry has been set
 */

/** @brief Identifier of the replay_protection_list_entry_set event */
#define sl_btmesh_evt_test_replay_protection_list_entry_set_id           0x012200a8

/***************************************************************************//**
 * @brief Data structure of the replay_protection_list_entry_set event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_test_replay_protection_list_entry_set_s
{
  uint16_t address; /**< Source address for the replay protection list entry */
  uint8_t  cancel;  /**< Nonzero when replay protection list update relates to a
                         cancelled segmented reception */
});

typedef struct sl_btmesh_evt_test_replay_protection_list_entry_set_s sl_btmesh_evt_test_replay_protection_list_entry_set_t;

/** @} */ // end addtogroup sl_btmesh_evt_test_replay_protection_list_entry_set

/**
 * @addtogroup sl_btmesh_evt_test_replay_protection_list_entry_cleared sl_btmesh_evt_test_replay_protection_list_entry_cleared
 * @{
 * @brief Indication that a replay protection list entry has been cleared
 */

/** @brief Identifier of the replay_protection_list_entry_cleared event */
#define sl_btmesh_evt_test_replay_protection_list_entry_cleared_id       0x022200a8

/***************************************************************************//**
 * @brief Data structure of the replay_protection_list_entry_cleared event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_test_replay_protection_list_entry_cleared_s
{
  uint16_t address; /**< Source address for the replay protection list entry */
});

typedef struct sl_btmesh_evt_test_replay_protection_list_entry_cleared_s sl_btmesh_evt_test_replay_protection_list_entry_cleared_t;

/** @} */ // end addtogroup sl_btmesh_evt_test_replay_protection_list_entry_cleared

/**
 * @addtogroup sl_btmesh_evt_test_replay_protection_list_saved sl_btmesh_evt_test_replay_protection_list_saved
 * @{
 * @brief Indication that replay protection list has been saved
 */

/** @brief Identifier of the replay_protection_list_saved event */
#define sl_btmesh_evt_test_replay_protection_list_saved_id               0x032200a8

/***************************************************************************//**
 * @brief Data structure of the replay_protection_list_saved event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_test_replay_protection_list_saved_s
{
  uint16_t result;      /**< SL_STATUS_OK if successful. Error code otherwise. */
  uint16_t saved_count; /**< Number of entries saved successfully */
  uint16_t total_count; /**< Number of entries in the list in total */
});

typedef struct sl_btmesh_evt_test_replay_protection_list_saved_s sl_btmesh_evt_test_replay_protection_list_saved_t;

/** @} */ // end addtogroup sl_btmesh_evt_test_replay_protection_list_saved

/**
 * @addtogroup sl_btmesh_evt_test_replay_protection_list_full sl_btmesh_evt_test_replay_protection_list_full
 * @{
 * @brief Indication that replay protection list is full when trying to process
 * a message
 */

/** @brief Identifier of the replay_protection_list_full event */
#define sl_btmesh_evt_test_replay_protection_list_full_id                0x042200a8

/** @} */ // end addtogroup sl_btmesh_evt_test_replay_protection_list_full

/***************************************************************************//**
 *
 * Get the network transmit state of a node.
 *
 * @param[out] count Number of network layer transmissions beyond the initial
 *   one. Range: 0-7.
 * @param[out] interval Transmit interval steps. The interval between
 *   transmissions is a random value between 10*(1+steps) and 10*(2+steps)
 *   milliseconds. For example, for a value of 2, the interval is between 30 and
 *   40 milliseconds. Range: 0-31.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_get_nettx(uint8_t *count, uint8_t *interval);

/***************************************************************************//**
 *
 * Set the network transmit state of a node locally. Normally, the network
 * transmit state is controlled by the Provisioner. This command overrides any
 * setting done by the Provisioner.
 *
 * @param[in] count Number of network layer transmissions beyond the initial
 *   one. Range: 0-7.
 * @param[in] interval Transmit interval steps. The interval between
 *   transmissions is a random value between 10*(1+steps) and 10*(2+steps)
 *   milliseconds. For example, for a value of 2 the interval would be between
 *   30 and 40 milliseconds. Range: 0-31.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_set_nettx(uint8_t count, uint8_t interval);

/***************************************************************************//**
 *

 *
 * @param[out] enabled State value indicating whether the relay functionality is
 *   not enabled on the node (0), is enabled on the node (1), or is not
 *   available (2).
 * @param[out] count Number of relay transmissions beyond the initial one.
 *   Range: 0-7.
 * @param[out] interval Relay retransmit interval steps. The interval between
 *   transmissions is 10*(1+steps) milliseconds. Range: 0-31.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_get_relay(uint8_t *enabled,
                                     uint8_t *count,
                                     uint8_t *interval);

/***************************************************************************//**
 *
 * Set the relay state and the relay retransmit state of a node locally.
 * Normally, these states are controlled by the Provisioner. This command
 * overrides any settings done by the Provisioner.
 *
 * @param[in] enabled Indicates whether the relay functionality is enabled on
 *   the node (1) or not (0); value indicating disabled (2) can't be set.
 * @param[in] count Number of relay transmissions beyond the initial one. Range:
 *   0-7.
 * @param[in] interval Relay retransmit interval steps. The interval between
 *   transmissions is 10*(1+steps) milliseconds. Range: 0-31.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_set_relay(uint8_t enabled,
                                     uint8_t count,
                                     uint8_t interval);

/***************************************************************************//**
 *
 * By default, IV index update is limited in how often the update procedure can
 * be performed. This test command can be called to set IV update test mode
 * where any time limits are ignored.
 *
 * @param[in] mode Whether test mode is enabled (1) or disabled (0).
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_set_ivupdate_test_mode(uint8_t mode);

/***************************************************************************//**
 *
 * Get the current IV update test mode. See @ref
 * sl_btmesh_test_set_ivupdate_test_mode for details.
 *
 * @param[out] mode Indicates whether test mode is enabled (1) or disabled (0).
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_get_ivupdate_test_mode(uint8_t *mode);

/***************************************************************************//**
 *
 * Forcefully change the IV update state on the device. Normally, the state
 * changes as a result of an IV index update procedure progressing from one
 * state to the next.
 *
 * @param[in] state Indicates whether IV update state should be entered (1) or
 *   exited (0).
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_set_ivupdate_state(uint8_t state);

/***************************************************************************//**
 *
 * Send secure network beacons for every network key on the device, regardless
 * of beacon configuration state or how many beacons sent by other devices have
 * been observed.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_send_beacons();

/***************************************************************************//**
 *
 * Bind a Model to an Appkey locally.
 *
 * @param[in] elem_index The index of the target Element, 0 is primary element
 * @param[in] appkey_index The Appkey to use for binding
 * @param[in] vendor_id Vendor ID for vendor-specific models. Use 0xffff for SIG
 *   models.
 * @param[in] model_id Model ID
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_bind_local_model_app(uint16_t elem_index,
                                                uint16_t appkey_index,
                                                uint16_t vendor_id,
                                                uint16_t model_id);

/***************************************************************************//**
 *
 * Remove a binding between a model and an application key locally.
 *
 * @param[in] elem_index The index of the target element, 0 is primary element
 * @param[in] appkey_index The Appkey to use for binding
 * @param[in] vendor_id Vendor ID for vendor-specific models. Use 0xffff for SIG
 *   models.
 * @param[in] model_id Model ID
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_unbind_local_model_app(uint16_t elem_index,
                                                  uint16_t appkey_index,
                                                  uint16_t vendor_id,
                                                  uint16_t model_id);

/***************************************************************************//**
 *
 * Add an address to a local model's subscription list.
 *
 * @param[in] elem_index The index of the target element, 0 is the primary
 *   element
 * @param[in] vendor_id Vendor ID for vendor-specific models. Use 0xffff for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID
 * @param[in] sub_address The address to add to the subscription list
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_add_local_model_sub(uint16_t elem_index,
                                               uint16_t vendor_id,
                                               uint16_t model_id,
                                               uint16_t sub_address);

/***************************************************************************//**
 *
 * Remove an address from a local model's subscription list.
 *
 * @param[in] elem_index The index of the target element, 0 is the primary
 *   element
 * @param[in] vendor_id Vendor ID for vendor-specific models. Use 0xffff for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID
 * @param[in] sub_address The address to remove from the subscription list
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_remove_local_model_sub(uint16_t elem_index,
                                                  uint16_t vendor_id,
                                                  uint16_t model_id,
                                                  uint16_t sub_address);

/***************************************************************************//**
 *
 * Add a virtual address to a local model's subscription list.
 *
 * @param[in] elem_index The index of the target element, 0 is the primary
 *   element
 * @param[in] vendor_id Vendor ID for vendor-specific models. Use 0xffff for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID
 * @param[in] sub_address_len Length of data in @p sub_address
 * @param[in] sub_address The Label UUID to add to the subscription list. The
 *   array must be exactly 16 bytes long.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_add_local_model_sub_va(uint16_t elem_index,
                                                  uint16_t vendor_id,
                                                  uint16_t model_id,
                                                  size_t sub_address_len,
                                                  const uint8_t* sub_address);

/***************************************************************************//**
 *
 * Remove a virtual address from a local model's subscription list.
 *
 * @param[in] elem_index The index of the target element, 0 is the primary
 *   element
 * @param[in] vendor_id Vendor ID for vendor-specific models. Use 0xffff for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID
 * @param[in] sub_address_len Length of data in @p sub_address
 * @param[in] sub_address The Label UUID to remove from the subscription list.
 *   The array must be exactly 16 bytes long.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_remove_local_model_sub_va(uint16_t elem_index,
                                                     uint16_t vendor_id,
                                                     uint16_t model_id,
                                                     size_t sub_address_len,
                                                     const uint8_t* sub_address);

/***************************************************************************//**
 *
 * Get all entries in a local model's subscription list.
 *
 * @param[in] elem_index The index of the target element, 0 is the primary
 *   element
 * @param[in] vendor_id Vendor ID for vendor-specific models. Use 0xffff for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID
 * @param[in] max_addresses_size Size of output buffer passed in @p addresses
 * @param[out] addresses_len On return, set to the length of output data written
 *   to @p addresses
 * @param[out] addresses List of 16-bit Mesh addresses; empty if not subscribed
 *   to any address. Ignore if the result code is non-zero.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_get_local_model_sub(uint16_t elem_index,
                                               uint16_t vendor_id,
                                               uint16_t model_id,
                                               size_t max_addresses_size,
                                               size_t *addresses_len,
                                               uint8_t *addresses);

/***************************************************************************//**
 *
 * Set a local model's publication address, key, and parameters.
 *
 * @param[in] elem_index The index of the target element, where 0 is the primary
 *   element
 * @param[in] appkey_index The application key index to use for the application
 *   messages published
 * @param[in] vendor_id Vendor ID for vendor-specific models. Use 0xffff for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID
 * @param[in] pub_address The address to publish to
 * @param[in] ttl Time-to-Live value for published messages. Valid values:
 *   range[0 - 127] and 255.
 * @param[in] period Publication period encoded as step count and step
 *   resolution. The encoding is as follows:
 *     - <b>Bits 0..5:</b> Step count
 *     - <b>Bits 6..7:</b> Step resolution:
 *         - 00: 100 milliseconds
 *         - 01: 1 second
 *         - 10: 10 seconds
 *         - 11: 10 minutes
 * @param[in] retrans @parblock
 *   Retransmission count and interval, which controls number of times that the
 *   model re-publishes the same message after the initial publish transmission
 *   and the cadence of retransmissions.
 *
 *   Retransmission count is encoded in the three low bits of the value, ranging
 *   from 0 to 7. Default value is 0 (no retransmissions).
 *
 *   Retransmission interval is encoded in the five high bits of the value,
 *   ranging from 0 to 31, in 50-millisecond units. Value of 0 corresponds to 50
 *   ms, while value of 31 corresponds to 1600 ms.
 *   @endparblock
 * @param[in] credentials Friendship credentials flag
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_set_local_model_pub(uint16_t elem_index,
                                               uint16_t appkey_index,
                                               uint16_t vendor_id,
                                               uint16_t model_id,
                                               uint16_t pub_address,
                                               uint8_t ttl,
                                               uint8_t period,
                                               uint8_t retrans,
                                               uint8_t credentials);

/***************************************************************************//**
 *
 * Set a model's publication virtual address, key, and parameters.
 *
 * @param[in] elem_index The index of the target element, 0 is the primary
 *   element
 * @param[in] appkey_index The application key index to use for the published
 *   messages
 * @param[in] vendor_id Vendor ID of the configured model. Use 0xffff for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID of the configured model
 * @param[in] ttl Publication time-to-live value. Valid values: range[0 - 127]
 *   and 255.
 * @param[in] period Publication period encoded as step count and step
 *   resolution. The encoding is as follows:
 *     - <b>Bits 0..5:</b> Step count
 *     - <b>Bits 6..7:</b> Step resolution:
 *         - 00: 100 milliseconds
 *         - 01: 1 second
 *         - 10: 10 seconds
 *         - 11: 10 minutes
 * @param[in] retrans See documentation of @ref
 *   sl_btmesh_test_set_local_model_pub for details.
 * @param[in] credentials Friendship credentials flag
 * @param[in] pub_address_len Length of data in @p pub_address
 * @param[in] pub_address The Label UUID to publish to. The byte array must be
 *   exactly 16 bytes long.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_set_local_model_pub_va(uint16_t elem_index,
                                                  uint16_t appkey_index,
                                                  uint16_t vendor_id,
                                                  uint16_t model_id,
                                                  uint8_t ttl,
                                                  uint8_t period,
                                                  uint8_t retrans,
                                                  uint8_t credentials,
                                                  size_t pub_address_len,
                                                  const uint8_t* pub_address);

/***************************************************************************//**
 *
 * Get a local model's publication address, key, and parameters.
 *
 * @param[in] elem_index The index of the target element, 0 is the primary
 *   element
 * @param[in] vendor_id Vendor ID for vendor-specific models. Use 0xffff for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID
 * @param[out] appkey_index The application key index used for the application
 *   messages published
 * @param[out] pub_address The address published to
 * @param[out] ttl Time-to-Live value for published messages
 * @param[out] period Publication period encoded as step count and step
 *   resolution. The encoding is as follows:
 *     - <b>Bits 0..5:</b> Step count
 *     - <b>Bits 6..7:</b> Step resolution:
 *         - 00: 100 milliseconds
 *         - 01: 1 second
 *         - 10: 10 seconds
 *         - 11: 10 minutes
 * @param[out] retrans See documentation of @ref
 *   sl_btmesh_test_set_local_model_pub for details.
 * @param[out] credentials Friendship credentials flag
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_get_local_model_pub(uint16_t elem_index,
                                               uint16_t vendor_id,
                                               uint16_t model_id,
                                               uint16_t *appkey_index,
                                               uint16_t *pub_address,
                                               uint8_t *ttl,
                                               uint8_t *period,
                                               uint8_t *retrans,
                                               uint8_t *credentials);

/***************************************************************************//**
 *
 * Set local node heartbeat subscription parameters. Normally heartbeat
 * subscription is controlled by the Provisioner.
 *
 * @param[in] subscription_source Source address for heartbeat messages. Must be
 *   either a unicast address or the unassigned address, in which case heartbeat
 *   messages are not processed.
 * @param[in] subscription_destination Destination address for heartbeat
 *   messages. The address must be either the unicast address of the primary
 *   element of the node, a group address, or the unassigned address. If it is
 *   the unassigned address, heartbeat messages are not processed.
 * @param[in] period_log Heartbeat subscription period setting. Valid values are
 *   as follows:
 *     - <b>0x00:</b> Heartbeat messages are not received
 *     - <b>0x01 .. 0x11:</b> Node will receive heartbeat messages for 2^(n-1)
 *       seconds
 *     - <b>0x12 .. 0xff:</b> Prohibited
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_set_local_heartbeat_subscription(uint16_t subscription_source,
                                                            uint16_t subscription_destination,
                                                            uint8_t period_log);

/***************************************************************************//**
 *
 * Get the local node heartbeat subscription state.
 *
 * @param[out] count Number of received heartbeat messages
 * @param[out] hop_min Minimum observed hop count in heartbeat messages
 * @param[out] hop_max Maximum observed hop count in heartbeat messages
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_get_local_heartbeat_subscription(uint16_t *count,
                                                            uint8_t *hop_min,
                                                            uint8_t *hop_max);

/***************************************************************************//**
 *
 * Get the heartbeat publication state of a local node.
 *
 * @param[out] publication_address Heartbeat publication address
 * @param[out] count Heartbeat publication remaining count
 * @param[out] period_log Heartbeat publication period setting. Valid values are
 *   as follows:
 *     - <b>0x00:</b> Heartbeat messages are not sent
 *     - <b>0x01 .. 0x11:</b> Node will send a heartbeat message every 2^(n-1)
 *       seconds
 *     - <b>0x12 .. 0xff:</b> Prohibited
 * @param[out] ttl Time-to-live parameter for heartbeat messages
 * @param[out] features Heartbeat trigger setting
 * @param[out] publication_netkey_index Index of the network key used to encrypt
 *   heartbeat messages
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_get_local_heartbeat_publication(uint16_t *publication_address,
                                                           uint8_t *count,
                                                           uint8_t *period_log,
                                                           uint8_t *ttl,
                                                           uint16_t *features,
                                                           uint16_t *publication_netkey_index);

/***************************************************************************//**
 *
 * Set the heartbeat publication state of a local node.
 *
 * @param[in] publication_address Heartbeat publication address. The address
 *   can't be a virtual address. Note that it can be the unassigned address, in
 *   which case the heartbeat publishing is disabled.
 * @param[in] count_log Heartbeat publication count setting. Valid values are as
 *   follows:
 *     - <b>0x00:</b> Heartbeat messages are not sent
 *     - <b>0x01 .. 0x11:</b> Node will send 2^(n-1) heartbeat messages
 *     - <b>0x12 .. 0xfe:</b> Prohibited
 *     - <b>0xff:</b> Hearbeat messages are sent indefinitely
 * @param[in] period_log Heartbeat publication period setting. Valid values are
 *   as follows:
 *     - <b>0x00:</b> Heartbeat messages are not sent
 *     - <b>0x01 .. 0x11:</b> Node will send a heartbeat message every 2^(n-1)
 *       seconds
 *     - <b>0x12 .. 0xff:</b> Prohibited
 * @param[in] ttl Time-to-live parameter for heartbeat messages. Valid values:
 *   range[0 - 127].
 * @param[in] features @parblock
 *   Heartbeat trigger setting. For bits set in the bitmask, reconfiguration of
 *   the node feature associated with the bit will result in the node emitting a
 *   heartbeat message. Valid values are as follows:
 *     - <b>Bit 0:</b> Relay feature
 *     - <b>Bit 1:</b> Proxy feature
 *     - <b>Bit 2:</b> Friend feature
 *     - <b>Bit 3:</b> Low power feature
 *
 *   Remaining bits are reserved for future use.
 *   @endparblock
 * @param[in] publication_netkey_index Index of the network key used to encrypt
 *   heartbeat messages
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_set_local_heartbeat_publication(uint16_t publication_address,
                                                           uint8_t count_log,
                                                           uint8_t period_log,
                                                           uint8_t ttl,
                                                           uint16_t features,
                                                           uint16_t publication_netkey_index);

/***************************************************************************//**
 *
 * Add a network or application key locally.
 *
 * @param[in] key_type Enum @ref sl_btmesh_test_key_type_t. 0 for network key, 1
 *   for application key
 * @param[in] key Key data
 * @param[in] key_index Index for the added key (must be unused)
 * @param[in] netkey_index Network key index to which the application key is
 *   bound; ignored for network keys
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_add_local_key(uint8_t key_type,
                                         aes_key_128 key,
                                         uint16_t key_index,
                                         uint16_t netkey_index);

/***************************************************************************//**
 *
 * Delete a network or application key locally.
 *
 * @param[in] key_type Enum @ref sl_btmesh_test_key_type_t. 0 for network key, 1
 *   for application key
 * @param[in] key_index Index of the key to delete
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_remove_local_key(uint8_t key_type,
                                            uint16_t key_index);

/***************************************************************************//**
 *
 * Update the network or application key value locally.
 *
 * Copies the existing network key value to the old value and replaces the
 * current value with the given key data.
 *
 * Note that the standard way to update keys on the Provisioner as well as on
 * nodes is to run the key refresh procedure. This command is for debugging
 * only.
 *
 * @param[in] key_type Enum @ref sl_btmesh_test_key_type_t. 0 for network key, 1
 *   for application key
 * @param[in] key Key data
 * @param[in] key_index Index for the key to update
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_update_local_key(uint8_t key_type,
                                            aes_key_128 key,
                                            uint16_t key_index);

/***************************************************************************//**
 *
 * Disable or enable advertisement bearer for sending.
 *
 * @param[in] state 0: disable advertisement, 1: enable advertisement.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_set_adv_bearer_state(uint8_t state);

/***************************************************************************//**
 *
 * Get the device key with the address of the node's primary element.
 *
 * @param[in] address Address of the node
 * @param[out] device_key Device key, 16-bytes
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_prov_get_device_key(uint16_t address,
                                               aes_key_128 *device_key);

/***************************************************************************//**
 *
 * Prepare the key refresh by feeding the new network key and all needed
 * application keys. The function can be called multiple times to include more
 * application keys. The network key must be the same in all calls. If the
 * network key is changed, the network key from the 1st command is used.
 * Sending application key data with length zero results in all initialization
 * data being forgotten unless this is done in the first prepare command i.e.,
 * trying to update only the network key. Also starting the key refresh
 * procedure results in all the preparation data being forgotten.
 *
 * @param[in] net_key New net key
 * @param[in] app_keys_len Length of data in @p app_keys
 * @param[in] app_keys list of new application keys, 16-bytes each
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_prov_prepare_key_refresh(aes_key_128 net_key,
                                                    size_t app_keys_len,
                                                    const uint8_t* app_keys);

/***************************************************************************//**
 *
 * Cancel sending a segmented message.
 *
 * @param[in] src_addr Source address for the segmented message
 * @param[in] dst_addr Destination address for the segmented message
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_cancel_segmented_tx(uint16_t src_addr,
                                               uint16_t dst_addr);

/***************************************************************************//**
 *
 * Set the IV index value of the node.
 *
 * @param[in] iv_index IV Index value to use
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_set_iv_index(uint32_t iv_index);

/***************************************************************************//**
 *
 * Set the current sequence number of an element.
 *
 * @param[in] elem_index The index of the target element, 0 is the primary
 *   element
 * @param[in] seqnum Sequence number to set on the target element
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_set_element_seqnum(uint16_t elem_index,
                                              uint32_t seqnum);

/***************************************************************************//**
 * @cond RESTRICTED
 *
 * Restricted/experimental API. Contact Silicon Labs sales for more information.
 *
 * Set the model-specific option.
 *
 * @param[in] elem_index The index of the target element, 0 is the primary
 *   element
 * @param[in] vendor_id Vendor ID for vendor-specific models. Use 0xffff for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID
 * @param[in] option Option to set
 * @param[in] value Value for the option
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @endcond
 ******************************************************************************/
sl_status_t sl_btmesh_test_set_model_option(uint16_t elem_index,
                                            uint16_t vendor_id,
                                            uint16_t model_id,
                                            uint8_t option,
                                            uint32_t value);

/***************************************************************************//**
 *
 * Get the application key bindings of a model.
 *
 * @param[in] elem_index The index of the target element, 0 is the primary
 *   element
 * @param[in] vendor_id Vendor ID for vendor-specific models. Use 0xffff for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID
 * @param[in] max_appkeys_size Size of output buffer passed in @p appkeys
 * @param[out] appkeys_len On return, set to the length of output data written
 *   to @p appkeys
 * @param[out] appkeys List of 16-bit application key indices; empty if model
 *   has not been bound to any application key.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_get_local_model_app_bindings(uint16_t elem_index,
                                                        uint16_t vendor_id,
                                                        uint16_t model_id,
                                                        size_t max_appkeys_size,
                                                        size_t *appkeys_len,
                                                        uint8_t *appkeys);

/***************************************************************************//**
 *
 * Get the replay protection list entry for an address.
 *
 * @param[in] address Source address to check
 * @param[out] seq Stored sequence number for the address; not a valid value if
 *   the result is not SL_STATUS_OK.
 * @param[out] seq_ivindex IV index in use at the time the sequence number was
 *   stored in the replay protection list; not a valid value if result is not
 *   SL_STATUS_OK.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_get_replay_protection_list_entry(uint16_t address,
                                                            uint32_t *seq,
                                                            uint32_t *seq_ivindex);

/***************************************************************************//**
 *
 * Clear the replay protection list entry for an address. Use this command
 * carefully because it may expose the node to replay attacks when misused.
 *
 * @param[in] address Source address to use in finding the entry
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_clear_replay_protection_list_entry(uint16_t address);

/***************************************************************************//**
 *
 * Enable or disable replay protection list diagnostic events. When enabled,
 * events related to the replay protection list changes are generated.
 *
 * @param[in] enable Enable (nonzero) or disable (zero) diagnostic events for
 *   replay protection list
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_set_replay_protection_list_diagnostics(uint8_t enable);

/***************************************************************************//**
 * @cond RESTRICTED
 *
 * Restricted/experimental API. Contact Silicon Labs sales for more information.
 *
 * Get a model-specific option.
 *
 * @param[in] elem_index The index of the target element, 0 is the primary
 *   element
 * @param[in] vendor_id Vendor ID for vendor-specific models. Use 0xffff for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID
 * @param[in] option Option to get.
 * @param[out] value Value for the option.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @endcond
 ******************************************************************************/
sl_status_t sl_btmesh_test_get_model_option(uint16_t elem_index,
                                            uint16_t vendor_id,
                                            uint16_t model_id,
                                            uint8_t option,
                                            uint32_t *value);

/***************************************************************************//**
 *
 * Get node default TTL state.
 *
 * @param[out] value Default time-to-live value. Valid value range is from 2 to
 *   127 for relayed PDUs, and 0 to indicate non-relayed PDUs
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_get_default_ttl(uint8_t *value);

/***************************************************************************//**
 *
 * Set node default TTL state.
 *
 * @param[in] set_value Default time-to-live value. See @ref
 *   sl_btmesh_test_get_default_ttl for details.
 * @param[out] value Default time-to-live value. See @ref
 *   sl_btmesh_test_get_default_ttl for details.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_set_default_ttl(uint8_t set_value, uint8_t *value);

/***************************************************************************//**
 *
 * Get node GATT proxy state.
 *
 * @param[out] value GATT proxy value of the node. Valid values are:
 *     - 0: GATT proxy feature is disabled
 *     - 1: GATT proxy feature is enabled
 *     - 2: GATT proxy feature is not supported
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_get_gatt_proxy(uint8_t *value);

/***************************************************************************//**
 *
 * Set node GATT proxy state.
 *
 * @param[in] set_value GATT proxy value to set. Valid values are:
 *     - 0: Proxy feature is disabled
 *     - 1: Proxy feature is enabled
 * @param[out] value GATT proxy state value. See @ref
 *   sl_btmesh_test_get_gatt_proxy for details.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_gatt_proxy_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_set_gatt_proxy(uint8_t set_value, uint8_t *value);

/***************************************************************************//**
 *
 * Get node identity state.
 *
 * @param[in] get_netkey_index Network key index for which the state is queried
 * @param[out] netkey_index Network key index for which the state is queried
 * @param[out] value Identity state of the node for the used network index.
 *   Valid values are as follows:
 *     - 0: Node identity advertising is disabled
 *     - 1: Node identity advertising is enabled
 *     - 2: Node identity advertising is not supported
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_get_identity(uint16_t get_netkey_index,
                                        uint16_t *netkey_index,
                                        uint8_t *value);

/***************************************************************************//**
 *
 * Set node identity state.
 *
 * @param[in] set_netkey_index Network key index for which the state is
 *   configured
 * @param[in] set_value Identity value to set. Valid values are:
 *     - 0: Node identity advertising is disabled
 *     - 1: Node identity advertising is enabled
 * @param[out] netkey_index Network key index for which the state is set
 * @param[out] value Identity state of the node for the used network index. See
 *   @ref sl_btmesh_test_get_identity for details
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_set_identity(uint16_t set_netkey_index,
                                        uint8_t set_value,
                                        uint16_t *netkey_index,
                                        uint8_t *value);

/***************************************************************************//**
 *
 * Get node friend state.
 *
 * @param[out] value Friend state value. Valid values are:
 *     - 0: Friend feature is not enabled
 *     - 1: Friend feature is enabled
 *     - 2: Friend feature is not supported
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_get_friend(uint8_t *value);

/***************************************************************************//**
 *
 * Set node friend state.
 *
 * @param[in] set_value Friend value to set. Valid values are:
 *     - 0: Friend feature is not enabled
 *     - 1: Friend feature is enabled
 * @param[out] value Friend state value. See @ref sl_btmesh_test_get_friend for
 *   detais.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_set_friend(uint8_t set_value, uint8_t *value);

/***************************************************************************//**
 *
 * Get node secure network beacon state.
 *
 * @param[out] value Secure network beacon value. Valid values are:
 *     - 0: Node is not broadcasting secure network beacons
 *     - 1: Node is broadcasting secure network beacons
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_get_beacon(uint8_t *value);

/***************************************************************************//**
 *
 * Set node secure network beacon state.
 *
 * @param[in] set_value Secure network beacon value to set. Valid values are:
 *     - 0: Node is not broadcasting secure network beacons
 *     - 1: Node is broadcasting secure network beacons
 * @param[out] value Secure network beacon value of the node.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_set_beacon(uint8_t set_value, uint8_t *value);

/***************************************************************************//**
 *
 * Get node private identity state.
 *
 * @param[in] get_netkey_index Network key index for which the state is queried
 * @param[out] netkey_index Network key index for which the state is queried
 * @param[out] value Private identity state of the node for the used network
 *   index. Valid values are as follows:
 *     - 0: Node private identity advertising is disabled
 *     - 1: Node private identity advertising is enabled
 *     - 2: Node private identity advertising is not supported
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_get_private_identity(uint16_t get_netkey_index,
                                                uint16_t *netkey_index,
                                                uint8_t *value);

/***************************************************************************//**
 *
 * Set node private identity state.
 *
 * @param[in] set_netkey_index Network key index for which the state is
 *   configured
 * @param[in] set_value Identity value to set. Valid values are:
 *     - 0: Node private identity advertising is disabled
 *     - 1: Node private identity advertising is enabled
 * @param[out] netkey_index Network key index for which the state is set
 * @param[out] value Identity state of the node for the used network index. See
 *   @ref sl_btmesh_test_get_private_identity for details
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_set_private_identity(uint16_t set_netkey_index,
                                                uint8_t set_value,
                                                uint16_t *netkey_index,
                                                uint8_t *value);

/***************************************************************************//**
 *
 * Set PB-ADV provisioning bearer timing. This controls how PB-ADV provisioning
 * PDUs are transmitted by the device; each transmission is preceded by a delay
 * that is computed by adding together a minimum interval with a random
 * variation.
 *
 * If this setting is not applied, a default interval of 20 ms and a default
 * variation of 30 ms is used, meaning that a single advertisement is sent with
 * a delay ranging from 20 to 50 ms.
 *
 * Note that using a non-default setting is against the Mesh specification and
 * should only used for troubleshooting.
 *
 * @param[in] pbadv_interval_ms PB-ADV Tx interval. Value is in milliseconds.
 *   Range: 10 to 1000.
 * @param[in] pbadv_variation_ms PB-ADV Tx interval variation. A random delay
 *   between 0 and this value is added to the PB-ADV Tx interval when
 *   transmitting a PB-ADV PDU. Value is in milliseconds. Range: 10 to 1000.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_set_adv_provisioning_bearer_timing(uint16_t pbadv_interval_ms,
                                                              uint16_t pbadv_variation_ms);

/***************************************************************************//**
 *
 * Change keyrefresh phase of of a network id.
 *
 * Update is done with following logic depending on current keyrefresh phase:
 *   - Phase 0 : return an errorcode for wrong phase
 *   - Phase 1 : set phase to 2
 *   - Phase 2 : set phase to 3
 *   - Phase 3 : Leave phase 3 as is
 *
 * This command is for debugging only
 *
 * @param[in] network_key_index Index for the key to put into next phase
 * @param[out] phase Key refresh phase state of the node for the given network
 *   key. Valid values are:
 *     - 0: Normal operation
 *     - 1: First phase of key refresh procedure
 *     - 2: Second phase of key refresh procedure
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_update_keyrefresh_phase(uint16_t network_key_index,
                                                   uint8_t *phase);

/***************************************************************************//**
 *
 * Send private network beacons for every network key on the device, regardless
 * of beacon configuration state or how many beacons sent by other devices have
 * been observed.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_send_private_beacons();

/***************************************************************************//**
 *
 * Set the non-default advertisement parameters used in mesh communications.
 * Call this command before @ref sl_btmesh_node_init or @ref sl_btmesh_prov_init
 * for the settings to take effect.
 *
 * @param[in] adv_interval_min Minimum advertisement interval. Value is in units
 *   of 0.625 ms. Default value is 32 (20 ms). Must be equal to or greater than
 *   32.
 * @param[in] adv_interval_max Maximum advertisement interval. Value is in units
 *   of 0.625 ms. Must be equal to or greater than the minimum interval. Default
 *   value is 32 (20 ms). Must be equal to or less than 16384.
 * @param[in] adv_repeat_packets Number of times to repeat each packet on all
 *   selected advertisement channels. Range: 1-5. Default value is 1.
 * @param[in] adv_channel_map Advertisement channel selection bitmask. Range:
 *   0x1-0x7. Default value: 7 (all channels)
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_set_adv_params(uint16_t adv_interval_min,
                                          uint16_t adv_interval_max,
                                          uint8_t adv_repeat_packets,
                                          uint8_t adv_channel_map);

/***************************************************************************//**
 *
 * Set the non-default scanning parameters used in mesh communications. Call
 * this command before @ref sl_btmesh_node_init or @ref sl_btmesh_prov_init for
 * the settings to take effect.
 *
 * @param[in] scan_interval Scan interval. Value is in units of 0.625 ms. Range:
 *   0x0004 to 0x4000 (time range of 2.5 ms to 10.24 s). Default value is 160
 *   (100 ms).
 * @param[in] scan_window Scan window. Value is in units of 0.625 ms. Must be
 *   equal to or less than the scan interval.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_set_scan_params(uint16_t scan_interval,
                                           uint16_t scan_window);

/***************************************************************************//**
 *
 * Set advertiser to use ramdom or public address. New setting will take effect
 * with the next advertisement.
 *
 * @param[in] address_type Address type:
 *     - <b>0:</b> Reset random address usage. Public address will used for
 *       advertisements.
 *     - <b>1:</b> Set random address usage. New random address will be
 *       generated and taken into use.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_test_adv_use_random_address(uint8_t address_type);

/** @} */ // end addtogroup sl_btmesh_test

/**
 * @addtogroup sl_btmesh_lpn Bluetooth Mesh Low Power Node API
 * @{
 *
 * @brief Bluetooth Mesh Low Power Node API
 *
 * These commands and events are for low-power operation, available in nodes
 * which have the LPN feature.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_lpn_init_id                                        0x00230028
#define sl_btmesh_cmd_lpn_deinit_id                                      0x01230028
#define sl_btmesh_cmd_lpn_establish_friendship_id                        0x03230028
#define sl_btmesh_cmd_lpn_poll_id                                        0x04230028
#define sl_btmesh_cmd_lpn_terminate_friendship_id                        0x05230028
#define sl_btmesh_cmd_lpn_config_id                                      0x06230028
#define sl_btmesh_rsp_lpn_init_id                                        0x00230028
#define sl_btmesh_rsp_lpn_deinit_id                                      0x01230028
#define sl_btmesh_rsp_lpn_establish_friendship_id                        0x03230028
#define sl_btmesh_rsp_lpn_poll_id                                        0x04230028
#define sl_btmesh_rsp_lpn_terminate_friendship_id                        0x05230028
#define sl_btmesh_rsp_lpn_config_id                                      0x06230028

/**
 * @brief Key values to identify LPN configurations
 */
typedef enum
{
  sl_btmesh_lpn_queue_length    = 0x0, /**< (0x0) Minimum queue length that the
                                            friend must support. Choose an
                                            appropriate length based on the
                                            expected message frequency and LPN
                                            sleep period because messages that
                                            do not fit into the friend queue are
                                            dropped. Note that the given value
                                            is rounded up to the nearest power
                                            of 2. Range: 2..128 */
  sl_btmesh_lpn_poll_timeout    = 0x1, /**< (0x1) Poll timeout in milliseconds,
                                            which is the longest time that LPN
                                            sleeps in between querying its
                                            friend for queued messages. Long
                                            poll timeout allows the LPN to sleep
                                            for longer periods, at the expense
                                            of increased latency for receiving
                                            messages. Note that the given value
                                            is rounded up to the nearest 100 ms
                                            Range: 1 s to 95 h 59 min 59 s 900
                                            ms */
  sl_btmesh_lpn_receive_delay   = 0x2, /**< (0x2) Receive delay in milliseconds.
                                            Receive delay is the time between
                                            the LPN sending a request and
                                            listening for a response. Receive
                                            delay allows the friend node time to
                                            prepare the message and LPN to
                                            sleep. Range: 10 ms to 255 ms The
                                            default receive delay in 10 ms. */
  sl_btmesh_lpn_request_retries = 0x3, /**< (0x3) Request retry is the number of
                                            retry attempts to repeat e.g., the
                                            friend poll message if the friend
                                            update was not received by the LPN.
                                            Range is from 0 to 10, default is 3 */
  sl_btmesh_lpn_retry_interval  = 0x4, /**< (0x4) Time interval between retry
                                            attempts in milliseconds. Range is 0
                                            to 100 ms. */
  sl_btmesh_lpn_clock_accuracy  = 0x5  /**< (0x5) Clock accuracy in ppm, which
                                            will be taken into account when
                                            opening and closing the receive
                                            window, and determining the poll
                                            timeout. Should be used with care,
                                            because inaccurate clock can
                                            increase the receive window lenght
                                            to up to 2,5 times in some cases.
                                            Default value is 0. */
} sl_btmesh_lpn_settings_t;

/**
 * @addtogroup sl_btmesh_evt_lpn_friendship_established sl_btmesh_evt_lpn_friendship_established
 * @{
 * @brief Indicate that a friendship has been established, received as a
 * response to the @ref sl_btmesh_lpn_establish_friendship command.
 */

/** @brief Identifier of the friendship_established event */
#define sl_btmesh_evt_lpn_friendship_established_id                      0x002300a8

/***************************************************************************//**
 * @brief Data structure of the friendship_established event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_lpn_friendship_established_s
{
  uint16_t netkey_index;   /**< Network key index */
  uint16_t friend_address; /**< Friend node address */
});

typedef struct sl_btmesh_evt_lpn_friendship_established_s sl_btmesh_evt_lpn_friendship_established_t;

/** @} */ // end addtogroup sl_btmesh_evt_lpn_friendship_established

/**
 * @addtogroup sl_btmesh_evt_lpn_friendship_failed sl_btmesh_evt_lpn_friendship_failed
 * @{
 * @brief Indicate that the friendship establishment has failed, received as a
 * response to the @ref sl_btmesh_lpn_establish_friendship command.
 */

/** @brief Identifier of the friendship_failed event */
#define sl_btmesh_evt_lpn_friendship_failed_id                           0x012300a8

/***************************************************************************//**
 * @brief Data structure of the friendship_failed event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_lpn_friendship_failed_s
{
  uint16_t netkey_index; /**< Network key index */
  uint16_t reason;       /**< Reason for friendship establishment failure */
});

typedef struct sl_btmesh_evt_lpn_friendship_failed_s sl_btmesh_evt_lpn_friendship_failed_t;

/** @} */ // end addtogroup sl_btmesh_evt_lpn_friendship_failed

/**
 * @addtogroup sl_btmesh_evt_lpn_friendship_terminated sl_btmesh_evt_lpn_friendship_terminated
 * @{
 * @brief Indicate that a friendship that was successfully established has been
 * terminated, received as a response to the @ref
 * sl_btmesh_lpn_terminate_friendship command or when friend not answering to a
 * POLL send by @ref sl_btmesh_lpn_poll command
 */

/** @brief Identifier of the friendship_terminated event */
#define sl_btmesh_evt_lpn_friendship_terminated_id                       0x022300a8

/***************************************************************************//**
 * @brief Data structure of the friendship_terminated event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_lpn_friendship_terminated_s
{
  uint16_t netkey_index; /**< Network key index */
  uint16_t reason;       /**< Reason for friendship termination */
});

typedef struct sl_btmesh_evt_lpn_friendship_terminated_s sl_btmesh_evt_lpn_friendship_terminated_t;

/** @} */ // end addtogroup sl_btmesh_evt_lpn_friendship_terminated

/***************************************************************************//**
 *
 * Initialize the Low Power node (LPN) mode. The node needs to be provisioned
 * before calling this command. After the LPN mode is initialized, the node
 * can't operate in the network without a Friend node. To establish a friendship
 * with a nearby Friend node, use the @ref sl_btmesh_lpn_establish_friendship
 * command. Make this call before calling the other commands in this class.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lpn_init();

/***************************************************************************//**
 *
 * Deinitialize the LPN functionality. After calling this command, a possible
 * friendship with a Friend node is terminated and the node can operate in the
 * network independently. After calling this command, do not call any other
 * command in this class before the Low Power mode is @ref sl_btmesh_lpn_init
 * again.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lpn_deinit();

/***************************************************************************//**
 *
 * Establish a friendship. After a friendship has been established, the node can
 * start saving power.
 *
 * @param[in] netkey_index Network key index used in friendship request
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_lpn_friendship_established
 *   - @ref sl_btmesh_evt_lpn_friendship_failed
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lpn_establish_friendship(uint16_t netkey_index);

/***************************************************************************//**
 *
 * Poll the Friend node for stored messages and security updates. This command
 * may be used if the application is expecting to receive messages at a specific
 * time. However, it is not required for correct operation, because the
 * procedure will be performed autonomously before the poll timeout expires.
 *
 * @param[in] netkey_index Network key index used in poll
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lpn_poll(uint16_t netkey_index);

/***************************************************************************//**
 *
 * Terminate an already established friendship. @ref
 * sl_btmesh_evt_lpn_friendship_terminated event will be emitted when the
 * friendship termination has been completed.
 *
 * @param[in] netkey_index Network key index used in friendship request
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_lpn_friendship_terminated
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lpn_terminate_friendship(uint16_t netkey_index);

/***************************************************************************//**
 *
 * Configure the parameters for friendship establishment and LPN behavior.
 *
 * @param[in] setting_id Enum @ref sl_btmesh_lpn_settings_t. Identifies the LPN
 *   setting to be updated.
 * @param[in] value New value for the given setting
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lpn_config(uint8_t setting_id, uint32_t value);

/** @} */ // end addtogroup sl_btmesh_lpn

/**
 * @addtogroup sl_btmesh_friend Bluetooth Mesh Friend Node API
 * @{
 *
 * @brief Bluetooth Mesh Friend Node API
 *
 * These commands and events are for the Friend operation, available in nodes
 * which have the Friend feature.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_friend_init_id                                     0x00240028
#define sl_btmesh_cmd_friend_deinit_id                                   0x01240028
#define sl_btmesh_rsp_friend_init_id                                     0x00240028
#define sl_btmesh_rsp_friend_deinit_id                                   0x01240028

/**
 * @addtogroup sl_btmesh_evt_friend_friendship_established sl_btmesh_evt_friend_friendship_established
 * @{
 * @brief Indicate that a friendship has been established.
 */

/** @brief Identifier of the friendship_established event */
#define sl_btmesh_evt_friend_friendship_established_id                   0x002400a8

/***************************************************************************//**
 * @brief Data structure of the friendship_established event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_friend_friendship_established_s
{
  uint16_t netkey_index; /**< Index of the network key used in friendship */
  uint16_t lpn_address;  /**< LPN node address */
});

typedef struct sl_btmesh_evt_friend_friendship_established_s sl_btmesh_evt_friend_friendship_established_t;

/** @} */ // end addtogroup sl_btmesh_evt_friend_friendship_established

/**
 * @addtogroup sl_btmesh_evt_friend_friendship_terminated sl_btmesh_evt_friend_friendship_terminated
 * @{
 * @brief Indicate that a friendship that was successfully established has been
 * terminated.
 */

/** @brief Identifier of the friendship_terminated event */
#define sl_btmesh_evt_friend_friendship_terminated_id                    0x012400a8

/***************************************************************************//**
 * @brief Data structure of the friendship_terminated event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_friend_friendship_terminated_s
{
  uint16_t netkey_index; /**< Index of the network key used in friendship */
  uint16_t lpn_address;  /**< LPN node address */
  uint16_t reason;       /**< Reason for friendship termination */
});

typedef struct sl_btmesh_evt_friend_friendship_terminated_s sl_btmesh_evt_friend_friendship_terminated_t;

/** @} */ // end addtogroup sl_btmesh_evt_friend_friendship_terminated

/***************************************************************************//**
 *
 * Initialize the Friend mode. The node needs to be provisioned before calling
 * this command. After the Friend mode is initialized, it is ready to accept
 * friend requests from low-power nodes. This call has to be made before calling
 * the other commands in this class.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_friend_init();

/***************************************************************************//**
 *
 * Deinitialize the Friend functionality. After calling this command, a possible
 * friendship with a Low Power node is terminated and all friendships are
 * terminated. After calling this command, don't call other commands in this
 * class before the Friend mode is @ref sl_btmesh_friend_init again.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_friend_deinit();

/** @} */ // end addtogroup sl_btmesh_friend

/**
 * @addtogroup sl_btmesh_proxy_client Bluetooth Mesh GATT Proxy Client
 * @{
 *
 * @brief Bluetooth Mesh GATT Proxy Client
 *
 * Initialize the GATT Proxy client-side functionality. Mesh proxy commands are
 * in the mesh_proxy class. This class allows the linker to drop the GATT Proxy
 * client code if it is not needed. It is enough to initialize this BGAPI class.
 * It contains no commands or events.
 */

/* Command and Response IDs */

/** @} */ // end addtogroup sl_btmesh_proxy_client

/**
 * @addtogroup sl_btmesh_proxy_server Bluetooth Mesh GATT Proxy Server
 * @{
 *
 * @brief Bluetooth Mesh GATT Proxy Server
 *
 * Initialize the GATT Proxy server-side functionality. This class allows the
 * linker to drop the GATT Proxy server code if it is not needed. It is enough
 * to initialize this BGAPI class. It contains no commands or events.
 */

/* Command and Response IDs */

/** @} */ // end addtogroup sl_btmesh_proxy_server

/**
 * @addtogroup sl_btmesh_config_client Bluetooth Mesh Configuration Client Model
 * @{
 *
 * @brief Bluetooth Mesh Configuration Client Model
 *
 * Bluetooth mesh stack API for the Mesh Configuration Client
 *
 * Commands in this class configure nodes in the Mesh network, which includes
 * key management, publish and subscribe settings manipulation, and node feature
 * configuration.
 *
 * Requests to nodes are asynchronous. A handle is assigned to each request that
 * is pending a reply from a node in the network. The handle can be used to
 * query the request status, and to identify the response event from the node.
 * Multiple requests can be made in parallel (as long as they are destined to
 * different nodes; only one pending request per node is allowed).
 *
 * <b>Request Management</b>
 *   - @ref sl_btmesh_config_client_cancel_request : Cancel a request
 *   - @ref sl_btmesh_config_client_get_request_status : Query current status of
 *     a request
 *   - @ref sl_btmesh_config_client_get_default_timeout : Get default request
 *     timeout
 *   - @ref sl_btmesh_config_client_set_default_timeout : Set default request
 *     timeout
 *
 * <b>Key and Mesh Network Management</b>
 *   - @ref sl_btmesh_config_client_add_netkey : Deploy a network key to a node
 *   - @ref sl_btmesh_config_client_remove_netkey : Remove a network key from a
 *     node
 *   - @ref sl_btmesh_config_client_list_netkeys : List network keys on a node
 *   - @ref sl_btmesh_config_client_add_appkey : Deploy an application key to a
 *     node
 *   - @ref sl_btmesh_config_client_remove_appkey : Remove an application key
 *     from a node
 *   - @ref sl_btmesh_config_client_list_appkeys : List application keys bound
 *     to a network key on a node
 *
 * <b>Node Configuration</b>
 *   - @ref sl_btmesh_config_client_get_dcd : Get device composition data of a
 *     node
 *   - @ref sl_btmesh_config_client_reset_node : Reset a node
 *   - @ref sl_btmesh_config_client_get_default_ttl : Get node default TTL state
 *     value
 *   - @ref sl_btmesh_config_client_set_default_ttl : Set node default TTL state
 *     value
 *   - @ref sl_btmesh_config_client_get_beacon : Get node secure network beacon
 *     state value
 *   - @ref sl_btmesh_config_client_set_beacon : Set node secure network beacon
 *     state value
 *   - @ref sl_btmesh_config_client_get_identity : Get node identity advertising
 *     state value
 *   - @ref sl_btmesh_config_client_set_identity : Set node identity advertising
 *     state value
 *   - @ref sl_btmesh_config_client_get_friend : Get node friend state value
 *   - @ref sl_btmesh_config_client_set_friend : Set node friend state value
 *   - @ref sl_btmesh_config_client_get_lpn_polltimeout : Get node LPN poll
 *     timeout state value
 *   - @ref sl_btmesh_config_client_get_gatt_proxy : Get node GATT proxy state
 *     value
 *   - @ref sl_btmesh_config_client_set_gatt_proxy : Set node GATT proxy state
 *     value
 *   - @ref sl_btmesh_config_client_get_relay : Get node relay state value
 *   - @ref sl_btmesh_config_client_set_relay : Set node relay state value
 *   - @ref sl_btmesh_config_client_get_network_transmit : Get node network
 *     transmit state value
 *   - @ref sl_btmesh_config_client_set_network_transmit : Set node network
 *     transmit state value
 *
 * <b>Model Configuration</b>
 *   - @ref sl_btmesh_config_client_bind_model : Bind a model to an application
 *     key
 *   - @ref sl_btmesh_config_client_unbind_model : Remove a model to application
 *     key binding
 *   - @ref sl_btmesh_config_client_list_bindings : List model to application
 *     key bindings on a node
 *   - @ref sl_btmesh_config_client_add_model_sub : Add a subscription address
 *     to a model
 *   - @ref sl_btmesh_config_client_add_model_sub_va : Add a virtual
 *     subscription address to a model
 *   - @ref sl_btmesh_config_client_remove_model_sub : Remove a subscription
 *     address from a model
 *   - @ref sl_btmesh_config_client_remove_model_sub_va : Remove a virtual
 *     subscription address from a model
 *   - @ref sl_btmesh_config_client_set_model_sub : Overwrite the subscription
 *     list of a model with an address
 *   - @ref sl_btmesh_config_client_set_model_sub_va : Overwrite the
 *     subscription list of a model with a virtual address
 *   - @ref sl_btmesh_config_client_clear_model_sub : Clear the subscription
 *     list of a model
 *   - @ref sl_btmesh_config_client_list_subs : Get the subscription list of a
 *     model
 *   - @ref sl_btmesh_config_client_get_model_pub : Get a model's publication
 *     parameters
 *   - @ref sl_btmesh_config_client_set_model_pub : Set a model's publication
 *     parameters
 *
 * <b>Heartbeat</b>
 *   - @ref sl_btmesh_config_client_get_heartbeat_pub : Get node heartbeat
 *     publication settings
 *   - @ref sl_btmesh_config_client_set_heartbeat_pub : Set node heartbeat
 *     publication settings
 *   - @ref sl_btmesh_config_client_get_heartbeat_sub : Get node heartbeat
 *     subscription settings
 *   - @ref sl_btmesh_config_client_set_heartbeat_sub : Set node heartbeat
 *     subscription settings
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_config_client_cancel_request_id                    0x00270028
#define sl_btmesh_cmd_config_client_get_request_status_id                0x01270028
#define sl_btmesh_cmd_config_client_get_default_timeout_id               0x2e270028
#define sl_btmesh_cmd_config_client_set_default_timeout_id               0x2f270028
#define sl_btmesh_cmd_config_client_add_netkey_id                        0x02270028
#define sl_btmesh_cmd_config_client_remove_netkey_id                     0x03270028
#define sl_btmesh_cmd_config_client_list_netkeys_id                      0x04270028
#define sl_btmesh_cmd_config_client_add_appkey_id                        0x05270028
#define sl_btmesh_cmd_config_client_remove_appkey_id                     0x06270028
#define sl_btmesh_cmd_config_client_list_appkeys_id                      0x07270028
#define sl_btmesh_cmd_config_client_bind_model_id                        0x08270028
#define sl_btmesh_cmd_config_client_unbind_model_id                      0x09270028
#define sl_btmesh_cmd_config_client_list_bindings_id                     0x0a270028
#define sl_btmesh_cmd_config_client_get_model_pub_id                     0x0b270028
#define sl_btmesh_cmd_config_client_set_model_pub_id                     0x0c270028
#define sl_btmesh_cmd_config_client_set_model_pub_va_id                  0x0d270028
#define sl_btmesh_cmd_config_client_add_model_sub_id                     0x0e270028
#define sl_btmesh_cmd_config_client_add_model_sub_va_id                  0x0f270028
#define sl_btmesh_cmd_config_client_remove_model_sub_id                  0x10270028
#define sl_btmesh_cmd_config_client_remove_model_sub_va_id               0x11270028
#define sl_btmesh_cmd_config_client_set_model_sub_id                     0x12270028
#define sl_btmesh_cmd_config_client_set_model_sub_va_id                  0x13270028
#define sl_btmesh_cmd_config_client_clear_model_sub_id                   0x14270028
#define sl_btmesh_cmd_config_client_list_subs_id                         0x15270028
#define sl_btmesh_cmd_config_client_get_heartbeat_pub_id                 0x16270028
#define sl_btmesh_cmd_config_client_set_heartbeat_pub_id                 0x17270028
#define sl_btmesh_cmd_config_client_get_heartbeat_sub_id                 0x19270028
#define sl_btmesh_cmd_config_client_set_heartbeat_sub_id                 0x1a270028
#define sl_btmesh_cmd_config_client_get_beacon_id                        0x1b270028
#define sl_btmesh_cmd_config_client_set_beacon_id                        0x1c270028
#define sl_btmesh_cmd_config_client_get_default_ttl_id                   0x1d270028
#define sl_btmesh_cmd_config_client_set_default_ttl_id                   0x1e270028
#define sl_btmesh_cmd_config_client_get_gatt_proxy_id                    0x1f270028
#define sl_btmesh_cmd_config_client_set_gatt_proxy_id                    0x20270028
#define sl_btmesh_cmd_config_client_get_relay_id                         0x21270028
#define sl_btmesh_cmd_config_client_set_relay_id                         0x22270028
#define sl_btmesh_cmd_config_client_get_network_transmit_id              0x23270028
#define sl_btmesh_cmd_config_client_set_network_transmit_id              0x24270028
#define sl_btmesh_cmd_config_client_get_identity_id                      0x25270028
#define sl_btmesh_cmd_config_client_set_identity_id                      0x26270028
#define sl_btmesh_cmd_config_client_get_friend_id                        0x27270028
#define sl_btmesh_cmd_config_client_set_friend_id                        0x28270028
#define sl_btmesh_cmd_config_client_get_key_refresh_phase_id             0x29270028
#define sl_btmesh_cmd_config_client_get_lpn_polltimeout_id               0x2b270028
#define sl_btmesh_cmd_config_client_get_dcd_id                           0x2c270028
#define sl_btmesh_cmd_config_client_reset_node_id                        0x2d270028
#define sl_btmesh_cmd_config_client_set_request_timeout_for_node_id      0x30270028
#define sl_btmesh_rsp_config_client_cancel_request_id                    0x00270028
#define sl_btmesh_rsp_config_client_get_request_status_id                0x01270028
#define sl_btmesh_rsp_config_client_get_default_timeout_id               0x2e270028
#define sl_btmesh_rsp_config_client_set_default_timeout_id               0x2f270028
#define sl_btmesh_rsp_config_client_add_netkey_id                        0x02270028
#define sl_btmesh_rsp_config_client_remove_netkey_id                     0x03270028
#define sl_btmesh_rsp_config_client_list_netkeys_id                      0x04270028
#define sl_btmesh_rsp_config_client_add_appkey_id                        0x05270028
#define sl_btmesh_rsp_config_client_remove_appkey_id                     0x06270028
#define sl_btmesh_rsp_config_client_list_appkeys_id                      0x07270028
#define sl_btmesh_rsp_config_client_bind_model_id                        0x08270028
#define sl_btmesh_rsp_config_client_unbind_model_id                      0x09270028
#define sl_btmesh_rsp_config_client_list_bindings_id                     0x0a270028
#define sl_btmesh_rsp_config_client_get_model_pub_id                     0x0b270028
#define sl_btmesh_rsp_config_client_set_model_pub_id                     0x0c270028
#define sl_btmesh_rsp_config_client_set_model_pub_va_id                  0x0d270028
#define sl_btmesh_rsp_config_client_add_model_sub_id                     0x0e270028
#define sl_btmesh_rsp_config_client_add_model_sub_va_id                  0x0f270028
#define sl_btmesh_rsp_config_client_remove_model_sub_id                  0x10270028
#define sl_btmesh_rsp_config_client_remove_model_sub_va_id               0x11270028
#define sl_btmesh_rsp_config_client_set_model_sub_id                     0x12270028
#define sl_btmesh_rsp_config_client_set_model_sub_va_id                  0x13270028
#define sl_btmesh_rsp_config_client_clear_model_sub_id                   0x14270028
#define sl_btmesh_rsp_config_client_list_subs_id                         0x15270028
#define sl_btmesh_rsp_config_client_get_heartbeat_pub_id                 0x16270028
#define sl_btmesh_rsp_config_client_set_heartbeat_pub_id                 0x17270028
#define sl_btmesh_rsp_config_client_get_heartbeat_sub_id                 0x19270028
#define sl_btmesh_rsp_config_client_set_heartbeat_sub_id                 0x1a270028
#define sl_btmesh_rsp_config_client_get_beacon_id                        0x1b270028
#define sl_btmesh_rsp_config_client_set_beacon_id                        0x1c270028
#define sl_btmesh_rsp_config_client_get_default_ttl_id                   0x1d270028
#define sl_btmesh_rsp_config_client_set_default_ttl_id                   0x1e270028
#define sl_btmesh_rsp_config_client_get_gatt_proxy_id                    0x1f270028
#define sl_btmesh_rsp_config_client_set_gatt_proxy_id                    0x20270028
#define sl_btmesh_rsp_config_client_get_relay_id                         0x21270028
#define sl_btmesh_rsp_config_client_set_relay_id                         0x22270028
#define sl_btmesh_rsp_config_client_get_network_transmit_id              0x23270028
#define sl_btmesh_rsp_config_client_set_network_transmit_id              0x24270028
#define sl_btmesh_rsp_config_client_get_identity_id                      0x25270028
#define sl_btmesh_rsp_config_client_set_identity_id                      0x26270028
#define sl_btmesh_rsp_config_client_get_friend_id                        0x27270028
#define sl_btmesh_rsp_config_client_set_friend_id                        0x28270028
#define sl_btmesh_rsp_config_client_get_key_refresh_phase_id             0x29270028
#define sl_btmesh_rsp_config_client_get_lpn_polltimeout_id               0x2b270028
#define sl_btmesh_rsp_config_client_get_dcd_id                           0x2c270028
#define sl_btmesh_rsp_config_client_reset_node_id                        0x2d270028
#define sl_btmesh_rsp_config_client_set_request_timeout_for_node_id      0x30270028

/**
 * @addtogroup sl_btmesh_evt_config_client_request_modified sl_btmesh_evt_config_client_request_modified
 * @{
 * @brief Pending request parameters have been updated
 *
 * The application may call @ref sl_btmesh_config_client_get_request_status to
 * retrieve the current status of the request. This event is generated when the
 * timeout of a request is extended because the request is acknowledged by a
 * Friend node on behalf of the LPN, which is the destination of the request.
 */

/** @brief Identifier of the request_modified event */
#define sl_btmesh_evt_config_client_request_modified_id                  0x002700a8

/***************************************************************************//**
 * @brief Data structure of the request_modified event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_request_modified_s
{
  uint32_t handle; /**< Request handle */
});

typedef struct sl_btmesh_evt_config_client_request_modified_s sl_btmesh_evt_config_client_request_modified_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_request_modified

/**
 * @addtogroup sl_btmesh_evt_config_client_netkey_status sl_btmesh_evt_config_client_netkey_status
 * @{
 * @brief This event is created when a response for an @ref
 * sl_btmesh_config_client_add_netkey or a @ref
 * sl_btmesh_config_client_remove_netkey request is received, or the request
 * times out.
 */

/** @brief Identifier of the netkey_status event */
#define sl_btmesh_evt_config_client_netkey_status_id                     0x012700a8

/***************************************************************************//**
 * @brief Data structure of the netkey_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_netkey_status_s
{
  uint16_t result; /**< SL_STATUS_OK if successful. Error code otherwise. */
  uint32_t handle; /**< Request handle */
});

typedef struct sl_btmesh_evt_config_client_netkey_status_s sl_btmesh_evt_config_client_netkey_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_netkey_status

/**
 * @addtogroup sl_btmesh_evt_config_client_netkey_list sl_btmesh_evt_config_client_netkey_list
 * @{
 * @brief This event contains a list of key indices for network keys that are
 * present on a node
 *
 * The list is requested using the @ref sl_btmesh_config_client_list_netkeys
 * command. More than one event may be generated. List contents are terminated
 * by a @ref sl_btmesh_evt_config_client_netkey_list_end event.
 */

/** @brief Identifier of the netkey_list event */
#define sl_btmesh_evt_config_client_netkey_list_id                       0x022700a8

/***************************************************************************//**
 * @brief Data structure of the netkey_list event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_netkey_list_s
{
  uint32_t   handle;         /**< Request handle */
  uint8array netkey_indices; /**< List of network key indices, two bytes per
                                  entry */
});

typedef struct sl_btmesh_evt_config_client_netkey_list_s sl_btmesh_evt_config_client_netkey_list_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_netkey_list

/**
 * @addtogroup sl_btmesh_evt_config_client_netkey_list_end sl_btmesh_evt_config_client_netkey_list_end
 * @{
 * @brief Terminating event for network key index list
 */

/** @brief Identifier of the netkey_list_end event */
#define sl_btmesh_evt_config_client_netkey_list_end_id                   0x032700a8

/***************************************************************************//**
 * @brief Data structure of the netkey_list_end event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_netkey_list_end_s
{
  uint16_t result; /**< SL_STATUS_OK if successful. Error code otherwise. */
  uint32_t handle; /**< Request handle */
});

typedef struct sl_btmesh_evt_config_client_netkey_list_end_s sl_btmesh_evt_config_client_netkey_list_end_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_netkey_list_end

/**
 * @addtogroup sl_btmesh_evt_config_client_appkey_status sl_btmesh_evt_config_client_appkey_status
 * @{
 * @brief This event is created when a response for an @ref
 * sl_btmesh_config_client_add_appkey or a @ref
 * sl_btmesh_config_client_remove_appkey request is received or the request
 * times out.
 */

/** @brief Identifier of the appkey_status event */
#define sl_btmesh_evt_config_client_appkey_status_id                     0x042700a8

/***************************************************************************//**
 * @brief Data structure of the appkey_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_appkey_status_s
{
  uint16_t result; /**< SL_STATUS_OK if successful. Error code otherwise. */
  uint32_t handle; /**< Request handle */
});

typedef struct sl_btmesh_evt_config_client_appkey_status_s sl_btmesh_evt_config_client_appkey_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_appkey_status

/**
 * @addtogroup sl_btmesh_evt_config_client_appkey_list sl_btmesh_evt_config_client_appkey_list
 * @{
 * @brief This event contains a list of key indices for application keys that
 * are present on a node and are bound to the network key specified in the
 * request
 *
 * The list is requested using the @ref sl_btmesh_config_client_list_appkeys
 * command. More than one event may be generated. List contents are terminated
 * by a @ref sl_btmesh_evt_config_client_appkey_list_end event.
 */

/** @brief Identifier of the appkey_list event */
#define sl_btmesh_evt_config_client_appkey_list_id                       0x052700a8

/***************************************************************************//**
 * @brief Data structure of the appkey_list event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_appkey_list_s
{
  uint32_t   handle;         /**< Request handle */
  uint8array appkey_indices; /**< List of application key indices, two bytes per
                                  entry */
});

typedef struct sl_btmesh_evt_config_client_appkey_list_s sl_btmesh_evt_config_client_appkey_list_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_appkey_list

/**
 * @addtogroup sl_btmesh_evt_config_client_appkey_list_end sl_btmesh_evt_config_client_appkey_list_end
 * @{
 * @brief Terminating event for application key index list
 */

/** @brief Identifier of the appkey_list_end event */
#define sl_btmesh_evt_config_client_appkey_list_end_id                   0x062700a8

/***************************************************************************//**
 * @brief Data structure of the appkey_list_end event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_appkey_list_end_s
{
  uint16_t result; /**< SL_STATUS_OK if successful. Error code otherwise. */
  uint32_t handle; /**< Request handle */
});

typedef struct sl_btmesh_evt_config_client_appkey_list_end_s sl_btmesh_evt_config_client_appkey_list_end_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_appkey_list_end

/**
 * @addtogroup sl_btmesh_evt_config_client_binding_status sl_btmesh_evt_config_client_binding_status
 * @{
 * @brief Status event for @ref sl_btmesh_config_client_bind_model and @ref
 * sl_btmesh_config_client_unbind_model application keys and models.
 */

/** @brief Identifier of the binding_status event */
#define sl_btmesh_evt_config_client_binding_status_id                    0x072700a8

/***************************************************************************//**
 * @brief Data structure of the binding_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_binding_status_s
{
  uint16_t result; /**< SL_STATUS_OK if successful. Error code otherwise. */
  uint32_t handle; /**< Request handle */
});

typedef struct sl_btmesh_evt_config_client_binding_status_s sl_btmesh_evt_config_client_binding_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_binding_status

/**
 * @addtogroup sl_btmesh_evt_config_client_bindings_list sl_btmesh_evt_config_client_bindings_list
 * @{
 * @brief This event contains a list of key indices for the application keys
 * which are bound to a model
 *
 * The list is requested using the @ref sl_btmesh_config_client_list_bindings
 * command. More than one such event may be generated; the list contents are
 * terminated by a @ref sl_btmesh_evt_config_client_bindings_list_end event.
 */

/** @brief Identifier of the bindings_list event */
#define sl_btmesh_evt_config_client_bindings_list_id                     0x082700a8

/***************************************************************************//**
 * @brief Data structure of the bindings_list event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_bindings_list_s
{
  uint32_t   handle;         /**< Request handle */
  uint8array appkey_indices; /**< List of application key indices, two bytes per
                                  entry */
});

typedef struct sl_btmesh_evt_config_client_bindings_list_s sl_btmesh_evt_config_client_bindings_list_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_bindings_list

/**
 * @addtogroup sl_btmesh_evt_config_client_bindings_list_end sl_btmesh_evt_config_client_bindings_list_end
 * @{
 * @brief Terminating event for model-application key bindings list
 */

/** @brief Identifier of the bindings_list_end event */
#define sl_btmesh_evt_config_client_bindings_list_end_id                 0x092700a8

/***************************************************************************//**
 * @brief Data structure of the bindings_list_end event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_bindings_list_end_s
{
  uint16_t result; /**< SL_STATUS_OK if successful. Error code otherwise. */
  uint32_t handle; /**< Request handle */
});

typedef struct sl_btmesh_evt_config_client_bindings_list_end_s sl_btmesh_evt_config_client_bindings_list_end_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_bindings_list_end

/**
 * @addtogroup sl_btmesh_evt_config_client_model_pub_status sl_btmesh_evt_config_client_model_pub_status
 * @{
 * @brief Status event for @ref sl_btmesh_config_client_get_model_pub, @ref
 * sl_btmesh_config_client_set_model_pub, @ref
 * sl_btmesh_config_client_set_model_pub_va commands.
 */

/** @brief Identifier of the model_pub_status event */
#define sl_btmesh_evt_config_client_model_pub_status_id                  0x0a2700a8

/***************************************************************************//**
 * @brief Data structure of the model_pub_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_model_pub_status_s
{
  uint16_t result;                 /**< SL_STATUS_OK if successful. Error code
                                        otherwise. */
  uint32_t handle;                 /**< Request handle */
  uint16_t address;                /**< The address to publish to. If this
                                        address is the unassigned address, the
                                        model is prevented from publishing. Note
                                        that, if state contains a Label UUID
                                        (full virtual address), the
                                        corresponding virtual address hash is
                                        returned in this parameter. */
  uint16_t appkey_index;           /**< The application key index used for the
                                        published messages */
  uint8_t  credentials;            /**< Friendship credentials flag */
  uint8_t  ttl;                    /**< Publication time-to-live value */
  uint32_t period_ms;              /**< Publication period in milliseconds */
  uint8_t  retransmit_count;       /**< Publication retransmission count */
  uint16_t retransmit_interval_ms; /**< Publication retransmission interval in
                                        milliseconds */
});

typedef struct sl_btmesh_evt_config_client_model_pub_status_s sl_btmesh_evt_config_client_model_pub_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_model_pub_status

/**
 * @addtogroup sl_btmesh_evt_config_client_model_sub_status sl_btmesh_evt_config_client_model_sub_status
 * @{
 * @brief Status event for @ref sl_btmesh_config_client_add_model_sub : add
 * subscription address, @ref sl_btmesh_config_client_add_model_sub_va : add
 * subscription virtual address, @ref sl_btmesh_config_client_remove_model_sub :
 * remove subscription address, @ref sl_btmesh_config_client_remove_model_sub_va
 * : remove subscription virtual address, @ref
 * sl_btmesh_config_client_set_model_sub : set subscription address, @ref
 * sl_btmesh_config_client_set_model_sub_va : set subscription virtual address,
 * and @ref sl_btmesh_config_client_clear_model_sub : clear subscription address
 * list commands
 */

/** @brief Identifier of the model_sub_status event */
#define sl_btmesh_evt_config_client_model_sub_status_id                  0x0b2700a8

/***************************************************************************//**
 * @brief Data structure of the model_sub_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_model_sub_status_s
{
  uint16_t result; /**< SL_STATUS_OK if successful. Error code otherwise. */
  uint32_t handle; /**< Request handle */
});

typedef struct sl_btmesh_evt_config_client_model_sub_status_s sl_btmesh_evt_config_client_model_sub_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_model_sub_status

/**
 * @addtogroup sl_btmesh_evt_config_client_subs_list sl_btmesh_evt_config_client_subs_list
 * @{
 * @brief This event contains a list of addresses that the queried model
 * subscribes to
 *
 * The list is requested using the @ref sl_btmesh_config_client_list_subs
 * command. More than one event may be generated. List contents are terminated
 * by a @ref sl_btmesh_evt_config_client_subs_list_end event. Note that, if the
 * subscription address list entry is a Label UUID (full virtual address), the
 * corresponding virtual address hash is returned in this event.
 */

/** @brief Identifier of the subs_list event */
#define sl_btmesh_evt_config_client_subs_list_id                         0x0c2700a8

/***************************************************************************//**
 * @brief Data structure of the subs_list event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_subs_list_s
{
  uint32_t   handle;    /**< Request handle */
  uint8array addresses; /**< List of subscription addresses, two bytes per entry */
});

typedef struct sl_btmesh_evt_config_client_subs_list_s sl_btmesh_evt_config_client_subs_list_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_subs_list

/**
 * @addtogroup sl_btmesh_evt_config_client_subs_list_end sl_btmesh_evt_config_client_subs_list_end
 * @{
 * @brief Terminating event for model subscription list
 */

/** @brief Identifier of the subs_list_end event */
#define sl_btmesh_evt_config_client_subs_list_end_id                     0x0d2700a8

/***************************************************************************//**
 * @brief Data structure of the subs_list_end event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_subs_list_end_s
{
  uint16_t result; /**< SL_STATUS_OK if successful. Error code otherwise. */
  uint32_t handle; /**< Request handle */
});

typedef struct sl_btmesh_evt_config_client_subs_list_end_s sl_btmesh_evt_config_client_subs_list_end_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_subs_list_end

/**
 * @addtogroup sl_btmesh_evt_config_client_heartbeat_pub_status sl_btmesh_evt_config_client_heartbeat_pub_status
 * @{
 * @brief Status event for @ref sl_btmesh_config_client_get_heartbeat_pub and
 * @ref sl_btmesh_config_client_set_heartbeat_pub commands
 */

/** @brief Identifier of the heartbeat_pub_status event */
#define sl_btmesh_evt_config_client_heartbeat_pub_status_id              0x0e2700a8

/***************************************************************************//**
 * @brief Data structure of the heartbeat_pub_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_heartbeat_pub_status_s
{
  uint16_t result;              /**< SL_STATUS_OK if successful. Error code
                                     otherwise. */
  uint32_t handle;              /**< Request handle */
  uint16_t destination_address; /**< Heartbeat publication destination address. */
  uint16_t netkey_index;        /**< Index of the network key used to encrypt
                                     heartbeat messages */
  uint8_t  count_log;           /**< Heartbeat publication count logarithm-of-2
                                     setting */
  uint8_t  period_log;          /**< Heartbeat publication period logarithm-of-2
                                     setting */
  uint8_t  ttl;                 /**< Time-to-live value for heartbeat messages */
  uint16_t features;            /**< Heartbeat trigger setting */
});

typedef struct sl_btmesh_evt_config_client_heartbeat_pub_status_s sl_btmesh_evt_config_client_heartbeat_pub_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_heartbeat_pub_status

/**
 * @addtogroup sl_btmesh_evt_config_client_heartbeat_sub_status sl_btmesh_evt_config_client_heartbeat_sub_status
 * @{
 * @brief Status event for @ref sl_btmesh_config_client_get_heartbeat_sub and
 * @ref sl_btmesh_config_client_set_heartbeat_sub commands
 */

/** @brief Identifier of the heartbeat_sub_status event */
#define sl_btmesh_evt_config_client_heartbeat_sub_status_id              0x0f2700a8

/***************************************************************************//**
 * @brief Data structure of the heartbeat_sub_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_heartbeat_sub_status_s
{
  uint16_t result;              /**< SL_STATUS_OK if successful. Error code
                                     otherwise. */
  uint32_t handle;              /**< Request handle */
  uint16_t source_address;      /**< Source address for heartbeat messages */
  uint16_t destination_address; /**< Destination address for heartbeat messages */
  uint8_t  period_log;          /**< Heartbeat subscription remaining period
                                     logarithm-of-2 value */
  uint8_t  count_log;           /**< Received heartbeat message count
                                     logarithm-of-2 value */
  uint8_t  min_hops;            /**< Minimum hop value seen in received
                                     heartbeat messages */
  uint8_t  max_hops;            /**< Minimum hop value seen in received
                                     heartbeat messages */
});

typedef struct sl_btmesh_evt_config_client_heartbeat_sub_status_s sl_btmesh_evt_config_client_heartbeat_sub_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_heartbeat_sub_status

/**
 * @addtogroup sl_btmesh_evt_config_client_beacon_status sl_btmesh_evt_config_client_beacon_status
 * @{
 * @brief Status event for @ref sl_btmesh_config_client_get_beacon and @ref
 * sl_btmesh_config_client_set_beacon commands.
 */

/** @brief Identifier of the beacon_status event */
#define sl_btmesh_evt_config_client_beacon_status_id                     0x102700a8

/***************************************************************************//**
 * @brief Data structure of the beacon_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_beacon_status_s
{
  uint16_t result; /**< SL_STATUS_OK if successful. Error code otherwise. */
  uint32_t handle; /**< Request handle */
  uint8_t  value;  /**< Secure network beacon state of the node. Valid values
                        are:
                          - 0: Node is not broadcasting secure network beacons
                          - 1: Node is broadcasting secure network beacons */
});

typedef struct sl_btmesh_evt_config_client_beacon_status_s sl_btmesh_evt_config_client_beacon_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_beacon_status

/**
 * @addtogroup sl_btmesh_evt_config_client_default_ttl_status sl_btmesh_evt_config_client_default_ttl_status
 * @{
 * @brief Status event for @ref sl_btmesh_config_client_get_default_ttl and @ref
 * sl_btmesh_config_client_set_default_ttl commands.
 */

/** @brief Identifier of the default_ttl_status event */
#define sl_btmesh_evt_config_client_default_ttl_status_id                0x112700a8

/***************************************************************************//**
 * @brief Data structure of the default_ttl_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_default_ttl_status_s
{
  uint16_t result; /**< SL_STATUS_OK if successful. Error code otherwise. */
  uint32_t handle; /**< Request handle */
  uint8_t  value;  /**< Default TTL value. Valid value range is from 2 to 127
                        for relayed PDUs, and 0 to indicate non-relayed PDUs. */
});

typedef struct sl_btmesh_evt_config_client_default_ttl_status_s sl_btmesh_evt_config_client_default_ttl_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_default_ttl_status

/**
 * @addtogroup sl_btmesh_evt_config_client_gatt_proxy_status sl_btmesh_evt_config_client_gatt_proxy_status
 * @{
 * @brief Status event for @ref sl_btmesh_config_client_get_gatt_proxy and @ref
 * sl_btmesh_config_client_set_gatt_proxy commands
 */

/** @brief Identifier of the gatt_proxy_status event */
#define sl_btmesh_evt_config_client_gatt_proxy_status_id                 0x122700a8

/***************************************************************************//**
 * @brief Data structure of the gatt_proxy_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_gatt_proxy_status_s
{
  uint16_t result; /**< SL_STATUS_OK if successful. Error code otherwise. */
  uint32_t handle; /**< Request handle */
  uint8_t  value;  /**< GATT proxy value of the node. Valid values are:
                          - 0: GATT proxy feature is disabled
                          - 1: GATT proxy feature is enabled
                          - 2: GATT proxy feature is not supported */
});

typedef struct sl_btmesh_evt_config_client_gatt_proxy_status_s sl_btmesh_evt_config_client_gatt_proxy_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_gatt_proxy_status

/**
 * @addtogroup sl_btmesh_evt_config_client_relay_status sl_btmesh_evt_config_client_relay_status
 * @{
 * @brief Status event for @ref sl_btmesh_config_client_get_relay and @ref
 * sl_btmesh_config_client_set_relay commands
 */

/** @brief Identifier of the relay_status event */
#define sl_btmesh_evt_config_client_relay_status_id                      0x132700a8

/***************************************************************************//**
 * @brief Data structure of the relay_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_relay_status_s
{
  uint16_t result;                 /**< SL_STATUS_OK if successful. Error code
                                        otherwise. */
  uint32_t handle;                 /**< Request handle */
  uint8_t  relay;                  /**< Relay state of the node. Valid values
                                        are as follows:
                                          - 0: Relaying disabled
                                          - 1: Relaying enabled
                                          - 2: Relaying not supported */
  uint8_t  retransmit_count;       /**< Relay retransmit count. Valid values
                                        range from 0 to 7; default value is 0
                                        (no retransmissions). */
  uint16_t retransmit_interval_ms; /**< Relay retransmit interval in
                                        milliseconds. Valid values range from 10
                                        ms to 320 ms, with a resolution of 10
                                        ms. The value will be zero if the
                                        retransmission count is zero. */
});

typedef struct sl_btmesh_evt_config_client_relay_status_s sl_btmesh_evt_config_client_relay_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_relay_status

/**
 * @addtogroup sl_btmesh_evt_config_client_network_transmit_status sl_btmesh_evt_config_client_network_transmit_status
 * @{
 * @brief Status event for @ref sl_btmesh_config_client_get_network_transmit and
 * @ref sl_btmesh_config_client_set_network_transmit commands
 */

/** @brief Identifier of the network_transmit_status event */
#define sl_btmesh_evt_config_client_network_transmit_status_id           0x142700a8

/***************************************************************************//**
 * @brief Data structure of the network_transmit_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_network_transmit_status_s
{
  uint16_t result;               /**< SL_STATUS_OK if successful. Error code
                                      otherwise. */
  uint32_t handle;               /**< Request handle */
  uint8_t  transmit_count;       /**< Network transmit count. Valid values range
                                      from 1 to 8; default value is 1 (single
                                      transmission; no retransmissions). */
  uint16_t transmit_interval_ms; /**< Network transmit interval in milliseconds.
                                      Valid values range from 10 ms to 320 ms,
                                      with a resolution of 10 ms. The value will
                                      be zero if the transmission count is set
                                      to one. */
});

typedef struct sl_btmesh_evt_config_client_network_transmit_status_s sl_btmesh_evt_config_client_network_transmit_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_network_transmit_status

/**
 * @addtogroup sl_btmesh_evt_config_client_identity_status sl_btmesh_evt_config_client_identity_status
 * @{
 * @brief Status event for @ref sl_btmesh_config_client_get_identity and @ref
 * sl_btmesh_config_client_set_identity commands.
 */

/** @brief Identifier of the identity_status event */
#define sl_btmesh_evt_config_client_identity_status_id                   0x152700a8

/***************************************************************************//**
 * @brief Data structure of the identity_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_identity_status_s
{
  uint16_t result; /**< SL_STATUS_OK if successful. Error code otherwise. */
  uint32_t handle; /**< Request handle */
  uint8_t  value;  /**< Identity state of the node for the used network index.
                        Valid values are as follows:
                          - 0: Node identity advertising is disabled
                          - 1: Node identity advertising is enabled
                          - 2: Node identity advertising is not supported */
});

typedef struct sl_btmesh_evt_config_client_identity_status_s sl_btmesh_evt_config_client_identity_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_identity_status

/**
 * @addtogroup sl_btmesh_evt_config_client_friend_status sl_btmesh_evt_config_client_friend_status
 * @{
 * @brief Status event for @ref sl_btmesh_config_client_get_friend and @ref
 * sl_btmesh_config_client_set_friend commands.
 */

/** @brief Identifier of the friend_status event */
#define sl_btmesh_evt_config_client_friend_status_id                     0x162700a8

/***************************************************************************//**
 * @brief Data structure of the friend_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_friend_status_s
{
  uint16_t result; /**< SL_STATUS_OK if successful. Error code otherwise. */
  uint32_t handle; /**< Request handle */
  uint8_t  value;  /**< Friend value to set. Valid values are:
                          - 0: Friend feature is not enabled
                          - 1: Friend feature is enabled
                          - 2: Friend feature is not supported */
});

typedef struct sl_btmesh_evt_config_client_friend_status_s sl_btmesh_evt_config_client_friend_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_friend_status

/**
 * @addtogroup sl_btmesh_evt_config_client_key_refresh_phase_status sl_btmesh_evt_config_client_key_refresh_phase_status
 * @{
 * @brief Status event for @ref sl_btmesh_config_client_get_key_refresh_phase
 * command.
 */

/** @brief Identifier of the key_refresh_phase_status event */
#define sl_btmesh_evt_config_client_key_refresh_phase_status_id          0x172700a8

/***************************************************************************//**
 * @brief Data structure of the key_refresh_phase_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_key_refresh_phase_status_s
{
  uint16_t result;       /**< SL_STATUS_OK if successful. Error code otherwise. */
  uint32_t handle;       /**< Request handle */
  uint16_t netkey_index; /**< Network key index for which the state applies */
  uint8_t  value;        /**< Key refresh phase state of the node for the given
                              network key. Valid values are:
                                - 0: Normal operation
                                - 1: First phase of key refresh procedure
                                - 2: Second phase of key refresh procedure */
});

typedef struct sl_btmesh_evt_config_client_key_refresh_phase_status_s sl_btmesh_evt_config_client_key_refresh_phase_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_key_refresh_phase_status

/**
 * @addtogroup sl_btmesh_evt_config_client_lpn_polltimeout_status sl_btmesh_evt_config_client_lpn_polltimeout_status
 * @{
 * @brief Status event for @ref sl_btmesh_config_client_get_lpn_polltimeout
 * command.
 */

/** @brief Identifier of the lpn_polltimeout_status event */
#define sl_btmesh_evt_config_client_lpn_polltimeout_status_id            0x182700a8

/***************************************************************************//**
 * @brief Data structure of the lpn_polltimeout_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_lpn_polltimeout_status_s
{
  uint16_t result;          /**< SL_STATUS_OK if successful. Error code
                                 otherwise. */
  uint32_t handle;          /**< Request handle */
  uint32_t poll_timeout_ms; /**< Poll timeout value, in milliseconds, for the
                                 specified LPN. The value reported is zero if
                                 the queried Friend does not have an ongoing
                                 friendship with the specified LPN. */
});

typedef struct sl_btmesh_evt_config_client_lpn_polltimeout_status_s sl_btmesh_evt_config_client_lpn_polltimeout_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_lpn_polltimeout_status

/**
 * @addtogroup sl_btmesh_evt_config_client_dcd_data sl_btmesh_evt_config_client_dcd_data
 * @{
 * @brief Event reporting queried composition data page contents
 *
 * The contents are requested using the @ref sl_btmesh_config_client_get_dcd
 * command. More than one event may be generated. Page contents are terminated
 * by a @ref sl_btmesh_evt_config_client_dcd_data_end event. Note that the
 * interpretation of the received data is page-specific. Page 0 contains the
 * element and model layout of the node.
 */

/** @brief Identifier of the dcd_data event */
#define sl_btmesh_evt_config_client_dcd_data_id                          0x192700a8

/***************************************************************************//**
 * @brief Data structure of the dcd_data event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_dcd_data_s
{
  uint32_t   handle; /**< Request handle */
  uint8_t    page;   /**< Composition data page containing data */
  uint8array data;   /**< Composition data page contents */
});

typedef struct sl_btmesh_evt_config_client_dcd_data_s sl_btmesh_evt_config_client_dcd_data_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_dcd_data

/**
 * @addtogroup sl_btmesh_evt_config_client_dcd_data_end sl_btmesh_evt_config_client_dcd_data_end
 * @{
 * @brief Terminating event for node composition data
 */

/** @brief Identifier of the dcd_data_end event */
#define sl_btmesh_evt_config_client_dcd_data_end_id                      0x1a2700a8

/***************************************************************************//**
 * @brief Data structure of the dcd_data_end event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_dcd_data_end_s
{
  uint16_t result; /**< SL_STATUS_OK if successful. Error code otherwise. */
  uint32_t handle; /**< Request handle */
});

typedef struct sl_btmesh_evt_config_client_dcd_data_end_s sl_btmesh_evt_config_client_dcd_data_end_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_dcd_data_end

/**
 * @addtogroup sl_btmesh_evt_config_client_reset_status sl_btmesh_evt_config_client_reset_status
 * @{
 * @brief Indicate a node has received a @ref
 * sl_btmesh_config_client_reset_node.
 */

/** @brief Identifier of the reset_status event */
#define sl_btmesh_evt_config_client_reset_status_id                      0x1b2700a8

/***************************************************************************//**
 * @brief Data structure of the reset_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_reset_status_s
{
  uint16_t result; /**< SL_STATUS_OK if successful. Error code otherwise. */
  uint32_t handle; /**< Request handle */
});

typedef struct sl_btmesh_evt_config_client_reset_status_s sl_btmesh_evt_config_client_reset_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_reset_status

/**
 * @addtogroup sl_btmesh_evt_config_client_obo_ack_received sl_btmesh_evt_config_client_obo_ack_received
 * @{
 * @brief Node received a transport layer Ack from a friend node , indicating
 * that the message has been placed in the Friend Queue.
 */

/** @brief Identifier of the obo_ack_received event */
#define sl_btmesh_evt_config_client_obo_ack_received_id                  0x1c2700a8

/***************************************************************************//**
 * @brief Data structure of the obo_ack_received event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_config_client_obo_ack_received_s
{
  uint16_t lpn_addr;    /**< Address of Friend node. */
  uint16_t friend_addr; /**< Address of Friend node. */
});

typedef struct sl_btmesh_evt_config_client_obo_ack_received_s sl_btmesh_evt_config_client_obo_ack_received_t;

/** @} */ // end addtogroup sl_btmesh_evt_config_client_obo_ack_received

/***************************************************************************//**
 *
 * Cancel an ongoing request releasing resources allocated at the Configuration
 * Client. Note that this call does no undo any setting a node may have made if
 * it had received the request already.
 *
 * @param[in] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_cancel_request(uint32_t handle);

/***************************************************************************//**
 *
 * Get the status of a pending request.
 *
 * @param[in] handle Request handle
 * @param[out] server_address Address of the Configuration Server
 * @param[out] opcode Message opcode used in the request. Opcodes are defined in
 *   the Bluetooth mesh stack 1.0 specification.
 * @param[out] age_ms Time in milliseconds that the request has been pending
 * @param[out] remaining_ms Time in milliseconds before the request times out.
 *   Note that timeout may be adjusted if it's determined that the request is
 *   destined to an LPN, which may respond very slowly.
 * @param[out] friend_acked If non-zero, response has been acknowledged by a
 *   Friend node, so it is destined to an LPN and may take a long time to
 *   complete.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_get_request_status(uint32_t handle,
                                                       uint16_t *server_address,
                                                       uint16_t *opcode,
                                                       uint32_t *age_ms,
                                                       uint32_t *remaining_ms,
                                                       uint8_t *friend_acked);

/***************************************************************************//**
 *
 * Get the default timeout for the configuration client requests. If there is no
 * response when the timeout expires, a configuration request is considered to
 * have failed and an event with an error result will be generated. Note that,
 * if the Bluetooth mesh stack notices the request is destined to an LPN by
 * receiving an on-behalf-of acknowledgment from a Friend node, the timeout in
 * use will be changed to the LPN default timeout.
 *
 * @param[out] timeout_ms Timeout in milliseconds. Default timeout is 5 s (5000
 *   ms).
 * @param[out] lpn_timeout_ms Timeout in milliseconds when communicating with an
 *   LPN node. Default LPN timeout is 120 s (120000 ms).
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_get_default_timeout(uint32_t *timeout_ms,
                                                        uint32_t *lpn_timeout_ms);

/***************************************************************************//**
 *
 * Set the default timeout for the configuration client requests.
 *
 * @param[in] timeout_ms Timeout in milliseconds. Default timeout is 5 s (5000
 *   ms).
 * @param[in] lpn_timeout_ms Timeout in milliseconds when communicating with an
 *   LPN node. Default LPN timeout is 120 s (120000 ms).
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_set_default_timeout(uint32_t timeout_ms,
                                                        uint32_t lpn_timeout_ms);

/***************************************************************************//**
 *
 * Add a network key to a node.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] netkey_index Index of the network key to add
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_netkey_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_add_netkey(uint16_t enc_netkey_index,
                                               uint16_t server_address,
                                               uint16_t netkey_index,
                                               uint32_t *handle);

/***************************************************************************//**
 *
 * Remove a network key from a node.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] netkey_index Index of the network key to remove
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_netkey_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_remove_netkey(uint16_t enc_netkey_index,
                                                  uint16_t server_address,
                                                  uint16_t netkey_index,
                                                  uint32_t *handle);

/***************************************************************************//**
 *
 * List the network keys on a node.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_netkey_list
 *   - @ref sl_btmesh_evt_config_client_netkey_list_end
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_list_netkeys(uint16_t enc_netkey_index,
                                                 uint16_t server_address,
                                                 uint32_t *handle);

/***************************************************************************//**
 *
 * Add an application key to a node.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] appkey_index Index of the application key to add
 * @param[in] netkey_index Index of the network key to bind the application key
 *   to on the node. Note that this may be different from the binding on other
 *   nodes or on the Configuration Client if desired.
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_appkey_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_add_appkey(uint16_t enc_netkey_index,
                                               uint16_t server_address,
                                               uint16_t appkey_index,
                                               uint16_t netkey_index,
                                               uint32_t *handle);

/***************************************************************************//**
 *
 * Remove an application key from a node.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] appkey_index Index of the application key to remove
 * @param[in] netkey_index Index of the network key bound to the application key
 *   to on the node. Note that this may be different from the binding on other
 *   nodes or on the Configuration Client.
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_appkey_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_remove_appkey(uint16_t enc_netkey_index,
                                                  uint16_t server_address,
                                                  uint16_t appkey_index,
                                                  uint16_t netkey_index,
                                                  uint32_t *handle);

/***************************************************************************//**
 *
 * List the application keys on a node.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] netkey_index Network key index for the key used as the query
 *   parameter. The result contains the indices of the application keys bound to
 *   this network key on the node.
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_appkey_list
 *   - @ref sl_btmesh_evt_config_client_appkey_list_end
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_list_appkeys(uint16_t enc_netkey_index,
                                                 uint16_t server_address,
                                                 uint16_t netkey_index,
                                                 uint32_t *handle);

/***************************************************************************//**
 *
 * Bind an application key to a model.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] elem_index Index of the element where the model to be configured
 *   resides on the node
 * @param[in] vendor_id Vendor ID for the model to configure. Use 0xFFFF for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID for the model to configure
 * @param[in] appkey_index Index of the application key to bind to the model
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_binding_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_bind_model(uint16_t enc_netkey_index,
                                               uint16_t server_address,
                                               uint8_t elem_index,
                                               uint16_t vendor_id,
                                               uint16_t model_id,
                                               uint16_t appkey_index,
                                               uint32_t *handle);

/***************************************************************************//**
 *
 * Unbind an application key from a model
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] elem_index Index of the element where the model to be configured
 *   resides on the node
 * @param[in] vendor_id Vendor ID for the model to configure. Use 0xFFFF for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID for the model to configure
 * @param[in] appkey_index Index of the application key to unbind from the model
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_binding_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_unbind_model(uint16_t enc_netkey_index,
                                                 uint16_t server_address,
                                                 uint8_t elem_index,
                                                 uint16_t vendor_id,
                                                 uint16_t model_id,
                                                 uint16_t appkey_index,
                                                 uint32_t *handle);

/***************************************************************************//**
 *
 * List application key bindings of a model.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] elem_index Index of the element where the model to be queried
 *   resides on the node
 * @param[in] vendor_id Vendor ID for the model to query. Use 0xFFFF for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID for the model to query
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_bindings_list
 *   - @ref sl_btmesh_evt_config_client_bindings_list_end
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_list_bindings(uint16_t enc_netkey_index,
                                                  uint16_t server_address,
                                                  uint8_t elem_index,
                                                  uint16_t vendor_id,
                                                  uint16_t model_id,
                                                  uint32_t *handle);

/***************************************************************************//**
 *
 * Get the model publication state.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] elem_index Index of the element where the model to be queried
 *   resides on the node
 * @param[in] vendor_id Vendor ID for the model to query. Use 0xFFFF for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID for the model to query
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_model_pub_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_get_model_pub(uint16_t enc_netkey_index,
                                                  uint16_t server_address,
                                                  uint8_t elem_index,
                                                  uint16_t vendor_id,
                                                  uint16_t model_id,
                                                  uint32_t *handle);

/***************************************************************************//**
 *
 * Set the model publication state.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] elem_index Index of the element where the model to be configured
 *   resides on the node
 * @param[in] vendor_id Vendor ID for the model to configure. Use 0xFFFF for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID for the model to configure
 * @param[in] address The address to publish to. It can be a unicast address, a
 *   virtual address, or a group address. It can also be the unassigned address
 *   to stop the model from publishing.
 * @param[in] appkey_index The application key index to use for the published
 *   messages.
 * @param[in] credentials Friendship credential flag. If zero, publication is
 *   done using normal credentials. If one, it is done with friendship
 *   credentials, meaning only the friend can decrypt the published message and
 *   relay it forward using the normal credentials. The default value is 0.
 * @param[in] ttl Publication time-to-live value. Valid values: range[0 - 127]
 *   and 255.
 * @param[in] period_ms Publication period in milliseconds. Note that the
 *   resolution of the publication period is limited by the specification to 100
 *   ms up to a period of 6.3 s, 1 s up to a period of 63 s, 10 s up to a period
 *   of 630 s, and 10 minutes above that. Maximum period allowed is 630 minutes.
 * @param[in] retransmit_count Publication retransmission count. Valid values
 *   range from 0 to 7.
 * @param[in] retransmit_interval_ms Publication retransmission interval in
 *   millisecond units. The range of value is 50 to 1600 ms, and the resolution
 *   of the value is 50 milliseconds.
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_model_pub_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_set_model_pub(uint16_t enc_netkey_index,
                                                  uint16_t server_address,
                                                  uint8_t elem_index,
                                                  uint16_t vendor_id,
                                                  uint16_t model_id,
                                                  uint16_t address,
                                                  uint16_t appkey_index,
                                                  uint8_t credentials,
                                                  uint8_t ttl,
                                                  uint32_t period_ms,
                                                  uint8_t retransmit_count,
                                                  uint16_t retransmit_interval_ms,
                                                  uint32_t *handle);

/***************************************************************************//**
 *
 * Set the model publication state with a full virtual publication address.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] elem_index Index of the element where the model to be configured
 *   resides on the node
 * @param[in] vendor_id Vendor ID for the model to configure. Use 0xFFFF for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID for the model to configure
 * @param[in] address The Label UUID (full virtual address) to publish to
 * @param[in] appkey_index The application key index to use for the published
 *   messages
 * @param[in] credentials Friendship credential flag. If zero, publication is
 *   done using normal credentials. If one, it is done with friendship
 *   credentials, meaning only the friend can decrypt the published message and
 *   relay it forward using the normal credentials. The default value is 0.
 * @param[in] ttl Publication time-to-live value. Valid values: range[0 - 127]
 *   and 255.
 * @param[in] period_ms Publication period in milliseconds. Note that the
 *   resolution of the publication period is limited by the specification to 100
 *   ms up to a period of 6.3 s, 1 s up to a period of 63 s, 10 s up to a period
 *   of 630 s, and 10 minutes above that. Maximum period allowed is 630 minutes.
 * @param[in] retransmit_count Publication retransmission count. Valid values
 *   range from 0 to 7.
 * @param[in] retransmit_interval_ms Publication retransmission interval in
 *   millisecond units. The range of value is 50 to 1600 ms. The resolution of
 *   the value is 50 milliseconds.
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_model_pub_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_set_model_pub_va(uint16_t enc_netkey_index,
                                                     uint16_t server_address,
                                                     uint8_t elem_index,
                                                     uint16_t vendor_id,
                                                     uint16_t model_id,
                                                     uuid_128 address,
                                                     uint16_t appkey_index,
                                                     uint8_t credentials,
                                                     uint8_t ttl,
                                                     uint32_t period_ms,
                                                     uint8_t retransmit_count,
                                                     uint16_t retransmit_interval_ms,
                                                     uint32_t *handle);

/***************************************************************************//**
 *
 * Add an address to the model subscription list.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] elem_index Index of the element where the model, which is about to
 *   be configured resides on the node
 * @param[in] vendor_id Vendor ID for the model to configure. Use 0xFFFF for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID for the model to configure
 * @param[in] sub_address The address to add to the subscription list. Note that
 *   the address has to be a group address.
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_model_sub_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_add_model_sub(uint16_t enc_netkey_index,
                                                  uint16_t server_address,
                                                  uint8_t elem_index,
                                                  uint16_t vendor_id,
                                                  uint16_t model_id,
                                                  uint16_t sub_address,
                                                  uint32_t *handle);

/***************************************************************************//**
 *
 * Add a Label UUID (full virtual address) to the model subscription list.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] elem_index Index of the element where the model, which is about to
 *   be configured resides on the node
 * @param[in] vendor_id Vendor ID for the model to configure. Use 0xFFFF for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID for the model to configure
 * @param[in] sub_address The full virtual address to add to the subscription
 *   list
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_model_sub_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_add_model_sub_va(uint16_t enc_netkey_index,
                                                     uint16_t server_address,
                                                     uint8_t elem_index,
                                                     uint16_t vendor_id,
                                                     uint16_t model_id,
                                                     uuid_128 sub_address,
                                                     uint32_t *handle);

/***************************************************************************//**
 *
 * Remove an address from the model subscription list.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] elem_index Index of the element where the model, which is about to
 *   be configured resides on the node
 * @param[in] vendor_id Vendor ID for the model to configure. Use 0xFFFF for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID for the model to configure
 * @param[in] sub_address The address to remove from the subscription list
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_model_sub_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_remove_model_sub(uint16_t enc_netkey_index,
                                                     uint16_t server_address,
                                                     uint8_t elem_index,
                                                     uint16_t vendor_id,
                                                     uint16_t model_id,
                                                     uint16_t sub_address,
                                                     uint32_t *handle);

/***************************************************************************//**
 *
 * Remove a Label UUID (full virtual address) from the model subscription list.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] elem_index Index of the element where the model to be configured
 *   resides on the node
 * @param[in] vendor_id Vendor ID for the model to configure. Use 0xFFFF for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID for the model to configure
 * @param[in] sub_address The full virtual address to remove from the
 *   subscription list
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_model_sub_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_remove_model_sub_va(uint16_t enc_netkey_index,
                                                        uint16_t server_address,
                                                        uint8_t elem_index,
                                                        uint16_t vendor_id,
                                                        uint16_t model_id,
                                                        uuid_128 sub_address,
                                                        uint32_t *handle);

/***************************************************************************//**
 *
 * Set (overwrite) model subscription address list to a single address.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] elem_index Index of the element where the model, which is about to
 *   be configured resides on the node
 * @param[in] vendor_id Vendor ID for the model to configure. Use 0xFFFF for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID for the model to configure
 * @param[in] sub_address The address to set as the subscription list
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_model_sub_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_set_model_sub(uint16_t enc_netkey_index,
                                                  uint16_t server_address,
                                                  uint8_t elem_index,
                                                  uint16_t vendor_id,
                                                  uint16_t model_id,
                                                  uint16_t sub_address,
                                                  uint32_t *handle);

/***************************************************************************//**
 *
 * Set (overwrite) model subscription address list to a single virtual address.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] elem_index Index of the element where the model, which is about to
 *   be configured resides on the node
 * @param[in] vendor_id Vendor ID for the model to configure. Use 0xFFFF for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID for the model to configure
 * @param[in] sub_address The full virtual address to set as the subscription
 *   list
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_model_sub_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_set_model_sub_va(uint16_t enc_netkey_index,
                                                     uint16_t server_address,
                                                     uint8_t elem_index,
                                                     uint16_t vendor_id,
                                                     uint16_t model_id,
                                                     uuid_128 sub_address,
                                                     uint32_t *handle);

/***************************************************************************//**
 *
 * Clear (empty) the model subscription address list.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] elem_index Index of the element where the model, which is about to
 *   be configured resides on the node
 * @param[in] vendor_id Vendor ID for the model to configure. Use 0xFFFF for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID for the model to configure
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_model_sub_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_clear_model_sub(uint16_t enc_netkey_index,
                                                    uint16_t server_address,
                                                    uint8_t elem_index,
                                                    uint16_t vendor_id,
                                                    uint16_t model_id,
                                                    uint32_t *handle);

/***************************************************************************//**
 *
 * Get the subscription address list of a model.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] elem_index Index of the element where the model, which is about to
 *   be queried resides on the node
 * @param[in] vendor_id Vendor ID for the model to query. Use 0xFFFF for
 *   Bluetooth SIG models.
 * @param[in] model_id Model ID for the model to query
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_subs_list
 *   - @ref sl_btmesh_evt_config_client_subs_list_end
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_list_subs(uint16_t enc_netkey_index,
                                              uint16_t server_address,
                                              uint8_t elem_index,
                                              uint16_t vendor_id,
                                              uint16_t model_id,
                                              uint32_t *handle);

/***************************************************************************//**
 *
 * Get the heartbeat publication state of a node.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_heartbeat_pub_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_get_heartbeat_pub(uint16_t enc_netkey_index,
                                                      uint16_t server_address,
                                                      uint32_t *handle);

/***************************************************************************//**
 *
 * Set the heartbeat publication state of a node.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] destination_address Heartbeat publication destination address. The
 *   address can't be a virtual address. Note that it can be the unassigned
 *   address, in which case the heartbeat publishing is disabled.
 * @param[in] netkey_index Index of the network key used to encrypt heartbeat
 *   messages
 * @param[in] count_log Heartbeat publication count logarithm-of-2 setting.
 *   Valid values are as follows:
 *     - <b>0x00:</b> Heartbeat messages are not sent
 *     - <b>0x01 .. 0x11:</b> Node will send 2^(n-1) heartbeat messages
 *     - <b>0x12 .. 0xfe:</b> Prohibited
 *     - <b>0xff:</b> Hearbeat messages are sent indefinitely
 * @param[in] period_log Heartbeat publication period logarithm-of-2 setting.
 *   Valid values are as follows:
 *     - <b>0x00:</b> Heartbeat messages are not sent
 *     - <b>0x01 .. 0x11:</b> Node will send a heartbeat message every 2^(n-1)
 *       seconds
 *     - <b>0x12 .. 0xff:</b> Prohibited
 * @param[in] ttl Time-to-live value for heartbeat messages
 * @param[in] features @parblock
 *   Heartbeat trigger setting. For bits set in the bitmask, reconfiguration of
 *   the node feature associated with the bit will result in the node emitting a
 *   heartbeat message. Valid values are as follows:
 *     - <b>Bit 0:</b> Relay feature
 *     - <b>Bit 1:</b> Proxy feature
 *     - <b>Bit 2:</b> Friend feature
 *     - <b>Bit 3:</b> Low power feature
 *
 *   Remaining bits are reserved for future use.
 *   @endparblock
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_heartbeat_pub_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_set_heartbeat_pub(uint16_t enc_netkey_index,
                                                      uint16_t server_address,
                                                      uint16_t destination_address,
                                                      uint16_t netkey_index,
                                                      uint8_t count_log,
                                                      uint8_t period_log,
                                                      uint8_t ttl,
                                                      uint16_t features,
                                                      uint32_t *handle);

/***************************************************************************//**
 *
 * Get the heartbeat subscription state of a node.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_heartbeat_sub_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_get_heartbeat_sub(uint16_t enc_netkey_index,
                                                      uint16_t server_address,
                                                      uint32_t *handle);

/***************************************************************************//**
 *
 * Set the heartbeat subscription state of a node.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] source_address Source address for heartbeat messages, which must
 *   be either a unicast address or the unassigned address, in which case
 *   heartbeat messages are not processed.
 * @param[in] destination_address Destination address for heartbeat messages.
 *   The address must be either the unicast address of the primary element of
 *   the node, a group address, or the unassigned address. If it is the
 *   unassigned address, heartbeat messages are not processed.
 * @param[in] period_log Heartbeat subscription period logarithm-of-2 setting.
 *   Valid values are as follows:
 *     - <b>0x00:</b> Heartbeat messages are not received
 *     - <b>0x01 .. 0x11:</b> Node will receive heartbeat messages for 2^(n-1)
 *       seconds
 *     - <b>0x12 .. 0xff:</b> Prohibited
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_heartbeat_sub_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_set_heartbeat_sub(uint16_t enc_netkey_index,
                                                      uint16_t server_address,
                                                      uint16_t source_address,
                                                      uint16_t destination_address,
                                                      uint8_t period_log,
                                                      uint32_t *handle);

/***************************************************************************//**
 *
 * Get node secure network beacon state.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_beacon_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_get_beacon(uint16_t enc_netkey_index,
                                               uint16_t server_address,
                                               uint32_t *handle);

/***************************************************************************//**
 *
 * Set node secure network beacon state.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] value Secure network beacon value to set. Valid values are:
 *     - 0: Node is not broadcasting secure network beacons
 *     - 1: Node is broadcasting secure network beacons
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_beacon_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_set_beacon(uint16_t enc_netkey_index,
                                               uint16_t server_address,
                                               uint8_t value,
                                               uint32_t *handle);

/***************************************************************************//**
 *
 * Get node default TTL state.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_default_ttl_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_get_default_ttl(uint16_t enc_netkey_index,
                                                    uint16_t server_address,
                                                    uint32_t *handle);

/***************************************************************************//**
 *
 * Set node default TTL state.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] value Default TTL value. Valid value range is from 2 to 127 for
 *   relayed PDUs, and 0 to indicate non-relayed PDUs
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_default_ttl_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_set_default_ttl(uint16_t enc_netkey_index,
                                                    uint16_t server_address,
                                                    uint8_t value,
                                                    uint32_t *handle);

/***************************************************************************//**
 *
 * Get node GATT proxy state.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_gatt_proxy_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_get_gatt_proxy(uint16_t enc_netkey_index,
                                                   uint16_t server_address,
                                                   uint32_t *handle);

/***************************************************************************//**
 *
 * Set node GATT proxy state.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] value GATT proxy value to set. Valid values are:
 *     - 0: Proxy feature is disabled
 *     - 1: Proxy feature is enabled
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_gatt_proxy_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_set_gatt_proxy(uint16_t enc_netkey_index,
                                                   uint16_t server_address,
                                                   uint8_t value,
                                                   uint32_t *handle);

/***************************************************************************//**
 *
 * Get node relay state.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_relay_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_get_relay(uint16_t enc_netkey_index,
                                              uint16_t server_address,
                                              uint32_t *handle);

/***************************************************************************//**
 *
 * Set node relay state.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] value Relay value to set. Valid values are:
 *     - 0: Relay feature is disabled
 *     - 1: Relay feature is enabled
 * @param[in] retransmit_count Relay retransmit count. Valid values range from 0
 *   to 7; default value is 0 (no retransmissions).
 * @param[in] retransmit_interval_ms Relay retransmit interval in milliseconds.
 *   Valid values range from 10 ms to 320 ms, with a resolution of 10 ms. The
 *   value is ignored if the retransmission count is set to zero.
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_relay_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_set_relay(uint16_t enc_netkey_index,
                                              uint16_t server_address,
                                              uint8_t value,
                                              uint8_t retransmit_count,
                                              uint16_t retransmit_interval_ms,
                                              uint32_t *handle);

/***************************************************************************//**
 *
 * Get node network transmit state.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_network_transmit_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_get_network_transmit(uint16_t enc_netkey_index,
                                                         uint16_t server_address,
                                                         uint32_t *handle);

/***************************************************************************//**
 *
 * Set node network transmit state.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] transmit_count Network transmit count. Valid values range from 1
 *   to 8; default value is 1 (single transmission; no retransmissions).
 * @param[in] transmit_interval_ms Network transmit interval in milliseconds.
 *   Valid values range from 10 ms to 320 ms, with a resolution of 10 ms. The
 *   value is ignored if the transmission count is set to one.
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_network_transmit_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_set_network_transmit(uint16_t enc_netkey_index,
                                                         uint16_t server_address,
                                                         uint8_t transmit_count,
                                                         uint16_t transmit_interval_ms,
                                                         uint32_t *handle);

/***************************************************************************//**
 *
 * Get node identity state.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] netkey_index Network key index for which the state is queried
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_identity_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_get_identity(uint16_t enc_netkey_index,
                                                 uint16_t server_address,
                                                 uint16_t netkey_index,
                                                 uint32_t *handle);

/***************************************************************************//**
 *
 * Set node identity state.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] netkey_index Network key index for which the state is configured
 * @param[in] value Identity value to set. Valid values are:
 *     - 0: Node identity advertising is disabled
 *     - 1: Node identity advertising is enabled
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_identity_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_set_identity(uint16_t enc_netkey_index,
                                                 uint16_t server_address,
                                                 uint16_t netkey_index,
                                                 uint8_t value,
                                                 uint32_t *handle);

/***************************************************************************//**
 *
 * Get node friend state.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_friend_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_get_friend(uint16_t enc_netkey_index,
                                               uint16_t server_address,
                                               uint32_t *handle);

/***************************************************************************//**
 *
 * Set node friend state.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] value Friend value to set. Valid values are:
 *     - 0: Friend feature is not enabled
 *     - 1: Friend feature is enabled
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_friend_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_set_friend(uint16_t enc_netkey_index,
                                               uint16_t server_address,
                                               uint8_t value,
                                               uint32_t *handle);

/***************************************************************************//**
 *
 * Get node key refresh phase state.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] netkey_index Network key index for which the state is queried
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_key_refresh_phase_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_get_key_refresh_phase(uint16_t enc_netkey_index,
                                                          uint16_t server_address,
                                                          uint16_t netkey_index,
                                                          uint32_t *handle);

/***************************************************************************//**
 *
 * Get the LPN poll timeout from a Friend node.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] lpn_address LPN address
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_lpn_polltimeout_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_get_lpn_polltimeout(uint16_t enc_netkey_index,
                                                        uint16_t server_address,
                                                        uint16_t lpn_address,
                                                        uint32_t *handle);

/***************************************************************************//**
 *
 * Get composition data of a device.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[in] page Composition data page to query
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_dcd_data
 *   - @ref sl_btmesh_evt_config_client_dcd_data_end
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_get_dcd(uint16_t enc_netkey_index,
                                            uint16_t server_address,
                                            uint8_t page,
                                            uint32_t *handle);

/***************************************************************************//**
 *
 * Request a node to unprovision itself. Use this function when a node is
 * removed from the network.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server_address Destination node primary element address
 * @param[out] handle Request handle
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_config_client_reset_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_reset_node(uint16_t enc_netkey_index,
                                               uint16_t server_address,
                                               uint32_t *handle);

/***************************************************************************//**
 *
 * Set request timeout for a specific node. Typically low power nodes have
 * longer latencies and default timeout may not be sufficient.
 *
 * Typically this command is issued after receiving a @ref
 * sl_btmesh_evt_config_client_obo_ack_received event indicating that a message
 * has been placed in the Friend Queue. To get an estimate for the needed
 * timeout, sending a @ref sl_btmesh_config_client_get_lpn_polltimeout request
 * to the Friend is recommended.
 *
 * @param[in] lpn_address Destination LPN node primary element address for which
 *   the timeout is to be set.
 * @param[in] timeout_ms Timeout in milliseconds.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 * 
 *
 ******************************************************************************/
sl_status_t sl_btmesh_config_client_set_request_timeout_for_node(uint16_t lpn_address,
                                                                 uint16_t timeout_ms);

/** @} */ // end addtogroup sl_btmesh_config_client

/**
 * @addtogroup sl_btmesh_mbt_client Bluetooth Mesh BLOB Transfer Client Model
 * @{
 *
 * @brief Bluetooth Mesh BLOB Transfer Client Model
 *
 * The Bluetooth Mesh BLOB Transfer (MBT) Client model is used to transfer
 * binary large objects (BLOBs) over a Bluetooth Mesh network. An MBT Client can
 * transfer a BLOB to one or more MBT Servers, either unicasting or multicasting
 * depending on the situation.
 *
 * There are two different modes in which the transfer may be performed: Push
 * BLOB Transfer mode and Pull BLOB Transfer mode. The MBT Client selects the
 * transfer mode based on the capabilities supported by the servers. The Pull
 * BLOB Transfer mode is typically used only if some of the MBT Servers are Low
 * Power Nodes. The MBT Client API usage flow is identical in both transfer
 * modes.
 *
 * The MBT Client model is normally used as a transport in a higher-layer model,
 * e.g., the Firmware Update models. Please refer to the documentation of such a
 * model for more information on how to interact with the MBT Client API in that
 * context.
 *
 * MBT Client command, which generates and sends a message to MBT Server, could
 * fail with SL_STATUS_BUSY or SL_STATUS_NO_MORE_RESOURCE return value.
 * SL_STATUS_BUSY return value implies underlying Tx-processing pipeline is full
 * and can't currently accept new Tx-messages. SL_STATUS_NO_MORE_RESOURCE
 * implies memory resources required for generating the required message are not
 * currently available. In both of the above cases application should reissue
 * the command at later time. Depending of the exact case @ref
 * sl_btmesh_evt_mbt_client_tx_complete could be used for timing the command
 * retry.
 *
 * If the command failed, then no @ref sl_btmesh_evt_mbt_client_tx_complete is
 * going to be emitted and the app's retry timer could be started immediately
 * after receiving the error. In a successful case, the retry timer should never
 * be started before receiving this event as the application might retry before
 * the previous message is sent out to all recipients.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_mbt_client_init_id                                 0x09280028
#define sl_btmesh_cmd_mbt_client_setup_id                                0x00280028
#define sl_btmesh_cmd_mbt_client_query_information_id                    0x01280028
#define sl_btmesh_cmd_mbt_client_get_server_status_id                    0x0d280028
#define sl_btmesh_cmd_mbt_client_add_server_id                           0x0e280028
#define sl_btmesh_cmd_mbt_client_configure_throttle_id                   0x0f280028
#define sl_btmesh_cmd_mbt_client_start_transfer_id                       0x02280028
#define sl_btmesh_cmd_mbt_client_start_block_id                          0x03280028
#define sl_btmesh_cmd_mbt_client_send_chunk_request_rsp_id               0x04280028
#define sl_btmesh_cmd_mbt_client_query_block_status_id                   0x05280028
#define sl_btmesh_cmd_mbt_client_get_status_id                           0x06280028
#define sl_btmesh_cmd_mbt_client_get_transfer_status_id                  0x0a280028
#define sl_btmesh_cmd_mbt_client_cancel_transfer_id                      0x0b280028
#define sl_btmesh_cmd_mbt_client_abort_id                                0x0c280028
#define sl_btmesh_rsp_mbt_client_init_id                                 0x09280028
#define sl_btmesh_rsp_mbt_client_setup_id                                0x00280028
#define sl_btmesh_rsp_mbt_client_query_information_id                    0x01280028
#define sl_btmesh_rsp_mbt_client_get_server_status_id                    0x0d280028
#define sl_btmesh_rsp_mbt_client_add_server_id                           0x0e280028
#define sl_btmesh_rsp_mbt_client_configure_throttle_id                   0x0f280028
#define sl_btmesh_rsp_mbt_client_start_transfer_id                       0x02280028
#define sl_btmesh_rsp_mbt_client_start_block_id                          0x03280028
#define sl_btmesh_rsp_mbt_client_send_chunk_request_rsp_id               0x04280028
#define sl_btmesh_rsp_mbt_client_query_block_status_id                   0x05280028
#define sl_btmesh_rsp_mbt_client_get_status_id                           0x06280028
#define sl_btmesh_rsp_mbt_client_get_transfer_status_id                  0x0a280028
#define sl_btmesh_rsp_mbt_client_cancel_transfer_id                      0x0b280028
#define sl_btmesh_rsp_mbt_client_abort_id                                0x0c280028

/**
 * @brief The MBT Server Status.
 */
typedef enum
{
  sl_btmesh_mbt_client_server_status_done                  = 0x0, /**< (0x0)
                                                                       Current
                                                                       procedure
                                                                       is done. */
  sl_btmesh_mbt_client_server_status_in_progress           = 0x1, /**< (0x1)
                                                                       Current
                                                                       procedure
                                                                       is in
                                                                       progress. */
  sl_btmesh_mbt_client_server_status_error                 = 0x2, /**< (0x2) MBT
                                                                       Server
                                                                       has
                                                                       failed,
                                                                       and is no
                                                                       longer
                                                                       part of
                                                                       the
                                                                       transfer. */
  sl_btmesh_mbt_client_server_status_successfully_complete = 0x3  /**< (0x3)
                                                                       Transfer
                                                                       successfully
                                                                       complete,
                                                                       server
                                                                       has
                                                                       reported
                                                                       to have
                                                                       received
                                                                       all the
                                                                       blocks of
                                                                       the BLOB
                                                                       successfully. */
} sl_btmesh_mbt_client_server_status_t;

/**
 * @brief The MBT Transfer modes.
 */
typedef enum
{
  sl_btmesh_mbt_client_mbt_transfer_mode_none = 0x0, /**< (0x0) No BLOB Transfer
                                                          mode. */
  sl_btmesh_mbt_client_mbt_transfer_mode_push = 0x1, /**< (0x1) Push BLOB
                                                          Transfer mode
                                                          (client-driven). */
  sl_btmesh_mbt_client_mbt_transfer_mode_pull = 0x2, /**< (0x2) Pull BLOB
                                                          Transfer mode
                                                          (server-driven). */
  sl_btmesh_mbt_client_mbt_transfer_mode_both = 0x3  /**< (0x3) Push and Pull
                                                          BLOB Transfer modes. */
} sl_btmesh_mbt_client_transfer_mode_t;

/**
 * @addtogroup sl_btmesh_evt_mbt_client_server_information_status sl_btmesh_evt_mbt_client_server_information_status
 * @{
 * @brief MBT Server information
 *
 * MBT Server information received as a response to the @ref
 * sl_btmesh_mbt_client_query_information command.
 */

/** @brief Identifier of the server_information_status event */
#define sl_btmesh_evt_mbt_client_server_information_status_id            0x002800a8

/***************************************************************************//**
 * @brief Data structure of the server_information_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_mbt_client_server_information_status_s
{
  uint16_t elem_index;         /**< The client model element index. */
  uint16_t server_address;     /**< The MBT Server address. */
  uint8_t  min_block_size_log; /**< The minimum supported block size as an
                                    exponent n in 2^n. */
  uint8_t  max_block_size_log; /**< The maximum supported block size as an
                                    exponent n in 2^n. */
  uint16_t max_chunks;         /**< The maximum number of chunks supported. */
  uint16_t max_chunk_size;     /**< The maximum supported chunk size. */
});

typedef struct sl_btmesh_evt_mbt_client_server_information_status_s sl_btmesh_evt_mbt_client_server_information_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_mbt_client_server_information_status

/**
 * @addtogroup sl_btmesh_evt_mbt_client_query_information_complete sl_btmesh_evt_mbt_client_query_information_complete
 * @{
 * @brief Query MBT Server information is complete
 *
 * All active servers have responded to the information query. The common values
 * supported by all servers are contained in this event for convenience.
 *
 * Note that if no common block size values are supported by all servers, the @p
 * block_size_log_min will be larger than the @p block_size_log_max. In such
 * case, the application should select a block size that is supported by as many
 * servers as possible based on the @ref
 * sl_btmesh_evt_mbt_client_server_information_status events received for each
 * server. Another option is to select either @p block_size_log_min or @p
 * block_size_log_max as is, which are both guaranteed to be supported by at
 * least one server.
 *
 * The BLOB transfer should be initiated next with @ref
 * sl_btmesh_mbt_client_start_transfer.
 */

/** @brief Identifier of the query_information_complete event */
#define sl_btmesh_evt_mbt_client_query_information_complete_id           0x042800a8

/***************************************************************************//**
 * @brief Data structure of the query_information_complete event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_mbt_client_query_information_complete_s
{
  uint16_t elem_index;               /**< The client model element index. */
  uint8_t  block_size_log_min;       /**< The minimum block size supported by
                                          all servers as an exponent n in 2^n. */
  uint8_t  block_size_log_max;       /**< The maximum block size supported by
                                          all servers as an exponent n in 2^n. */
  uint16_t max_chunk_size_min;       /**< The maximum chunk size supported by
                                          all servers. */
  uint16_t max_chunks_min;           /**< The maximum number of chunks supported
                                          by all servers. */
  uint8_t  supported_transfer_modes; /**< Enum @ref
                                          sl_btmesh_mbt_client_transfer_mode_t.
                                          Transfer modes supported by at least
                                          one server and the client. */
  uint8_t  common_transfer_modes;    /**< Enum @ref
                                          sl_btmesh_mbt_client_transfer_mode_t.
                                          Transfer modes supported by all
                                          servers and the client. */
});

typedef struct sl_btmesh_evt_mbt_client_query_information_complete_s sl_btmesh_evt_mbt_client_query_information_complete_t;

/** @} */ // end addtogroup sl_btmesh_evt_mbt_client_query_information_complete

/**
 * @addtogroup sl_btmesh_evt_mbt_client_server_transfer_status sl_btmesh_evt_mbt_client_server_transfer_status
 * @{
 * @brief The MBT Server's current transfer status
 *
 * This event contains the current transfer status received from a MBT Server.
 * The status is received as a response to @ref
 * sl_btmesh_mbt_client_start_transfer or @ref
 * sl_btmesh_mbt_client_get_transfer_status commands.
 */

/** @brief Identifier of the server_transfer_status event */
#define sl_btmesh_evt_mbt_client_server_transfer_status_id               0x012800a8

/***************************************************************************//**
 * @brief Data structure of the server_transfer_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_mbt_client_server_transfer_status_s
{
  uint16_t        elem_index;          /**< The client model element index. */
  uint16_t        server_address;      /**< The MBT Server address. */
  uint8_t         status;              /**< The status code for the last
                                            operation. */
  uint8_t         phase;               /**< The MBT Server phase. */
  sl_bt_uuid_64_t blob_id;             /**< The ID of the BLOB. */
  uint32_t        blob_size;           /**< The size of the BLOB. */
  uint8_t         block_size_log;      /**< The block size as an exponent n in
                                            2^n. */
  uint8array      blocks_not_received; /**< A bitfield indicating blocks which
                                            are not yet received. The least
                                            significant bit of the first byte in
                                            the array belongs to block zero. */
});

typedef struct sl_btmesh_evt_mbt_client_server_transfer_status_s sl_btmesh_evt_mbt_client_server_transfer_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_mbt_client_server_transfer_status

/**
 * @addtogroup sl_btmesh_evt_mbt_client_start_transfer_complete sl_btmesh_evt_mbt_client_start_transfer_complete
 * @{
 * @brief Transfer start is complete
 *
 * All active servers have responded to the transfer start command. The first
 * block start should be initiated with @ref sl_btmesh_mbt_client_start_block.
 */

/** @brief Identifier of the start_transfer_complete event */
#define sl_btmesh_evt_mbt_client_start_transfer_complete_id              0x052800a8

/***************************************************************************//**
 * @brief Data structure of the start_transfer_complete event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_mbt_client_start_transfer_complete_s
{
  uint16_t elem_index; /**< The client model element index. */
});

typedef struct sl_btmesh_evt_mbt_client_start_transfer_complete_s sl_btmesh_evt_mbt_client_start_transfer_complete_t;

/** @} */ // end addtogroup sl_btmesh_evt_mbt_client_start_transfer_complete

/**
 * @addtogroup sl_btmesh_evt_mbt_client_server_block_status sl_btmesh_evt_mbt_client_server_block_status
 * @{
 * @brief The MBT Server's current block status
 *
 * This event contains the current block status received from a MBT Server. The
 * status is received as a response to @ref sl_btmesh_mbt_client_start_block or
 * @ref sl_btmesh_mbt_client_query_block_status commands.
 */

/** @brief Identifier of the server_block_status event */
#define sl_btmesh_evt_mbt_client_server_block_status_id                  0x032800a8

/***************************************************************************//**
 * @brief Data structure of the server_block_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_mbt_client_server_block_status_s
{
  uint16_t   elem_index;     /**< The client model element index. */
  uint16_t   server_address; /**< The MBT Server address. */
  uint8_t    status;         /**< The status code for the last operation. */
  uint8array missing_chunks; /**< A bitfield indicating missing chunks. The
                                  least significant bit of the first byte in the
                                  array belongs to chunk zero. */
});

typedef struct sl_btmesh_evt_mbt_client_server_block_status_s sl_btmesh_evt_mbt_client_server_block_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_mbt_client_server_block_status

/**
 * @addtogroup sl_btmesh_evt_mbt_client_start_block_complete sl_btmesh_evt_mbt_client_start_block_complete
 * @{
 * @brief Block start is complete
 *
 * All active servers have responded to the block start command.
 */

/** @brief Identifier of the start_block_complete event */
#define sl_btmesh_evt_mbt_client_start_block_complete_id                 0x062800a8

/***************************************************************************//**
 * @brief Data structure of the start_block_complete event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_mbt_client_start_block_complete_s
{
  uint16_t elem_index;   /**< The client model element index. */
  uint16_t block_number; /**< The index number of the block. */
  uint32_t block_size;   /**< The size of the block. */
});

typedef struct sl_btmesh_evt_mbt_client_start_block_complete_s sl_btmesh_evt_mbt_client_start_block_complete_t;

/** @} */ // end addtogroup sl_btmesh_evt_mbt_client_start_block_complete

/**
 * @addtogroup sl_btmesh_evt_mbt_client_send_chunk_request sl_btmesh_evt_mbt_client_send_chunk_request
 * @{
 * @brief Request for chunk data
 *
 * The data for the next chunk to be sent is requested from the application. The
 * requested data should be provided with the @ref
 * sl_btmesh_mbt_client_send_chunk_request_rsp command.
 */

/** @brief Identifier of the send_chunk_request event */
#define sl_btmesh_evt_mbt_client_send_chunk_request_id                   0x072800a8

/***************************************************************************//**
 * @brief Data structure of the send_chunk_request event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_mbt_client_send_chunk_request_s
{
  uint16_t elem_index; /**< The client model element index. */
  uint32_t offset;     /**< The chunk offset from the beginning of the BLOB. */
  uint16_t length;     /**< The number of bytes requested. */
});

typedef struct sl_btmesh_evt_mbt_client_send_chunk_request_s sl_btmesh_evt_mbt_client_send_chunk_request_t;

/** @} */ // end addtogroup sl_btmesh_evt_mbt_client_send_chunk_request

/**
 * @addtogroup sl_btmesh_evt_mbt_client_server_partial_block_report sl_btmesh_evt_mbt_client_server_partial_block_report
 * @{
 * @brief The MBT Server's partial block report
 *
 * When the BLOB is transferred using the Pull BLOB Transfer mode, the MBT
 * Server requests chunks from the MBT Client by sending Partial Block Report
 * messages. The @p missing_chunks field indicates which chunks were requested.
 * If no bits are set in @p missing_chunks, then the MBT Server has received all
 * chunks of the current block.
 *
 * The MBT Client starts sending the requested chunks to the server. The chunk
 * data is requested from the application via the @ref
 * sl_btmesh_evt_mbt_client_send_chunk_request events.
 */

/** @brief Identifier of the server_partial_block_report event */
#define sl_btmesh_evt_mbt_client_server_partial_block_report_id          0x0c2800a8

/***************************************************************************//**
 * @brief Data structure of the server_partial_block_report event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_mbt_client_server_partial_block_report_s
{
  uint16_t   elem_index;     /**< The client model element index. */
  uint16_t   server_address; /**< The MBT Server address. */
  uint8array missing_chunks; /**< A bitfield indicating missing chunks. The
                                  least significant bit of the first byte in the
                                  array belongs to chunk zero. */
});

typedef struct sl_btmesh_evt_mbt_client_server_partial_block_report_s sl_btmesh_evt_mbt_client_server_partial_block_report_t;

/** @} */ // end addtogroup sl_btmesh_evt_mbt_client_server_partial_block_report

/**
 * @addtogroup sl_btmesh_evt_mbt_client_send_chunks_complete sl_btmesh_evt_mbt_client_send_chunks_complete
 * @{
 * @brief All chunks in the current block have been sent
 *
 * Each block is transferred in chunks, sending the chunks to the servers which
 * have them missing. This event indicates that a single round of sending those
 * chunks is complete.
 *
 * After receiving this event, the block status should be queried from the
 * servers with the @ref sl_btmesh_mbt_client_query_block_status command.
 */

/** @brief Identifier of the send_chunks_complete event */
#define sl_btmesh_evt_mbt_client_send_chunks_complete_id                 0x082800a8

/***************************************************************************//**
 * @brief Data structure of the send_chunks_complete event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_mbt_client_send_chunks_complete_s
{
  uint16_t elem_index; /**< The client model element index. */
});

typedef struct sl_btmesh_evt_mbt_client_send_chunks_complete_s sl_btmesh_evt_mbt_client_send_chunks_complete_t;

/** @} */ // end addtogroup sl_btmesh_evt_mbt_client_send_chunks_complete

/**
 * @addtogroup sl_btmesh_evt_mbt_client_query_block_status_complete sl_btmesh_evt_mbt_client_query_block_status_complete
 * @{
 * @brief Querying the current block status is complete
 *
 * All active servers have responded to the block status query. If the @p
 * transfer_complete is non-zero, all blocks of the BLOB have been successfully
 * transmitted and the transfer is complete. The MBT Client state is cleaned up
 * and a new transfer may be started.
 *
 * If the transfer is not complete, but the @p block_complete is non-zero, the
 * current block is successfully transmitted and the next block should be
 * started with @ref sl_btmesh_mbt_client_start_block command. If the @p
 * block_complete is zero, some servers still have chunks missing in the current
 * block. The MBT client starts resending those automatically, requesting the
 * data to send again from the application via @ref
 * sl_btmesh_evt_mbt_client_send_chunk_request events.
 */

/** @brief Identifier of the query_block_status_complete event */
#define sl_btmesh_evt_mbt_client_query_block_status_complete_id          0x092800a8

/***************************************************************************//**
 * @brief Data structure of the query_block_status_complete event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_mbt_client_query_block_status_complete_s
{
  uint16_t elem_index;        /**< The client model element index. */
  uint8_t  block_complete;    /**< Non-zero if the block is complete. */
  uint8_t  transfer_complete; /**< Non-zero if the transfer is complete. */
});

typedef struct sl_btmesh_evt_mbt_client_query_block_status_complete_s sl_btmesh_evt_mbt_client_query_block_status_complete_t;

/** @} */ // end addtogroup sl_btmesh_evt_mbt_client_query_block_status_complete

/**
 * @addtogroup sl_btmesh_evt_mbt_client_tx_complete sl_btmesh_evt_mbt_client_tx_complete
 * @{
 * @brief Previous command has been transmitted to all recipients
 *
 * When the previous command has been transmitted to all MBT Servers, either
 * multicasting or unicasting to each server one by one, this event indicates
 * that the whole transmission is completed. It may be used for timing retries,
 * in case all servers do not respond within an expected time after this event.
 */

/** @brief Identifier of the tx_complete event */
#define sl_btmesh_evt_mbt_client_tx_complete_id                          0x0a2800a8

/***************************************************************************//**
 * @brief Data structure of the tx_complete event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_mbt_client_tx_complete_s
{
  uint16_t elem_index; /**< The client model element index. */
});

typedef struct sl_btmesh_evt_mbt_client_tx_complete_s sl_btmesh_evt_mbt_client_tx_complete_t;

/** @} */ // end addtogroup sl_btmesh_evt_mbt_client_tx_complete

/**
 * @addtogroup sl_btmesh_evt_mbt_client_server_failed sl_btmesh_evt_mbt_client_server_failed
 * @{
 * @brief A server failed and was removed from the transfer
 *
 * This event indicates that a server timed out or reported an irrecoverable
 * error. A failed server is removed from the transfer.
 */

/** @brief Identifier of the server_failed event */
#define sl_btmesh_evt_mbt_client_server_failed_id                        0x0b2800a8

/***************************************************************************//**
 * @brief Data structure of the server_failed event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_mbt_client_server_failed_s
{
  uint16_t elem_index;     /**< The client model element index. */
  uint16_t server_address; /**< The MBT Server address. */
  uint8_t  timeout;        /**< Non-zero if a timeout occurred. */
  uint8_t  error;          /**< Non-zero if an error occurred. */
});

typedef struct sl_btmesh_evt_mbt_client_server_failed_s sl_btmesh_evt_mbt_client_server_failed_t;

/** @} */ // end addtogroup sl_btmesh_evt_mbt_client_server_failed

/**
 * @addtogroup sl_btmesh_evt_mbt_client_transfer_complete sl_btmesh_evt_mbt_client_transfer_complete
 * @{
 * @brief The transfer is complete
 *
 * This event indicates that the transfer is complete, either successfully or
 * unsuccessfully. Servers which have failed before the transfer is complete,
 * indicated by @ref sl_btmesh_evt_mbt_client_server_failed event, have failed
 * to receive the full object. All other servers have received the full object
 * successfully. The @ref sl_btmesh_mbt_client_get_status command may be used to
 * get the current status of each server to determine if the transfer to a
 * specific server succeeded or not. Servers which successfully completed the
 * transfer are marked with status done.
 */

/** @brief Identifier of the transfer_complete event */
#define sl_btmesh_evt_mbt_client_transfer_complete_id                    0x0d2800a8

/***************************************************************************//**
 * @brief Data structure of the transfer_complete event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_mbt_client_transfer_complete_s
{
  uint16_t elem_index; /**< The client model element index. */
});

typedef struct sl_btmesh_evt_mbt_client_transfer_complete_s sl_btmesh_evt_mbt_client_transfer_complete_t;

/** @} */ // end addtogroup sl_btmesh_evt_mbt_client_transfer_complete

/***************************************************************************//**
 *
 * Initialize the MBT Client.
 *
 * @param[in] elem_index The client model element index.
 * @param[in] max_servers The maximum number of servers supported
 *   simultaneously. Maximum supported value is 1008.
 * @param[in] max_blocks The maximum number of blocks into which a BLOB can be
 *   broken down. Maximum supported value is 1888.
 * @param[in] max_chunks_per_block The maximum number of chunks into which a
 *   block can be broken down. Maximum supported value is 2000.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_mbt_client_init(uint16_t elem_index,
                                      uint16_t max_servers,
                                      uint16_t max_blocks,
                                      uint16_t max_chunks_per_block);

/***************************************************************************//**
 *
 * Set up a new BLOB transfer.
 *
 * This is the first step in starting a transfer. Basic information of the BLOB
 * and the transfer are defined with this command.
 *
 * After setting up a transfer, the information of the servers should be queried
 * with @ref sl_btmesh_mbt_client_query_information.
 *
 * @param[in] elem_index The client model element index.
 * @param[in] blob_id The ID of the BLOB.
 * @param[in] blob_size The size of the BLOB.
 * @param[in] appkey_index Application key index to use for communicating with
 *   the servers.
 * @param[in] ttl The time-to-live value to use when communicating with the
 *   servers. Valid values: 0, range:[2-127] and 255.
 * @param[in] timeout_base If a Server does not respond within time frame of:
 *   (10000 * (timeout_base + 2)) + (100 * ttl) in units of ms, it will be
 *   marked as inactive.
 * @param[in] supported_transfer_modes A bitfield of supported transfer modes.
 *   At least one must be supported. If multiple transfer modes are supported by
 *   both the client and servers, Push mode will be preferred.
 *     - Bit 0: Push BLOB Transfer mode
 *     - Bit 1: Pull BLOB Transfer mode
 * @param[in] group_address The group address to use if the multicast_threshold
 *   is exceeded. Zero to only use unicast.
 * @param[in] virtual_address If the group address is a VA hash, this needs to
 *   contain the full virtual address label UUID. Otherwise, the value of this
 *   field will be ignored.
 * @param[in] multicast_threshold If the number of servers for any step exceeds
 *   or is equal to this number, the group address will be used. Otherwise,
 *   servers will be looped through one by one. Value of 0 disables the feature.
 * @param[in] servers_len Length of data in @p servers
 * @param[in] servers List of MBT Server addresses, represented as little endian
 *   two byte sequences.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_mbt_client_setup(uint16_t elem_index,
                                       sl_bt_uuid_64_t blob_id,
                                       uint32_t blob_size,
                                       uint16_t appkey_index,
                                       uint8_t ttl,
                                       uint16_t timeout_base,
                                       uint8_t supported_transfer_modes,
                                       uint16_t group_address,
                                       uuid_128 virtual_address,
                                       uint16_t multicast_threshold,
                                       size_t servers_len,
                                       const uint8_t* servers);

/***************************************************************************//**
 *
 * Query MBT Server information.
 *
 * After setting up a new BLOB transfer with @ref sl_btmesh_mbt_client_setup,
 * the server information needs to be queried with this command. The received
 * query result for each server is carried in a separate @ref
 * sl_btmesh_evt_mbt_client_server_information_status event. This information
 * should be used by the application to determine the size of blocks in which
 * the BLOB is divided when transmitting, and the size of chunks in which each
 * block is divided.
 *
 * The @ref sl_btmesh_evt_mbt_client_query_information_complete event indicates
 * that all servers have responded to the query. If this is not received in a
 * timely manner, the query may be resent by calling this command again.
 *
 * After the information query is successfully completed, the BLOB transfer
 * should be initiated with @ref sl_btmesh_mbt_client_start_transfer.
 *
 * @param[in] elem_index The client model element index.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_mbt_client_server_information_status
 *   - @ref sl_btmesh_evt_mbt_client_query_information_complete
 *
 ******************************************************************************/
sl_status_t sl_btmesh_mbt_client_query_information(uint16_t elem_index);

/***************************************************************************//**
 *
 * Check the status of a MBT Server participating in the transfer, as recorded
 * in the Client's internal bookkeeping.
 *
 * @param[in] elem_index The client model element index.
 * @param[in] server_index The index of the MBT Server which status to check.
 * @param[out] server_address The MBT Server address to identify which server
 *   the information belongs to.
 * @param[out] current_procedure_status Enum @ref
 *   sl_btmesh_mbt_client_server_status_t. The MBT Server Status.
 * @param[out] rx_blocks BLOB Transfer blocks delivered and confirmed.
 * @param[out] rx_chunks @parblock
 *   BLOB Transfer current block chunks delivered and confirmed.
 *
 *   <b>Note</b> : parameter is only valid in the push transfer mode.
 *   @endparblock
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_mbt_client_get_server_status(uint16_t elem_index,
                                                   uint16_t server_index,
                                                   uint16_t *server_address,
                                                   uint8_t *current_procedure_status,
                                                   uint16_t *rx_blocks,
                                                   uint16_t *rx_chunks);

/***************************************************************************//**
 *
 * Add MBT Server. Server can be added after calling @ref
 * sl_btmesh_mbt_client_setup and before calling @ref
 * sl_btmesh_mbt_client_query_information.
 *
 * @param[in] elem_index The client model element index.
 * @param[in] server_address The address of the MBT Server to add.
 *
 * @return SL_STATUS_OK if successful. SL_STATUS_BT_MESH_LIMIT_REACHED if the
 *   configured maximun storage has been reached. SL_STATUS_INVALID_STATE if
 *   invalid state, as proper call time for this API specified within the
 *   description of the API. SL_STATUS_BT_MESH_DOES_NOT_EXIST if the MBT Client
 *   model does not exist within the element specifed by the input parameter.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_mbt_client_add_server(uint16_t elem_index,
                                            uint16_t server_address);

/***************************************************************************//**
 *
 * Configure the MBT Client's sender.
 *
 * This command is used to configure sender of the MBT Client.
 *
 * @param[in] elem_index The client model element index.
 * @param[in] throttle_delay_ms Controls the delay in milliseconds between
 *   batches of messages. Works in conjunction with `throttle_concurrent` to
 *   control message sending rate. When the first transmission in the current
 *   batch completes, the sender will wait for this delay before refilling the
 *   batch up to the `throttle_concurrent` limit with new transmissions.
 * @param[in] throttle_concurrent Defines how many messages are being sent
 *   concurrently. If set to 0, the maximum supported value is used.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_mbt_client_configure_throttle(uint16_t elem_index,
                                                    uint16_t throttle_delay_ms,
                                                    uint8_t throttle_concurrent);

/***************************************************************************//**
 *
 * Initiate the BLOB transfer.
 *
 * After receiving @ref sl_btmesh_evt_mbt_client_query_information_complete
 * event, the BLOB transfer should be initiated by this command. Each server's
 * response is carried in a separate @ref
 * sl_btmesh_evt_mbt_client_server_transfer_status event.
 *
 * The @ref sl_btmesh_evt_mbt_client_start_transfer_complete event indicates
 * that all servers have responded to this command. If this is not received in a
 * timely manner, the initiation request may be resent by calling this command
 * again.
 *
 * After the transfer start is successfully completed, the first block should be
 * started with @ref sl_btmesh_mbt_client_start_block.
 *
 * When the transfer is eventually complete, the @ref
 * sl_btmesh_evt_mbt_client_transfer_complete event is emitted.
 *
 * @param[in] elem_index The client model element index.
 * @param[in] block_size_log The block size as an exponent n in 2^n. Range: 6 -
 *   32.
 * @param[in] transfer_mode Enum @ref sl_btmesh_mbt_client_transfer_mode_t. The
 *   transfer mode to use for the transfer. Must select exactly one mode from
 *   sl_btmesh_evt_mbt_client_query_information_complete_s.supported_transfer_modes.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_mbt_client_server_transfer_status
 *   - @ref sl_btmesh_evt_mbt_client_start_transfer_complete
 *   - @ref sl_btmesh_evt_mbt_client_transfer_complete
 *
 ******************************************************************************/
sl_status_t sl_btmesh_mbt_client_start_transfer(uint16_t elem_index,
                                                uint8_t block_size_log,
                                                uint8_t transfer_mode);

/***************************************************************************//**
 *
 * Initiate block start.
 *
 * After receiving @ref sl_btmesh_evt_mbt_client_start_transfer_complete event
 * in the beginning of the transfer, or @ref
 * sl_btmesh_evt_mbt_client_query_block_status_complete event indicating the
 * previous block is complete, the next incomplete block should be started by
 * this command. Each server's response is carried in in a separate @ref
 * sl_btmesh_evt_mbt_client_server_block_status event.
 *
 * The @ref sl_btmesh_evt_mbt_client_start_block_complete event indicates that
 * all servers have responded to this command. If this is not received in a
 * timely manner, the initiation request may be resent by calling this command
 * again.
 *
 * After the block start is successfully completed, the MBT Client starts
 * sending chunks to the servers. The data to send is requested from the
 * application via @ref sl_btmesh_evt_mbt_client_send_chunk_request events.
 *
 * @param[in] elem_index The client model element index.
 * @param[in] chunk_size The block is divided into chunks of this size. The
 *   chunk size has to be large enough so that the number of chunks in which a
 *   block gets divided doesn't exceed any server's maximum, see @ref
 *   sl_btmesh_evt_mbt_client_query_information_complete_s.max_chunks_min.
 *   Range: 1 - 241.
 * @param[out] block_number The index number of the block.
 * @param[out] block_size The size of the block.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_mbt_client_server_block_status
 *   - @ref sl_btmesh_evt_mbt_client_start_block_complete
 *   - @ref sl_btmesh_evt_mbt_client_send_chunk_request
 *
 ******************************************************************************/
sl_status_t sl_btmesh_mbt_client_start_block(uint16_t elem_index,
                                             uint16_t chunk_size,
                                             uint16_t *block_number,
                                             uint32_t *block_size);

/***************************************************************************//**
 *
 * Provide requested chunk data.
 *
 * This command is used to provide the chunk data requested by the MBT Client in
 * a @ref sl_btmesh_evt_mbt_client_send_chunk_request event.
 *
 * @param[in] elem_index The client model element index.
 * @param[in] data_len Length of data in @p data
 * @param[in] data The chunk data.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_mbt_client_send_chunk_request_rsp(uint16_t elem_index,
                                                        size_t data_len,
                                                        const uint8_t* data);

/***************************************************************************//**
 *
 * Query the current block status.
 *
 * After receiving @ref sl_btmesh_evt_mbt_client_send_chunks_complete, the block
 * statuses of the MBT Servers should be queried to determine if any chunks are
 * still missing. Each server's response is carried in a separate @ref
 * sl_btmesh_evt_mbt_client_server_block_status event.
 *
 * The @ref sl_btmesh_evt_mbt_client_query_block_status_complete event indicates
 * that all servers have responded to this command. If this is not received in a
 * timely manner, the block status query may be resent by calling this command
 * again.
 *
 * After the block status query is successfully completed, but some servers
 * still have chunks missing, the MBT Client starts resending those
 * automatically. See @ref sl_btmesh_evt_mbt_client_query_block_status_complete
 * for more information.
 *
 * @param[in] elem_index The client model element index.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_mbt_client_server_block_status
 *   - @ref sl_btmesh_evt_mbt_client_query_block_status_complete
 *   - @ref sl_btmesh_evt_mbt_client_send_chunk_request
 *
 ******************************************************************************/
sl_status_t sl_btmesh_mbt_client_query_block_status(uint16_t elem_index);

/***************************************************************************//**
 *
 * Get the status of the current transfer.
 *
 * This command returns the MBT Client's current state and the status of each
 * server in the current transfer.
 *
 * @param[in] elem_index The client model element index.
 * @param[out] state The current client state
 *     - 0 = inactive
 *     - 1 = transfer set up, ready to start
 *     - 2 = collecting server info
 *     - 3 = sending transfer start to servers
 *     - 4 = sending block start to servers
 *     - 5 = sending chunks to servers
 *     - 6 = querying received chunk statuses from servers
 *     - 7 = transfer complete (successfully, unsuccessfully, or partially
 *       successfully)
 * @param[out] total_blocks Total number of BLOB Transfer blocks.
 * @param[out] total_chunks Total number of BLOB Transfer chunks within the
 *   current block, 0 if there's no active Block.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_mbt_client_get_status(uint16_t elem_index,
                                            uint8_t *state,
                                            uint16_t *total_blocks,
                                            uint16_t *total_chunks);

/***************************************************************************//**
 *
 * Get the transfer status of an MBT Server.
 *
 * This command can be used to check whether an MBT Server (or several, if a
 * group address is used) is already taking part in a transfer. This command
 * does not need to be used when executing a transfer.
 *
 * The @ref sl_btmesh_evt_mbt_client_server_transfer_status event contains the
 * status response received from the server.
 *
 * @param[in] elem_index The client model element index.
 * @param[in] server_address The MBT Server address. May also be a group or
 *   virtual address.
 * @param[in] virtual_address If the server address is a VA hash, this needs to
 *   contain the full virtual address label UUID. Otherwise, the value of this
 *   field will be ignored.
 * @param[in] appkey_index Application key index to use for encrypting the
 *   message.
 * @param[in] ttl The time-to-live value to use. Valid values: 0, range:[2-127]
 *   and 255.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_mbt_client_server_transfer_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_mbt_client_get_transfer_status(uint16_t elem_index,
                                                     uint16_t server_address,
                                                     uuid_128 virtual_address,
                                                     uint16_t appkey_index,
                                                     uint8_t ttl);

/***************************************************************************//**
 *
 * Cancel a transfer.
 *
 * This command may be used to cancel a transfer at any point after it has been
 * set up with the @ref sl_btmesh_mbt_client_setup command. Each server's
 * response is carried in a separate @ref
 * sl_btmesh_evt_mbt_client_server_transfer_status event.
 *
 * The @ref sl_btmesh_evt_mbt_client_transfer_complete event indicates that all
 * servers have responded to the cancellation. If this is not received in a
 * timely manner, the cancellation may be retried by calling this command again.
 *
 * After the transfer is successfully canceled, a new transfer may be set up
 * with the @ref sl_btmesh_mbt_client_setup command.
 *
 * It's also possible to use this for canceling a previous transfer on a list of
 * MBT Servers in order to start a new one. In this case, the cancel would be
 * issued right after setting up the transfer with the ID of the BLOB to be
 * canceled.
 *
 * @param[in] elem_index The client model element index.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_mbt_client_server_transfer_status
 *   - @ref sl_btmesh_evt_mbt_client_transfer_complete
 *
 ******************************************************************************/
sl_status_t sl_btmesh_mbt_client_cancel_transfer(uint16_t elem_index);

/***************************************************************************//**
 *
 * Abort a previously set up BLOB transfer.
 *
 * Abort a transfer and clear the Client's state. Note that this function does
 * not inform the Servers that the Client has aborted the transfer.
 *
 * @param[in] elem_index The client model element index.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_mbt_client_abort(uint16_t elem_index);

/** @} */ // end addtogroup sl_btmesh_mbt_client

/**
 * @addtogroup sl_btmesh_mbt_server Bluetooth Mesh BLOB Transfer Server Model
 * @{
 *
 * @brief Bluetooth Mesh BLOB Transfer Server Model
 *
 * The Bluetooth Mesh BLOB Transfer (MBT) Server model is used to receive binary
 * large objects (BLOBs) over a Bluetooth Mesh network from a MBT Client.
 *
 * There are two different modes in which the transfer may be performed: Push
 * BLOB Transfer mode and Pull BLOB Transfer mode. The MBT Server may support
 * both, or only one of these modes. The MBT Server API usage flow is identical
 * in both transfer modes.
 *
 * A Low Power Node (LPN) typically only supports the Pull BLOB Transfer mode,
 * as that mode makes it possible for the MBT Server to throttle the transfer
 * speed. This is necessary for ensuring that no more chunks than the Friend
 * node can store in its Friend Queue are transmitted at once. While an LPN is
 * receiving a transfer, the application needs to make sure that messages are
 * polled from the Friend node at least every 30 seconds. If the LPN's Poll
 * Timeout is larger than this, the application may call @ref sl_btmesh_lpn_poll
 * to satisfy this requirement.
 *
 * The MBT Server model is normally used as a transport in a higher-layer model,
 * e.g., the Firmware Update models. Please refer to the documentation of such a
 * model for more information on how to interact with the MBT Server API in that
 * context.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_mbt_server_init_id                                 0x02290028
#define sl_btmesh_cmd_mbt_server_start_id                                0x00290028
#define sl_btmesh_cmd_mbt_server_get_transfer_status_id                  0x01290028
#define sl_btmesh_cmd_mbt_server_transfer_complete_id                    0x03290028
#define sl_btmesh_cmd_mbt_server_abort_id                                0x04290028
#define sl_btmesh_cmd_mbt_server_set_pull_mode_parameters_id             0x05290028
#define sl_btmesh_cmd_mbt_server_transfer_start_rsp_id                   0x06290028
#define sl_btmesh_cmd_mbt_server_enable_block_start_req_id               0x07290028
#define sl_btmesh_cmd_mbt_server_block_start_rsp_id                      0x08290028
#define sl_btmesh_rsp_mbt_server_init_id                                 0x02290028
#define sl_btmesh_rsp_mbt_server_start_id                                0x00290028
#define sl_btmesh_rsp_mbt_server_get_transfer_status_id                  0x01290028
#define sl_btmesh_rsp_mbt_server_transfer_complete_id                    0x03290028
#define sl_btmesh_rsp_mbt_server_abort_id                                0x04290028
#define sl_btmesh_rsp_mbt_server_set_pull_mode_parameters_id             0x05290028
#define sl_btmesh_rsp_mbt_server_transfer_start_rsp_id                   0x06290028
#define sl_btmesh_rsp_mbt_server_enable_block_start_req_id               0x07290028
#define sl_btmesh_rsp_mbt_server_block_start_rsp_id                      0x08290028

/**
 * @brief The MBT Status codes.
 */
typedef enum
{
  sl_btmesh_mbt_server_status_success                   = 0x0,  /**< (0x0) The
                                                                     Client
                                                                     message was
                                                                     successfully
                                                                     received by
                                                                     the Server. */
  sl_btmesh_mbt_server_status_invalid_block_number      = 0x1,  /**< (0x1) The
                                                                     Block
                                                                     number is
                                                                     higher than
                                                                     the total
                                                                     number of
                                                                     blocks
                                                                     indicates. */
  sl_btmesh_mbt_server_status_wrong_block_size          = 0x2,  /**< (0x2) The
                                                                     Block Size
                                                                     is outside
                                                                     of the
                                                                     boundaries
                                                                     specified
                                                                     by the
                                                                     Server's
                                                                     Min and Max
                                                                     Block Size
                                                                     Log
                                                                     capabilities. */
  sl_btmesh_mbt_server_status_wrong_chunk_size          = 0x3,  /**< (0x3) The
                                                                     Chunk Size
                                                                     is too
                                                                     large for
                                                                     the Server
                                                                     to accept,
                                                                     or so small
                                                                     that the
                                                                     Block's
                                                                     total
                                                                     chunks
                                                                     number
                                                                     would
                                                                     exceed the
                                                                     Server's
                                                                     Max Chunks
                                                                     capability. */
  sl_btmesh_mbt_server_status_invalid_state             = 0x4,  /**< (0x4) The
                                                                     Server is
                                                                     in a state
                                                                     where it
                                                                     cannot
                                                                     process the
                                                                     message. */
  sl_btmesh_mbt_server_status_invalid_parameter         = 0x5,  /**< (0x5) A
                                                                     parameter
                                                                     in the
                                                                     message is
                                                                     outside of
                                                                     its valid
                                                                     range. */
  sl_btmesh_mbt_server_status_wrong_object_id           = 0x6,  /**< (0x6) The
                                                                     Server is
                                                                     not
                                                                     expecting
                                                                     to receive
                                                                     this BLOB
                                                                     ID. */
  sl_btmesh_mbt_server_status_storage_limit             = 0x7,  /**< (0x7) Not
                                                                     enough
                                                                     space to
                                                                     store the
                                                                     BLOB. */
  sl_btmesh_mbt_server_status_unsupported_transfer_mode = 0x8,  /**< (0x8)
                                                                     Transfer
                                                                     Mode not
                                                                     supported. */
  sl_btmesh_mbt_server_status_internal_error            = 0x9,  /**< (0x9)
                                                                     Internal
                                                                     error
                                                                     occurred. */
  sl_btmesh_mbt_server_status_information_unavailable   = 0xa,  /**< (0xa) The
                                                                     requested
                                                                     information
                                                                     is not
                                                                     available.
                                                                     A Block Get
                                                                     was sent,
                                                                     but no
                                                                     Block is
                                                                     active. */
  sl_btmesh_mbt_server_status_malformed_message         = 0xfe, /**< (0xfe) A
                                                                     malformed
                                                                     message was
                                                                     received. */
  sl_btmesh_mbt_server_status_timeout                   = 0xff  /**< (0xff)
                                                                     Timeout
                                                                     occurred. */
} sl_btmesh_mbt_server_status_t;

/**
 * @brief The MBT Server Phase.
 */
typedef enum
{
  sl_btmesh_mbt_server_phase_inactive           = 0x0, /**< (0x0) The Server has
                                                            not been given a
                                                            BLOB ID to expect. */
  sl_btmesh_mbt_server_phase_idle               = 0x1, /**< (0x1) The Server has
                                                            a BLOB ID to expect,
                                                            but is not (yet)
                                                            receiving the BLOB. */
  sl_btmesh_mbt_server_phase_waiting_for_block  = 0x2, /**< (0x2) The Server is
                                                            waiting for the next
                                                            Block Start message. */
  sl_btmesh_mbt_server_phase_waiting_for_chunks = 0x3, /**< (0x3) The Server is
                                                            waiting for Chunk
                                                            messages. */
  sl_btmesh_mbt_server_phase_complete           = 0x4, /**< (0x4) The Server has
                                                            received the full
                                                            BLOB. */
  sl_btmesh_mbt_server_phase_suspended          = 0x5  /**< (0x5) The Server has
                                                            timed out and gone
                                                            to Suspended phase. */
} sl_btmesh_mbt_server_phase_t;

/**
 * @addtogroup sl_btmesh_evt_mbt_server_state_changed sl_btmesh_evt_mbt_server_state_changed
 * @{
 * @brief The state of the MBT Server has changed
 *
 * This event may be used by the application to determine the current state and
 * progress of the transfer.
 */

/** @brief Identifier of the state_changed event */
#define sl_btmesh_evt_mbt_server_state_changed_id                        0x062900a8

/***************************************************************************//**
 * @brief Data structure of the state_changed event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_mbt_server_state_changed_s
{
  uint16_t        elem_index;   /**< The server model element index. */
  uint8_t         new_state;    /**< The new state
                                       - 0: Inactive, not expecting any transfer
                                       - 1: Waiting for the BLOB transfer to
                                         start
                                       - 2: Waiting for a new block to start
                                       - 3: Waiting for chunks of the current
                                         block
                                       - 4: The BLOB has been transferred
                                         successfully
                                       - 5: Timeout occurred and the transfer is
                                         suspended */
  sl_bt_uuid_64_t blob_id;      /**< The ID of the BLOB. */
  uint32_t        blob_size;    /**< The size of the BLOB. */
  uint16_t        block_number; /**< The index number of the current block. */
  uint32_t        block_size;   /**< The size of the block. */
  uint16_t        chunk_size;   /**< The current block is divided into chunks of
                                     this size. */
});

typedef struct sl_btmesh_evt_mbt_server_state_changed_s sl_btmesh_evt_mbt_server_state_changed_t;

/** @} */ // end addtogroup sl_btmesh_evt_mbt_server_state_changed

/**
 * @addtogroup sl_btmesh_evt_mbt_server_transfer_start_req sl_btmesh_evt_mbt_server_transfer_start_req
 * @{
 * @brief The MBT Client has requested the transfer start
 *
 * The command @ref sl_btmesh_mbt_server_transfer_start_rsp must be called to
 * accept or reject the transfer start request.
 *
 * After the transfer has been started, the start of the first block is
 * indicated by the @ref sl_btmesh_evt_mbt_server_block_start event.
 */

/** @brief Identifier of the transfer_start_req event */
#define sl_btmesh_evt_mbt_server_transfer_start_req_id                   0x002900a8

/***************************************************************************//**
 * @brief Data structure of the transfer_start_req event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_mbt_server_transfer_start_req_s
{
  uint16_t        elem_index;     /**< The server model element index. */
  uint16_t        client_address; /**< The address of the MBT Client. */
  sl_bt_uuid_64_t blob_id;        /**< The ID of the BLOB. */
  uint32_t        blob_size;      /**< The size of the BLOB. */
  uint8_t         block_size_log; /**< The block size as an exponent n in 2^n. */
  uint8_t         transfer_mode;  /**< Transfer mode: 1 = Push Mode
                                       (client-driven), 2 = Pull Mode
                                       (server-driven). */
});

typedef struct sl_btmesh_evt_mbt_server_transfer_start_req_s sl_btmesh_evt_mbt_server_transfer_start_req_t;

/** @} */ // end addtogroup sl_btmesh_evt_mbt_server_transfer_start_req

/**
 * @addtogroup sl_btmesh_evt_mbt_server_block_start sl_btmesh_evt_mbt_server_block_start
 * @{
 * @brief The transfer of a new block has started
 *
 * If the block start request has been enabled with @ref
 * sl_btmesh_mbt_server_enable_block_start_req, the @ref
 * sl_btmesh_mbt_server_block_start_rsp command must be called to accept or
 * reject the block. The new block starts immediately after it has been
 * accepted.
 *
 * After the new block has been started, the MBT Client starts sending the
 * chunks of the block. The chunk data is carried in @ref
 * sl_btmesh_evt_mbt_server_chunk events. When all chunks of the block have been
 * successfully received, a @ref sl_btmesh_evt_mbt_server_block_complete event
 * is emitted.
 */

/** @brief Identifier of the block_start event */
#define sl_btmesh_evt_mbt_server_block_start_id                          0x012900a8

/***************************************************************************//**
 * @brief Data structure of the block_start event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_mbt_server_block_start_s
{
  uint16_t        elem_index;     /**< The server model element index. */
  uint16_t        client_address; /**< The address of the MBT Client. */
  sl_bt_uuid_64_t blob_id;        /**< The ID of the BLOB. */
  uint16_t        block_number;   /**< The index number of the new block. */
  uint16_t        chunk_size;     /**< The new block is divided into chunks of
                                       this size. */
  uint16_t        block_size;     /**< The size of the new block. */
});

typedef struct sl_btmesh_evt_mbt_server_block_start_s sl_btmesh_evt_mbt_server_block_start_t;

/** @} */ // end addtogroup sl_btmesh_evt_mbt_server_block_start

/**
 * @addtogroup sl_btmesh_evt_mbt_server_chunk sl_btmesh_evt_mbt_server_chunk
 * @{
 * @brief Received a new chunk of BLOB data
 *
 * Each chunk is delivered to the application with this event only once when it
 * is first received. The offset of the chunk `data` relative to the whole BLOB
 * is in `total_offset`, and relative to the current block in `block_offset`.
 */

/** @brief Identifier of the chunk event */
#define sl_btmesh_evt_mbt_server_chunk_id                                0x032900a8

/***************************************************************************//**
 * @brief Data structure of the chunk event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_mbt_server_chunk_s
{
  uint16_t   elem_index;     /**< The server model element index. */
  uint16_t   client_address; /**< The address of the MBT Client. */
  uint16_t   block_number;   /**< The index number of the current block. */
  uint32_t   total_offset;   /**< The offset of the chunk in the BLOB. */
  uint32_t   block_offset;   /**< The offset of the chunk in the current block. */
  uint8array data;           /**< The chunk data. */
});

typedef struct sl_btmesh_evt_mbt_server_chunk_s sl_btmesh_evt_mbt_server_chunk_t;

/** @} */ // end addtogroup sl_btmesh_evt_mbt_server_chunk

/**
 * @addtogroup sl_btmesh_evt_mbt_server_block_complete sl_btmesh_evt_mbt_server_block_complete
 * @{
 * @brief The transfer of the current block is complete
 *
 * If there are more blocks to transfer, a new block will be started by the MBT
 * Client, indicated by the @ref sl_btmesh_evt_mbt_server_block_start event.
 * Otherwise, the transfer completed successfully, and a @ref
 * sl_btmesh_evt_mbt_server_state_changed event with the proper `new_state` is
 * emitted.
 */

/** @brief Identifier of the block_complete event */
#define sl_btmesh_evt_mbt_server_block_complete_id                       0x042900a8

/***************************************************************************//**
 * @brief Data structure of the block_complete event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_mbt_server_block_complete_s
{
  uint16_t        elem_index;     /**< The server model element index. */
  uint16_t        client_address; /**< The address of the MBT Client. */
  sl_bt_uuid_64_t blob_id;        /**< The ID of the BLOB. */
  uint16_t        block_number;   /**< The index number of the completed block. */
  uint16_t        block_size;     /**< The size of the completed block. */
});

typedef struct sl_btmesh_evt_mbt_server_block_complete_s sl_btmesh_evt_mbt_server_block_complete_t;

/** @} */ // end addtogroup sl_btmesh_evt_mbt_server_block_complete

/**
 * @addtogroup sl_btmesh_evt_mbt_server_transfer_cancel sl_btmesh_evt_mbt_server_transfer_cancel
 * @{
 * @brief The MBT Client has canceled the transfer
 */

/** @brief Identifier of the transfer_cancel event */
#define sl_btmesh_evt_mbt_server_transfer_cancel_id                      0x072900a8

/***************************************************************************//**
 * @brief Data structure of the transfer_cancel event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_mbt_server_transfer_cancel_s
{
  uint16_t        elem_index;     /**< The server model element index. */
  uint16_t        client_address; /**< The address of the MBT Client. */
  sl_bt_uuid_64_t blob_id;        /**< The ID of the BLOB. */
});

typedef struct sl_btmesh_evt_mbt_server_transfer_cancel_s sl_btmesh_evt_mbt_server_transfer_cancel_t;

/** @} */ // end addtogroup sl_btmesh_evt_mbt_server_transfer_cancel

/**
 * @addtogroup sl_btmesh_evt_mbt_server_partial_block_report_tx_complete sl_btmesh_evt_mbt_server_partial_block_report_tx_complete
 * @{
 * @brief The MBT Server has sent partial block report
 */

/** @brief Identifier of the partial_block_report_tx_complete event */
#define sl_btmesh_evt_mbt_server_partial_block_report_tx_complete_id     0x082900a8

/***************************************************************************//**
 * @brief Data structure of the partial_block_report_tx_complete event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_mbt_server_partial_block_report_tx_complete_s
{
  uint16_t   elem_index;           /**< The server model element index. */
  uint16_t   num_requested_chunks; /**< Number of requested chunks. */
  uint8array requested_chunks;     /**< The array of requested chunks in
                                        little-endian uint16 format. */
});

typedef struct sl_btmesh_evt_mbt_server_partial_block_report_tx_complete_s sl_btmesh_evt_mbt_server_partial_block_report_tx_complete_t;

/** @} */ // end addtogroup sl_btmesh_evt_mbt_server_partial_block_report_tx_complete

/***************************************************************************//**
 *
 * Initialize the MBT Server.
 *
 * Note that the configuration parameters `min_block_size_log`,
 * `max_chunks_per_block` and `max_blob_size` have memory usage implications.
 * The server needs to store 1 bit of information per chunk, so every additional
 * 8 chunks supported add 1 byte of RAM usage. Each potential block needs 1 bit
 * of information, so the server must allocate 1 byte of RAM for every 8
 * potential blocks. Supporting small blocks and large objects, or large blocks
 * and small chunks, will use more memory. The values should be chosen to
 * conservatively satisfy the use cases of the node if RAM usage is of concern.
 *
 * @param[in] elem_index The server model element index.
 * @param[in] min_block_size_log The minimum supported block size as an exponent
 *   n in 2^n. Range: 6 - 32.
 * @param[in] max_block_size_log @parblock
 *   The maximum supported block size as an exponent n in 2^n. Must be less than
 *   or equal to `max_chunks_per_block * max_chunk_size`. Range: 6 - 32.
 *   @endparblock
 * @param[in] max_chunks_per_block The maximum number of chunks into which a
 *   block can be broken down.
 * @param[in] max_chunk_size The maximum supported chunk size. Range: 8 - 241.
 * @param[in] max_blob_size The maximum supported BLOB size.
 * @param[in] supported_transfer_modes A bitfield of supported transfer modes.
 *   At least one must be supported.
 *     - Bit 0: Push BLOB Transfer mode
 *     - Bit 1: Pull BLOB Transfer mode
 * @param[in] pull_mode_chunks_to_request In Pull BLOB Transfer mode, how many
 *   chunks to request at a time. Can be set to zero if this transfer mode is
 *   not supported.
 * @param[in] pull_mode_retry_interval_ms In Pull BLOB Transfer mode, retry
 *   requesting chunks with this interval in milliseconds. Can be set to zero if
 *   this transfer mode is not supported.
 * @param[in] pull_mode_retry_count In Pull BLOB Transfer mode, retry requesting
 *   chunks this many times before giving up. Note that if the timeout set by
 *   @ref sl_btmesh_mbt_server_start occurs before the retry attempts are
 *   exhausted, no more retries will be attempted. Can be set to zero if this
 *   transfer mode is not supported.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_mbt_server_init(uint16_t elem_index,
                                      uint8_t min_block_size_log,
                                      uint8_t max_block_size_log,
                                      uint16_t max_chunks_per_block,
                                      uint16_t max_chunk_size,
                                      uint32_t max_blob_size,
                                      uint8_t supported_transfer_modes,
                                      uint16_t pull_mode_chunks_to_request,
                                      uint16_t pull_mode_retry_interval_ms,
                                      uint16_t pull_mode_retry_count);

/***************************************************************************//**
 *
 * Start receiving a BLOB.
 *
 * This is used to declare a BLOB to the MBT Server for receiving. The `blob_id`
 * identifies the expected BLOB. The MBT Server will wait for the MBT Client to
 * start the transfer, which is indicated by @ref
 * sl_btmesh_evt_mbt_server_transfer_start_req.
 *
 * @param[in] elem_index The server model element index.
 * @param[in] blob_id The ID of the BLOB.
 * @param[in] timeout_10s If no messages are received from the MBT Client within
 *   this time frame, the transfer will be suspended.
 * @param[in] ttl The time-to-live value to use when communicating with the MBT
 *   Client. Valid values: 0, range:[2-127] and 255.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_mbt_server_transfer_start_req
 *
 ******************************************************************************/
sl_status_t sl_btmesh_mbt_server_start(uint16_t elem_index,
                                       sl_bt_uuid_64_t blob_id,
                                       uint16_t timeout_10s,
                                       uint8_t ttl);

/***************************************************************************//**
 *
 * Get the status of the transfer.
 *
 * @param[in] elem_index The server model element index.
 * @param[out] state The current state
 *     - 0: Inactive, not expecting any transfer
 *     - 1: Waiting for the BLOB transfer to start
 *     - 2: Waiting for a new block to start
 *     - 3: Waiting for chunks of the current block
 *     - 4: The BLOB has been transferred successfully
 *     - 5: Timeout occurred and the transfer is suspended
 * @param[out] blob_id The ID of the BLOB.
 * @param[out] blob_size The size of the BLOB.
 * @param[out] block_number The index number of the current block.
 * @param[in] max_blocks_missing_mask_size Size of output buffer passed in @p
 *   blocks_missing_mask
 * @param[out] blocks_missing_mask_len On return, set to the length of output
 *   data written to @p blocks_missing_mask
 * @param[out] blocks_missing_mask A bitfield indicating missing blocks. The
 *   least significant bit of the first byte in the array belongs to block zero.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_mbt_server_get_transfer_status(uint16_t elem_index,
                                                     uint8_t *state,
                                                     sl_bt_uuid_64_t *blob_id,
                                                     uint32_t *blob_size,
                                                     uint16_t *block_number,
                                                     size_t max_blocks_missing_mask_size,
                                                     size_t *blocks_missing_mask_len,
                                                     uint8_t *blocks_missing_mask);

/***************************************************************************//**
 *
 * To be called when the transfer is completed. This is only needed when
 * implementing your vendor model that uses MBT Server. The DFU models use this
 * API internally.
 *
 * @param[in] elem_index The server model element index.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_mbt_server_transfer_complete(uint16_t elem_index);

/***************************************************************************//**
 *
 * Abort the ongoing transfer, if any.
 *
 * Abort a transfer and clear the Server's state. Note that this function does
 * not inform the Client that the Server has aborted the transfer.
 *
 * @param[in] elem_index The server model element index.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_mbt_server_abort(uint16_t elem_index);

/***************************************************************************//**
 *
 * Set the retry interval and the retry count parameters in the Pull BLOB
 * Transfer Mode. The new values will take effect at the next transfer start.
 * The command does not modify parameters of the ongoing transfer.
 *
 * @param[in] elem_index The server model element index.
 * @param[in] pull_mode_retry_interval_ms The retry interval in milliseconds
 *   (ms).
 * @param[in] pull_mode_retry_count The retry count.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_mbt_server_set_pull_mode_parameters(uint16_t elem_index,
                                                          uint16_t pull_mode_retry_interval_ms,
                                                          uint16_t pull_mode_retry_count);

/***************************************************************************//**
 *
 * Response to the @ref sl_btmesh_evt_mbt_server_transfer_start_req event.
 *
 * @param[in] elem_index The server model element index.
 * @param[in] status mesh_mbt_status_success to accept,
 *   mesh_mbt_status_internal_error and mesh_mbt_status_storage_limit to reject.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_mbt_server_transfer_start_rsp(uint16_t elem_index,
                                                    uint8_t status);

/***************************************************************************//**
 *
 * Enable the MBT Server block start request.
 *
 * This changes the behavior of the @ref sl_btmesh_evt_mbt_server_block_start
 * event, making it a request instead of a notification. If this is enabled,
 * @ref sl_btmesh_mbt_server_block_start_rsp must be called after receiving the
 * event.
 *
 * @param[in] elem_index The server model element index.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_mbt_server_enable_block_start_req(uint16_t elem_index);

/***************************************************************************//**
 *
 * Accept or reject an MBT Server block start request.
 *
 * This command must be called after receiving the @ref
 * sl_btmesh_evt_mbt_server_block_start event to accept or reject the block. If
 * some preparation is needed for the block, e.g. erasing flash pages, those
 * should be done before accepting the block. The block might also be rejected
 * if implementation-specific errors occur.
 *
 * @param[in] elem_index The server model element index.
 * @param[in] status @ref sl_btmesh_mbt_server_status_success to accept, @ref
 *   sl_btmesh_mbt_server_status_internal_error to reject.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_mbt_server_block_start_rsp(uint16_t elem_index,
                                                 uint8_t status);

/** @} */ // end addtogroup sl_btmesh_mbt_server

/**
 * @addtogroup sl_btmesh_sensor_server Bluetooth Mesh Sensor Server Model
 * @{
 *
 * @brief Bluetooth Mesh Sensor Server Model
 *
 * This class provides the commands and messages to interface with the Sensor
 * Server model.
 *
 * A Sensor State consists of four states:
 *   - Sensor Descriptor
 *   - Sensor Setting
 *   - Sensor Cadence
 *   - Measurement value
 *
 * A multisensor setup is possible by having multiple sensor states within the
 * same model, provided that each sensor has a unique Sensor Property ID.
 *
 * Sensor Descriptor states are constant. Therefore, the stack can cache them
 * and enumerate the present sensors to clients when requested.
 *
 * Currently, the Sensor Server model does not cache the measurement data,
 * sensor settings, or cadence. When a client is querying sensor data, the
 * requests will be propagated to the application.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_sensor_server_init_id                              0x00470028
#define sl_btmesh_cmd_sensor_server_deinit_id                            0x01470028
#define sl_btmesh_cmd_sensor_server_send_descriptor_status_id            0x02470028
#define sl_btmesh_cmd_sensor_server_send_status_id                       0x03470028
#define sl_btmesh_cmd_sensor_server_send_column_status_id                0x04470028
#define sl_btmesh_cmd_sensor_server_send_series_status_id                0x05470028
#define sl_btmesh_rsp_sensor_server_init_id                              0x00470028
#define sl_btmesh_rsp_sensor_server_deinit_id                            0x01470028
#define sl_btmesh_rsp_sensor_server_send_descriptor_status_id            0x02470028
#define sl_btmesh_rsp_sensor_server_send_status_id                       0x03470028
#define sl_btmesh_rsp_sensor_server_send_column_status_id                0x04470028
#define sl_btmesh_rsp_sensor_server_send_series_status_id                0x05470028

/**
 * @addtogroup sl_btmesh_evt_sensor_server_get_request sl_btmesh_evt_sensor_server_get_request
 * @{
 * @brief Indicate an incoming Sensor Get request to get the Sensor Data
 * state(s)
 *
 * This event must be replied to by sending the Sensor Status message.
 */

/** @brief Identifier of the get_request event */
#define sl_btmesh_evt_sensor_server_get_request_id                       0x014700a8

/***************************************************************************//**
 * @brief Data structure of the get_request event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_sensor_server_get_request_s
{
  uint16_t server_address; /**< Destination server address */
  uint16_t elem_index;     /**< Server model element index */
  uint16_t client_address; /**< Source client address */
  uint16_t appkey_index;   /**< The application key index to use */
  uint8_t  flags;          /**< No flags defined currently */
  uint16_t property_id;    /**< Property ID for the sensor. Range: 0x0001 -
                                0x0ffff for a specific device property, or
                                0x0000 to get the values of all sensors. */
});

typedef struct sl_btmesh_evt_sensor_server_get_request_s sl_btmesh_evt_sensor_server_get_request_t;

/** @} */ // end addtogroup sl_btmesh_evt_sensor_server_get_request

/**
 * @addtogroup sl_btmesh_evt_sensor_server_get_column_request sl_btmesh_evt_sensor_server_get_column_request
 * @{
 * @brief Indicate an incoming Sensor Column Get message to get Sensor Series
 * Column state
 *
 * The event must be replied to by sending a Sensor Column Status message.
 */

/** @brief Identifier of the get_column_request event */
#define sl_btmesh_evt_sensor_server_get_column_request_id                0x024700a8

/***************************************************************************//**
 * @brief Data structure of the get_column_request event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_sensor_server_get_column_request_s
{
  uint16_t   server_address; /**< Destination server address */
  uint16_t   elem_index;     /**< Server model element index */
  uint16_t   client_address; /**< Source client model address */
  uint16_t   appkey_index;   /**< The application key index to use */
  uint8_t    flags;          /**< No flags defined currently */
  uint16_t   property_id;    /**< Property ID for the sensor. Range: 0x0001 -
                                  0x0ffff for a specific device property, the
                                  value 0x0000 is prohibited. */
  uint8array column_ids;     /**< Raw value identifying a column */
});

typedef struct sl_btmesh_evt_sensor_server_get_column_request_s sl_btmesh_evt_sensor_server_get_column_request_t;

/** @} */ // end addtogroup sl_btmesh_evt_sensor_server_get_column_request

/**
 * @addtogroup sl_btmesh_evt_sensor_server_get_series_request sl_btmesh_evt_sensor_server_get_series_request
 * @{
 * @brief Indicate an incoming Sensor Series Get message to get the Sensor
 * Series Column states
 *
 * This event must be replied to by sending a Sensor Series Status message.
 */

/** @brief Identifier of the get_series_request event */
#define sl_btmesh_evt_sensor_server_get_series_request_id                0x034700a8

/***************************************************************************//**
 * @brief Data structure of the get_series_request event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_sensor_server_get_series_request_s
{
  uint16_t   server_address; /**< Destination server address */
  uint16_t   elem_index;     /**< Server model element index */
  uint16_t   client_address; /**< Source client address */
  uint16_t   appkey_index;   /**< The application key index to use */
  uint8_t    flags;          /**< No flags defined currently */
  uint16_t   property_id;    /**< Property ID for the sensor. Range: 0x0001 -
                                  0x0ffff for a specific device property, the
                                  value 0x0000 is prohibited. */
  uint8array column_ids;     /**< Optional raw values identifying starting and
                                  ending columns */
});

typedef struct sl_btmesh_evt_sensor_server_get_series_request_s sl_btmesh_evt_sensor_server_get_series_request_t;

/** @} */ // end addtogroup sl_btmesh_evt_sensor_server_get_series_request

/**
 * @addtogroup sl_btmesh_evt_sensor_server_publish sl_btmesh_evt_sensor_server_publish
 * @{
 * @brief Indicate that the publishing period timer elapsed and the app should
 * publish its state.
 */

/** @brief Identifier of the publish event */
#define sl_btmesh_evt_sensor_server_publish_id                           0x044700a8

/***************************************************************************//**
 * @brief Data structure of the publish event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_sensor_server_publish_s
{
  uint16_t elem_index; /**< Client model element index */
  uint32_t period_ms;  /**< The current publishing period that can be used to
                            estimate the next tick, e.g., when the state should
                            be reported at higher frequency. */
});

typedef struct sl_btmesh_evt_sensor_server_publish_s sl_btmesh_evt_sensor_server_publish_t;

/** @} */ // end addtogroup sl_btmesh_evt_sensor_server_publish

/***************************************************************************//**
 *
 * Initialize the Sensor Server model with Sensor Descriptors present at the
 * element. The descriptors are cached and Descriptor Get queries are served
 * without propagating it to the application. All incoming client queries are
 * checked against the cached data. However, only valid requests related to
 * existing sensors are propagated to the the application.
 *
 * @param[in] elem_index Server model element index
 * @param[in] descriptors_len Length of data in @p descriptors
 * @param[in] descriptors @parblock
 *   Sensor Descriptor State structures submitted as a byte array
 *
 *   A sensor descriptor represents the attributes describing the sensor data.
 *   It does not change throughout the lifetime of the element.
 *
 *   The following fields are required:
 *     - Sensor Property ID: 16 bits
 *     - Sensor Positive Tolerance: 12 bits
 *     - Sensor Negative Tolerance: 12 bits
 *     - Sensor Sampling Function: 8 bits
 *     - Sensor Measurement Period: 8 bits
 *     - Sensor Update Interval: 8 bits
 *
 *   Sensor Property ID is a 2-octet value referencing a device property that
 *   describes the meaning and the format of data reported by the sensor. The
 *   value 0x0000 is prohibited. Valid range is 0x0001 to 0xFFFF.
 *   @endparblock
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_sensor_server_init(uint16_t elem_index,
                                         size_t descriptors_len,
                                         const uint8_t* descriptors);

/***************************************************************************//**
 *
 * Deinitialize the Sensor Server functionality.
 * Note that the heap reserved space cannot be freed or reallocated.
 * Reinitializing with greater number of sensors than before will eventually
 * return an out of memory error until the device is reset.
 *
 * @param[in] elem_index Server model element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_sensor_server_deinit(uint16_t elem_index);

/***************************************************************************//**
 *
 * Send a Descriptor Status message either as a reply to a Get Descriptor client
 * request.
 *
 * @param[in] client_address Destination client address. The address 0x0000 can
 *   be used to publish the message according to model configuration
 * @param[in] elem_index Server model element index
 * @param[in] appkey_index The application key index to use.
 * @param[in] flags No flags defined currently
 * @param[in] descriptors_len Length of data in @p descriptors
 * @param[in] descriptors Serialized Sensor Descriptor states for all sensors
 *   within the element consisting one or more 8 bytes structures as follows:
 *     - Sensor Property ID: 16 bits
 *     - Sensor Positive Tolerance: 12 bits
 *     - Sensor Negative Tolerance: 12 bits
 *     - Sensor Sampling Function: 8 bits
 *     - Sensor Measurement Period: 8 bits
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_sensor_server_send_descriptor_status(uint16_t client_address,
                                                           uint16_t elem_index,
                                                           uint16_t appkey_index,
                                                           uint8_t flags,
                                                           size_t descriptors_len,
                                                           const uint8_t* descriptors);

/***************************************************************************//**
 *
 * Send Sensor Status message as a reply to a Get client request or as an
 * unsolicited message.
 *
 * @param[in] client_address Destination client address. The address 0x0000 can
 *   be used to publish the message according to model configuration
 * @param[in] elem_index Setup Server model element index
 * @param[in] appkey_index The application key index to use
 * @param[in] flags No flags defined currently
 * @param[in] sensor_data_len Length of data in @p sensor_data
 * @param[in] sensor_data @parblock
 *   Serialized Sensor Data consisting of one or more Sensor state for each
 *   sensor within the element. To simplify processing, the byte array is in TLV
 *   format:
 *     - 1st Property ID: 16 bits
 *     - Value Length: 8 bits
 *     - Value: variable
 *     - 2nd Property ID: 16 bits
 *     - Value Length: 8 bits
 *     - Value: variable
 *     - ...
 *
 *   If sensor data was requested for a Property ID that does not exist within
 *   the element, the reply must contain the given Property ID with zero length.
 *   @endparblock
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_sensor_server_send_status(uint16_t client_address,
                                                uint16_t elem_index,
                                                uint16_t appkey_index,
                                                uint8_t flags,
                                                size_t sensor_data_len,
                                                const uint8_t* sensor_data);

/***************************************************************************//**
 *
 * Send Column Status message as a response to a Column Get client request or as
 * an unsolicited message
 *
 * @param[in] client_address Destination client address. The address 0x0000 can
 *   be used to publish the message according to model configuration
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use.
 * @param[in] flags No flags defined currently
 * @param[in] property_id Property ID for the sensor. Range: 0x0001 - 0x0ffff
 *   for a specific device property, the value 0x0000 is prohibited.
 * @param[in] sensor_data_len Length of data in @p sensor_data
 * @param[in] sensor_data @parblock
 *   Byte array containing the serialized Sensor Series Column state in the
 *   following format:
 *     - Sensor Raw Value X, variable length raw value representing the left
 *       corner of a column
 *     - Sensor Column Width, variable length raw value representing the width
 *       of the column
 *     - Sensor Raw Value Y, variable length raw value representing the height
 *       of the column
 *
 *   If the Property ID or the column ID (Raw value X) does not exist, the reply
 *   must contain only these two fields, omitting the optional Column Width and
 *   Raw Value Y fields.
 *   @endparblock
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_sensor_server_send_column_status(uint16_t client_address,
                                                       uint16_t elem_index,
                                                       uint16_t appkey_index,
                                                       uint8_t flags,
                                                       uint16_t property_id,
                                                       size_t sensor_data_len,
                                                       const uint8_t* sensor_data);

/***************************************************************************//**
 *
 * Send Series Status message as a response to a Series Get client request or as
 * an unsolicited message.
 *
 * @param[in] client_address Destination client address. The address 0x0000 can
 *   be used to publish the message according to model configuration
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use.
 * @param[in] flags No flags defined currently
 * @param[in] property_id Property ID for the sensor. Range: 0x0001 - 0x0ffff
 *   for a specific device property, the value 0x0000 is prohibited.
 * @param[in] sensor_data_len Length of data in @p sensor_data
 * @param[in] sensor_data @parblock
 *   Byte array containing the serialized sequence of Sensor Series Column
 *   states in the following format:
 *     - 1st Sensor Raw Value X, variable length raw value representing the left
 *       corner of a column
 *     - 1st Sensor Column Width, variable length raw value representing the
 *       width of the column
 *     - 1st Sensor Raw Value Y, variable length raw value representing the
 *       height of the column
 *     - ...
 *     - Nth Sensor Raw Value X, variable length raw value representing the left
 *       corner of a column
 *     - Nth Sensor Column Width, variable length raw value representing the
 *       width of the column
 *     - Nth Sensor Raw Value Y, variable length raw value representing the
 *       height of the column
 *
 *   If Property ID does not exist in the element, the reply must contain only
 *   the given Property ID, omitting the other optional fields to column
 *   identifiers and column values.
 *   @endparblock
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_sensor_server_send_series_status(uint16_t client_address,
                                                       uint16_t elem_index,
                                                       uint16_t appkey_index,
                                                       uint8_t flags,
                                                       uint16_t property_id,
                                                       size_t sensor_data_len,
                                                       const uint8_t* sensor_data);

/** @} */ // end addtogroup sl_btmesh_sensor_server

/**
 * @addtogroup sl_btmesh_sensor_setup_server Bluetooth Mesh Sensor Setup Server Model
 * @{
 *
 * @brief Bluetooth Mesh Sensor Setup Server Model
 *
 * This class provides the commands and messages to interface with the Sensor
 * Setup Server model. Elements containing sensor model must have a setup server
 * model attached. Therefore, it is initialized/deinitialized internally
 * together with the server model.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_sensor_setup_server_send_cadence_status_id         0x00480028
#define sl_btmesh_cmd_sensor_setup_server_send_settings_status_id        0x01480028
#define sl_btmesh_cmd_sensor_setup_server_send_setting_status_id         0x02480028
#define sl_btmesh_rsp_sensor_setup_server_send_cadence_status_id         0x00480028
#define sl_btmesh_rsp_sensor_setup_server_send_settings_status_id        0x01480028
#define sl_btmesh_rsp_sensor_setup_server_send_setting_status_id         0x02480028

/**
 * @addtogroup sl_btmesh_evt_sensor_setup_server_get_cadence_request sl_btmesh_evt_sensor_setup_server_get_cadence_request
 * @{
 * @brief Indicate an incoming Sensor Cadence Get request
 *
 * This event must be replied to by sending a Sensor Cadence Status message.
 */

/** @brief Identifier of the get_cadence_request event */
#define sl_btmesh_evt_sensor_setup_server_get_cadence_request_id         0x004800a8

/***************************************************************************//**
 * @brief Data structure of the get_cadence_request event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_sensor_setup_server_get_cadence_request_s
{
  uint16_t server_address; /**< Address the request was directed to, either the
                                server's unicast address or a group address the
                                server subscribes to */
  uint16_t elem_index;     /**< Setup Server model element index */
  uint16_t client_address; /**< Requesting client model's address */
  uint16_t appkey_index;   /**< The application key index to use */
  uint8_t  flags;          /**< No flags defined currently */
  uint16_t property_id;    /**< Property ID for the sensor. Range: 0x0001 -
                                0x0ffff for a specific device property, the
                                value 0x0000 is prohibited. */
});

typedef struct sl_btmesh_evt_sensor_setup_server_get_cadence_request_s sl_btmesh_evt_sensor_setup_server_get_cadence_request_t;

/** @} */ // end addtogroup sl_btmesh_evt_sensor_setup_server_get_cadence_request

/**
 * @addtogroup sl_btmesh_evt_sensor_setup_server_set_cadence_request sl_btmesh_evt_sensor_setup_server_set_cadence_request
 * @{
 * @brief Indicate an incoming Sensor Cadence Set request, which can be replied
 * to by sending a Sensor Cadence Status message
 *
 * Only Sensor Cadence Set (acknowledged) request results in a direct reply. In
 * addition, configuration changes must be reported by publishing the updated
 * cadence state according to model configuration.
 */

/** @brief Identifier of the set_cadence_request event */
#define sl_btmesh_evt_sensor_setup_server_set_cadence_request_id         0x014800a8

/***************************************************************************//**
 * @brief Data structure of the set_cadence_request event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_sensor_setup_server_set_cadence_request_s
{
  uint16_t   server_address; /**< Address the request was directed to, either
                                  the server's unicast address or a group
                                  address the server subscribes to */
  uint16_t   elem_index;     /**< Setup Server model element index */
  uint16_t   client_address; /**< Requesting client model's address */
  uint16_t   appkey_index;   /**< The application key index to use */
  uint8_t    flags;          /**< Bit 1 (0x02) defines whether response is
                                  required. If set to 1, the client sent the
                                  message with SET CADENCE opcode and expects a
                                  CADENCE STATUS message in return. */
  uint16_t   property_id;    /**< Property ID for the sensor. Range: 0x0001 -
                                  0x0ffff for a specific device property, the
                                  value 0x0000 is prohibited. */
  uint8_t    period_divisor; /**< Fast Cadence Period Divisor, 7 bits defining
                                  the divisor for the Publish Period */
  uint8_t    trigger_type;   /**< Status Trigger Type, 1 bit: 0 = discrete
                                  value, 1 = delta percentage */
  uint8array params;         /**< Optional byte array containing serialized
                                  fields of Sensor Cadence state, excluding the
                                  property ID, period divisor, and trigger type
                                    - Fast Cadence Period Divisor, 7 bits
                                    - Status Trigger type, 1 bit (0 = discrete
                                      value, 1 = percentage)
                                    - Status Trigger Delta Down, variable length
                                    - Status Trigger Delta Up, variable length
                                    - Status Min Interval, 8 bits, representing
                                      a power of 2 milliseconds. Valid range is
                                      0-26
                                    - Fast Cadence Low, variable length, lower
                                      bound for the fast cadence range
                                    - Low Cadence Low, variable length, higher
                                      bound for the fast cadence range */
});

typedef struct sl_btmesh_evt_sensor_setup_server_set_cadence_request_s sl_btmesh_evt_sensor_setup_server_set_cadence_request_t;

/** @} */ // end addtogroup sl_btmesh_evt_sensor_setup_server_set_cadence_request

/**
 * @addtogroup sl_btmesh_evt_sensor_setup_server_get_settings_request sl_btmesh_evt_sensor_setup_server_get_settings_request
 * @{
 * @brief Indicate an incoming Sensor Settings Get message to fetch the Sensor
 * Setting Property IDs configured for the given Sensor
 *
 * This event must be replied to by sending a Sensor Settings Status message.
 */

/** @brief Identifier of the get_settings_request event */
#define sl_btmesh_evt_sensor_setup_server_get_settings_request_id        0x024800a8

/***************************************************************************//**
 * @brief Data structure of the get_settings_request event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_sensor_setup_server_get_settings_request_s
{
  uint16_t server_address; /**< Address the request was directed to, either the
                                server's unicast address or a group address the
                                server subscribes to */
  uint16_t elem_index;     /**< Setup Server model element index */
  uint16_t client_address; /**< Requesting client model's address */
  uint16_t appkey_index;   /**< The application key index to use */
  uint8_t  flags;          /**< No flags defined currently */
  uint16_t property_id;    /**< Property ID for the sensor. Range: 0x0001 -
                                0x0ffff for a specific device property, the
                                value 0x0000 is prohibited. */
});

typedef struct sl_btmesh_evt_sensor_setup_server_get_settings_request_s sl_btmesh_evt_sensor_setup_server_get_settings_request_t;

/** @} */ // end addtogroup sl_btmesh_evt_sensor_setup_server_get_settings_request

/**
 * @addtogroup sl_btmesh_evt_sensor_setup_server_get_setting_request sl_btmesh_evt_sensor_setup_server_get_setting_request
 * @{
 * @brief Indicate an incoming Sensor Get Setting request to fetch the value of
 * a setting to a given sensor of a setting given by its ID
 *
 * This event must be replied to by sending a Sensor Setting Status message.
 */

/** @brief Identifier of the get_setting_request event */
#define sl_btmesh_evt_sensor_setup_server_get_setting_request_id         0x034800a8

/***************************************************************************//**
 * @brief Data structure of the get_setting_request event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_sensor_setup_server_get_setting_request_s
{
  uint16_t server_address; /**< Address the request was directed to, either the
                                server's unicast address, or a group address the
                                server subscribes to */
  uint16_t elem_index;     /**< Setup Server model element index */
  uint16_t client_address; /**< Requesting client model's address */
  uint16_t appkey_index;   /**< The application key index to use */
  uint8_t  flags;          /**< No flags defined currently */
  uint16_t property_id;    /**< Property ID for the sensor. Range: 0x0001 -
                                0x0ffff for a specific device property, the
                                value 0x0000 is prohibited. */
  uint16_t setting_id;     /**< Sensor Setting Property ID field identifying the
                                device property of a setting. Range: 0x0001 -
                                0xffff, 0x0000 is prohibited. */
});

typedef struct sl_btmesh_evt_sensor_setup_server_get_setting_request_s sl_btmesh_evt_sensor_setup_server_get_setting_request_t;

/** @} */ // end addtogroup sl_btmesh_evt_sensor_setup_server_get_setting_request

/**
 * @addtogroup sl_btmesh_evt_sensor_setup_server_set_setting_request sl_btmesh_evt_sensor_setup_server_set_setting_request
 * @{
 * @brief Indicate an incoming Sensor Set Setting request, which can be replied
 * to by sending a Sensor Setting Status message
 *
 * Only Setting Set (acknowledged) request is replied directly to the client. In
 * addition, configuration changes must be reported by publishing the new state
 * according to model publishing configuration.
 */

/** @brief Identifier of the set_setting_request event */
#define sl_btmesh_evt_sensor_setup_server_set_setting_request_id         0x044800a8

/***************************************************************************//**
 * @brief Data structure of the set_setting_request event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_sensor_setup_server_set_setting_request_s
{
  uint16_t   server_address; /**< Address the request was directed to, either
                                  the server's unicast address, or a group
                                  address the server subscribes to. */
  uint16_t   elem_index;     /**< Setup Server model element index */
  uint16_t   client_address; /**< Requesting client address */
  uint16_t   appkey_index;   /**< The application key index to use */
  uint8_t    flags;          /**< Bit 1 (0x02) defines whether response is
                                  required. If set to 1, the client sent the
                                  message with SET SETTING opcode and expects a
                                  SETTING STATUS message in return. */
  uint16_t   property_id;    /**< Property ID for the sensor. Range: 0x0001 -
                                  0x0ffff for a specific device property, the
                                  value 0x0000 is prohibited. */
  uint16_t   setting_id;     /**< Sensor Setting Property ID field identifying
                                  the device property of a setting. Range:
                                  0x0001 - 0xffff, 0x0000 is prohibited. */
  uint8array raw_value;      /**< Sensor Setting raw value. Size and
                                  representation depends on the type defined by
                                  the Sensor Setting Property ID. */
});

typedef struct sl_btmesh_evt_sensor_setup_server_set_setting_request_s sl_btmesh_evt_sensor_setup_server_set_setting_request_t;

/** @} */ // end addtogroup sl_btmesh_evt_sensor_setup_server_set_setting_request

/**
 * @addtogroup sl_btmesh_evt_sensor_setup_server_publish sl_btmesh_evt_sensor_setup_server_publish
 * @{
 * @brief Indicate that the publishing period timer elapsed and the app
 * should/can publish its state.
 */

/** @brief Identifier of the publish event */
#define sl_btmesh_evt_sensor_setup_server_publish_id                     0x054800a8

/***************************************************************************//**
 * @brief Data structure of the publish event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_sensor_setup_server_publish_s
{
  uint16_t elem_index; /**< Client model element index */
  uint32_t period_ms;  /**< The current publishing period that can be used to
                            estimate the next tick, e.g., when the state should
                            be reported at higher frequency. */
});

typedef struct sl_btmesh_evt_sensor_setup_server_publish_s sl_btmesh_evt_sensor_setup_server_publish_t;

/** @} */ // end addtogroup sl_btmesh_evt_sensor_setup_server_publish

/***************************************************************************//**
 *
 * Reply to a Get/Set Cadence client request with a Cadence Status message. Only
 * Cadence Set (acknowledged) must be answered by sending the status message to
 * the client. In addition, configuration changes must be published according to
 * model publishing configuration.
 *
 * @param[in] client_address Destination client address The address 0x0000 can
 *   be used to publish the message according model configuration instead of a
 *   direct reply.
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use
 * @param[in] flags No flags defined currently
 * @param[in] property_id Property ID for the sensor. Range: 0x0001 - 0x0ffff
 *   for a specific device property, the value 0x0000 is prohibited.
 * @param[in] params_len Length of data in @p params
 * @param[in] params Optional byte array containing the serialized Sensor
 *   Cadence state, excluding the property ID. If not empty, the state consists
 *   of the following fields:
 *     - Fast Cadence Period Divisor, 7 bits
 *     - Status Trigger type, 1 bits (0 = discrete value, 1 = percentage)
 *     - Status Trigger Delta Down, variable length
 *     - Status Trigger Delta Up, variable length
 *     - Status Min Interval, 8 bits, representing a power of 2 milliseconds.
 *       Valid range is 0-26
 *     - Fast Cadence Low, variable length, lower bound for the fast cadence
 *       range
 *     - Low Cadence Low, variable length, higher bound for the fast cadence
 *       range
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_sensor_setup_server_send_cadence_status(uint16_t client_address,
                                                              uint16_t elem_index,
                                                              uint16_t appkey_index,
                                                              uint8_t flags,
                                                              uint16_t property_id,
                                                              size_t params_len,
                                                              const uint8_t* params);

/***************************************************************************//**
 *
 * Reply to a Get Settings client request with a Settings Status message.
 *
 * @param[in] client_address Destination client model address
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use
 * @param[in] flags No flags defined currently
 * @param[in] property_id Property ID for the sensor. Range: 0x0001 - 0x0ffff
 *   for a specific device property, the value 0x0000 is prohibited.
 * @param[in] setting_ids_len Length of data in @p setting_ids
 * @param[in] setting_ids Array of 16-bit Setting Property IDs of the settings
 *   the given sensor has
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_sensor_setup_server_send_settings_status(uint16_t client_address,
                                                               uint16_t elem_index,
                                                               uint16_t appkey_index,
                                                               uint8_t flags,
                                                               uint16_t property_id,
                                                               size_t setting_ids_len,
                                                               const uint8_t* setting_ids);

/***************************************************************************//**
 *
 * Reply to a Get/Set Setting client request with a Setting Status message. Only
 * Set Setting (acknowledged) request must be answered by sending a reply to the
 * unicast address of the sender. In addition, configuration changes must be
 * published if model publishing is set up.
 *
 * @param[in] client_address Destination client model address
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use
 * @param[in] flags No flags defined currently
 * @param[in] property_id Property ID for the sensor. Range: 0x0001 - 0x0ffff
 *   for a specific device property, the value 0x0000 is prohibited.
 * @param[in] setting_id Sensor Setting Property ID field identifying the device
 *   property of a setting. Range: 0x0001 - 0xffff, 0x0000 is prohibited.
 * @param[in] raw_value_len Length of data in @p raw_value
 * @param[in] raw_value Sensor Setting raw value. Size and representation
 *   depends on the type defined by the Sensor Setting Property ID.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_sensor_setup_server_send_setting_status(uint16_t client_address,
                                                              uint16_t elem_index,
                                                              uint16_t appkey_index,
                                                              uint8_t flags,
                                                              uint16_t property_id,
                                                              uint16_t setting_id,
                                                              size_t raw_value_len,
                                                              const uint8_t* raw_value);

/** @} */ // end addtogroup sl_btmesh_sensor_setup_server

/**
 * @addtogroup sl_btmesh_sensor_client Bluetooth Mesh Sensor Client Model
 * @{
 *
 * @brief Bluetooth Mesh Sensor Client Model
 *
 * This class provides the commands and messages to interface with the Sensor
 * Client model.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_sensor_client_init_id                              0x00490028
#define sl_btmesh_cmd_sensor_client_deinit_id                            0x01490028
#define sl_btmesh_cmd_sensor_client_get_descriptor_id                    0x02490028
#define sl_btmesh_cmd_sensor_client_get_id                               0x03490028
#define sl_btmesh_cmd_sensor_client_get_column_id                        0x04490028
#define sl_btmesh_cmd_sensor_client_get_series_id                        0x05490028
#define sl_btmesh_cmd_sensor_client_get_cadence_id                       0x06490028
#define sl_btmesh_cmd_sensor_client_set_cadence_id                       0x07490028
#define sl_btmesh_cmd_sensor_client_get_settings_id                      0x08490028
#define sl_btmesh_cmd_sensor_client_get_setting_id                       0x09490028
#define sl_btmesh_cmd_sensor_client_set_setting_id                       0x0a490028
#define sl_btmesh_rsp_sensor_client_init_id                              0x00490028
#define sl_btmesh_rsp_sensor_client_deinit_id                            0x01490028
#define sl_btmesh_rsp_sensor_client_get_descriptor_id                    0x02490028
#define sl_btmesh_rsp_sensor_client_get_id                               0x03490028
#define sl_btmesh_rsp_sensor_client_get_column_id                        0x04490028
#define sl_btmesh_rsp_sensor_client_get_series_id                        0x05490028
#define sl_btmesh_rsp_sensor_client_get_cadence_id                       0x06490028
#define sl_btmesh_rsp_sensor_client_set_cadence_id                       0x07490028
#define sl_btmesh_rsp_sensor_client_get_settings_id                      0x08490028
#define sl_btmesh_rsp_sensor_client_get_setting_id                       0x09490028
#define sl_btmesh_rsp_sensor_client_set_setting_id                       0x0a490028

/**
 * @addtogroup sl_btmesh_evt_sensor_client_descriptor_status sl_btmesh_evt_sensor_client_descriptor_status
 * @{
 * @brief Indicate an incoming Sensor Descriptor Status message, received as a
 * response to the @ref sl_btmesh_sensor_client_get_descriptor command
 */

/** @brief Identifier of the descriptor_status event */
#define sl_btmesh_evt_sensor_client_descriptor_status_id                 0x004900a8

/***************************************************************************//**
 * @brief Data structure of the descriptor_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_sensor_client_descriptor_status_s
{
  uint16_t   client_address; /**< Destination client model address */
  uint16_t   elem_index;     /**< Client model element index */
  uint16_t   server_address; /**< Source server model address */
  uint16_t   appkey_index;   /**< The application key index to use. */
  uint8_t    flags;          /**< No flags defined currently */
  uint8array descriptors;    /**< One or more Sensor Descriptor states (N times
                                  8 bytes) As a reply to a Get message
                                  referencing a sensor that does not exist, the
                                  array contains only the requested Property ID.
                                  Format of the Descriptor state is as follows:
                                    - Property ID, 16 bits
                                    - Sensor Positive Tolerance, 12 bits
                                    - Sensor Negative Tolerance, 12 bits
                                    - Sensor Sampling Function, 8 bits
                                    - Sensor Measurement Period, 8 bits
                                    - Sensor Update Interval, 8 bits
                                  {/ul> */
});

typedef struct sl_btmesh_evt_sensor_client_descriptor_status_s sl_btmesh_evt_sensor_client_descriptor_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_sensor_client_descriptor_status

/**
 * @addtogroup sl_btmesh_evt_sensor_client_cadence_status sl_btmesh_evt_sensor_client_cadence_status
 * @{
 * @brief Indicate an incoming Sensor Cadence Status message, received as a
 * response to the @ref sl_btmesh_sensor_client_get_cadence or @ref
 * sl_btmesh_sensor_client_set_cadence command
 */

/** @brief Identifier of the cadence_status event */
#define sl_btmesh_evt_sensor_client_cadence_status_id                    0x014900a8

/***************************************************************************//**
 * @brief Data structure of the cadence_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_sensor_client_cadence_status_s
{
  uint16_t   client_address; /**< Destination client model address */
  uint16_t   elem_index;     /**< Client model element index */
  uint16_t   server_address; /**< Source server model address */
  uint16_t   appkey_index;   /**< The application key index to use. */
  uint8_t    flags;          /**< No flags defined currently */
  uint16_t   property_id;    /**< Property ID for the sensor. Range: 0x0001 -
                                  0x0ffff for a specific device property, the
                                  value 0x0000 is prohibited. */
  uint8array params;         /**< Optional byte array containing serialized
                                  fields of Sensor Cadence state, excluding the
                                  property ID
                                    - Fast Cadence Period Divisor, 7 bits
                                    - Status Trigger type, 1 bit (0 = discrete
                                      value, 1 = percentage)
                                    - Status Trigger Delta Down, variable length
                                    - Status Trigger Delta Up, variable length
                                    - Status Min Interval, 8 bits, representing
                                      a power of 2 milliseconds. Valid range is
                                      0-26
                                    - Fast Cadence Low, variable length, lower
                                      bound for the fast cadence range
                                    - Low Cadence Low, variable length, higher
                                      bound for the fast cadence range */
});

typedef struct sl_btmesh_evt_sensor_client_cadence_status_s sl_btmesh_evt_sensor_client_cadence_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_sensor_client_cadence_status

/**
 * @addtogroup sl_btmesh_evt_sensor_client_settings_status sl_btmesh_evt_sensor_client_settings_status
 * @{
 * @brief Indicate an incoming Sensor Settings Status message, received as a
 * response to the @ref sl_btmesh_sensor_client_get_settings command or
 * unsolicited message
 */

/** @brief Identifier of the settings_status event */
#define sl_btmesh_evt_sensor_client_settings_status_id                   0x024900a8

/***************************************************************************//**
 * @brief Data structure of the settings_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_sensor_client_settings_status_s
{
  uint16_t   client_address; /**< Destination client model address */
  uint16_t   elem_index;     /**< Client model element index */
  uint16_t   server_address; /**< Source server model address */
  uint16_t   appkey_index;   /**< The application key index to use */
  uint8_t    flags;          /**< No flags defined currently */
  uint16_t   property_id;    /**< Property ID for the sensor. Range: 0x0001 -
                                  0x0ffff for a specific device property, the
                                  value 0x0000 is prohibited. */
  uint8array setting_ids;    /**< Array of 16-bit Setting Property IDs of the
                                  settings the given sensor has */
});

typedef struct sl_btmesh_evt_sensor_client_settings_status_s sl_btmesh_evt_sensor_client_settings_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_sensor_client_settings_status

/**
 * @addtogroup sl_btmesh_evt_sensor_client_setting_status sl_btmesh_evt_sensor_client_setting_status
 * @{
 * @brief Indicate an incoming Sensor Setting Status message, received as a
 * response to the @ref sl_btmesh_sensor_client_get_setting or @ref
 * sl_btmesh_sensor_client_set_setting command or unsolicited message
 */

/** @brief Identifier of the setting_status event */
#define sl_btmesh_evt_sensor_client_setting_status_id                    0x034900a8

/***************************************************************************//**
 * @brief Data structure of the setting_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_sensor_client_setting_status_s
{
  uint16_t   client_address; /**< Destination client model address */
  uint16_t   elem_index;     /**< Client model element index */
  uint16_t   server_address; /**< Source server model address */
  uint16_t   appkey_index;   /**< The application key index to use */
  uint8_t    flags;          /**< No flags defined currently */
  uint16_t   property_id;    /**< Property ID for the sensor. Range: 0x0001 -
                                  0x0ffff for a specific device property, the
                                  value 0x0000 is prohibited. */
  uint16_t   setting_id;     /**< Sensor Setting Property ID field identifying
                                  the device property of a setting. Range:
                                  0x0001 - 0xffff, 0x0000 is prohibited. */
  uint8array raw_value;      /**< Sensor Setting raw value. Size and
                                  representation depends on the type defined by
                                  the Sensor Setting Property ID. */
});

typedef struct sl_btmesh_evt_sensor_client_setting_status_s sl_btmesh_evt_sensor_client_setting_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_sensor_client_setting_status

/**
 * @addtogroup sl_btmesh_evt_sensor_client_status sl_btmesh_evt_sensor_client_status
 * @{
 * @brief Indicate an incoming Sensor Status event as a published data state
 * from the sensor server or as a reply to the @ref sl_btmesh_sensor_client_get
 * command
 */

/** @brief Identifier of the status event */
#define sl_btmesh_evt_sensor_client_status_id                            0x044900a8

/***************************************************************************//**
 * @brief Data structure of the status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_sensor_client_status_s
{
  uint16_t   client_address; /**< Destination client model address */
  uint16_t   elem_index;     /**< Client model element index */
  uint16_t   server_address; /**< Source server model address */
  uint16_t   appkey_index;   /**< The application key index to use */
  uint8_t    flags;          /**< No flags defined currently */
  uint8array sensor_data;    /**< Serialized Sensor Data consisting of one or
                                  more Sensor state for each sensor within the
                                  element. To simplify processing, the byte
                                  array is in TLV format:
                                    - 1st Property ID: 16 bits
                                    - Value Length: 8 bits
                                    - Value: variable
                                    - 2nd Property ID: 16 bits
                                    - Value Length: 8 bits
                                    - Value: variable
                                    - ...

                                  If the requested Property ID does not exist at
                                  the server element, the reply contains only
                                  the given Property ID and zero length. */
});

typedef struct sl_btmesh_evt_sensor_client_status_s sl_btmesh_evt_sensor_client_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_sensor_client_status

/**
 * @addtogroup sl_btmesh_evt_sensor_client_column_status sl_btmesh_evt_sensor_client_column_status
 * @{
 * @brief Indicate an incoming Sensor Column Status event, received as a
 * response to the @ref sl_btmesh_sensor_client_get_column command or
 * unsolicited message
 */

/** @brief Identifier of the column_status event */
#define sl_btmesh_evt_sensor_client_column_status_id                     0x054900a8

/***************************************************************************//**
 * @brief Data structure of the column_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_sensor_client_column_status_s
{
  uint16_t   client_address; /**< Destination client model address */
  uint16_t   elem_index;     /**< Client model element index */
  uint16_t   server_address; /**< Source server model address */
  uint16_t   appkey_index;   /**< The application key index to use */
  uint8_t    flags;          /**< No flags defined currently */
  uint16_t   property_id;    /**< Property ID for the sensor. Range: 0x0001 -
                                  0x0ffff for a specific device property, the
                                  value 0x0000 is prohibited. */
  uint8array sensor_data;    /**< Byte array containing the serialized Sensor
                                  Series Column state in the following format:
                                    - Sensor Raw Value X, variable length raw
                                      value representing the left corner of a
                                      column
                                    - Sensor Column Width, variable length raw
                                      value representing the width of the column
                                    - Sensor Raw Value Y, variable length raw
                                      value representing the height of the
                                      column

                                  If the requested Property ID or column ID does
                                  not exist at the server elements, the reply
                                  status message contains only these two fields,
                                  omitting column width and height values. */
});

typedef struct sl_btmesh_evt_sensor_client_column_status_s sl_btmesh_evt_sensor_client_column_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_sensor_client_column_status

/**
 * @addtogroup sl_btmesh_evt_sensor_client_series_status sl_btmesh_evt_sensor_client_series_status
 * @{
 * @brief Indicate an incoming Sensor Series Status message, received as a
 * response to the @ref sl_btmesh_sensor_client_get_series command or
 * unsolicited message
 */

/** @brief Identifier of the series_status event */
#define sl_btmesh_evt_sensor_client_series_status_id                     0x064900a8

/***************************************************************************//**
 * @brief Data structure of the series_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_sensor_client_series_status_s
{
  uint16_t   client_address; /**< Destination client model address */
  uint16_t   elem_index;     /**< Client model element index */
  uint16_t   server_address; /**< Source server model address */
  uint16_t   appkey_index;   /**< The application key index to use */
  uint8_t    flags;          /**< No flags defined currently */
  uint16_t   property_id;    /**< Property ID for the sensor. Range: 0x0001 -
                                  0x0ffff for a specific device property, the
                                  value 0x0000 is prohibited. */
  uint8array sensor_data;    /**< Byte array containing the serialized sequence
                                  of Sensor Serier Column states in the
                                  following format:
                                    - 1st Sensor Raw Value X, variable length
                                      raw value representing the left corner of
                                      a column
                                    - 1st Sensor Column Width, variable length
                                      raw value representing the width of the
                                      column
                                    - 1st Sensor Raw Value Y, variable length
                                      raw value representing the height of the
                                      column
                                    - ...
                                    - Nth Sensor Raw Value X, variable length
                                      raw value representing the left corner of
                                      a column
                                    - Nth Sensor Column Width, variable length
                                      raw value representing the width of the
                                      column
                                    - Nth Sensor Raw Value Y, variable length
                                      raw value representing the height of the
                                      column

                                  If a Property ID or column requested does not
                                  exist at the server element, the reply Series
                                  Status message contains only the given
                                  Property ID. */
});

typedef struct sl_btmesh_evt_sensor_client_series_status_s sl_btmesh_evt_sensor_client_series_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_sensor_client_series_status

/**
 * @addtogroup sl_btmesh_evt_sensor_client_publish sl_btmesh_evt_sensor_client_publish
 * @{
 * @brief Indicate that the publishing period timer elapsed and the app
 * should/can publish its state or any request.
 */

/** @brief Identifier of the publish event */
#define sl_btmesh_evt_sensor_client_publish_id                           0x074900a8

/***************************************************************************//**
 * @brief Data structure of the publish event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_sensor_client_publish_s
{
  uint16_t elem_index; /**< Client model element index */
  uint32_t period_ms;  /**< The current publishing period that can be used to
                            estimate the next tick, e.g., when the state should
                            be reported at higher frequency. */
});

typedef struct sl_btmesh_evt_sensor_client_publish_s sl_btmesh_evt_sensor_client_publish_t;

/** @} */ // end addtogroup sl_btmesh_evt_sensor_client_publish

/***************************************************************************//**
 *
 * Initialize the Sensor Client model. Sensor Client does not have any internal
 * configuration, it only activates the model in the Bluetooth mesh stack.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_sensor_client_init();

/***************************************************************************//**
 *
 * Deinitialize the Sensor Client model. There are no sensor-specific
 * configurations to reset. Normally, models are initialized at boot and never
 * deinitialized.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_sensor_client_deinit();

/***************************************************************************//**
 *
 * Get the Sensor Descriptor state of one specific sensor or all sensors within
 * a model. Results in a Sensor Descriptor Status event
 *
 * @param[in] server_address Destination server model address. The address
 *   0x0000 can be used to publish the message according to model configuration
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use.
 * @param[in] flags No flags defined currently
 * @param[in] property_id ProperyID for the sensor (optional). Range: 0x0001 -
 *   0xffff for a specific device property ID or 0x0000 to get all (the value
 *   0x0000 is prohibited as a real ID).
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_sensor_client_descriptor_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_sensor_client_get_descriptor(uint16_t server_address,
                                                   uint16_t elem_index,
                                                   uint16_t appkey_index,
                                                   uint8_t flags,
                                                   uint16_t property_id);

/***************************************************************************//**
 *
 * Send a Sensor Get message to fetch the Sensor Data state of one specific
 * sensor given by its Property ID, results in a Sensor Status event. The
 * Property ID 0x0000 can be used to fetch all sensor values present at a server
 * element.
 *
 * @param[in] server_address Destination server model address. The address
 *   0x0000 can be used to publish the message according to model configuration
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use.
 * @param[in] flags No flags defined currently
 * @param[in] property_id Property ID for the sensor. Range: 0x0001 - 0x0ffff
 *   for a specific device property, or 0x0000 when not used to get values for
 *   all sensors present in the element.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_sensor_client_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_sensor_client_get(uint16_t server_address,
                                        uint16_t elem_index,
                                        uint16_t appkey_index,
                                        uint8_t flags,
                                        uint16_t property_id);

/***************************************************************************//**
 *
 * Get a Sensor Series Column state, results in a Sensor Column Status event.
 *
 * @param[in] server_address Destination server model address. The address
 *   0x0000 can be used to publish the message according to model configuration
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use.
 * @param[in] flags No flags defined currently
 * @param[in] property_id Property ID for the sensor. Range: 0x0001 - 0x0ffff
 *   for a specific device property, the value 0x0000 is prohibited.
 * @param[in] column_id_len Length of data in @p column_id
 * @param[in] column_id Raw value identifying a column
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_sensor_client_column_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_sensor_client_get_column(uint16_t server_address,
                                               uint16_t elem_index,
                                               uint16_t appkey_index,
                                               uint8_t flags,
                                               uint16_t property_id,
                                               size_t column_id_len,
                                               const uint8_t* column_id);

/***************************************************************************//**
 *
 * Get a Sensor Series Column state, which results in a Sensor Series Status
 * event.
 *
 * @param[in] server_address Destination server model address. The address
 *   0x0000 can be used to publish the message according to model configuration
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use.
 * @param[in] flags No flags defined currently
 * @param[in] property_id Property ID for the sensor. Range: 0x0001 - 0x0ffff
 *   for a specific device property, the value 0x0000 is prohibited.
 * @param[in] column_ids_len Length of data in @p column_ids
 * @param[in] column_ids Raw values identifying starting and ending columns
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_sensor_client_series_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_sensor_client_get_series(uint16_t server_address,
                                               uint16_t elem_index,
                                               uint16_t appkey_index,
                                               uint8_t flags,
                                               uint16_t property_id,
                                               size_t column_ids_len,
                                               const uint8_t* column_ids);

/***************************************************************************//**
 *
 * Send a Sensor Get Cadence message to get the Sensor Cadence state, which
 * results in a Sensor Cadence Status message.
 *
 * @param[in] server_address Destination server model address. The address
 *   0x0000 can be used to publish the message according to model configuration
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use
 * @param[in] flags No flags defined currently
 * @param[in] property_id Property ID for the sensor. Range: 0x0001 - 0x0ffff
 *   for a specific device property, the value 0x0000 is prohibited.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_sensor_client_cadence_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_sensor_client_get_cadence(uint16_t server_address,
                                                uint16_t elem_index,
                                                uint16_t appkey_index,
                                                uint8_t flags,
                                                uint16_t property_id);

/***************************************************************************//**
 *
 * Send a Sensor Cadence Set message, either acknowledged or unacknowledged,
 * depending on the message flags. Acknowledged message results in a Cadence
 * Status reply message and event. The server must publish its new state in any
 * case.
 *
 * @param[in] server_address Destination server model address. The address
 *   0x0000 can be used to publish the message according to model configuration
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use
 * @param[in] flags Bit 1 (0x02) defines whether response is required.
 *   If set to 1, SET CADENCE message will be sent, zero will send SET CADENCE
 *   UNACKNOWLEDGED
 * @param[in] property_id Property ID for the sensor. Range: 0x0001 - 0x0ffff
 *   for a specific device property, the value 0x0000 is prohibited.
 * @param[in] params_len Length of data in @p params
 * @param[in] params Byte array containing serialized fields of Sensor Cadence
 *   state, excluding the property ID
 *     - Fast Cadence Period Divisor, 7 bits
 *     - Status Trigger type, 1 bit (0 = discrete value, 1 = percentage)
 *     - Status Trigger Delta Down, variable length
 *     - Status Trigger Delta Up, variable length
 *     - Status Min Interval, 8 bits, representing a power of 2 milliseconds.
 *       Valid range is 0-26
 *     - Fast Cadence Low, variable length, lower bound for the fast cadence
 *       range
 *     - Low Cadence Low, variable length, higher bound for the fast cadence
 *       range
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_sensor_client_cadence_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_sensor_client_set_cadence(uint16_t server_address,
                                                uint16_t elem_index,
                                                uint16_t appkey_index,
                                                uint8_t flags,
                                                uint16_t property_id,
                                                size_t params_len,
                                                const uint8_t* params);

/***************************************************************************//**
 *
 * Send a Sensor Settings Get message to fetch the Sensor Property IDs present
 * for the given sensor, which results in a Sensor Settings Status event.
 *
 * @param[in] server_address Destination server model address. The address
 *   0x0000 can be used to publish the message according to model configuration
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use
 * @param[in] flags No flags defined currently
 * @param[in] property_id Property ID for the sensor. Range: 0x0001 - 0x0ffff
 *   for a specific device property, the value 0x0000 is prohibited.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_sensor_client_settings_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_sensor_client_get_settings(uint16_t server_address,
                                                 uint16_t elem_index,
                                                 uint16_t appkey_index,
                                                 uint8_t flags,
                                                 uint16_t property_id);

/***************************************************************************//**
 *
 * Send a Sensor Get Setting message to get the value of a specific setting for
 * the given sensor, which results in a Sensor Setting Status event.
 *
 * @param[in] server_address Destination server model address. The address
 *   0x0000 can be used to publish the message according to model configuration
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use.
 * @param[in] flags No flags defined currently
 * @param[in] property_id Property ID for the sensor. Range: 0x0001 - 0x0ffff
 *   for a specific device property, the value 0x0000 is prohibited.
 * @param[in] setting_id Sensor Setting Property ID field identifying the device
 *   property of a setting. Range: 0x0001 - 0xffff, 0x0000 is prohibited.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_sensor_client_setting_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_sensor_client_get_setting(uint16_t server_address,
                                                uint16_t elem_index,
                                                uint16_t appkey_index,
                                                uint8_t flags,
                                                uint16_t property_id,
                                                uint16_t setting_id);

/***************************************************************************//**
 *
 * Send Sensor Setting Set message to update the value of a specific setting for
 * the given sensor, either acknowledged or unacknowledged depending on the
 * message flags. Only acknowledged requests will have a direct Sensor Setting
 * Status reply. The server must publish its new state in any case.
 *
 * @param[in] server_address Destination server model address. The address
 *   0x0000 can be used to publish the message according to model configuration
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use
 * @param[in] flags Bit 1 (0x02) defines whether response is required.
 *   If set to 1, SET SETTING message is sent, zero will use SET SETTING
 *   UNACKNOWLEDGED.
 * @param[in] property_id Property ID for the sensor. Range: 0x0001 - 0x0ffff
 *   for a specific device property, the value 0x0000 is prohibited.
 * @param[in] setting_id Sensor Setting Property ID field identifying the device
 *   property of a setting. Range: 0x0001 - 0xffff, 0x0000 is prohibited.
 * @param[in] raw_value_len Length of data in @p raw_value
 * @param[in] raw_value Sensor Setting raw value. Size and representation
 *   depends on the type defined by the Sensor Setting Property ID.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_sensor_client_setting_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_sensor_client_set_setting(uint16_t server_address,
                                                uint16_t elem_index,
                                                uint16_t appkey_index,
                                                uint8_t flags,
                                                uint16_t property_id,
                                                uint16_t setting_id,
                                                size_t raw_value_len,
                                                const uint8_t* raw_value);

/** @} */ // end addtogroup sl_btmesh_sensor_client

/**
 * @addtogroup sl_btmesh_fw_update_client Bluetooth Mesh Firmware Update Client Model
 * @{
 *
 * @brief Bluetooth Mesh Firmware Update Client Model
 *
 * This class provides the commands and messages to interface with the Firmware
 * Update Client model.
 *
 * If Firmware Distribution Server is used only the @ref
 * sl_btmesh_fw_update_client_init function shall be called as all of the other
 * functions are called by the Distribution Server inside the BT Mesh stack
 * automatically.
 *
 * The MBT Client must be initialized before the Update Client.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_fw_update_client_init_id                           0x004a0028
#define sl_btmesh_cmd_fw_update_client_deinit_id                         0x014a0028
#define sl_btmesh_cmd_fw_update_client_get_info_id                       0x084a0028
#define sl_btmesh_cmd_fw_update_client_get_update_id                     0x094a0028
#define sl_btmesh_cmd_fw_update_client_start_update_id                   0x0a4a0028
#define sl_btmesh_cmd_fw_update_client_cancel_update_id                  0x0b4a0028
#define sl_btmesh_cmd_fw_update_client_apply_update_id                   0x0c4a0028
#define sl_btmesh_cmd_fw_update_client_check_metadata_id                 0x0d4a0028
#define sl_btmesh_rsp_fw_update_client_init_id                           0x004a0028
#define sl_btmesh_rsp_fw_update_client_deinit_id                         0x014a0028
#define sl_btmesh_rsp_fw_update_client_get_info_id                       0x084a0028
#define sl_btmesh_rsp_fw_update_client_get_update_id                     0x094a0028
#define sl_btmesh_rsp_fw_update_client_start_update_id                   0x0a4a0028
#define sl_btmesh_rsp_fw_update_client_cancel_update_id                  0x0b4a0028
#define sl_btmesh_rsp_fw_update_client_apply_update_id                   0x0c4a0028
#define sl_btmesh_rsp_fw_update_client_check_metadata_id                 0x0d4a0028

/**
 * @brief The Additional Information value for the firmware update candidate.
 * This value is reported in the Update Server's Metadata Status message, and in
 * the Update Status message if an update is active. This value indicates what
 * will happen to the Target Node after the new firmware is applied successfully
 * and the node is rebooted.
 */
typedef enum
{
  sl_btmesh_fw_update_client_additional_info_none                       = 0x0, /**<
                                                                                    (0x0)
                                                                                    No
                                                                                    changes
                                                                                    to
                                                                                    Device
                                                                                    Composition
                                                                                    Data. */
  sl_btmesh_fw_update_client_additional_info_dcd_change_no_rp           = 0x1, /**<
                                                                                    (0x1)
                                                                                    Device
                                                                                    Composition
                                                                                    Data
                                                                                    will
                                                                                    change,
                                                                                    but
                                                                                    Remote
                                                                                    Provisioning,
                                                                                    and
                                                                                    thus
                                                                                    Composition
                                                                                    Data
                                                                                    Refresh
                                                                                    is
                                                                                    not
                                                                                    supported.
                                                                                    The
                                                                                    node
                                                                                    will
                                                                                    need
                                                                                    to
                                                                                    be
                                                                                    re-provisioned
                                                                                    to
                                                                                    take
                                                                                    its
                                                                                    new
                                                                                    models
                                                                                    into
                                                                                    use. */
  sl_btmesh_fw_update_client_additional_info_dcd_change_rp_supported    = 0x2, /**<
                                                                                    (0x2)
                                                                                    Device
                                                                                    Composition
                                                                                    Data
                                                                                    will
                                                                                    change,
                                                                                    and
                                                                                    can
                                                                                    be
                                                                                    refreshed. */
  sl_btmesh_fw_update_client_additional_info_unprovisioned_after_update = 0x3  /**<
                                                                                    (0x3)
                                                                                    The
                                                                                    node
                                                                                    will
                                                                                    be
                                                                                    unprovisioned
                                                                                    after
                                                                                    the
                                                                                    update. */
} sl_btmesh_fw_update_client_additional_info_t;

/**
 * @addtogroup sl_btmesh_evt_fw_update_client_info_status_current_fwid sl_btmesh_evt_fw_update_client_info_status_current_fwid
 * @{
 * @brief Report information about firmware images installed on a node, FWID
 * part of the Information Status message
 *
 * Sent as a response to @ref sl_btmesh_fw_update_client_get_info
 */

/** @brief Identifier of the info_status_current_fwid event */
#define sl_btmesh_evt_fw_update_client_info_status_current_fwid_id       0x004a00a8

/***************************************************************************//**
 * @brief Data structure of the info_status_current_fwid event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_update_client_info_status_current_fwid_s
{
  uint16_t   elem_index;     /**< Client model element index */
  uint16_t   server_address; /**< Unicast address of the server */
  uint8_t    index;          /**< Index of this FWID in the Firmware Information
                                  List */
  uint8_t    count;          /**< Total number of entries in the Firmware
                                  Information List */
  uint8array current_fwid;   /**< Current Firmware ID */
});

typedef struct sl_btmesh_evt_fw_update_client_info_status_current_fwid_s sl_btmesh_evt_fw_update_client_info_status_current_fwid_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_update_client_info_status_current_fwid

/**
 * @addtogroup sl_btmesh_evt_fw_update_client_info_status_update_uri sl_btmesh_evt_fw_update_client_info_status_update_uri
 * @{
 * @brief Report information about firmware images installed on a node, Update
 * URI part of the Information Status message
 *
 * Send as a response to @ref sl_btmesh_fw_update_client_get_info
 */

/** @brief Identifier of the info_status_update_uri event */
#define sl_btmesh_evt_fw_update_client_info_status_update_uri_id         0x014a00a8

/***************************************************************************//**
 * @brief Data structure of the info_status_update_uri event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_update_client_info_status_update_uri_s
{
  uint16_t   elem_index;     /**< Client model element index */
  uint16_t   server_address; /**< Unicast address of the server */
  uint8_t    index;          /**< Index of this URI in the Firmware Information
                                  List */
  uint8_t    count;          /**< Total number of entries in the Firmware
                                  Information List */
  uint8array update_uri;     /**< Update URI */
});

typedef struct sl_btmesh_evt_fw_update_client_info_status_update_uri_s sl_btmesh_evt_fw_update_client_info_status_update_uri_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_update_client_info_status_update_uri

/**
 * @addtogroup sl_btmesh_evt_fw_update_client_update_status sl_btmesh_evt_fw_update_client_update_status
 * @{
 * @brief Report the status of a firmware update
 *
 * Sent in response to a @ref sl_btmesh_fw_update_client_get_update @ref
 * sl_btmesh_fw_update_client_start_update @ref
 * sl_btmesh_fw_update_client_cancel_update @ref
 * sl_btmesh_fw_update_client_apply_update messages.
 */

/** @brief Identifier of the update_status event */
#define sl_btmesh_evt_fw_update_client_update_status_id                  0x024a00a8

/***************************************************************************//**
 * @brief Data structure of the update_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_update_client_update_status_s
{
  uint16_t        elem_index;             /**< Client model element index */
  uint16_t        server_address;         /**< Unicast address of the server */
  uint8_t         status;                 /**< Enum @ref
                                               sl_btmesh_fw_update_server_update_status_t.
                                               Status code returned */
  uint8_t         phase;                  /**< Enum @ref
                                               sl_btmesh_fw_update_server_update_phase_t.
                                               Phase of the Update Server */
  uint8_t         additional_information; /**< Enum @ref
                                               sl_btmesh_fw_update_client_additional_info_t.
                                               Additional information about what
                                               will happen to the node after the
                                               update is applied. 0 = no changes
                                               to DCD; 1 = DCD will change but
                                               Composition Data Refresh is not
                                               supported; 2 = DCD change and
                                               Composition Data Refresh is
                                               supported; 3 = node will become
                                               unprovisioned. */
  uint16_t        update_timeout_base;    /**< Base value for BLOB Transfer
                                               Timeout */
  uint8_t         distributor_ttl;        /**< TTL used by the Distributor.
                                               Ignore if no update is active. */
  sl_bt_uuid_64_t blob_id;                /**< BLOB ID of the firmware being
                                               transferred. Ignore if no update
                                               is active. */
  uint8_t         fw_index;               /**< Index of the firmware being
                                               updated on the node. */
});

typedef struct sl_btmesh_evt_fw_update_client_update_status_s sl_btmesh_evt_fw_update_client_update_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_update_client_update_status

/**
 * @addtogroup sl_btmesh_evt_fw_update_client_metadata_status sl_btmesh_evt_fw_update_client_metadata_status
 * @{
 * @brief Report whether a server can accept a firmware update
 *
 * Sent in response to a @ref sl_btmesh_fw_update_client_check_metadata
 */

/** @brief Identifier of the metadata_status event */
#define sl_btmesh_evt_fw_update_client_metadata_status_id                0x0a4a00a8

/***************************************************************************//**
 * @brief Data structure of the metadata_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_update_client_metadata_status_s
{
  uint16_t elem_index;             /**< Client model element index */
  uint16_t server_address;         /**< Unicast address of the server */
  uint8_t  status;                 /**< Enum @ref
                                        sl_btmesh_fw_update_server_update_status_t.
                                        Status code returned */
  uint8_t  additional_information; /**< Enum @ref
                                        sl_btmesh_fw_update_client_additional_info_t.
                                        Additional information about what will
                                        happen to the node after the update is
                                        applied. 0 = no changes to DCD; 1 = DCD
                                        will change but Composition Data Refresh
                                        is not supported; 2 = DCD change and
                                        Composition Data Refresh is supported; 3
                                        = node will become unprovisioned. */
  uint8_t  fw_index;               /**< Index of the firmware that was checked
                                        against the metadata. */
});

typedef struct sl_btmesh_evt_fw_update_client_metadata_status_s sl_btmesh_evt_fw_update_client_metadata_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_update_client_metadata_status

/***************************************************************************//**
 *
 * Initialize the Firmware Update Client model.
 *
 * @param[in] elem_index Client model element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_update_client_init(uint16_t elem_index);

/***************************************************************************//**
 *
 * Deinitialize the Firmware Update Client model.
 *
 * @param[in] elem_index Client model element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_update_client_deinit(uint16_t elem_index);

/***************************************************************************//**
 *
 * Query information about the firmware images installed on a node. The received
 * query result is carried in @ref
 * sl_btmesh_evt_fw_update_client_info_status_current_fwid and @ref
 * sl_btmesh_evt_fw_update_client_info_status_update_uri events.
 *
 * @param[in] elem_index Client model element index
 * @param[in] dst Destination address
 * @param[in] virtual_address If the destination address is a VA hash, this
 *   needs to contain the full virtual address label UUID. Otherwise, the value
 *   of this field will be ignored.
 * @param[in] appkey_index Application key index to use for encrypting the
 *   message.
 * @param[in] ttl The time-to-live value to use. Valid values: 0, range:[2-127]
 *   and 255.
 * @param[in] first_index Index of the first requested entry
 * @param[in] max_entries Maximum number of entries
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_update_client_info_status_current_fwid
 *   - @ref sl_btmesh_evt_fw_update_client_info_status_update_uri
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_update_client_get_info(uint16_t elem_index,
                                                uint16_t dst,
                                                uuid_128 virtual_address,
                                                uint16_t appkey_index,
                                                uint8_t ttl,
                                                uint8_t first_index,
                                                uint8_t max_entries);

/***************************************************************************//**
 *
 * Get the current status of the server. @ref
 * sl_btmesh_evt_fw_update_client_update_status is sent as a response.
 *
 * @param[in] elem_index Client model element index
 * @param[in] dst Destination address
 * @param[in] virtual_address If the destination address is a VA hash, this
 *   needs to contain the full virtual address label UUID. Otherwise, the value
 *   of this field will be ignored.
 * @param[in] appkey_index Application key index to use for encrypting the
 *   message.
 * @param[in] ttl The time-to-live value to use. Valid values: 0, range:[2-127]
 *   and 255.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_update_client_update_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_update_client_get_update(uint16_t elem_index,
                                                  uint16_t dst,
                                                  uuid_128 virtual_address,
                                                  uint16_t appkey_index,
                                                  uint8_t ttl);

/***************************************************************************//**
 *
 * Start a firmware update on a server. @ref
 * sl_btmesh_evt_fw_update_client_update_status is sent as a response.
 *
 * @param[in] elem_index Client model element index
 * @param[in] dst Destination address
 * @param[in] virtual_address If the destination address is a VA hash, this
 *   needs to contain the full virtual address label UUID. Otherwise, the value
 *   of this field will be ignored.
 * @param[in] appkey_index Application key index to use for encrypting the
 *   message.
 * @param[in] ttl The time-to-live value to use for sending this message. Valid
 *   values: 0, range:[2-127] and 255.
 * @param[in] update_ttl The time-to-live value the node should use when sending
 *   its status responses in the firmware image transfer. Valid values: 0,
 *   range:[2-127] and 255.
 * @param[in] update_timeout_base Update Timeout Base value to inform to the
 *   node.
 * @param[in] blob_id The BLOB ID of the firmware image that is going to be
 *   transferred.
 * @param[in] fw_index Index of the firmware on the node to be updated.
 * @param[in] metadata_len Length of data in @p metadata
 * @param[in] metadata Metadata associated with the firmware image to be
 *   transferred.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_update_client_update_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_update_client_start_update(uint16_t elem_index,
                                                    uint16_t dst,
                                                    uuid_128 virtual_address,
                                                    uint16_t appkey_index,
                                                    uint8_t ttl,
                                                    uint8_t update_ttl,
                                                    uint16_t update_timeout_base,
                                                    sl_bt_uuid_64_t blob_id,
                                                    uint8_t fw_index,
                                                    size_t metadata_len,
                                                    const uint8_t* metadata);

/***************************************************************************//**
 *
 * Cancel a firmware update and delete any stored information about the update
 * on a server. @ref sl_btmesh_evt_fw_update_client_update_status is sent as a
 * response.
 *
 * @param[in] elem_index Client model element index
 * @param[in] dst Destination address
 * @param[in] virtual_address If the destination address is a VA hash, this
 *   needs to contain the full virtual address label UUID. Otherwise, the value
 *   of this field will be ignored.
 * @param[in] appkey_index Application key index to use for encrypting the
 *   message.
 * @param[in] ttl The time-to-live value to use. Valid values: 0, range:[2-127]
 *   and 255.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_update_client_update_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_update_client_cancel_update(uint16_t elem_index,
                                                     uint16_t dst,
                                                     uuid_128 virtual_address,
                                                     uint16_t appkey_index,
                                                     uint8_t ttl);

/***************************************************************************//**
 *
 * Apply a firmware image that has been transferred to a server. @ref
 * sl_btmesh_evt_fw_update_client_update_status is sent as a response.
 *
 * @param[in] elem_index Client model element index
 * @param[in] dst Destination address
 * @param[in] virtual_address If the destination address is a VA hash, this
 *   needs to contain the full virtual address label UUID. Otherwise, the value
 *   of this field will be ignored.
 * @param[in] appkey_index Application key index to use for encrypting the
 *   message.
 * @param[in] ttl The time-to-live value to use. Valid values: 0, range:[2-127]
 *   and 255.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_update_client_update_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_update_client_apply_update(uint16_t elem_index,
                                                    uint16_t dst,
                                                    uuid_128 virtual_address,
                                                    uint16_t appkey_index,
                                                    uint8_t ttl);

/***************************************************************************//**
 *
 * Check whether the node can accept a firmware based on the firmware's
 * metadata. @ref sl_btmesh_evt_fw_update_client_metadata_status is sent as a
 * response.
 *
 * @param[in] elem_index Client model element index
 * @param[in] dst Destination address
 * @param[in] virtual_address If the destination address is a VA hash, this
 *   needs to contain the full virtual address label UUID. Otherwise, the value
 *   of this field will be ignored.
 * @param[in] appkey_index Application key index to use for encrypting the
 *   message.
 * @param[in] ttl The time-to-live value to use. Valid values: 0, range:[2-127]
 *   and 255.
 * @param[in] fw_index Index of the firmware on the node to check the metadata
 *   against.
 * @param[in] metadata_len Length of data in @p metadata
 * @param[in] metadata Metadata to check.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_update_client_metadata_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_update_client_check_metadata(uint16_t elem_index,
                                                      uint16_t dst,
                                                      uuid_128 virtual_address,
                                                      uint16_t appkey_index,
                                                      uint8_t ttl,
                                                      uint8_t fw_index,
                                                      size_t metadata_len,
                                                      const uint8_t* metadata);

/** @} */ // end addtogroup sl_btmesh_fw_update_client

/**
 * @addtogroup sl_btmesh_lc_client Bluetooth Mesh Light Control Client Model
 * @{
 *
 * @brief Bluetooth Mesh Light Control Client Model
 *
 * Bluetooth Mesh LC Client model API provides functionality to send and receive
 * messages to/from the LC Server and LC Setup Server models.
 *
 * Throughout the API, the client model that's used is identified by its element
 * address and model ID, while the server model responding to the client model
 * requests is identified by its element address and model ID.
 *
 * The API has functions for querying server model states and requesting server
 * model state changes
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_lc_client_init_id                                  0x004c0028
#define sl_btmesh_cmd_lc_client_get_mode_id                              0x014c0028
#define sl_btmesh_cmd_lc_client_set_mode_id                              0x024c0028
#define sl_btmesh_cmd_lc_client_get_om_id                                0x044c0028
#define sl_btmesh_cmd_lc_client_set_om_id                                0x054c0028
#define sl_btmesh_cmd_lc_client_get_light_onoff_id                       0x074c0028
#define sl_btmesh_cmd_lc_client_set_light_onoff_id                       0x084c0028
#define sl_btmesh_cmd_lc_client_get_property_id                          0x094c0028
#define sl_btmesh_cmd_lc_client_set_property_id                          0x0a4c0028
#define sl_btmesh_rsp_lc_client_init_id                                  0x004c0028
#define sl_btmesh_rsp_lc_client_get_mode_id                              0x014c0028
#define sl_btmesh_rsp_lc_client_set_mode_id                              0x024c0028
#define sl_btmesh_rsp_lc_client_get_om_id                                0x044c0028
#define sl_btmesh_rsp_lc_client_set_om_id                                0x054c0028
#define sl_btmesh_rsp_lc_client_get_light_onoff_id                       0x074c0028
#define sl_btmesh_rsp_lc_client_set_light_onoff_id                       0x084c0028
#define sl_btmesh_rsp_lc_client_get_property_id                          0x094c0028
#define sl_btmesh_rsp_lc_client_set_property_id                          0x0a4c0028

/**
 * @addtogroup sl_btmesh_evt_lc_client_mode_status sl_btmesh_evt_lc_client_mode_status
 * @{
 * @brief Indicate an incoming LC Mode Status message, received as a response to
 * @ref sl_btmesh_lc_client_get_mode or @ref sl_btmesh_lc_client_set_mode
 */

/** @brief Identifier of the mode_status event */
#define sl_btmesh_evt_lc_client_mode_status_id                           0x004c00a8

/***************************************************************************//**
 * @brief Data structure of the mode_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_lc_client_mode_status_s
{
  uint16_t destination_address; /**< Address of the client or the group address
                                     to which it was published. */
  uint16_t elem_index;          /**< Index of the element for which received the
                                     status. */
  uint16_t server_address;      /**< Device which sent the status. */
  uint16_t appkey_index;        /**< Appkey used by server_address. */
  uint8_t  mode_status_value;   /**< Value reported by server. */
});

typedef struct sl_btmesh_evt_lc_client_mode_status_s sl_btmesh_evt_lc_client_mode_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_lc_client_mode_status

/**
 * @addtogroup sl_btmesh_evt_lc_client_om_status sl_btmesh_evt_lc_client_om_status
 * @{
 * @brief Indicate an incoming LC Occupancy Mode Status message, received as a
 * response to @ref sl_btmesh_lc_client_get_om or @ref
 * sl_btmesh_lc_client_set_om
 */

/** @brief Identifier of the om_status event */
#define sl_btmesh_evt_lc_client_om_status_id                             0x014c00a8

/***************************************************************************//**
 * @brief Data structure of the om_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_lc_client_om_status_s
{
  uint16_t destination_address; /**< Address of the client or the group address
                                     to which it was published. */
  uint16_t elem_index;          /**< Index of the element for which received the
                                     status. */
  uint16_t server_address;      /**< Device which sent the status. */
  uint16_t appkey_index;        /**< Appkey used by server_address. */
  uint8_t  om_status_value;     /**< Value reported by server. */
});

typedef struct sl_btmesh_evt_lc_client_om_status_s sl_btmesh_evt_lc_client_om_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_lc_client_om_status

/**
 * @addtogroup sl_btmesh_evt_lc_client_light_onoff_status sl_btmesh_evt_lc_client_light_onoff_status
 * @{
 * @brief Indicate an incoming LC Light OnOff Status message, received as a
 * response to @ref sl_btmesh_lc_client_get_light_onoff or @ref
 * sl_btmesh_lc_client_set_light_onoff
 */

/** @brief Identifier of the light_onoff_status event */
#define sl_btmesh_evt_lc_client_light_onoff_status_id                    0x024c00a8

/***************************************************************************//**
 * @brief Data structure of the light_onoff_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_lc_client_light_onoff_status_s
{
  uint16_t destination_address; /**< Address of the client or the group address
                                     to which it was published. */
  uint16_t elem_index;          /**< Index of the element for which received the
                                     status. */
  uint16_t server_address;      /**< Device which sent the status. */
  uint16_t appkey_index;        /**< Appkey used by server_address. */
  uint8_t  present_light_onoff; /**< Present value of the Light OnOff state */
  uint8_t  target_light_onoff;  /**< Target value of the Light OnOff state */
  uint32_t remaining_time_ms;   /**< Time (in milliseconds) remaining in
                                     transition */
});

typedef struct sl_btmesh_evt_lc_client_light_onoff_status_s sl_btmesh_evt_lc_client_light_onoff_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_lc_client_light_onoff_status

/**
 * @addtogroup sl_btmesh_evt_lc_client_property_status sl_btmesh_evt_lc_client_property_status
 * @{
 * @brief Indicate an incoming LC Property Status message, received as a
 * response to @ref sl_btmesh_lc_client_get_property or @ref
 * sl_btmesh_lc_client_set_property
 */

/** @brief Identifier of the property_status event */
#define sl_btmesh_evt_lc_client_property_status_id                       0x034c00a8

/***************************************************************************//**
 * @brief Data structure of the property_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_lc_client_property_status_s
{
  uint16_t   destination_address; /**< Address of the client or the group
                                       address to which it was published. */
  uint16_t   elem_index;          /**< Index of the element for which received
                                       the status. */
  uint16_t   server_address;      /**< Device which sent the status. */
  uint16_t   appkey_index;        /**< Appkey used by server_address. */
  uint16_t   property_id;         /**< Property ID */
  uint8array property_value;      /**< Property value */
});

typedef struct sl_btmesh_evt_lc_client_property_status_s sl_btmesh_evt_lc_client_property_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_lc_client_property_status

/***************************************************************************//**
 *
 * Initialize the LC Client model. LC Client does not have any internal
 * configuration. It only activates the model in the mesh stack.
 *
 * @param[in] elem_index Index of the client element.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lc_client_init(uint16_t elem_index);

/***************************************************************************//**
 *
 * Get the mode status.
 *
 * @param[in] server_address Device to be queried. The address 0x0000 can be
 *   used to publish the message according to the model configuration.
 * @param[in] elem_index Index of the client element.
 * @param[in] appkey_index Appkey used by server_address.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_lc_client_mode_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lc_client_get_mode(uint16_t server_address,
                                         uint16_t elem_index,
                                         uint16_t appkey_index);

/***************************************************************************//**
 *
 * Set mode
 *
 * @param[in] server_address Destination server model address. The address
 *   0x0000 can be used to publish the message according to the model
 *   configuration.
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use.
 * @param[in] flags Bit 1 (0x02) defines whether a response is required.
 *   If set to 1, SET PROPERTY message will be sent. Zero will send SET PROPERTY
 *   UNACKNOWLEDGED.
 * @param[in] mode Mode value to set
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_lc_client_mode_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lc_client_set_mode(uint16_t server_address,
                                         uint16_t elem_index,
                                         uint16_t appkey_index,
                                         uint8_t flags,
                                         uint8_t mode);

/***************************************************************************//**
 *
 * Get the OM status.
 *
 * @param[in] server_address Device to be queried. The address 0x0000 can be
 *   used to publish the message according to the model configuration.
 * @param[in] elem_index Index of the client element.
 * @param[in] appkey_index Appkey used by server_address.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_lc_client_om_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lc_client_get_om(uint16_t server_address,
                                       uint16_t elem_index,
                                       uint16_t appkey_index);

/***************************************************************************//**
 *
 * Set occupancy mode.
 *
 * @param[in] server_address Destination server model address. The address
 *   0x0000 can be used to publish the message according to the model
 *   configuration.
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use.
 * @param[in] flags Bit 1 (0x02) defines whether response is required.
 *   If set to 1, SET PROPERTY message will be sent. Zero will send SET PROPERTY
 *   UNACKNOWLEDGED.
 * @param[in] mode Mode value to set
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_lc_client_om_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lc_client_set_om(uint16_t server_address,
                                       uint16_t elem_index,
                                       uint16_t appkey_index,
                                       uint8_t flags,
                                       uint8_t mode);

/***************************************************************************//**
 *
 * Get the Light OnOff status.
 *
 * @param[in] server_address Device to be queried. The address 0x0000 can be
 *   used to publish the message according to the model configuration.
 * @param[in] elem_index Index of the client element.
 * @param[in] appkey_index Appkey used by server_address.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_lc_client_light_onoff_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lc_client_get_light_onoff(uint16_t server_address,
                                                uint16_t elem_index,
                                                uint16_t appkey_index);

/***************************************************************************//**
 *
 * Set the Light OnOff.
 *
 * @param[in] server_address Destination server model address. The address
 *   0x0000 can be used to publish the message according to the model
 *   configuration.
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use.
 * @param[in] flags Bit 1 (0x02) defines whether response is required.
 *   If set to 1, SET PROPERTY message will be sent. Zero will send SET PROPERTY
 *   UNACKNOWLEDGED.
 * @param[in] target_state The target value of the Light LC Light OnOff state
 * @param[in] tid Transaction identifier
 * @param[in] transition_time_ms Transition time in milliseconds. Value of
 *   0xFFFFFFFF will cause this parameter as well as the "delay" parameter to be
 *   omitted.
 * @param[in] message_delay_ms Message execution delay in milliseconds. If the
 *   "transition_time" is 0xFFFFFFFF, this parameter is ignored. If both the
 *   transition time and the delay are zero, the transition is immediate. Valid
 *   range: 0-1275.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_lc_client_light_onoff_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lc_client_set_light_onoff(uint16_t server_address,
                                                uint16_t elem_index,
                                                uint16_t appkey_index,
                                                uint8_t flags,
                                                uint8_t target_state,
                                                uint8_t tid,
                                                uint32_t transition_time_ms,
                                                uint16_t message_delay_ms);

/***************************************************************************//**
 *
 * Get the Property status.
 *
 * @param[in] server_address Device to be queried. The address 0x0000 can be
 *   used to publish the message according to the model configuration.
 * @param[in] elem_index Index of the client element.
 * @param[in] appkey_index Appkey used by server_address.
 * @param[in] property_id The property ID to query.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_lc_client_property_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lc_client_get_property(uint16_t server_address,
                                             uint16_t elem_index,
                                             uint16_t appkey_index,
                                             uint16_t property_id);

/***************************************************************************//**
 *
 * Set a particular property.
 *
 * @param[in] server_address Destination server model address. The address
 *   0x0000 can be used to publish the message according to the model
 *   configuration.
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use.
 * @param[in] flags Bit 1 (0x02) defines whether response is required.
 *   If set to 1, SET PROPERTY message will be sent. Zero will send SET PROPERTY
 *   UNACKNOWLEDGED.
 * @param[in] property_id Property ID for the LC Server. Range: 0x0001 - 0x0ffff
 *   for a specific device property, the value 0x0000 is prohibited.
 * @param[in] params_len Length of data in @p params
 * @param[in] params Byte array containing serialized fields of LC Property,
 *   excluding the property ID
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_lc_client_property_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lc_client_set_property(uint16_t server_address,
                                             uint16_t elem_index,
                                             uint16_t appkey_index,
                                             uint8_t flags,
                                             uint16_t property_id,
                                             size_t params_len,
                                             const uint8_t* params);

/** @} */ // end addtogroup sl_btmesh_lc_client

/**
 * @addtogroup sl_btmesh_lc_server Bluetooth Mesh Light Control Server Model
 * @{
 *
 * @brief Bluetooth Mesh Light Control Server Model
 *
 * Bluetooth Mesh Light Control Server model functionality.
 *
 * All LC Server state resides in and is own by the Model (stack). The state
 * update notification events to the application are informational: the
 * application is not required to react to them. The application may choose to
 * save the LC Server state in persistent storage and set the states in the LC
 * Server following a restart. To do this the application can utilize the
 * notification events and update command.
 *
 * Each LC Server instance requires that a Lightness Server is initialized in
 * the element preceding the LC Server element: LC Server controls the Lightness
 * Server residing in the preceding element. Each LC Server instance requires
 * that a generic OnOff Server is initialized in the same element as the LC
 * Server.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_lc_server_init_id                                  0x004d0028
#define sl_btmesh_cmd_lc_server_deinit_id                                0x014d0028
#define sl_btmesh_cmd_lc_server_update_mode_id                           0x024d0028
#define sl_btmesh_cmd_lc_server_update_om_id                             0x034d0028
#define sl_btmesh_cmd_lc_server_update_light_onoff_id                    0x044d0028
#define sl_btmesh_cmd_lc_server_init_all_properties_id                   0x054d0028
#define sl_btmesh_cmd_lc_server_set_publish_mask_id                      0x064d0028
#define sl_btmesh_cmd_lc_server_set_regulator_interval_id                0x074d0028
#define sl_btmesh_cmd_lc_server_set_event_mask_id                        0x084d0028
#define sl_btmesh_cmd_lc_server_get_lc_state_id                          0x094d0028
#define sl_btmesh_rsp_lc_server_init_id                                  0x004d0028
#define sl_btmesh_rsp_lc_server_deinit_id                                0x014d0028
#define sl_btmesh_rsp_lc_server_update_mode_id                           0x024d0028
#define sl_btmesh_rsp_lc_server_update_om_id                             0x034d0028
#define sl_btmesh_rsp_lc_server_update_light_onoff_id                    0x044d0028
#define sl_btmesh_rsp_lc_server_init_all_properties_id                   0x054d0028
#define sl_btmesh_rsp_lc_server_set_publish_mask_id                      0x064d0028
#define sl_btmesh_rsp_lc_server_set_regulator_interval_id                0x074d0028
#define sl_btmesh_rsp_lc_server_set_event_mask_id                        0x084d0028
#define sl_btmesh_rsp_lc_server_get_lc_state_id                          0x094d0028

/**
 * @brief These values define the possible states of Light Controller.
 */
typedef enum
{
  sl_btmesh_lc_server_lc_state_off                 = 0x0, /**< (0x0) The
                                                               controller is
                                                               turned off and
                                                               does not control
                                                               lighting. */
  sl_btmesh_lc_server_lc_state_standby             = 0x1, /**< (0x1) The
                                                               controller is
                                                               turned on and
                                                               awaits an event
                                                               from an occupancy
                                                               sensor or a
                                                               manual switch. */
  sl_btmesh_lc_server_lc_state_fade_on             = 0x2, /**< (0x2) The
                                                               controller has
                                                               been triggered
                                                               and gradually
                                                               transitions to
                                                               the Run phase,
                                                               gradually dimming
                                                               the lights up.> */
  sl_btmesh_lc_server_lc_state_run                 = 0x3, /**< (0x3) The lights
                                                               are on and the
                                                               timer counts down
                                                               (but may be
                                                               re-triggered by a
                                                               sensor or a
                                                               switch event). */
  sl_btmesh_lc_server_lc_state_fade                = 0x4, /**< (0x4) The Run
                                                               timer has expired
                                                               and the
                                                               controller
                                                               gradually
                                                               transitions to
                                                               the Prolong
                                                               state. */
  sl_btmesh_lc_server_lc_state_prolong             = 0x5, /**< (0x5) The lights
                                                               are at a lower
                                                               level and the
                                                               timer counts down
                                                               (but may be
                                                               re-triggered by a
                                                               sensor or a
                                                               switch event). */
  sl_btmesh_lc_server_lc_state_fade_standby_auto   = 0x6, /**< (0x6) The
                                                               controller
                                                               gradually returns
                                                               to the Standby
                                                               state */
  sl_btmesh_lc_server_lc_state_fade_standby_manual = 0x7  /**< (0x7) The
                                                               controller
                                                               gradually returns
                                                               to the Standby
                                                               state after
                                                               external event. */
} sl_btmesh_lc_server_lc_state_t;

/**
 * @brief These values identify optional diagnostic events that provide more
 * information to the application about LC behavior.
 */
typedef enum
{
  sl_btmesh_lc_server_lc_event_state_updated        = 0x1, /**< (0x1) Event
                                                                reporting LC
                                                                Server state
                                                                machine state
                                                                changes along
                                                                with the
                                                                remaining state
                                                                timer. */
  sl_btmesh_lc_server_lc_event_regulator_debug_info = 0x2  /**< (0x2) Event
                                                                reporting LC
                                                                Server PI
                                                                regulator
                                                                integral term
                                                                and regulator
                                                                output. */
} sl_btmesh_lc_server_lc_debug_events_t;

/**
 * @addtogroup sl_btmesh_evt_lc_server_mode_updated sl_btmesh_evt_lc_server_mode_updated
 * @{
 * @brief LC Mode state has been updated
 *
 * The update could be triggered by a reception of a client message or by an LC
 * Server State Machine action.
 */

/** @brief Identifier of the mode_updated event */
#define sl_btmesh_evt_lc_server_mode_updated_id                          0x004d00a8

/***************************************************************************//**
 * @brief Data structure of the mode_updated event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_lc_server_mode_updated_s
{
  uint16_t destination_address; /**< Message destination address if triggered by
                                     a message, 0 otherwise. */
  uint16_t elem_index;          /**< Index of the element where the update
                                     happened */
  uint16_t client_address;      /**< Message source address if triggered by a
                                     message, 0 otherwise. */
  uint16_t appkey_index;        /**< Message appkey index if triggered by a
                                     message, 0xFFFF otherwise. */
  uint8_t  mode_value;          /**< The value the LC Mode state is being set
                                     to. */
  uint8_t  manual_override;     /**< Light Control Mode state can be set to zero
                                     by a binding from Light Lightness Linear
                                     when it is modified by an action from the
                                     application or a Light Lightness Client
                                     command. In this case, the parameter is set
                                     to 0x01. In all other cases, this parameter
                                     is zero. For example, when LC Mode is
                                     modified by the application or by a LC
                                     Client command, this parameter will be set
                                     to 0. */
});

typedef struct sl_btmesh_evt_lc_server_mode_updated_s sl_btmesh_evt_lc_server_mode_updated_t;

/** @} */ // end addtogroup sl_btmesh_evt_lc_server_mode_updated

/**
 * @addtogroup sl_btmesh_evt_lc_server_om_updated sl_btmesh_evt_lc_server_om_updated
 * @{
 * @brief LC Occupancy Mode state has been updated
 *
 * The update could be triggered by a reception of a client message or by an LC
 * Server State Machine action.
 */

/** @brief Identifier of the om_updated event */
#define sl_btmesh_evt_lc_server_om_updated_id                            0x014d00a8

/***************************************************************************//**
 * @brief Data structure of the om_updated event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_lc_server_om_updated_s
{
  uint16_t destination_address; /**< Message destination address if triggered by
                                     a message, 0 otherwise. */
  uint16_t elem_index;          /**< Index of the element where the update
                                     happened */
  uint16_t client_address;      /**< Message source address if triggered by a
                                     message, 0 otherwise. */
  uint16_t appkey_index;        /**< Message appkey index if triggered by a
                                     message, 0xFFFF otherwise. */
  uint8_t  om_value;            /**< The value the LC Occupancy Mode state is
                                     being set to. */
});

typedef struct sl_btmesh_evt_lc_server_om_updated_s sl_btmesh_evt_lc_server_om_updated_t;

/** @} */ // end addtogroup sl_btmesh_evt_lc_server_om_updated

/**
 * @addtogroup sl_btmesh_evt_lc_server_light_onoff_updated sl_btmesh_evt_lc_server_light_onoff_updated
 * @{
 * @brief LC Light OnOff state has been updated
 *
 * The update could be triggered by a reception of a client message or by an LC
 * Server State Machine action.
 */

/** @brief Identifier of the light_onoff_updated event */
#define sl_btmesh_evt_lc_server_light_onoff_updated_id                   0x024d00a8

/***************************************************************************//**
 * @brief Data structure of the light_onoff_updated event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_lc_server_light_onoff_updated_s
{
  uint16_t destination_address; /**< Message destination address if triggered by
                                     a message, 0 otherwise. */
  uint16_t elem_index;          /**< Index of the element where the update
                                     happened */
  uint16_t source_address;      /**< Message source address if triggered by a
                                     message, 0 otherwise. */
  uint16_t appkey_index;        /**< Message appkey index if triggered by a
                                     message, 0xFFFF otherwise. */
  uint8_t  onoff_state;         /**< The target value of the Light LC Light
                                     OnOff state. */
  uint32_t onoff_trans_time_ms; /**< Amount of time (in milliseconds) the
                                     element will take to transition to the
                                     target state from the present state. */
});

typedef struct sl_btmesh_evt_lc_server_light_onoff_updated_s sl_btmesh_evt_lc_server_light_onoff_updated_t;

/** @} */ // end addtogroup sl_btmesh_evt_lc_server_light_onoff_updated

/**
 * @addtogroup sl_btmesh_evt_lc_server_occupancy_updated sl_btmesh_evt_lc_server_occupancy_updated
 * @{
 * @brief LC Occupancy state has been updated
 *
 * The update could be triggered by a reception of a sensor message or by an LC
 * Server State Machine action.
 */

/** @brief Identifier of the occupancy_updated event */
#define sl_btmesh_evt_lc_server_occupancy_updated_id                     0x034d00a8

/***************************************************************************//**
 * @brief Data structure of the occupancy_updated event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_lc_server_occupancy_updated_s
{
  uint16_t destination_address; /**< Message destination address if triggered by
                                     a message, 0 otherwise. */
  uint16_t elem_index;          /**< Index of the element where the update
                                     happened */
  uint16_t source_address;      /**< Message source address if triggered by a
                                     message, 0 otherwise. */
  uint16_t appkey_index;        /**< Message appkey index if triggered by a
                                     message, 0xFFFF otherwise. */
  uint8_t  occupancy_value;     /**< The updated value of the LC Occupancy state */
});

typedef struct sl_btmesh_evt_lc_server_occupancy_updated_s sl_btmesh_evt_lc_server_occupancy_updated_t;

/** @} */ // end addtogroup sl_btmesh_evt_lc_server_occupancy_updated

/**
 * @addtogroup sl_btmesh_evt_lc_server_ambient_lux_level_updated sl_btmesh_evt_lc_server_ambient_lux_level_updated
 * @{
 * @brief LC Ambient LuxLevel state has been updated
 *
 * The update is triggered by a reception of a sensor message.
 */

/** @brief Identifier of the ambient_lux_level_updated event */
#define sl_btmesh_evt_lc_server_ambient_lux_level_updated_id             0x044d00a8

/***************************************************************************//**
 * @brief Data structure of the ambient_lux_level_updated event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_lc_server_ambient_lux_level_updated_s
{
  uint16_t destination_address;     /**< Message destination address */
  uint16_t elem_index;              /**< Index of the element where the update
                                         happened */
  uint16_t source_address;          /**< Message source address */
  uint16_t appkey_index;            /**< Message appkey index */
  uint32_t ambient_lux_level_value; /**< The updated value of the LC Ambient
                                         LuxLevel state */
});

typedef struct sl_btmesh_evt_lc_server_ambient_lux_level_updated_s sl_btmesh_evt_lc_server_ambient_lux_level_updated_t;

/** @} */ // end addtogroup sl_btmesh_evt_lc_server_ambient_lux_level_updated

/**
 * @addtogroup sl_btmesh_evt_lc_server_linear_output_updated sl_btmesh_evt_lc_server_linear_output_updated
 * @{
 * @brief LC Linear Output state has been updated
 *
 * The update is triggered by an LC Server State Machine action.
 */

/** @brief Identifier of the linear_output_updated event */
#define sl_btmesh_evt_lc_server_linear_output_updated_id                 0x054d00a8

/***************************************************************************//**
 * @brief Data structure of the linear_output_updated event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_lc_server_linear_output_updated_s
{
  uint16_t elem_index;          /**< Index of the element where the update
                                     happened */
  uint16_t linear_output_value; /**< The updated value of the LC Linear Output
                                     state */
});

typedef struct sl_btmesh_evt_lc_server_linear_output_updated_s sl_btmesh_evt_lc_server_linear_output_updated_t;

/** @} */ // end addtogroup sl_btmesh_evt_lc_server_linear_output_updated

/**
 * @addtogroup sl_btmesh_evt_lc_server_state_updated sl_btmesh_evt_lc_server_state_updated
 * @{
 * @brief LC state machine state has been updated
 *
 * The update is triggered by LC mode switching on or off and transitions
 * between phases.
 */

/** @brief Identifier of the state_updated event */
#define sl_btmesh_evt_lc_server_state_updated_id                         0x064d00a8

/***************************************************************************//**
 * @brief Data structure of the state_updated event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_lc_server_state_updated_s
{
  uint16_t elem_index;      /**< Index of the element where the update happened */
  uint8_t  state;           /**< The updated value of the LC state */
  uint32_t transition_time; /**< Transition time defined for the current LC
                                 state. */
});

typedef struct sl_btmesh_evt_lc_server_state_updated_s sl_btmesh_evt_lc_server_state_updated_t;

/** @} */ // end addtogroup sl_btmesh_evt_lc_server_state_updated

/**
 * @addtogroup sl_btmesh_evt_lc_server_regulator_debug_info sl_btmesh_evt_lc_server_regulator_debug_info
 * @{
 * @brief LC regulator calculation details
 */

/** @brief Identifier of the regulator_debug_info event */
#define sl_btmesh_evt_lc_server_regulator_debug_info_id                  0x074d00a8

/***************************************************************************//**
 * @brief Data structure of the regulator_debug_info event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_lc_server_regulator_debug_info_s
{
  uint16_t elem_index; /**< Index of the element where LC server is located */
  uint16_t i;          /**< Integral term */
  uint16_t l;          /**< Regulator output */
});

typedef struct sl_btmesh_evt_lc_server_regulator_debug_info_s sl_btmesh_evt_lc_server_regulator_debug_info_t;

/** @} */ // end addtogroup sl_btmesh_evt_lc_server_regulator_debug_info

/***************************************************************************//**
 *
 * Initialize the LC Server model. The server does not have any internal
 * configuration. The command only activates the model in the mesh stack.
 *
 * Each LC Server instance requires that a Lightness Server is initialized in
 * the element preceding the LC Server element: LC Server controls the Lightness
 * Server residing in the preceding element. Each LC Server instance requires
 * that a generic OnOff Server is initialized in the same element as the LC
 * Server.
 *
 * LC properties are initialized as follows:
 *
 * PropertyID: PropertyValue 0x002B: 0x111111, 0x002C: 0x011111, 0x002D:
 * 0x001111, 0x002E: 0xf000, 0x002F: 0x0f00, 0x0030: 0x00f0, 0x031: 50, 0x032:
 * 25.0, 0x0033: 250.0, 0x0034: 80.0, 0x0035: 80.0, 0x0036: 3000, 0x0037: 3000,
 * 0x0038: 3000, 0x0039: 3000, 0x003A: 0, 0x003B: 3000, 0x003C: 3000
 *
 * PI Regulator interval (T) is initialized to 50ms
 *
 * The rest of the state values are initialized to zero
 *
 * @param[in] elem_index Index of the element.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lc_server_init(uint16_t elem_index);

/***************************************************************************//**
 *
 * De-initializes the LC Server model.
 *
 * @param[in] elem_index Index of the element.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lc_server_deinit(uint16_t elem_index);

/***************************************************************************//**
 *
 * Update the LC Server model Mode state in the stack. Application may choose to
 * directly set the model state in the stack, this function will pass the state
 * value to the LC Server model.
 *
 * @param[in] elem_index Index of the element.
 * @param[in] mode Mode value. Valid range: 0-1.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lc_server_update_mode(uint16_t elem_index, uint8_t mode);

/***************************************************************************//**
 *
 * Update the LC Server model Occupancy Mode state in the stack. The application
 * may choose to directly set the model state in the stack. This function will
 * pass the state value to the LC Server model.
 *
 * @param[in] elem_index Index of the element.
 * @param[in] om Occupancy Mode value. Valid range: 0-1.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lc_server_update_om(uint16_t elem_index, uint8_t om);

/***************************************************************************//**
 *
 * Update the LC Server model Light OnOff state in the stack. The application
 * may choose to directly set the model state in the stack. This function will
 * pass the state value to the LC Server model.
 *
 * @param[in] elem_index Index of the element.
 * @param[in] light_onoff Light OnOff value. Valid range: 0-1.
 * @param[in] transition_time_ms Amount of time (in milliseconds) that the
 *   element will take to transition to the target state from the present state.
 *   If set to 0, the transition will be immediate.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lc_server_update_light_onoff(uint16_t elem_index,
                                                   uint8_t light_onoff,
                                                   uint32_t transition_time_ms);

/***************************************************************************//**
 *
 * Initialize all LC properties in one attempt. The following values are used:
 *
 * PropertyID: PropertyValue 0x002B: 0x111111, 0x002C: 0x011111, 0x002D:
 * 0x001111, 0x002E: 0xf000, 0x002F: 0x0f00, 0x0030: 0x00f0, 0x031: 50, 0x032:
 * 25.0, 0x0033: 250.0, 0x0034: 80.0, 0x0035: 80.0, 0x0036: 3000, 0x0037: 3000,
 * 0x0038: 3000, 0x0039: 3000, 0x003A: 0, 0x003B: 3000, 0x003C: 3000
 *
 * @param[in] elem_index Index of the element.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lc_server_init_all_properties(uint16_t elem_index);

/***************************************************************************//**
 *
 * Update the bitmask that controls which messages are sent when the LC Server
 * publishes. By default, the bitmask will be enabled to publish all three
 * status messages.
 *
 * @param[in] elem_index Index of the element.
 * @param[in] status_type @parblock
 *   The type of status message to turn on/off. Options for this are:
 *
 *   LC Mode Status 0x8294 LC Occupancy Mode Status 0x8298 LC Light On Off
 *   Status 0x829C
 *   @endparblock
 * @param[in] value Turn on or off the status message. Non zero - enable,
 *   otherwise disable.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lc_server_set_publish_mask(uint16_t elem_index,
                                                 uint16_t status_type,
                                                 uint8_t value);

/***************************************************************************//**
 *
 * Update the summation interval (T) at which the PI regulator is run. Only
 * valid when the regulator is disabled (Light LC Mode is 0).
 *
 * @param[in] elem_index Index of the element.
 * @param[in] value Valid values are 1 ms-100 ms. (Default: 50 ms)
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lc_server_set_regulator_interval(uint16_t elem_index,
                                                       uint8_t value);

/***************************************************************************//**
 *
 * Enable or disable additional diagnostics events. See lc_debug_events.
 *
 * @param[in] elem_index Index of the element.
 * @param[in] event_type @parblock
 *   The type of event to enable/disable. Options are:
 *
 *   lc_event_state_updated = 0x01, state_updated event report state changes
 *   lc_event_regulator_debug_info = 0x02, regulator_debug_info Regulator
 *   calculation details
 *   @endparblock
 * @param[in] value Valid values are 0 and 1 to disable or enable the event
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lc_server_set_event_mask(uint16_t elem_index,
                                               uint16_t event_type,
                                               uint8_t value);

/***************************************************************************//**
 *
 * Fetch the current LC state. States can be as Off, Standby, Fade On, Run,
 * Fade, Prolong, Fade Standby Auto, Fade Standby Manual
 *
 * @param[in] elem_index Index of the element.
 * @param[out] state The current state of LC state machine
 * @param[out] transition_time Transition time left for the current LC state.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lc_server_get_lc_state(uint16_t elem_index,
                                             uint8_t *state,
                                             uint32_t *transition_time);

/** @} */ // end addtogroup sl_btmesh_lc_server

/**
 * @addtogroup sl_btmesh_lc_setup_server Bluetooth Mesh Light Control Setup Server Model
 * @{
 *
 * @brief Bluetooth Mesh Light Control Setup Server Model
 *
 * Bluetooth Mesh Light Control model Setup Server functionality.
 *
 * This class provides the API that the LC Setup server uses to inform the
 * application of its received events. The API is informational: application is
 * not required to react to these events.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_lc_setup_server_update_property_id                 0x004e0028
#define sl_btmesh_rsp_lc_setup_server_update_property_id                 0x004e0028

/**
 * @addtogroup sl_btmesh_evt_lc_setup_server_set_property sl_btmesh_evt_lc_setup_server_set_property
 * @{
 * @brief LC Property Set from the Client
 */

/** @brief Identifier of the set_property event */
#define sl_btmesh_evt_lc_setup_server_set_property_id                    0x004e00a8

/***************************************************************************//**
 * @brief Data structure of the set_property event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_lc_setup_server_set_property_s
{
  uint16_t   destination_address; /**< Address of the server or the group
                                       address to which it was published. */
  uint16_t   elem_index;          /**< Index of the element which received the
                                       command. */
  uint16_t   client_address;      /**< Device which sent the request. */
  uint16_t   appkey_index;        /**< Appkey used by client_address. */
  uint16_t   property_id;         /**< Property ID */
  uint8array property_value;      /**< Property value */
});

typedef struct sl_btmesh_evt_lc_setup_server_set_property_s sl_btmesh_evt_lc_setup_server_set_property_t;

/** @} */ // end addtogroup sl_btmesh_evt_lc_setup_server_set_property

/***************************************************************************//**
 *
 * Update the LC Server property. The application may choose to directly set
 * model properties in the stack. This function will pass the property value to
 * the LC Setup Server and on to the LC Server model.
 *
 * @param[in] elem_index Client model element index
 * @param[in] property_id Property ID for the LC Server. Range: 0x0001 - 0x0ffff
 *   for a specific device property, the value 0x0000 is prohibited.
 * @param[in] params_len Length of data in @p params
 * @param[in] params Byte array containing serialized fields of LC Property,
 *   excluding the property ID
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_lc_setup_server_update_property(uint16_t elem_index,
                                                      uint16_t property_id,
                                                      size_t params_len,
                                                      const uint8_t* params);

/** @} */ // end addtogroup sl_btmesh_lc_setup_server

/**
 * @addtogroup sl_btmesh_scene_client Bluetooth Mesh Scene Client Model
 * @{
 *
 * @brief Bluetooth Mesh Scene Client Model
 *
 * Bluetooth Mesh Scene Client model functionality to send and receive messages
 * to/from the Scene Server and Scene Setup Server models.
 *
 * Throughout the API, the client model that is used is identified by its
 * element address and model ID, while the server model responding to the client
 * model requests is identified by its element address and model ID.
 *
 * The API has functions for querying server model states and requesting server
 * model state changes.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_scene_client_init_id                               0x004f0028
#define sl_btmesh_cmd_scene_client_get_id                                0x014f0028
#define sl_btmesh_cmd_scene_client_get_register_id                       0x024f0028
#define sl_btmesh_cmd_scene_client_recall_id                             0x034f0028
#define sl_btmesh_cmd_scene_client_store_id                              0x044f0028
#define sl_btmesh_cmd_scene_client_delete_id                             0x054f0028
#define sl_btmesh_cmd_scene_client_deinit_id                             0x064f0028
#define sl_btmesh_rsp_scene_client_init_id                               0x004f0028
#define sl_btmesh_rsp_scene_client_get_id                                0x014f0028
#define sl_btmesh_rsp_scene_client_get_register_id                       0x024f0028
#define sl_btmesh_rsp_scene_client_recall_id                             0x034f0028
#define sl_btmesh_rsp_scene_client_store_id                              0x044f0028
#define sl_btmesh_rsp_scene_client_delete_id                             0x054f0028
#define sl_btmesh_rsp_scene_client_deinit_id                             0x064f0028

/**
 * @addtogroup sl_btmesh_evt_scene_client_status sl_btmesh_evt_scene_client_status
 * @{
 * @brief Event indicating an incoming Scene Status message, response to @ref
 * sl_btmesh_scene_client_get or @ref sl_btmesh_scene_client_recall command or
 * unsolicited message
 */

/** @brief Identifier of the status event */
#define sl_btmesh_evt_scene_client_status_id                             0x004f00a8

/***************************************************************************//**
 * @brief Data structure of the status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_scene_client_status_s
{
  uint16_t destination_address; /**< Address of the client or the group address
                                     to which it was published. */
  uint16_t elem_index;          /**< Index of the element which received the
                                     status. */
  uint16_t server_address;      /**< Device which sent the status. */
  uint16_t appkey_index;        /**< Appkey used by client_address. */
  uint8_t  status;              /**< Status of the request. */
  uint16_t current_scene;       /**< Currently selected scene. */
  uint16_t target_scene;        /**< Scene to be transitioning to. */
  uint32_t remaining_time_ms;   /**< Time (in milliseconds) remaining in
                                     transition. */
});

typedef struct sl_btmesh_evt_scene_client_status_s sl_btmesh_evt_scene_client_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_scene_client_status

/**
 * @addtogroup sl_btmesh_evt_scene_client_register_status sl_btmesh_evt_scene_client_register_status
 * @{
 * @brief Event indicating an incoming Scene Register Status message, response
 * to @ref sl_btmesh_scene_client_get_register or @ref
 * sl_btmesh_scene_client_store or @ref sl_btmesh_scene_client_delete command
 */

/** @brief Identifier of the register_status event */
#define sl_btmesh_evt_scene_client_register_status_id                    0x014f00a8

/***************************************************************************//**
 * @brief Data structure of the register_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_scene_client_register_status_s
{
  uint16_t   destination_address; /**< Address of the client or the group
                                       address to which it was published. */
  uint16_t   elem_index;          /**< Index of the element for which received
                                       the status. */
  uint16_t   server_address;      /**< Device which sent the status. */
  uint16_t   appkey_index;        /**< Appkey used by client_address. */
  uint8_t    status;              /**< Status of the request. */
  uint16_t   current_scene;       /**< Currently selected scene. */
  uint8array scenes;              /**< List of stored scenes on the server
                                       derived from the Scene Register. It
                                       contains a variable length array of
                                       16-bit values representing Scene Numbers
                                       in little endian format. */
});

typedef struct sl_btmesh_evt_scene_client_register_status_s sl_btmesh_evt_scene_client_register_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_scene_client_register_status

/***************************************************************************//**
 *
 * Initialize the Scene Client model. The Scene Client does not have any
 * internal configuration. It only activates the model in the mesh stack.
 *
 * @param[in] elem_index Index of the client element.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_scene_client_init(uint16_t elem_index);

/***************************************************************************//**
 *
 * Scene Get command.
 *
 * @param[in] server_address Device to be queried. The address 0x0000 can be
 *   used to publish the message according to model configuration
 * @param[in] elem_index Index of the client element.
 * @param[in] appkey_index Appkey used by server_address.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_scene_client_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_scene_client_get(uint16_t server_address,
                                       uint16_t elem_index,
                                       uint16_t appkey_index);

/***************************************************************************//**
 *
 * Scene Register Get command
 *
 * @param[in] server_address Destination server model address. The address
 *   0x0000 can be used to publish the message according to model configuration
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_scene_client_register_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_scene_client_get_register(uint16_t server_address,
                                                uint16_t elem_index,
                                                uint16_t appkey_index);

/***************************************************************************//**
 *
 * Recall a scene.
 *
 * @param[in] server_address Destination server model address. The address
 *   0x0000 can be used to publish the message according to model configuration
 * @param[in] elem_index Index of the client element.
 * @param[in] selected_scene Scene of interest
 * @param[in] appkey_index Appkey used by server_address.
 * @param[in] flags Bit 1 (0x02) defines whether response is required.
 *   If set to 1, SET PROPERTY message will be sent, zero will send SET PROPERTY
 *   UNACKNOWLEDGED
 * @param[in] tid Transaction ID
 * @param[in] transition_time_ms Amount of time (in milliseconds) allotted for
 *   the transition to take place. Value of 0xFFFFFFFF will cause this parameter
 *   as well as the "delay" parameter to be omitted. The transition will be
 *   immediate if both the transition time and the delay are zero.
 * @param[in] delay_ms Message execution delay in milliseconds. If the
 *   "transition_time" is 0xFFFFFFFF, this parameter is ignored. If both the
 *   transition time and the delay are zero, the transition is immediate. Valid
 *   range: 0-1275.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_scene_client_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_scene_client_recall(uint16_t server_address,
                                          uint16_t elem_index,
                                          uint16_t selected_scene,
                                          uint16_t appkey_index,
                                          uint8_t flags,
                                          uint8_t tid,
                                          uint32_t transition_time_ms,
                                          uint16_t delay_ms);

/***************************************************************************//**
 *
 * Store a scene.
 *
 * @param[in] server_address Destination server model address. The address
 *   0x0000 can be used to publish the message according to the model
 *   configuration.
 * @param[in] elem_index Client model element index
 * @param[in] selected_scene Scene of interest
 * @param[in] appkey_index The application key index to use.
 * @param[in] flags Bit 1 (0x02) defines whether response is required.
 *   If set to 1, SET PROPERTY message will be sent. Zero will send SET PROPERTY
 *   UNACKNOWLEDGED.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_scene_client_register_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_scene_client_store(uint16_t server_address,
                                         uint16_t elem_index,
                                         uint16_t selected_scene,
                                         uint16_t appkey_index,
                                         uint8_t flags);

/***************************************************************************//**
 *
 * Delete a scene.
 *
 * @param[in] server_address Destination server model address. The address
 *   0x0000 can be used to publish the message according to the model
 *   configuration.
 * @param[in] elem_index Client model element index
 * @param[in] selected_scene Scene of interest
 * @param[in] appkey_index The application key index to use.
 * @param[in] flags Bit 1 (0x02) defines whether response is required.
 *   If set to 1, SET PROPERTY message will be sent. Zero will send SET PROPERTY
 *   UNACKNOWLEDGED.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_scene_client_register_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_scene_client_delete(uint16_t server_address,
                                          uint16_t elem_index,
                                          uint16_t selected_scene,
                                          uint16_t appkey_index,
                                          uint8_t flags);

/***************************************************************************//**
 *
 * Deinitialize the Scene Client model. This deactivates the model in the mesh
 * stack.
 *
 * @param[in] elem_index Index of the client element.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_scene_client_deinit(uint16_t elem_index);

/** @} */ // end addtogroup sl_btmesh_scene_client

/**
 * @addtogroup sl_btmesh_scene_server Bluetooth Mesh Scene Server Model
 * @{
 *
 * @brief Bluetooth Mesh Scene Server Model
 *
 * Bluetooth Mesh Scene model Server functionality.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_scene_server_init_id                               0x00500028
#define sl_btmesh_cmd_scene_server_deinit_id                             0x01500028
#define sl_btmesh_cmd_scene_server_reset_register_id                     0x02500028
#define sl_btmesh_cmd_scene_server_enable_compact_recall_events_id       0x03500028
#define sl_btmesh_rsp_scene_server_init_id                               0x00500028
#define sl_btmesh_rsp_scene_server_deinit_id                             0x01500028
#define sl_btmesh_rsp_scene_server_reset_register_id                     0x02500028
#define sl_btmesh_rsp_scene_server_enable_compact_recall_events_id       0x03500028

/**
 * @addtogroup sl_btmesh_evt_scene_server_get sl_btmesh_evt_scene_server_get
 * @{
 * @brief Get the status.
 */

/** @brief Identifier of the get event */
#define sl_btmesh_evt_scene_server_get_id                                0x015000a8

/***************************************************************************//**
 * @brief Data structure of the get event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_scene_server_get_s
{
  uint16_t destination_address; /**< Address of the server or the group address
                                     to which it was published. */
  uint16_t elem_index;          /**< Index of the element which received the
                                     command. */
  uint16_t client_address;      /**< Device which sent the request. */
  uint16_t appkey_index;        /**< Appkey used by client_address. */
});

typedef struct sl_btmesh_evt_scene_server_get_s sl_btmesh_evt_scene_server_get_t;

/** @} */ // end addtogroup sl_btmesh_evt_scene_server_get

/**
 * @addtogroup sl_btmesh_evt_scene_server_register_get sl_btmesh_evt_scene_server_register_get
 * @{
 * @brief Get the status of a register.
 */

/** @brief Identifier of the register_get event */
#define sl_btmesh_evt_scene_server_register_get_id                       0x025000a8

/***************************************************************************//**
 * @brief Data structure of the register_get event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_scene_server_register_get_s
{
  uint16_t destination_address; /**< Address of the server or the group address
                                     to which it was published. */
  uint16_t elem_index;          /**< Index of the element which received the
                                     command. */
  uint16_t client_address;      /**< Device which sent the request. */
  uint16_t appkey_index;        /**< Appkey used by client_address. */
});

typedef struct sl_btmesh_evt_scene_server_register_get_s sl_btmesh_evt_scene_server_register_get_t;

/** @} */ // end addtogroup sl_btmesh_evt_scene_server_register_get

/**
 * @addtogroup sl_btmesh_evt_scene_server_recall sl_btmesh_evt_scene_server_recall
 * @{
 * @brief Recall a scene.
 */

/** @brief Identifier of the recall event */
#define sl_btmesh_evt_scene_server_recall_id                             0x035000a8

/***************************************************************************//**
 * @brief Data structure of the recall event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_scene_server_recall_s
{
  uint16_t destination_address; /**< Address of the server or the group address
                                     to which it was published. */
  uint16_t elem_index;          /**< Index of the element which received the
                                     command. */
  uint16_t selected_scene;      /**< Scene of interest */
  uint16_t client_address;      /**< Device which sent the request. */
  uint16_t appkey_index;        /**< Appkey used by client_address. */
  uint32_t transition_time_ms;  /**< Time (in milliseconds) allotted for the
                                     transition to take place */
});

typedef struct sl_btmesh_evt_scene_server_recall_s sl_btmesh_evt_scene_server_recall_t;

/** @} */ // end addtogroup sl_btmesh_evt_scene_server_recall

/**
 * @addtogroup sl_btmesh_evt_scene_server_publish sl_btmesh_evt_scene_server_publish
 * @{
 * @brief Indicate that the publishing period timer elapsed and the app
 * should/can publish its state or any request.
 */

/** @brief Identifier of the publish event */
#define sl_btmesh_evt_scene_server_publish_id                            0x045000a8

/***************************************************************************//**
 * @brief Data structure of the publish event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_scene_server_publish_s
{
  uint16_t elem_index; /**< Client model element index */
  uint32_t period_ms;  /**< The current publishing period that can be used to
                            estimate the next tick, e.g., when the state should
                            be reported at higher frequency. */
});

typedef struct sl_btmesh_evt_scene_server_publish_s sl_btmesh_evt_scene_server_publish_t;

/** @} */ // end addtogroup sl_btmesh_evt_scene_server_publish

/**
 * @addtogroup sl_btmesh_evt_scene_server_compact_recall sl_btmesh_evt_scene_server_compact_recall
 * @{
 * @brief Recall a scene.
 */

/** @brief Identifier of the compact_recall event */
#define sl_btmesh_evt_scene_server_compact_recall_id                     0x055000a8

/***************************************************************************//**
 * @brief Data structure of the compact_recall event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_scene_server_compact_recall_s
{
  uint8array states; /**< Byte array containind recalled states. Array consist
                          of 5 byte chunks as follows:

                            - Element id as 16bit unsigned integer in little
                              endian format
                            - Model id as 16bit unsigned integer in little
                              endian format
                            - Model-specific state type, identifying the kind of
                              state recalled See get state types list for
                              details.

                          after this event application can request recalled
                          states with @ref
                          sl_btmesh_generic_server_get_cached_state command */
});

typedef struct sl_btmesh_evt_scene_server_compact_recall_s sl_btmesh_evt_scene_server_compact_recall_t;

/** @} */ // end addtogroup sl_btmesh_evt_scene_server_compact_recall

/***************************************************************************//**
 *
 * Initialize the Scene Server model. Server does not have any internal
 * configurations. The command only activates the model in the mesh stack.
 *
 * @param[in] elem_index Index of the element.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_scene_server_init(uint16_t elem_index);

/***************************************************************************//**
 *
 * De-initializes the Scene Server model.
 *
 * @param[in] elem_index Index of the element.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_scene_server_deinit(uint16_t elem_index);

/***************************************************************************//**
 *
 * Reset register value. This command should be invoked if state of a model has
 * been modified in such a manner that it can't be in the scene, which is
 * indicated by the scene register.
 *
 * @param[in] elem_index Index of the element. This can be either element of the
 *   updated model or the element of the scene model responsible for controlling
 *   the scene of the updated model.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_scene_server_reset_register(uint16_t elem_index);

/***************************************************************************//**
 *
 * Switch to compact reporting for recalled states. Compact state reduces amount
 * buffering memory needed by the scene recall and is recommended for devices
 * with big amount of models or for devices in environment with lots of
 * bluetooth advertisement traffic.
 *
 * When compact mode is active @ref sl_btmesh_evt_scene_server_compact_recall is
 * generated instead of several @ref sl_btmesh_evt_generic_server_state_recall
 * events.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_scene_server_enable_compact_recall_events();

/** @} */ // end addtogroup sl_btmesh_scene_server

/**
 * @addtogroup sl_btmesh_scene_setup_server Bluetooth Mesh Scene Setup Server Model
 * @{
 *
 * @brief Bluetooth Mesh Scene Setup Server Model
 *
 * Bluetooth Mesh Scene model Setup Server functionality.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_scene_setup_server_init_id                         0x00510028
#define sl_btmesh_cmd_scene_setup_server_deinit_id                       0x01510028
#define sl_btmesh_rsp_scene_setup_server_init_id                         0x00510028
#define sl_btmesh_rsp_scene_setup_server_deinit_id                       0x01510028

/**
 * @addtogroup sl_btmesh_evt_scene_setup_server_store sl_btmesh_evt_scene_setup_server_store
 * @{
 * @brief Scene Store from the Client
 */

/** @brief Identifier of the store event */
#define sl_btmesh_evt_scene_setup_server_store_id                        0x005100a8

/***************************************************************************//**
 * @brief Data structure of the store event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_scene_setup_server_store_s
{
  uint16_t destination_address; /**< Address of the server or the group address
                                     to which it was published. */
  uint16_t elem_index;          /**< Index of the element which received the
                                     command. */
  uint16_t scene_id;            /**< Scene ID */
  uint16_t client_address;      /**< Device which sent the request. */
  uint16_t appkey_index;        /**< Appkey used by client_address. */
});

typedef struct sl_btmesh_evt_scene_setup_server_store_s sl_btmesh_evt_scene_setup_server_store_t;

/** @} */ // end addtogroup sl_btmesh_evt_scene_setup_server_store

/**
 * @addtogroup sl_btmesh_evt_scene_setup_server_delete sl_btmesh_evt_scene_setup_server_delete
 * @{
 * @brief Scene Delete from the Client
 */

/** @brief Identifier of the delete event */
#define sl_btmesh_evt_scene_setup_server_delete_id                       0x015100a8

/***************************************************************************//**
 * @brief Data structure of the delete event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_scene_setup_server_delete_s
{
  uint16_t destination_address; /**< Address of the server or the group address
                                     to which it was published. */
  uint16_t elem_index;          /**< Index of the element which received the
                                     command. */
  uint16_t scene_id;            /**< Scene ID */
  uint16_t client_address;      /**< Device which sent the request. */
  uint16_t appkey_index;        /**< Appkey used by client_address. */
});

typedef struct sl_btmesh_evt_scene_setup_server_delete_s sl_btmesh_evt_scene_setup_server_delete_t;

/** @} */ // end addtogroup sl_btmesh_evt_scene_setup_server_delete

/**
 * @addtogroup sl_btmesh_evt_scene_setup_server_publish sl_btmesh_evt_scene_setup_server_publish
 * @{
 * @brief Indicate that the publishing period timer elapsed and the app
 * should/can publish its state or any request.
 */

/** @brief Identifier of the publish event */
#define sl_btmesh_evt_scene_setup_server_publish_id                      0x025100a8

/***************************************************************************//**
 * @brief Data structure of the publish event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_scene_setup_server_publish_s
{
  uint16_t elem_index; /**< Client model element index */
  uint32_t period_ms;  /**< The current publishing period that can be used to
                            estimate the next tick, e.g., when the state should
                            be reported at higher frequency. */
});

typedef struct sl_btmesh_evt_scene_setup_server_publish_s sl_btmesh_evt_scene_setup_server_publish_t;

/** @} */ // end addtogroup sl_btmesh_evt_scene_setup_server_publish

/***************************************************************************//**
 *
 * Initialize the Scene Setup Server model. Server does not have any internal
 * configurations. The command only activates the model in the mesh stack.
 *
 * @param[in] elem_index Index of the element.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_scene_setup_server_init(uint16_t elem_index);

/***************************************************************************//**
 *
 * Deinitialize the Scene Setup Server model. This deactivates the model in the
 * mesh stack.
 *
 * @param[in] elem_index Index of the element.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_scene_setup_server_deinit(uint16_t elem_index);

/** @} */ // end addtogroup sl_btmesh_scene_setup_server

/**
 * @addtogroup sl_btmesh_scheduler_client Bluetooth Mesh Scheduler Client Model
 * @{
 *
 * @brief Bluetooth Mesh Scheduler Client Model
 *
 * This class provides commands and messages to interface with the Scheduler
 * Client model. Scheduler models uses multiple fields to define the occurrence
 * of an event and the type of event to be triggered.
 *
 * For the description of these fields, please see <a
 * href="#cls_mesh_scheduler_server">Scheduler Server</a>
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_scheduler_client_init_id                           0x00540028
#define sl_btmesh_cmd_scheduler_client_deinit_id                         0x01540028
#define sl_btmesh_cmd_scheduler_client_get_id                            0x02540028
#define sl_btmesh_cmd_scheduler_client_get_action_id                     0x03540028
#define sl_btmesh_cmd_scheduler_client_set_action_id                     0x04540028
#define sl_btmesh_rsp_scheduler_client_init_id                           0x00540028
#define sl_btmesh_rsp_scheduler_client_deinit_id                         0x01540028
#define sl_btmesh_rsp_scheduler_client_get_id                            0x02540028
#define sl_btmesh_rsp_scheduler_client_get_action_id                     0x03540028
#define sl_btmesh_rsp_scheduler_client_set_action_id                     0x04540028

/**
 * @addtogroup sl_btmesh_evt_scheduler_client_status sl_btmesh_evt_scheduler_client_status
 * @{
 * @brief Scheduler Status event, received as a response to the @ref
 * sl_btmesh_scheduler_client_get command.
 */

/** @brief Identifier of the status event */
#define sl_btmesh_evt_scheduler_client_status_id                         0x005400a8

/***************************************************************************//**
 * @brief Data structure of the status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_scheduler_client_status_s
{
  uint16_t client_address; /**< Destination client model address */
  uint16_t elem_index;     /**< Client model element index */
  uint16_t server_address; /**< Source server model address */
  uint16_t appkey_index;   /**< The application key index to use. */
  uint16_t scheduler;      /**< Bit field indicating defined Actions in the
                                Schedule Register */
});

typedef struct sl_btmesh_evt_scheduler_client_status_s sl_btmesh_evt_scheduler_client_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_scheduler_client_status

/**
 * @addtogroup sl_btmesh_evt_scheduler_client_action_status sl_btmesh_evt_scheduler_client_action_status
 * @{
 * @brief Scheduler Action Status event, received as a response to the @ref
 * sl_btmesh_scheduler_client_set_action or @ref
 * sl_btmesh_scheduler_client_get_action command
 *
 * For the description of these fields, see <a
 * href="#cls_mesh_scheduler_server">Scheduler Server</a>
 */

/** @brief Identifier of the action_status event */
#define sl_btmesh_evt_scheduler_client_action_status_id                  0x015400a8

/***************************************************************************//**
 * @brief Data structure of the action_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_scheduler_client_action_status_s
{
  uint16_t client_address;     /**< Destination client model address */
  uint16_t elem_index;         /**< Client model element index */
  uint16_t server_address;     /**< Source server model address */
  uint16_t appkey_index;       /**< The application key index to use. */
  uint8_t  index;              /**< Index of the Scheduler Register */
  uint8_t  year;               /**< Scheduled year for the action */
  uint16_t month;              /**< Scheduled month for the action */
  uint8_t  day;                /**< Scheduled day of the month for the action */
  uint8_t  hour;               /**< Scheduled hour for the action */
  uint8_t  minute;             /**< Scheduled minute for the action */
  uint8_t  second;             /**< Scheduled second for the action */
  uint8_t  day_of_week;        /**< Scheduled days of the week for the action */
  uint8_t  action;             /**< Action to be performed at the scheduled time */
  uint32_t transition_time_ms; /**< Transition time for this action */
  uint16_t scene_number;       /**< Scene number to be used for some actions */
});

typedef struct sl_btmesh_evt_scheduler_client_action_status_s sl_btmesh_evt_scheduler_client_action_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_scheduler_client_action_status

/***************************************************************************//**
 *
 * Initializes the Scheduler Client model
 *
 * @param[in] elem_index Client model element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_scheduler_client_init(uint16_t elem_index);

/***************************************************************************//**
 *
 * Deinitialize the Scheduler Client model
 *
 * @param[in] elem_index Client model element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_scheduler_client_deinit(uint16_t elem_index);

/***************************************************************************//**
 *
 * Send a Scheduler Get message to get the Scheduler Register status of a
 * Scheduler Server.
 *
 * @param[in] server_address Destination server model address
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_scheduler_client_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_scheduler_client_get(uint16_t server_address,
                                           uint16_t elem_index,
                                           uint16_t appkey_index);

/***************************************************************************//**
 *
 * Send a Scheduler Action Get message to get action defined by a Schedule
 * Register state entry.
 *
 * @param[in] server_address Destination server model address
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use.
 * @param[in] index Index of the Scheduler Register entry to get
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_scheduler_client_action_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_scheduler_client_get_action(uint16_t server_address,
                                                  uint16_t elem_index,
                                                  uint16_t appkey_index,
                                                  uint8_t index);

/***************************************************************************//**
 *
 * Send a Scheduler Action Set message to set the given entry of the Scheduler
 * Register state.
 *
 * For the description of these fields, see <a
 * href="#cls_mesh_scheduler_server">Scheduler Server</a>
 *
 * @param[in] server_address Destination server model address
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use.
 * @param[in] flags Bit 1 (0x02) defines whether response is required, othewise
 *   the unacknowledged message is used.
 * @param[in] index Index of the Scheduler Register entry to set
 * @param[in] year Scheduled year for the action
 * @param[in] month Scheduled month for the action
 * @param[in] day Scheduled day of the month for the action
 * @param[in] hour Scheduled hour for the action
 * @param[in] minute Scheduled minute for the action
 * @param[in] second Scheduled second for the action
 * @param[in] day_of_week Scheduled days of the week for the action
 * @param[in] action Action to be performed at the scheduled time
 * @param[in] transition_time_ms Transition time for this action
 * @param[in] scene_number Scene number to be used for some actions
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_scheduler_client_action_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_scheduler_client_set_action(uint16_t server_address,
                                                  uint16_t elem_index,
                                                  uint16_t appkey_index,
                                                  uint8_t flags,
                                                  uint8_t index,
                                                  uint8_t year,
                                                  uint16_t month,
                                                  uint8_t day,
                                                  uint8_t hour,
                                                  uint8_t minute,
                                                  uint8_t second,
                                                  uint8_t day_of_week,
                                                  uint8_t action,
                                                  uint32_t transition_time_ms,
                                                  uint16_t scene_number);

/** @} */ // end addtogroup sl_btmesh_scheduler_client

/**
 * @addtogroup sl_btmesh_scheduler_server Bluetooth Mesh Scheduler Server Model
 * @{
 *
 * @brief Bluetooth Mesh Scheduler Server Model
 *
 * This class provides commands and messages to interface with the Scheduler
 * Server model
 *
 * The scheduler server uses multiple fields to define the occurrence of an
 * event and the type of event to be triggered. The field definitions are as
 * follows.
 *
 * <b>Year</b>
 *
 * The year field represents 2 last significant digits of the year of the
 * occurrence of the scheduled event.
 *   - 0x00-0x63: 2 least significant digits of the year (0-99)
 *   - 0x64: Any year
 *   - All other values are prohibited
 *
 * <b>Month</b>
 *
 * The month field represents the months of the occurrences of the scheduled
 * event, using a bitfield with one bit for each month.
 *
 *   - Bit 0: Scheduled in January
 *   - Bit 1: February
 *   - Bit 2: March
 *   - Bit 3: April
 *   - Bit 4: May
 *   - Bit 5: June
 *   - Bit 6: July
 *   - Bit 7: August
 *   - Bit 8: September
 *   - Bit 9: October
 *   - Bit 10: November
 *   - Bit 11: December
 *
 * <b>Day</b>
 *
 * The Day field represents the day of the month of the occurrence of the
 * scheduled event. If the day of the month has a number that is larger than the
 * number of days in the month, the event occurs in the last day of the month.
 * For example, in February if the day field holds the value 29, the action is
 * triggered on February 28th in a non-leap year or February 29th in a leap
 * year.
 *
 *   - 0x00: Any day
 *   - 0x01-0x1F: Day of the month (1-31)
 *   - All other values are prohibited
 *
 * <b>Hour</b>
 *
 * The Hour field represents the hour of the occurrence of the scheduled event.
 *
 *   - 0x00-0x17: Hour of the day (0-23)
 *   - 0x18: Any hour of the day
 *   - 0x19: Once a day (at a random hour)
 *   - All other values are prohibited
 *
 * <b>Minute</b>
 *
 * The Minute field represents the minute of the occurrence of the scheduled
 * event.
 *
 *   - 0x00-0x3B: Minute of the hour (0-59)
 *   - 0x3C: Any minute of the hour
 *   - 0x3D: Every 15 minutes (0, 15, 30, 45)
 *   - 0x3E: Every 20 minutes (0, 20, 40)
 *   - 0x3F: Once an hour (at a random minute)
 *   - All other values are prohibited
 *
 * <b>Second</b>
 *
 * The Second field represents the second of the occurrence of the scheduled
 * event.
 *
 *   - 0x00-0x3B: Seconds of the minute (0-59)
 *   - 0x3C: Any second of the minute
 *   - 0x3D: Every 15 seconds (0, 15, 30, 45)
 *   - 0x3E: Every 20 seconds (0, 20, 40)
 *   - 0x3F: Once a minute (at a random second)
 *   - All other values are prohibited
 *
 * <b>Day of Week</b>
 *
 * The DayOfWeek field represents the days of the week when the scheduled event
 * will trigger. The week days are represented by a bitfield, by one bit for
 * each day.
 *
 *   - Bit 0: Scheduled on Mondays
 *   - Bit 1: Scheduled on Tuesdays
 *   - Bit 2: Scheduled on Wednesdays
 *   - Bit 3: Scheduled on Thursdays
 *   - Bit 4: Scheduled on Fridays
 *   - Bit 5: Scheduled on Saturdays
 *   - Bit 6: Scheduled on Sundays
 *
 * <b>Action</b>
 *
 * The action field represents an action to be executed for a scheduled event
 *
 *   - 0x00: Turn Off
 *   - 0x01: Turn On
 *   - 0x02: Scene Recall
 *   - 0x0F: No action
 *   - All other values are prohibited
 *
 * <b>Transition time</b>
 *
 * This is a 1-octet value that consists of two fields: a 2-bit bit field
 * representing the step resolution and a 6-bit bit field representing the
 * number of transition steps.
 *
 * Bit 0-1: Transition Step Resolution
 *   - 0b00: The Default Transition Step Resolution is 100 milliseconds
 *   - 0b01: 1 second resolution
 *   - 0b10: 10 seconds resolution
 *   - 0b11: 10 minutes resolution
 *
 * Bit 2-7: Transition Number of Steps
 *   - 0x00: The Transition Time is immediate
 *   - 0x01-0x3E: The number of steps
 *   - 0x3F: The value is unknown. The state cannot be set to this value, but an
 *     element may report an unknown value if a transition is higher than 0x3E
 *     or not determined
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_scheduler_server_init_id                           0x00550028
#define sl_btmesh_cmd_scheduler_server_deinit_id                         0x01550028
#define sl_btmesh_cmd_scheduler_server_get_id                            0x02550028
#define sl_btmesh_cmd_scheduler_server_get_action_id                     0x03550028
#define sl_btmesh_cmd_scheduler_server_set_action_id                     0x04550028
#define sl_btmesh_rsp_scheduler_server_init_id                           0x00550028
#define sl_btmesh_rsp_scheduler_server_deinit_id                         0x01550028
#define sl_btmesh_rsp_scheduler_server_get_id                            0x02550028
#define sl_btmesh_rsp_scheduler_server_get_action_id                     0x03550028
#define sl_btmesh_rsp_scheduler_server_set_action_id                     0x04550028

/**
 * @addtogroup sl_btmesh_evt_scheduler_server_action_changed sl_btmesh_evt_scheduler_server_action_changed
 * @{
 * @brief Notification of a Scheduler Action change as the result of a Scheduler
 * Set message
 *
 * For the description of these fields, see <a
 * href="#cls_mesh_scheduler_server">Scheduler Server.</a>
 */

/** @brief Identifier of the action_changed event */
#define sl_btmesh_evt_scheduler_server_action_changed_id                 0x015500a8

/***************************************************************************//**
 * @brief Data structure of the action_changed event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_scheduler_server_action_changed_s
{
  uint16_t elem_index;         /**< Scheduler server model element index */
  uint8_t  index;              /**< Index of the Scheduler Register */
  uint8_t  year;               /**< Scheduled year for the action */
  uint16_t month;              /**< Scheduled month for the action */
  uint8_t  day;                /**< Scheduled day of the month for the action */
  uint8_t  hour;               /**< Scheduled hour for the action */
  uint8_t  minute;             /**< Scheduled minute for the action */
  uint8_t  second;             /**< Scheduled second for the action */
  uint8_t  day_of_week;        /**< Scheduled days of the week for the action */
  uint8_t  action;             /**< Action to be performed at the scheduled time */
  uint32_t transition_time_ms; /**< Transition time for this action */
  uint16_t scene_number;       /**< Scene number to be used for some actions */
});

typedef struct sl_btmesh_evt_scheduler_server_action_changed_s sl_btmesh_evt_scheduler_server_action_changed_t;

/** @} */ // end addtogroup sl_btmesh_evt_scheduler_server_action_changed

/**
 * @addtogroup sl_btmesh_evt_scheduler_server_scene_changed sl_btmesh_evt_scheduler_server_scene_changed
 * @{
 * @brief Notification that scheduled scene change took place
 */

/** @brief Identifier of the scene_changed event */
#define sl_btmesh_evt_scheduler_server_scene_changed_id                  0x025500a8

/***************************************************************************//**
 * @brief Data structure of the scene_changed event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_scheduler_server_scene_changed_s
{
  uint16_t elem_index;         /**< Scheduler server model element index */
  uint32_t transition_time_ms; /**< Transition time for this action */
  uint16_t scene_number;       /**< Scene number being activated */
});

typedef struct sl_btmesh_evt_scheduler_server_scene_changed_s sl_btmesh_evt_scheduler_server_scene_changed_t;

/** @} */ // end addtogroup sl_btmesh_evt_scheduler_server_scene_changed

/**
 * @addtogroup sl_btmesh_evt_scheduler_server_action_triggered sl_btmesh_evt_scheduler_server_action_triggered
 * @{
 * @brief Notification about a Scheduler Action that had its deadline expired
 */

/** @brief Identifier of the action_triggered event */
#define sl_btmesh_evt_scheduler_server_action_triggered_id               0x035500a8

/***************************************************************************//**
 * @brief Data structure of the action_triggered event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_scheduler_server_action_triggered_s
{
  uint16_t elem_index;         /**< Scheduler server model element index */
  uint8_t  index;              /**< Index of the Scheduler Register */
  uint8_t  action;             /**< Action to be performed at the scheduled time */
  uint32_t transition_time_ms; /**< Transition time for this action */
  uint16_t scene_number;       /**< Scene number to be used for some actions */
});

typedef struct sl_btmesh_evt_scheduler_server_action_triggered_s sl_btmesh_evt_scheduler_server_action_triggered_t;

/** @} */ // end addtogroup sl_btmesh_evt_scheduler_server_action_triggered

/***************************************************************************//**
 *
 * Initialize the Scheduler Server model
 *
 * @param[in] elem_index Scheduler server model element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_scheduler_server_init(uint16_t elem_index);

/***************************************************************************//**
 *
 * Deinitialize the Scheduler Server model
 *
 * @param[in] elem_index Scheduler server model element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_scheduler_server_deinit(uint16_t elem_index);

/***************************************************************************//**
 *
 * Get Scheduler Register status of Scheduler Server.
 *
 * @param[in] elem_index Scheduler server model element index
 * @param[out] status Scheduler Server Scheduler Register status
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_scheduler_server_get(uint16_t elem_index,
                                           uint16_t *status);

/***************************************************************************//**
 *
 * Get the Scheduler Action defined by a Schedule Register state entry.
 *
 * For the description of returned fields, see <a
 * href="#cls_mesh_scheduler_server">Scheduler Server</a>
 *
 * @param[in] elem_index Scheduler server model element index
 * @param[in] index Index of the Scheduler Register entry to get
 * @param[out] index_ Index of the Scheduler Register entry to set
 * @param[out] year Scheduled year for the action
 * @param[out] month Scheduled month for the action
 * @param[out] day Scheduled day of the month for the action
 * @param[out] hour Scheduled hour for the action
 * @param[out] minute Scheduled minute for the action
 * @param[out] second Scheduled second for the action
 * @param[out] day_of_week Scheduled days of the week for the action
 * @param[out] action Action to be performed at the scheduled time
 * @param[out] transition_time_ms Transition time for this action
 * @param[out] scene_number Scene number to be used for some actions
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_scheduler_server_get_action(uint16_t elem_index,
                                                  uint8_t index,
                                                  uint8_t *index_,
                                                  uint8_t *year,
                                                  uint16_t *month,
                                                  uint8_t *day,
                                                  uint8_t *hour,
                                                  uint8_t *minute,
                                                  uint8_t *second,
                                                  uint8_t *day_of_week,
                                                  uint8_t *action,
                                                  uint32_t *transition_time_ms,
                                                  uint16_t *scene_number);

/***************************************************************************//**
 *
 * Set the given Scheduler Action entry of the Scheduler Register state.
 *
 * For the description of these fields, see <a
 * href="#cls_mesh_scheduler_server">Scheduler Server</a>
 *
 * @param[in] elem_index Scheduler server model element index
 * @param[in] index Index of the Scheduler Register entry to set
 * @param[in] year Scheduled year for the action
 * @param[in] month Scheduled month for the action
 * @param[in] day Scheduled day of the month for the action
 * @param[in] hour Scheduled hour for the action
 * @param[in] minute Scheduled minute for the action
 * @param[in] second Scheduled second for the action
 * @param[in] day_of_week Scheduled days of the week for the action
 * @param[in] action Action to be performed at the scheduled time
 * @param[in] transition_time_ms Transition time for this action
 * @param[in] scene_number Scene number to be used for some actions
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_scheduler_server_set_action(uint16_t elem_index,
                                                  uint8_t index,
                                                  uint8_t year,
                                                  uint16_t month,
                                                  uint8_t day,
                                                  uint8_t hour,
                                                  uint8_t minute,
                                                  uint8_t second,
                                                  uint8_t day_of_week,
                                                  uint8_t action,
                                                  uint32_t transition_time_ms,
                                                  uint16_t scene_number);

/** @} */ // end addtogroup sl_btmesh_scheduler_server

/**
 * @addtogroup sl_btmesh_fw_update_server Bluetooth Mesh Firmware Update Server Model
 * @{
 *
 * @brief Bluetooth Mesh Firmware Update Server Model
 *
 * This class provides the commands and messages to interface with the Firmware
 * Update Server model.
 *
 * The MBT Server must be initialized before the Update Server.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_fw_update_server_init_id                           0x00570028
#define sl_btmesh_cmd_fw_update_server_deinit_id                         0x01570028
#define sl_btmesh_cmd_fw_update_server_check_fw_metadata_rsp_id          0x02570028
#define sl_btmesh_cmd_fw_update_server_update_start_rsp_id               0x03570028
#define sl_btmesh_cmd_fw_update_server_verify_fw_rsp_id                  0x04570028
#define sl_btmesh_cmd_fw_update_server_distributor_self_update_rsp_id    0x05570028
#define sl_btmesh_rsp_fw_update_server_init_id                           0x00570028
#define sl_btmesh_rsp_fw_update_server_deinit_id                         0x01570028
#define sl_btmesh_rsp_fw_update_server_check_fw_metadata_rsp_id          0x02570028
#define sl_btmesh_rsp_fw_update_server_update_start_rsp_id               0x03570028
#define sl_btmesh_rsp_fw_update_server_verify_fw_rsp_id                  0x04570028
#define sl_btmesh_rsp_fw_update_server_distributor_self_update_rsp_id    0x05570028

/**
 * @brief The Update Phase of the Firmware Update Server.
 */
typedef enum
{
  sl_btmesh_fw_update_server_update_phase_idle                 = 0x0, /**<
                                                                           (0x0)
                                                                           No
                                                                           firmware
                                                                           transfer
                                                                           is in
                                                                           progress. */
  sl_btmesh_fw_update_server_update_phase_transfer_error       = 0x1, /**<
                                                                           (0x1)
                                                                           Error
                                                                           occurred
                                                                           during
                                                                           firmware
                                                                           image
                                                                           BLOB
                                                                           transfer. */
  sl_btmesh_fw_update_server_update_phase_transfer_in_progress = 0x2, /**<
                                                                           (0x2)
                                                                           Firmware
                                                                           image
                                                                           BLOB
                                                                           transfer
                                                                           is in
                                                                           progress. */
  sl_btmesh_fw_update_server_update_phase_verifying_update     = 0x3, /**<
                                                                           (0x3)
                                                                           Firmware
                                                                           image
                                                                           transfer
                                                                           completed,
                                                                           and
                                                                           the
                                                                           node
                                                                           is
                                                                           verifying
                                                                           the
                                                                           image. */
  sl_btmesh_fw_update_server_update_phase_verification_success = 0x4, /**<
                                                                           (0x4)
                                                                           Firmware
                                                                           image
                                                                           verification
                                                                           succeeded. */
  sl_btmesh_fw_update_server_update_phase_verification_failed  = 0x5, /**<
                                                                           (0x5)
                                                                           Firmware
                                                                           image
                                                                           verification
                                                                           failed. */
  sl_btmesh_fw_update_server_update_phase_apply_in_progress    = 0x6  /**<
                                                                           (0x6)
                                                                           The
                                                                           node
                                                                           is
                                                                           applying
                                                                           the
                                                                           firmware
                                                                           image. */
} sl_btmesh_fw_update_server_update_phase_t;

/**
 * @brief Status codes reported by the Firmware Update Server.
 */
typedef enum
{
  sl_btmesh_fw_update_server_update_status_success               = 0x0, /**<
                                                                             (0x0)
                                                                             The
                                                                             message
                                                                             was
                                                                             processed
                                                                             successfully. */
  sl_btmesh_fw_update_server_update_status_out_of_resources      = 0x1, /**<
                                                                             (0x1)
                                                                             Insufficient
                                                                             resources
                                                                             on
                                                                             the
                                                                             node. */
  sl_btmesh_fw_update_server_update_status_wrong_phase           = 0x2, /**<
                                                                             (0x2)
                                                                             The
                                                                             operation
                                                                             cannot
                                                                             be
                                                                             performed
                                                                             while
                                                                             the
                                                                             server
                                                                             is
                                                                             in
                                                                             the
                                                                             current
                                                                             phase. */
  sl_btmesh_fw_update_server_update_status_internal_error        = 0x3, /**<
                                                                             (0x3)
                                                                             An
                                                                             internal
                                                                             error
                                                                             occurred
                                                                             on
                                                                             the
                                                                             node. */
  sl_btmesh_fw_update_server_update_status_wrong_fw_index        = 0x4, /**<
                                                                             (0x4)
                                                                             The
                                                                             message
                                                                             contained
                                                                             a
                                                                             firmware
                                                                             index
                                                                             value
                                                                             that
                                                                             is
                                                                             not
                                                                             expected. */
  sl_btmesh_fw_update_server_update_status_metadata_check_failed = 0x5, /**<
                                                                             (0x5)
                                                                             The
                                                                             firmware
                                                                             metadata
                                                                             check
                                                                             failed. */
  sl_btmesh_fw_update_server_update_status_temporarily_unable    = 0x6, /**<
                                                                             (0x6)
                                                                             The
                                                                             server
                                                                             cannot
                                                                             start
                                                                             a
                                                                             firmware
                                                                             update
                                                                             right
                                                                             now. */
  sl_btmesh_fw_update_server_update_status_blob_transfer_busy    = 0x7  /**<
                                                                             (0x7)
                                                                             The
                                                                             local
                                                                             MBT
                                                                             Server
                                                                             model
                                                                             is
                                                                             busy
                                                                             with
                                                                             another
                                                                             BLOB
                                                                             transfer. */
} sl_btmesh_fw_update_server_update_status_t;

/**
 * @brief Used by the implementation to accept or reject the image and select
 * the status code reported by the Firmware Update Server.
 */
typedef enum
{
  sl_btmesh_fw_update_server_update_start_response_type_accept                       = 0x0, /**<
                                                                                                 (0x0)
                                                                                                 Accept;
                                                                                                 the
                                                                                                 MBT
                                                                                                 Server
                                                                                                 will
                                                                                                 start
                                                                                                 receiving
                                                                                                 the
                                                                                                 image. */
  sl_btmesh_fw_update_server_update_start_response_type_fw_already_exists            = 0x1, /**<
                                                                                                 (0x1)
                                                                                                 Accept;
                                                                                                 the
                                                                                                 MBT
                                                                                                 Server
                                                                                                 will
                                                                                                 not
                                                                                                 start
                                                                                                 receiving
                                                                                                 the
                                                                                                 image,
                                                                                                 as
                                                                                                 the
                                                                                                 image
                                                                                                 has
                                                                                                 already
                                                                                                 been
                                                                                                 received. */
  sl_btmesh_fw_update_server_update_start_response_type_reject_out_of_resources      = 0x2, /**<
                                                                                                 (0x2)
                                                                                                 Reject;
                                                                                                 insufficient
                                                                                                 resources
                                                                                                 on
                                                                                                 the
                                                                                                 node. */
  sl_btmesh_fw_update_server_update_start_response_type_reject_internal_error        = 0x3, /**<
                                                                                                 (0x3)
                                                                                                 Reject;
                                                                                                 An
                                                                                                 internal
                                                                                                 error
                                                                                                 occurred
                                                                                                 on
                                                                                                 the
                                                                                                 node. */
  sl_btmesh_fw_update_server_update_start_response_type_reject_metadata_check_failed = 0x4, /**<
                                                                                                 (0x4)
                                                                                                 Reject;
                                                                                                 the
                                                                                                 firmware
                                                                                                 metadata
                                                                                                 indicated
                                                                                                 that
                                                                                                 the
                                                                                                 image
                                                                                                 is
                                                                                                 not
                                                                                                 suitable
                                                                                                 for
                                                                                                 this
                                                                                                 node. */
  sl_btmesh_fw_update_server_update_start_response_type_reject_temporarily_unable    = 0x5  /**<
                                                                                                 (0x5)
                                                                                                 Reject;
                                                                                                 The
                                                                                                 server
                                                                                                 cannot
                                                                                                 start
                                                                                                 a
                                                                                                 firmware
                                                                                                 update
                                                                                                 right
                                                                                                 now. */
} sl_btmesh_fw_update_server_update_start_response_type_t;

/**
 * @addtogroup sl_btmesh_evt_fw_update_server_check_fw_metadata_req sl_btmesh_evt_fw_update_server_check_fw_metadata_req
 * @{
 * @brief The command @ref sl_btmesh_fw_update_server_check_fw_metadata_rsp must
 * be issued to inform the client whether metadata looks suitable or not.
 */

/** @brief Identifier of the check_fw_metadata_req event */
#define sl_btmesh_evt_fw_update_server_check_fw_metadata_req_id          0x005700a8

/***************************************************************************//**
 * @brief Data structure of the check_fw_metadata_req event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_update_server_check_fw_metadata_req_s
{
  uint16_t   elem_index;     /**< Server model element index */
  uint16_t   client_address; /**< Address of the Update Client */
  uint8_t    fw_index;       /**< Index of the currently installed firmware to
                                  check for updating. */
  uint8array metadata;       /**< Metadata for the incoming firmware. May be
                                  zero-length. */
});

typedef struct sl_btmesh_evt_fw_update_server_check_fw_metadata_req_s sl_btmesh_evt_fw_update_server_check_fw_metadata_req_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_update_server_check_fw_metadata_req

/**
 * @addtogroup sl_btmesh_evt_fw_update_server_update_start_req sl_btmesh_evt_fw_update_server_update_start_req
 * @{
 * @brief The firmware update is being started
 *
 * The command @ref sl_btmesh_fw_update_server_update_start_rsp should be issued
 * as a response.
 */

/** @brief Identifier of the update_start_req event */
#define sl_btmesh_evt_fw_update_server_update_start_req_id               0x035700a8

/***************************************************************************//**
 * @brief Data structure of the update_start_req event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_update_server_update_start_req_s
{
  uint16_t        elem_index;     /**< Server model element index */
  uint16_t        client_address; /**< Address of the Update Client */
  sl_bt_uuid_64_t blob_id;        /**< BLOB ID of the firmware being
                                       transferred. */
  uint8_t         fw_index;       /**< Index of the firmware to update */
  uint8array      metadata;       /**< Metadata for the FWID in the following
                                       update_start_req event. May be
                                       zero-length. */
});

typedef struct sl_btmesh_evt_fw_update_server_update_start_req_s sl_btmesh_evt_fw_update_server_update_start_req_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_update_server_update_start_req

/**
 * @addtogroup sl_btmesh_evt_fw_update_server_verify_fw_req sl_btmesh_evt_fw_update_server_verify_fw_req
 * @{
 * @brief The firmware has been downloaded completely, and needs to be verified
 *
 * The application should respond with the command @ref
 * sl_btmesh_fw_update_server_verify_fw_rsp to indicate whether the image passed
 * the application-specific checks.
 */

/** @brief Identifier of the verify_fw_req event */
#define sl_btmesh_evt_fw_update_server_verify_fw_req_id                  0x045700a8

/***************************************************************************//**
 * @brief Data structure of the verify_fw_req event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_update_server_verify_fw_req_s
{
  uint16_t elem_index; /**< Server model element index */
});

typedef struct sl_btmesh_evt_fw_update_server_verify_fw_req_s sl_btmesh_evt_fw_update_server_verify_fw_req_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_update_server_verify_fw_req

/**
 * @addtogroup sl_btmesh_evt_fw_update_server_apply sl_btmesh_evt_fw_update_server_apply
 * @{
 * @brief Apply a firmware image that has been transferred to a server.
 */

/** @brief Identifier of the apply event */
#define sl_btmesh_evt_fw_update_server_apply_id                          0x055700a8

/***************************************************************************//**
 * @brief Data structure of the apply event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_update_server_apply_s
{
  uint16_t elem_index;     /**< Server model element index */
  uint16_t client_address; /**< Address of the Update Client */
});

typedef struct sl_btmesh_evt_fw_update_server_apply_s sl_btmesh_evt_fw_update_server_apply_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_update_server_apply

/**
 * @addtogroup sl_btmesh_evt_fw_update_server_update_cancelled sl_btmesh_evt_fw_update_server_update_cancelled
 * @{
 * @brief Cancel a firmware update and delete any stored information about the
 * update on the server.
 */

/** @brief Identifier of the update_cancelled event */
#define sl_btmesh_evt_fw_update_server_update_cancelled_id               0x065700a8

/***************************************************************************//**
 * @brief Data structure of the update_cancelled event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_update_server_update_cancelled_s
{
  uint16_t elem_index;     /**< Server model element index */
  uint16_t client_address; /**< Address of the Update Client */
});

typedef struct sl_btmesh_evt_fw_update_server_update_cancelled_s sl_btmesh_evt_fw_update_server_update_cancelled_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_update_server_update_cancelled

/**
 * @addtogroup sl_btmesh_evt_fw_update_server_distributor_self_update_req sl_btmesh_evt_fw_update_server_distributor_self_update_req
 * @{
 * @brief This event can only be generated if both Distribution Server and
 * Update Server functionality are used on the same node
 *
 * This event signals that the Initiator has added this node to the list of
 * nodes to be updated; that is, the node is asked to update itself using an
 * image already present in the Distribution Server's Firmware List. The command
 * @ref sl_btmesh_fw_update_server_distributor_self_update_rsp must be called to
 * accept or reject the self-update request.
 */

/** @brief Identifier of the distributor_self_update_req event */
#define sl_btmesh_evt_fw_update_server_distributor_self_update_req_id    0x075700a8

/***************************************************************************//**
 * @brief Data structure of the distributor_self_update_req event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_update_server_distributor_self_update_req_s
{
  uint16_t elem_index;     /**< Server model element index */
  uint16_t client_address; /**< Address of the Update Client; in this case it is
                                the address of the local element on which the
                                Distribution Server and Update Client reside. It
                                can be the same element as the Update Server's
                                element, or a different one. */
  uint8_t  fw_index;       /**< Index of the firmware on the Update Server to
                                update */
  uint16_t fw_list_index;  /**< Index of the firmware image in the Distribution
                                Server's Firmware List. As the Distribution
                                Server may have multiple images in its Firmware
                                List, this field indicates which one to use for
                                updating. */
});

typedef struct sl_btmesh_evt_fw_update_server_distributor_self_update_req_s sl_btmesh_evt_fw_update_server_distributor_self_update_req_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_update_server_distributor_self_update_req

/***************************************************************************//**
 *
 * Initializes the Firmware Update Server model.
 *
 * @param[in] elem_index Server model element index
 * @param[in] num_installed_fw Number of installed FWs
 * @param[in] max_metadata_len Maximum length of metadata supported. Any request
 *   with a larger metadata value will be rejected automatically.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_update_server_init(uint16_t elem_index,
                                            uint8_t num_installed_fw,
                                            uint8_t max_metadata_len);

/***************************************************************************//**
 *
 * Deinitializes the Firmware Update Server model.
 *
 * @param[in] elem_index Server model element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_update_server_deinit(uint16_t elem_index);

/***************************************************************************//**
 *
 * Response for the @ref sl_btmesh_evt_fw_update_server_check_fw_metadata_req
 * event. Report whether a server can accept a firmware update.
 *
 * @param[in] elem_index Server model element index
 * @param[in] response_type Enum @ref
 *   sl_btmesh_fw_update_server_update_start_response_type_t. The status code to
 *   indicate to the client.
 * @param[in] additional_information Enum @ref
 *   sl_btmesh_fw_update_client_additional_info_t. Additional information about
 *   what will happen to the node after the update is applied. 0 = no changes to
 *   DCD; 1 = DCD will change but Composition Data Refresh is not supported; 2 =
 *   DCD change and Composition Data Refresh is supported; 3 = node will become
 *   unprovisioned.
 * @param[in] fw_index Index of the firmware to be updated.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_update_server_check_fw_metadata_rsp(uint16_t elem_index,
                                                             uint8_t response_type,
                                                             uint8_t additional_information,
                                                             uint8_t fw_index);

/***************************************************************************//**
 *
 * Response for the @ref sl_btmesh_evt_fw_update_server_update_start_req event.
 * After the firmware BLOB Transfer reception is complete, the event @ref
 * sl_btmesh_evt_fw_update_server_verify_fw_req will be generated.
 *
 * @param[in] elem_index Server model element index
 * @param[in] response_type Enum @ref
 *   sl_btmesh_fw_update_server_update_start_response_type_t. Accept or reject
 *   the image.
 * @param[in] additional_information Enum @ref
 *   sl_btmesh_fw_update_client_additional_info_t. Additional information about
 *   what will happen to the node after the update is applied. 0 = no changes to
 *   DCD; 1 = DCD will change but Composition Data Refresh is not supported; 2 =
 *   DCD change and Composition Data Refresh is supported; 3 = node will become
 *   unprovisioned.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_update_server_verify_fw_req
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_update_server_update_start_rsp(uint16_t elem_index,
                                                        uint8_t response_type,
                                                        uint8_t additional_information);

/***************************************************************************//**
 *
 * Response for the @ref sl_btmesh_evt_fw_update_server_verify_fw_req event. If
 * the image was accepted, the event @ref sl_btmesh_evt_fw_update_server_apply
 * will be generated when the Update Client asks the node to apply the new
 * firmware and reboot.
 *
 * @param[in] elem_index Server model element index
 * @param[in] accept 0 to reject the firmware image, nonzero to accept. If
 *   rejected, the server will indicate verification failure to the client when
 *   queried. If accepted, the server will wait for an apply command from the
 *   client.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_update_server_apply
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_update_server_verify_fw_rsp(uint16_t elem_index,
                                                     uint8_t accept);

/***************************************************************************//**
 *
 * Response for the @ref
 * sl_btmesh_evt_fw_update_server_distributor_self_update_req event. This
 * command can only be used if both Distribution Server and Update Server
 * functionality are used on the same node. If the self-update is accepted, the
 * event @ref sl_btmesh_evt_fw_update_server_verify_fw_req will be generated to
 * confirm that the firmware image passed the application-specific checks, just
 * like in a regular update.
 *
 * @param[in] elem_index Server model element index
 * @param[in] response_type Enum @ref
 *   sl_btmesh_fw_update_server_update_start_response_type_t. Accept or reject
 *   the image.
 * @param[in] additional_information Enum @ref
 *   sl_btmesh_fw_update_client_additional_info_t. Additional information about
 *   what will happen to the node after the update is applied. 0 = no changes to
 *   DCD; 1 = DCD will change but Composition Data Refresh is not supported; 2 =
 *   DCD change and Composition Data Refresh is supported; 3 = node will become
 *   unprovisioned.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_update_server_verify_fw_req
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_update_server_distributor_self_update_rsp(uint16_t elem_index,
                                                                   uint8_t response_type,
                                                                   uint8_t additional_information);

/** @} */ // end addtogroup sl_btmesh_fw_update_server

/**
 * @addtogroup sl_btmesh_time_server Bluetooth Mesh Time Server Model
 * @{
 *
 * @brief Bluetooth Mesh Time Server Model
 *
 * This class provides the commands and messages to interface with the Time
 * Server model
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_time_server_init_id                                0x00520028
#define sl_btmesh_cmd_time_server_deinit_id                              0x01520028
#define sl_btmesh_cmd_time_server_get_time_id                            0x02520028
#define sl_btmesh_cmd_time_server_set_time_id                            0x03520028
#define sl_btmesh_cmd_time_server_get_time_zone_offset_new_id            0x04520028
#define sl_btmesh_cmd_time_server_set_time_zone_offset_new_id            0x05520028
#define sl_btmesh_cmd_time_server_get_tai_utc_delta_new_id               0x06520028
#define sl_btmesh_cmd_time_server_set_tai_utc_delta_new_id               0x07520028
#define sl_btmesh_cmd_time_server_get_time_role_id                       0x08520028
#define sl_btmesh_cmd_time_server_set_time_role_id                       0x09520028
#define sl_btmesh_cmd_time_server_get_datetime_id                        0x0a520028
#define sl_btmesh_cmd_time_server_publish_id                             0x0b520028
#define sl_btmesh_cmd_time_server_status_id                              0x0c520028
#define sl_btmesh_rsp_time_server_init_id                                0x00520028
#define sl_btmesh_rsp_time_server_deinit_id                              0x01520028
#define sl_btmesh_rsp_time_server_get_time_id                            0x02520028
#define sl_btmesh_rsp_time_server_set_time_id                            0x03520028
#define sl_btmesh_rsp_time_server_get_time_zone_offset_new_id            0x04520028
#define sl_btmesh_rsp_time_server_set_time_zone_offset_new_id            0x05520028
#define sl_btmesh_rsp_time_server_get_tai_utc_delta_new_id               0x06520028
#define sl_btmesh_rsp_time_server_set_tai_utc_delta_new_id               0x07520028
#define sl_btmesh_rsp_time_server_get_time_role_id                       0x08520028
#define sl_btmesh_rsp_time_server_set_time_role_id                       0x09520028
#define sl_btmesh_rsp_time_server_get_datetime_id                        0x0a520028
#define sl_btmesh_rsp_time_server_publish_id                             0x0b520028
#define sl_btmesh_rsp_time_server_status_id                              0x0c520028

/**
 * @addtogroup sl_btmesh_evt_time_server_time_updated sl_btmesh_evt_time_server_time_updated
 * @{
 * @brief Indicate that Time State has been updated by external source, either
 * Time Set message from a Time Client, or a Time Status message
 */

/** @brief Identifier of the time_updated event */
#define sl_btmesh_evt_time_server_time_updated_id                        0x005200a8

/***************************************************************************//**
 * @brief Data structure of the time_updated event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_time_server_time_updated_s
{
  uint16_t elem_index;       /**< Server model element index */
  uint64_t tai_seconds;      /**< The current TAI time in seconds since the
                                  epoch, 40-bit value */
  uint8_t  subsecond;        /**< The sub-second in units of 1/256th second.
                                  Range is 0-255. */
  uint8_t  uncertainty;      /**< The estimated uncertainty in 10-milliseconds
                                  steps. Range is 0-255, representing up 2.55
                                  seconds. */
  uint8_t  time_authority;   /**< 0: No Time Authority, the element does not
                                  have a trusted source of time such as GPS or
                                  NTP. 1: Time Authority, the element has a
                                  trusted source of time or a battery-backed
                                  properly initialized RTC. Other values are
                                  prohibited. */
  int32_t  tai_utc_delta;    /**< Current difference between TAI and UTC in
                                  seconds. Range is -255 to 32512. */
  int16_t  time_zone_offset; /**< The local time zone offset in 15-minute
                                  increments. Range is -64 to 191, representing
                                  \-16 to 47.75 hours. */
});

typedef struct sl_btmesh_evt_time_server_time_updated_s sl_btmesh_evt_time_server_time_updated_t;

/** @} */ // end addtogroup sl_btmesh_evt_time_server_time_updated

/**
 * @addtogroup sl_btmesh_evt_time_server_time_zone_offset_updated sl_btmesh_evt_time_server_time_zone_offset_updated
 * @{
 * @brief Indicate that upcoming time zone offset change has been updated by
 * external source
 */

/** @brief Identifier of the time_zone_offset_updated event */
#define sl_btmesh_evt_time_server_time_zone_offset_updated_id            0x015200a8

/***************************************************************************//**
 * @brief Data structure of the time_zone_offset_updated event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_time_server_time_zone_offset_updated_s
{
  uint16_t elem_index;               /**< Server model element index */
  int16_t  time_zone_offset_current; /**< Current local time zone offset. Range
                                          is -64 to 191, representing -16 to
                                          47.75 hours. */
  int16_t  time_zone_offset_new;     /**< Upcoming local time zone offset. Range
                                          is -64 to 191, representing -16 to
                                          47.75 hours. */
  uint64_t tai_of_zone_change;       /**< Absolute TAI time when the Time Zone
                                          Offset will change from Current to New */
});

typedef struct sl_btmesh_evt_time_server_time_zone_offset_updated_s sl_btmesh_evt_time_server_time_zone_offset_updated_t;

/** @} */ // end addtogroup sl_btmesh_evt_time_server_time_zone_offset_updated

/**
 * @addtogroup sl_btmesh_evt_time_server_tai_utc_delta_updated sl_btmesh_evt_time_server_tai_utc_delta_updated
 * @{
 * @brief Indicate that the upcoming TAI-UTC Delta has been updated by external
 * source
 */

/** @brief Identifier of the tai_utc_delta_updated event */
#define sl_btmesh_evt_time_server_tai_utc_delta_updated_id               0x025200a8

/***************************************************************************//**
 * @brief Data structure of the tai_utc_delta_updated event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_time_server_tai_utc_delta_updated_s
{
  uint16_t elem_index;            /**< Server model element index */
  int32_t  tai_utc_delta_current; /**< Current difference between TAI and UTC is
                                       seconds. Range is -255 to 32512. */
  int32_t  tai_utc_delta_new;     /**< Upcoming difference between TAI and UTC
                                       is seconds. Range is -255 to 32512. */
  uint64_t tai_of_delta_change;   /**< TAI Seconds time of the upcoming TAI-UTC
                                       Delta change */
});

typedef struct sl_btmesh_evt_time_server_tai_utc_delta_updated_s sl_btmesh_evt_time_server_tai_utc_delta_updated_t;

/** @} */ // end addtogroup sl_btmesh_evt_time_server_tai_utc_delta_updated

/**
 * @addtogroup sl_btmesh_evt_time_server_time_role_updated sl_btmesh_evt_time_server_time_role_updated
 * @{
 * @brief Indicate that Time Role has been updated by external source by Time
 * Role Set message from a Time Client.
 */

/** @brief Identifier of the time_role_updated event */
#define sl_btmesh_evt_time_server_time_role_updated_id                   0x035200a8

/***************************************************************************//**
 * @brief Data structure of the time_role_updated event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_time_server_time_role_updated_s
{
  uint16_t elem_index; /**< Server model element index */
  uint8_t  time_role;  /**< Time Role */
});

typedef struct sl_btmesh_evt_time_server_time_role_updated_s sl_btmesh_evt_time_server_time_role_updated_t;

/** @} */ // end addtogroup sl_btmesh_evt_time_server_time_role_updated

/***************************************************************************//**
 *
 * Initializes the Time Server model
 *
 * @param[in] elem_index Server model element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_time_server_init(uint16_t elem_index);

/***************************************************************************//**
 *
 * Deinitialize the Time Server model
 *
 * @param[in] elem_index Server model element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_time_server_deinit(uint16_t elem_index);

/***************************************************************************//**
 *
 * Get the current time from Time Server
 *
 * @param[in] elem_index Server model element index
 * @param[out] tai_seconds The current TAI time in seconds
 * @param[out] subsecond The sub-second time in units of 1/256th second
 * @param[out] uncertainty The estimated uncertainty in 10 millisecond steps
 * @param[out] time_authority 0 = No Time Authority, 1 = Time Authority
 * @param[out] time_zone_offset Current local time zone offset. Range is -64 to
 *   191, representing -16 to 47.75 hours.
 * @param[out] tai_utc_delta Current difference between TAI and UTC in seconds.
 *   Range is -255 to 32512.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_time_server_get_time(uint16_t elem_index,
                                           uint64_t *tai_seconds,
                                           uint8_t *subsecond,
                                           uint8_t *uncertainty,
                                           uint8_t *time_authority,
                                           int16_t *time_zone_offset,
                                           int32_t *tai_utc_delta);

/***************************************************************************//**
 *
 * Set the current time for the element.
 *
 * @param[in] elem_index Server model element index
 * @param[in] tai_seconds The current TAI time in seconds
 * @param[in] subsecond The sub-second time in units of 1/256th second
 * @param[in] uncertainty The estimated uncertainty in 10 millisecond steps
 * @param[in] time_authority 0 = No Time Authority, 1 = Time Authority
 * @param[in] time_zone_offset Current local time zone offset. Range is -64 to
 *   191, representing -16 to 47.75 hours.
 * @param[in] tai_utc_delta Current difference between TAI and UTC in seconds.
 *   Range is -255 to 32512.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_time_server_set_time(uint16_t elem_index,
                                           uint64_t tai_seconds,
                                           uint8_t subsecond,
                                           uint8_t uncertainty,
                                           uint8_t time_authority,
                                           int16_t time_zone_offset,
                                           int32_t tai_utc_delta);

/***************************************************************************//**
 *
 * Get the upcoming time zone offset from Time Server
 *
 * @param[in] elem_index Server model element index
 * @param[out] new_offset Upcoming local time zone offset. Range is -64 to 191,
 *   representing -16 to 47.75 hours.
 * @param[out] tai_of_zone_change Absolute TAI time when the Time Zone Offset
 *   will change from Current to New
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_time_server_get_time_zone_offset_new(uint16_t elem_index,
                                                           int16_t *new_offset,
                                                           uint64_t *tai_of_zone_change);

/***************************************************************************//**
 *
 * Set the upcoming time zone offset for the element
 *
 * @param[in] elem_index Server model element index
 * @param[in] new_offset Upcoming local time zone offset. Range is -64 to 191,
 *   representing -16 to 47.75 hours.
 * @param[in] tai_of_zone_change Absolute TAI time when the Time Zone Offset
 *   will change from Current to New
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_time_server_set_time_zone_offset_new(uint16_t elem_index,
                                                           int16_t new_offset,
                                                           uint64_t tai_of_zone_change);

/***************************************************************************//**
 *
 * Get the upcoming TAI-UTC delta for the element
 *
 * @param[in] elem_index Server model element index
 * @param[out] new_delta Upcoming difference between TAI and UTC in seconds
 * @param[out] tai_of_delta_change Absolute TAI time when the TAI-UTC Delta will
 *   change from Current to New
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_time_server_get_tai_utc_delta_new(uint16_t elem_index,
                                                        int32_t *new_delta,
                                                        uint64_t *tai_of_delta_change);

/***************************************************************************//**
 *
 * Set the upcoming TAI-UTC delta for the element
 *
 * @param[in] elem_index Server model element index
 * @param[in] new_delta Upcoming difference between TAI and UTC in seconds.
 *   Range is -255 to 32512.
 * @param[in] tai_of_delta_change Absolute TAI time when the TAI-UTC Delta will
 *   change from Current to New
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_time_server_set_tai_utc_delta_new(uint16_t elem_index,
                                                        int32_t new_delta,
                                                        uint64_t tai_of_delta_change);

/***************************************************************************//**
 *
 * Get Time Role for the element
 *
 * @param[in] elem_index Server model element index
 * @param[out] time_role Time Role of the element
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_time_server_get_time_role(uint16_t elem_index,
                                                uint8_t *time_role);

/***************************************************************************//**
 *
 * Set Time Role for the element
 *
 * @param[in] elem_index Server model element index
 * @param[in] time_role Time Role of the element. Range is 0 to 3.
 *     - 0 = Time Role None, The element does not participate in propagation of
 *       time information.
 *     - 1 = Time Role Authority, The element publishes Time Status messages but
 *       does not process received Time Status messages.
 *     - 2 = Time Role Relay, The element processes received and publishes Time
 *       Status messages.
 *     - 3 = Time Role Client, The element does not publish but processes
 *       received Time Status messages.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_time_server_set_time_role(uint16_t elem_index,
                                                uint8_t time_role);

/***************************************************************************//**
 *
 * Return the date and time from the Time Server
 *
 * @param[in] elem_index Server model element index
 * @param[out] year Year
 * @param[out] month Month
 * @param[out] day Day
 * @param[out] hour Hour
 * @param[out] min Minutes
 * @param[out] sec Seconds
 * @param[out] ms Milliseconds
 * @param[out] timezone Local time zone offset. Range is -64 to 191,
 *   representing -16 to 47.75 hours.
 * @param[out] day_of_week Day of week, 0..6 represents Monday to Sunday
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_time_server_get_datetime(uint16_t elem_index,
                                               uint16_t *year,
                                               uint8_t *month,
                                               uint8_t *day,
                                               uint8_t *hour,
                                               uint8_t *min,
                                               uint8_t *sec,
                                               uint16_t *ms,
                                               int16_t *timezone,
                                               uint8_t *day_of_week);

/***************************************************************************//**
 *
 * Publish Time Status containing the current time. Permitted only for Time
 * Server having the role of Time Authority.
 *
 * @param[in] elem_index Element index of the Time Server
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_time_server_publish(uint16_t elem_index);

/***************************************************************************//**
 *
 * Send a Time Status message containing the current time as an unsolicited
 * message. Permitted only for Time Server having the role of Time Authority.
 *
 * @param[in] destination_address Destination address
 * @param[in] elem_index Element index of the Time Server
 * @param[in] appkey_index The application key index to use
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_time_server_status(uint16_t destination_address,
                                         uint16_t elem_index,
                                         uint16_t appkey_index);

/** @} */ // end addtogroup sl_btmesh_time_server

/**
 * @addtogroup sl_btmesh_time_client Bluetooth Mesh Time Client Model
 * @{
 *
 * @brief Bluetooth Mesh Time Client Model
 *
 * This class provides the commands and messages to interface with the Time
 * Client model
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_time_client_init_id                                0x00530028
#define sl_btmesh_cmd_time_client_deinit_id                              0x01530028
#define sl_btmesh_cmd_time_client_get_time_id                            0x02530028
#define sl_btmesh_cmd_time_client_set_time_id                            0x03530028
#define sl_btmesh_cmd_time_client_get_time_zone_id                       0x04530028
#define sl_btmesh_cmd_time_client_set_time_zone_id                       0x05530028
#define sl_btmesh_cmd_time_client_get_tai_utc_delta_id                   0x06530028
#define sl_btmesh_cmd_time_client_set_tai_utc_delta_id                   0x07530028
#define sl_btmesh_cmd_time_client_get_time_role_id                       0x08530028
#define sl_btmesh_cmd_time_client_set_time_role_id                       0x09530028
#define sl_btmesh_rsp_time_client_init_id                                0x00530028
#define sl_btmesh_rsp_time_client_deinit_id                              0x01530028
#define sl_btmesh_rsp_time_client_get_time_id                            0x02530028
#define sl_btmesh_rsp_time_client_set_time_id                            0x03530028
#define sl_btmesh_rsp_time_client_get_time_zone_id                       0x04530028
#define sl_btmesh_rsp_time_client_set_time_zone_id                       0x05530028
#define sl_btmesh_rsp_time_client_get_tai_utc_delta_id                   0x06530028
#define sl_btmesh_rsp_time_client_set_tai_utc_delta_id                   0x07530028
#define sl_btmesh_rsp_time_client_get_time_role_id                       0x08530028
#define sl_btmesh_rsp_time_client_set_time_role_id                       0x09530028

/**
 * @brief These values define the Time Role types used by the stack.
 */
typedef enum
{
  sl_btmesh_time_client_time_role_none      = 0x0, /**< (0x0) The element does
                                                        not participate in
                                                        propagation of time
                                                        information. */
  sl_btmesh_time_client_time_role_authority = 0x1, /**< (0x1) The element
                                                        publishes Time Status
                                                        messages but does not
                                                        process received Time
                                                        Status messages. */
  sl_btmesh_time_client_time_role_relay     = 0x2, /**< (0x2) The element
                                                        processes received and
                                                        publishes Time Status
                                                        messages. */
  sl_btmesh_time_client_time_role_client    = 0x3  /**< (0x3) The element does
                                                        not publish but
                                                        processes received Time
                                                        Status messages. */
} sl_btmesh_time_client_time_roles_t;

/**
 * @addtogroup sl_btmesh_evt_time_client_time_status sl_btmesh_evt_time_client_time_status
 * @{
 * @brief Time Status event, response to @ref sl_btmesh_time_client_set_time or
 * @ref sl_btmesh_time_client_get_time command or unsolicited message.
 */

/** @brief Identifier of the time_status event */
#define sl_btmesh_evt_time_client_time_status_id                         0x005300a8

/***************************************************************************//**
 * @brief Data structure of the time_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_time_client_time_status_s
{
  uint16_t client_address;   /**< Client address */
  uint16_t elem_index;       /**< Client model element index */
  uint16_t server_address;   /**< Unicast address of the server */
  uint16_t appkey_index;     /**< App key index. */
  uint64_t tai_seconds;      /**< The current TAI time in seconds */
  uint8_t  subsecond;        /**< The sub-second time in units of 1/256th second */
  uint8_t  uncertainty;      /**< The estimated uncertainty in 10-millisecond
                                  steps */
  uint8_t  time_authority;   /**< 0 = No Time Authority, 1 = Time Authority */
  int32_t  tai_utc_delta;    /**< Current difference between TAI and UTC in
                                  seconds. Range is -255 to 32512. */
  int16_t  time_zone_offset; /**< The local time zone offset in 15-minute
                                  increments. Range is -64 to 191, representing
                                  \-16 to 47.75 hours. */
});

typedef struct sl_btmesh_evt_time_client_time_status_s sl_btmesh_evt_time_client_time_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_time_client_time_status

/**
 * @addtogroup sl_btmesh_evt_time_client_time_zone_status sl_btmesh_evt_time_client_time_zone_status
 * @{
 * @brief Time Zone Status event, response to @ref
 * sl_btmesh_time_client_set_time_zone or @ref
 * sl_btmesh_time_client_get_time_zone command or unsolicited message.
 */

/** @brief Identifier of the time_zone_status event */
#define sl_btmesh_evt_time_client_time_zone_status_id                    0x015300a8

/***************************************************************************//**
 * @brief Data structure of the time_zone_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_time_client_time_zone_status_s
{
  uint16_t client_address;           /**< Client address */
  uint16_t elem_index;               /**< Client model element index */
  uint16_t server_address;           /**< Unicast address of the server */
  uint16_t appkey_index;             /**< App key index. */
  int16_t  time_zone_offset_current; /**< Current local time zone offset. Range
                                          is -64 to 191, representing -16 to
                                          47.75 hours. */
  int16_t  time_zone_offset_new;     /**< Upcoming local time zone offset */
  uint64_t tai_of_zone_change;       /**< TAI seconds time of the upcoming Time
                                          Zone offset change */
});

typedef struct sl_btmesh_evt_time_client_time_zone_status_s sl_btmesh_evt_time_client_time_zone_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_time_client_time_zone_status

/**
 * @addtogroup sl_btmesh_evt_time_client_tai_utc_delta_status sl_btmesh_evt_time_client_tai_utc_delta_status
 * @{
 * @brief TAI-UTC Delta Status event, response to @ref
 * sl_btmesh_time_client_set_tai_utc_delta or @ref
 * sl_btmesh_time_client_get_tai_utc_delta command or unsolicited message.
 */

/** @brief Identifier of the tai_utc_delta_status event */
#define sl_btmesh_evt_time_client_tai_utc_delta_status_id                0x025300a8

/***************************************************************************//**
 * @brief Data structure of the tai_utc_delta_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_time_client_tai_utc_delta_status_s
{
  uint16_t client_address;        /**< Client address */
  uint16_t elem_index;            /**< Client model element index */
  uint16_t server_address;        /**< Unicast address of the server */
  uint16_t appkey_index;          /**< App key index. */
  int32_t  tai_utc_delta_current; /**< Current difference between TAI and UTC.
                                       Range is -255 to 32512. */
  int32_t  tai_utc_delta_new;     /**< Upcoming difference between TAI and UTC
                                       in seconds. Range is -255 to 32512. */
  uint64_t tai_of_delta_change;   /**< TAI seconds time of the upcoming TAI-UTC
                                       delta change */
});

typedef struct sl_btmesh_evt_time_client_tai_utc_delta_status_s sl_btmesh_evt_time_client_tai_utc_delta_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_time_client_tai_utc_delta_status

/**
 * @addtogroup sl_btmesh_evt_time_client_time_role_status sl_btmesh_evt_time_client_time_role_status
 * @{
 * @brief Time Role Status event, response to @ref
 * sl_btmesh_time_client_set_time_role or @ref
 * sl_btmesh_time_client_get_time_role command or unsolicited message.
 */

/** @brief Identifier of the time_role_status event */
#define sl_btmesh_evt_time_client_time_role_status_id                    0x035300a8

/***************************************************************************//**
 * @brief Data structure of the time_role_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_time_client_time_role_status_s
{
  uint16_t client_address; /**< Client address */
  uint16_t elem_index;     /**< Client model element index */
  uint16_t server_address; /**< Unicast address of the server */
  uint16_t appkey_index;   /**< App key index. */
  uint8_t  time_role;      /**< The Time Role for the element */
});

typedef struct sl_btmesh_evt_time_client_time_role_status_s sl_btmesh_evt_time_client_time_role_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_time_client_time_role_status

/***************************************************************************//**
 *
 * Initialize the Time Client model
 *
 * @param[in] elem_index Client model element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_time_client_init(uint16_t elem_index);

/***************************************************************************//**
 *
 * Deinitialize the Time Client model
 *
 * @param[in] elem_index Client model element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_time_client_deinit(uint16_t elem_index);

/***************************************************************************//**
 *
 * Send a Time Get message to Time Server
 *
 * @param[in] server_address Destination server model address
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_time_client_time_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_time_client_get_time(uint16_t server_address,
                                           uint16_t elem_index,
                                           uint16_t appkey_index);

/***************************************************************************//**
 *
 * Send a Time Set message to Time Server
 *
 * @param[in] server_address Destination server model address
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use.
 * @param[in] tai_seconds The current TAI time in seconds since the epoch,
 *   40-bit value
 * @param[in] subsecond The sub-second in units of 1/256th second. Range is
 *   0-255.
 * @param[in] uncertainty The estimated uncertainty in 10-milliseconds steps.
 *   Range is 0-255, representing up to 2.55 seconds.
 * @param[in] time_authority 0: No Time Authority, the element does not have a
 *   trusted source of time such as GPS or NTP. 1: Time Authority, the element
 *   has a trusted source of time or a battery-backed properly initialized RTC.
 *   Other values are prohibited.
 * @param[in] tai_utc_delta Current difference between TAI and UTC in seconds.
 *   Range is -255 to 32512.
 * @param[in] time_zone_offset The local time zone offset in 15-minute
 *   increments. Range is -64 to 191, representing -16 to 47.75 hours.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_time_client_time_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_time_client_set_time(uint16_t server_address,
                                           uint16_t elem_index,
                                           uint16_t appkey_index,
                                           uint64_t tai_seconds,
                                           uint8_t subsecond,
                                           uint8_t uncertainty,
                                           uint8_t time_authority,
                                           int32_t tai_utc_delta,
                                           int16_t time_zone_offset);

/***************************************************************************//**
 *
 * Send a Time Zone Get message to Time Server
 *
 * @param[in] server_address Destination server model address
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_time_client_time_zone_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_time_client_get_time_zone(uint16_t server_address,
                                                uint16_t elem_index,
                                                uint16_t appkey_index);

/***************************************************************************//**
 *
 * Send a Time Zone Set message to Time Server to set the Time Zone New state
 *
 * @param[in] server_address Destination server model address
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use.
 * @param[in] time_zone_offset_new Upcoming local time zone offset. Range is -64
 *   to 191, representing -16 to 47.75 hours.
 * @param[in] tai_of_zone_change TAI Seconds time of upcoming Time Zone offset
 *   change
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_time_client_time_zone_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_time_client_set_time_zone(uint16_t server_address,
                                                uint16_t elem_index,
                                                uint16_t appkey_index,
                                                int16_t time_zone_offset_new,
                                                uint64_t tai_of_zone_change);

/***************************************************************************//**
 *
 * Send a TAI-UTC Delta Get message to Time Server
 *
 * @param[in] server_address Destination server model address
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_time_client_tai_utc_delta_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_time_client_get_tai_utc_delta(uint16_t server_address,
                                                    uint16_t elem_index,
                                                    uint16_t appkey_index);

/***************************************************************************//**
 *
 * Send a TAI-UTC Delta Set message to Time Server, which responds with a
 * TAI-UTC Delta Status message.
 *
 * @param[in] server_address Destination server model address
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use.
 * @param[in] tai_utc_delta_new Upcoming difference between TAI and UTC is
 *   seconds. Range is -255 to 32512.
 * @param[in] tai_of_delta_change TAI Seconds time of the upcoming TAI-UTC Delta
 *   change
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_time_client_tai_utc_delta_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_time_client_set_tai_utc_delta(uint16_t server_address,
                                                    uint16_t elem_index,
                                                    uint16_t appkey_index,
                                                    int32_t tai_utc_delta_new,
                                                    uint64_t tai_of_delta_change);

/***************************************************************************//**
 *
 * Send a Time Role Get message to Time Server, which responds with a Time Role
 * Status message.
 *
 * @param[in] server_address Destination server model address
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_time_client_time_role_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_time_client_get_time_role(uint16_t server_address,
                                                uint16_t elem_index,
                                                uint16_t appkey_index);

/***************************************************************************//**
 *
 * Send Time Role Set message to Time Server, which responds with a Time Role
 * Status message.
 *
 * @param[in] server_address Destination server model address
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index The application key index to use.
 * @param[in] time_role The Time Role for the element. Range is 0 to 3.
 *     - 0 = Time Role None, The element does not participate in propagation of
 *       time information.
 *     - 1 = Time Role Authority, The element publishes Time Status messages but
 *       does not process received Time Status messages.
 *     - 2 = Time Role Relay, The element processes received and publishes Time
 *       Status messages.
 *     - 3 = Time Role Client, The element does not publish but processes
 *       received Time Status messages.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_time_client_time_role_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_time_client_set_time_role(uint16_t server_address,
                                                uint16_t elem_index,
                                                uint16_t appkey_index,
                                                uint8_t time_role);

/** @} */ // end addtogroup sl_btmesh_time_client

/**
 * @addtogroup sl_btmesh_fw_dist_server Bluetooth Mesh Firmware Distribution Server Model
 * @{
 *
 * @brief Bluetooth Mesh Firmware Distribution Server Model
 *
 * The Bluetooth Mesh Firmware Distribution Server Model is used to receive new
 * firmware images from the Firmware Distribution Client model and distribute
 * them to Target Nodes. As a transport layer Bluetooth Mesh BLOB Transfer (MBT)
 * models are in use.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_fw_dist_server_init_id                             0x00580028
#define sl_btmesh_cmd_fw_dist_server_deinit_id                           0x01580028
#define sl_btmesh_cmd_fw_dist_server_upload_start_rsp_id                 0x02580028
#define sl_btmesh_cmd_fw_dist_server_execute_distribution_step_id        0x03580028
#define sl_btmesh_cmd_fw_dist_server_upload_oob_start_rsp_id             0x08580028
#define sl_btmesh_cmd_fw_dist_server_set_upload_oob_metadata_id          0x09580028
#define sl_btmesh_cmd_fw_dist_server_set_upload_oob_image_size_id        0x0a580028
#define sl_btmesh_cmd_fw_dist_server_set_upload_oob_progress_id          0x0b580028
#define sl_btmesh_cmd_fw_dist_server_get_node_status_by_index_id         0x0c580028
#define sl_btmesh_cmd_fw_dist_server_get_node_status_by_address_id       0x0d580028
#define sl_btmesh_cmd_fw_dist_server_delete_rsp_id                       0x0e580028
#define sl_btmesh_cmd_fw_dist_server_dist_start_rsp_id                   0x13580028
#define sl_btmesh_cmd_fw_dist_server_set_multicast_threshold_id          0x14580028
#define sl_btmesh_cmd_fw_dist_server_delete_all_rsp_id                   0x0f580028
#define sl_btmesh_cmd_fw_dist_server_resume_rsp_id                       0x10580028
#define sl_btmesh_cmd_fw_dist_server_configure_throttle_id               0x15580028
#define sl_btmesh_rsp_fw_dist_server_init_id                             0x00580028
#define sl_btmesh_rsp_fw_dist_server_deinit_id                           0x01580028
#define sl_btmesh_rsp_fw_dist_server_upload_start_rsp_id                 0x02580028
#define sl_btmesh_rsp_fw_dist_server_execute_distribution_step_id        0x03580028
#define sl_btmesh_rsp_fw_dist_server_upload_oob_start_rsp_id             0x08580028
#define sl_btmesh_rsp_fw_dist_server_set_upload_oob_metadata_id          0x09580028
#define sl_btmesh_rsp_fw_dist_server_set_upload_oob_image_size_id        0x0a580028
#define sl_btmesh_rsp_fw_dist_server_set_upload_oob_progress_id          0x0b580028
#define sl_btmesh_rsp_fw_dist_server_get_node_status_by_index_id         0x0c580028
#define sl_btmesh_rsp_fw_dist_server_get_node_status_by_address_id       0x0d580028
#define sl_btmesh_rsp_fw_dist_server_delete_rsp_id                       0x0e580028
#define sl_btmesh_rsp_fw_dist_server_dist_start_rsp_id                   0x13580028
#define sl_btmesh_rsp_fw_dist_server_set_multicast_threshold_id          0x14580028
#define sl_btmesh_rsp_fw_dist_server_delete_all_rsp_id                   0x0f580028
#define sl_btmesh_rsp_fw_dist_server_resume_rsp_id                       0x10580028
#define sl_btmesh_rsp_fw_dist_server_configure_throttle_id               0x15580028

/**
 * @brief The state machine states of the Distribution Server's distribution
 * procedure.
 */
typedef enum
{
  sl_btmesh_fw_dist_server_dist_step_idle                   = 0x0, /**< (0x0) No
                                                                        distribution
                                                                        is
                                                                        active. */
  sl_btmesh_fw_dist_server_dist_step_starting_update        = 0x1, /**< (0x1)
                                                                        Sending
                                                                        Update
                                                                        Start
                                                                        messages. */
  sl_btmesh_fw_dist_server_dist_step_transferring_image     = 0x2, /**< (0x2)
                                                                        Firmware
                                                                        image
                                                                        BLOB
                                                                        transfer
                                                                        is in
                                                                        progress. */
  sl_btmesh_fw_dist_server_dist_step_checking_verification  = 0x3, /**< (0x3)
                                                                        Sending
                                                                        Update
                                                                        Get
                                                                        messages
                                                                        to check
                                                                        verification
                                                                        status. */
  sl_btmesh_fw_dist_server_dist_step_waiting_for_apply      = 0x4, /**< (0x4)
                                                                        Park
                                                                        state if
                                                                        Update
                                                                        Policy
                                                                        is
                                                                        Verify
                                                                        Only:
                                                                        wait for
                                                                        Distribution
                                                                        Apply.
                                                                        If the
                                                                        policy
                                                                        is
                                                                        Verify
                                                                        and
                                                                        Apply,
                                                                        this
                                                                        state is
                                                                        skipped. */
  sl_btmesh_fw_dist_server_dist_step_applying_update        = 0x5, /**< (0x5)
                                                                        Sending
                                                                        Update
                                                                        Apply
                                                                        messages. */
  sl_btmesh_fw_dist_server_dist_step_checking_update_result = 0x6, /**< (0x6)
                                                                        Sending
                                                                        Information
                                                                        Get
                                                                        messages
                                                                        to
                                                                        determine
                                                                        if new
                                                                        firmware
                                                                        was
                                                                        applied. */
  sl_btmesh_fw_dist_server_dist_step_completed              = 0x7, /**< (0x7)
                                                                        Completed
                                                                        successfully
                                                                        for at
                                                                        least
                                                                        one
                                                                        Target
                                                                        Node. */
  sl_btmesh_fw_dist_server_dist_step_failed                 = 0x8, /**< (0x8)
                                                                        Distribution
                                                                        failed
                                                                        for all
                                                                        Target
                                                                        Nodes. */
  sl_btmesh_fw_dist_server_dist_step_cancelling             = 0x9, /**< (0x9)
                                                                        Cancelling
                                                                        an
                                                                        ongoing
                                                                        distribution. */
  sl_btmesh_fw_dist_server_dist_step_suspended              = 0xa  /**< (0xa)
                                                                        Firmware
                                                                        image
                                                                        BLOB
                                                                        transfer
                                                                        is
                                                                        temporarily
                                                                        suspended. */
} sl_btmesh_fw_dist_server_dist_step_t;

/**
 * @brief The Update Policy determines whether the Distribution Server will send
 * Update Apply messages to the Target Nodes immediately, or wait for the
 * Distribution Client to send it the Distribution Apply message.
 */
typedef enum
{
  sl_btmesh_fw_dist_server_dfu_policy_verify_only      = 0x0, /**< (0x0) The
                                                                   Distribution
                                                                   Server will
                                                                   wait for the
                                                                   Distribution
                                                                   Apply
                                                                   message, and
                                                                   then send the
                                                                   Update Apply
                                                                   messages. */
  sl_btmesh_fw_dist_server_dfu_policy_verify_and_apply = 0x1  /**< (0x1) The
                                                                   Distribution
                                                                   Server will
                                                                   send the
                                                                   Update Apply
                                                                   messages
                                                                   immediately,
                                                                   after the
                                                                   firmware
                                                                   image
                                                                   verification
                                                                   is complete. */
} sl_btmesh_fw_dist_server_dfu_policy_t;

/**
 * @brief The phase of the Target Node as determined by the Distribution Server.
 */
typedef enum
{
  sl_btmesh_fw_dist_server_dist_node_phase_idle                 = 0x0, /**<
                                                                            (0x0)
                                                                            No
                                                                            firmware
                                                                            transfer
                                                                            is
                                                                            in
                                                                            progress. */
  sl_btmesh_fw_dist_server_dist_node_phase_transfer_error       = 0x1, /**<
                                                                            (0x1)
                                                                            Error
                                                                            occurred
                                                                            during
                                                                            firmware
                                                                            image
                                                                            BLOB
                                                                            transfer. */
  sl_btmesh_fw_dist_server_dist_node_phase_transfer_in_progress = 0x2, /**<
                                                                            (0x2)
                                                                            Firmware
                                                                            image
                                                                            BLOB
                                                                            transfer
                                                                            is
                                                                            in
                                                                            progress. */
  sl_btmesh_fw_dist_server_dist_node_phase_verifying_update     = 0x3, /**<
                                                                            (0x3)
                                                                            Firmware
                                                                            image
                                                                            transfer
                                                                            completed,
                                                                            and
                                                                            the
                                                                            node
                                                                            is
                                                                            verifying
                                                                            the
                                                                            image. */
  sl_btmesh_fw_dist_server_dist_node_phase_verification_success = 0x4, /**<
                                                                            (0x4)
                                                                            Firmware
                                                                            image
                                                                            verification
                                                                            succeeded. */
  sl_btmesh_fw_dist_server_dist_node_phase_verification_failed  = 0x5, /**<
                                                                            (0x5)
                                                                            Firmware
                                                                            image
                                                                            verification
                                                                            failed. */
  sl_btmesh_fw_dist_server_dist_node_phase_apply_in_progress    = 0x6, /**<
                                                                            (0x6)
                                                                            The
                                                                            node
                                                                            is
                                                                            applying
                                                                            the
                                                                            firmware
                                                                            update. */
  sl_btmesh_fw_dist_server_dist_node_phase_transfer_cancelled   = 0x7, /**<
                                                                            (0x7)
                                                                            Firmware
                                                                            transfer
                                                                            was
                                                                            cancelled. */
  sl_btmesh_fw_dist_server_dist_node_phase_apply_success        = 0x8, /**<
                                                                            (0x8)
                                                                            Firmware
                                                                            was
                                                                            applied
                                                                            successfully. */
  sl_btmesh_fw_dist_server_dist_node_phase_apply_failed         = 0x9, /**<
                                                                            (0x9)
                                                                            Firmware
                                                                            applying
                                                                            failed. */
  sl_btmesh_fw_dist_server_dist_node_phase_unknown              = 0xa  /**<
                                                                            (0xa)
                                                                            The
                                                                            phase
                                                                            of
                                                                            the
                                                                            node
                                                                            has
                                                                            not
                                                                            been
                                                                            retrieved
                                                                            yet. */
} sl_btmesh_fw_dist_server_dist_node_phase_t;

/**
 * @brief Status codes reported by the Firmware Distribution Server.
 */
typedef enum
{
  sl_btmesh_fw_dist_server_dist_status_success                = 0x0, /**< (0x0)
                                                                          The
                                                                          message
                                                                          was
                                                                          processed
                                                                          successfully. */
  sl_btmesh_fw_dist_server_dist_status_out_of_resources       = 0x1, /**< (0x1)
                                                                          Insufficient
                                                                          resources
                                                                          on the
                                                                          node. */
  sl_btmesh_fw_dist_server_dist_status_wrong_phase            = 0x2, /**< (0x2)
                                                                          The
                                                                          operation
                                                                          cannot
                                                                          be
                                                                          performed
                                                                          while
                                                                          the
                                                                          server
                                                                          is in
                                                                          the
                                                                          current
                                                                          phase. */
  sl_btmesh_fw_dist_server_dist_status_internal_error         = 0x3, /**< (0x3)
                                                                          An
                                                                          internal
                                                                          error
                                                                          occurred
                                                                          on the
                                                                          node. */
  sl_btmesh_fw_dist_server_dist_status_fwid_not_found         = 0x4, /**< (0x4)
                                                                          The
                                                                          requested
                                                                          firmware
                                                                          image
                                                                          is not
                                                                          stored
                                                                          on the
                                                                          server. */
  sl_btmesh_fw_dist_server_dist_status_invalid_appkey_index   = 0x5, /**< (0x5)
                                                                          The
                                                                          appkey
                                                                          identified
                                                                          by the
                                                                          appkey
                                                                          index
                                                                          is not
                                                                          known. */
  sl_btmesh_fw_dist_server_dist_status_receivers_list_empty   = 0x6, /**< (0x6)
                                                                          There
                                                                          are no
                                                                          Target
                                                                          Nodes
                                                                          in the
                                                                          Distribution
                                                                          Receivers
                                                                          List. */
  sl_btmesh_fw_dist_server_dist_status_busy_with_distribution = 0x7, /**< (0x7)
                                                                          Another
                                                                          distribution
                                                                          is
                                                                          already
                                                                          in
                                                                          progress. */
  sl_btmesh_fw_dist_server_dist_status_busy_with_upload       = 0x8, /**< (0x8)
                                                                          Another
                                                                          upload
                                                                          is
                                                                          already
                                                                          in
                                                                          progress. */
  sl_btmesh_fw_dist_server_dist_status_uri_not_supported      = 0x9, /**< (0x9)
                                                                          The
                                                                          URI
                                                                          scheme
                                                                          name
                                                                          indicated
                                                                          by the
                                                                          URI is
                                                                          not
                                                                          supported. */
  sl_btmesh_fw_dist_server_dist_status_uri_malformed          = 0xa, /**< (0xa)
                                                                          The
                                                                          format
                                                                          of the
                                                                          URI is
                                                                          invalid. */
  sl_btmesh_fw_dist_server_dist_status_uri_unreachable        = 0xb, /**< (0xb)
                                                                          The
                                                                          URI is
                                                                          currently
                                                                          unreachable. */
  sl_btmesh_fw_dist_server_dist_status_new_fw_not_available   = 0xc, /**< (0xc)
                                                                          No new
                                                                          firmware
                                                                          was
                                                                          available
                                                                          with
                                                                          given
                                                                          the
                                                                          URI +
                                                                          Current
                                                                          FWID. */
  sl_btmesh_fw_dist_server_dist_status_suspend_failed         = 0xd  /**< (0xd)
                                                                          Suspending
                                                                          a
                                                                          distribution
                                                                          failed. */
} sl_btmesh_fw_dist_server_dist_status_t;

/**
 * @addtogroup sl_btmesh_evt_fw_dist_server_nodes_added sl_btmesh_evt_fw_dist_server_nodes_added
 * @{
 * @brief Event indicating nodes were added to the Distributor Server's Nodes
 * List
 */

/** @brief Identifier of the nodes_added event */
#define sl_btmesh_evt_fw_dist_server_nodes_added_id                      0x005800a8

/***************************************************************************//**
 * @brief Data structure of the nodes_added event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_server_nodes_added_s
{
  uint16_t   elem_index;      /**< Server model element index */
  uint16_t   client_address;  /**< Address of the Distribution Client */
  uint16_t   total_nodes;     /**< Total number of nodes now on the list */
  uint16_t   num_added_nodes; /**< Number of nodes added */
  uint8array added_nodes;     /**< Address - firmware index pairs that were
                                   added */
});

typedef struct sl_btmesh_evt_fw_dist_server_nodes_added_s sl_btmesh_evt_fw_dist_server_nodes_added_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_server_nodes_added

/**
 * @addtogroup sl_btmesh_evt_fw_dist_server_nodes_deleted sl_btmesh_evt_fw_dist_server_nodes_deleted
 * @{
 * @brief Event indicating the Nodes List was cleared
 */

/** @brief Identifier of the nodes_deleted event */
#define sl_btmesh_evt_fw_dist_server_nodes_deleted_id                    0x015800a8

/***************************************************************************//**
 * @brief Data structure of the nodes_deleted event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_server_nodes_deleted_s
{
  uint16_t elem_index;     /**< Server model element index */
  uint16_t client_address; /**< Address of the Distribution Client */
});

typedef struct sl_btmesh_evt_fw_dist_server_nodes_deleted_s sl_btmesh_evt_fw_dist_server_nodes_deleted_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_server_nodes_deleted

/**
 * @addtogroup sl_btmesh_evt_fw_dist_server_dist_start_req sl_btmesh_evt_fw_dist_server_dist_start_req
 * @{
 * @brief None
 */

/** @brief Identifier of the dist_start_req event */
#define sl_btmesh_evt_fw_dist_server_dist_start_req_id                   0x025800a8

/***************************************************************************//**
 * @brief Data structure of the dist_start_req event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_server_dist_start_req_s
{
  uint16_t elem_index;        /**< Server model element index */
  uint16_t client_address;    /**< Address of the Distribution Client */
  uint16_t group_address;     /**< Group address of the distribution */
  uuid_128 virtual_address;   /**< If the group address is a VA hash, this
                                   contains the full virtual address label UUID
                                   of the distribution. Otherwise, the value of
                                   this field should be ignored. */
  uint8_t  ttl;               /**< Distribution TTL */
  uint8_t  apply_immediately; /**< Enum @ref
                                   sl_btmesh_fw_dist_server_dfu_policy_t. 1 if
                                   the Update Policy is Verify and Apply,
                                   otherwise 0. */
  uint16_t fw_list_index;     /**< Index of the firmware in the Distributor's FW
                                   List */
  uint16_t timeout_base;      /**< Timeout base */
  uint8_t  transfer_mode;     /**< Transfer Mode to use in the distribution */
});

typedef struct sl_btmesh_evt_fw_dist_server_dist_start_req_s sl_btmesh_evt_fw_dist_server_dist_start_req_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_server_dist_start_req

/**
 * @addtogroup sl_btmesh_evt_fw_dist_server_dist_cancel sl_btmesh_evt_fw_dist_server_dist_cancel
 * @{
 * @brief None
 */

/** @brief Identifier of the dist_cancel event */
#define sl_btmesh_evt_fw_dist_server_dist_cancel_id                      0x035800a8

/***************************************************************************//**
 * @brief Data structure of the dist_cancel event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_server_dist_cancel_s
{
  uint16_t elem_index;     /**< Server model element index */
  uint16_t client_address; /**< Address of the Distribution Client */
});

typedef struct sl_btmesh_evt_fw_dist_server_dist_cancel_s sl_btmesh_evt_fw_dist_server_dist_cancel_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_server_dist_cancel

/**
 * @addtogroup sl_btmesh_evt_fw_dist_server_dist_cancel_complete sl_btmesh_evt_fw_dist_server_dist_cancel_complete
 * @{
 * @brief None
 */

/** @brief Identifier of the dist_cancel_complete event */
#define sl_btmesh_evt_fw_dist_server_dist_cancel_complete_id             0x045800a8

/***************************************************************************//**
 * @brief Data structure of the dist_cancel_complete event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_server_dist_cancel_complete_s
{
  uint16_t elem_index;     /**< Server model element index */
  uint16_t client_address; /**< Address of the Distribution Client */
});

typedef struct sl_btmesh_evt_fw_dist_server_dist_cancel_complete_s sl_btmesh_evt_fw_dist_server_dist_cancel_complete_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_server_dist_cancel_complete

/**
 * @addtogroup sl_btmesh_evt_fw_dist_server_upload_start_metadata sl_btmesh_evt_fw_dist_server_upload_start_metadata
 * @{
 * @brief None
 */

/** @brief Identifier of the upload_start_metadata event */
#define sl_btmesh_evt_fw_dist_server_upload_start_metadata_id            0x055800a8

/***************************************************************************//**
 * @brief Data structure of the upload_start_metadata event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_server_upload_start_metadata_s
{
  uint16_t   elem_index;     /**< Server model element index */
  uint16_t   client_address; /**< Address of the Distribution Client */
  uint8array metadata;       /**< Metadata of the FW */
});

typedef struct sl_btmesh_evt_fw_dist_server_upload_start_metadata_s sl_btmesh_evt_fw_dist_server_upload_start_metadata_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_server_upload_start_metadata

/**
 * @addtogroup sl_btmesh_evt_fw_dist_server_upload_start_req sl_btmesh_evt_fw_dist_server_upload_start_req
 * @{
 * @brief None
 */

/** @brief Identifier of the upload_start_req event */
#define sl_btmesh_evt_fw_dist_server_upload_start_req_id                 0x065800a8

/***************************************************************************//**
 * @brief Data structure of the upload_start_req event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_server_upload_start_req_s
{
  uint16_t        elem_index;     /**< Server model element index */
  uint16_t        client_address; /**< Address of the Distribution Client */
  uint32_t        size;           /**< Upload size */
  sl_bt_uuid_64_t blob_id;        /**< BLOB ID */
  uint8array      fwid;           /**< FWID of the firmware image */
});

typedef struct sl_btmesh_evt_fw_dist_server_upload_start_req_s sl_btmesh_evt_fw_dist_server_upload_start_req_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_server_upload_start_req

/**
 * @addtogroup sl_btmesh_evt_fw_dist_server_upload_cancel sl_btmesh_evt_fw_dist_server_upload_cancel
 * @{
 * @brief None
 */

/** @brief Identifier of the upload_cancel event */
#define sl_btmesh_evt_fw_dist_server_upload_cancel_id                    0x085800a8

/***************************************************************************//**
 * @brief Data structure of the upload_cancel event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_server_upload_cancel_s
{
  uint16_t elem_index;     /**< Server model element index */
  uint16_t client_address; /**< Address of the Distribution Client */
});

typedef struct sl_btmesh_evt_fw_dist_server_upload_cancel_s sl_btmesh_evt_fw_dist_server_upload_cancel_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_server_upload_cancel

/**
 * @addtogroup sl_btmesh_evt_fw_dist_server_dist_state_changed sl_btmesh_evt_fw_dist_server_dist_state_changed
 * @{
 * @brief None
 */

/** @brief Identifier of the dist_state_changed event */
#define sl_btmesh_evt_fw_dist_server_dist_state_changed_id               0x0b5800a8

/***************************************************************************//**
 * @brief Data structure of the dist_state_changed event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_server_dist_state_changed_s
{
  uint16_t elem_index;       /**< Server model element index */
  uint8_t  state;            /**< Enum @ref
                                  sl_btmesh_fw_dist_server_dist_step_t. New
                                  state of the Distribution */
  uint16_t num_active_nodes; /**< Number of active nodes */
});

typedef struct sl_btmesh_evt_fw_dist_server_dist_state_changed_s sl_btmesh_evt_fw_dist_server_dist_state_changed_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_server_dist_state_changed

/**
 * @addtogroup sl_btmesh_evt_fw_dist_server_node_failed sl_btmesh_evt_fw_dist_server_node_failed
 * @{
 * @brief Indicates that an Target Node failed, either due to an error or due to
 * timeout
 */

/** @brief Identifier of the node_failed event */
#define sl_btmesh_evt_fw_dist_server_node_failed_id                      0x0c5800a8

/***************************************************************************//**
 * @brief Data structure of the node_failed event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_server_node_failed_s
{
  uint16_t elem_index;    /**< Server model element index */
  uint16_t address;       /**< Address of the Update Server */
  uint8_t  update_phase;  /**< Enum @ref
                               sl_btmesh_fw_dist_server_dist_node_phase_t.
                               Update Phase of the Update Server. 0xa if not
                               known yet. */
  uint8_t  update_status; /**< Enum @ref
                               sl_btmesh_fw_update_server_update_status_t. Last
                               Update Status message received from the Update
                               Server */
  uint8_t  mbt_status;    /**< Last status code received from the MBT Server.
                               0xa if not known yet. */
  uint8_t  progress;      /**< 0-100, percentage of BLOB Transfer octets
                               delivered and confirmed */
  uint8_t  fw_index;      /**< Index of the firmware being updated */
});

typedef struct sl_btmesh_evt_fw_dist_server_node_failed_s sl_btmesh_evt_fw_dist_server_node_failed_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_server_node_failed

/**
 * @addtogroup sl_btmesh_evt_fw_dist_server_upload_complete_metadata sl_btmesh_evt_fw_dist_server_upload_complete_metadata
 * @{
 * @brief None
 */

/** @brief Identifier of the upload_complete_metadata event */
#define sl_btmesh_evt_fw_dist_server_upload_complete_metadata_id         0x0d5800a8

/***************************************************************************//**
 * @brief Data structure of the upload_complete_metadata event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_server_upload_complete_metadata_s
{
  uint16_t   elem_index;     /**< Server model element index */
  uint16_t   client_address; /**< Address of the Distribution Client */
  uint8array metadata;       /**< Metadata of the firmware image */
});

typedef struct sl_btmesh_evt_fw_dist_server_upload_complete_metadata_s sl_btmesh_evt_fw_dist_server_upload_complete_metadata_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_server_upload_complete_metadata

/**
 * @addtogroup sl_btmesh_evt_fw_dist_server_upload_complete sl_btmesh_evt_fw_dist_server_upload_complete
 * @{
 * @brief None
 */

/** @brief Identifier of the upload_complete event */
#define sl_btmesh_evt_fw_dist_server_upload_complete_id                  0x0e5800a8

/***************************************************************************//**
 * @brief Data structure of the upload_complete event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_server_upload_complete_s
{
  uint16_t        elem_index;     /**< Server model element index */
  uint16_t        client_address; /**< Address of the Distribution Client */
  uint32_t        size;           /**< Upload size */
  sl_bt_uuid_64_t blob_id;        /**< BLOB ID */
  uint8array      fwid;           /**< FWID of the firmware image */
});

typedef struct sl_btmesh_evt_fw_dist_server_upload_complete_s sl_btmesh_evt_fw_dist_server_upload_complete_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_server_upload_complete

/**
 * @addtogroup sl_btmesh_evt_fw_dist_server_upload_failed sl_btmesh_evt_fw_dist_server_upload_failed
 * @{
 * @brief None
 */

/** @brief Identifier of the upload_failed event */
#define sl_btmesh_evt_fw_dist_server_upload_failed_id                    0x0f5800a8

/***************************************************************************//**
 * @brief Data structure of the upload_failed event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_server_upload_failed_s
{
  uint16_t        elem_index;     /**< Server model element index */
  uint16_t        client_address; /**< Address of the Distribution Client */
  uint32_t        size;           /**< Upload size */
  sl_bt_uuid_64_t blob_id;        /**< BLOB ID */
  uint8array      fwid;           /**< FWID of the firmware image */
});

typedef struct sl_btmesh_evt_fw_dist_server_upload_failed_s sl_btmesh_evt_fw_dist_server_upload_failed_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_server_upload_failed

/**
 * @addtogroup sl_btmesh_evt_fw_dist_server_upload_oob_start_uri sl_btmesh_evt_fw_dist_server_upload_oob_start_uri
 * @{
 * @brief URI for the Upload OOB Start request event @ref
 * sl_btmesh_evt_fw_dist_server_upload_oob_start_req that follows.
 */

/** @brief Identifier of the upload_oob_start_uri event */
#define sl_btmesh_evt_fw_dist_server_upload_oob_start_uri_id             0x105800a8

/***************************************************************************//**
 * @brief Data structure of the upload_oob_start_uri event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_server_upload_oob_start_uri_s
{
  uint16_t   elem_index;     /**< Server model element index */
  uint16_t   client_address; /**< Address of the Distribution Client */
  uint8array uri;            /**< URI to use for retrieving a new firmware image */
});

typedef struct sl_btmesh_evt_fw_dist_server_upload_oob_start_uri_s sl_btmesh_evt_fw_dist_server_upload_oob_start_uri_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_server_upload_oob_start_uri

/**
 * @addtogroup sl_btmesh_evt_fw_dist_server_upload_oob_start_req sl_btmesh_evt_fw_dist_server_upload_oob_start_req
 * @{
 * @brief Upload OOB Start request
 *
 * The preceding event @ref sl_btmesh_evt_fw_dist_server_upload_oob_start_uri
 * contains the URI to use in combination with the FWID of this event. The
 * application should execute the Check Firmware OOB procedure and then respond
 * to this event with the command @ref
 * sl_btmesh_fw_dist_server_upload_oob_start_rsp according to the information
 * received OOB.
 */

/** @brief Identifier of the upload_oob_start_req event */
#define sl_btmesh_evt_fw_dist_server_upload_oob_start_req_id             0x115800a8

/***************************************************************************//**
 * @brief Data structure of the upload_oob_start_req event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_server_upload_oob_start_req_s
{
  uint16_t   elem_index;     /**< Server model element index */
  uint16_t   client_address; /**< Address of the Distribution Client */
  uint8array fwid;           /**< FWID to use for retrieving a new firmware
                                  image */
});

typedef struct sl_btmesh_evt_fw_dist_server_upload_oob_start_req_s sl_btmesh_evt_fw_dist_server_upload_oob_start_req_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_server_upload_oob_start_req

/**
 * @addtogroup sl_btmesh_evt_fw_dist_server_fw_delete_req sl_btmesh_evt_fw_dist_server_fw_delete_req
 * @{
 * @brief None
 */

/** @brief Identifier of the fw_delete_req event */
#define sl_btmesh_evt_fw_dist_server_fw_delete_req_id                    0x125800a8

/***************************************************************************//**
 * @brief Data structure of the fw_delete_req event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_server_fw_delete_req_s
{
  uint16_t   elem_index;     /**< Server model element index */
  uint16_t   client_address; /**< Address of the Distribution Client */
  uint8array fwid;           /**< FWID of the firmware that was deleted */
});

typedef struct sl_btmesh_evt_fw_dist_server_fw_delete_req_s sl_btmesh_evt_fw_dist_server_fw_delete_req_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_server_fw_delete_req

/**
 * @addtogroup sl_btmesh_evt_fw_dist_server_fw_delete_all_req sl_btmesh_evt_fw_dist_server_fw_delete_all_req
 * @{
 * @brief None
 */

/** @brief Identifier of the fw_delete_all_req event */
#define sl_btmesh_evt_fw_dist_server_fw_delete_all_req_id                0x135800a8

/***************************************************************************//**
 * @brief Data structure of the fw_delete_all_req event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_server_fw_delete_all_req_s
{
  uint16_t elem_index;     /**< Server model element index */
  uint16_t client_address; /**< Address of the Distribution Client */
});

typedef struct sl_btmesh_evt_fw_dist_server_fw_delete_all_req_s sl_btmesh_evt_fw_dist_server_fw_delete_all_req_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_server_fw_delete_all_req

/**
 * @addtogroup sl_btmesh_evt_fw_dist_server_dist_suspend sl_btmesh_evt_fw_dist_server_dist_suspend
 * @{
 * @brief The distribution was suspended temporarily.
 */

/** @brief Identifier of the dist_suspend event */
#define sl_btmesh_evt_fw_dist_server_dist_suspend_id                     0x155800a8

/***************************************************************************//**
 * @brief Data structure of the dist_suspend event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_server_dist_suspend_s
{
  uint16_t elem_index;     /**< Server model element index */
  uint16_t client_address; /**< Address of the Distribution Client */
});

typedef struct sl_btmesh_evt_fw_dist_server_dist_suspend_s sl_btmesh_evt_fw_dist_server_dist_suspend_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_server_dist_suspend

/**
 * @addtogroup sl_btmesh_evt_fw_dist_server_resume_req sl_btmesh_evt_fw_dist_server_resume_req
 * @{
 * @brief The distribution resuming is requested.
 */

/** @brief Identifier of the resume_req event */
#define sl_btmesh_evt_fw_dist_server_resume_req_id                       0x165800a8

/***************************************************************************//**
 * @brief Data structure of the resume_req event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_server_resume_req_s
{
  uint16_t elem_index;     /**< Server model element index */
  uint16_t client_address; /**< Address of the Distribution Client */
  uint16_t group_address;  /**< Group address to use for the distribution */
  uint16_t app_index;      /**< Appkey index to use for the distribution */
  uint16_t ttl;            /**< TTL to use for the distribution */
  uint16_t update_policy;  /**< Update Policy to use for the distribution */
  uint16_t fw_index;       /**< Distribution Server's Firmware Image List Index
                                of the image to transfer */
  uint16_t timeout_base;   /**< Timeout Base to use for the distribution */
  uint16_t transfer_mode;  /**< BLOB Transfer Mode to use for transferring the
                                image */
});

typedef struct sl_btmesh_evt_fw_dist_server_resume_req_s sl_btmesh_evt_fw_dist_server_resume_req_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_server_resume_req

/***************************************************************************//**
 *
 * Initializes the Firmware Distribution Server model. Before initializing the
 * Distribution Server model, the Update Client and MBT Server models on the
 * same element must be initialized.
 *
 * @param[in] elem_index Server model element index
 * @param[in] max_node_list_size Maximum number of nodes that can be updated
 *   simultaneously
 * @param[in] max_fw_entries Maximum number of firmware images that can be
 *   stored
 * @param[in] max_fw_size Maximum size of a single firmware image
 * @param[in] total_store_size Maximum combined size of all firmware images
 * @param[in] multicast_threshold This value will be passed to the MBT Client.
 *   If the number of servers for any step exceeds or is equal to this number,
 *   the group address will be used. Otherwise, servers will be looped through
 *   one by one. Value of 0 disables the feature.
 * @param[in] oob_supported_uri_scheme_names_len Length of data in @p
 *   oob_supported_uri_scheme_names
 * @param[in] oob_supported_uri_scheme_names To indicate Upload OOB support,
 *   list the supported URI Scheme Names UTF-8 code points here (see
 *   https://www.bluetooth.com/specifications/assigned-numbers/uri-scheme-name-string-mapping/).
 *   To indicate Upload OOB is not supported, give a zero-length string.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_server_init(uint16_t elem_index,
                                          uint16_t max_node_list_size,
                                          uint16_t max_fw_entries,
                                          uint32_t max_fw_size,
                                          uint32_t total_store_size,
                                          uint16_t multicast_threshold,
                                          size_t oob_supported_uri_scheme_names_len,
                                          const uint8_t* oob_supported_uri_scheme_names);

/***************************************************************************//**
 *
 * Deinitializes the Firmware Distribution Server model. The corresponding
 * Update Client and MBT Server models can be deinitialized before or after this
 * model.
 *
 * @param[in] elem_index Server model element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_server_deinit(uint16_t elem_index);

/***************************************************************************//**
 *
 * Response for the upload_start_req event
 *
 * @param[in] elem_index Server model element index
 * @param[in] status Enum @ref sl_btmesh_fw_dist_server_dist_status_t. Status
 *   code to the client
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_server_upload_start_rsp(uint16_t elem_index,
                                                      uint8_t status);

/***************************************************************************//**
 *
 * Execute the next action for the distribution state machine.
 *
 * @param[in] elem_index Server model element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_server_execute_distribution_step(uint16_t elem_index);

/***************************************************************************//**
 *
 * Response to the @ref sl_btmesh_evt_fw_dist_server_upload_oob_start_req event.
 * The application should run the Check Firmware OOB procedure (as detailed in
 * the specification) in order to check that the given URI is reachable, and a
 * new firmware image is available.
 *
 * When the size of the image is known, @ref
 * sl_btmesh_fw_dist_server_set_upload_oob_image_size must be called to inform
 * the stack about the size of the image.
 *
 * If the image has an associated set of metadata, @ref
 * sl_btmesh_fw_dist_server_set_upload_oob_metadata must be called to inform the
 * stack about the image metadata.
 *
 * The command @ref sl_btmesh_fw_dist_server_set_upload_oob_progress must be
 * called at minimum when the transfer completes or an error is encountered. It
 * can be called earlier to inform the stack about the transfer progress.
 *
 * @param[in] elem_index Server model element index
 * @param[in] status_code @parblock
 *   Enum @ref sl_btmesh_fw_dist_server_dist_status_t. Status code to use for
 *   the response. Must be one of the following:
 *     - <b>0:</b> success - retrieving the firmware archive was started or will
 *       start shortly
 *     - <b>1:</b> out of resources
 *     - <b>3:</b> internal error occurred
 *     - <b>9:</b> URI scheme name not supported
 *     - <b>10:</b> URI malformed
 *     - <b>11:</b> URI was unreachable
 *     - <b>12:</b> No new firmware was available
 *
 *   All other status codes are reserved or automatically sent by the stack.
 *   @endparblock
 * @param[in] fwid_len Length of data in @p fwid
 * @param[in] fwid Firmware ID of the image that will be retrieved OOB. The
 *   Firmware ID is retrieved by executing the Check Firmware OOB procedure.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_server_upload_oob_start_rsp(uint16_t elem_index,
                                                          uint8_t status_code,
                                                          size_t fwid_len,
                                                          const uint8_t* fwid);

/***************************************************************************//**
 *
 * Set the metadata of an image retrieved via OOB Upload. This command can be
 * used after @ref sl_btmesh_fw_dist_server_upload_oob_start_rsp, and cannot be
 * used after the OOB Upload completes. If the uploaded firmware image has no
 * associated metadata, this command does not need to be called.
 *
 * @param[in] elem_index Server model element index
 * @param[in] metadata_len Length of data in @p metadata
 * @param[in] metadata Metadata of the image. May be zero-length.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_server_set_upload_oob_metadata(uint16_t elem_index,
                                                             size_t metadata_len,
                                                             const uint8_t* metadata);

/***************************************************************************//**
 *
 * Set the size of the image being retrieved via OOB Upload. This command must
 * be used after @ref sl_btmesh_fw_dist_server_upload_oob_start_rsp, as soon as
 * the size of the image is known. The upload cannot be completed with @ref
 * sl_btmesh_fw_dist_server_set_upload_oob_progress before the size is set.
 *
 * @param[in] elem_index Server model element index
 * @param[in] size Size of the new firmware image
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_server_set_upload_oob_image_size(uint16_t elem_index,
                                                               uint32_t size);

/***************************************************************************//**
 *
 * Set the progress of the Upload OOB image retrieval. Used to inform the stack
 * about the progress, and to indicate when the process is complete.
 *
 * @param[in] elem_index Server model element index
 * @param[in] error @parblock
 *   Set to zero if no error occurred. If nonzero, the transfer will be
 *   considered a failure, and the Upload Phase will be set to Transfer Error. A
 *   nonzero value cannot be used in conjuction with progress 100.
 *
 *   Note that there is no way in the specification to indicate what kind of
 *   error occurred during the OOB retrieval. Only the initial start request can
 *   be rejected with a specific error code; see @ref
 *   sl_btmesh_fw_dist_server_upload_oob_start_rsp.
 *   @endparblock
 * @param[in] progress Progress of the transfer, in percentage. Range: 0-100. If
 *   set 100, the upload will be completed, and the fw_list_index return
 *   parameter is populated with the Firmware List Index that was assigned to
 *   the new image.
 * @param[out] fw_list_index Firmware List Index that was assigned to the image,
 *   if progress was set to 100. Otherwise will be set to 0xffff, and should be
 *   ignored.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_server_set_upload_oob_progress(uint16_t elem_index,
                                                             uint8_t error,
                                                             uint8_t progress,
                                                             uint16_t *fw_list_index);

/***************************************************************************//**
 *
 * Retrieve information about the ongoing update on a node
 *
 * @param[in] elem_index Server model element index
 * @param[in] entry_index Node entry index
 * @param[out] address Address of the Update Server
 * @param[out] retrieved_phase Enum @ref
 *   sl_btmesh_fw_dist_server_dist_node_phase_t. Phase of the Update Server
 * @param[out] update_server_status Enum @ref
 *   sl_btmesh_fw_update_server_update_status_t. Last Update Status message
 *   received from the Update Server
 * @param[out] mbt_server_status Enum @ref sl_btmesh_mbt_server_status_t. Last
 *   Transfer Status message received from the BLOB Transfer Server
 * @param[out] transfer_progress 0-100, percentage of BLOB Transfer octets
 *   delivered and confirmed
 * @param[out] fw_index Index of the firmware being updated
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_server_get_node_status_by_index(uint16_t elem_index,
                                                              uint16_t entry_index,
                                                              uint16_t *address,
                                                              uint8_t *retrieved_phase,
                                                              uint8_t *update_server_status,
                                                              uint8_t *mbt_server_status,
                                                              uint8_t *transfer_progress,
                                                              uint8_t *fw_index);

/***************************************************************************//**
 *
 * Retrieve information about the ongoing update by a node address
 *
 * @param[in] elem_index Server model element index
 * @param[in] node_address Node mesh address
 * @param[out] address Address of the Update Server
 * @param[out] retrieved_phase Enum @ref
 *   sl_btmesh_fw_dist_server_dist_node_phase_t. Phase of the Update Server
 * @param[out] update_server_status Enum @ref
 *   sl_btmesh_fw_update_server_update_status_t. Last Update Status message
 *   received from the Update Server
 * @param[out] mbt_server_status Enum @ref sl_btmesh_mbt_server_status_t. Last
 *   Transfer Status message received from the BLOB Transfer Server
 * @param[out] transfer_progress 0-100, percentage of BLOB Transfer octets
 *   delivered and confirmed
 * @param[out] fw_index Index of the firmware being updated
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_server_get_node_status_by_address(uint16_t elem_index,
                                                                uint16_t node_address,
                                                                uint16_t *address,
                                                                uint8_t *retrieved_phase,
                                                                uint8_t *update_server_status,
                                                                uint8_t *mbt_server_status,
                                                                uint8_t *transfer_progress,
                                                                uint8_t *fw_index);

/***************************************************************************//**
 *
 * Response to the @ref sl_btmesh_evt_fw_dist_server_fw_delete_req event. The
 * application should check that requested firmware image exists and can be
 * deleted. The stack will ignore incoming @ref
 * sl_btmesh_fw_dist_client_start_distribution, @ref
 * sl_btmesh_fw_dist_client_start_upload, @ref sl_btmesh_fw_dist_client_get_fw,
 * @ref sl_btmesh_fw_dist_client_get_fw_by_index, @ref
 * sl_btmesh_fw_dist_client_delete_all_fw commands until the response is called.
 *
 * @param[in] elem_index Server model element index
 * @param[in] status @parblock
 *   Enum @ref sl_btmesh_fw_dist_server_dist_status_t. Status code to use for
 *   the response. Must be one of the following:
 *     - <b>0:</b> success - requested firmware entry deleted by the application
 *     - <b>1:</b> out of resources
 *     - <b>3:</b> internal error occurred
 *
 *   All other status codes are reserved or automatically sent by the stack.
 *   @endparblock
 * @param[in] fwid_len Length of data in @p fwid
 * @param[in] fwid Firmware ID of the image that was requested to be deleted.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_server_delete_rsp(uint16_t elem_index,
                                                uint16_t status,
                                                size_t fwid_len,
                                                const uint8_t* fwid);

/***************************************************************************//**
 *
 * Response to the @ref sl_btmesh_evt_fw_dist_server_dist_start_req event. The
 * stack will ignore incoming @ref sl_btmesh_fw_dist_client_start_distribution,
 * @ref sl_btmesh_fw_dist_client_start_upload, @ref
 * sl_btmesh_fw_dist_client_get_fw, @ref
 * sl_btmesh_fw_dist_client_get_fw_by_index, @ref
 * sl_btmesh_fw_dist_client_delete_all_fw commands until the response is called.
 *
 * @param[in] elem_index Server model element index
 * @param[in] status mesh_dfu_dist_status_success to accept,
 *   mesh_dfu_dist_status_internal_error to reject
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_server_dist_start_rsp(uint16_t elem_index,
                                                    uint16_t status);

/***************************************************************************//**
 *
 * Set the multicast threshold value.
 *
 * @param[in] elem_index Element index
 * @param[in] multicast_threshold This value will be passed to the MBT Client.
 *   If the number of servers for any step exceeds or is equal to this number,
 *   the group address will be used. Otherwise, servers will be looped through
 *   one by one. Value of 0 disables the feature.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_server_set_multicast_threshold(uint16_t elem_index,
                                                             uint16_t multicast_threshold);

/***************************************************************************//**
 *
 * Response to the @ref sl_btmesh_evt_fw_dist_server_fw_delete_all_req event.
 * The application should check that all firmware images can be deleted. The
 * stack will ignore incoming @ref sl_btmesh_fw_dist_client_start_distribution,
 * @ref sl_btmesh_fw_dist_client_start_upload, @ref
 * sl_btmesh_fw_dist_client_get_fw, @ref
 * sl_btmesh_fw_dist_client_get_fw_by_index, @ref
 * sl_btmesh_fw_dist_client_delete_all_fw commands until the response is called.
 *
 * @param[in] elem_index Server model element index
 * @param[in] status @parblock
 *   Enum @ref sl_btmesh_fw_dist_server_dist_status_t. Status code to use for
 *   the response. Must be one of the following:
 *     - <b>0:</b> success - firmware entries deleted by the application
 *     - <b>1:</b> out of resources
 *     - <b>3:</b> internal error occurred
 *
 *   All other status codes are reserved or automatically sent by the stack.
 *   @endparblock
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_server_delete_all_rsp(uint16_t elem_index,
                                                    uint16_t status);

/***************************************************************************//**
 *
 * Response to the @ref sl_btmesh_evt_fw_dist_server_resume_req event.
 *
 * @param[in] elem_index Server model element index
 * @param[in] status @parblock
 *   Enum @ref sl_btmesh_fw_dist_server_dist_status_t. Status code to use for
 *   the response. Must be one of the following:
 *     - <b>0:</b> success - resume distribution
 *     - <b>1:</b> out of resources
 *     - <b>3:</b> internal error occurred
 *
 *   All other status codes are reserved or automatically sent by the stack.
 *   @endparblock
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_server_resume_rsp(uint16_t elem_index,
                                                uint16_t status);

/***************************************************************************//**
 *
 * Configure the sender parameters for the Distributor Server. This command is
 * used to set the parameters that will be used when sending messages to the
 * Target Nodes.
 *
 * @param[in] elem_index Server model element index
 * @param[in] throttle_delay_ms Controls the delay in milliseconds between
 *   batches of messages. Works in conjunction with `throttle_concurrent` to
 *   control message sending rate. When the first transmission in the current
 *   batch completes, the sender will wait for this delay before refilling the
 *   batch up to the `throttle_concurrent` limit with new transmissions.
 * @param[in] throttle_concurrent Defines how many messages are being sent
 *   concurrently. If set to 0, the maximum supported value is used.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_server_configure_throttle(uint16_t elem_index,
                                                        uint16_t throttle_delay_ms,
                                                        uint8_t throttle_concurrent);

/** @} */ // end addtogroup sl_btmesh_fw_dist_server

/**
 * @addtogroup sl_btmesh_fw_dist_client Bluetooth Mesh Firmware Distribution Client Model
 * @{
 *
 * @brief Bluetooth Mesh Firmware Distribution Client Model
 *
 * The Bluetooth Mesh Firmware Distribution Client Model is used to transfer new
 * firmware images to Bluetooth Mesh Firmware Distribution Server Model and
 * monitor progress of a firmware update. As a transport layer Bluetooth Mesh
 * BLOB Transfer (MBT) models are in use.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_fw_dist_client_init_id                             0x00590028
#define sl_btmesh_cmd_fw_dist_client_deinit_id                           0x01590028
#define sl_btmesh_cmd_fw_dist_client_setup_id                            0x02590028
#define sl_btmesh_cmd_fw_dist_client_setup_upload_id                     0x03590028
#define sl_btmesh_cmd_fw_dist_client_set_upload_metadata_id              0x04590028
#define sl_btmesh_cmd_fw_dist_client_get_receivers_id                    0x0e590028
#define sl_btmesh_cmd_fw_dist_client_get_id                              0x0d590028
#define sl_btmesh_cmd_fw_dist_client_start_distribution_id               0x0f590028
#define sl_btmesh_cmd_fw_dist_client_cancel_distribution_id              0x10590028
#define sl_btmesh_cmd_fw_dist_client_apply_distribution_id               0x11590028
#define sl_btmesh_cmd_fw_dist_client_suspend_distribution_id             0x12590028
#define sl_btmesh_cmd_fw_dist_client_add_receivers_id                    0x13590028
#define sl_btmesh_cmd_fw_dist_client_delete_all_receivers_id             0x14590028
#define sl_btmesh_cmd_fw_dist_client_get_capabilities_id                 0x15590028
#define sl_btmesh_cmd_fw_dist_client_get_upload_id                       0x16590028
#define sl_btmesh_cmd_fw_dist_client_start_upload_id                     0x17590028
#define sl_btmesh_cmd_fw_dist_client_cancel_upload_id                    0x19590028
#define sl_btmesh_cmd_fw_dist_client_get_fw_id                           0x1a590028
#define sl_btmesh_cmd_fw_dist_client_get_fw_by_index_id                  0x1b590028
#define sl_btmesh_cmd_fw_dist_client_delete_fw_id                        0x1c590028
#define sl_btmesh_cmd_fw_dist_client_delete_all_fw_id                    0x1d590028
#define sl_btmesh_cmd_fw_dist_client_prepare_oob_upload_id               0x1e590028
#define sl_btmesh_cmd_fw_dist_client_start_oob_upload_id                 0x1f590028
#define sl_btmesh_cmd_fw_dist_client_suspend_upload_id                   0x20590028
#define sl_btmesh_cmd_fw_dist_client_resume_upload_id                    0x21590028
#define sl_btmesh_rsp_fw_dist_client_init_id                             0x00590028
#define sl_btmesh_rsp_fw_dist_client_deinit_id                           0x01590028
#define sl_btmesh_rsp_fw_dist_client_setup_id                            0x02590028
#define sl_btmesh_rsp_fw_dist_client_setup_upload_id                     0x03590028
#define sl_btmesh_rsp_fw_dist_client_set_upload_metadata_id              0x04590028
#define sl_btmesh_rsp_fw_dist_client_get_receivers_id                    0x0e590028
#define sl_btmesh_rsp_fw_dist_client_get_id                              0x0d590028
#define sl_btmesh_rsp_fw_dist_client_start_distribution_id               0x0f590028
#define sl_btmesh_rsp_fw_dist_client_cancel_distribution_id              0x10590028
#define sl_btmesh_rsp_fw_dist_client_apply_distribution_id               0x11590028
#define sl_btmesh_rsp_fw_dist_client_suspend_distribution_id             0x12590028
#define sl_btmesh_rsp_fw_dist_client_add_receivers_id                    0x13590028
#define sl_btmesh_rsp_fw_dist_client_delete_all_receivers_id             0x14590028
#define sl_btmesh_rsp_fw_dist_client_get_capabilities_id                 0x15590028
#define sl_btmesh_rsp_fw_dist_client_get_upload_id                       0x16590028
#define sl_btmesh_rsp_fw_dist_client_start_upload_id                     0x17590028
#define sl_btmesh_rsp_fw_dist_client_cancel_upload_id                    0x19590028
#define sl_btmesh_rsp_fw_dist_client_get_fw_id                           0x1a590028
#define sl_btmesh_rsp_fw_dist_client_get_fw_by_index_id                  0x1b590028
#define sl_btmesh_rsp_fw_dist_client_delete_fw_id                        0x1c590028
#define sl_btmesh_rsp_fw_dist_client_delete_all_fw_id                    0x1d590028
#define sl_btmesh_rsp_fw_dist_client_prepare_oob_upload_id               0x1e590028
#define sl_btmesh_rsp_fw_dist_client_start_oob_upload_id                 0x1f590028
#define sl_btmesh_rsp_fw_dist_client_suspend_upload_id                   0x20590028
#define sl_btmesh_rsp_fw_dist_client_resume_upload_id                    0x21590028

/**
 * @brief The Upload Phase of the Firmware Distribution Server.
 */
typedef enum
{
  sl_btmesh_fw_dist_client_upload_phase_idle                 = 0x0, /**< (0x0)
                                                                         No
                                                                         Upload
                                                                         in
                                                                         progress. */
  sl_btmesh_fw_dist_client_upload_phase_transfer_in_progress = 0x1, /**< (0x1)
                                                                         Upload
                                                                         in
                                                                         progress. */
  sl_btmesh_fw_dist_client_upload_phase_transfer_error       = 0x2, /**< (0x2)
                                                                         Upload
                                                                         or
                                                                         Upload
                                                                         OOB
                                                                         encountered
                                                                         a
                                                                         transfer
                                                                         error. */
  sl_btmesh_fw_dist_client_upload_phase_transfer_completed   = 0x3  /**< (0x3)
                                                                         Upload
                                                                         or
                                                                         Upload
                                                                         OOB
                                                                         completed
                                                                         successfully. */
} sl_btmesh_fw_dist_client_upload_phase_t;

/**
 * @brief The Upload Type of an ongoing or finished upload.
 */
typedef enum
{
  sl_btmesh_fw_dist_client_dist_upload_type_in_band     = 0x0, /**< (0x0)
                                                                    In-band
                                                                    upload via
                                                                    BLOB
                                                                    Transfer. */
  sl_btmesh_fw_dist_client_dist_upload_type_out_of_band = 0x1  /**< (0x1)
                                                                    Out-of-band
                                                                    upload. */
} sl_btmesh_fw_dist_client_dist_upload_type_t;

/**
 * @brief Distribution Server OOB Support capability values.
 */
typedef enum
{
  sl_btmesh_fw_dist_client_dist_oob_not_supported = 0x0, /**< (0x0) Upload OOB
                                                              not supported. */
  sl_btmesh_fw_dist_client_dist_oob_supported     = 0x1  /**< (0x1) Upload OOB
                                                              supported. */
} sl_btmesh_fw_dist_client_dist_oob_t;

/**
 * @brief The Distribution Phase of the Firmware Distribution Server.
 */
typedef enum
{
  sl_btmesh_fw_dist_client_dist_phase_idle        = 0x0, /**< (0x0) No firmware
                                                              distribution is in
                                                              progress. */
  sl_btmesh_fw_dist_client_dist_phase_active      = 0x1, /**< (0x1) Firmware
                                                              distribution in
                                                              progress; sending
                                                              Update Start or
                                                              BLOB Transfer
                                                              messages. */
  sl_btmesh_fw_dist_client_dist_phase_transferred = 0x2, /**< (0x2) Firmware
                                                              distribution in
                                                              progress; BLOB
                                                              Transfer is
                                                              complete, waiting
                                                              for verification.
                                                              If the Update
                                                              Policy is Verify
                                                              Only, the
                                                              Distribution
                                                              Server will park
                                                              here until it
                                                              receives a
                                                              Distribution Apply
                                                              message from the
                                                              Distribution
                                                              Client. */
  sl_btmesh_fw_dist_client_dist_phase_applying    = 0x3, /**< (0x3) Firmware
                                                              distribution in
                                                              progress; sending
                                                              Update Apply
                                                              messages. */
  sl_btmesh_fw_dist_client_dist_phase_completed   = 0x4, /**< (0x4) Firmware
                                                              distribution
                                                              completed
                                                              successfully for
                                                              at least one
                                                              Target Node. */
  sl_btmesh_fw_dist_client_dist_phase_failed      = 0x5, /**< (0x5) Firmware
                                                              distribution
                                                              failed for all
                                                              Target Nodes. */
  sl_btmesh_fw_dist_client_dist_phase_cancelling  = 0x6, /**< (0x6) Cancelling
                                                              firmware
                                                              distribution. */
  sl_btmesh_fw_dist_client_dist_phase_suspended   = 0x7  /**< (0x7) Firmware
                                                              distribution is
                                                              temporarily
                                                              suspended. */
} sl_btmesh_fw_dist_client_dist_phase_t;

/**
 * @addtogroup sl_btmesh_evt_fw_dist_client_receivers_status sl_btmesh_evt_fw_dist_client_receivers_status
 * @{
 * @brief Firmware Distribution Nodes Status
 */

/** @brief Identifier of the receivers_status event */
#define sl_btmesh_evt_fw_dist_client_receivers_status_id                 0x035900a8

/***************************************************************************//**
 * @brief Data structure of the receivers_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_client_receivers_status_s
{
  uint16_t elem_index;      /**< Client model element index */
  uint16_t server_address;  /**< Unicast address of the server */
  uint8_t  status;          /**< Enum @ref
                                 sl_btmesh_fw_dist_server_dist_status_t. Status
                                 code returned */
  uint16_t nodes_list_size; /**< Number of entries currently in the node list */
});

typedef struct sl_btmesh_evt_fw_dist_client_receivers_status_s sl_btmesh_evt_fw_dist_client_receivers_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_client_receivers_status

/**
 * @addtogroup sl_btmesh_evt_fw_dist_client_receivers_list sl_btmesh_evt_fw_dist_client_receivers_list
 * @{
 * @brief Firmware Distribution Node List item
 */

/** @brief Identifier of the receivers_list event */
#define sl_btmesh_evt_fw_dist_client_receivers_list_id                   0x045900a8

/***************************************************************************//**
 * @brief Data structure of the receivers_list event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_client_receivers_list_s
{
  uint16_t elem_index;      /**< Client model element index */
  uint16_t server_address;  /**< Unicast address of the server */
  uint16_t index;           /**< Index of the entry reported */
  uint16_t nodes_list_size; /**< Number of entries currently in the node list */
  uint16_t address;         /**< Address of the Update Server */
  uint8_t  phase;           /**< Enum @ref
                                 sl_btmesh_fw_dist_server_dist_node_phase_t.
                                 Phase of the Update Server */
  uint8_t  update_status;   /**< Last Update Status message received from the
                                 Update Server */
  uint8_t  transfer_status; /**< Last Transfer Status message received from the
                                 BLOB Transfer Server */
  uint8_t  progress;        /**< 0-100, percentage of BLOB Transfer octets
                                 delivered and confirmed */
  uint8_t  fw_index;        /**< Index of the firmware being updated */
});

typedef struct sl_btmesh_evt_fw_dist_client_receivers_list_s sl_btmesh_evt_fw_dist_client_receivers_list_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_client_receivers_list

/**
 * @addtogroup sl_btmesh_evt_fw_dist_client_receivers_list_end sl_btmesh_evt_fw_dist_client_receivers_list_end
 * @{
 * @brief Firmware Distribution Node List end
 */

/** @brief Identifier of the receivers_list_end event */
#define sl_btmesh_evt_fw_dist_client_receivers_list_end_id               0x055900a8

/***************************************************************************//**
 * @brief Data structure of the receivers_list_end event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_client_receivers_list_end_s
{
  uint16_t elem_index;      /**< Client model element index */
  uint16_t server_address;  /**< Unicast address of the server */
  uint16_t start_index;     /**< Starting index */
  uint16_t num_reported;    /**< Number of entries reported */
  uint16_t nodes_list_size; /**< Number of entries currently in the node list */
});

typedef struct sl_btmesh_evt_fw_dist_client_receivers_list_end_s sl_btmesh_evt_fw_dist_client_receivers_list_end_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_client_receivers_list_end

/**
 * @addtogroup sl_btmesh_evt_fw_dist_client_capabilities_status sl_btmesh_evt_fw_dist_client_capabilities_status
 * @{
 * @brief Firmware Distribution Capabilities Status
 */

/** @brief Identifier of the capabilities_status event */
#define sl_btmesh_evt_fw_dist_client_capabilities_status_id              0x065900a8

/***************************************************************************//**
 * @brief Data structure of the capabilities_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_client_capabilities_status_s
{
  uint16_t   elem_index;             /**< Client model element index */
  uint16_t   server_address;         /**< Unicast address of the server */
  uint16_t   max_nodes_list_size;    /**< Maximum number of entries the node
                                          list can contain */
  uint16_t   max_fw_list_size;       /**< Maximum number of entries the Firmware
                                          List can contain */
  uint32_t   max_fw_size;            /**< Maximum size of a single firmware
                                          image */
  uint32_t   max_upload_space;       /**< Maximum number of bytes for firmware
                                          images */
  uint32_t   remaining_upload_space; /**< Remaining number of bytes for firmware
                                          images */
  uint8_t    oob_supported;          /**< Enum @ref
                                          sl_btmesh_fw_dist_client_dist_oob_t.
                                          OOB firmware retrieval (using the
                                          Update URI received from an Update
                                          Server) supported */
  uint8array uri_scheme_names;       /**< Supported URI scheme names. Empty if
                                          OOB is not supported. */
});

typedef struct sl_btmesh_evt_fw_dist_client_capabilities_status_s sl_btmesh_evt_fw_dist_client_capabilities_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_client_capabilities_status

/**
 * @addtogroup sl_btmesh_evt_fw_dist_client_distribution_status sl_btmesh_evt_fw_dist_client_distribution_status
 * @{
 * @brief Firmware Distribution Status
 */

/** @brief Identifier of the distribution_status event */
#define sl_btmesh_evt_fw_dist_client_distribution_status_id              0x075900a8

/***************************************************************************//**
 * @brief Data structure of the distribution_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_client_distribution_status_s
{
  uint16_t elem_index;        /**< Client model element index */
  uint16_t server_address;    /**< Unicast address of the server */
  uint8_t  status;            /**< Enum @ref
                                   sl_btmesh_fw_dist_server_dist_status_t.
                                   Status code returned */
  uint8_t  phase;             /**< Enum @ref
                                   sl_btmesh_fw_dist_client_dist_phase_t. Phase
                                   of the Distribution Server */
  uint16_t group_address;     /**< Distribution group address. Ignore if no
                                   distribution is active. */
  uint16_t appkey_index;      /**< Distribution app key index. Ignore if no
                                   distribution is active. */
  uint8_t  distributor_ttl;   /**< TTL used by the Distributor. Ignore if no
                                   distribution is active. */
  uint16_t timeout_base;      /**< Timeout Base value used in BLOB Transfer */
  uint8_t  transfer_mode;     /**< Transfer Mode used in BLOB Transfer */
  uint8_t  apply_immediately; /**< 1 if the Update Policy is Verify and Apply,
                                   otherwise 0. Ignore if no distribution is
                                   active. */
  uint16_t fw_list_index;     /**< Index of the firmware in the Distributor's
                                   Firmware List */
});

typedef struct sl_btmesh_evt_fw_dist_client_distribution_status_s sl_btmesh_evt_fw_dist_client_distribution_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_client_distribution_status

/**
 * @addtogroup sl_btmesh_evt_fw_dist_client_upload_status sl_btmesh_evt_fw_dist_client_upload_status
 * @{
 * @brief Firmware Distribution Upload Status
 */

/** @brief Identifier of the upload_status event */
#define sl_btmesh_evt_fw_dist_client_upload_status_id                    0x085900a8

/***************************************************************************//**
 * @brief Data structure of the upload_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_client_upload_status_s
{
  uint16_t   elem_index;     /**< Client model element index */
  uint16_t   server_address; /**< Unicast address of the server */
  uint8_t    status;         /**< Enum @ref
                                  sl_btmesh_fw_dist_server_dist_status_t. Status
                                  code returned */
  uint8_t    phase;          /**< Upload Phase of the Distribution Server */
  uint8_t    type;           /**< Enum @ref
                                  sl_btmesh_fw_dist_client_dist_upload_type_t.
                                  Upload Type of an upload (0:in-band
                                  1:out-of-band). Ignore if no upload is active. */
  uint8_t    progress;       /**< 0-100, percentage of BLOB Transfer octets or
                                  out-of-band data delivered and confirmed.
                                  Ignore if no upload is active. */
  uint8array fwid;           /**< FWID of the firmware being transferred. Empty
                                  if no upload is active. */
});

typedef struct sl_btmesh_evt_fw_dist_client_upload_status_s sl_btmesh_evt_fw_dist_client_upload_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_client_upload_status

/**
 * @addtogroup sl_btmesh_evt_fw_dist_client_fw_status sl_btmesh_evt_fw_dist_client_fw_status
 * @{
 * @brief Firmware Distribution Firmware Status
 */

/** @brief Identifier of the fw_status event */
#define sl_btmesh_evt_fw_dist_client_fw_status_id                        0x095900a8

/***************************************************************************//**
 * @brief Data structure of the fw_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_client_fw_status_s
{
  uint16_t   elem_index;     /**< Client model element index */
  uint16_t   server_address; /**< Unicast address of the server */
  uint8_t    status;         /**< Enum @ref
                                  sl_btmesh_fw_dist_server_dist_status_t. Status
                                  code returned */
  uint16_t   num_entries;    /**< Number of images stored on the Distributor */
  uint16_t   index;          /**< Index of the queried firmware on the server.
                                  0xffff if not found. */
  uint8array fwid;           /**< FWID of the firmware that was queried. Empty
                                  if get_by_index was used and no such index
                                  exists. */
});

typedef struct sl_btmesh_evt_fw_dist_client_fw_status_s sl_btmesh_evt_fw_dist_client_fw_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_client_fw_status

/**
 * @addtogroup sl_btmesh_evt_fw_dist_client_upload_complete sl_btmesh_evt_fw_dist_client_upload_complete
 * @{
 * @brief Firmware Distribution Upload procedure was completed successfully
 */

/** @brief Identifier of the upload_complete event */
#define sl_btmesh_evt_fw_dist_client_upload_complete_id                  0x0a5900a8

/***************************************************************************//**
 * @brief Data structure of the upload_complete event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_client_upload_complete_s
{
  uint16_t        elem_index;     /**< Client model element index */
  uint16_t        server_address; /**< Unicast address of the server */
  sl_bt_uuid_64_t blob_id;        /**< The ID of the BLOB. */
  uint32_t        blob_size;      /**< The size of the BLOB. */
  uint8array      fwid;           /**< FWID of the firmware that was uploaded. */
});

typedef struct sl_btmesh_evt_fw_dist_client_upload_complete_s sl_btmesh_evt_fw_dist_client_upload_complete_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_client_upload_complete

/**
 * @addtogroup sl_btmesh_evt_fw_dist_client_upload_failed sl_btmesh_evt_fw_dist_client_upload_failed
 * @{
 * @brief Firmware Distribution Upload procedure failed
 */

/** @brief Identifier of the upload_failed event */
#define sl_btmesh_evt_fw_dist_client_upload_failed_id                    0x0b5900a8

/***************************************************************************//**
 * @brief Data structure of the upload_failed event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_dist_client_upload_failed_s
{
  uint16_t        elem_index;       /**< Client model element index */
  uint16_t        server_address;   /**< Unicast address of the server */
  uint8_t         status_code;      /**< Reason for failure: BLOB Transfer
                                         status code, or 0xff for timeout, 0 if
                                         a DFU layer error happened (failure
                                         result in dist_status_code) */
  uint8_t         dist_status_code; /**< Reason for failure: Distribution status
                                         code, or 0xff for timeout, 0 if a MBT
                                         layer error happened (failure result in
                                         status_code) */
  uint8_t         progress;         /**< 0-100, percentage of BLOB Transfer
                                         octets data delivered and confirmed. */
  sl_bt_uuid_64_t blob_id;          /**< The ID of the BLOB. */
  uint32_t        blob_size;        /**< The size of the BLOB. */
  uint8array      fwid;             /**< FWID of the firmware which uploading
                                         failed. */
});

typedef struct sl_btmesh_evt_fw_dist_client_upload_failed_s sl_btmesh_evt_fw_dist_client_upload_failed_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_dist_client_upload_failed

/***************************************************************************//**
 *
 * Initialize the Firmware Distribution Client model. An MBT Client on the same
 * element must have been initialized prior to making this call.
 *
 * @param[in] elem_index Client model element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_init(uint16_t elem_index);

/***************************************************************************//**
 *
 * Deinitialize the Firmware Distribution Client model. The corresponding MBT
 * Client instance can be deinitialized before or after the Distribution Client.
 *
 * @param[in] elem_index Client model element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_deinit(uint16_t elem_index);

/***************************************************************************//**
 *
 * Set up the communication parameters for a Distribution Client instance.
 *
 * @param[in] elem_index Client model element index
 * @param[in] appkey_index Application key index
 * @param[in] ttl The time-to-live value to use. Valid values: 0, range:[2-127]
 *   and 255.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_setup(uint16_t elem_index,
                                           uint16_t appkey_index,
                                           uint8_t ttl);

/***************************************************************************//**
 *
 * Set up an in-band Upload of a firmware image. Note that only one upload can
 * be executed at a time. This command must be used before setting the upload
 * image metadata via @ref sl_btmesh_fw_dist_client_set_upload_metadata.
 *
 * @param[in] elem_index Client model element index
 * @param[in] server_address Address of the target Distributor
 * @param[in] ttl The time-to-live value to use for the upload. Valid values: 0,
 *   range:[2-127] and 255.
 * @param[in] timeout_base Timeout Base value for the BLOB Transfer
 * @param[in] blob_id BLOB ID to use
 * @param[in] size Size of the firmware image
 * @param[in] fwid_len Length of data in @p fwid
 * @param[in] fwid Firmware ID of the firmware image
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_setup_upload(uint16_t elem_index,
                                                  uint16_t server_address,
                                                  uint8_t ttl,
                                                  uint16_t timeout_base,
                                                  sl_bt_uuid_64_t blob_id,
                                                  uint32_t size,
                                                  size_t fwid_len,
                                                  const uint8_t* fwid);

/***************************************************************************//**
 *
 * Set the metadata for an in-band Upload of a firmware image.
 *
 * @param[in] elem_index Client model element index
 * @param[in] metadata_len Length of data in @p metadata
 * @param[in] metadata Metadata for the firmware image
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_set_upload_metadata(uint16_t elem_index,
                                                         size_t metadata_len,
                                                         const uint8_t* metadata);

/***************************************************************************//**
 *
 * Send a Receivers Get message. This function is used to get a slice of the
 * Receivers List on the Distribution Server. The response is a Receivers List
 * message.
 *
 * @param[in] elem_index Client model element index
 * @param[in] dst Address of the Distributor
 * @param[in] start_index Starting index of node statuses to request
 * @param[in] max_entries Maximum number of node statuses to request
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_dist_client_receivers_list
 *   - @ref sl_btmesh_evt_fw_dist_client_receivers_list_end
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_get_receivers(uint16_t elem_index,
                                                   uint16_t dst,
                                                   uint16_t start_index,
                                                   uint16_t max_entries);

/***************************************************************************//**
 *
 * Send a Distribution Get message. This function is used to determine the
 * parameters of an ongoing distribution, if any. The response is a Distribution
 * Status message.
 *
 * @param[in] elem_index Client model element index
 * @param[in] dst Address of the Distributor
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_dist_client_distribution_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_get(uint16_t elem_index, uint16_t dst);

/***************************************************************************//**
 *
 * Send a Distribution Start message. This function is used to start a new
 * distribution process. The Receivers List must have been populated via @ref
 * sl_btmesh_fw_dist_client_add_receivers, and the firmware image must have been
 * uploaded to the Distribution Server. The response is a Distribution Status
 * message.
 *
 * @param[in] elem_index Client model element index
 * @param[in] dst Address of the Distributor
 * @param[in] dist_appkey_index Application key to use for the communication
 *   between the Distributor and Target Nodes
 * @param[in] dist_ttl The time-to-live value for the Distributor to use when
 *   communicating with the Target Nodes. Valid values: 0, range:[2-127] and 255
 * @param[in] dist_timeout_base The Timeout Base value to use in the firmware
 *   image BLOB Transfer
 * @param[in] transfer_mode 1 = use Push Mode, 2 = use Pull Mode (typically for
 *   LPNs)
 * @param[in] update_policy Enum @ref sl_btmesh_fw_dist_server_dfu_policy_t. 0 =
 *   Verify Only (Initiator sends Distribution Apply to Distributor to apply the
 *   update), 1 = Verify and Apply (Distributor applies update automatically)
 * @param[in] fw_list_index Index of the firmware image in the Distributor's
 *   Firmware List
 * @param[in] group_address Group address to use. Zero to only use unicast.
 * @param[in] virtual_address If the group address is a VA hash, this needs to
 *   contain the full virtual address label UUID. Otherwise, the value of this
 *   field will be ignored.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_dist_client_distribution_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_start_distribution(uint16_t elem_index,
                                                        uint16_t dst,
                                                        uint8_t dist_appkey_index,
                                                        uint8_t dist_ttl,
                                                        uint16_t dist_timeout_base,
                                                        uint8_t transfer_mode,
                                                        uint8_t update_policy,
                                                        uint16_t fw_list_index,
                                                        uint16_t group_address,
                                                        uuid_128 virtual_address);

/***************************************************************************//**
 *
 * Send a Distribution Cancel message. This function is used to cancel a
 * distribution process. The response is a Distribution Status message.
 *
 * @param[in] elem_index Client model element index
 * @param[in] dst Address of the Distributor
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_dist_client_distribution_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_cancel_distribution(uint16_t elem_index,
                                                         uint16_t dst);

/***************************************************************************//**
 *
 * Send a Distribution Apply message. This function is used to trigger applying
 * the firmware update on the Target Nodes. This should only be used if the
 * Update Policy of the distribution is Verify Only. The response is a
 * Distribution Status message.
 *
 * @param[in] elem_index Client model element index
 * @param[in] dst Address of the Distributor
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_dist_client_distribution_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_apply_distribution(uint16_t elem_index,
                                                        uint16_t dst);

/***************************************************************************//**
 *
 * Send a Distribution Suspend message. This function is used to suspend a
 * distribution when the BLOB Transfer is still in progress. It cannot be used
 * after the image has already been fully transferred. The response is a
 * Distribution Status message.
 *
 * @param[in] elem_index Client model element index
 * @param[in] dst Address of the Distributor
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_dist_client_distribution_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_suspend_distribution(uint16_t elem_index,
                                                          uint16_t dst);

/***************************************************************************//**
 *
 * Send a Receivers Add message. This function is used to add Target Nodes to
 * the Distribution Server's Receivers List. The response is a Receivers Status
 * message.
 *
 * @param[in] elem_index Client model element index
 * @param[in] dst Address of the Distributor
 * @param[in] nodes_len Length of data in @p nodes
 * @param[in] nodes List of (address [uint16], fw_index [uint8]) pairs.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_dist_client_receivers_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_add_receivers(uint16_t elem_index,
                                                   uint16_t dst,
                                                   size_t nodes_len,
                                                   const uint8_t* nodes);

/***************************************************************************//**
 *
 * Send a Receivers Delete All message. This function is used to clear the
 * Receivers List on the Distribution Server. The response is a Receivers Status
 * message.
 *
 * @param[in] elem_index Client model element index
 * @param[in] dst Address of the Distributor
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_dist_client_receivers_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_delete_all_receivers(uint16_t elem_index,
                                                          uint16_t dst);

/***************************************************************************//**
 *
 * Send a Capabilities Get message. This function is used to determine the
 * Distribution Capabilities of a Distribution Server. The response is a
 * Capabilities Status message.
 *
 * @param[in] elem_index Client model element index
 * @param[in] dst Address of the Distributor
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_dist_client_capabilities_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_get_capabilities(uint16_t elem_index,
                                                      uint16_t dst);

/***************************************************************************//**
 *
 * Send an Upload Get message. This function is used to check if an upload is
 * ongoing, and what its progress is. The response is an Upload Status message.
 *
 * @param[in] elem_index Client model element index
 * @param[in] dst Address of the Distributor
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_dist_client_upload_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_get_upload(uint16_t elem_index,
                                                uint16_t dst);

/***************************************************************************//**
 *
 * Start an upload previously set up with @ref
 * sl_btmesh_fw_dist_client_setup_upload. Calling this function will send an
 * Upload Start message. The response is an Upload Status message. Note that if
 * the remote end does not respond, this function may need to be retried. Note
 * that upon sending the Upload Start message a response timer is started for a
 * time frame of: (10000 * (timeout_base + 2)) + (100 * ttl) in units of ms.
 *
 * @param[in] elem_index Client model element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_dist_client_upload_status
 *   - @ref sl_btmesh_evt_fw_dist_client_upload_failed
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_start_upload(uint16_t elem_index);

/***************************************************************************//**
 *
 * Cancel an upload previously started with @ref
 * sl_btmesh_fw_dist_client_start_upload.
 *
 * @param[in] elem_index Client model element index
 * @param[in] dst Address of the Distributor
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_dist_client_upload_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_cancel_upload(uint16_t elem_index,
                                                   uint16_t dst);

/***************************************************************************//**
 *
 * Send a Firmware Get message. This function is used to check if a firmware
 * image with the given FWID exists on the Distribution Server, and if it does,
 * to retrieve its Firmware List Index. The response is a Firmware Status
 * message.
 *
 * @param[in] elem_index Client model element index
 * @param[in] dst Address of the Distributor
 * @param[in] fwid_len Length of data in @p fwid
 * @param[in] fwid Firmware ID of the firmware image
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_dist_client_fw_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_get_fw(uint16_t elem_index,
                                            uint16_t dst,
                                            size_t fwid_len,
                                            const uint8_t* fwid);

/***************************************************************************//**
 *
 * Send a Firmware Get By Index message. This function is used to check if a
 * firmware with a particular index exists on the Distribution Server, and if it
 * does, to retrieve its FWID. The response is a Firmware Status message.
 *
 * @param[in] elem_index Client model element index
 * @param[in] dst Address of the Distributor
 * @param[in] index Index to check
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_dist_client_fw_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_get_fw_by_index(uint16_t elem_index,
                                                     uint16_t dst,
                                                     uint16_t index);

/***************************************************************************//**
 *
 * Send a Firmware Delete message. This function is used to delete a particular
 * firmware on the Distribution Server. The response is a Firmware Status
 * message.
 *
 * @param[in] elem_index Client model element index
 * @param[in] dst Address of the Distributor
 * @param[in] fwid_len Length of data in @p fwid
 * @param[in] fwid Firmware ID of the firmware image
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_dist_client_fw_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_delete_fw(uint16_t elem_index,
                                               uint16_t dst,
                                               size_t fwid_len,
                                               const uint8_t* fwid);

/***************************************************************************//**
 *
 * Send a Firmware Delete All message. This function is used to delete all
 * firmware images on the Distribution Server. The response is a Firmware Status
 * message.
 *
 * @param[in] elem_index Client model element index
 * @param[in] dst Address of the Distributor
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_dist_client_fw_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_delete_all_fw(uint16_t elem_index,
                                                   uint16_t dst);

/***************************************************************************//**
 *
 * Prepare an OOB Upload. This must be called before calling the @ref
 * sl_btmesh_fw_dist_client_start_oob_upload command. Calling this multiple
 * times will overwrite the state of the previous OOB Upload.
 *
 * @param[in] elem_index Client model element index
 * @param[in] fwid_len Length of data in @p fwid
 * @param[in] fwid FWID used to form the query string for the Distribution
 *   Server's Check Firmware OOB and Store Firmware OOB procedures.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_prepare_oob_upload(uint16_t elem_index,
                                                        size_t fwid_len,
                                                        const uint8_t* fwid);

/***************************************************************************//**
 *
 * Start an OOB Upload. The response is an @ref
 * sl_btmesh_evt_fw_dist_client_upload_status event. This command must be
 * repeated by the application if the remote end does not respond. If the
 * event's status field is zero, the request was accepted; otherwise it was
 * rejected. If the event indicates that the request was accepted, the type
 * field will be set to 1 to denote an out-of-band upload.
 *
 * @ref sl_btmesh_fw_dist_client_prepare_oob_upload must have been called prior
 * to using this command.
 *
 * To poll the status of the OOB Upload, the application can either repeat this
 * command, or use the command @ref sl_btmesh_fw_dist_client_get_upload.
 *
 * When a @ref sl_btmesh_evt_fw_dist_client_upload_status event with type set to
 * 1 and progress set to 100 is received, the application can consider the
 * upload complete.
 *
 * @param[in] elem_index Client model element index
 * @param[in] dst Destination address
 * @param[in] appkey_index Application key index to use for encrypting the
 *   message.
 * @param[in] ttl The time-to-live value to use. Valid values: 0, range:[2-127]
 *   and 255.
 * @param[in] uri_len Length of data in @p uri
 * @param[in] uri URI used to form the query string for the Distribution
 *   Server's Check Firmware OOB and Store Firmware OOB procedures.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_dist_client_upload_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_start_oob_upload(uint16_t elem_index,
                                                      uint16_t dst,
                                                      uint16_t appkey_index,
                                                      uint8_t ttl,
                                                      size_t uri_len,
                                                      const uint8_t* uri);

/***************************************************************************//**
 *
 * Suspend an Upload's BLOB Transfer. An Upload that isn't yet in the BLOB
 * Transfer phase, or which has failed, cannot be suspended. The suspension is
 * entirely local; there is no message sent over the air. The remote receiver's
 * BLOB Transfer Server will go to the suspended phase when the Upload Timeout
 * timer expires. When the Upload is resumed, it will continue from where it
 * previously left off.
 *
 * @param[in] elem_index Client model element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_suspend_upload(uint16_t elem_index);

/***************************************************************************//**
 *
 * Resume a suspended upload. The command @ref
 * sl_btmesh_fw_dist_client_start_upload must be used after this command, and it
 * may need to be repeated if the remote end does not respond. When the Upload
 * is resumed, the BLOB Transfer will continue from where it previously left
 * off.
 *
 * @param[in] elem_index Client model element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_dist_client_resume_upload(uint16_t elem_index);

/** @} */ // end addtogroup sl_btmesh_fw_dist_client

/**
 * @addtogroup sl_btmesh_remote_provisioning_client Bluetooth Mesh Remote Provisioning Client Model
 * @{
 *
 * @brief Bluetooth Mesh Remote Provisioning Client Model
 *
 * Remote Provisioning Client model is used to provision devices which are not
 * within one hop from the provisioner.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_remote_provisioning_client_init_id                 0x00300028
#define sl_btmesh_cmd_remote_provisioning_client_get_scan_capabilities_id 0x01300028
#define sl_btmesh_cmd_remote_provisioning_client_start_scan_id           0x02300028
#define sl_btmesh_cmd_remote_provisioning_client_get_scan_status_id      0x03300028
#define sl_btmesh_cmd_remote_provisioning_client_stop_scan_id            0x04300028
#define sl_btmesh_cmd_remote_provisioning_client_start_extended_scan_id  0x05300028
#define sl_btmesh_cmd_remote_provisioning_client_open_link_id            0x06300028
#define sl_btmesh_cmd_remote_provisioning_client_get_link_status_id      0x07300028
#define sl_btmesh_cmd_remote_provisioning_client_close_link_id           0x08300028
#define sl_btmesh_rsp_remote_provisioning_client_init_id                 0x00300028
#define sl_btmesh_rsp_remote_provisioning_client_get_scan_capabilities_id 0x01300028
#define sl_btmesh_rsp_remote_provisioning_client_start_scan_id           0x02300028
#define sl_btmesh_rsp_remote_provisioning_client_get_scan_status_id      0x03300028
#define sl_btmesh_rsp_remote_provisioning_client_stop_scan_id            0x04300028
#define sl_btmesh_rsp_remote_provisioning_client_start_extended_scan_id  0x05300028
#define sl_btmesh_rsp_remote_provisioning_client_open_link_id            0x06300028
#define sl_btmesh_rsp_remote_provisioning_client_get_link_status_id      0x07300028
#define sl_btmesh_rsp_remote_provisioning_client_close_link_id           0x08300028

/**
 * @brief Possible remote provisioning link states.
 */
typedef enum
{
  sl_btmesh_remote_provisioning_client_link_state_idle                     = 0x0, /**<
                                                                                       (0x0)
                                                                                       Idle
                                                                                       state */
  sl_btmesh_remote_provisioning_client_link_state_opening                  = 0x1, /**<
                                                                                       (0x1)
                                                                                       Link
                                                                                       Opening */
  sl_btmesh_remote_provisioning_client_link_state_active                   = 0x2, /**<
                                                                                       (0x2)
                                                                                       Link
                                                                                       active */
  sl_btmesh_remote_provisioning_client_link_state_outbound_packet_transfer = 0x3, /**<
                                                                                       (0x3)
                                                                                       Outbound
                                                                                       packet
                                                                                       transfer */
  sl_btmesh_remote_provisioning_client_link_state_closing                  = 0x4  /**<
                                                                                       (0x4)
                                                                                       Link
                                                                                       Closing */
} sl_btmesh_remote_provisioning_client_link_state_t;

/**
 * @brief NPPI operation type
 */
typedef enum
{
  sl_btmesh_remote_provisioning_client_nppi_device_key_update = 0x0,  /**<
                                                                           (0x0)
                                                                           Update
                                                                           Device
                                                                           Key */
  sl_btmesh_remote_provisioning_client_nppi_address_update    = 0x1,  /**<
                                                                           (0x1)
                                                                           Update
                                                                           device
                                                                           address.
                                                                           Note
                                                                           that
                                                                           also
                                                                           DCD
                                                                           will
                                                                           be
                                                                           updated
                                                                           if
                                                                           updated
                                                                           DCD
                                                                           is
                                                                           available */
  sl_btmesh_remote_provisioning_client_nppi_dcd_update        = 0x2,  /**<
                                                                           (0x2)
                                                                           Update
                                                                           device
                                                                           DCD */
  sl_btmesh_remote_provisioning_client_remote_provisioning    = 0xff  /**<
                                                                           (0xff)
                                                                           Provision
                                                                           a
                                                                           remote
                                                                           device. */
} sl_btmesh_remote_provisioning_client_nppi_operation_t;

/**
 * @addtogroup sl_btmesh_evt_remote_provisioning_client_scan_capabilities sl_btmesh_evt_remote_provisioning_client_scan_capabilities
 * @{
 * @brief Scan capabilities of the remote provisioning server.
 */

/** @brief Identifier of the scan_capabilities event */
#define sl_btmesh_evt_remote_provisioning_client_scan_capabilities_id    0x003000a8

/***************************************************************************//**
 * @brief Data structure of the scan_capabilities event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_remote_provisioning_client_scan_capabilities_s
{
  uint16_t result;    /**< SL_STATUS_OK if successful. Error code otherwise. */
  uint16_t server;    /**< Server address. */
  uint8_t  max_items; /**< The maximum number of UUIDs that can be reported
                           during scanning. */
  uint8_t  active;    /**< Indication if active scan is supported. */
});

typedef struct sl_btmesh_evt_remote_provisioning_client_scan_capabilities_s sl_btmesh_evt_remote_provisioning_client_scan_capabilities_t;

/** @} */ // end addtogroup sl_btmesh_evt_remote_provisioning_client_scan_capabilities

/**
 * @addtogroup sl_btmesh_evt_remote_provisioning_client_scan_status sl_btmesh_evt_remote_provisioning_client_scan_status
 * @{
 * @brief The Remote Provisioning Scan Status.
 */

/** @brief Identifier of the scan_status event */
#define sl_btmesh_evt_remote_provisioning_client_scan_status_id          0x013000a8

/***************************************************************************//**
 * @brief Data structure of the scan_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_remote_provisioning_client_scan_status_s
{
  uint16_t result;      /**< SL_STATUS_OK if successful. Error code otherwise. */
  uint16_t server;      /**< Server address. */
  uint8_t  status;      /**< Enum @ref
                             sl_btmesh_remote_provisioning_server_status_t.
                             server status. */
  uint8_t  state;       /**< Enum @ref
                             sl_btmesh_remote_provisioning_server_scan_state_t.
                             Scan state. */
  uint8_t  max_reports; /**< Maximum number of scanned items to be reported. */
  uint8_t  timeout_sec; /**< Maximum available time for time left until the end
                             of the Remote Provisioning Scan procedure. */
});

typedef struct sl_btmesh_evt_remote_provisioning_client_scan_status_s sl_btmesh_evt_remote_provisioning_client_scan_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_remote_provisioning_client_scan_status

/**
 * @addtogroup sl_btmesh_evt_remote_provisioning_client_scan_report sl_btmesh_evt_remote_provisioning_client_scan_report
 * @{
 * @brief The Remote Provisioning Scan Report
 */

/** @brief Identifier of the scan_report event */
#define sl_btmesh_evt_remote_provisioning_client_scan_report_id          0x023000a8

/***************************************************************************//**
 * @brief Data structure of the scan_report event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_remote_provisioning_client_scan_report_s
{
  uint16_t server; /**< Server address. */
  int8_t   rssi;   /**< measured signal stength in dBm. */
  uuid_128 uuid;   /**< UUID of the unprovisioned device found */
  uint16_t oob;    /**< OOB information. */
  uint32_t uri;    /**< Hash of the out-of-band URI, which is received in <a
                        href="#evt_mesh_prov_uri">a separate event</a> . If the
                        URI bit (bit 1) is not set in the OOB capabilities
                        bitfield, this field is ignored. */
});

typedef struct sl_btmesh_evt_remote_provisioning_client_scan_report_s sl_btmesh_evt_remote_provisioning_client_scan_report_t;

/** @} */ // end addtogroup sl_btmesh_evt_remote_provisioning_client_scan_report

/**
 * @addtogroup sl_btmesh_evt_remote_provisioning_client_extended_scan_report sl_btmesh_evt_remote_provisioning_client_extended_scan_report
 * @{
 * @brief Remote Provisioning Server report of the advertising data requested by
 * the client.
 */

/** @brief Identifier of the extended_scan_report event */
#define sl_btmesh_evt_remote_provisioning_client_extended_scan_report_id 0x033000a8

/***************************************************************************//**
 * @brief Data structure of the extended_scan_report event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_remote_provisioning_client_extended_scan_report_s
{
  uint16_t   result;        /**< SL_STATUS_OK if successful. Error code
                                 otherwise. */
  uint16_t   server;        /**< Server address. */
  uint8_t    status;        /**< Enum @ref
                                 sl_btmesh_remote_provisioning_server_status_t.
                                 server status. */
  uuid_128   uuid;          /**< UUID of the unprovisioned device found */
  uint16_t   oob;           /**< OOB information. 0 if not present */
  uint8array ad_structures; /**< Concatenated list of AD Structures that match
                                 the AD Types requested by the client. */
});

typedef struct sl_btmesh_evt_remote_provisioning_client_extended_scan_report_s sl_btmesh_evt_remote_provisioning_client_extended_scan_report_t;

/** @} */ // end addtogroup sl_btmesh_evt_remote_provisioning_client_extended_scan_report

/**
 * @addtogroup sl_btmesh_evt_remote_provisioning_client_link_status sl_btmesh_evt_remote_provisioning_client_link_status
 * @{
 * @brief Remote Provisioning link Status.
 */

/** @brief Identifier of the link_status event */
#define sl_btmesh_evt_remote_provisioning_client_link_status_id          0x053000a8

/***************************************************************************//**
 * @brief Data structure of the link_status event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_remote_provisioning_client_link_status_s
{
  uint16_t result; /**< SL_STATUS_OK if successful. Error code otherwise. */
  uint16_t server; /**< Server address. */
  uint8_t  status; /**< Enum @ref sl_btmesh_remote_provisioning_server_status_t.
                        server status. */
  uint8_t  state;  /**< Enum @ref
                        sl_btmesh_remote_provisioning_client_link_state_t. link
                        state. */
});

typedef struct sl_btmesh_evt_remote_provisioning_client_link_status_s sl_btmesh_evt_remote_provisioning_client_link_status_t;

/** @} */ // end addtogroup sl_btmesh_evt_remote_provisioning_client_link_status

/**
 * @addtogroup sl_btmesh_evt_remote_provisioning_client_link_report sl_btmesh_evt_remote_provisioning_client_link_report
 * @{
 * @brief Remote Provisioning link report
 *
 * Used by the Remote Provisioning Server to report the state change of a
 * provisioning bearer link or the Node Provisioning Protocol Interface.
 */

/** @brief Identifier of the link_report event */
#define sl_btmesh_evt_remote_provisioning_client_link_report_id          0x063000a8

/***************************************************************************//**
 * @brief Data structure of the link_report event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_remote_provisioning_client_link_report_s
{
  uint16_t server; /**< Server address. */
  uint8_t  status; /**< Enum @ref sl_btmesh_remote_provisioning_server_status_t.
                        server status. */
  uint8_t  state;  /**< Enum @ref
                        sl_btmesh_remote_provisioning_client_link_state_t. link
                        state. */
  uint8_t  reason; /**< 0: success, 1: procedure timeout 2:procedure failed,
                        0xff not available. */
});

typedef struct sl_btmesh_evt_remote_provisioning_client_link_report_s sl_btmesh_evt_remote_provisioning_client_link_report_t;

/** @} */ // end addtogroup sl_btmesh_evt_remote_provisioning_client_link_report

/***************************************************************************//**
 *
 * Initialize the remote provisioning client.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_remote_provisioning_client_init();

/***************************************************************************//**
 *
 * Get the value of the Remote Provisioning Scan Capabilities state.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server Server address.
 * @param[in] elem_index Client element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_remote_provisioning_client_scan_capabilities
 *
 ******************************************************************************/
sl_status_t sl_btmesh_remote_provisioning_client_get_scan_capabilities(uint16_t enc_netkey_index,
                                                                       uint16_t server,
                                                                       uint16_t elem_index);

/***************************************************************************//**
 *
 * Start scanning for remote unprovisioned devices, within immediate radio range
 * of the Remote Provisioning Server.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server Server address
 * @param[in] elem_index Client element index
 * @param[in] item_limit Maximum number of scanned items to be reported.
 * @param[in] timeout_sec Time limit for a scan (in seconds). Valid value range:
 *   [1 - 255].
 * @param[in] single_scan Whether to perform single UUID scan or multiple target
 *   scan.
 * @param[in] uuid UUID of the device to scan. Valid only for single UUID scan
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_remote_provisioning_client_scan_status
 *   - @ref sl_btmesh_evt_remote_provisioning_client_scan_report
 *
 ******************************************************************************/
sl_status_t sl_btmesh_remote_provisioning_client_start_scan(uint16_t enc_netkey_index,
                                                            uint16_t server,
                                                            uint16_t elem_index,
                                                            uint8_t item_limit,
                                                            uint8_t timeout_sec,
                                                            uint8_t single_scan,
                                                            uuid_128 uuid);

/***************************************************************************//**
 *
 * Request scan status.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server Server address.
 * @param[in] elem_index Client element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_remote_provisioning_client_scan_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_remote_provisioning_client_get_scan_status(uint16_t enc_netkey_index,
                                                                 uint16_t server,
                                                                 uint16_t elem_index);

/***************************************************************************//**
 *
 * Stop Remote Provisioning Scan procedure.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server Server address.
 * @param[in] elem_index Client element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_remote_provisioning_client_scan_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_remote_provisioning_client_stop_scan(uint16_t enc_netkey_index,
                                                           uint16_t server,
                                                           uint16_t elem_index);

/***************************************************************************//**
 *
 * Request additional information about a specific unprovisioned device or about
 * the Remote Provisioning Server itself.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server Server address.
 * @param[in] elem_index Client element index
 * @param[in] timeout_sec Time limit for a scan (in seconds). Used only if
 *   selected mode is 1 and then valid value range: [1 - 5].
 * @param[in] mode If 1 request additional information about a specific
 *   unprovisioned device. If 0 request additional information about the Remote
 *   Provisioning Server itself.
 * @param[in] uuid If mode is 1 identifies the Device UUID of the unprovisioned
 *   device for which additional information is requested.
 * @param[in] ad_types_len Length of data in @p ad_types
 * @param[in] ad_types list AD_types to be reported. List maximun size being 16.
 *   See
 *   https://www.bluetooth.com/specifications/assigned-numbers/generic-access-profile/.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_remote_provisioning_client_extended_scan_report
 *
 ******************************************************************************/
sl_status_t sl_btmesh_remote_provisioning_client_start_extended_scan(uint16_t enc_netkey_index,
                                                                     uint16_t server,
                                                                     uint16_t elem_index,
                                                                     uint8_t timeout_sec,
                                                                     uint8_t mode,
                                                                     uuid_128 uuid,
                                                                     size_t ad_types_len,
                                                                     const uint8_t* ad_types);

/***************************************************************************//**
 *
 * Establish the provisioning bearer between a node supporting the Remote
 * Provisioning Server model and an unprovisioned device, or to open the Node
 * Provisioning Protocol Interface.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server Server address.
 * @param[in] elem_index Client element index
 * @param[in] timeout_sec Time limit for a scan (in seconds).
 * @param[in] dkri Enum @ref
 *   sl_btmesh_remote_provisioning_client_nppi_operation_t. Selection of Network
 *   Interface Operation or remote provisioning.
 * @param[in] uuid UUID of the device to provision. Valid if NPPI(dkri)
 *   procedure is not @ref
 *   sl_btmesh_remote_provisioning_client_remote_provisioning
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_remote_provisioning_client_link_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_remote_provisioning_client_open_link(uint16_t enc_netkey_index,
                                                           uint16_t server,
                                                           uint16_t elem_index,
                                                           uint8_t timeout_sec,
                                                           uint8_t dkri,
                                                           uuid_128 uuid);

/***************************************************************************//**
 *
 * Get the Remote Provisioning Link state of a Remote Provisioning Server model.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server Server address.
 * @param[in] elem_index Client element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_remote_provisioning_client_link_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_remote_provisioning_client_get_link_status(uint16_t enc_netkey_index,
                                                                 uint16_t server,
                                                                 uint16_t elem_index);

/***************************************************************************//**
 *
 * Close remote provisioning link.
 *
 * @param[in] enc_netkey_index Network key used to encrypt the request on the
 *   network layer
 * @param[in] server Server address.
 * @param[in] elem_index Client element index
 * @param[in] reason 0: success, 2:procedure failed, other not allowed.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_remote_provisioning_client_link_status
 *
 ******************************************************************************/
sl_status_t sl_btmesh_remote_provisioning_client_close_link(uint16_t enc_netkey_index,
                                                            uint16_t server,
                                                            uint16_t elem_index,
                                                            uint8_t reason);

/** @} */ // end addtogroup sl_btmesh_remote_provisioning_client

/**
 * @addtogroup sl_btmesh_remote_provisioning_server Bluetooth Mesh Remote Provisioning Server Model
 * @{
 *
 * @brief Bluetooth Mesh Remote Provisioning Server Model
 *
 * Remote Provisioning Server model provides tunneling of the provisioning
 * commands for the provisioning process.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_remote_provisioning_server_init_id                 0x00310028
#define sl_btmesh_cmd_remote_provisioning_server_open_link_id            0x01310028
#define sl_btmesh_cmd_remote_provisioning_server_set_default_bearer_id   0x02310028
#define sl_btmesh_rsp_remote_provisioning_server_init_id                 0x00310028
#define sl_btmesh_rsp_remote_provisioning_server_open_link_id            0x01310028
#define sl_btmesh_rsp_remote_provisioning_server_set_default_bearer_id   0x02310028

/**
 * @brief Remote provisionin scan states
 */
typedef enum
{
  sl_btmesh_remote_provisioning_server_idle     = 0x0, /**< (0x0) Idle state */
  sl_btmesh_remote_provisioning_server_multiple = 0x1, /**< (0x1) Multiple
                                                            Devices Scan */
  sl_btmesh_remote_provisioning_server_single   = 0x2  /**< (0x2) Single Device
                                                            Scan */
} sl_btmesh_remote_provisioning_server_scan_state_t;

/**
 * @brief Remote provisionnin server status
 */
typedef enum
{
  sl_btmesh_remote_provisioning_server_success                                  = 0x0, /**<
                                                                                            (0x0)
                                                                                            Success */
  sl_btmesh_remote_provisioning_server_scan_cannot_Start                        = 0x1, /**<
                                                                                            (0x1)
                                                                                            Scanning
                                                                                            Cannot
                                                                                            Start */
  sl_btmesh_remote_provisioning_server_invalid_state                            = 0x2, /**<
                                                                                            (0x2)
                                                                                            Invalid
                                                                                            State */
  sl_btmesh_remote_provisioning_server_limited_resources                        = 0x3, /**<
                                                                                            (0x3)
                                                                                            Limited
                                                                                            Resources */
  sl_btmesh_remote_provisioning_server_link_cannot_open                         = 0x4, /**<
                                                                                            (0x4)
                                                                                            Link
                                                                                            Cannot
                                                                                            Open */
  sl_btmesh_remote_provisioning_server_link_open_failed                         = 0x5, /**<
                                                                                            (0x5)
                                                                                            Link
                                                                                            open
                                                                                            failed */
  sl_btmesh_remote_provisioning_server_link_closed_by_device                    = 0x6, /**<
                                                                                            (0x6)
                                                                                            Link
                                                                                            Closed
                                                                                            by
                                                                                            Device */
  sl_btmesh_remote_provisioning_server_link_closed_by_server                    = 0x7, /**<
                                                                                            (0x7)
                                                                                            Link
                                                                                            closed
                                                                                            by
                                                                                            server */
  sl_btmesh_remote_provisioning_server_link_closed_by_client                    = 0x8, /**<
                                                                                            (0x8)
                                                                                            Link
                                                                                            closed
                                                                                            by
                                                                                            client */
  sl_btmesh_remote_provisioning_server_link_closed_as_cannot_receive_pdu        = 0x9, /**<
                                                                                            (0x9)
                                                                                            Link
                                                                                            closed
                                                                                            as
                                                                                            cannot
                                                                                            receive
                                                                                            PDU */
  sl_btmesh_remote_provisioning_server_link_closed_as_cannot_send_pdu           = 0xa, /**<
                                                                                            (0xa)
                                                                                            Link
                                                                                            closed
                                                                                            as
                                                                                            cannot
                                                                                            send
                                                                                            PDU */
  sl_btmesh_remote_provisioning_server_link_closed_as_cannot_deliver_pdu_report = 0xb  /**<
                                                                                            (0xb)
                                                                                            Link
                                                                                            closed
                                                                                            as
                                                                                            cannot
                                                                                            deliver
                                                                                            PDU
                                                                                            report */
} sl_btmesh_remote_provisioning_server_status_t;

/**
 * @addtogroup sl_btmesh_evt_remote_provisioning_server_link_open_request sl_btmesh_evt_remote_provisioning_server_link_open_request
 * @{
 * @brief This event indicates that server has received link-open request and
 * stack needs an LE-Connection
 *
 * This should be provided by the @ref
 * sl_btmesh_remote_provisioning_server_open_link command.
 */

/** @brief Identifier of the link_open_request event */
#define sl_btmesh_evt_remote_provisioning_server_link_open_request_id    0x003100a8

/***************************************************************************//**
 * @brief Data structure of the link_open_request event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_remote_provisioning_server_link_open_request_s
{
  bd_addr address;      /**< Address of the device to connect to */
  uint8_t address_type; /**< Provisinee device address type. Values:
                               - <b>0:</b> Public address
                               - <b>1:</b> Random address */
});

typedef struct sl_btmesh_evt_remote_provisioning_server_link_open_request_s sl_btmesh_evt_remote_provisioning_server_link_open_request_t;

/** @} */ // end addtogroup sl_btmesh_evt_remote_provisioning_server_link_open_request

/**
 * @addtogroup sl_btmesh_evt_remote_provisioning_server_link_close_request sl_btmesh_evt_remote_provisioning_server_link_close_request
 * @{
 * @brief Request to close LE-Connection
 */

/** @brief Identifier of the link_close_request event */
#define sl_btmesh_evt_remote_provisioning_server_link_close_request_id   0x013100a8

/***************************************************************************//**
 * @brief Data structure of the link_close_request event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_remote_provisioning_server_link_close_request_s
{
  uint8_t connection; /**< Handle that was assigned to the remote provisioning
                           and can now be closed */
});

typedef struct sl_btmesh_evt_remote_provisioning_server_link_close_request_s sl_btmesh_evt_remote_provisioning_server_link_close_request_t;

/** @} */ // end addtogroup sl_btmesh_evt_remote_provisioning_server_link_close_request

/***************************************************************************//**
 *
 * Initialize the remote provisioning server.
 *
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_remote_provisioning_server_init();

/***************************************************************************//**
 *
 * This command is used to pass connction handle to the remote provisioning
 * server after having received @ref
 * sl_btmesh_evt_remote_provisioning_server_link_open_request event.
 *
 * @param[in] connection Handle that was assigned to the connection
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 * 
 *
 ******************************************************************************/
sl_status_t sl_btmesh_remote_provisioning_server_open_link(uint8_t connection);

/***************************************************************************//**
 *
 * Set default bearer to be used in remote provisioning.
 *
 * @param[in] bearer Default bearer for provisioning.
 *     - <b>1</b> PB-ADV
 *     - <b>2</b> PB-GATT
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 * 
 *
 ******************************************************************************/
sl_status_t sl_btmesh_remote_provisioning_server_set_default_bearer(uint8_t bearer);

/** @} */ // end addtogroup sl_btmesh_remote_provisioning_server

/**
 * @addtogroup sl_btmesh_fw_standalone_updater Bluetooth Mesh Standalone Updater
 * @{
 *
 * @brief Bluetooth Mesh Standalone Updater
 *
 * The Standalone Updater combines the functionality of a local Distribution
 * Server and an external Distribution Client. It uses an Update Client and MBT
 * Client model internally to distribute firmware images to Target Nodes.
 *
 * The Standalone Updater is not a model per se. It achieves the same
 * functionality as a Distribution Client controlling a Distribution Server via
 * the local loopback interface, but without the ovearhead of any message
 * handling.
 */

/* Command and Response IDs */
#define sl_btmesh_cmd_fw_standalone_updater_init_id                      0x005a0028
#define sl_btmesh_cmd_fw_standalone_updater_deinit_id                    0x015a0028
#define sl_btmesh_cmd_fw_standalone_updater_setup_id                     0x025a0028
#define sl_btmesh_cmd_fw_standalone_updater_set_metadata_id              0x035a0028
#define sl_btmesh_cmd_fw_standalone_updater_set_multicast_threshold_id   0x045a0028
#define sl_btmesh_cmd_fw_standalone_updater_add_receiver_id              0x055a0028
#define sl_btmesh_cmd_fw_standalone_updater_delete_all_receivers_id      0x065a0028
#define sl_btmesh_cmd_fw_standalone_updater_start_id                     0x075a0028
#define sl_btmesh_cmd_fw_standalone_updater_execute_distribution_step_id 0x085a0028
#define sl_btmesh_cmd_fw_standalone_updater_apply_id                     0x095a0028
#define sl_btmesh_cmd_fw_standalone_updater_get_state_id                 0x0a5a0028
#define sl_btmesh_cmd_fw_standalone_updater_get_node_status_by_index_id  0x0b5a0028
#define sl_btmesh_cmd_fw_standalone_updater_get_node_status_by_address_id 0x0c5a0028
#define sl_btmesh_cmd_fw_standalone_updater_cancel_id                    0x0d5a0028
#define sl_btmesh_cmd_fw_standalone_updater_configure_throttle_id        0x0e5a0028
#define sl_btmesh_rsp_fw_standalone_updater_init_id                      0x005a0028
#define sl_btmesh_rsp_fw_standalone_updater_deinit_id                    0x015a0028
#define sl_btmesh_rsp_fw_standalone_updater_setup_id                     0x025a0028
#define sl_btmesh_rsp_fw_standalone_updater_set_metadata_id              0x035a0028
#define sl_btmesh_rsp_fw_standalone_updater_set_multicast_threshold_id   0x045a0028
#define sl_btmesh_rsp_fw_standalone_updater_add_receiver_id              0x055a0028
#define sl_btmesh_rsp_fw_standalone_updater_delete_all_receivers_id      0x065a0028
#define sl_btmesh_rsp_fw_standalone_updater_start_id                     0x075a0028
#define sl_btmesh_rsp_fw_standalone_updater_execute_distribution_step_id 0x085a0028
#define sl_btmesh_rsp_fw_standalone_updater_apply_id                     0x095a0028
#define sl_btmesh_rsp_fw_standalone_updater_get_state_id                 0x0a5a0028
#define sl_btmesh_rsp_fw_standalone_updater_get_node_status_by_index_id  0x0b5a0028
#define sl_btmesh_rsp_fw_standalone_updater_get_node_status_by_address_id 0x0c5a0028
#define sl_btmesh_rsp_fw_standalone_updater_cancel_id                    0x0d5a0028
#define sl_btmesh_rsp_fw_standalone_updater_configure_throttle_id        0x0e5a0028

/**
 * @addtogroup sl_btmesh_evt_fw_standalone_updater_dist_state_changed sl_btmesh_evt_fw_standalone_updater_dist_state_changed
 * @{
 * @brief Distribution state changed
 */

/** @brief Identifier of the dist_state_changed event */
#define sl_btmesh_evt_fw_standalone_updater_dist_state_changed_id        0x005a00a8

/***************************************************************************//**
 * @brief Data structure of the dist_state_changed event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_standalone_updater_dist_state_changed_s
{
  uint16_t elem_index;       /**< Element index */
  uint8_t  state;            /**< Enum @ref
                                  sl_btmesh_fw_dist_server_dist_step_t. New
                                  state of the Distribution */
  uint16_t num_active_nodes; /**< Number of active nodes */
});

typedef struct sl_btmesh_evt_fw_standalone_updater_dist_state_changed_s sl_btmesh_evt_fw_standalone_updater_dist_state_changed_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_standalone_updater_dist_state_changed

/**
 * @addtogroup sl_btmesh_evt_fw_standalone_updater_node_failed sl_btmesh_evt_fw_standalone_updater_node_failed
 * @{
 * @brief Indicates that an Target Node failed, either due to an error or due to
 * timeout
 */

/** @brief Identifier of the node_failed event */
#define sl_btmesh_evt_fw_standalone_updater_node_failed_id               0x015a00a8

/***************************************************************************//**
 * @brief Data structure of the node_failed event
 ******************************************************************************/
PACKSTRUCT( struct sl_btmesh_evt_fw_standalone_updater_node_failed_s
{
  uint16_t elem_index;    /**< Element index */
  uint16_t address;       /**< Address of the Update Server */
  uint8_t  update_phase;  /**< Enum @ref
                               sl_btmesh_fw_dist_server_dist_node_phase_t.
                               Update Phase of the Update Server. 0xa if not
                               known yet. */
  uint8_t  update_status; /**< Enum @ref
                               sl_btmesh_fw_update_server_update_status_t. Last
                               Update Status message received from the Update
                               Server */
  uint8_t  mbt_status;    /**< Last status code received from the MBT Server.
                               0xa if not known yet. */
  uint8_t  progress;      /**< 0-100, percentage of BLOB Transfer octets
                               delivered and confirmed */
  uint8_t  fw_index;      /**< Index of the firmware being updated */
});

typedef struct sl_btmesh_evt_fw_standalone_updater_node_failed_s sl_btmesh_evt_fw_standalone_updater_node_failed_t;

/** @} */ // end addtogroup sl_btmesh_evt_fw_standalone_updater_node_failed

/***************************************************************************//**
 *
 * Initializes the Standalone Updater. Before initializing the Standalone
 * Updater, the Update Client and MBT Client models on the same element must be
 * initialized.
 *
 * The Distribution Server is not supported on the same element at the same
 * time.
 *
 * @param[in] elem_index Standalone Updater element index (that is, the element
 *   index of the Update Client and MBT Client)
 * @param[in] max_node_list_size Maximum number of nodes that can be updated
 *   simultaneously
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_standalone_updater_init(uint16_t elem_index,
                                                 uint16_t max_node_list_size);

/***************************************************************************//**
 *
 * Deinitializes the Standalone Updater. The corresponding Update Client and MBT
 * Client models can be deinitialized before or after this model.
 *
 * @param[in] elem_index Element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_standalone_updater_deinit(uint16_t elem_index);

/***************************************************************************//**
 *
 * Set up parameters for a firmware distribution
 *
 * @param[in] elem_index Element index
 * @param[in] fw_size Firmware image size
 * @param[in] blob_id BLOB ID to use for the image
 * @param[in] fwid_len Length of data in @p fwid
 * @param[in] fwid Firmware ID of the firmware image
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_standalone_updater_setup(uint16_t elem_index,
                                                  uint32_t fw_size,
                                                  sl_bt_uuid_64_t blob_id,
                                                  size_t fwid_len,
                                                  const uint8_t* fwid);

/***************************************************************************//**
 *
 * Set the metadata of the image to be distributed. Should be called after the
 * @ref sl_btmesh_fw_standalone_updater_setup command. Cannot be called after
 * the distribution is started.
 *
 * @param[in] elem_index Element index
 * @param[in] metadata_len Length of data in @p metadata
 * @param[in] metadata Metadata of the image. May be zero-length.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_standalone_updater_set_metadata(uint16_t elem_index,
                                                         size_t metadata_len,
                                                         const uint8_t* metadata);

/***************************************************************************//**
 *
 * Set the multicast threshold value.
 *
 * @param[in] elem_index Element index
 * @param[in] multicast_threshold This value will be passed to the MBT Client.
 *   If the number of servers for any step exceeds or is equal to this number,
 *   the group address will be used. Otherwise, servers will be looped through
 *   one by one. Value of 0 disables the feature.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_standalone_updater_set_multicast_threshold(uint16_t elem_index,
                                                                    uint16_t multicast_threshold);

/***************************************************************************//**
 *
 * Add a receiver to the distribution
 *
 * @param[in] elem_index Element index
 * @param[in] address Address of the Target Node
 * @param[in] fw_index Index of the firmware on the Target Node to update
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_standalone_updater_add_receiver(uint16_t elem_index,
                                                         uint16_t address,
                                                         uint8_t fw_index);

/***************************************************************************//**
 *
 * Clears the list of receivers
 *
 * @param[in] elem_index Element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_standalone_updater_delete_all_receivers(uint16_t elem_index);

/***************************************************************************//**
 *
 * Start the firmware image distribution. The command @ref
 * sl_btmesh_fw_standalone_updater_execute_distribution_step is used to progress
 * through the procedures.
 *
 * The Receivers List must have been populated via the @ref
 * sl_btmesh_fw_standalone_updater_add_receiver command. The distribution
 * parameters must have been set via the @ref
 * sl_btmesh_fw_standalone_updater_setup and @ref
 * sl_btmesh_fw_standalone_updater_set_metadata commands.
 *
 * @param[in] elem_index Element index
 * @param[in] dist_appkey_index Application key index to use when communicating
 *   with the Target Nodes
 * @param[in] dist_ttl The time-to-live value to use when communicating with the
 *   Target Nodes. Valid values: 0, range:[2-127] and 255.
 * @param[in] dist_timeout_base The Timeout Base value to use in the firmware
 *   image BLOB Transfer and Update Client operations
 * @param[in] transfer_mode 1 = use Push Mode, 2 = use Pull Mode (typically for
 *   LPNs)
 * @param[in] group_address Group address to use. Zero to only use unicast.
 * @param[in] virtual_address If the group address is a VA hash, this needs to
 *   contain the full virtual address label UUID. Otherwise, the value of this
 *   field will be ignored.
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 * @b Events
 *   - @ref sl_btmesh_evt_fw_standalone_updater_dist_state_changed
 *   - @ref sl_btmesh_evt_fw_standalone_updater_node_failed
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_standalone_updater_start(uint16_t elem_index,
                                                  uint16_t dist_appkey_index,
                                                  uint8_t dist_ttl,
                                                  uint16_t dist_timeout_base,
                                                  uint8_t transfer_mode,
                                                  uint16_t group_address,
                                                  uuid_128 virtual_address);

/***************************************************************************//**
 *
 * Execute the next action for the distribution state machine. Must be repeated
 * by the application if the Target Nodes do not respond in a timely manner.
 * Calling this function sends out the necessary messages to all nodes that have
 * not responded. Note that the distribution step dist_step_transferring_image
 * is executed via the MBT Client API, and thus the command does not need to be
 * called in that state.
 *
 * @param[in] elem_index Element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_standalone_updater_execute_distribution_step(uint16_t elem_index);

/***************************************************************************//**
 *
 * Enter the apply step after the distribution is in the "verification complete"
 * step. The Standalone Updater can proceed to the apply step immediately after
 * the verification is complete. This state transition exists only for keeping
 * the API as similar as possible to the Distribution Server. The
 * execute_distribution_step command must be called to send out the Update Apply
 * messages.
 *
 * @param[in] elem_index Element index
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_standalone_updater_apply(uint16_t elem_index);

/***************************************************************************//**
 *
 * Get the current state of the distribution process. If an error is returned,
 * the other return values should be ignored.
 *
 * @param[in] elem_index Element index
 * @param[out] state Current state of the distribution
 * @param[out] total_nodes Number of nodes originally in the Receivers List
 * @param[out] num_active_nodes Number of active nodes
 *
 * @return SL_STATUS_OK if successful. Error code otherwise.
 *
 ******************************************************************************/
sl_status_t sl_btmesh_fw_standalone_updater_get_state(uint16_t elem_index,
                                                      uint8_t *st