/*******************************************************************************
 * @file  lwipopts.h
 * @brief
 *******************************************************************************
 * # License
 * <b>Copyright 2023 Silicon Laboratories Inc. www.silabs.com</b>
 *******************************************************************************
 *
 * The licensor of this software is Silicon Laboratories Inc. Your use of this
 * software is governed by the terms of Silicon Labs Master Software License
 * Agreement (MSLA) available at
 * www.silabs.com/about-us/legal/master-software-license-agreement. This
 * software is distributed to you in Source Code format and is governed by the
 * sections of the MSLA applicable to Source Code.
 *
 ******************************************************************************/
#ifndef __LWIPOPTS_H__
#define __LWIPOPTS_H__

#define LWIP_NETIF_EXT_STATUS_CALLBACK 0
#define LWIP_EVENT_API 0
#define LWIP_CALLBACK_API 1

/**
 * NO_SYS==1: Provides VERY minimal functionality. Otherwise,
 * use lwIP facilities.
 */
#define NO_SYS 0

/* ---------- Memory options ---------- */
/* MEM_ALIGNMENT: should be set to the alignment of the CPU for which
   lwIP is compiled. 4 byte alignment -> define MEM_ALIGNMENT to 4, 2
   byte alignment -> define MEM_ALIGNMENT to 2. */
#define MEM_ALIGNMENT 4

/* MEM_SIZE: the size of the heap memory. If the application will send
a lot of data that needs to be copied, this should be set high. */
#define MEM_SIZE (3 * 1024)

/* MEMP_NUM_PBUF: the number of memp struct pbufs. If the application sends a lot of data out of ROM (or other static
 * memory), this should be set high. */
#define MEMP_NUM_PBUF 10

/* MEMP_NUM_UDP_PCB: the number of UDP protocol control blocks. One per active UDP "connection". */
#define MEMP_NUM_UDP_PCB 6

/* MEMP_NUM_TCP_PCB: the number of simulatenously active TCP connections. */
#define MEMP_NUM_TCP_PCB 10

/* MEMP_NUM_TCP_PCB_LISTEN: the number of listening TCP connections. */
#define MEMP_NUM_TCP_PCB_LISTEN 5

/* MEMP_NUM_TCP_SEG: the number of simultaneously queued TCP segments. */
#define MEMP_NUM_TCP_SEG 8

/* MEMP_NUM_SYS_TIMEOUT: the number of simulateously active timeouts. */
#define MEMP_NUM_SYS_TIMEOUT 10

/* ---------- Pbuf options ---------- */
/* PBUF_POOL_SIZE: the number of buffers in the pbuf pool. */
#define PBUF_POOL_SIZE 8

/* PBUF_POOL_BUFSIZE: the size of each pbuf in the pbuf pool. */
#define PBUF_POOL_BUFSIZE 1524 + 16

/* ---------- NETIF API Supoort ---------- */
#define LWIP_NETIF_API 1

/* ---------- IPV4 Supoort ---------- */
#define LWIP_IPV4 1

/* ---------- ARP Supoort ---------- */
#define LWIP_ARP 1

/* ---------- IGMP Supoort ---------- */
#define LWIP_IGMP 1

#define LWIP_ICMP 1

#define LWIP_DHCP 1

/* ---------- IPV6 Supoort ---------- */
#define LWIP_IPV6 1

#define LWIP_ETHERNET (LWIP_IPV6)

#define LWIP_IPV6_MLD (LWIP_IPV6)

#define LWIP_IPV6_REASS (LWIP_IPV6)

#define LWIP_IPV6_FRAG (LWIP_IPV6)

#define LWIP_IPV6_AUTOCONFIG (LWIP_IPV6)

#define LWIP_IPV6_ROUTER_SUPPORT (LWIP_IPV6)

#define LWIP_ICMP6 (LWIP_IPV6)

#define LWIP_IPV6_DHCP6 0

/* ---------- TCP options ---------- */
#define LWIP_TCP 1
#define TCP_TTL 255

/* Controls if TCP should queue segments that arrive out of order. Define to 0 if your device is low on memory. */
#define TCP_QUEUE_OOSEQ 0

/* TCP Maximum segment size. */
#define TCP_MSS (1500 - 40) /* TCP_MSS = (Ethernet MTU - IP header size - TCP header size) */

/* TCP sender buffer space (bytes). */
#define TCP_SND_BUF (4 * TCP_MSS)

/*  TCP_SND_QUEUELEN: TCP sender buffer space (pbufs). This must be at least as much as (2 * TCP_SND_BUF/TCP_MSS) for
 * things to work. */
#define TCP_SND_QUEUELEN (2 * TCP_SND_BUF / TCP_MSS)

/* TCP receive window. */
#define TCP_WND (8 * TCP_MSS) //(2*TCP_MSS)

/* ---------- ICMP options ---------- */
#define LWIP_ICMP 1

/* ---------- DHCP options ---------- */
#define LWIP_DHCP 1
#define ETHARP_SUPPORT_STATIC_ENTRIES 1

/* ---------- UDP options ---------- */
#define LWIP_UDP 1
#define UDP_TTL 255

/* ---------- Statistics options ---------- */
#define LWIP_STATS 0

/* ---------- link callback options ---------- */
/* LWIP_NETIF_LINK_CALLBACK==1: Support a callback function from an interface whenever the link changes (i.e., link
 * down) */
#define LWIP_NETIF_LINK_CALLBACK 1

/*
   --------------------------------------
   ---------- Checksum options ----------
   --------------------------------------
*/

#define CHECKSUM_BY_HARDWARE 0

/* CHECKSUM_GEN_IP==1: Generate checksums in software for outgoing IP packets.*/
#define CHECKSUM_GEN_IP 1

/* CHECKSUM_GEN_UDP==1: Generate checksums in software for outgoing UDP packets.*/
#define CHECKSUM_GEN_UDP 1

/* CHECKSUM_GEN_TCP==1: Generate checksums in software for outgoing TCP packets.*/
#define CHECKSUM_GEN_TCP 1

/* CHECKSUM_CHECK_IP==1: Check checksums in software for incoming IP packets.*/
#define CHECKSUM_CHECK_IP 1

/* CHECKSUM_CHECK_UDP==1: Check checksums in software for incoming UDP packets.*/
#define CHECKSUM_CHECK_UDP 1

/* CHECKSUM_CHECK_TCP==1: Check checksums in software for incoming TCP packets.*/
#define CHECKSUM_CHECK_TCP 1

/* CHECKSUM_CHECK_ICMP==1: Check checksums by hardware for incoming ICMP packets.*/
#define CHECKSUM_GEN_ICMP 1

//#define LWIP_CHECKSUM_ON_COPY           1

/*
   ----------------------------------------------
   ---------- Sequential layer options ----------
   ----------------------------------------------
*/
/**
 * LWIP_NETCONN==1: Enable Netconn API (require to use api_lib.c)
 */
#define LWIP_NETCONN 0

/*
   ------------------------------------
   ---------- Socket options ----------
   ------------------------------------
*/
/**
 * LWIP_SOCKET==1: Enable Socket API (require to use sockets.c)
 */
#define LWIP_SOCKET 1

#ifndef LWIP_DNS
#define LWIP_DNS 0
#endif
/*
   ------------------------------------
   ---------- httpd options ----------
   ------------------------------------
*/
/** Set this to 1 to include "fsdata_custom.c" instead of "fsdata.c" for the
 * file system (to prevent changing the file included in CVS) */
#define HTTPD_USE_CUSTOM_FSDATA 0

/*----- Default Value for LWIP_HTTPD_CGI: 0 ---*/
#define LWIP_HTTPD_CGI 1

/*----- Default Value for LWIP_HTTPD_SSI: 0 ---*/
#define LWIP_HTTPD_SSI 1

/*----- Default Value for LWIP_HTTPD_MAX_TAG_NAME_LEN: 8 ---*/
#define LWIP_HTTPD_MAX_TAG_NAME_LEN 16

/* LwIP Stack Parameters (modified compared to initialization value in opt.h) -*/
#define LWIP_HTTPD_DYNAMIC_HEADERS 1
/*
   ---------------------------------
   ---------- OS options ----------
   ---------------------------------
*/
#define TCPIP_THREAD_NAME "TCP/IP"
#define TCPIP_THREAD_STACKSIZE 1000
#define TCPIP_MBOX_SIZE 6
#define DEFAULT_UDP_RECVMBOX_SIZE 6
#define DEFAULT_TCP_RECVMBOX_SIZE 6
#define DEFAULT_ACCEPTMBOX_SIZE 6
#define DEFAULT_THREAD_STACKSIZE 500

#if defined(SLI_SI91X_FREERTOS_SUPPORT)
//#include "cmsis_os.h"
#define TCPIP_THREAD_PRIO 2
#elif defined(SLI_SI91X_MICIRUM_SUPPORT)
#define TCPIP_THREAD_PRIO 1
#endif

#ifndef LWIP_TESTMODE
#define LWIP_TESTMODE 1
#endif
/*
   ------------------------------------------------
   ---------- Network Interfaces options ----------
   ------------------------------------------------
*/

#define LWIP_NETIF_TX_SINGLE_PBUF 1
#define LWIP_MDNS_RESPONDER LWIP_UDP
#define LWIP_NUM_NETIF_CLIENT_DATA (LWIP_MDNS_RESPONDER)

// #define LWIP_DEBUG LWIP_DBG_ON
// #define MDNS_DEBUG LWIP_DBG_ON

#endif /* __LWIPOPTS_H__ */
