/***************************************************************************//**
 * @file app_settings.c
 * @brief Application settings handler
 *******************************************************************************
 * # License
 * <b>Copyright 2020 Silicon Laboratories Inc. www.silabs.com</b>
 *******************************************************************************
 *
 * The licensor of this software is Silicon Laboratories Inc. Your use of this
 * software is governed by the terms of Silicon Labs Master Software License
 * Agreement (MSLA) available at
 * www.silabs.com/about-us/legal/master-software-license-agreement. This
 * software is distributed to you in Source Code format and is governed by the
 * sections of the MSLA applicable to Source Code.
 *
 ******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "sl_wisun_ip6string.h"
#include "nvm3.h"
#include "app_settings.h"
#include "sl_wisun_types.h"
#include "sl_wisun_cli_settings.h"
#include "sl_memory_manager.h"
#include "sl_wisun_keychain.h"
#include "sl_wisun_api.h"
#include "sl_wisun_trace_api.h"
#include "sl_rail_features.h"

#ifdef SL_CATALOG_WISUN_FAN_CERTIFICATION_PRESENT

  // Defaults for Wi-SUN FAN Certification.

  #define APP_SETTINGS_WISUN_DEFAULT_NETWORK_NAME  "WiSUN PAN"
  #define APP_SETTINGS_WISUN_DEFAULT_PHY_CONFIG_TYPE  SL_WISUN_PHY_CONFIG_FAN11
  #define APP_SETTINGS_WISUN_DEFAULT_REGULATORY_DOMAIN  1 // NA
  #define APP_SETTINGS_WISUN_DEFAULT_OPERATING_CLASS  1
  #define APP_SETTINGS_WISUN_DEFAULT_OPERATING_MODE  0x1b
  #define APP_SETTINGS_WISUN_DEFAULT_CHAN_PLAN_ID  1
  #define APP_SETTINGS_WISUN_DEFAULT_PHY_MODE_ID  2
  #define APP_SETTINGS_WISUN_DEFAULT_NETWORK_SIZE  SL_WISUN_NETWORK_SIZE_CERTIFICATION
  #define APP_SETTINGS_WISUN_DEFAULT_ALLOWED_CHANNELS  "10"

#elif defined SL_CATALOG_WISUN_CONFIG_PRESENT

  // Defaults from Wi-SUN Configurator.

  #include "sl_wisun_config.h"

  #ifdef WISUN_CONFIG_NETWORK_NAME
    #define APP_SETTINGS_WISUN_DEFAULT_NETWORK_NAME  WISUN_CONFIG_NETWORK_NAME
  #endif
  #ifdef WISUN_CONFIG_DEFAULT_PHY_FAN10
    #define APP_SETTINGS_WISUN_DEFAULT_PHY_CONFIG_TYPE  SL_WISUN_PHY_CONFIG_FAN10
    #define APP_SETTINGS_WISUN_DEFAULT_REGULATORY_DOMAIN  WISUN_CONFIG_REGULATORY_DOMAIN
    #define APP_SETTINGS_WISUN_DEFAULT_OPERATING_CLASS  WISUN_CONFIG_OPERATING_CLASS
    #define APP_SETTINGS_WISUN_DEFAULT_OPERATING_MODE  WISUN_CONFIG_OPERATING_MODE
  #elif defined WISUN_CONFIG_DEFAULT_PHY_FAN11
    #define APP_SETTINGS_WISUN_DEFAULT_PHY_CONFIG_TYPE  SL_WISUN_PHY_CONFIG_FAN11
    #define APP_SETTINGS_WISUN_DEFAULT_REGULATORY_DOMAIN  WISUN_CONFIG_REGULATORY_DOMAIN
    #define APP_SETTINGS_WISUN_DEFAULT_CHAN_PLAN_ID  WISUN_CONFIG_CHANNEL_PLAN_ID
    #define APP_SETTINGS_WISUN_DEFAULT_PHY_MODE_ID  WISUN_CONFIG_PHY_MODE_ID
  #endif
  #ifdef WISUN_CONFIG_NETWORK_SIZE
    #define APP_SETTINGS_WISUN_DEFAULT_NETWORK_SIZE  WISUN_CONFIG_NETWORK_SIZE
  #endif
  #ifdef WISUN_CONFIG_ALLOWED_CHANNELS
    #define APP_SETTINGS_WISUN_DEFAULT_ALLOWED_CHANNELS  WISUN_CONFIG_ALLOWED_CHANNELS
  #endif
  #ifdef WISUN_CONFIG_TX_POWER
    #define APP_SETTINGS_WISUN_DEFAULT_TX_POWER_DDBM  (WISUN_CONFIG_TX_POWER * 10)
  #endif
  #ifdef WISUN_CONFIG_DEVICE_TYPE
    #define APP_SETTINGS_WISUN_DEFAULT_DEVICE_TYPE  WISUN_CONFIG_DEVICE_TYPE
  #endif
  #ifdef WISUN_CONFIG_MODE_SWITCH_PHYS
    #define APP_SETTINGS_WISUN_DEFAULT_RX_PHY_MODE_IDS  WISUN_CONFIG_MODE_SWITCH_PHYS
  #endif
  #ifdef WISUN_CONFIG_MODE_SWITCH_PHYS_NUMBER
    #define APP_SETTINGS_WISUN_DEFAULT_RX_PHY_MODE_IDS_COUNT  WISUN_CONFIG_MODE_SWITCH_PHYS_NUMBER
  #endif
  #ifdef WISUN_CONFIG_DEVICE_PROFILE
    #define APP_SETTINGS_WISUN_DEFAULT_LFN_PROFILE  WISUN_CONFIG_DEVICE_PROFILE
  #endif

#endif

// Defaults from wisun_default_phy component.

#include "sl_wisun_default_phy.h"

#ifndef APP_SETTINGS_WISUN_DEFAULT_NETWORK_NAME
  #define APP_SETTINGS_WISUN_DEFAULT_NETWORK_NAME  "Wi-SUN Network"
#endif
#ifndef APP_SETTINGS_WISUN_DEFAULT_PHY_CONFIG_TYPE
  #define APP_SETTINGS_WISUN_DEFAULT_PHY_CONFIG_TYPE  SL_WISUN_DEFAULT_PHY_CONFIG_TYPE
#endif
#ifndef APP_SETTINGS_WISUN_DEFAULT_REGULATORY_DOMAIN
  #define APP_SETTINGS_WISUN_DEFAULT_REGULATORY_DOMAIN  SL_WISUN_DEFAULT_REGULATORY_DOMAIN
#endif
#ifndef APP_SETTINGS_WISUN_DEFAULT_OPERATING_CLASS
  #define APP_SETTINGS_WISUN_DEFAULT_OPERATING_CLASS  SL_WISUN_DEFAULT_OPERATING_CLASS
#endif
#ifndef APP_SETTINGS_WISUN_DEFAULT_OPERATING_MODE
  #define APP_SETTINGS_WISUN_DEFAULT_OPERATING_MODE  SL_WISUN_DEFAULT_OPERATING_MODE
#endif
#ifndef APP_SETTINGS_WISUN_DEFAULT_CHAN_PLAN_ID
  #define APP_SETTINGS_WISUN_DEFAULT_CHAN_PLAN_ID  SL_WISUN_DEFAULT_CHAN_PLAN_ID
#endif
#ifndef APP_SETTINGS_WISUN_DEFAULT_PHY_MODE_ID
  #define APP_SETTINGS_WISUN_DEFAULT_PHY_MODE_ID  SL_WISUN_DEFAULT_PHY_MODE_ID
#endif

#ifndef APP_SETTINGS_WISUN_DEFAULT_NETWORK_SIZE
  #define APP_SETTINGS_WISUN_DEFAULT_NETWORK_SIZE  SL_WISUN_NETWORK_SIZE_SMALL
#endif
#ifndef APP_SETTINGS_WISUN_DEFAULT_ALLOWED_CHANNELS
  #define APP_SETTINGS_WISUN_DEFAULT_ALLOWED_CHANNELS  "0-255"
#endif
#ifndef APP_SETTINGS_WISUN_DEFAULT_TX_POWER_DDBM
  #define APP_SETTINGS_WISUN_DEFAULT_TX_POWER_DDBM  200
#endif
#ifndef APP_SETTINGS_WISUN_DEFAULT_DEVICE_TYPE
  #define APP_SETTINGS_WISUN_DEFAULT_DEVICE_TYPE  DEFAULT_DEVICE_TYPE
#endif
#ifndef APP_SETTINGS_WISUN_DEFAULT_RX_PHY_MODE_IDS
  #define APP_SETTINGS_WISUN_DEFAULT_RX_PHY_MODE_IDS  { 0 }
#endif
#ifndef APP_SETTINGS_WISUN_DEFAULT_RX_PHY_MODE_IDS_COUNT
  #define APP_SETTINGS_WISUN_DEFAULT_RX_PHY_MODE_IDS_COUNT  0
#endif
#ifndef APP_SETTINGS_WISUN_DEFAULT_LFN_PROFILE
  #define APP_SETTINGS_WISUN_DEFAULT_LFN_PROFILE SL_WISUN_LFN_PROFILE_TEST
#endif

#define APP_SETTINGS_WISUN_DEFAULT_UC_DWELL_INTERVAL  255
#define APP_SETTINGS_WISUN_DEFAULT_CH0_FREQUENCY  863100
#define APP_SETTINGS_WISUN_DEFAULT_NUMBER_OF_CHANNELS  69
#define APP_SETTINGS_WISUN_DEFAULT_CHANNEL_SPACING  100
#define APP_SETTINGS_WISUN_DEFAULT_REGULATION  SL_WISUN_REGULATION_NONE
#define APP_SETTINGS_WISUN_DEFAULT_CRC_TYPE SL_WISUN_4_BYTES_CRC
#define APP_SETTINGS_WISUN_DEFAULT_STF_LENGTH 4
#define APP_SETTINGS_WISUN_DEFAULT_PREAMBLE_LENGTH 56
#define APP_SETTINGS_WISUN_MAX_NEIGHBOR_COUNT 32
#define APP_SETTINGS_WISUN_MAX_CHILD_COUNT 22
#define APP_SETTINGS_WISUN_MAX_SECURITY_NEIGHBOR_COUNT 300
#define APP_SETTINGS_WISUN_DEFAULT_PREFERRED_PAN_ID 0xFFFF
#define APP_SETTINGS_WISUN_DEFAULT_KEYCHAIN SL_WISUN_KEYCHAIN_AUTOMATIC
#define APP_SETTINGS_WISUN_DEFAULT_KEYCHAIN_INDEX 0
#define APP_SETTINGS_WISUN_DEFAULT_MAX_HOP_COUNT 24
#define APP_SETTINGS_WISUN_DEFAULT_LOWPAN_MTU 1576
#define APP_SETTINGS_WISUN_DEFAULT_IPV6_MRU 1504
#define APP_SETTINGS_WISUN_DEFAULT_MAX_EDFE_FRAGMENT_COUNT 5
#define APP_SETTINGS_WISUN_DEFAULT_SOCKET_RX_BUFFER_SIZE 2048
#define APP_SETTINGS_WISUN_DEFAULT_EAP_IDENTITY "Anonymous"

#define APP_SETTINGS_MAC_DEFAULT_MIN_BE 3
#define APP_SETTINGS_MAC_DEFAULT_MAX_BE 5
#define APP_SETTINGS_MAC_DEFAULT_BACKOFF_PERIOD_US 0
#define APP_SETTINGS_MAC_DEFAULT_MAX_CCA_RETRIES 8
#define APP_SETTINGS_MAC_DEFAULT_MAX_FRAME_RETRIES 19

#ifndef APP_SETTINGS_APP_DEFAULT_AUTOCONNECT
  #define APP_SETTINGS_APP_DEFAULT_AUTOCONNECT  0
#endif

typedef enum
{
  app_settings_domain_wisun       = 0x00,
  app_settings_domain_statistics  = 0x01,
  app_settings_domain_app         = 0x02,
  app_settings_domain_info        = 0x03,
  app_settings_domain_ping        = 0x04,
  app_settings_domain_mac         = 0x05
} app_settings_domain_t;

const char *app_settings_domain_str[] =
{
  "wisun",
  "statistics",
  "app",
  "info",
  "ping",
  "mac",
  NULL,
};

typedef enum
{
  app_statistics_domain_phy        = 0x00,
  app_statistics_domain_mac        = 0x01,
  app_statistics_domain_fhss       = 0x02,
  app_statistics_domain_wisun      = 0x03,
  app_statistics_domain_network    = 0x04,
  app_statistics_domain_regulation = 0x05,
  app_statistics_domain_heap       = 0x06
} app_statistics_domain_t;

const char *app_statistics_domain_str[] =
{
  "phy",
  "mac",
  "fhss",
  "wisun",
  "network",
  "regulation",
  "heap",
  NULL,
};

typedef enum
{
  app_info_domain_network = 0x00,
  app_info_domain_rpl     = 0x01
} app_info_domain_t;

const char *app_info_domain_str[] =
{
  "network",
  "rpl",
  NULL,
};

static const app_settings_wisun_t app_settings_wisun_default = {
  .allowed_channels = APP_SETTINGS_WISUN_DEFAULT_ALLOWED_CHANNELS,
  .network_name = APP_SETTINGS_WISUN_DEFAULT_NETWORK_NAME,
  .regulatory_domain = APP_SETTINGS_WISUN_DEFAULT_REGULATORY_DOMAIN,
  .operating_class = APP_SETTINGS_WISUN_DEFAULT_OPERATING_CLASS,
  .operating_mode = APP_SETTINGS_WISUN_DEFAULT_OPERATING_MODE,
  .network_size = APP_SETTINGS_WISUN_DEFAULT_NETWORK_SIZE,
  .tx_power_ddbm = APP_SETTINGS_WISUN_DEFAULT_TX_POWER_DDBM,
  .uc_dwell_interval_ms = APP_SETTINGS_WISUN_DEFAULT_UC_DWELL_INTERVAL,
  .number_of_channels = APP_SETTINGS_WISUN_DEFAULT_NUMBER_OF_CHANNELS,
  .ch0_frequency = APP_SETTINGS_WISUN_DEFAULT_CH0_FREQUENCY,
  .channel_spacing = APP_SETTINGS_WISUN_DEFAULT_CHANNEL_SPACING,
  .trace_filter = {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
  .regulation = APP_SETTINGS_WISUN_DEFAULT_REGULATION,
  .regulation_warning_threshold = 50,
  .regulation_alert_threshold = 90,
  .device_type = APP_SETTINGS_WISUN_DEFAULT_DEVICE_TYPE,
  .chan_plan_id = APP_SETTINGS_WISUN_DEFAULT_CHAN_PLAN_ID,
  .phy_mode_id = APP_SETTINGS_WISUN_DEFAULT_PHY_MODE_ID,
  .phy_config_type = APP_SETTINGS_WISUN_DEFAULT_PHY_CONFIG_TYPE,
  .rx_phy_mode_ids = APP_SETTINGS_WISUN_DEFAULT_RX_PHY_MODE_IDS,
  .rx_phy_mode_ids_count = APP_SETTINGS_WISUN_DEFAULT_RX_PHY_MODE_IDS_COUNT,
  .lfn_profile = APP_SETTINGS_WISUN_DEFAULT_LFN_PROFILE,
  .crc_type = APP_SETTINGS_WISUN_DEFAULT_CRC_TYPE,
  .stf_length = APP_SETTINGS_WISUN_DEFAULT_STF_LENGTH,
  .preamble_length = APP_SETTINGS_WISUN_DEFAULT_PREAMBLE_LENGTH,
  .max_neighbor_count = APP_SETTINGS_WISUN_MAX_NEIGHBOR_COUNT,
  .max_child_count = APP_SETTINGS_WISUN_MAX_CHILD_COUNT,
  .max_security_neighbor_count = APP_SETTINGS_WISUN_MAX_SECURITY_NEIGHBOR_COUNT,
  .preferred_pan_id = APP_SETTINGS_WISUN_DEFAULT_PREFERRED_PAN_ID,
  .keychain = APP_SETTINGS_WISUN_DEFAULT_KEYCHAIN,
  .keychain_index = APP_SETTINGS_WISUN_DEFAULT_KEYCHAIN_INDEX,
  .direct_connect_pmk = { 0x34, 0xba, 0x32, 0x26, 0xa0, 0xb2, 0xad, 0x66, 0x7c, 0x9f, 0x66, 0x02, 0xe5, 0xdb, 0x75, 0x77,
                          0xdd, 0xbd, 0x5d, 0x2b, 0x34, 0x3a, 0x93, 0x06, 0x2b, 0x90, 0xc0, 0x7b, 0xe2, 0x8e, 0x4e, 0x54 },
  .max_hop_count = APP_SETTINGS_WISUN_DEFAULT_MAX_HOP_COUNT,
  .lowpan_mtu = APP_SETTINGS_WISUN_DEFAULT_LOWPAN_MTU,
  .ipv6_mru = APP_SETTINGS_WISUN_DEFAULT_IPV6_MRU,
  .max_edfe_fragment_count = APP_SETTINGS_WISUN_DEFAULT_MAX_EDFE_FRAGMENT_COUNT,
  .socket_rx_buffer_size = APP_SETTINGS_WISUN_DEFAULT_SOCKET_RX_BUFFER_SIZE,
  .eap_identity = APP_SETTINGS_WISUN_DEFAULT_EAP_IDENTITY,
};

static const app_settings_ping_t app_settings_ping_default = {
  .identifier = 1,
  .sequence_number = 1,
  .packet_interval = 1000,
  .packet_length = 40,
  .pattern = "0123456789"
};

static const app_settings_app_t app_settings_app_default = {
  .printable_data_as_hex = false,
  .printable_data_length = APP_UTIL_PRINTABLE_DATA_MAX_LENGTH,
  .autoconnect = APP_SETTINGS_APP_DEFAULT_AUTOCONNECT,
  .pti_state = true
};

static const app_settings_mac_t app_settings_mac_default = {
  .min_be = APP_SETTINGS_MAC_DEFAULT_MIN_BE,
  .max_be = APP_SETTINGS_MAC_DEFAULT_MAX_BE,
  .backoff_period_us = APP_SETTINGS_MAC_DEFAULT_BACKOFF_PERIOD_US,
  .max_cca_retries = APP_SETTINGS_MAC_DEFAULT_MAX_CCA_RETRIES,
  .max_frame_retries = APP_SETTINGS_MAC_DEFAULT_MAX_FRAME_RETRIES
};

app_settings_wisun_t app_settings_wisun;
app_settings_ping_t app_settings_ping;
app_settings_app_t app_settings_app;
app_settings_mac_t app_settings_mac;

const app_saving_item_t app_saving_item_wisun = {
  .data = &app_settings_wisun,
  .data_size = sizeof(app_settings_wisun),
  .default_val = &app_settings_wisun_default
};

const app_saving_item_t app_saving_item_ping = {
  .data = &app_settings_ping,
  .data_size = sizeof(app_settings_ping),
  .default_val = &app_settings_ping_default
};

const app_saving_item_t app_saving_item_app = {
  .data = &app_settings_app,
  .data_size = sizeof(app_settings_app),
  .default_val = &app_settings_app_default
};

const app_saving_item_t app_saving_item_mac = {
  .data = &app_settings_mac,
  .data_size = sizeof(app_settings_mac),
  .default_val = &app_settings_mac_default
};

const app_saving_item_t *saving_settings[] = {
  &app_saving_item_wisun,
  &app_saving_item_ping,
  &app_saving_item_app,
  &app_saving_item_mac,
  NULL
};

static sl_wisun_statistics_t app_statistics;
static sl_wisun_network_info_t app_network_info;
static sl_wisun_rpl_info_t app_rpl_info;

#if SLI_WISUN_DISABLE_SECURITY
uint32_t app_security_state = 1;
#endif

#ifdef SL_CATALOG_POWER_MANAGER_PRESENT
uint32_t lfn_em_time_ms[EM_MAX + 1] = {0};
#endif

static const app_enum_t app_settings_wisun_phy_config_type_enum[] =
{
  { "FAN 1.0", SL_WISUN_PHY_CONFIG_FAN10 },
  { "FAN 1.1", SL_WISUN_PHY_CONFIG_FAN11 },
  { "explicit", SL_WISUN_PHY_CONFIG_EXPLICIT },
  { "IDs", SL_WISUN_PHY_CONFIG_IDS},
  { NULL, 0 }
};

static const app_enum_t app_settings_wisun_network_size_enum[] =
{
  { "automatic", SL_WISUN_NETWORK_SIZE_AUTOMATIC },
  { "small", SL_WISUN_NETWORK_SIZE_SMALL },
  { "medium", SL_WISUN_NETWORK_SIZE_MEDIUM },
  { "large", SL_WISUN_NETWORK_SIZE_LARGE },
  { "test", SL_WISUN_NETWORK_SIZE_TEST },
  { "certification", SL_WISUN_NETWORK_SIZE_CERTIFICATION },
  { NULL, 0 }
};

static const app_enum_t app_settings_wisun_reg_domain_enum[] =
{
  { "WW", SL_WISUN_REGULATORY_DOMAIN_WW },
  { "NA", SL_WISUN_REGULATORY_DOMAIN_NA },
  { "JP", SL_WISUN_REGULATORY_DOMAIN_JP },
  { "EU", SL_WISUN_REGULATORY_DOMAIN_EU },
  { "CN", SL_WISUN_REGULATORY_DOMAIN_CN },
  { "IN", SL_WISUN_REGULATORY_DOMAIN_IN },
  { "MX", SL_WISUN_REGULATORY_DOMAIN_MX },
  { "BZ", SL_WISUN_REGULATORY_DOMAIN_BZ },
  { "AZ/NZ", SL_WISUN_REGULATORY_DOMAIN_AZ },
  { "AZ", SL_WISUN_REGULATORY_DOMAIN_AZ },
  { "NZ", SL_WISUN_REGULATORY_DOMAIN_NZ },
  { "KR", SL_WISUN_REGULATORY_DOMAIN_KR },
  { "PH", SL_WISUN_REGULATORY_DOMAIN_PH },
  { "MY", SL_WISUN_REGULATORY_DOMAIN_MY },
  { "HK", SL_WISUN_REGULATORY_DOMAIN_HK },
  { "SG", SL_WISUN_REGULATORY_DOMAIN_SG },
  { "TH", SL_WISUN_REGULATORY_DOMAIN_TH },
  { "VN", SL_WISUN_REGULATORY_DOMAIN_VN },
  { NULL, 0 }
};

static const app_enum_t app_settings_wisun_operating_class_enum[] =
{
  { "application", SL_WISUN_OPERATING_CLASS_APP },
  { NULL, 0 }
};

const app_enum_t app_settings_wisun_join_state_enum_ffn[] =
{
  { "Disconnected", SL_WISUN_JOIN_STATE_DISCONNECTED },
  { "Join state 1: Select PAN", SL_WISUN_JOIN_STATE_SELECT_PAN },
  { "Join state 2: Authenticate", SL_WISUN_JOIN_STATE_AUTHENTICATE },
  { "Join state 3: Acquire PAN Config", SL_WISUN_JOIN_STATE_ACQUIRE_PAN_CONFIG },
  { "Join state 4: Configure Routing - parent selection", SL_WISUN_JOIN_STATE_CONFIGURE_ROUTING },
  { "Join state 4: Configure Routing - DHCP", SL_WISUN_JOIN_STATE_DHCP },
  { "Join state 4: Configure Routing - address registration", SL_WISUN_JOIN_STATE_EARO },
  { "Join state 4: Configure Routing - DAO registration", SL_WISUN_JOIN_STATE_DAO },
  { "Join state 5: Operational", SL_WISUN_JOIN_STATE_OPERATIONAL },
  { "Disconnecting", SL_WISUN_JOIN_STATE_DISCONNECTING },
  { NULL, 0 }
};

const app_enum_t app_settings_wisun_join_state_enum_lfn[] =
{
  { "Disconnected", SL_WISUN_JOIN_STATE_DISCONNECTED },
  { "Join state 1: Select PAN", SL_WISUN_JOIN_STATE_SELECT_PAN },
  { "Join state 2: Authenticate", SL_WISUN_JOIN_STATE_AUTHENTICATE },
  { "Join state 3: Acquire PAN Config", SL_WISUN_JOIN_STATE_ACQUIRE_PAN_CONFIG },
  { "Join state 4: Configure IP Layer - DHCP", SL_WISUN_JOIN_STATE_CONFIGURE_ROUTING },
  { "Join state 4: Configure IP Layer - address registration", SL_WISUN_JOIN_STATE_EARO },
  { "Join state 5: Operational", SL_WISUN_JOIN_STATE_OPERATIONAL },
  { "Disconnecting", SL_WISUN_JOIN_STATE_DISCONNECTING },
  { NULL, 0 }
};

const app_enum_t app_settings_wisun_crc_type_enum[] =
{
  { "no-crc", SL_WISUN_NO_CRC },
  { "2-bytes", SL_WISUN_2_BYTES_CRC },
  { "4-bytes", SL_WISUN_4_BYTES_CRC },
  { NULL, 0 }
};

const app_enum_t app_settings_wisun_neighbor_type_enum[] =
{
  { "Primary parent", SL_WISUN_NEIGHBOR_TYPE_PRIMARY_PARENT },
  { "Secondary parent", SL_WISUN_NEIGHBOR_TYPE_SECONDARY_PARENT },
  { "Child", SL_WISUN_NEIGHBOR_TYPE_CHILD },
  { NULL, 0 }
};

const app_enum_t app_settings_wisun_is_lfn_enum[] =
{
  { "FFN", 0 },
  { "LFN", 1 },
  { NULL, 0 }
};


static const app_enum_t app_settings_wisun_regulation_enum[] =
{
  { "none", SL_WISUN_REGULATION_NONE },
  { "arib", SL_WISUN_REGULATION_ARIB },
  { "wpc", SL_WISUN_REGULATION_WPC },
  { "etsi", SL_WISUN_REGULATION_ETSI },
  { NULL, 0 }
};

static const app_enum_t app_settings_wisun_device_type_enum[] =
{
#ifdef HAVE_FFN
  { "FFN", SL_WISUN_ROUTER },
#endif
#ifdef HAVE_LFN
  { "LFN", SL_WISUN_LFN },
#endif
  { NULL, 0 }
};

static const app_enum_t app_settings_wisun_lfn_profile_enum[] =
{
  { "test", SL_WISUN_LFN_PROFILE_TEST },
  { "balanced", SL_WISUN_LFN_PROFILE_BALANCED },
  { "eco", SL_WISUN_LFN_PROFILE_ECO },
  { NULL, 0 }
};

static const app_enum_t app_settings_wisun_keychain_enum[] =
{
  { "automatic", SL_WISUN_KEYCHAIN_AUTOMATIC },
  { "builtin", SL_WISUN_KEYCHAIN_BUILTIN },
  { "nvm", SL_WISUN_KEYCHAIN_NVM },
  { NULL, 0 }
};

static sl_status_t app_settings_get_fan10_phy_config(char *value_str,
                                                     const char *key_str,
                                                     const app_settings_entry_t *entry);
static sl_status_t app_settings_get_fan11_phy_config(char *value_str,
                                                     const char *key_str,
                                                     const app_settings_entry_t *entry);
static sl_status_t app_settings_get_explicit_phy_config(char *value_str,
                                                        const char *key_str,
                                                        const app_settings_entry_t *entry);
static sl_status_t app_settings_get_custom_phy_config(char *value_str,
                                                      const char *key_str,
                                                      const app_settings_entry_t *entry);
static sl_status_t app_settings_get_custom_fsk_oqpsk_phy_config(char *value_str,
                                                                const char *key_str,
                                                                const app_settings_entry_t *entry);
static sl_status_t app_settings_get_custom_ofdm_phy_config(char *value_str,
                                                           const char *key_str,
                                                           const app_settings_entry_t *entry);
static sl_status_t app_settings_get_ids_phy_config(char *value_str,
                                                   const char *key_str,
                                                   const app_settings_entry_t *entry);
static sl_status_t app_settings_get_fan10_and_fan11_phy_config(char *value_str,
                                                               const char *key_str,
                                                               const app_settings_entry_t *entry);
static sl_status_t app_settings_get_fan11_and_explicit_and_ids_phy_config(char *value_str,
                                                                          const char *key_str,
                                                                          const app_settings_entry_t *entry);
static sl_status_t app_settings_get_ip_addresses(char *value_str,
                                                 const char *key_str,
                                                 const app_settings_entry_t *entry);
static sl_status_t app_settings_get_border_router(char *value_str,
                                                  const char *key_str,
                                                  const app_settings_entry_t *entry);
static sl_status_t app_settings_get_parents(char *value_str,
                                            const char *key_str,
                                            const app_settings_entry_t *entry);
static sl_status_t app_settings_get_neighbors(char *value_str,
                                              const char *key_str,
                                              const app_settings_entry_t *entry);
static sl_status_t app_settings_get_statistics(char *value_str,
                                               const char *key_str,
                                               const app_settings_entry_t *entry);
static sl_status_t app_settings_get_statistics_regulation(char *value_str,
                                                          const char *key_str,
                                                          const app_settings_entry_t *entry);
static sl_status_t app_settings_get_async_channel_mask_str(char *value_str,
                                                           const char *key_str,
                                                           const app_settings_entry_t *entry);
static sl_status_t app_settings_get_unicast_channel_mask_str(char *value_str,
                                                             const char *key_str,
                                                             const app_settings_entry_t *entry);
static sl_status_t app_settings_get_broadcast_channel_mask_str(char *value_str,
                                                               const char *key_str,
                                                               const app_settings_entry_t *entry);
static sl_status_t app_settings_set_allowed_channels(const char *value_str,
                                                     const char *key_str,
                                                     const app_settings_entry_t *entry);
static sl_status_t app_settings_set_trace_filter(const char *value_str,
                                                 const char *key_str,
                                                 const app_settings_entry_t *entry);
static sl_status_t app_settings_get_trace_filter(char *value_str,
                                                 const char *key_str,
                                                 const app_settings_entry_t *entry);
static sl_status_t app_settings_get_join_state(char *value_str,
                                               const char *key_str,
                                               const app_settings_entry_t *entry);
static sl_status_t app_settings_set_mac_address(const char *value_str,
                                                const char *key_str,
                                                const app_settings_entry_t *entry);
static sl_status_t app_settings_get_mac_address(char *value_str,
                                                const char *key_str,
                                                const app_settings_entry_t *entry);
static sl_status_t app_settings_set_regulation_warning_threshold(const char *value_str,
                                                                 const char *key_str,
                                                                 const app_settings_entry_t *entry);
static sl_status_t app_settings_set_regulation_alert_threshold(const char *value_str,
                                                               const char *key_str,
                                                               const app_settings_entry_t *entry);

#if SL_RAIL_IEEE802154_SUPPORTS_G_MODE_SWITCH
static sl_status_t app_settings_set_rx_mdr_capable(const char *value_str,
                                                   const char *key_str,
                                                   const app_settings_entry_t *entry);
static sl_status_t app_settings_set_rx_phy_mode_ids(const char *value_str,
                                                    const char *key_str,
                                                    const app_settings_entry_t *entry);
static sl_status_t app_settings_get_rx_phy_mode_ids(char *value_str,
                                                    const char *key_str,
                                                    const app_settings_entry_t *entry);
#endif

static sl_status_t app_settings_get_network_info(char *value_str,
                                                 const char *key_str,
                                                 const app_settings_entry_t *entry);
static sl_status_t app_settings_get_rpl_info(char *value_str,
                                             const char *key_str,
                                             const app_settings_entry_t *entry);

static sl_status_t app_settings_set_direct_connect_pmk(const char *value_str,
                                                       const char *key_str,
                                                       const app_settings_entry_t *entry);
static sl_status_t app_settings_get_direct_connect_pmk(char *value_str,
                                                       const char *key_str,
                                                       const app_settings_entry_t *entry);
static sl_status_t app_settings_set_tx_power(const char *value_str,
                                             const char *key_str,
                                             const app_settings_entry_t *entry);
static sl_status_t app_settings_get_tx_power(char *value_str,
                                             const char *key_str,
                                             const app_settings_entry_t *entry);
static sl_status_t app_settings_set_preferred_pan(const char *value_str,
                                                  const char *key_str,
                                                  const app_settings_entry_t *entry);

const app_settings_entry_t app_settings_entries[] =
{
  {
    .key = "join_state",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = NULL,
    .input_enum_list = NULL,
    .output_enum_list = app_settings_wisun_join_state_enum_ffn,
    .set_handler = NULL,
    .get_handler = app_settings_get_join_state,
    .description = "Wi-SUN join state"
  },
  {
    .key = "network_name",
    .domain = app_settings_domain_wisun,
    .value_size = SL_WISUN_NETWORK_NAME_SIZE+1,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.network_name,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_string,
    .get_handler = app_settings_get_string,
    .description = "Wi-SUN network [string] max 32"
  },
  {
    .key = "phy_config_type",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.phy_config_type,
    .input_enum_list = NULL,
    .output_enum_list = app_settings_wisun_phy_config_type_enum,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "PHY configuration type (FAN 1.0|FAN 1.1|explicit|IDs)"
  },
  {
    .key = "regulatory_domain",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.regulatory_domain,
    .input_enum_list = app_settings_wisun_reg_domain_enum,
    .output_enum_list = app_settings_wisun_reg_domain_enum,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_fan10_and_fan11_phy_config,
    .description = "Wi-SUN regulatory domain [uint8]"
  },
  {
    .key = "operating_class",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.operating_class,
    .input_enum_list = app_settings_wisun_operating_class_enum,
    .output_enum_list = app_settings_wisun_operating_class_enum,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_fan10_phy_config,
    .description = "Wi-SUN operation class [uint8]"
  },
  {
    .key = "operating_mode",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT | APP_SETTINGS_OUTPUT_FLAG_HEX,
    .value = &app_settings_wisun.operating_mode,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_fan10_phy_config,
    .description = "Wi-SUN operating mode [uint16]"
  },
  {
    .key = "fec",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.fec,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_fan10_phy_config,
    .description = "1 to use FEC, 0 otherwise [bool] (0|1)"
  },
  {
    .key = "chan_plan_id",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.chan_plan_id,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_fan11_phy_config,
    .description = "Wi-SUN channel plan ID [uint8]"
  },
  {
    .key = "phy_mode_id",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.phy_mode_id,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_fan11_and_explicit_and_ids_phy_config,
    .description = "Wi-SUN PHY mode ID [uint8]"
  },
  {
    .key = "ch0_frequency",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.ch0_frequency,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_explicit_phy_config,
    .description = "Ch0 frequency in kHz [uint32]"
  },
  {
    .key = "number_of_channels",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.number_of_channels,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_explicit_phy_config,
    .description = "Number of channels [uint16]"
  },
  {
    .key = "channel_spacing",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.channel_spacing,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_explicit_phy_config,
    .description = "Channel spacing in kHz [uint16]"
  },
  {
    .key = "crc_type",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.crc_type,
    .input_enum_list = app_settings_wisun_crc_type_enum,
    .output_enum_list = app_settings_wisun_crc_type_enum,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_custom_phy_config,
    .description = "Custom CRC type [uint8]"
  },
  {
    .key = "preamble_length",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.preamble_length,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_custom_fsk_oqpsk_phy_config,
    .description = "Custom preamble length [uint8]"
  },
  {
    .key = "stf_length",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.stf_length,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_custom_ofdm_phy_config,
    .description = "Custom STF length [uint8]"
  },
  {
    .key = "protocol_id",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.protocol_id,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_ids_phy_config,
    .description = "Radioconf protocol ID [uint16]"
  },
  {
    .key = "channel_id",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.channel_id,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_ids_phy_config,
    .description = "Radioconf channel ID [uint16]"
  },
  {
    .key = "network_size",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.network_size,
    .input_enum_list = app_settings_wisun_network_size_enum,
    .output_enum_list = app_settings_wisun_network_size_enum,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Wi-SUN network size [uint8]"
  },
  {
    .key = "tx_power",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT | APP_SETTINGS_INPUT_FLAG_SIGNED,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT | APP_SETTINGS_OUTPUT_FLAG_SIGNED,
    .value = &app_settings_wisun.tx_power_ddbm,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_tx_power,
    .get_handler = app_settings_get_tx_power,
    .description = "TX power in dBm [int8]"
  },
  {
    .key = "tx_power_ddbm",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT | APP_SETTINGS_INPUT_FLAG_SIGNED,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT | APP_SETTINGS_OUTPUT_FLAG_SIGNED,
    .value = &app_settings_wisun.tx_power_ddbm,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_tx_power,
    .get_handler = app_settings_get_tx_power,
    .description = "TX power in ddBm [int16]"
  },
  {
    .key = "unicast_dwell_interval",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.uc_dwell_interval_ms,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Unicast dwell interval in ms [uint8]"
  },
  {
    .key = "mac_address",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_NONE,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = NULL,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_mac_address,
    .get_handler = app_settings_get_mac_address,
    .description = "MAC address"
  },
  {
    .key = "ip_addresses",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_NONE,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = NULL,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_ip_addresses,
    .description = "IPv6 addresses"
  },
  {
    .key = "border_router",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_NONE,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = NULL,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_border_router,
    .description = "Border router IP address"
  },
  {
    .key = "parents",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_NONE,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = NULL,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_parents,
    .description = "Primary and secondary parent IP addresses"
  },
  {
    .key = "neighbors",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_NONE,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = NULL,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_neighbors,
    .description = "RPL neighbors (parents and children)"
  },
  {
    .key = "async_channel_mask",
    .domain = app_settings_domain_wisun,
    .value_size = APP_UTIL_PRINTABLE_DATA_MAX_LENGTH+1,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = NULL,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_async_channel_mask_str,
    .description = "Excluded channel mask applied to async frames"
  },
  {
    .key = "unicast_channel_mask",
    .domain = app_settings_domain_wisun,
    .value_size = APP_UTIL_PRINTABLE_DATA_MAX_LENGTH+1,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = NULL,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_unicast_channel_mask_str,
    .description = "Excluded channel mask applied to unicast frequency hopping"
  },
  {
    .key = "broadcast_channel_mask",
    .domain = app_settings_domain_wisun,
    .value_size = APP_UTIL_PRINTABLE_DATA_MAX_LENGTH+1,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = NULL,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_broadcast_channel_mask_str,
    .description = "Excluded channel mask applied to broadcast frequency hopping"
  },
  {
    .key = "allowed_channels",
    .domain = app_settings_domain_wisun,
    .value_size = APP_UTIL_PRINTABLE_DATA_MAX_LENGTH+1,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = app_settings_wisun.allowed_channels,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_allowed_channels,
    .get_handler = app_settings_get_string,
    .description = "Allowed channel ranges (e.g. 0-54,57-60,64,67-68)"
  },
  {
    .key = "trace_filter",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_NONE,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = NULL,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_trace_filter,
    .get_handler = app_settings_get_trace_filter,
    .description = "Trace filter group list, 1 bit per group"
  },
  {
    .key = "regulation",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.regulation,
    .input_enum_list = app_settings_wisun_regulation_enum,
    .output_enum_list = app_settings_wisun_regulation_enum,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Regional regulation [uint8]"
  },
  {
    .key = "regulation_warning_threshold",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_OUTPUT_FLAG_SIGNED,
    .output = APP_SETTINGS_OUTPUT_FLAG_SIGNED,
    .value = &app_settings_wisun.regulation_warning_threshold,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_regulation_warning_threshold,
    .get_handler = app_settings_get_integer,
    .description = "Transmission warning threshold in percent (-1 to disable) [int8]"
  },
  {
    .key = "regulation_alert_threshold",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_OUTPUT_FLAG_SIGNED,
    .output = APP_SETTINGS_OUTPUT_FLAG_SIGNED,
    .value = &app_settings_wisun.regulation_alert_threshold,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_regulation_alert_threshold,
    .get_handler = app_settings_get_integer,
    .description = "Transmission alert threshold in percent (-1 to disable) [int8]"
  },
  {
    .key = "device_type",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.device_type,
    .input_enum_list = app_settings_wisun_device_type_enum,
    .output_enum_list = app_settings_wisun_device_type_enum,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Device type [uint8]"
  },
#if SL_RAIL_IEEE802154_SUPPORTS_G_MODE_SWITCH
  {
    .key = "rx_mdr_capable",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.rx_mdr_capable,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_rx_mdr_capable,
    .get_handler = app_settings_get_integer,
    .description = "Indicate if MDR Command is supported"
  },
  {
    .key = "rx_phy_mode_ids_count",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.rx_phy_mode_ids_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = NULL,
    .description = "Number of PhyModeId configured"
  },
  {
    .key = "rx_phy_mode_ids",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.rx_phy_mode_ids,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_rx_phy_mode_ids,
    .get_handler = app_settings_get_rx_phy_mode_ids,
    .description = "List of PhyModeId to advertise in POM-IE"
  },
#endif
  {
    .key = "lfn_profile",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.lfn_profile,
    .input_enum_list = app_settings_wisun_lfn_profile_enum,
    .output_enum_list = app_settings_wisun_lfn_profile_enum,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Wi-SUN LFN profile [uint8]"
  },
  {
    .key = "phy",
    .domain = app_settings_domain_statistics,
    .value_size = APP_SETTINGS_VALUE_SIZE_NONE,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = (void *)SL_WISUN_STATISTICS_TYPE_PHY,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_statistics,
    .description = "PHY statistics and errors"
  },
  {
    .key = "mac",
    .domain = app_settings_domain_statistics,
    .value_size = APP_SETTINGS_VALUE_SIZE_NONE,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = (void *)SL_WISUN_STATISTICS_TYPE_MAC,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_statistics,
    .description = "MAC statistics and errors"
  },
  {
    .key = "fhss",
    .domain = app_settings_domain_statistics,
    .value_size = APP_SETTINGS_VALUE_SIZE_NONE,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = (void *)SL_WISUN_STATISTICS_TYPE_FHSS,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_statistics,
    .description = "FHSS statistics and errors"
  },
  {
    .key = "wisun",
    .domain = app_settings_domain_statistics,
    .value_size = APP_SETTINGS_VALUE_SIZE_NONE,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = (void *)SL_WISUN_STATISTICS_TYPE_WISUN,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_statistics,
    .description = "Wi-SUN statistics and errors"
  },
  {
    .key = "network",
    .domain = app_settings_domain_statistics,
    .value_size = APP_SETTINGS_VALUE_SIZE_NONE,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = (void *)SL_WISUN_STATISTICS_TYPE_NETWORK,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_statistics,
    .description = "Network statistics and errors"
  },
  {
    .key = "heap",
    .domain = app_settings_domain_statistics,
    .value_size = APP_SETTINGS_VALUE_SIZE_NONE,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = (void *)SL_WISUN_STATISTICS_TYPE_HEAP,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_statistics,
    .description = "Heap usage statistics"
  },
  {
    .key = "regulation",
    .domain = app_settings_domain_statistics,
    .value_size = APP_SETTINGS_VALUE_SIZE_NONE,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = (void *)SL_WISUN_STATISTICS_TYPE_REGULATION,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_statistics_regulation,
    .description = "Regional regulation statistics and errors"
  },
  {
    .key = "printable_data_as_hex",
    .domain = app_settings_domain_app,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_app.printable_data_as_hex,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Print socket data as hex [bool] (0|1)"
  },
  {
    .key = "printable_data_length",
    .domain = app_settings_domain_app,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_app.printable_data_length,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Socket data line length [uint8]"
  },
  {
    .key = "autoconnect",
    .domain = app_settings_domain_app,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_app.autoconnect,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Enable or disable autoconnect [bool] (0|1)"
  },
  {
    .key = "pti_state",
    .domain = app_settings_domain_app,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_app.pti_state,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Disable or enable PTI [bool] (0|1)"
  },
#if SLI_WISUN_DISABLE_SECURITY
  {
    .key = "security_state",
    .domain = app_settings_domain_app,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_security_state,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Set the security state [uint32]"
  },
#endif
  {
    .key = "network",
    .domain = app_settings_domain_info,
    .value_size = APP_SETTINGS_VALUE_SIZE_NONE,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = NULL,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_network_info,
    .description = "Wi-SUN network information"
  },
  {
    .key = "rpl",
    .domain = app_settings_domain_info,
    .value_size = APP_SETTINGS_VALUE_SIZE_NONE,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = NULL,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_rpl_info,
    .description = "Wi-SUN RPL information"
  },
  {
    .key = "max_neighbor_count",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.max_neighbor_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Neighbor table size [uint8]"
  },
  {
    .key = "max_child_count",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.max_child_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Maximum number of RPL children [uint8]"
  },
  {
    .key = "max_security_neighbor_count",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.max_security_neighbor_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Maximum number of neighbor in security table [uint16]"
  },
  {
    .key = "preferred_pan_id",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.preferred_pan_id,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_preferred_pan,
    .get_handler = app_settings_get_integer,
    .description = "Preferred PAN ID [uint16]"
  },
  {
    .key = "identifier",
    .domain = app_settings_domain_ping,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_ping.identifier,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Identifier [uint16]"
  },
  {
    .key = "sequence_number",
    .domain = app_settings_domain_ping,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_ping.sequence_number,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Sequence number [uint16]"
  },
  {
    .key = "packet_interval",
    .domain = app_settings_domain_ping,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_ping.packet_interval,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Packet interval in ms [uint16]"
  },
  {
    .key = "packet_length",
    .domain = app_settings_domain_ping,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_ping.packet_length,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Packet length [uint16]"
  },
  {
    .key = "pattern",
    .domain = app_settings_domain_ping,
    .value_size = SL_WISUN_PING_PATTERN_SIZE+1,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = app_settings_ping.pattern,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_string,
    .get_handler = app_settings_get_string,
    .description = "Pattern [string] max 16"
  },
  {
    .key = "keychain",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.keychain,
    .input_enum_list = app_settings_wisun_keychain_enum,
    .output_enum_list = app_settings_wisun_keychain_enum,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Keychain [uint8]"
  },
  {
    .key = "keychain_index",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.keychain_index,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Built-in keychain index [uint8]"
  },
  {
    .key = "direct_connect_pmk",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_NONE,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.direct_connect_pmk,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_direct_connect_pmk,
    .get_handler = app_settings_get_direct_connect_pmk,
    .description = "Set Direct Connect PMK [string]"
  },
  {
    .key = "max_hop_count",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.max_hop_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Maximum hop count requested in PAN [uint8]"
  },
  {
    .key = "lowpan_mtu",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.lowpan_mtu,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "6Lowpan MTU in bytes [uint16]"
  },
  {
    .key = "ipv6_mru",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.ipv6_mru,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "IPv6 MRU in bytes [uint16]"
  },
  {
    .key = "max_edfe_fragment_count",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.max_edfe_fragment_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Maximum EDFE fragment count [uint16]"
  },
  {
    .key = "socket_rx_buffer_size",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.socket_rx_buffer_size,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Socket receiver buffer size in bytes [uint16]"
  },
  {
    .key = "min_be",
    .domain = app_settings_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_mac.min_be,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Minimum CSMA-CA backoff exponent [uint8]"
  },
  {
    .key = "max_be",
    .domain = app_settings_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_mac.max_be,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Maximum CSMA-CA backoff exponent [uint8]"
  },
  {
    .key = "backoff_period_us",
    .domain = app_settings_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_mac.backoff_period_us,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Length of one backoff period in microseconds [uint16]"
  },
  {
    .key = "max_cca_retries",
    .domain = app_settings_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_mac.max_cca_retries,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Maximum number of CCA retries [uint8]"
  },
  {
    .key = "max_frame_retries",
    .domain = app_settings_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_mac.max_frame_retries,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Maximum number of transmission retries [uint8]"
  },
  {
    .key = "eap_identity",
    .domain = app_settings_domain_wisun,
    .value_size = SL_WISUN_EAP_IDENTITY_SIZE+1,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_settings_wisun.eap_identity,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_string,
    .get_handler = app_settings_get_string,
    .description = "EAP identity [string] max 32"
  },
  {
    .key = NULL,
    .domain = 0,
    .value_size = APP_SETTINGS_VALUE_SIZE_NONE,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = NULL,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = NULL,
    .description = NULL
  }
};

static const app_settings_entry_t app_trace_filter_entries[] = {
  {
    .key = "000-031",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT | APP_SETTINGS_OUTPUT_FLAG_HEX | APP_SETTINGS_OUTPUT_FLAG_FIXEDSIZE,
    .value = &app_settings_wisun.trace_filter[0],
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Trace filter bitfield 000-031 [uint32]"
  },
  {
    .key = "032-063",
    .domain = app_settings_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT | APP_SETTINGS_OUTPUT_FLAG_HEX | APP_SETTINGS_OUTPUT_FLAG_FIXEDSIZE,
    .value = &app_settings_wisun.trace_filter[4],
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = app_settings_set_integer,
    .get_handler = app_settings_get_integer,
    .description = "Trace filter bitfield 032-063 [uint32]"
  },
  {
    .key = NULL,
    .domain = 0,
    .value_size = APP_SETTINGS_VALUE_SIZE_NONE,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = NULL,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = NULL,
    .description = NULL
  }
};

static const app_settings_entry_t app_statistics_entries[] =
{
  {
    .key = "crc_fails",
    .domain = app_statistics_domain_phy,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.phy.crc_fails,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Crc fails"
  },
  {
    .key = "tx_timeouts",
    .domain = app_statistics_domain_phy,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.phy.tx_timeouts,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Tx timeouts"
  },
  {
    .key = "rx_timeouts",
    .domain = app_statistics_domain_phy,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.phy.rx_timeouts,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Rx timeouts"
  },
  {
    .key = "tx_queue_size",
    .domain = app_statistics_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.mac.tx_queue_size,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Tx queue size"
  },
  {
    .key = "tx_queue_peak",
    .domain = app_statistics_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.mac.tx_queue_peak,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Tx queue peak"
  },
  {
    .key = "rx_ms_count",
    .domain = app_statistics_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.mac.rx_ms_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Rx count using mode switch"
  },
  {
    .key = "tx_ms_count",
    .domain = app_statistics_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.mac.tx_ms_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Tx count using mode switch"
  },
  {
    .key = "rx_ms_failed",
    .domain = app_statistics_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.mac.rx_ms_failed_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Rx failed count using mode switch"
  },
  {
    .key = "tx_ms_failed",
    .domain = app_statistics_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.mac.tx_ms_failed_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Tx failed count using mode switch"
  },
  {
    .key = "rx_count",
    .domain = app_statistics_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.mac.rx_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Rx count"
  },
  {
    .key = "tx_count",
    .domain = app_statistics_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.mac.tx_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Tx count"
  },
  {
    .key = "bc_rx_count",
    .domain = app_statistics_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.mac.bc_rx_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Bc rx count"
  },
  {
    .key = "bc_tx_count",
    .domain = app_statistics_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.mac.bc_tx_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Bc tx count"
  },
  {
    .key = "rx_drop_count",
    .domain = app_statistics_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.mac.rx_drop_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Rx drop count"
  },
  {
    .key = "tx_bytes",
    .domain = app_statistics_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.mac.tx_bytes,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Tx bytes"
  },
  {
    .key = "rx_bytes",
    .domain = app_statistics_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.mac.rx_bytes,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Rx bytes"
  },
  {
    .key = "tx_failed_count",
    .domain = app_statistics_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.mac.tx_failed_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Tx failed count"
  },
  {
    .key = "retry_count",
    .domain = app_statistics_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.mac.retry_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Retry count"
  },
  {
    .key = "cca_attempts_count",
    .domain = app_statistics_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.mac.cca_attempts_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Cca attempts count"
  },
  {
    .key = "failed_cca_count",
    .domain = app_statistics_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.mac.failed_cca_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Failed cca count"
  },
  {
    .key = "drift_compensation",
    .domain = app_statistics_domain_fhss,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT | APP_SETTINGS_INPUT_FLAG_SIGNED,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT | APP_SETTINGS_OUTPUT_FLAG_SIGNED,
    .value = &app_statistics.fhss.drift_compensation,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Drift compensation"
  },
  {
    .key = "hop_count",
    .domain = app_statistics_domain_fhss,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.fhss.hop_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Hop count"
  },
  {
    .key = "synch_interval",
    .domain = app_statistics_domain_fhss,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.fhss.synch_interval,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Synch interval"
  },
  {
    .key = "prev_avg_synch_fix",
    .domain = app_statistics_domain_fhss,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT | APP_SETTINGS_INPUT_FLAG_SIGNED,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT | APP_SETTINGS_OUTPUT_FLAG_SIGNED,
    .value = &app_statistics.fhss.prev_avg_synch_fix,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Prev avf synch fix"
  },
  {
    .key = "synch_lost",
    .domain = app_statistics_domain_fhss,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.fhss.synch_lost,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Synch lost"
  },
  {
    .key = "unknown_neighbor",
    .domain = app_statistics_domain_fhss,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.fhss.unknown_neighbor,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Unknown neighbor"
  },
  {
    .key = "pan_control_rx_count",
    .domain = app_statistics_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.wisun.pan_control_rx_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "PAN control rx count"
  },
  {
    .key = "pan_control_tx_count",
    .domain = app_statistics_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.wisun.pan_control_tx_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "PAN control tx count"
  },
#ifdef SL_CATALOG_POWER_MANAGER_PRESENT
  {
    .key = "em0_time_ms",
    .domain = app_statistics_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &lfn_em_time_ms[SL_POWER_MANAGER_EM0],
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Get EM0 time in milliseconds"
  },
    {
    .key = "em1_time_ms",
    .domain = app_statistics_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &lfn_em_time_ms[SL_POWER_MANAGER_EM1],
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Get EM1 time in milliseconds"
  },
  {
    .key = "em2_time_ms",
    .domain = app_statistics_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &lfn_em_time_ms[SL_POWER_MANAGER_EM2],
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Get EM2 time in milliseconds"
  },
    {
    .key = "em3_time_ms",
    .domain = app_statistics_domain_wisun,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &lfn_em_time_ms[SL_POWER_MANAGER_EM3],
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Get EM3 time in milliseconds"
  },
#endif
  {
    .key = "ip_rx_count",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.ip_rx_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Ip rx count"
  },
  {
    .key = "ip_tx_count",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.ip_tx_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Ip tx count"
  },
  {
    .key = "ip_rx_drop",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.ip_rx_drop,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Ip rx drop"
  },
  {
    .key = "ip_cksum_error",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.ip_cksum_error,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Ip cksum error"
  },
  {
    .key = "ip_tx_bytes",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.ip_tx_bytes,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Ip tx bytes"
  },
  {
    .key = "ip_rx_bytes",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.ip_rx_bytes,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Ip rx bytes"
  },
  {
    .key = "ip_routed_up",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.ip_routed_up,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Ip routed up"
  },
  {
    .key = "ip_no_route",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.ip_no_route,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Ip no route"
  },
  {
    .key = "frag_rx_errors",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.frag_rx_errors,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Frag rx errors"
  },
  {
    .key = "frag_tx_errors",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.frag_tx_errors,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Frag tx errors"
  },
  {
    .key = "rpl_route_routecost_better_change",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.rpl_route_routecost_better_change,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Rpl route routecost better change"
  },
  {
    .key = "ip_routeloop_detect",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.ip_routeloop_detect,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Ip routeloop detect"
  },
  {
    .key = "rpl_memory_overflow",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.rpl_memory_overflow,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Rpl memory overflow"
  },
  {
    .key = "rpl_parent_tx_fail",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.rpl_parent_tx_fail,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Rpl parent tx fail"
  },
  {
    .key = "rpl_unknown_instance",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.rpl_unknown_instance,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Rpl unknown instance"
  },
  {
    .key = "rpl_local_repair",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.rpl_local_repair,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Rpl local repair"
  },
  {
    .key = "rpl_global_repair",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.rpl_global_repair,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Rpl global repair"
  },
  {
    .key = "rpl_malformed_message",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.rpl_malformed_message,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Rpl malformed message"
  },
  {
    .key = "rpl_time_no_next_hop",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.rpl_time_no_next_hop,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Rpl time no next hop"
  },
  {
    .key = "rpl_total_memory",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.rpl_total_memory,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Rpl total memory"
  },
  {
    .key = "buf_alloc",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.buf_alloc,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Buf alloc"
  },
  {
    .key = "buf_headroom_realloc",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.buf_headroom_realloc,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Buf headroom realloc"
  },
  {
    .key = "buf_headroom_shuffle",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.buf_headroom_shuffle,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Buf headroom shuffle"
  },
  {
    .key = "buf_headroom_fail",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.buf_headroom_fail,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Buf headroom fail"
  },
  {
    .key = "etx_1st_parent",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.etx_1st_parent,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Etx 1st parent"
  },
  {
    .key = "etx_2nd_parent",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.etx_2nd_parent,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Etx 2nd parent"
  },
  {
    .key = "adapt_layer_tx_queue_size",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.adapt_layer_tx_queue_size,
    .input_enum_list =  NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Adapt layer tx queue size"
  },
  {
    .key = "adapt_layer_tx_queue_peak",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.adapt_layer_tx_queue_peak,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Adapt layer tx queue peak"
  },
  {
    .key = "mpl_new_messages_count",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.mpl_new_messages_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "MPL new messages count (first RX of messages)"
  },
  {
    .key = "mpl_rx_count",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.mpl_rx_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "MPL total RX count"
  },
  {
    .key = "mpl_forwarded_messages_count",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.mpl_forwarded_messages_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "MPL forwarded messages count"
  },
  {
    .key = "mpl_freed_messages_count",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.mpl_freed_messages_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "MPL freed messages count (when MPL memory is full)"
  },
  {
    .key = "mpl_not_tx_count",
    .domain = app_statistics_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.network.mpl_not_tx_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Number of MPL messages that are deleted and never transmitted"
  },
  {
    .key = "tx_duration_ms",
    .domain = app_statistics_domain_regulation,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.regulation.tx_duration_ms,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Total transmission duration during last hour in milliseconds"
  },
  {
    .key = "tx_duration_per_channel_ms",
    .domain = app_statistics_domain_regulation,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.regulation.tx_duration_per_channel_ms,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Total transmission duration per channel during last hour in milliseconds"
  },
  {
    .key = "duty_cycle_warning_threshold_ms",
    .domain = app_statistics_domain_regulation,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.regulation.duty_cycle_warning_threshold_ms,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Duty cycle warning threshold in milliseconds"
  },
  {
    .key = "duty_cycle_warning_threshold_per_channel_ms",
    .domain = app_statistics_domain_regulation,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.regulation.duty_cycle_warning_threshold_per_channel_ms,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Duty cycle warning threshold per channel in milliseconds"
  },
  {
    .key = "duty_cycle_alert_threshold_ms",
    .domain = app_statistics_domain_regulation,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.regulation.duty_cycle_alert_threshold_ms,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Duty cycle alert threshold in milliseconds"
  },
  {
    .key = "duty_cycle_alert_threshold_per_channel_ms",
    .domain = app_statistics_domain_regulation,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.regulation.duty_cycle_alert_threshold_per_channel_ms,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Duty cycle alert threshold per channel in milliseconds"
  },
  {
    .key = "arena",
    .domain = app_statistics_domain_heap,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.heap.arena,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Maximum heap usage"
  },
  {
    .key = "uordblks",
    .domain = app_statistics_domain_heap,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.heap.uordblks,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Current heap usage"
  },
  {
    .key = "free",
    .domain = app_statistics_domain_heap,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.heap.free,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Current free heap"
  },
  {
    .key = "total",
    .domain = app_statistics_domain_heap,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.heap.total,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Total available heap"
  },
  {
    .key = "idle_duration_s",
    .domain = app_statistics_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.mac.idle_duration_s,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Total duration of radio idle state since MAC started"
  },
  {
    .key = "rx_availability_percentage",
    .domain = app_statistics_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.mac.rx_availability_percentage,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Device RX availability percentage"
  },
  {
    .key = "tx_duration_ms",
    .domain = app_statistics_domain_mac,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT32,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_statistics.mac.radio_tx_duration_ms,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Total transmission duration since boot"
  },
  {
    .key = NULL,
    .domain = 0,
    .value_size = APP_SETTINGS_VALUE_SIZE_NONE,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = NULL,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = NULL,
    .description = NULL
  }
};

static const app_settings_entry_t app_info_entries[] =
{
  {
    .key = "pan_id",
    .domain = app_info_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_network_info.pan_id,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "PAN ID"
  },
  {
    .key = "hop_count",
    .domain = app_info_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_network_info.hop_count,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Hop count"
  },
  {
    .key = NULL,
    .domain = 0,
    .value_size = APP_SETTINGS_VALUE_SIZE_NONE,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = NULL,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = NULL,
    .description = NULL
  }
};

static const app_settings_entry_t app_rpl_entries[] =
{
  {
    .key = "instance_id",
    .domain = app_info_domain_rpl,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_rpl_info.instance_id,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Instance ID"
  },
  {
    .key = "dodag_version_number",
    .domain = app_info_domain_rpl,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_rpl_info.dodag_version_number,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "DODAG Version Number"
  },
  {
    .key = "dodag_rank",
    .domain = app_info_domain_rpl,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_rpl_info.dodag_rank,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "DODAG Rank"
  },
  {
    .key = "grounded",
    .domain = app_info_domain_rpl,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_rpl_info.grounded,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Grounded"
  },
  {
    .key = "mode_of_operation",
    .domain = app_info_domain_rpl,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_rpl_info.mode_of_operation,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Mode of Operation (MOF)"
  },
  {
    .key = "dodag_preference",
    .domain = app_info_domain_rpl,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_rpl_info.dodag_preference,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "DODAG Preference"
  },
  {
    .key = "dodag_dtsn",
    .domain = app_info_domain_rpl,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_rpl_info.dodag_dtsn,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Destination Advertisement Trigger Sequence Number (DTSN)"
  },
  {
    .key = "dio_interval_min",
    .domain = app_info_domain_rpl,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_rpl_info.dio_interval_min,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "DIO Minimum Interval"
  },
  {
    .key = "dio_interval_doublings",
    .domain = app_info_domain_network,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_rpl_info.dio_interval_doublings,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "DIO Interval Doublings"
  },
  {
    .key = "dio_redundancy_constant",
    .domain = app_info_domain_rpl,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_rpl_info.dio_redundancy_constant,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "DIO Redundancy Constant"
  },
  {
    .key = "default_lifetime",
    .domain = app_info_domain_rpl,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT8,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_rpl_info.default_lifetime,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Default Lifetime"
  },
  {
    .key = "lifetime_unit",
    .domain = app_info_domain_rpl,
    .value_size = APP_SETTINGS_VALUE_SIZE_UINT16,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = &app_rpl_info.lifetime_unit,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = app_settings_get_integer,
    .description = "Lifetime Unit"
  },
  {
    .key = NULL,
    .domain = 0,
    .value_size = APP_SETTINGS_VALUE_SIZE_NONE,
    .input = APP_SETTINGS_INPUT_FLAG_DEFAULT,
    .output = APP_SETTINGS_OUTPUT_FLAG_DEFAULT,
    .value = NULL,
    .input_enum_list = NULL,
    .output_enum_list = NULL,
    .set_handler = NULL,
    .get_handler = NULL,
    .description = NULL
  }
};

static sl_status_t app_settings_get_phy_config(char *value_str,
                                               const char *key_str,
                                               const app_settings_entry_t *entry,
                                               bool unused)
{
  sl_status_t status;

  status = app_settings_get_integer(value_str, key_str, entry);
  if (status == SL_STATUS_OK && unused) {
    strcat(value_str, " (unused)");
  }

  return status;
}

static sl_status_t app_settings_get_fan10_phy_config(char *value_str,
                                                     const char *key_str,
                                                     const app_settings_entry_t *entry)
{
  bool used = app_settings_wisun.phy_config_type == SL_WISUN_PHY_CONFIG_FAN10;
  return app_settings_get_phy_config(value_str, key_str, entry, !used);
}

static sl_status_t app_settings_get_fan11_phy_config(char *value_str,
                                                     const char *key_str,
                                                     const app_settings_entry_t *entry)
{
  bool used = app_settings_wisun.phy_config_type == SL_WISUN_PHY_CONFIG_FAN11;
  return app_settings_get_phy_config(value_str, key_str, entry, !used);
}

static sl_status_t app_settings_get_explicit_phy_config(char *value_str,
                                                        const char *key_str,
                                                        const app_settings_entry_t *entry)
{
  bool used = (app_settings_wisun.phy_config_type == SL_WISUN_PHY_CONFIG_EXPLICIT
               || app_settings_wisun.phy_config_type == SL_WISUN_PHY_CONFIG_CUSTOM_FSK
               || app_settings_wisun.phy_config_type == SL_WISUN_PHY_CONFIG_CUSTOM_OFDM
               || app_settings_wisun.phy_config_type == SL_WISUN_PHY_CONFIG_CUSTOM_OQPSK);
  return app_settings_get_phy_config(value_str, key_str, entry, !used);
}

static sl_status_t app_settings_get_custom_phy_config(char *value_str,
                                                      const char *key_str,
                                                      const app_settings_entry_t *entry)
{
  bool used = (app_settings_wisun.phy_config_type == SL_WISUN_PHY_CONFIG_CUSTOM_FSK
               || app_settings_wisun.phy_config_type == SL_WISUN_PHY_CONFIG_CUSTOM_OFDM
               || app_settings_wisun.phy_config_type == SL_WISUN_PHY_CONFIG_CUSTOM_OQPSK);
  return app_settings_get_phy_config(value_str, key_str, entry, !used);
}

static sl_status_t app_settings_get_custom_fsk_oqpsk_phy_config(char *value_str,
                                                                const char *key_str,
                                                                const app_settings_entry_t *entry)
{
  bool used = (app_settings_wisun.phy_config_type == SL_WISUN_PHY_CONFIG_CUSTOM_FSK
               || app_settings_wisun.phy_config_type == SL_WISUN_PHY_CONFIG_CUSTOM_OQPSK);
  return app_settings_get_phy_config(value_str, key_str, entry, !used);
}

static sl_status_t app_settings_get_custom_ofdm_phy_config(char *value_str,
                                                           const char *key_str,
                                                           const app_settings_entry_t *entry)
{
  bool used = (app_settings_wisun.phy_config_type == SL_WISUN_PHY_CONFIG_CUSTOM_OFDM);
  return app_settings_get_phy_config(value_str, key_str, entry, !used);
}

static sl_status_t app_settings_get_ids_phy_config(char *value_str,
                                                   const char *key_str,
                                                   const app_settings_entry_t *entry)
{
  bool used = app_settings_wisun.phy_config_type == SL_WISUN_PHY_CONFIG_IDS;
  return app_settings_get_phy_config(value_str, key_str, entry, !used);
}

static sl_status_t app_settings_get_fan10_and_fan11_phy_config(char *value_str,
                                                               const char *key_str,
                                                               const app_settings_entry_t *entry)
{
  bool used = app_settings_wisun.phy_config_type == SL_WISUN_PHY_CONFIG_FAN10
           || app_settings_wisun.phy_config_type == SL_WISUN_PHY_CONFIG_FAN11;
  return app_settings_get_phy_config(value_str, key_str, entry, !used);
}

static sl_status_t app_settings_get_fan11_and_explicit_and_ids_phy_config(char *value_str,
                                                                          const char *key_str,
                                                                          const app_settings_entry_t *entry)
{
  bool used = app_settings_wisun.phy_config_type == SL_WISUN_PHY_CONFIG_FAN11
           || app_settings_wisun.phy_config_type == SL_WISUN_PHY_CONFIG_EXPLICIT
           || app_settings_wisun.phy_config_type == SL_WISUN_PHY_CONFIG_CUSTOM_FSK
           || app_settings_wisun.phy_config_type == SL_WISUN_PHY_CONFIG_CUSTOM_OFDM
           || app_settings_wisun.phy_config_type == SL_WISUN_PHY_CONFIG_CUSTOM_OQPSK
           || app_settings_wisun.phy_config_type == SL_WISUN_PHY_CONFIG_IDS;
  return app_settings_get_phy_config(value_str, key_str, entry, !used);
}

static sl_status_t app_settings_get_ip_addresses(char *value_str,
                                                 const char *key_str,
                                                 const app_settings_entry_t *entry)
{
  sl_status_t ret;
  in6_addr_t address;
  int address_count = 0;
  int count;
  (void)key_str;
  (void)entry;

  count = sprintf(value_str, "[");
  value_str += count;

  // Global address
  ret = sl_wisun_get_ip_address(SL_WISUN_IP_ADDRESS_TYPE_GLOBAL, &address);
  if (ret == SL_STATUS_OK) {
    count = ip6tos(address.address, value_str);
    value_str += count;
    address_count++;
  }

  // Link-local address
  ret = sl_wisun_get_ip_address(SL_WISUN_IP_ADDRESS_TYPE_LINK_LOCAL, &address);
  if (ret == SL_STATUS_OK) {
    if (address_count) {
      count = sprintf(value_str, ", ");
      value_str += count;
    }
    count = ip6tos(address.address, value_str);
    value_str += count;
    address_count++;
  }

  count = sprintf(value_str, "]");
  value_str += count;

  // Prevent parent from printing anything if no address was found
  return address_count ? SL_STATUS_OK : SL_STATUS_FAIL;
}

static sl_status_t app_settings_get_border_router(char *value_str,
                                                  const char *key_str,
                                                  const app_settings_entry_t *entry)
{
  sl_status_t ret;
  in6_addr_t address;
  (void)key_str;
  (void)entry;

  ret = sl_wisun_get_ip_address(SL_WISUN_IP_ADDRESS_TYPE_BORDER_ROUTER, &address);
  if (ret == SL_STATUS_OK) {
    ip6tos(address.address, value_str);
  }

  return ret;
}

static sl_status_t app_settings_get_parents(char *value_str,
                                            const char *key_str,
                                            const app_settings_entry_t *entry)
{
  sl_status_t ret;
  in6_addr_t address;
  int address_count = 0;
  int count;
  (void)key_str;
  (void)entry;

  count = sprintf(value_str, "[");
  value_str += count;

  // Primary parent
  ret = sl_wisun_get_ip_address(SL_WISUN_IP_ADDRESS_TYPE_PRIMARY_PARENT, &address);
  if (ret == SL_STATUS_OK) {
    count = ip6tos(address.address, value_str);
    value_str += count;
    address_count++;
  }

  // Secondary parent
  ret = sl_wisun_get_ip_address(SL_WISUN_IP_ADDRESS_TYPE_SECONDARY_PARENT, &address);
  if (ret == SL_STATUS_OK) {
    if (address_count) {
      count = sprintf(value_str, ", ");
      value_str += count;
    }
    count = ip6tos(address.address, value_str);
    value_str += count;
    address_count++;
  }

  count = sprintf(value_str, "]");
  value_str += count;

  // Prevent parent from printing anything if no address was found
  return address_count ? SL_STATUS_OK : SL_STATUS_FAIL;
}

static sl_status_t app_settings_get_neighbors(char *value_str,
                                              const char *key_str,
                                              const app_settings_entry_t *entry)
{
  sl_status_t ret;
  uint8_t neighbor_count, idx, j;
  sl_wisun_mac_address_t *neighbor_mac_addresses;
  (void)key_str;
  (void)entry;

  ret = sl_wisun_get_neighbor_count(&neighbor_count);
  if (ret == SL_STATUS_OK && neighbor_count) {
    neighbor_mac_addresses = sl_malloc(neighbor_count * sizeof(sl_wisun_mac_address_t));
    if (neighbor_mac_addresses) {
      ret = sl_wisun_get_neighbors(&neighbor_count, neighbor_mac_addresses);
      if (ret == SL_STATUS_OK) {
        printf("%s.%s = [\r\n", app_settings_domain_str[entry->domain], entry->key);
        for (idx = 0; idx < neighbor_count; idx++) {
          sl_wisun_neighbor_info_t neighbor_info;
          app_util_get_mac_address_string(value_str, neighbor_mac_addresses + idx);
          printf("  %s\r\n", value_str);
          ret = sl_wisun_get_neighbor_info(neighbor_mac_addresses + idx, &neighbor_info);
          if (ret == SL_STATUS_OK)  {
            app_util_get_string(value_str, neighbor_info.type, app_settings_wisun_neighbor_type_enum, false, false, 0);
            printf("    type = %s\r\n", value_str);
            ip6tos(neighbor_info.link_local_address.address, value_str);
            printf("    ll = %s\r\n", value_str);
            ip6tos(neighbor_info.global_address.address, value_str);
            if (strcmp(value_str, "::")) {
              printf("    gua = %s\r\n", value_str);
            }
            printf("    lifetime = %lu\r\n", neighbor_info.lifetime);
            printf("    mac_tx_count = %lu\r\n", neighbor_info.mac_tx_count);
            printf("    mac_tx_failed_count = %lu\r\n", neighbor_info.mac_tx_failed_count);
            printf("    mac_tx_ms_count = %lu\r\n", neighbor_info.mac_tx_ms_count);
            printf("    mac_tx_ms_failed_count = %lu\r\n", neighbor_info.mac_tx_ms_failed_count);
            printf("    mac_rx_count = %lu\r\n", neighbor_info.mac_rx_count);
            if (neighbor_info.rpl_rank != 0xFFFF) {
              printf("    rpl_rank = %hu\r\n", neighbor_info.rpl_rank);
            }
            if (neighbor_info.etx != 0xFFFF) {
              printf("    etx = %hu\r\n", neighbor_info.etx);
            }
            if (neighbor_info.rsl_out != 0xFF) {
              printf("    rsl_out = %d dBm\r\n", (int16_t)neighbor_info.rsl_out - 174);
            }
            if (neighbor_info.rsl_in != 0xFF) {
              printf("    rsl_in = %d dBm\r\n", (int16_t)neighbor_info.rsl_in - 174);
            }
            app_util_get_string(value_str, neighbor_info.is_lfn, app_settings_wisun_is_lfn_enum, false, false, 0);
            printf("    device_type = %s\r\n", value_str);
            printf("    phy_mode_ids = ");
            if (neighbor_info.phy_mode_id_count != 0) {
              for (j = 0; j < neighbor_info.phy_mode_id_count - 1; j++) {
                printf("0x%02x,", neighbor_info.phy_mode_ids[j]);
              }
              // Last one without coma
              printf("0x%02x\r\n", neighbor_info.phy_mode_ids[j]);
            } else {
              printf("none\r\n");
            }
            printf("    is_mdr_command_capable = %s\r\n", neighbor_info.is_mdr_command_capable ? "true" : "false");
          }
        }
        printf("]\r\n");
      }
      sl_free(neighbor_mac_addresses);
    }
  }

  // Prevent parent from printing anything
  return SL_STATUS_FAIL;
}

static sl_status_t app_settings_get_statistics(char *value_str,
                                               const char *key_str,
                                               const app_settings_entry_t *entry)
{
  sl_status_t ret;
  const app_settings_entry_t *iter;
  sl_wisun_statistics_type_t statistics_type;

  // API statistics type is stored in the settings entry value.
  #ifdef __GNUC__
  statistics_type = (sl_wisun_statistics_type_t) entry->value;
  #elif defined __ICCARM__
  statistics_type = (sl_wisun_statistics_type_t)((uint32_t) entry->value & 0xFF);
  #endif

  // Update statistics
  ret = sl_wisun_get_statistics(statistics_type, &app_statistics);
  if (ret != SL_STATUS_OK) {
    printf("[Failed to retrieve statistics: %lu]\r\n", ret);
    return SL_STATUS_FAIL;
  }

  iter = app_statistics_entries;
  while (iter->key) {
    if (!strcmp(entry->key, app_statistics_domain_str[iter->domain])) {
      if (!key_str || !strcmp(iter->key, key_str)) {
        if (iter->get_handler) {
          ret = iter->get_handler(value_str, NULL, iter);
          if (ret == SL_STATUS_OK) {
            printf("%s.%s.%s = %s\r\n", app_settings_domain_str[entry->domain], app_statistics_domain_str[iter->domain], iter->key, value_str);
          }
        }
      }
    }
    iter++;
  }

  // Prevent parent from printing anything
  return SL_STATUS_FAIL;
}

static sl_status_t app_settings_get_statistics_regulation(char *value_str,
                                                          const char *key_str,
                                                          const app_settings_entry_t *entry)
{
  sl_status_t ret;
  const app_settings_entry_t *iter;

  // Update statistics
  ret = sl_wisun_get_statistics(SL_WISUN_STATISTICS_TYPE_REGULATION, &app_statistics);
  if (ret != SL_STATUS_OK) {
    return ret;
  }

  iter = app_statistics_entries;
  while (iter->key) {
    if (!strcmp(entry->key, app_statistics_domain_str[iter->domain])) {
      if (!key_str || !strcmp(iter->key, key_str)) {
        if (iter->get_handler) {
          ret = iter->get_handler(value_str, NULL, iter);
          if (ret == SL_STATUS_OK) {
            printf("%s.%s.%s = %s\r\n", app_settings_domain_str[entry->domain], app_statistics_domain_str[iter->domain], iter->key, value_str);
          }
        }
      }
    }
    iter++;
  }

  // Prevent parent from printing anything
  return SL_STATUS_FAIL;
}

static sl_status_t app_settings_get_channel_mask_str(sl_wisun_channel_mask_type_t type, char *value_str)


{
  sl_status_t status;
  sl_wisun_channel_mask_t channel_mask;
  uint32_t channel_mask_len;

  status = sl_wisun_get_excluded_channel_mask(type, &channel_mask, &channel_mask_len);

  strcpy(value_str, "--");

  if (status == SL_STATUS_OK && channel_mask_len > 0) {
    sprintf(value_str, "%02x", channel_mask.mask[0]);
    value_str += 2;
    for (uint32_t i = 1; i < (channel_mask_len + 7) / 8; i++) {
      sprintf(value_str, ":%02x", channel_mask.mask[i]);
      value_str += 3;
    }
  }

  return SL_STATUS_OK;
}

static sl_status_t app_settings_get_async_channel_mask_str(char *value_str,
                                                           const char *key_str,
                                                           const app_settings_entry_t *entry)
{
  (void)key_str;
  (void)entry;

  return app_settings_get_channel_mask_str(SL_WISUN_CHANNEL_MASK_TYPE_EFFECTIVE_ASYNC, value_str);
}

static sl_status_t app_settings_get_unicast_channel_mask_str(char *value_str,
                                                             const char *key_str,
                                                             const app_settings_entry_t *entry)
{
  (void)key_str;
  (void)entry;

  return app_settings_get_channel_mask_str(SL_WISUN_CHANNEL_MASK_TYPE_EFFECTIVE_UNICAST, value_str);
}

static sl_status_t app_settings_get_broadcast_channel_mask_str(char *value_str,
                                                               const char *key_str,
                                                               const app_settings_entry_t *entry)
{
  (void)key_str;
  (void)entry;

  return app_settings_get_channel_mask_str(SL_WISUN_CHANNEL_MASK_TYPE_EFFECTIVE_BROADCAST, value_str);
}

static sl_status_t app_ranges_to_mask(const char *str, uint8_t *mask, uint32_t size)
{
  char *endptr;
  uint32_t cur, end, index;
  memset(mask, 0, size * sizeof(uint8_t));
  do {
    if (*str == '\0') {
      return SL_STATUS_FAIL;
    }
    cur = strtoul(str, &endptr, 0);
    if (*endptr == '-') {
      str = endptr + 1;
      end = strtoul(str, &endptr, 0);
    } else {
      end = cur;
    }
    if (*endptr != '\0' && *endptr != ',') {
      return SL_STATUS_FAIL;
    }
    if (cur > end) {
      return SL_STATUS_FAIL;
    }
    for (; cur <= end; cur++) {
      index = cur / 8;
      if (index < size) {
        mask[index] |= 1 << (cur % 8);
      } else {
        return SL_STATUS_FAIL;
      }
    }
    str = endptr + 1;
  } while (*endptr != '\0');

  return SL_STATUS_OK;
}

sl_status_t app_settings_get_channel_mask(const char *str, sl_wisun_channel_mask_t *channel_mask)
{
  return app_ranges_to_mask(str, channel_mask->mask, SL_WISUN_CHANNEL_MASK_SIZE);
}

static sl_status_t app_settings_set_allowed_channels(const char *value_str,
                                                     const char *key_str,
                                                     const app_settings_entry_t *entry)
{
  sl_status_t ret;
  sl_wisun_channel_mask_t channel_mask;
  (void)key_str;
  (void)entry;

  ret = app_settings_get_channel_mask(value_str, &channel_mask);
  if (ret == SL_STATUS_OK) {
    ret = app_settings_set_string(value_str, key_str, entry);
  }

  return ret;
}

static sl_status_t app_settings_set_trace_filter(const char *value_str,
                                                 const char *key_str,
                                                 const app_settings_entry_t *entry)
{
  sl_status_t ret;
  const app_settings_entry_t *iter;
  (void)entry;

  if (!value_str) {
    return SL_STATUS_INVALID_KEY;
  }

  iter = app_trace_filter_entries;
  while (iter->key) {
    if (!key_str || !strcmp(iter->key, key_str)) {
      if (iter->set_handler) {
        ret = iter->set_handler(value_str, key_str, iter);
        if (ret != SL_STATUS_OK) {
          return ret;
        } else {
          return sl_wisun_set_trace_filter(app_settings_wisun.trace_filter);
        }
      } else {
        return SL_STATUS_PERMISSION;
      }
    }
    iter++;
  }

  return SL_STATUS_OK;
}

static sl_status_t app_settings_get_trace_filter(char *value_str,
                                                 const char *key_str,
                                                 const app_settings_entry_t *entry)
{
  sl_status_t ret;
  const app_settings_entry_t *iter;
  char trace_filter_string[128];
  (void)value_str;

  iter = app_trace_filter_entries;
  while (iter->key) {
    if (!key_str || !strcmp(iter->key, key_str)) {
      if (iter->get_handler) {
        ret = iter->get_handler(trace_filter_string, NULL, iter);
        if (ret == SL_STATUS_OK) {
          printf("%s.%s.%s = %s\r\n", app_settings_domain_str[entry->domain], entry->key, iter->key, trace_filter_string);
        }
      }
    }
    iter++;
  }

  // Prevent parent from printing anything
  return SL_STATUS_FAIL;
}

static sl_status_t app_settings_get_join_state(char *value_str,
                                               const char *key_str,
                                               const app_settings_entry_t *entry)
{
  sl_status_t ret;
  sl_wisun_join_state_t join_state;
  uint32_t value;
  app_settings_entry_t ent = *entry;

  ret = sl_wisun_get_join_state(&join_state);
  if (ret == SL_STATUS_OK) {
    if (app_settings_wisun.device_type == SL_WISUN_LFN) {
      ent.output_enum_list = app_settings_wisun_join_state_enum_lfn;
    }
    value = join_state;
    ent.value = &value;
    ret = app_settings_get_integer(value_str, key_str, (const app_settings_entry_t *)&ent);
  }

  return ret;
}

static sl_status_t app_settings_set_mac_address(const char *value_str,
                                                const char *key_str,
                                                const app_settings_entry_t *entry)
{
  sl_status_t ret;
  sl_wisun_mac_address_t address;
  (void)key_str;
  (void)entry;

  ret = app_util_get_mac_address(&address, value_str);
  if (ret == SL_STATUS_OK) {
    ret = sl_wisun_set_mac_address(&address);
  }

  return ret;
}

static sl_status_t app_settings_get_mac_address(char *value_str,
                                                const char *key_str,
                                                const app_settings_entry_t *entry)
{
  sl_status_t ret;
  sl_wisun_mac_address_t address;
  (void)key_str;
  (void)entry;

  ret = sl_wisun_get_mac_address(&address);
  if (ret == SL_STATUS_OK) {
    ret = app_util_get_mac_address_string(value_str, &address);
  }

  return ret;
}

static sl_status_t app_settings_set_regulation_warning_threshold(const char *value_str,
                                                                 const char *key_str,
                                                                 const app_settings_entry_t *entry)
{
  sl_status_t ret;

  ret = app_settings_set_integer(value_str, key_str, entry);
  if (ret == SL_STATUS_OK) {
    ret = sl_wisun_set_regulation_tx_thresholds(app_settings_wisun.regulation_warning_threshold,
                                                app_settings_wisun.regulation_alert_threshold);
  }

  return ret;
}

static sl_status_t app_settings_set_regulation_alert_threshold(const char *value_str,
                                                               const char *key_str,
                                                               const app_settings_entry_t *entry)
{
  sl_status_t ret;

  ret = app_settings_set_integer(value_str, key_str, entry);
  if (ret == SL_STATUS_OK) {
    ret = sl_wisun_set_regulation_tx_thresholds(app_settings_wisun.regulation_warning_threshold,
                                                app_settings_wisun.regulation_alert_threshold);
  }

  return ret;
}

#if SL_RAIL_IEEE802154_SUPPORTS_G_MODE_SWITCH
static sl_status_t app_settings_set_rx_mdr_capable(const char *value_str,
                                                   const char *key_str,
                                                   const app_settings_entry_t *entry)
{
  sl_status_t ret;
  uint8_t phy_mode_id_count, is_mdr_command_capable;
  uint8_t phy_mode_id[SL_WISUN_MAX_PHY_MODE_ID_COUNT];
  uint8_t *phy_mode_id_p, *phy_mode_id_count_p;

  ret = app_settings_set_integer(value_str, key_str, entry);

  if (ret == SL_STATUS_OK) {

    if (app_settings_wisun.rx_phy_mode_ids_count == 0) {
      // Check if default PhyList can be retrieved from device
      if (sl_wisun_get_pom_ie(&phy_mode_id_count, phy_mode_id, &is_mdr_command_capable) == SL_STATUS_OK) {
        phy_mode_id_p = phy_mode_id;
        phy_mode_id_count_p = &phy_mode_id_count;
      } else {
        // POM-IE not available yet
        return ret;
      }
    } else {
      phy_mode_id_p = app_settings_wisun.rx_phy_mode_ids;
      phy_mode_id_count_p = &app_settings_wisun.rx_phy_mode_ids_count;
    }

    ret = sl_wisun_set_pom_ie(*phy_mode_id_count_p,
                              phy_mode_id_p,
                              app_settings_wisun.rx_mdr_capable);
  }

  return ret;
}

static sl_status_t app_settings_set_rx_phy_mode_ids(const char *value_str,
                                                    const char *key_str,
                                                    const app_settings_entry_t *entry)
{
  sl_status_t ret = SL_STATUS_OK;
  char *end_str, *start_str;
  uint8_t phy_mode_id;
  uint8_t phy_mode_ids_count = 0;
  uint8_t phy_mode_ids[SL_WISUN_MAX_PHY_MODE_ID_COUNT];


  (void)key_str;
  (void)entry;

  start_str = (char *) value_str;

  do {
    phy_mode_id = strtoul(start_str, &end_str, 0);

    if (end_str == start_str) {
      // No integer found
      ret = SL_STATUS_INVALID_PARAMETER;
      break;
    }

    phy_mode_ids[phy_mode_ids_count] = phy_mode_id;
    phy_mode_ids_count++;

    if (*end_str == ',') {
      // list separator, point to next element
      end_str++;
    }
    start_str = end_str;
  } while (*end_str != '\0');

  if (ret == SL_STATUS_OK) {
    memcpy(app_settings_wisun.rx_phy_mode_ids, phy_mode_ids, phy_mode_ids_count);
    app_settings_wisun.rx_phy_mode_ids_count = phy_mode_ids_count;

    ret = sl_wisun_set_pom_ie(app_settings_wisun.rx_phy_mode_ids_count,
                              app_settings_wisun.rx_phy_mode_ids,
                              app_settings_wisun.rx_mdr_capable);
  }

  return ret;
}

static sl_status_t app_settings_get_rx_phy_mode_ids(char *value_str,
                                                    const char *key_str,
                                                    const app_settings_entry_t *entry)
{
  int i;
  sl_status_t ret = SL_STATUS_OK;
  uint8_t phy_mode_id_count, is_mdr_command_capable;
  uint8_t phy_mode_id[SL_WISUN_MAX_PHY_MODE_ID_COUNT];
  uint8_t *phy_mode_id_p, *phy_mode_id_count_p;

  (void) key_str;
  (void) entry;


  if (app_settings_wisun.rx_phy_mode_ids_count == 0) {
    // Check if default PhyList can be retrieved from device
    if (sl_wisun_get_pom_ie(&phy_mode_id_count, phy_mode_id, &is_mdr_command_capable) == SL_STATUS_OK) {
      phy_mode_id_p = phy_mode_id;
      phy_mode_id_count_p = &phy_mode_id_count;
    } else {
      ret = SL_STATUS_FAIL;
    }
  } else {
    phy_mode_id_p = app_settings_wisun.rx_phy_mode_ids;
    phy_mode_id_count_p = &app_settings_wisun.rx_phy_mode_ids_count;
  }

  if (ret == SL_STATUS_OK) {
    if (*phy_mode_id_count_p == 0) {
      sprintf(value_str, "no PhyModeId set");
    } else {
      for (i = 0; i < *phy_mode_id_count_p - 1; i ++) {
        sprintf(value_str, "0x%02x,", phy_mode_id_p[i]);
        value_str += 5;
      }
      // Last one without coma
      sprintf(value_str, "0x%02x", phy_mode_id_p[i]);
    }
  }

  return SL_STATUS_OK;
}
#endif

static sl_status_t app_settings_get_network_info(char *value_str,
                                                 const char *key_str,
                                                 const app_settings_entry_t *entry)
{
  sl_status_t ret;
  const app_settings_entry_t *iter;

  ret = sl_wisun_get_network_info(&app_network_info);
  if (ret != SL_STATUS_OK) {
    printf("[Failed to retrieve Wi-SUN network information: %lu]\r\n", ret);
    return SL_STATUS_FAIL;
  }

  iter = app_info_entries;
  while (iter->key) {
    if (!strcmp(entry->key, app_info_domain_str[iter->domain])) {
      if (!key_str || !strcmp(iter->key, key_str)) {
        if (iter->get_handler) {
          ret = iter->get_handler(value_str, NULL, iter);
          if (ret == SL_STATUS_OK) {
            printf("%s.%s.%s = %s\r\n", app_settings_domain_str[entry->domain], app_info_domain_str[iter->domain], iter->key, value_str);
          }
        }
      }
    }
    iter++;
  }

  // Prevent parent from printing anything
  return SL_STATUS_FAIL;
}

static sl_status_t app_settings_get_rpl_info(char *value_str,
                                             const char *key_str,
                                             const app_settings_entry_t *entry)
{
  sl_status_t ret;
  const app_settings_entry_t *iter;

  if (app_settings_wisun.device_type == SL_WISUN_LFN) {
    printf("[LFN has no RPL info]\r\n");
    return SL_STATUS_NOT_AVAILABLE;
  }

  ret = sl_wisun_get_rpl_info(&app_rpl_info);
  if (ret != SL_STATUS_OK) {
    printf("[Failed to retrieve Wi-SUN RPL information: %lu]\r\n", ret);
    return SL_STATUS_FAIL;
  }

  iter = app_rpl_entries;
  while (iter->key) {
    if (!strcmp(entry->key, app_info_domain_str[iter->domain])) {
      if (!key_str || !strcmp(iter->key, key_str)) {
        if (iter->get_handler) {
          ret = iter->get_handler(value_str, NULL, iter);
          if (ret == SL_STATUS_OK) {
            printf("%s.%s.%s = %s\r\n", app_settings_domain_str[entry->domain], app_info_domain_str[iter->domain], iter->key, value_str);
          }
        }
      }
    }
    iter++;
  }

  // Prevent parent from printing anything
  return SL_STATUS_FAIL;
}

static sl_status_t app_settings_set_direct_connect_pmk(const char *value_str,
                                                       const char *key_str,
                                                       const app_settings_entry_t *entry)
{
  (void)key_str;

  return app_util_get_byte_array(value_str, entry->value, SL_WISUN_PMK_LEN);
}

static sl_status_t app_settings_get_direct_connect_pmk(char *value_str,
                                                       const char *key_str,
                                                       const app_settings_entry_t *entry)
{
  (void) key_str;

  const uint8_t *pmk = (uint8_t *) entry->value;
  int offset = 0;

  for (int i = 0; i < SL_WISUN_PMK_LEN; i++) {
    offset += sprintf(value_str + offset, "%02x", pmk[i]);
    if (i + 1 < SL_WISUN_PMK_LEN) {
      offset += sprintf(value_str + offset, ":");
    }
  }
  return SL_STATUS_OK;
}

static sl_status_t app_settings_set_tx_power(const char *value_str,
                                             const char *key_str,
                                             const app_settings_entry_t *entry)
{
  (void)key_str;
  sl_status_t ret;
  uint32_t value;

  ret = app_util_get_integer(&value,
                             value_str,
                             entry->input_enum_list,
                             entry->input & APP_SETTINGS_INPUT_FLAG_SIGNED);

  if (ret == SL_STATUS_OK) {
    // If entry is in dBm, multiply by 10 to have ddBm
    if (entry->value_size == APP_SETTINGS_VALUE_SIZE_UINT8) {
      value = value * 10;
    }
    app_settings_wisun.tx_power_ddbm = (int16_t)value;
  }

  return ret;
}

static sl_status_t app_settings_get_tx_power(char *value_str,
                                             const char *key_str,
                                             const app_settings_entry_t *entry)
{
  sl_status_t status = SL_STATUS_OK;
  int16_t tx_power_ddbm;
  (void)key_str;

  tx_power_ddbm = app_settings_wisun.tx_power_ddbm;
  if (entry->value_size == APP_SETTINGS_VALUE_SIZE_UINT8) {
    // If entry is in dBm, print in float format
    sprintf(value_str, "%d.%d",
            tx_power_ddbm / 10,
            abs(tx_power_ddbm % 10));
  } else if (entry->value_size == APP_SETTINGS_VALUE_SIZE_UINT16) {
    sprintf(value_str, "%d", tx_power_ddbm);
  } else {
    status = SL_STATUS_FAIL;
  }

  return status;
}

static sl_status_t app_settings_set_preferred_pan(const char *value_str,
                                                  const char *key_str,
                                                  const app_settings_entry_t *entry)
{
  (void)key_str;
  sl_status_t ret;
  sl_wisun_join_state_t join_state;
  uint32_t value;

  ret = app_util_get_integer(&value,
                             value_str,
                             entry->input_enum_list,
                             entry->input & APP_SETTINGS_INPUT_FLAG_SIGNED);

  if (ret == SL_STATUS_OK) {
    app_settings_wisun.preferred_pan_id = (uint16_t)value;
  } else {
    return ret;
  }

  ret = sl_wisun_get_join_state(&join_state);
  if (ret != SL_STATUS_OK) {
    return ret;
  }

  if (join_state != SL_WISUN_JOIN_STATE_DISCONNECTED && join_state != SL_WISUN_JOIN_STATE_DISCONNECTING) {
    ret = sl_wisun_set_preferred_pan(app_settings_wisun.preferred_pan_id);
    if (ret == SL_STATUS_OK) {
      printf("[Preferred PAN set]\r\n");
    } else {
      printf("[Failed: unable to set preferred PAN: %lu]\r\n", ret);
    }
  }

  return ret;
}
