/**
 * @file
 * @brief Application Timer module
 *
 * @copyright 2018 Silicon Laboratories Inc.
 */

/* Z-Wave includes */
#include <AppTimer.h>
#include <SwTimerLiaison.h>
#include <ZW_system_startup_api.h>
#include <zpal_retention_register.h>
#include "zpal_log.h"

#include <assert.h>
#include <FreeRTOS.h>
#include <task.h>

/**
 * First (zero based) retention register to use for persisting application
 * timers during Deep Sleep. Other retention registers used for Deep Sleep persistent
 * app timers are defined as offsets from this value.
 */
#define FIRST_APP_TIMER_RETENTION_REGISTER        (ZPAL_RETENTION_REGISTER_PROTOCOL_RESERVED_COUNT + 0)

/** Retention register to use for persisting the task tick value at power down */
#define TASKTICK_AT_POWERDOWN_RETENTION_REGISTER  (FIRST_APP_TIMER_RETENTION_REGISTER + 0)

/**
 * Retention register to use for persisting the task tick value when the timer
 * values are saved to retention registers
 */
#define TASKTICK_AT_SAVETIMERS_RETENTION_REGISTER (FIRST_APP_TIMER_RETENTION_REGISTER + 1)

/**
 * First retention register to use for persisting the Deep Sleep persistent application
 * timers during Deep Sleep. (actual number of registers used is determined by
 * how many times AppTimerDeepSleepPersistentRegister() is called).
 */
#define TIMER_VALUES_BEGIN_RETENTION_REGISTER     (FIRST_APP_TIMER_RETENTION_REGISTER + 2)

/**
 * We have 8 retention registers allocated for ZAF/App timers.
 * 6 for persistent timers and 2 are for admin purpose.
 */
_Static_assert(APP_TIMER_RETENTION_REGISTER_RESERVED_COUNT == 8, "STATIC_ASSERT_FAILED_retention_register_count");

/**
 * On wakeup from Deep Sleep, if the difference between expected timeout of an
 * Deep Sleep persistent application timer and the elapsed time at wake up, is
 * smaller than this value then the timer callback will be activated.
 */
#define APP_TIMER_TRIGGER_DELTA_MS 10

// Using AppTimer singleton
extern SAppTimer g_AppTimer;
// Using state variable from AppTimer
extern bool g_deepSleepTimersLoaded;

/* This function will be called in the correct task context */
void AppTimerDeepSleepCallbackWrapper(SSwTimer* pTimer)
{
  ZPAL_LOG_DEBUG(ZPAL_LOG_ZAF_APP_TIMER, "AppTimerDeepSleepCallbackWrapper timerId=%d\n", pTimer->Id);
  AppTimerDeepSleepPersistentSaveAll();

  assert(g_AppTimer.DeepSleepPersistent[pTimer->Id] && g_AppTimer.pDeepSleepCallback[pTimer->Id]);

  if (g_AppTimer.pDeepSleepCallback[pTimer->Id]) {
    ZPAL_LOG_DEBUG(ZPAL_LOG_ZAF_APP_TIMER, "Calling g_AppTimer.pDeepSleepCallback[%d] = %p\n", pTimer->Id, g_AppTimer.pDeepSleepCallback[pTimer->Id]);
    (g_AppTimer.pDeepSleepCallback[pTimer->Id])(pTimer);
  }
}

bool AppTimerDeepSleepPersistentRegister(SSwTimer* pTimer,
                                         bool bAutoReload,
                                         void(*pCallback)(SSwTimer* pTimer))
{
  /* We don't support auto reload of Deep Sleep persistent timers (at least it has
   * not been tested - it might actually work now) */
  assert(false == bAutoReload);

  /* Check that we have a retention register available for this new persistent timer */
  uint32_t count = 0;
  for (uint32_t timerId = 0; timerId < MAX_NUM_APP_TIMERS; timerId++) {
    if (true == g_AppTimer.DeepSleepPersistent[timerId]) {
      count++;
    }
  }
  if (count >= MAX_NUM_PERSISTENT_APP_TIMERS) {
    /* All timer retention registers are taken */
    ZPAL_LOG_DEBUG(ZPAL_LOG_ZAF_APP_TIMER, "AppTimerDeepSleepPersistentRegister: Max number of registrations exceeded (%d)\n", MAX_NUM_PERSISTENT_APP_TIMERS);
    return false;
  }

  /* We register AppTimerDeepSleepCallbackWrapper() as the call back in order to
   * update the timer status in retention registers when the timer expires.
   * The actual callback is saved to g_AppTimer.pDeepSleepCallback and will be
   * called by AppTimerDeepSleepCallbackWrapper() */
  ESwTimerLiaisonStatus status = TimerLiaisonRegister(&g_AppTimer.TimerLiaison,
                                                      pTimer,
                                                      bAutoReload,
                                                      AppTimerDeepSleepCallbackWrapper);

  ZPAL_LOG_DEBUG(ZPAL_LOG_ZAF_APP_TIMER, "AppTimerDeepSleepPersistentRegister() id=%d pCallback=%p\n", pTimer->Id, pCallback);
  if (status == ESWTIMERLIAISON_STATUS_SUCCESS) {
    g_AppTimer.DeepSleepPersistent[pTimer->Id] = true;
    g_AppTimer.pDeepSleepCallback[pTimer->Id]  = pCallback;
    return true;
  }

  return false;
}

ESwTimerStatus AppTimerDeepSleepPersistentStart(SSwTimer* pTimer, uint32_t iTimeout)
{
  ZPAL_LOG_DEBUG(ZPAL_LOG_ZAF_APP_TIMER, "AppTimerDeepSleepPersistentStart() id=%d, timeout=%u\n", pTimer->Id, iTimeout);
  ESwTimerStatus status = TimerStart(pTimer, iTimeout);
  AppTimerDeepSleepPersistentSaveAll();
  return status;
}

ESwTimerStatus AppTimerDeepSleepPersistentRestart(SSwTimer* pTimer)
{
  ZPAL_LOG_DEBUG(ZPAL_LOG_ZAF_APP_TIMER, "AppTimerDeepSleepPersistentRestart() id=%d\n", pTimer->Id);
  ESwTimerStatus status = TimerRestart(pTimer);
  AppTimerDeepSleepPersistentSaveAll();
  return status;
}

ESwTimerStatus AppTimerDeepSleepPersistentStop(SSwTimer* pTimer)
{
  ZPAL_LOG_DEBUG(ZPAL_LOG_ZAF_APP_TIMER, "AppTimerDeepSleepPersistentStop() id=%d\n", pTimer->Id);
  ESwTimerStatus status = TimerStop(pTimer);
  AppTimerDeepSleepPersistentSaveAll();
  return status;
}

/*
 * How the Deep Sleep persistent application timers are saved to RTCC retention registers
 *
 * For example the following list of APPLICATION TIMERS (contained in struct
 * g_AppTimer) is assumed (ordered by timer id)
 *
 * 0 (DeepSleepPersistent = false)
 * 1 (DeepSleepPersistent = false)
 * 2 (DeepSleepPersistent = true) MsUntilTimeout=30000
 * 3 (DeepSleepPersistent = true) MsUntilTimeout=0xFFFFFFFF (not active when saved)
 * 4 (DeepSleepPersistent = false)
 * 5 (DeepSleepPersistent = true) MsUntilTimeout=20000
 * 6 (DeepSleepPersistent = false)
 * 7 (DeepSleepPersistent = false)
 *
 * - DeepSleepPersistent is the flag in g_AppTimer ("true" implies that the
 *   timer should be saved, and it also implies that the timer exist)
 *
 * - MsUntilTimeout is the calculated number of milliseconds remaining
 *   before the timer times out. If equal to 0xFFFFFFFF then the timer is
 *   not active.
 *
 * The RETENTION REGISTERS will only contain the timer values for the Deep Sleep
 * persistent timers plus the task tick values when the timer values were
 * saved and when the device is going to sleep in Deep Sleep:
 *
 * 0 TaskTick at power-down
 * 1 TaskTick at save timers
 * 2 timerValue_ms=30000
 * 3 timerValue_ms=0xFFFFFFFF
 * 4 timerValue_ms=20000
 *
 * When the device wakes up from Deep Sleep the values in the retention registers
 * together with the time spent in Deep Sleep hibernate are used to determine if a
 * timer has expired or what value should be used to start it again to have
 * it time out at the right moment relative to its original start time.
 */
void AppTimerDeepSleepPersistentSaveAll(void)
{
  uint32_t reg = TIMER_VALUES_BEGIN_RETENTION_REGISTER;

  // Don't touch the retention registers until they are loaded
  if (false == g_deepSleepTimersLoaded) {
    return;
  }

  uint32_t taskTickCount = xTaskGetTickCount();
  zpal_retention_register_write(TASKTICK_AT_SAVETIMERS_RETENTION_REGISTER, taskTickCount);

  ZPAL_LOG_DEBUG(ZPAL_LOG_ZAF_APP_TIMER, "AppTimerDeepSleepPersistentSaveAll tick: %u\n", taskTickCount);

  for (uint32_t timerId = 0; timerId < MAX_NUM_APP_TIMERS; timerId++) {
    if (true == g_AppTimer.DeepSleepPersistent[timerId]) {
      SSwTimer *pTimer        = g_AppTimer.aTimerPointerArray[timerId];
      uint32_t  timerValue_ms = UINT32_MAX;

      TimerGetMsUntilTimeout(pTimer, taskTickCount, &timerValue_ms);

      ZPAL_LOG_DEBUG(ZPAL_LOG_ZAF_APP_TIMER, "Saving value for timer %d: %u (0x%x) ms\n", timerId, timerValue_ms, timerValue_ms);

      zpal_retention_register_write(reg, timerValue_ms);
      reg++;
    }
  }
}

void AppTimerDeepSleepPersistentLoadAll(zpal_reset_reason_t resetReason)
{
  uint32_t tickValueAtPowerDown  = 0;
  uint32_t tickValueAtSaveTimers = 0;
  uint8_t  valIdx                = 0;
  uint32_t savedTimerValue       = 0;
  uint32_t elapsedMsFromSaveTimerValuesToSleep = 0;
  uint32_t elapsedMsFromTimerValueSave         = 0;
  uint32_t durationDiffMs = 0;

  g_deepSleepTimersLoaded = true;

  /* Do nothing if we did not wake up from Deep Sleep */
  if (ZPAL_RESET_REASON_DEEP_SLEEP_EXT_INT != resetReason && ZPAL_RESET_REASON_DEEP_SLEEP_WUT != resetReason) {
    // It is safe to persist the registers now
    AppTimerDeepSleepPersistentSaveAll();
    return;
  }

  if (true == IsWakeupCausedByRtccTimeout()) {
    ZPAL_LOG_DEBUG(ZPAL_LOG_ZAF_APP_TIMER, "\nRTCC wakeup!\n");
  }

  /* Read the task tick values saved before sleeping in Deep Sleep */
  zpal_retention_register_read(TASKTICK_AT_POWERDOWN_RETENTION_REGISTER, &tickValueAtPowerDown);
  ZPAL_LOG_DEBUG(ZPAL_LOG_ZAF_APP_TIMER, "Loaded tickValueAtPowerDown: %u\n", tickValueAtPowerDown);

  zpal_retention_register_read(TASKTICK_AT_SAVETIMERS_RETENTION_REGISTER, &tickValueAtSaveTimers);
  ZPAL_LOG_DEBUG(ZPAL_LOG_ZAF_APP_TIMER, "Loaded tickValueAtSaveTimers: %u\n", tickValueAtSaveTimers);

  if ((0 == tickValueAtPowerDown) || (0 == tickValueAtSaveTimers)) {
    /* Retention registers are still at initial value - nothing to process now */
    return;
  }

  /* How many ms before power-down were the timer values saved to
   * retention registers? (NB: one tick = one millisecond) */
  if (tickValueAtPowerDown >= tickValueAtSaveTimers) {
    elapsedMsFromSaveTimerValuesToSleep = tickValueAtPowerDown - tickValueAtSaveTimers;
  } else {
    /* The 32-bit task tick has wrapped around
     * (VERY unlikely for a sleeping node) */
    elapsedMsFromSaveTimerValuesToSleep = (UINT32_MAX - tickValueAtSaveTimers) + tickValueAtPowerDown;
  }

  /* How many ms since the timer values were saved, including the
   * time spent sleeping */
  elapsedMsFromTimerValueSave = elapsedMsFromSaveTimerValuesToSleep
                                + GetCompletedSleepDurationMs();

  ZPAL_LOG_DEBUG(ZPAL_LOG_ZAF_APP_TIMER, "elapsedMsFromTimerValueSave=%u\n", elapsedMsFromTimerValueSave);

  /* Read saved timer values from retention registers into array
   * while looking for smallest value larger than savedBeforePowerdownMs */
  for (uint8_t timerId = 0; timerId < MAX_NUM_APP_TIMERS; timerId++) {
    if (true == g_AppTimer.DeepSleepPersistent[timerId]) {
      if (ZPAL_STATUS_OK == zpal_retention_register_read(TIMER_VALUES_BEGIN_RETENTION_REGISTER + valIdx, &savedTimerValue)) {
        SSwTimer *pTimer = g_AppTimer.aTimerPointerArray[timerId];

        ZPAL_LOG_DEBUG(ZPAL_LOG_ZAF_APP_TIMER, "Loaded value for timer %d: %u ms\n", timerId, savedTimerValue);

        /* We don't expect any timer to expire before power down without being restarted */
        assert(savedTimerValue > elapsedMsFromSaveTimerValuesToSleep);

        /* How close is the timer to its expire time? */
        if (savedTimerValue > elapsedMsFromTimerValueSave) {
          durationDiffMs = savedTimerValue - elapsedMsFromTimerValueSave;
        } else {
          //Time is past expiration. Make sure the callback is called.
          durationDiffMs = 0;
        }

        ZPAL_LOG_DEBUG(ZPAL_LOG_ZAF_APP_TIMER, "durationDiffMs=%u\n", durationDiffMs);

        /* If the timer timeout is within APP_TIMER_TRIGGER_DELTA_MS milliseconds
         * of the wakeup time - no matter if the wakeup was caused by an RTCC
         * timeout or e.g. pin interrupt - we simply activate its callback now.
         * (A 10 ms difference means the timeout is so close to the wake-up event
         * that the timer could run out before the task tick and scheduler is
         * started. In any case, for fast repeating timers (if any), we would
         * never get here anyway since we only enter Deep Sleep hibernate if we are
         * expected to sleep for at least 4000 ms (see enterPowerDown())
         */
        if (durationDiffMs < APP_TIMER_TRIGGER_DELTA_MS) {
          ZPAL_LOG_DEBUG(ZPAL_LOG_ZAF_APP_TIMER, "Timer %d has expired. Activating callback.\n", timerId);

          /* Activate the callback for the expired timer (for Deep Sleep persistent timer
           * this will call the wrapper AppTimerDeepSleepCallcackWrapper that will call
           * AppTimerDeepSleepPersistentSaveAll and the actual callback)
           */
          TimerLiaisonExpiredTimerCallback(pTimer);
        } else {
          /* Reduce all saved timer values (for active timers that have not yet expired) by
           * number of milliseconds elapsed since it was last saved. Then start the timer
           * with this new value.
           */
          if ((UINT32_MAX != savedTimerValue) && (elapsedMsFromTimerValueSave < savedTimerValue)) {
            uint32_t newTimerValue = savedTimerValue - elapsedMsFromTimerValueSave;
            ZPAL_LOG_DEBUG(ZPAL_LOG_ZAF_APP_TIMER, "Setting timer %d to %u ms\n", timerId, newTimerValue);
            /* We call TimerStart() here instead of AppTimerDeepSleepPersistentStart()
             * to avoid AppTimerDeepSleepPersistentSaveAll() being called multiple
             * times. Instead we call AppTimerDeepSleepPersistentSaveAll() once for
             * all (if needed) outside the loop */
            __attribute__((unused)) ESwTimerStatus timerStatus = TimerStart(pTimer, newTimerValue);
            assert(ESWTIMER_STATUS_SUCCESS == timerStatus);
          }
        }
      }
      valIdx++; // Only increment when we have processed an Deep Sleep persistent timer
    }
  }

  // Always update the retention registers when done loading
  AppTimerDeepSleepPersistentSaveAll();
}

uint32_t AppTimerDeepSleepGetFirstRetentionRegister(void)
{
  return FIRST_APP_TIMER_RETENTION_REGISTER;
}

uint32_t AppTimerDeepSleepGetLastRetentionRegister(void)
{
  uint32_t count = 0;
  for (uint32_t timerId = 0; timerId < MAX_NUM_APP_TIMERS; timerId++) {
    if (true == g_AppTimer.DeepSleepPersistent[timerId]) {
      count++;
    }
  }
  return TIMER_VALUES_BEGIN_RETENTION_REGISTER + count - 1;
}

void AppTimerDeepSleepPersistentResetStorage(void)
{
  uint32_t first       = AppTimerDeepSleepGetFirstRetentionRegister();
  uint32_t last        = AppTimerDeepSleepGetLastRetentionRegister();

  ZPAL_LOG_DEBUG(ZPAL_LOG_ZAF_APP_TIMER, "\nResetDeepSleepPersistentAppTimerStorage first=%u, last=%ux\n", first, last);

  assert(first < last);
  for (uint32_t reg = first; reg <= last; reg++) {
    zpal_retention_register_write(reg, 0);
  }
}

// Weak function from protocol
// Overwriting it at compile time instead of passing it on runtime
void ZW_AppPowerDownCallBack(void)
{
  uint32_t taskTickCount = xTaskGetTickCount();

  /* Called while the scheduler is disabled just before being forced into
   * Deep Sleep hibernate. If printing to serial line we need to delay the function
   * return to allow for the serial buffer content to be flushed */
#if defined(ZPAL_LOG_ZAF_APP_TIMER)
  ZPAL_LOG_DEBUG(ZPAL_LOG_ZAF_APP_TIMER, "Saving task tick: %u\n", taskTickCount);
  for (int i = 0; i < 2000; i++) {
    __asm__ ("nop");                          // Allow the serial line to flush before sleeping
  }
#endif

  zpal_retention_register_write(TASKTICK_AT_POWERDOWN_RETENTION_REGISTER, taskTickCount);
}
