/**
 * @file
 * @brief Credential Learn module for Command Class User Credential.
 *
 * @details The Credential Learn module provides functions to start and cancel
 * the local learning process for credentials.
 *
 * @copyright 2023 Silicon Laboratories Inc.
 */

#ifndef CC_USER_CREDENTIAL_LEARN_H
#define CC_USER_CREDENTIAL_LEARN_H

#include "CC_UserCredential.h"
#include "zaf_transport_tx.h"
#include "zaf_event_distributor_soc.h"
#include "AppTimer.h"
#include "zpal_power_manager.h"
#include "zw_power_manager_ids.h"

/**
 * @return true if a Credential Learn operation is in currently progress
 */
bool is_credential_learn_in_progress();

/**
 * @brief Cancels the learning process for credentials.
 *
 * @details This function is responsible for canceling the learning process for
 * credentials.
 * It is called when the learning process needs to be stopped or aborted.
 */
void credential_learn_reset();

/**
 * @brief Called upon receiving a Credential Learn Start frame.
 *
 * @details Checks and processes a request to initiate the Credential Learn
 * process, which can either add or replace a credential based on the input
 * data from the local device.
 *
 * @param[in] operation_type  The type of operation to be performed at the end of the learning process.
 * @param[in] p_learn_target  Pointer to the identifying data of the targeted
 *                            credential.
 * @param[in] timeout_seconds The maximum allowed duration of the learning process.
 * @param[in] p_rx_options    Pointer to the structure containing the received frame 's details.
 * @return The result of the attempt to start the learn process.
 */
received_frame_status_t CC_UserCredential_CredentialLearnStart_handler(
  const u3c_operation_type_t operation_type,
  const u3c_credential_identifier_t * const p_learn_target,
  const uint8_t timeout_seconds,
  RECEIVE_OPTIONS_TYPE_EX * p_rx_options);

/**
 * @brief Called upon receiving a Credential Learn Cancel frame.
 *
 * @details Processes a request to cancel an ongoing Credential Learn process.
 *
 * @param[in] input  Pointer to the structure containing the input data for the handler.
 * @return The result of the attempt to cancel the learn process.
 */
received_frame_status_t CC_UserCredential_CredentialLearnCancel_handler(
  const cc_handler_input_t * const input);

/**
 * @brief Handles events related to the user credential learning process.
 *
 * @details Processes various events generated by the application during the credential learning process.
 *
 * @param[in] event  The event identifier.
 * @param[in] p_data Pointer to the data associated with the event.
 *
 */
void CC_UserCredential_learn_event_handler(
  const uint8_t event, const void * p_data);

#endif // CC_USER_CREDENTIAL_LEARN_H
