/***************************************************************************//**
 * @file
 * @brief Routines for the Comms Hub Function plugin.
 *******************************************************************************
 * # License
 * <b>Copyright 2018 Silicon Laboratories Inc. www.silabs.com</b>
 *******************************************************************************
 *
 * The licensor of this software is Silicon Laboratories Inc. Your use of this
 * software is governed by the terms of Silicon Labs Master Software License
 * Agreement (MSLA) available at
 * www.silabs.com/about-us/legal/master-software-license-agreement. This
 * software is distributed to you in Source Code format and is governed by the
 * sections of the MSLA applicable to Source Code.
 *
 ******************************************************************************/

#include "app/framework/include/af.h"
#include "app/framework/util/af-main.h"
#include "app/framework/util/common.h"
#include "app/framework/plugin/gbcs-device-log/gbcs-device-log.h"
#include "app/framework/plugin/meter-mirror/meter-mirror.h"
#include "app/framework/plugin/sleepy-message-queue/sleepy-message-queue.h"
#include "app/framework/plugin/events-server/events-server.h"
#include "comms-hub-function.h"
#include "comms-hub-tunnel-endpoints.h"
#include "tunnel-manager.h"
#include "zigbee-security-manager.h"

// default sleep message timeout is 24 hours
#define DEFAULT_SLEEPY_MSG_TIMEOUT_SEC (60 * 60 * 24)
static uint32_t defaultMessageTimeout = DEFAULT_SLEEPY_MSG_TIMEOUT_SEC;

// Tunnel Manager Header values per GNBCS spec
#define TUNNEL_MANAGER_HEADER_GET          0x01
#define TUNNEL_MANAGER_HEADER_GET_RESPONSE 0x02
#define TUNNEL_MANAGER_HEADER_PUT          0x03

static uint8_t currentDeviceLogEntry = 0;
static bool discoveryInProgress = false;

static const uint8_t PLUGIN_NAME[] = "CommsHubFunction";

static sl_802154_short_addr_t tunnelTargetNodeId = SL_ZIGBEE_NULL_NODE_ID;
static uint8_t tunnelTargetAttempts = 0;

// Unfortunately both the attempt to do node ID discovery and Match descriptor
// are each considered separate "attempts" and thus reduce the number of total attempts,
// even in the case that both succeed without any issues.
// However differentiating between node ID discovery attempts (to reset this back to zero)
// and keep track of Match Descriptor events separates is complicated and ultimately less important.
// So we set the attempts high enough such that in the succesful case we don't hit the max attempts,
// one or two failures won't hit the limit.  But repeated failures will (for example, a node offline
// and unresponsive).
#define MAX_TUNNEL_TARGET_ATTEMPTS 5

sl_zigbee_af_event_t sl_zigbee_af_comms_hub_function_tunnel_check_event;
#define tunnelCheckEventControl (&sl_zigbee_af_comms_hub_function_tunnel_check_event)
static void tunnelCheckEventHandler(sl_zigbee_af_event_t * event);

#define PLUGIN_DEBUG
#if defined(PLUGIN_DEBUG)
#define pluginDebugPrint(...)   sl_zigbee_af_core_print(__VA_ARGS__)
#define pluginDebugPrintln(...) sl_zigbee_af_core_println(__VA_ARGS__)
#define pluginDebugExec(x)      (x)
#define pluginDebugPrintBuffer(buffer, len, withSpace) sl_zigbee_af_print_buffer(SL_ZIGBEE_AF_PRINT_CORE, (buffer), (len), (withSpace))
#else
#define pluginDebugPrint(...)
#define pluginDebugPrintln(...)
#define pluginDebugExec(x)
#define pluginDebugPrintBuffer(x, y, z)
#endif

// As long as this plugin requires the sleepy-message-queue plugin, this #define
// will exist.
static uint16_t sleepyMessageUseCaseCodes[SL_ZIGBEE_AF_PLUGIN_SLEEPY_MESSAGE_QUEUE_SLEEPY_QUEUE_SIZE];

//------------------------------------------------------------------------------
// Forward Declarations

static sl_zigbee_af_plugin_comms_hub_function_status_t setTunnelMessagePending(sl_802154_long_addr_t deviceId);
static sl_zigbee_af_plugin_comms_hub_function_status_t clearTunnelMessagePending(sl_802154_long_addr_t deviceId);
static void tunnelDiscoveryCallback(const sl_zigbee_af_service_discovery_result_t *result);
static void initiateDiscovery(sl_802154_short_addr_t nodeId, sl_802154_long_addr_t deviceEui64);
static void checkForAnyDeviceThatNeedsTunnelCreated(void);
static bool checkForSpecificDeviceThatNeedsTunnelCreated(sl_802154_short_addr_t nodeId,
                                                         sl_802154_long_addr_t deviceEui64);

//------------------------------------------------------------------------------
// API functions

void sl_zigbee_af_comms_hub_function_init_cb(uint8_t init_level)
{
  switch (init_level) {
    case SL_ZIGBEE_INIT_LEVEL_EVENT:
    {
      sl_zigbee_af_network_event_init(&sl_zigbee_af_comms_hub_function_tunnel_check_event,
                                      tunnelCheckEventHandler);
      break;
    }

    case SL_ZIGBEE_INIT_LEVEL_LOCAL_DATA:
    {
      discoveryInProgress = false;
      break;
    }
  }
}

sl_zigbee_af_plugin_comms_hub_function_status_t sl_zigbee_af_comms_hub_function_send(sl_802154_long_addr_t destinationDeviceId,
                                                                                     uint16_t length,
                                                                                     uint8_t *payload,
                                                                                     uint16_t messageCode)
{
  sl_zigbee_af_g_b_c_s_device_log_info_t deviceInfo;
  sl_zigbee_af_sleepy_message_t sleepyMessage;
  sl_zigbee_af_sleepy_message_id_t sleepyMessageId;
  sl_zigbee_af_plugin_comms_hub_function_status_t status;
  sl_zigbee_af_zcl_event_t event;
  uint8_t * eventData = event.eventData;
  bool isSleepyDevice = false;

  eventData[0] = 0x00;

  //  Check to make sure the destination device is in the device log.
  if (!sl_zigbee_af_gbcs_device_log_get(destinationDeviceId, &deviceInfo)) {
    char * msg = "CHF: Given destination device ID has not been added to the GBCS device log";

    sl_zigbee_af_comms_hub_function_println(msg);
    status = SL_ZIGBEE_AF_CHF_STATUS_NO_ACCESS;
    goto kickout;
  }

  // If it's a sleepy device then we must queue the message for later delivery
  // using the GET, GET_RESPONSE, PUT protocol as defined in the GBCS spec.
  if (sl_zigbee_af_gbcs_device_log_is_sleepy_type(deviceInfo.deviceType)) {
    sl_zigbee_af_plugin_comms_hub_function_status_t chfStatus;
    isSleepyDevice = true;

    // set mirror notification flags and queue the data for the sleepy device
    chfStatus = setTunnelMessagePending(destinationDeviceId);
    if (chfStatus != SL_ZIGBEE_AF_CHF_STATUS_SUCCESS) {
      char * msg = "CHF: Unable to set mirror notification flags and queue the data for the sleepy device ";

      sl_zigbee_af_comms_hub_function_println(msg);
      status = chfStatus;
      goto kickout;
    }

    memcpy(sleepyMessage.dstEui64, destinationDeviceId, EUI64_SIZE);
    sleepyMessage.length = length;
    sleepyMessage.payload = payload;
    sleepyMessageId = sl_zigbee_af_sleepy_message_queue_store_message(&sleepyMessage, defaultMessageTimeout);
    if (sleepyMessageId == SL_ZIGBEE_AF_PLUGIN_SLEEPY_MESSAGE_INVALID_ID) {
      char * msg = "CHF: Unable to add message to sleepy message queue ";

      sl_zigbee_af_comms_hub_function_println(msg);
      status = SL_ZIGBEE_AF_CHF_STATUS_TOO_MANY_PEND_MESSAGES;
      goto kickout;
    }

    sleepyMessageUseCaseCodes[sleepyMessageId] = messageCode;
    status = SL_ZIGBEE_AF_CHF_STATUS_SUCCESS;
  } else {
    isSleepyDevice = false;

    // Not a sleepy device so send the message now.
    if (sli_zigbee_af_comms_hub_function_tunnel_send_data(destinationDeviceId, 0, NULL, length, payload)) {
      status = SL_ZIGBEE_AF_CHF_STATUS_SUCCESS;
    } else {
      char * msg = "CHF: Unable to send the message through tunnel to destination";
      status = SL_ZIGBEE_AF_CHF_STATUS_TUNNEL_FAILURE;

      sl_zigbee_af_comms_hub_function_println(msg);
    }

    sl_zigbee_af_comms_hub_function_send_cb(status,
                                            destinationDeviceId,
                                            length,
                                            payload);
  }

  kickout:
  if ((status == SL_ZIGBEE_AF_CHF_STATUS_SUCCESS)
      && (isSleepyDevice)) {
    // msg is queued for later delivery.
    // can't really determine if msg is sent/actioned or not yet.
  } else {
    if (status == SL_ZIGBEE_AF_CHF_STATUS_SUCCESS) {
      event.eventId = GBCS_EVENT_ID_IMM_HAN_CMD_RXED_ACTED;
    } else {
      event.eventId = GBCS_EVENT_ID_IMM_HAN_CMD_RXED_NOT_ACTED;
    }
    // EMAPPFWKV2-1315 - "For any Event Log entries relating to Event Codes
    // 0x0054 and 0x0055, the Device shall record the Commands received on the
    // Network Interface by including the Message Code in the Event Log"
    eventData[0] = 0x02;
    eventData[1] = HIGH_BYTE(messageCode);
    eventData[2] = LOW_BYTE(messageCode);

    event.eventTime = sl_zigbee_af_get_current_time();
    sl_zigbee_af_events_server_add_event(SL_ZIGBEE_AF_PLUGIN_GAS_PROXY_FUNCTION_REMOTE_COMMSHUB_ENDPOINT,
                                         SL_ZIGBEE_ZCL_EVENT_LOG_ID_GENERAL_EVENT_LOG,
                                         &event);
  }
  return status;
}

void sli_zigbee_af_comms_hub_function_set_default_timeout(uint32_t timeout)
{
  defaultMessageTimeout = timeout;
}

//------------------------------------------------------------------------------
// Callback Functions

#ifdef EZSP_HOST
void sli_zigbee_af_comms_hub_function_trust_center_join_callback(sl_802154_short_addr_t newNodeId,
                                                                 sl_802154_long_addr_t newNodeEui64,
                                                                 sl_zigbee_device_update_t status,
                                                                 sl_zigbee_join_decision_t decision,
                                                                 sl_802154_short_addr_t parentOfNewNode)
#else // !EZSP_HOST
void sli_zigbee_af_comms_hub_function_trust_center_join_callback(sl_802154_short_addr_t newNodeId,
                                                                 sl_802154_long_addr_t newNodeEui64,
                                                                 sl_zigbee_device_update_t status,
                                                                 sl_802154_short_addr_t parentOfNewNode)
#endif // EZSP_HOST
{
  pluginDebugPrint("%s: TrustCenterJoin 0x%04X ", PLUGIN_NAME, newNodeId);
  pluginDebugExec(sl_zigbee_af_print_big_endian_eui64(newNodeEui64));
#ifdef EZSP_HOST
  pluginDebugPrintln(" 0x%04X 0x%02X 0x%02X", parentOfNewNode, status, decision);
#else // !EZSP_HOST
  pluginDebugPrintln(" 0x%04X 0x%02X", parentOfNewNode, status);
#endif // EZSP_HOST

  // If a device is leaving or rejoining the trust center we may have knowledge of a
  // tunnel previous established with that device.  If so remove all knowledge
  // of that tunnel because it is no longer valid.
  sli_zigbee_af_comms_hub_function_tunnel_cleanup(newNodeEui64);

  if (SL_ZIGBEE_DEVICE_LEFT != status) {
    // If the device did a first time join, then it will not have
    // done key establishment yet and so attempting to initiate a tunnel
    // to the device is premature.

    // Our event handler will periodically kick off and see if the device
    // is done with key establishment and thus will initiate a tunnel to
    // it.

    // If the device did a rejoin then we could try to initiate a tunnel
    // to the device immediately, but our event will periodically kick off to do that.
    // More importantly we can't call checkForSpecificDeviceThatNeedsTunnelCreated()
    // now because our currentDeviceLogEntry variable won't necessarily correspond to the
    // device that is doing a rejoin here.  Better let the event fire later.

    // Cleanest thing to do is assume the tunnel needs to be re-created.
    // Although the local device may have a tunnel up and running fine,
    // the remote device (e.g. ESME) may have forgotten its tunnel.
  }
}

/** @brief Key Establishment
 *
 * A callback by the key-establishment code to indicate an event has occurred.
 * For error codes this is purely a notification.  For non-error status codes
 * (besides LINK_KEY_ESTABLISHED), it is the application's chance to allow or
 * disallow the operation.  If the application returns true then the key
 * establishment is allowed to proceed.  If it returns false, then key
 * establishment is aborted.  LINK_KEY_ESTABLISHED is a notification of success.
 *
 * @param status   Ver.: always
 * @param amInitiator   Ver.: always
 * @param partnerShortId   Ver.: always
 * @param delayInSeconds   Ver.: always
 */
bool sl_zigbee_af_key_establishment_event_cb(sl_zigbee_af_key_establishment_notify_message_t status,
                                             bool amInitiator,
                                             sl_802154_short_addr_t partnerShortId,
                                             uint8_t delayInSeconds)
{
  sl_802154_long_addr_t partnerEui;
  sl_zigbee_af_g_b_c_s_device_log_info_t deviceInfo;

  pluginDebugPrintln("%s: KeyEstablishmentEventCallback 0x%02X, 0x%02X, 0x%04X, 0x%02X",
                     PLUGIN_NAME,
                     status,
                     amInitiator,
                     partnerShortId,
                     delayInSeconds);

  /*
   * As defined in section 10.2.2.1 of the GBCS version 0.8:
   *
   * "When a Communications Hub has successfully established a shared secret key
   * using CBKE with a Device of type ESME, HCALCS or PPMID, the CHF shall send a
   * RequestTunnel command to the Device to request a tunnel association with the
   * Device.
   */

  if (status == LINK_KEY_ESTABLISHED) {
    if (SL_STATUS_OK != sl_zigbee_lookup_eui64_by_node_id(partnerShortId, partnerEui)) {
      // We're only giving the tunnel creation a best effort at link key establishment
      // stage.  If it fails for any reason we'll just wait until the host app
      // attempts to send data to try the tunnel creation again.
      return true;
    }

    if (false == sl_zigbee_af_gbcs_device_log_get(partnerEui, &deviceInfo)) {
      // This is not one the devices in the device log so no need to bring up a
      // tunnel to this device.
      return true;
    }

    // If this is not a sleepy device then bring up the tunnel.
    if (!sl_zigbee_af_gbcs_device_log_is_sleepy_type(deviceInfo.deviceType)) {
      // We'll look for all devices that support the Tunneling cluster. Once a device
      // responds we'll add their information to the endpoint table.
      sl_zigbee_af_find_devices_by_profile_and_cluster(partnerShortId,
                                                       SE_PROFILE_ID,
                                                       ZCL_TUNNELING_CLUSTER_ID,
                                                       SL_ZIGBEE_AF_SERVER_CLUSTER_DISCOVERY,
                                                       tunnelDiscoveryCallback);
    }
  }

  // Always allow key establishment to continue.
  return true;
}

/** @brief tunnelDiscoveryCallback
 *
 * This function is called when device endpoint discovery completes.
 * It adds the device and all endpoints that support the Tunneling cluster to the list.
 *
 * @param result Contains the list of discovered endpoints.
 */
static void tunnelDiscoveryCallback(const sl_zigbee_af_service_discovery_result_t *result)
{
  pluginDebugPrintln("%s: Discovery callback, cluster 0x%04X, status:0x%02X",
                     PLUGIN_NAME,
                     result->zdoRequestClusterId,
                     result->status);
  if (result->status != SL_ZIGBEE_AF_BROADCAST_SERVICE_DISCOVERY_RESPONSE_RECEIVED) {
    discoveryInProgress = false;
  }

  if (result->zdoRequestClusterId == MATCH_DESCRIPTORS_REQUEST) {
    const sl_zigbee_af_endpoint_list_t* epList;
    sl_802154_long_addr_t eui64;
    sl_status_t status;

    if (result->status == SL_ZIGBEE_AF_UNICAST_SERVICE_DISCOVERY_COMPLETE_WITH_RESPONSE) {
      epList = (const sl_zigbee_af_endpoint_list_t*)result->responseData;
      status = sl_zigbee_lookup_eui64_by_node_id(result->matchAddress, eui64);
      if ( (status == SL_STATUS_OK) && (epList->count >= 1) ) {
        sl_zigbee_af_add_tunneling_endpoint(result->matchAddress, (uint8_t *)epList->list, epList->count);
        sli_zigbee_af_comms_hub_function_tunnel_create(eui64, epList->list[0]);
      } else {
        // Failed to store endpoint.  Try with default endpoint.
        sl_zigbee_af_comms_hub_function_println("Error: Failure to find address or endpoint, status=0x%02X, nodeId=0x%04X, epCount=%d",
                                                status,
                                                result->matchAddress,
                                                epList->count);
      }
    }
  } else if (result->zdoRequestClusterId == NETWORK_ADDRESS_REQUEST) {
    if (result->status == SL_ZIGBEE_AF_BROADCAST_SERVICE_DISCOVERY_COMPLETE_WITH_RESPONSE
        || result->status == SL_ZIGBEE_AF_BROADCAST_SERVICE_DISCOVERY_COMPLETE) {
      pluginDebugPrintln("%s: Broadcast node ID discovery complete.", PLUGIN_NAME);
    }

    if (result->status == SL_ZIGBEE_AF_BROADCAST_SERVICE_DISCOVERY_COMPLETE_WITH_RESPONSE) {
      sl_802154_long_addr_t deviceEui64;
      sl_zigbee_af_g_b_c_s_device_log_info_t deviceInfo;
      tunnelTargetNodeId = result->matchAddress;
      pluginDebugPrintln("%s: Recorded node ID for 0x%04X", PLUGIN_NAME, tunnelTargetNodeId);
      if (tunnelTargetNodeId != SL_ZIGBEE_NULL_NODE_ID
          && sl_zigbee_af_gbcs_device_log_retrieve_by_index(currentDeviceLogEntry,
                                                            deviceEui64,
                                                            &deviceInfo)) {
        sl_zigbee_af_event_set_active(tunnelCheckEventControl);
      }
    }
  }
}

/*
 * @brief Logging timed out message to CHF Event Log.
 */
static void log_timed_out_message_event(sl_zigbee_af_sleepy_message_t * sleepyMessage)
{
#if defined(ZCL_USING_EVENTS_CLUSTER_SERVER)
  sl_zigbee_af_zcl_event_t event;

  event.eventData[0] = 0x00;
  event.eventTime = sl_zigbee_af_get_current_time();
  event.eventId = GBCS_EVENT_ID_GSME_CMD_NOT_RETRVD;

  pluginDebugPrintln("CHF: Adding timed out message to CHF Event Log");
  sl_zigbee_af_events_server_add_event(SL_ZIGBEE_AF_PLUGIN_GAS_PROXY_FUNCTION_REMOTE_COMMSHUB_ENDPOINT,
                                       SL_ZIGBEE_ZCL_EVENT_LOG_ID_GENERAL_EVENT_LOG,
                                       &event);
#endif
}

/** @brief Message Timed Out
 *
 * This function is called by the sleepy message queue when a message times out.
 *  The plugin will invalidate the entry in the queue after giving the
 * application a chance to perform any actions on the timed-out message.
 *
 * @param sleepyMsgId   Ver.: always
 */
void sl_zigbee_af_sleepy_message_queue_message_timed_out_cb(sl_zigbee_af_sleepy_message_id_t sleepyMessageId)
{
  sl_zigbee_af_sleepy_message_t sleepyMessage;

  if (sl_zigbee_af_sleepy_message_queue_get_pending_message(sleepyMessageId, &sleepyMessage)) {
    sl_zigbee_af_sleepy_message_queue_remove_message(sleepyMessageId);
    log_timed_out_message_event(&sleepyMessage);
    sl_zigbee_af_comms_hub_function_send_cb(SL_ZIGBEE_AF_CHF_STATUS_SEND_TIMEOUT,
                                            sleepyMessage.dstEui64,
                                            sleepyMessage.length,
                                            sleepyMessage.payload);
  }
}

/** @brief Device Removed
 *
 * This callback is called by the plugin when a device is removed from the
 * device log.
 *
 * @param deviceId Identifier of the device removed  Ver.: always
 */
void sl_zigbee_af_gbcs_device_log_device_removed_cb(sl_802154_long_addr_t deviceId)
{
  sl_zigbee_af_sleepy_message_t sleepyMessage;
  sl_zigbee_af_sleepy_message_id_t sleepyMessageId = sl_zigbee_af_sleepy_message_queue_get_pending_message_id(deviceId);

  pluginDebugPrint("CHF: DeviceLogDeviceRemoved ");
  pluginDebugExec(sl_zigbee_af_print_big_endian_eui64(deviceId));
  pluginDebugPrintln("");

  // Remove all pending messages to the device, reset the functional notification flags
  // attribute, and teardown the tunnel associated with the device.
  while (sleepyMessageId != SL_ZIGBEE_AF_PLUGIN_SLEEPY_MESSAGE_INVALID_ID) {
    sl_zigbee_af_sleepy_message_queue_get_pending_message(sleepyMessageId, &sleepyMessage);
    sl_zigbee_af_sleepy_message_queue_remove_message(sleepyMessageId);
    sl_zigbee_af_comms_hub_function_send_cb(SL_ZIGBEE_AF_CHF_STATUS_NO_ACCESS,
                                            sleepyMessage.dstEui64,
                                            sleepyMessage.length,
                                            sleepyMessage.payload);
    sleepyMessageId = sl_zigbee_af_sleepy_message_queue_get_pending_message_id(deviceId);
  }
  clearTunnelMessagePending(deviceId);
  sli_zigbee_af_comms_hub_function_tunnel_destroy(deviceId);
}

/**
 * @brief Tunnel Accept
 *
 * This callback is called by the tunnel manager when a tunnel is requested. The
 * given device identifier should be checked against the Device Log to verify
 * whether tunnels from the device should be accepted or not.
 *
 * @param deviceId Identifier of the device from which a tunnel is requested
 * @return true is the tunnel should be allowed, false otherwise
 */
bool sli_zigbee_af_comms_hub_function_tunnel_accept_callback(sl_802154_long_addr_t deviceId)
{
  sl_zigbee_af_g_b_c_s_device_log_info_t deviceInfo;

  pluginDebugPrint("CHF: TunnelAccept ");
  pluginDebugExec(sl_zigbee_af_print_big_endian_eui64(deviceId));
  pluginDebugPrintln("");

  return sl_zigbee_af_gbcs_device_log_get(deviceId, &deviceInfo);
}

/** @brief Tunnel Data Received
 *
 * This callback is called by the tunnel manager when data is received over a tunnel.
 *
 * @param senderDeviceId Identifier of the device from which the data was received
 * @param length The length of the data received
 * @param payload The data received
 */
void sli_zigbee_af_comms_hub_function_tunnel_data_received_callback(sl_802154_long_addr_t senderDeviceId,
                                                                    uint16_t length,
                                                                    uint8_t *payload)
{
  sl_zigbee_af_g_b_c_s_device_log_info_t deviceInfo;
  uint8_t tunnelHeader[2];
  uint8_t pendingMessages;
  uint16_t dataLen;
  uint8_t *data;
  sl_zigbee_af_sleepy_message_id_t sleepyMessageId = SL_ZIGBEE_AF_PLUGIN_SLEEPY_MESSAGE_INVALID_ID;
  sl_zigbee_af_sleepy_message_t sleepyMessage;
  uint16_t messageCode;

  pluginDebugPrint("CHF: TunnelDataReceived ");
  pluginDebugExec(sl_zigbee_af_print_big_endian_eui64(senderDeviceId));
  pluginDebugPrint(" [");
  pluginDebugPrintBuffer(payload, length, false);
  pluginDebugPrintln("]");

  //  Check to make sure the destination device is in the device log.
  if (!sl_zigbee_af_gbcs_device_log_get(senderDeviceId, &deviceInfo)) {
    sli_zigbee_af_comms_hub_function_tunnel_destroy(senderDeviceId);
    sl_zigbee_af_comms_hub_function_println("Given destination device ID has not been configured in the GBCS device log");
    return;
  }

  if (0 == length) {
    return;
  }

  // If it's a sleepy device then check for the GET and PUT message headers
  if (sl_zigbee_af_gbcs_device_log_is_sleepy_type(deviceInfo.deviceType)) {
    if (*payload == TUNNEL_MANAGER_HEADER_GET) {
      bool result;
      // GET-RESPONSE (the concatenation 0x02 || number of Remote Party Messages remaining):
      // this is used by the CHF to send a Remote Party Message to the GSME. It also indicates
      // how many Remote Party Messages have yet to be retrieved;
      pendingMessages = sl_zigbee_af_sleepy_message_queue_get_num_messages(senderDeviceId);
      if (pendingMessages > 0) {
        sleepyMessageId = sl_zigbee_af_sleepy_message_queue_get_pending_message_id(senderDeviceId);
        sl_zigbee_af_sleepy_message_queue_get_pending_message(sleepyMessageId, &sleepyMessage);
        sl_zigbee_af_sleepy_message_queue_remove_message(sleepyMessageId);
        pendingMessages--;
        dataLen = sleepyMessage.length;
        data = sleepyMessage.payload;
        messageCode = sleepyMessageUseCaseCodes[sleepyMessageId];
      } else {
        dataLen = 0;
        data = NULL;
      }
      tunnelHeader[0] = TUNNEL_MANAGER_HEADER_GET_RESPONSE;
      tunnelHeader[1] = pendingMessages;
      result = sli_zigbee_af_comms_hub_function_tunnel_send_data(senderDeviceId, 2, tunnelHeader, dataLen, data);

      // If we sent or attempted to send a message from the sleepy queue then
      // we need to let the calling application know the status of that message.
      if (sleepyMessageId != SL_ZIGBEE_AF_PLUGIN_SLEEPY_MESSAGE_INVALID_ID) {
        // log status of sent message to sleepy device.
        // EMAPPFWKV2-1315 - "For any Event Log entries relating to Event Codes
        // 0x0054 and 0x0055, the Device shall record the Commands received on the
        // Network Interface by including the Message Code in the Event Log"
        sl_zigbee_af_zcl_event_t event;
        event.eventData[0] = 0x02;
        event.eventData[1] = HIGH_BYTE(messageCode);
        event.eventData[2] = LOW_BYTE(messageCode);
        event.eventId = (result) ? GBCS_EVENT_ID_IMM_HAN_CMD_RXED_ACTED : GBCS_EVENT_ID_IMM_HAN_CMD_RXED_NOT_ACTED;
        event.eventTime = sl_zigbee_af_get_current_time();
        sl_zigbee_af_events_server_add_event(SL_ZIGBEE_AF_PLUGIN_GAS_PROXY_FUNCTION_REMOTE_COMMSHUB_ENDPOINT,
                                             SL_ZIGBEE_ZCL_EVENT_LOG_ID_GENERAL_EVENT_LOG,
                                             &event);
        sl_zigbee_af_comms_hub_function_send_cb((result) ? SL_ZIGBEE_AF_CHF_STATUS_SUCCESS : SL_ZIGBEE_AF_CHF_STATUS_TUNNEL_FAILURE,
                                                senderDeviceId,
                                                dataLen,
                                                data);
      }

      // If there are no more pending messages then clean the FNF attribute in the mirror.
      if (pendingMessages == 0) {
        clearTunnelMessagePending(senderDeviceId);
      }
    } else if (*payload == TUNNEL_MANAGER_HEADER_PUT) {
      sl_zigbee_af_comms_hub_function_received_cb(senderDeviceId, length - 1, payload + 1);
    } else {
      // Not sure what this is so let's just pass it up to the application to deal with
      sl_zigbee_af_comms_hub_function_received_cb(senderDeviceId, length, payload);
    }
  }
  // Data from a non-sleepy device just pass it on to the app.
  else {
    sl_zigbee_af_comms_hub_function_received_cb(senderDeviceId, length, payload);
  }
}

/** @brief Upfate Functional Notification Flags routines
 */
sl_zigbee_af_plugin_comms_hub_function_status_t sli_zigbee_af_update_functional_notification_flags_by_endpoint(uint8_t endpoint,
                                                                                                               uint32_t resetMask,
                                                                                                               uint32_t setMask)
{
  sl_zigbee_af_status_t status;
  uint32_t notificationFlags;

  status = sl_zigbee_af_read_client_attribute(endpoint,
                                              ZCL_SIMPLE_METERING_CLUSTER_ID,
                                              ZCL_FUNCTIONAL_NOTIFICATION_FLAGS_ATTRIBUTE_ID,
                                              (uint8_t *)&notificationFlags,
                                              4);
  if (status != SL_ZIGBEE_ZCL_STATUS_SUCCESS) {
    sl_zigbee_af_comms_hub_function_println("Unable to read the functional notification flags attribute: 0x%02X", status);
    return SL_ZIGBEE_AF_CHF_STATUS_FNF_ATTR_FAILURE;
  }

  notificationFlags &= resetMask;
  notificationFlags |= setMask;

  status = sl_zigbee_af_write_client_attribute(endpoint,
                                               ZCL_SIMPLE_METERING_CLUSTER_ID,
                                               ZCL_FUNCTIONAL_NOTIFICATION_FLAGS_ATTRIBUTE_ID,
                                               (uint8_t *)&notificationFlags,
                                               ZCL_BITMAP32_ATTRIBUTE_TYPE);
  if (status != SL_ZIGBEE_ZCL_STATUS_SUCCESS) {
    sl_zigbee_af_comms_hub_function_println("Unable to write the functional notification flags attribute: 0x%02X", status);
    return SL_ZIGBEE_AF_CHF_STATUS_FNF_ATTR_FAILURE;
  }

  return SL_ZIGBEE_AF_CHF_STATUS_SUCCESS;
}

sl_zigbee_af_plugin_comms_hub_function_status_t sli_zigbee_af_update_functional_notification_flags_by_eui64(sl_802154_long_addr_t deviceId,
                                                                                                            uint32_t resetMask,
                                                                                                            uint32_t setMask)
{
  uint8_t mirrorEndpoint;

  if (!sl_zigbee_af_meter_mirror_get_endpoint_by_eui64(deviceId, &mirrorEndpoint)) {
    sl_zigbee_af_comms_hub_function_println("Mirror endpoint for given EUI64 has not been configured");
    return SL_ZIGBEE_AF_CHF_STATUS_NO_MIRROR;
  }

  return sli_zigbee_af_update_functional_notification_flags_by_endpoint(mirrorEndpoint,
                                                                        resetMask,
                                                                        setMask);
}

//------------------------------------------------------------------------------
// Internal Functions

static sl_zigbee_af_plugin_comms_hub_function_status_t setTunnelMessagePending(sl_802154_long_addr_t deviceId)
{
  return sli_zigbee_af_update_functional_notification_flags_by_eui64(deviceId,
                                                                     0xFFFFFFFF,
                                                                     SL_ZIGBEE_AF_METERING_FNF_TUNNEL_MESSAGE_PENDING);
}

static sl_zigbee_af_plugin_comms_hub_function_status_t clearTunnelMessagePending(sl_802154_long_addr_t deviceId)
{
  return sli_zigbee_af_update_functional_notification_flags_by_eui64(deviceId,
                                                                     ~SL_ZIGBEE_AF_METERING_FNF_TUNNEL_MESSAGE_PENDING,
                                                                     0);
}

static void tunnelCheckEventHandler(sl_zigbee_af_event_t * event)
{
  uint32_t delay = (SL_ZIGBEE_AF_PLUGIN_COMMS_HUB_FUNCTION_TUNNEL_CHECK_PERIOD_SECONDS
                    * MILLISECOND_TICKS_PER_SECOND);
  if (delay <= SL_ZIGBEE_MAX_EVENT_DELAY_MS) {
    sl_zigbee_af_event_set_delay_ms(tunnelCheckEventControl,
                                    delay);
  }
  if (discoveryInProgress) {
    return;
  }
  checkForAnyDeviceThatNeedsTunnelCreated();
}

static void initiateDiscovery(sl_802154_short_addr_t nodeId, sl_802154_long_addr_t deviceEui64)
{
  sl_status_t status;

  // In case we need to discover multiple devices AND
  // our periodic event kicks off again while waiting for responses,
  // we can't perform another discovery.
  // This is because we only support a single discovery at one
  // time.  We can't perform multiple discoveries.  It is a limitation of
  // the service discovery code.
  discoveryInProgress = true;

  if (nodeId == SL_ZIGBEE_NULL_NODE_ID) {
    pluginDebugPrintln("%s: Initiating node ID discovery.", PLUGIN_NAME);
    status = sl_zigbee_af_find_node_id(deviceEui64, tunnelDiscoveryCallback);
    if (status != SL_STATUS_OK) {
      sl_zigbee_af_core_println("%s: Failed to initiate node ID discovery for tunnel.",
                                PLUGIN_NAME);
    }
  } else {
    pluginDebugPrintln("%s: Initiating endpoint discovery.", PLUGIN_NAME);
    status = sl_zigbee_af_find_devices_by_profile_and_cluster(nodeId,
                                                              SE_PROFILE_ID,
                                                              ZCL_TUNNELING_CLUSTER_ID,
                                                              SL_ZIGBEE_AF_SERVER_CLUSTER_DISCOVERY,
                                                              tunnelDiscoveryCallback);
    if (status != SL_STATUS_OK) {
      sl_zigbee_af_core_println("%s: Failed to initiate tunnel service discovery to 0x%04X",
                                PLUGIN_NAME,
                                nodeId);
    }
  }
}

static bool deviceTypeRequiresTunnelInitiated(sl_zigbee_af_g_b_c_s_device_log_info_t deviceInfo)
{
  // Per the GBCS spec section 10.2.2.1,
  // only these devices require tunnel initiated to them.
  return (deviceInfo.deviceType == SL_ZIGBEE_AF_GBCS_ESME_DEVICE_TYPE
          || deviceInfo.deviceType == SL_ZIGBEE_AF_GBCS_HCALCS_DEVICE_TYPE
          || deviceInfo.deviceType == SL_ZIGBEE_AF_GBCS_PPMID_DEVICE_TYPE);
}

static bool checkForSpecificDeviceThatNeedsTunnelCreated(sl_802154_short_addr_t nodeId,
                                                         sl_802154_long_addr_t deviceEui64)
{
  sl_zigbee_af_g_b_c_s_device_log_info_t deviceInfo;
  pluginDebugPrint("%s: Checking whether device needs tunnel created ", PLUGIN_NAME);
  pluginDebugExec(sl_zigbee_af_print_big_endian_eui64(deviceEui64));
  if (sl_zigbee_af_gbcs_device_log_get(deviceEui64, &deviceInfo)
      && deviceTypeRequiresTunnelInitiated(deviceInfo)
      && !sli_zigbee_af_comms_hub_function_tunnel_exists(deviceEui64)) {
    sl_zigbee_sec_man_aps_key_metadata_t key_info;
    sl_zigbee_sec_man_context_t context;
    sl_zigbee_sec_man_init_context(&context);
    context.core_key_type = SL_ZB_SEC_MAN_KEY_TYPE_APP_LINK;
    context.flags |= ZB_SEC_MAN_FLAG_KEY_INDEX_IS_VALID;
    sl_status_t status = sl_zigbee_sec_man_export_link_key_by_eui(deviceEui64, &context, NULL, &key_info);
    if (((context.key_index != 0xFF)
         && (SL_STATUS_OK == status)
         && (key_info.bitmask & SL_ZIGBEE_KEY_IS_AUTHORIZED))
#ifdef SL_ZIGBEE_TEST
        || (sl_zigbee_af_is_full_smart_energy_security_present() == SL_ZIGBEE_AF_INVALID_KEY_ESTABLISHMENT_SUITE)
#endif //SL_ZIGBEE_TEST
        ) {
      pluginDebugPrintln(": YES.");
      initiateDiscovery(nodeId, deviceEui64);
      tunnelTargetAttempts++;
      pluginDebugPrintln("Initiate discovery attempt: %d", tunnelTargetAttempts);
      return true;
    }
  }
  pluginDebugPrintln(": No");
  return false;
}

static void checkForAnyDeviceThatNeedsTunnelCreated(void)
{
  sl_802154_long_addr_t deviceEui64;
  sl_zigbee_af_g_b_c_s_device_log_info_t deviceInfo;

  uint8_t max = sl_zigbee_af_gbcs_device_log_max_size();

  for (; currentDeviceLogEntry < max; currentDeviceLogEntry++) {
    if (sl_zigbee_af_gbcs_device_log_retrieve_by_index(currentDeviceLogEntry, deviceEui64, &deviceInfo)) {
      if (tunnelTargetAttempts < MAX_TUNNEL_TARGET_ATTEMPTS
          && checkForSpecificDeviceThatNeedsTunnelCreated(tunnelTargetNodeId,
                                                          deviceEui64)) {
        sl_zigbee_af_core_print("%s: Device needs tunnel created", PLUGIN_NAME);
        sl_zigbee_af_print_big_endian_eui64(deviceEui64);
        sl_zigbee_af_core_println("");
        return;
      }
    }
    tunnelTargetAttempts = 0;
    tunnelTargetNodeId = SL_ZIGBEE_NULL_NODE_ID;
  }
  currentDeviceLogEntry = 0;
}

void sl_zigbee_af_comms_hub_function_stack_status_cb(sl_status_t status)
{
  if (status != SL_STATUS_NETWORK_UP) {
    if (status == SL_STATUS_NETWORK_DOWN) {
      sl_zigbee_af_event_set_inactive(tunnelCheckEventControl);
    } else {
      sl_zigbee_af_event_set_delay_ms(tunnelCheckEventControl, 0);
    }
    return;
  }

  tunnelTargetNodeId = SL_ZIGBEE_NULL_NODE_ID;
  currentDeviceLogEntry = 0;
  tunnelTargetAttempts = 0;
  discoveryInProgress = false;

  sl_zigbee_af_core_println("%s: Setting up event for monitoring tunnels.", PLUGIN_NAME);
  sl_zigbee_af_event_set_delay_ms(tunnelCheckEventControl,
                                  (SL_ZIGBEE_AF_PLUGIN_COMMS_HUB_FUNCTION_TUNNEL_CHECK_PERIOD_SECONDS
                                   * MILLISECOND_TICKS_PER_SECOND));
}
