/** Authenticated encryption with associated data(AEAD).
 *
 * @file
 * @copyright Copyright 2023 Secure-IC S.A.S.
 * This file relies on Secure-IC S.A.S. software and patent portfolio.
 * This file cannot be used nor duplicated without prior approval from Secure-IC S.A.S.
 *
 * Examples:
 * The following examples show typical sequences of function calls for
 * encryption and decryption a message.
   @code
   1. One-shot operation
      a. Encryption
         sx_aead_create_aesgcm_enc(ctx, ...)
         sx_aead_feed_aad(ctx, aad, aadsz)
         sx_aead_crypt(ctx, datain, datainz, dataout)
         sx_aead_produce_tag(ctx, tag)
         sx_aead_wait(ctx)
      b. Decryption
         sx_aead_create_aesgcm_dec(ctx, ...)
         sx_aead_feed_aad(ctx, aad, aadsz)
         sx_aead_crypt(ctx, datain, datainz, dataout)
         sx_aead_verify_tag(ctx, tag)
         sx_aead_wait(ctx)
   2. Context-saving operation
         a. Encryption
         First round:
            sx_aead_create_aesccm_enc(ctx)
            sx_aead_feed_aad(ctx, aad, aadsz)
            sx_aead_crypt(ctx, 'first chunk')
            sx_aead_save_state(ctx)
            sx_aead_wait(ctx)
         Intermediary rounds:
            sx_aead_resume_state(ctx)
            sx_aead_crypt(ctx, 'n-th chunk')
            sx_aead_save_state(ctx)
            sx_aead_wait(ctx)
         Last round:
            sx_aead_resume_state(ctx)
            sx_aead_crypt(ctx, 'last chunk')
            sx_aead_produce_tag(ctx, tag)
            sx_aead_wait(ctx)
         b. Decryption
         First round:
            sx_aead_create_aesccm_dec(ctx)
            sx_aead_feed_aad(ctx, aad, aadsz)
            sx_aead_crypt(ctx, 'first chunk')
            sx_aead_save_state(ctx)
            sx_aead_wait(ctx)
         Intermediary rounds:
            sx_aead_resume_state(ctx)
            sx_aead_crypt(ctx, 'n-th chunk')
            sx_aead_save_state(ctx)
            sx_aead_wait(ctx)
         Last round:
            sx_aead_resume_state(ctx)
            sx_aead_crypt(ctx, 'last chunk')
            sx_aead_verify_tag(ctx, tag)
            sx_aead_wait(ctx)
   @endcode
 */

#ifndef AEAD_HEADER_FILE
#define AEAD_HEADER_FILE

#ifdef __cplusplus
extern "C" {
#endif

#include <stddef.h>
#include "sl_code_classification.h"

/** Initialization vector (IV) size, in bytes, for GCM encryption/decryption */
#define SX_GCM_IV_SZ 12u

/** Size, in bytes, of GCM authentication tag */
#define SX_GCM_TAG_SZ 16u

struct sxaead;
struct sxdataref;


/** Adds AAD chunks
 *
 * This function is used for adding AAD buffer given by \p aad. The function
 * will return immediately.
 *
 * @param[in,out] c AEAD operation context
 * @param[in] aad additional authentication data(AAD), with size \p aadsz
 * @param[in] aadsz size, in bytes, of the additional authenticated data(AAD),
 *                  can be 0 if \p aad is empty
 * @return ::SX_OK
 * @return ::SX_ERR_UNITIALIZED_OBJ
 * @return ::SX_ERR_TOO_BIG
 *
 * @pre - one of the sx_aead_create_*() functions must be called first
 *
 * @remark - the additional authentication data can be empty(\p aadsz = 0)
 * @remark - \p aad buffer should not be changed until the operation is
 *           completed.
 */
SL_CODE_CLASSIFY(SL_CODE_COMPONENT_AES_COMMON, SL_CODE_CLASS_SECURITY)
int sx_aead_feed_aad(struct sxaead *c, const char *aad, size_t aadsz);


/** Adds multiple AAD chunks
 *
 * The function will return immediately.
 *
 * @param[in,out] c AEAD operation context
 * @param[in] aadin head of the array containing input buffers to be
 *                  authenticated, with count \p countaadin
 * @param[in] countaadin number of input buffers
 * @return ::SX_OK
 * @return ::SX_ERR_UNITIALIZED_OBJ
 * @return ::SX_ERR_TOO_BIG
 * @return ::SX_ERR_TOO_SMALL
 * @return ::SX_ERR_FEED_COUNT_EXCEEDED
 * @return ::SX_ERR_FEED_AFTER_FEED
 *
 * @remark - the additional authentication data can not
 *           empty(\p countaadin != 0)
 * @remark - buffers contained in \p aadin should not be changed until
 *           the operation is completed.
 * @remark: This function can not be used with the context saving feature.
 *          It should be directly followed by either sx_aead_crypt(),
 *          sx_aead_multifeed_crypt(), sx_aead_produce_tag() or sx_aead_verify_tag()
 * @remark - this function can not be called after having called sx_aead_feed_aad().
 * @remark - this function can not be called after having called
 *           sx_aead_multifeed_aad().
 */
int sx_aead_multifeed_aad(struct sxaead *c, struct sxdataref *aadin,
   size_t countaadin);


/** Adds data to be encrypted or decrypted.
 *
 * This function is used for adding data to be processed. The function will
 * return immediately.
 *
 * The result of the operation will be transfered to \p dataout after the
 * operation is successfully completed.
 *
 * For context saving, \p datain size(\p datainsz) must be a multiple of 16
 * bytes for AES GCM and CCM and a multiple of 64 bytes for ChaCha20Poly1305,
 * except the last buffer.
 *
 * @param[in,out] c AEAD operation context
 * @param[in] datain data to be encrypted or decryoted, with size \p datainsz
 * @param[in] datainsz size, in bytes, of the data to be encrypted or decrypted
 * @param[out] dataout encrypted or decrypted data, must have \p datainsz bytes
 * @return ::SX_OK
 * @return ::SX_ERR_UNITIALIZED_OBJ
 * @return ::SX_ERR_TOO_BIG
 * @return ::SX_ERR_FEED_AFTER_FEED
 *
 * @pre - one of the sx_aead_create_*() functions must be called first
 *
 * @remark - \p datain buffer should not be changed until the operation is
 *           completed.
 * @remark - this function can not be called after having called sx_aead_crypt().
 * @remark - this function can not be called after having called sx_aead_multifeed_crypt().
 */
SL_CODE_CLASSIFY(SL_CODE_COMPONENT_AES_COMMON, SL_CODE_CLASS_SECURITY)
int sx_aead_crypt(struct sxaead *c, const char *datain, size_t datainsz,
    char *dataout);


/** Adds multiple data to be encrypted or decrypted.
 *
 * This function is used when multiple data inputs and outputs need to be
 * used. The number of inputs and outputs can be different, however the total
 * input size must be equal to the total output size.
 *
 * The function will return immediately.
 *
 * In order to start the operation sx_aead_produce_tag() or
 * sx_aead_verify_tag() must be called.
 *
 * @param[in,out] c AEAD operation context
 * @param[in] datain head of the array containing input buffers be encrypted
 *                   or decrypted, with count \p countdatain
 * @param[in] countdatain number of input buffers
 * @param[in] dataout head of the array containing output buffers where the encrypted or
 *                    decrypted data will be stored, with count \p countdataout
 * @param[in] countdataout number of output buffers
 * @return ::SX_OK
 * @return ::SX_ERR_UNITIALIZED_OBJ
 * @return ::SX_ERR_TOO_BIG
 * @return ::SX_ERR_TOO_SMALL
 * @return ::SX_ERR_FEED_COUNT_EXCEEDED
 * @return ::SX_ERR_FEED_UNBALANCED
 * @return ::SX_ERR_FEED_AFTER_FEED
 *
 * @pre - one of the sx_aead_create_*() functions must be called first
 *
 * @remark - this function can be called even if data size, \p countdatain or
 *           \p countdataout, is 0.
 * @remark - this function can be called only once before calling
 *           the hardware.
 * @remark - this function should not be used with the context saving feature (hardware
 *           should not be called with sx_aead_save_state()).
 * @remark - this function can not be called after having called sx_aead_crypt().
 * @remark - this function can not be called after having called sx_aead_multifeed_crypt().
 * @remark - the total maximum number of IN buffer is SX_AEAD_DATA_EXTRA_DESC_IN + 1 and
 *           the total maximum number of OUT buffers is SX_AEAD_DATA_EXTRA_DESC_OUT + 1
 */
int sx_aead_multifeed_crypt(struct sxaead *c,
    const struct sxdataref *datain, size_t countdatain,
    const struct sxdataref *dataout, size_t countdataout);


/** Starts an AEAD encryption and tag computation.
 *
 * The function will return immediately.
 *
 * The computed tag will be transfered to \p tag only after the operation is
 * successfully completed.
 *
 * The user shall check operation status with sx_aead_status() or sx_aead_wait().
 *
 * @param[in,out] c AEAD operation context
 * @param[out] tag authentication tag
 * @return ::SX_OK
 * @return ::SX_ERR_UNITIALIZED_OBJ
 * @return ::SX_ERR_TOO_SMALL
 * @return ::SX_ERR_INCOMPATIBLE_HW
 *
 * @pre - one of the sx_aead_feed_aad() or sx_aead_crypt() functions must be
 *        called first
 *
 * @remark - if used with context saving(last chunk), the fed data size for
 *         the last chunk can not be 0
 */
SL_CODE_CLASSIFY(SL_CODE_COMPONENT_AES_COMMON, SL_CODE_CLASS_SECURITY)
int sx_aead_produce_tag(struct sxaead *c, char *tag);


/** Starts an AEAD decryption and tag validation.
 *
 * The function will return immediately.
 *
 * The user shall check operation status with sx_aead_status() or sx_aead_wait().
 *
 * @param[in,out] c AEAD operation context
 * @param[in] tag authentication tag
 * @return ::SX_OK
 * @return ::SX_ERR_UNITIALIZED_OBJ
 * @return ::SX_ERR_TOO_SMALL
 * @return ::SX_ERR_INCOMPATIBLE_HW
 *
 * @pre - one of the sx_aead_feed_aad() or sx_aead_crypt() functions must be
 *        called first
 *
 * @remark - \p tag buffer should not be changed until the operation is
 *           completed.
 * @remark - if used with context saving(last chunk), the fed data size for
 *         the last chunk can not be 0
 */
SL_CODE_CLASSIFY(SL_CODE_COMPONENT_AES_COMMON, SL_CODE_CLASS_SECURITY)
int sx_aead_verify_tag(struct sxaead *c, const char *tag);


/** Resumes AEAD operation in context-saving.
 *
 * This function shall be called when using context-saving to load the state
 * that was previously saved by sx_aead_save_state() in the sxaead
 * operation context \p c. It must be called with the same sxaead operation
 * context \p c that was used with sx_aead_save_state(). It will reserve
 * all hardware resources required to run the partial AEAD operation.
 * Previously used mode and direction are already stored in sxaead \p c.
 *
 * After successful execution of this function, the context \p c can be passed
 * to any of the AEAD functions, except the sx_aead_create_*()
 * functions.
 *
 * @param[in,out] c AEAD operation context
 * @return ::SX_OK
 * @return ::SX_ERR_UNITIALIZED_OBJ
 * @return ::SX_ERR_CONTEXT_SAVING_NOT_SUPPORTED
 * @return ::SX_ERR_INVALID_KEYREF
 * @return ::SX_ERR_INVALID_KEY_SZ
 * @return ::SX_ERR_INCOMPATIBLE_HW
 * @return ::SX_ERR_RETRY
 *
 * @pre - sx_aead_create_aes*() and sx_aead_save_state() functions
 *        must be called before, for first part of the message.
 * @pre - must be called for each part of the message(besides first), before
 *        sx_aead_crypt() or sx_aead_save_state().
 *
 * @remark - the user must not change the key until all parts of the message to
 *           be encrypted/decrypted are processed.
 */
int sx_aead_resume_state(struct sxaead *c);


/** Starts a partial AEAD operation.
 *
 * This function is used to start a partial encryption or decryption of
 * \p datain. The function will return immediately.
 *
 * The partial result will be transfered to \p dataout only after the operation
 * is successfully completed. The user shall check operation status with
 * sx_aead_status() or sx_aead_wait().
 *
 * @param[in,out] c AEAD operation context
 * @return ::SX_OK
 * @return ::SX_ERR_UNITIALIZED_OBJ
 * @return ::SX_ERR_CONTEXT_SAVING_NOT_SUPPORTED
 * @return ::SX_ERR_WRONG_SIZE_GRANULARITY
 *
 * @pre - sx_aead_crypt() should be called first.
 *
 * @remark - the user must not change the key until all parts of the message to
 *           be encrypted/decrypted are processed.
 * @remark - when in context saving, the sizes of the chunks fed must be
 *           multiple of 16 bytes, besides the last chunk that can be any size,
 *           but not 0
 */
int sx_aead_save_state(struct sxaead *c);


/** Waits until the given AEAD operation has finished
 *
 * This function returns when the AEAD operation was successfully completed,
 * or when an error has occurred that caused the operation to terminate.
 *
 * The return value of this function is the operation status.
 *
 * After this call, all resources have been released and \p c cannot be used
 * again unless sx_aead_create_*() is used.
 *
 * making \p c unusable for a new operation without calling, first, one of the
 * sx_aead_create_*() functions.
 *
 * @param[in,out] c AEAD operation context
 * @return ::SX_OK
 * @return ::SX_ERR_UNITIALIZED_OBJ
 * @return ::SX_ERR_DMA_FAILED
 * @return ::SX_ERR_INVALID_TAG
 *
 * @pre - sx_aead_encrypt or sx_aead_decrypt() function must be called first
 *
 * @see sx_aead_status().
 *
 * @remark - this function is blocking until operation finishes.
 */
SL_CODE_CLASSIFY(SL_CODE_COMPONENT_AES_COMMON, SL_CODE_CLASS_SECURITY)
int sx_aead_wait(struct sxaead *c);


/** Returns the AEAD operation status.
 *
 * If the operation is still ongoing, return ::SX_ERR_HW_PROCESSING.
 * In that case, the user can retry later.
 *
 * When this function returns with a code different than ::SX_ERR_HW_PROCESSING,
 * the AEAD operation has ended and all resources used by the AEAD operation
 * context \p c have been released. In this case, \p c cannot be used for a new
 * operation until one of the sx_aead_create_*() functions is called again.
 *
 * @param[in,out] c AEAD operation context
 * @return ::SX_OK
 * @return ::SX_ERR_UNITIALIZED_OBJ
 * @return ::SX_ERR_HW_PROCESSING
 * @return ::SX_ERR_DMA_FAILED
 * @return ::SX_ERR_INVALID_TAG
 *
 * @pre - sx_aead_encrypt or sx_aead_decrypt() function must be called first
 *
 * @remark -  if authentication fails during decryption, ::SX_ERR_INVALID_TAG
 *            will be returned. In this case, the decrypted text is not valid
 *            and shall not be used.
 */
SL_CODE_CLASSIFY(SL_CODE_COMPONENT_AES_COMMON, SL_CODE_CLASS_SECURITY)
int sx_aead_status(struct sxaead *c);


/** Shortens the tag size of the AEAD operation.
 *
 * This function is optional and should be used only when a smaller tag than
 * maximum size is required. This function sets the new tag size that will be
 * used by sx_aead_produce_tag and sx_aead_verify_tag. Based on this new value,
 * the tag will be truncated and remaining data, up to the maximum tag size
 * per algorithm, will be discarded.
 *
 * The function will return immediately.
 *
 * @param[in,out] c AEAD operation context
 * @param[in] tagsz new size for the tag
 * @return ::SX_OK
 * @return ::SX_ERR_UNITIALIZED_OBJ
 * @return ::SX_ERR_INVALID_TAG_SIZE
 * @return ::SX_ERR_INCOMPATIBLE_HW
 *
 * @pre - one of the sx_aead_create_*()functions must be called first
 *
 * @remark - AES/SM4 CCM tag size is user provided and it must be between 4 and
 * 16 bytes, multiple of 2. If this function is called, the new tag size must be
 * between 4 and the value specified during create, sx_aead_create_*ccm_*().
 */
SL_CODE_CLASSIFY(SL_CODE_COMPONENT_AES_GCM, SL_CODE_CLASS_SECURITY)
int sx_aead_truncate_tag(struct sxaead *c, const size_t tagsz);

#ifdef __cplusplus
}
#endif

#endif
