/*
 * FreeRTOS Kernel V11.1.0
 * Copyright (C) 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
 * the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * https://www.FreeRTOS.org
 * https://github.com/FreeRTOS
 *
 */

/* Defining MPU_WRAPPERS_INCLUDED_FROM_API_FILE prevents task.h from redefining
 * all the API functions to use the MPU wrappers.  That should only be done when
 * task.h is included from an application file. */
#define MPU_WRAPPERS_INCLUDED_FROM_API_FILE

/* Scheduler includes. */
#include "FreeRTOS.h"
#include "task.h"
#include "queue.h"
#include "timers.h"
#include "event_groups.h"
#include "stream_buffer.h"
#include "mpu_prototypes.h"
#include "mpu_syscall_numbers.h"

#undef MPU_WRAPPERS_INCLUDED_FROM_API_FILE
/*-----------------------------------------------------------*/

#if ( (configENABLE_MPU == 1) && (configUSE_MPU_WRAPPERS_V1 == 0) )

    #if (INCLUDE_xTaskDelayUntil == 1)

BaseType_t MPU_xTaskDelayUntil(TickType_t * const pxPreviousWakeTime,
                               const TickType_t xTimeIncrement) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

BaseType_t MPU_xTaskDelayUntil(TickType_t * const pxPreviousWakeTime,
                               const TickType_t xTimeIncrement)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xTaskDelayUntilImpl                       \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xTaskDelayUntil_Unpriv                        \n"
    " MPU_xTaskDelayUntil_Priv:                             \n"
    "     b MPU_xTaskDelayUntilImpl                         \n"
    " MPU_xTaskDelayUntil_Unpriv:                           \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xTaskDelayUntil) : "memory"
  );
}

    #endif /* if ( INCLUDE_xTaskDelayUntil == 1 ) */
/*-----------------------------------------------------------*/

    #if (INCLUDE_xTaskAbortDelay == 1)

BaseType_t MPU_xTaskAbortDelay(TaskHandle_t xTask) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

BaseType_t MPU_xTaskAbortDelay(TaskHandle_t xTask)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xTaskAbortDelayImpl                       \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xTaskAbortDelay_Unpriv                        \n"
    " MPU_xTaskAbortDelay_Priv:                             \n"
    "     b MPU_xTaskAbortDelayImpl                         \n"
    " MPU_xTaskAbortDelay_Unpriv:                           \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xTaskAbortDelay) : "memory"
  );
}

    #endif /* if ( INCLUDE_xTaskAbortDelay == 1 ) */
/*-----------------------------------------------------------*/

    #if (INCLUDE_vTaskDelay == 1)

void MPU_vTaskDelay(const TickType_t xTicksToDelay) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

void MPU_vTaskDelay(const TickType_t xTicksToDelay)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_vTaskDelayImpl                            \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_vTaskDelay_Unpriv                             \n"
    " MPU_vTaskDelay_Priv:                                  \n"
    "     b MPU_vTaskDelayImpl                              \n"
    " MPU_vTaskDelay_Unpriv:                                \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_vTaskDelay) : "memory"
  );
}

    #endif /* if ( INCLUDE_vTaskDelay == 1 ) */
/*-----------------------------------------------------------*/

    #if (INCLUDE_uxTaskPriorityGet == 1)

UBaseType_t MPU_uxTaskPriorityGet(const TaskHandle_t xTask) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

UBaseType_t MPU_uxTaskPriorityGet(const TaskHandle_t xTask)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_uxTaskPriorityGetImpl                     \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_uxTaskPriorityGet_Unpriv                      \n"
    " MPU_uxTaskPriorityGet_Priv:                           \n"
    "     b MPU_uxTaskPriorityGetImpl                       \n"
    " MPU_uxTaskPriorityGet_Unpriv:                         \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_uxTaskPriorityGet) : "memory"
  );
}

    #endif /* if ( INCLUDE_uxTaskPriorityGet == 1 ) */
/*-----------------------------------------------------------*/

    #if (INCLUDE_eTaskGetState == 1)

eTaskState MPU_eTaskGetState(TaskHandle_t xTask) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

eTaskState MPU_eTaskGetState(TaskHandle_t xTask)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_eTaskGetStateImpl                         \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_eTaskGetState_Unpriv                          \n"
    " MPU_eTaskGetState_Priv:                               \n"
    "     b MPU_eTaskGetStateImpl                           \n"
    " MPU_eTaskGetState_Unpriv:                             \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_eTaskGetState) : "memory"
  );
}

    #endif /* if ( INCLUDE_eTaskGetState == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_TRACE_FACILITY == 1)

void MPU_vTaskGetInfo(TaskHandle_t xTask,
                      TaskStatus_t * pxTaskStatus,
                      BaseType_t xGetFreeStackSpace,
                      eTaskState eState) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

void MPU_vTaskGetInfo(TaskHandle_t xTask,
                      TaskStatus_t * pxTaskStatus,
                      BaseType_t xGetFreeStackSpace,
                      eTaskState eState)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_vTaskGetInfoImpl                          \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_vTaskGetInfo_Unpriv                           \n"
    " MPU_vTaskGetInfo_Priv:                                \n"
    "     b MPU_vTaskGetInfoImpl                            \n"
    " MPU_vTaskGetInfo_Unpriv:                              \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_vTaskGetInfo) : "memory"
  );
}

    #endif /* if ( configUSE_TRACE_FACILITY == 1 ) */
/*-----------------------------------------------------------*/

    #if (INCLUDE_xTaskGetIdleTaskHandle == 1)

TaskHandle_t MPU_xTaskGetIdleTaskHandle(void) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

TaskHandle_t MPU_xTaskGetIdleTaskHandle(void)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xTaskGetIdleTaskHandleImpl                \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xTaskGetIdleTaskHandle_Unpriv                 \n"
    " MPU_xTaskGetIdleTaskHandle_Priv:                      \n"
    "     b MPU_xTaskGetIdleTaskHandleImpl                  \n"
    " MPU_xTaskGetIdleTaskHandle_Unpriv:                    \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xTaskGetIdleTaskHandle) : "memory"
  );
}

    #endif /* if ( INCLUDE_xTaskGetIdleTaskHandle == 1 ) */
/*-----------------------------------------------------------*/

    #if (INCLUDE_vTaskSuspend == 1)

void MPU_vTaskSuspend(TaskHandle_t xTaskToSuspend) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

void MPU_vTaskSuspend(TaskHandle_t xTaskToSuspend)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_vTaskSuspendImpl                          \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_vTaskSuspend_Unpriv                           \n"
    " MPU_vTaskSuspend_Priv:                                \n"
    "     b MPU_vTaskSuspendImpl                            \n"
    " MPU_vTaskSuspend_Unpriv:                              \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_vTaskSuspend) : "memory"
  );
}

    #endif /* if ( INCLUDE_vTaskSuspend == 1 ) */
/*-----------------------------------------------------------*/

    #if (INCLUDE_vTaskSuspend == 1)

void MPU_vTaskResume(TaskHandle_t xTaskToResume) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

void MPU_vTaskResume(TaskHandle_t xTaskToResume)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_vTaskResumeImpl                           \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_vTaskResume_Unpriv                            \n"
    " MPU_vTaskResume_Priv:                                 \n"
    "     b MPU_vTaskResumeImpl                             \n"
    " MPU_vTaskResume_Unpriv:                               \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_vTaskResume) : "memory"
  );
}

    #endif /* if ( INCLUDE_vTaskSuspend == 1 ) */
/*-----------------------------------------------------------*/

TickType_t MPU_xTaskGetTickCount(void) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

TickType_t MPU_xTaskGetTickCount(void)       /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xTaskGetTickCountImpl                     \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xTaskGetTickCount_Unpriv                      \n"
    " MPU_xTaskGetTickCount_Priv:                           \n"
    "     b MPU_xTaskGetTickCountImpl                       \n"
    " MPU_xTaskGetTickCount_Unpriv:                         \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xTaskGetTickCount) : "memory"
  );
}
/*-----------------------------------------------------------*/

UBaseType_t MPU_uxTaskGetNumberOfTasks(void) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

UBaseType_t MPU_uxTaskGetNumberOfTasks(void)       /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_uxTaskGetNumberOfTasksImpl                \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_uxTaskGetNumberOfTasks_Unpriv                 \n"
    " MPU_uxTaskGetNumberOfTasks_Priv:                      \n"
    "     b MPU_uxTaskGetNumberOfTasksImpl                  \n"
    " MPU_uxTaskGetNumberOfTasks_Unpriv:                    \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_uxTaskGetNumberOfTasks) : "memory"
  );
}
/*-----------------------------------------------------------*/

    #if (configGENERATE_RUN_TIME_STATS == 1)

configRUN_TIME_COUNTER_TYPE MPU_ulTaskGetRunTimeCounter(const TaskHandle_t xTask) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

configRUN_TIME_COUNTER_TYPE MPU_ulTaskGetRunTimeCounter(const TaskHandle_t xTask)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_ulTaskGetRunTimeCounterImpl               \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_ulTaskGetRunTimeCounter_Unpriv                \n"
    " MPU_ulTaskGetRunTimeCounter_Priv:                     \n"
    "     b MPU_ulTaskGetRunTimeCounterImpl                 \n"
    " MPU_ulTaskGetRunTimeCounter_Unpriv:                   \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_ulTaskGetRunTimeCounter) : "memory"
  );
}

    #endif /* if ( ( configGENERATE_RUN_TIME_STATS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configGENERATE_RUN_TIME_STATS == 1)

configRUN_TIME_COUNTER_TYPE MPU_ulTaskGetRunTimePercent(const TaskHandle_t xTask) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

configRUN_TIME_COUNTER_TYPE MPU_ulTaskGetRunTimePercent(const TaskHandle_t xTask)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_ulTaskGetRunTimePercentImpl               \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_ulTaskGetRunTimePercent_Unpriv                \n"
    " MPU_ulTaskGetRunTimePercent_Priv:                     \n"
    "     b MPU_ulTaskGetRunTimePercentImpl                 \n"
    " MPU_ulTaskGetRunTimePercent_Unpriv:                   \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_ulTaskGetRunTimePercent) : "memory"
  );
}

    #endif /* if ( ( configGENERATE_RUN_TIME_STATS == 1 ) */
/*-----------------------------------------------------------*/

    #if ( (configGENERATE_RUN_TIME_STATS == 1) && (INCLUDE_xTaskGetIdleTaskHandle == 1) )

configRUN_TIME_COUNTER_TYPE MPU_ulTaskGetIdleRunTimePercent(void) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

configRUN_TIME_COUNTER_TYPE MPU_ulTaskGetIdleRunTimePercent(void)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_ulTaskGetIdleRunTimePercentImpl           \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_ulTaskGetIdleRunTimePercent_Unpriv            \n"
    " MPU_ulTaskGetIdleRunTimePercent_Priv:                 \n"
    "     b MPU_ulTaskGetIdleRunTimePercentImpl             \n"
    " MPU_ulTaskGetIdleRunTimePercent_Unpriv:               \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_ulTaskGetIdleRunTimePercent) : "memory"
  );
}

    #endif /* if ( ( configGENERATE_RUN_TIME_STATS == 1 ) && ( INCLUDE_xTaskGetIdleTaskHandle == 1 ) ) */
/*-----------------------------------------------------------*/

    #if ( (configGENERATE_RUN_TIME_STATS == 1) && (INCLUDE_xTaskGetIdleTaskHandle == 1) )

configRUN_TIME_COUNTER_TYPE MPU_ulTaskGetIdleRunTimeCounter(void) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

configRUN_TIME_COUNTER_TYPE MPU_ulTaskGetIdleRunTimeCounter(void)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_ulTaskGetIdleRunTimeCounterImpl           \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_ulTaskGetIdleRunTimeCounter_Unpriv            \n"
    " MPU_ulTaskGetIdleRunTimeCounter_Priv:                 \n"
    "     b MPU_ulTaskGetIdleRunTimeCounterImpl             \n"
    " MPU_ulTaskGetIdleRunTimeCounter_Unpriv:               \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_ulTaskGetIdleRunTimeCounter) : "memory"
  );
}

    #endif /* if ( ( configGENERATE_RUN_TIME_STATS == 1 ) && ( INCLUDE_xTaskGetIdleTaskHandle == 1 ) ) */
/*-----------------------------------------------------------*/

    #if (configUSE_APPLICATION_TASK_TAG == 1)

void MPU_vTaskSetApplicationTaskTag(TaskHandle_t xTask,
                                    TaskHookFunction_t pxHookFunction) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

void MPU_vTaskSetApplicationTaskTag(TaskHandle_t xTask,
                                    TaskHookFunction_t pxHookFunction)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_vTaskSetApplicationTaskTagImpl            \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_vTaskSetApplicationTaskTag_Unpriv             \n"
    " MPU_vTaskSetApplicationTaskTag_Priv:                  \n"
    "     b MPU_vTaskSetApplicationTaskTagImpl              \n"
    " MPU_vTaskSetApplicationTaskTag_Unpriv:                \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_vTaskSetApplicationTaskTag) : "memory"
  );
}

    #endif /* if ( configUSE_APPLICATION_TASK_TAG == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_APPLICATION_TASK_TAG == 1)

TaskHookFunction_t MPU_xTaskGetApplicationTaskTag(TaskHandle_t xTask) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

TaskHookFunction_t MPU_xTaskGetApplicationTaskTag(TaskHandle_t xTask)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xTaskGetApplicationTaskTagImpl            \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xTaskGetApplicationTaskTag_Unpriv             \n"
    " MPU_xTaskGetApplicationTaskTag_Priv:                  \n"
    "     b MPU_xTaskGetApplicationTaskTagImpl              \n"
    " MPU_xTaskGetApplicationTaskTag_Unpriv:                \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xTaskGetApplicationTaskTag) : "memory"
  );
}

    #endif /* if ( configUSE_APPLICATION_TASK_TAG == 1 ) */
/*-----------------------------------------------------------*/

    #if (configNUM_THREAD_LOCAL_STORAGE_POINTERS != 0)

void MPU_vTaskSetThreadLocalStoragePointer(TaskHandle_t xTaskToSet,
                                           BaseType_t xIndex,
                                           void * pvValue) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

void MPU_vTaskSetThreadLocalStoragePointer(TaskHandle_t xTaskToSet,
                                           BaseType_t xIndex,
                                           void * pvValue)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_vTaskSetThreadLocalStoragePointerImpl     \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_vTaskSetThreadLocalStoragePointer_Unpriv      \n"
    " MPU_vTaskSetThreadLocalStoragePointer_Priv:           \n"
    "     b MPU_vTaskSetThreadLocalStoragePointerImpl       \n"
    " MPU_vTaskSetThreadLocalStoragePointer_Unpriv:         \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_vTaskSetThreadLocalStoragePointer) : "memory"
  );
}

    #endif /* if ( configNUM_THREAD_LOCAL_STORAGE_POINTERS != 0 ) */
/*-----------------------------------------------------------*/

    #if (configNUM_THREAD_LOCAL_STORAGE_POINTERS != 0)

void * MPU_pvTaskGetThreadLocalStoragePointer(TaskHandle_t xTaskToQuery,
                                              BaseType_t xIndex) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

void * MPU_pvTaskGetThreadLocalStoragePointer(TaskHandle_t xTaskToQuery,
                                              BaseType_t xIndex)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_pvTaskGetThreadLocalStoragePointerImpl    \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_pvTaskGetThreadLocalStoragePointer_Unpriv     \n"
    " MPU_pvTaskGetThreadLocalStoragePointer_Priv:          \n"
    "     b MPU_pvTaskGetThreadLocalStoragePointerImpl      \n"
    " MPU_pvTaskGetThreadLocalStoragePointer_Unpriv:        \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_pvTaskGetThreadLocalStoragePointer) : "memory"
  );
}

    #endif /* if ( configNUM_THREAD_LOCAL_STORAGE_POINTERS != 0 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_TRACE_FACILITY == 1)

UBaseType_t MPU_uxTaskGetSystemState(TaskStatus_t * const pxTaskStatusArray,
                                     const UBaseType_t uxArraySize,
                                     configRUN_TIME_COUNTER_TYPE * const pulTotalRunTime) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

UBaseType_t MPU_uxTaskGetSystemState(TaskStatus_t * const pxTaskStatusArray,
                                     const UBaseType_t uxArraySize,
                                     configRUN_TIME_COUNTER_TYPE * const pulTotalRunTime)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_uxTaskGetSystemStateImpl                  \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_uxTaskGetSystemState_Unpriv                   \n"
    " MPU_uxTaskGetSystemState_Priv:                        \n"
    "     b MPU_uxTaskGetSystemStateImpl                    \n"
    " MPU_uxTaskGetSystemState_Unpriv:                      \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_uxTaskGetSystemState) : "memory"
  );
}

    #endif /* if ( configUSE_TRACE_FACILITY == 1 ) */
/*-----------------------------------------------------------*/

    #if (INCLUDE_uxTaskGetStackHighWaterMark == 1)

UBaseType_t MPU_uxTaskGetStackHighWaterMark(TaskHandle_t xTask) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

UBaseType_t MPU_uxTaskGetStackHighWaterMark(TaskHandle_t xTask)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_uxTaskGetStackHighWaterMarkImpl           \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_uxTaskGetStackHighWaterMark_Unpriv            \n"
    " MPU_uxTaskGetStackHighWaterMark_Priv:                 \n"
    "     b MPU_uxTaskGetStackHighWaterMarkImpl             \n"
    " MPU_uxTaskGetStackHighWaterMark_Unpriv:               \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_uxTaskGetStackHighWaterMark) : "memory"
  );
}

    #endif /* if ( INCLUDE_uxTaskGetStackHighWaterMark == 1 ) */
/*-----------------------------------------------------------*/

    #if (INCLUDE_uxTaskGetStackHighWaterMark2 == 1)

configSTACK_DEPTH_TYPE MPU_uxTaskGetStackHighWaterMark2(TaskHandle_t xTask) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

configSTACK_DEPTH_TYPE MPU_uxTaskGetStackHighWaterMark2(TaskHandle_t xTask)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_uxTaskGetStackHighWaterMark2Impl          \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_uxTaskGetStackHighWaterMark2_Unpriv           \n"
    " MPU_uxTaskGetStackHighWaterMark2_Priv:                \n"
    "     b MPU_uxTaskGetStackHighWaterMark2Impl            \n"
    " MPU_uxTaskGetStackHighWaterMark2_Unpriv:              \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_uxTaskGetStackHighWaterMark2) : "memory"
  );
}

    #endif /* if ( INCLUDE_uxTaskGetStackHighWaterMark2 == 1 ) */
/*-----------------------------------------------------------*/

    #if ( (INCLUDE_xTaskGetCurrentTaskHandle == 1) || (configUSE_MUTEXES == 1) )

TaskHandle_t MPU_xTaskGetCurrentTaskHandle(void) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

TaskHandle_t MPU_xTaskGetCurrentTaskHandle(void)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xTaskGetCurrentTaskHandleImpl             \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xTaskGetCurrentTaskHandle_Unpriv              \n"
    " MPU_xTaskGetCurrentTaskHandle_Priv:                   \n"
    "     b MPU_xTaskGetCurrentTaskHandleImpl               \n"
    " MPU_xTaskGetCurrentTaskHandle_Unpriv:                 \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xTaskGetCurrentTaskHandle) : "memory"
  );
}

    #endif /* if ( ( INCLUDE_xTaskGetCurrentTaskHandle == 1 ) || ( configUSE_MUTEXES == 1 ) ) */
/*-----------------------------------------------------------*/

    #if (INCLUDE_xTaskGetSchedulerState == 1)

BaseType_t MPU_xTaskGetSchedulerState(void) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

BaseType_t MPU_xTaskGetSchedulerState(void)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xTaskGetSchedulerStateImpl                \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xTaskGetSchedulerState_Unpriv                 \n"
    " MPU_xTaskGetSchedulerState_Priv:                      \n"
    "     b MPU_xTaskGetSchedulerStateImpl                  \n"
    " MPU_xTaskGetSchedulerState_Unpriv:                    \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xTaskGetSchedulerState) : "memory"
  );
}

    #endif /* if ( INCLUDE_xTaskGetSchedulerState == 1 ) */
/*-----------------------------------------------------------*/

void MPU_vTaskSetTimeOutState(TimeOut_t * const pxTimeOut) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

void MPU_vTaskSetTimeOutState(TimeOut_t * const pxTimeOut)       /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_vTaskSetTimeOutStateImpl                  \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_vTaskSetTimeOutState_Unpriv                   \n"
    " MPU_vTaskSetTimeOutState_Priv:                        \n"
    "     b MPU_vTaskSetTimeOutStateImpl                    \n"
    " MPU_vTaskSetTimeOutState_Unpriv:                      \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_vTaskSetTimeOutState) : "memory"
  );
}
/*-----------------------------------------------------------*/

BaseType_t MPU_xTaskCheckForTimeOut(TimeOut_t * const pxTimeOut,
                                    TickType_t * const pxTicksToWait) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

BaseType_t MPU_xTaskCheckForTimeOut(TimeOut_t * const pxTimeOut,
                                    TickType_t * const pxTicksToWait)       /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xTaskCheckForTimeOutImpl                  \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xTaskCheckForTimeOut_Unpriv                   \n"
    " MPU_xTaskCheckForTimeOut_Priv:                        \n"
    "     b MPU_xTaskCheckForTimeOutImpl                    \n"
    " MPU_xTaskCheckForTimeOut_Unpriv:                      \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xTaskCheckForTimeOut) : "memory"
  );
}
/*-----------------------------------------------------------*/

    #if (configUSE_TASK_NOTIFICATIONS == 1)

BaseType_t MPU_xTaskGenericNotifyEntry(const xTaskGenericNotifyParams_t * pxParams) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

BaseType_t MPU_xTaskGenericNotifyEntry(const xTaskGenericNotifyParams_t * pxParams)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xTaskGenericNotifyImpl                    \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xTaskGenericNotify_Unpriv                     \n"
    " MPU_xTaskGenericNotify_Priv:                          \n"
    "     b MPU_xTaskGenericNotifyImpl                      \n"
    " MPU_xTaskGenericNotify_Unpriv:                        \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xTaskGenericNotify) : "memory"
  );
}

    #endif /* if ( configUSE_TASK_NOTIFICATIONS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_TASK_NOTIFICATIONS == 1)

BaseType_t MPU_xTaskGenericNotifyWaitEntry(const xTaskGenericNotifyWaitParams_t * pxParams) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

BaseType_t MPU_xTaskGenericNotifyWaitEntry(const xTaskGenericNotifyWaitParams_t * pxParams)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xTaskGenericNotifyWaitImpl                \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xTaskGenericNotifyWait_Unpriv                 \n"
    " MPU_xTaskGenericNotifyWait_Priv:                      \n"
    "     b MPU_xTaskGenericNotifyWaitImpl                  \n"
    " MPU_xTaskGenericNotifyWait_Unpriv:                    \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xTaskGenericNotifyWait) : "memory"
  );
}

    #endif /* if ( configUSE_TASK_NOTIFICATIONS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_TASK_NOTIFICATIONS == 1)

uint32_t MPU_ulTaskGenericNotifyTake(UBaseType_t uxIndexToWaitOn,
                                     BaseType_t xClearCountOnExit,
                                     TickType_t xTicksToWait) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

uint32_t MPU_ulTaskGenericNotifyTake(UBaseType_t uxIndexToWaitOn,
                                     BaseType_t xClearCountOnExit,
                                     TickType_t xTicksToWait)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_ulTaskGenericNotifyTakeImpl               \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_ulTaskGenericNotifyTake_Unpriv                \n"
    " MPU_ulTaskGenericNotifyTake_Priv:                     \n"
    "     b MPU_ulTaskGenericNotifyTakeImpl                 \n"
    " MPU_ulTaskGenericNotifyTake_Unpriv:                   \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_ulTaskGenericNotifyTake) : "memory"
  );
}

    #endif /* if ( configUSE_TASK_NOTIFICATIONS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_TASK_NOTIFICATIONS == 1)

BaseType_t MPU_xTaskGenericNotifyStateClear(TaskHandle_t xTask,
                                            UBaseType_t uxIndexToClear) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

BaseType_t MPU_xTaskGenericNotifyStateClear(TaskHandle_t xTask,
                                            UBaseType_t uxIndexToClear)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xTaskGenericNotifyStateClearImpl          \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xTaskGenericNotifyStateClear_Unpriv           \n"
    " MPU_xTaskGenericNotifyStateClear_Priv:                \n"
    "     b MPU_xTaskGenericNotifyStateClearImpl            \n"
    " MPU_xTaskGenericNotifyStateClear_Unpriv:              \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xTaskGenericNotifyStateClear) : "memory"
  );
}

    #endif /* if ( configUSE_TASK_NOTIFICATIONS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_TASK_NOTIFICATIONS == 1)

uint32_t MPU_ulTaskGenericNotifyValueClear(TaskHandle_t xTask,
                                           UBaseType_t uxIndexToClear,
                                           uint32_t ulBitsToClear) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

uint32_t MPU_ulTaskGenericNotifyValueClear(TaskHandle_t xTask,
                                           UBaseType_t uxIndexToClear,
                                           uint32_t ulBitsToClear)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_ulTaskGenericNotifyValueClearImpl         \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_ulTaskGenericNotifyValueClear_Unpriv          \n"
    " MPU_ulTaskGenericNotifyValueClear_Priv:               \n"
    "     b MPU_ulTaskGenericNotifyValueClearImpl           \n"
    " MPU_ulTaskGenericNotifyValueClear_Unpriv:             \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_ulTaskGenericNotifyValueClear) : "memory"
  );
}

    #endif /* if ( configUSE_TASK_NOTIFICATIONS == 1 ) */
/*-----------------------------------------------------------*/

BaseType_t MPU_xQueueGenericSend(QueueHandle_t xQueue,
                                 const void * const pvItemToQueue,
                                 TickType_t xTicksToWait,
                                 const BaseType_t xCopyPosition) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

BaseType_t MPU_xQueueGenericSend(QueueHandle_t xQueue,
                                 const void * const pvItemToQueue,
                                 TickType_t xTicksToWait,
                                 const BaseType_t xCopyPosition)       /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xQueueGenericSendImpl                     \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xQueueGenericSend_Unpriv                      \n"
    " MPU_xQueueGenericSend_Priv:                           \n"
    "     b MPU_xQueueGenericSendImpl                       \n"
    " MPU_xQueueGenericSend_Unpriv:                         \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xQueueGenericSend) : "memory"
  );
}
/*-----------------------------------------------------------*/

UBaseType_t MPU_uxQueueMessagesWaiting(const QueueHandle_t xQueue) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

UBaseType_t MPU_uxQueueMessagesWaiting(const QueueHandle_t xQueue)       /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_uxQueueMessagesWaitingImpl                \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_uxQueueMessagesWaiting_Unpriv                 \n"
    " MPU_uxQueueMessagesWaiting_Priv:                      \n"
    "     b MPU_uxQueueMessagesWaitingImpl                  \n"
    " MPU_uxQueueMessagesWaiting_Unpriv:                    \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_uxQueueMessagesWaiting) : "memory"
  );
}
/*-----------------------------------------------------------*/

UBaseType_t MPU_uxQueueSpacesAvailable(const QueueHandle_t xQueue) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

UBaseType_t MPU_uxQueueSpacesAvailable(const QueueHandle_t xQueue)       /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_uxQueueSpacesAvailableImpl                \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_uxQueueSpacesAvailable_Unpriv                 \n"
    " MPU_uxQueueSpacesAvailable_Priv:                      \n"
    "     b MPU_uxQueueSpacesAvailableImpl                  \n"
    " MPU_uxQueueSpacesAvailable_Unpriv:                    \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_uxQueueSpacesAvailable) : "memory"
  );
}
/*-----------------------------------------------------------*/

BaseType_t MPU_xQueueReceive(QueueHandle_t xQueue,
                             void * const pvBuffer,
                             TickType_t xTicksToWait) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

BaseType_t MPU_xQueueReceive(QueueHandle_t xQueue,
                             void * const pvBuffer,
                             TickType_t xTicksToWait)       /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xQueueReceiveImpl                         \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xQueueReceive_Unpriv                          \n"
    " MPU_xQueueReceive_Priv:                               \n"
    "     b MPU_xQueueReceiveImpl                           \n"
    " MPU_xQueueReceive_Unpriv:                             \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xQueueReceive) : "memory"
  );
}
/*-----------------------------------------------------------*/

BaseType_t MPU_xQueuePeek(QueueHandle_t xQueue,
                          void * const pvBuffer,
                          TickType_t xTicksToWait) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

BaseType_t MPU_xQueuePeek(QueueHandle_t xQueue,
                          void * const pvBuffer,
                          TickType_t xTicksToWait)       /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xQueuePeekImpl                            \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xQueuePeek_Unpriv                             \n"
    " MPU_xQueuePeek_Priv:                                  \n"
    "     b MPU_xQueuePeekImpl                              \n"
    " MPU_xQueuePeek_Unpriv:                                \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xQueuePeek) : "memory"
  );
}
/*-----------------------------------------------------------*/

BaseType_t MPU_xQueueSemaphoreTake(QueueHandle_t xQueue,
                                   TickType_t xTicksToWait) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

BaseType_t MPU_xQueueSemaphoreTake(QueueHandle_t xQueue,
                                   TickType_t xTicksToWait)       /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xQueueSemaphoreTakeImpl                   \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xQueueSemaphoreTake_Unpriv                    \n"
    " MPU_xQueueSemaphoreTake_Priv:                         \n"
    "     b MPU_xQueueSemaphoreTakeImpl                     \n"
    " MPU_xQueueSemaphoreTake_Unpriv:                       \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xQueueSemaphoreTake) : "memory"
  );
}
/*-----------------------------------------------------------*/

    #if ( (configUSE_MUTEXES == 1) && (INCLUDE_xSemaphoreGetMutexHolder == 1) )

TaskHandle_t MPU_xQueueGetMutexHolder(QueueHandle_t xSemaphore) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

TaskHandle_t MPU_xQueueGetMutexHolder(QueueHandle_t xSemaphore)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xQueueGetMutexHolderImpl                  \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xQueueGetMutexHolder_Unpriv                   \n"
    " MPU_xQueueGetMutexHolder_Priv:                        \n"
    "     b MPU_xQueueGetMutexHolderImpl                    \n"
    " MPU_xQueueGetMutexHolder_Unpriv:                      \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xQueueGetMutexHolder) : "memory"
  );
}

    #endif /* if ( ( configUSE_MUTEXES == 1 ) && ( INCLUDE_xSemaphoreGetMutexHolder == 1 ) ) */
/*-----------------------------------------------------------*/

    #if (configUSE_RECURSIVE_MUTEXES == 1)

BaseType_t MPU_xQueueTakeMutexRecursive(QueueHandle_t xMutex,
                                        TickType_t xTicksToWait) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

BaseType_t MPU_xQueueTakeMutexRecursive(QueueHandle_t xMutex,
                                        TickType_t xTicksToWait)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xQueueTakeMutexRecursiveImpl              \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xQueueTakeMutexRecursive_Unpriv               \n"
    " MPU_xQueueTakeMutexRecursive_Priv:                    \n"
    "     b MPU_xQueueTakeMutexRecursiveImpl                \n"
    " MPU_xQueueTakeMutexRecursive_Unpriv:                  \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xQueueTakeMutexRecursive) : "memory"
  );
}

    #endif /* if ( configUSE_RECURSIVE_MUTEXES == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_RECURSIVE_MUTEXES == 1)

BaseType_t MPU_xQueueGiveMutexRecursive(QueueHandle_t pxMutex) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

BaseType_t MPU_xQueueGiveMutexRecursive(QueueHandle_t pxMutex)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xQueueGiveMutexRecursiveImpl              \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xQueueGiveMutexRecursive_Unpriv               \n"
    " MPU_xQueueGiveMutexRecursive_Priv:                    \n"
    "     b MPU_xQueueGiveMutexRecursiveImpl                \n"
    " MPU_xQueueGiveMutexRecursive_Unpriv:                  \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xQueueGiveMutexRecursive) : "memory"
  );
}

    #endif /* if ( configUSE_RECURSIVE_MUTEXES == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_QUEUE_SETS == 1)

QueueSetMemberHandle_t MPU_xQueueSelectFromSet(QueueSetHandle_t xQueueSet,
                                               const TickType_t xTicksToWait) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

QueueSetMemberHandle_t MPU_xQueueSelectFromSet(QueueSetHandle_t xQueueSet,
                                               const TickType_t xTicksToWait)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xQueueSelectFromSetImpl                   \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xQueueSelectFromSet_Unpriv                    \n"
    " MPU_xQueueSelectFromSet_Priv:                         \n"
    "     b MPU_xQueueSelectFromSetImpl                     \n"
    " MPU_xQueueSelectFromSet_Unpriv:                       \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xQueueSelectFromSet) : "memory"
  );
}

    #endif /* if ( configUSE_QUEUE_SETS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_QUEUE_SETS == 1)

BaseType_t MPU_xQueueAddToSet(QueueSetMemberHandle_t xQueueOrSemaphore,
                              QueueSetHandle_t xQueueSet) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

BaseType_t MPU_xQueueAddToSet(QueueSetMemberHandle_t xQueueOrSemaphore,
                              QueueSetHandle_t xQueueSet)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xQueueAddToSetImpl                        \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xQueueAddToSet_Unpriv                         \n"
    " MPU_xQueueAddToSet_Priv:                              \n"
    "     b MPU_xQueueAddToSetImpl                          \n"
    " MPU_xQueueAddToSet_Unpriv:                            \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xQueueAddToSet) : "memory"
  );
}

    #endif /* if ( configUSE_QUEUE_SETS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configQUEUE_REGISTRY_SIZE > 0)

void MPU_vQueueAddToRegistry(QueueHandle_t xQueue,
                             const char * pcName) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

void MPU_vQueueAddToRegistry(QueueHandle_t xQueue,
                             const char * pcName)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_vQueueAddToRegistryImpl                   \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_vQueueAddToRegistry_Unpriv                    \n"
    " MPU_vQueueAddToRegistry_Priv:                         \n"
    "     b MPU_vQueueAddToRegistryImpl                     \n"
    " MPU_vQueueAddToRegistry_Unpriv:                       \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_vQueueAddToRegistry) : "memory"
  );
}

    #endif /* if ( configQUEUE_REGISTRY_SIZE > 0 ) */
/*-----------------------------------------------------------*/

    #if (configQUEUE_REGISTRY_SIZE > 0)

void MPU_vQueueUnregisterQueue(QueueHandle_t xQueue) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

void MPU_vQueueUnregisterQueue(QueueHandle_t xQueue)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_vQueueUnregisterQueueImpl                 \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_vQueueUnregisterQueue_Unpriv                  \n"
    " MPU_vQueueUnregisterQueue_Priv:                       \n"
    "     b MPU_vQueueUnregisterQueueImpl                   \n"
    " MPU_vQueueUnregisterQueue_Unpriv:                     \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_vQueueUnregisterQueue) : "memory"
  );
}

    #endif /* if ( configQUEUE_REGISTRY_SIZE > 0 ) */
/*-----------------------------------------------------------*/

    #if (configQUEUE_REGISTRY_SIZE > 0)

const char * MPU_pcQueueGetName(QueueHandle_t xQueue) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

const char * MPU_pcQueueGetName(QueueHandle_t xQueue)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_pcQueueGetNameImpl                        \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_pcQueueGetName_Unpriv                         \n"
    " MPU_pcQueueGetName_Priv:                              \n"
    "     b MPU_pcQueueGetNameImpl                          \n"
    " MPU_pcQueueGetName_Unpriv:                            \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_pcQueueGetName) : "memory"
  );
}

    #endif /* if ( configQUEUE_REGISTRY_SIZE > 0 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_TIMERS == 1)

void * MPU_pvTimerGetTimerID(const TimerHandle_t xTimer) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

void * MPU_pvTimerGetTimerID(const TimerHandle_t xTimer)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_pvTimerGetTimerIDImpl                     \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_pvTimerGetTimerID_Unpriv                      \n"
    " MPU_pvTimerGetTimerID_Priv:                           \n"
    "     b MPU_pvTimerGetTimerIDImpl                       \n"
    " MPU_pvTimerGetTimerID_Unpriv:                         \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_pvTimerGetTimerID) : "memory"
  );
}

    #endif /* if ( configUSE_TIMERS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_TIMERS == 1)

void MPU_vTimerSetTimerID(TimerHandle_t xTimer,
                          void * pvNewID) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

void MPU_vTimerSetTimerID(TimerHandle_t xTimer,
                          void * pvNewID)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_vTimerSetTimerIDImpl                      \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_vTimerSetTimerID_Unpriv                       \n"
    " MPU_vTimerSetTimerID_Priv:                            \n"
    "     b MPU_vTimerSetTimerIDImpl                        \n"
    " MPU_vTimerSetTimerID_Unpriv:                          \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_vTimerSetTimerID) : "memory"
  );
}

    #endif /* if ( configUSE_TIMERS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_TIMERS == 1)

BaseType_t MPU_xTimerIsTimerActive(TimerHandle_t xTimer) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

BaseType_t MPU_xTimerIsTimerActive(TimerHandle_t xTimer)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xTimerIsTimerActiveImpl                   \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xTimerIsTimerActive_Unpriv                    \n"
    " MPU_xTimerIsTimerActive_Priv:                         \n"
    "     b MPU_xTimerIsTimerActiveImpl                     \n"
    " MPU_xTimerIsTimerActive_Unpriv:                       \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xTimerIsTimerActive) : "memory"
  );
}

    #endif /* if ( configUSE_TIMERS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_TIMERS == 1)

TaskHandle_t MPU_xTimerGetTimerDaemonTaskHandle(void) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

TaskHandle_t MPU_xTimerGetTimerDaemonTaskHandle(void)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xTimerGetTimerDaemonTaskHandleImpl        \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xTimerGetTimerDaemonTaskHandle_Unpriv         \n"
    " MPU_xTimerGetTimerDaemonTaskHandle_Priv:              \n"
    "     b MPU_xTimerGetTimerDaemonTaskHandleImpl          \n"
    " MPU_xTimerGetTimerDaemonTaskHandle_Unpriv:            \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xTimerGetTimerDaemonTaskHandle) : "memory"
  );
}

    #endif /* if ( configUSE_TIMERS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_TIMERS == 1)

BaseType_t MPU_xTimerGenericCommandFromTaskEntry(const xTimerGenericCommandFromTaskParams_t * pxParams) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

BaseType_t MPU_xTimerGenericCommandFromTaskEntry(const xTimerGenericCommandFromTaskParams_t * pxParams)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xTimerGenericCommandFromTaskImpl          \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xTimerGenericCommandFromTask_Unpriv           \n"
    " MPU_xTimerGenericCommandFromTask_Priv:                \n"
    "     b MPU_xTimerGenericCommandFromTaskImpl            \n"
    " MPU_xTimerGenericCommandFromTask_Unpriv:              \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xTimerGenericCommandFromTask) : "memory"
  );
}

    #endif /* if ( configUSE_TIMERS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_TIMERS == 1)

const char * MPU_pcTimerGetName(TimerHandle_t xTimer) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

const char * MPU_pcTimerGetName(TimerHandle_t xTimer)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_pcTimerGetNameImpl                        \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_pcTimerGetName_Unpriv                         \n"
    " MPU_pcTimerGetName_Priv:                              \n"
    "     b MPU_pcTimerGetNameImpl                          \n"
    " MPU_pcTimerGetName_Unpriv:                            \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_pcTimerGetName) : "memory"
  );
}

    #endif /* if ( configUSE_TIMERS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_TIMERS == 1)

void MPU_vTimerSetReloadMode(TimerHandle_t xTimer,
                             const BaseType_t uxAutoReload) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

void MPU_vTimerSetReloadMode(TimerHandle_t xTimer,
                             const BaseType_t uxAutoReload)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_vTimerSetReloadModeImpl                   \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_vTimerSetReloadMode_Unpriv                    \n"
    " MPU_vTimerSetReloadMode_Priv:                         \n"
    "     b MPU_vTimerSetReloadModeImpl                     \n"
    " MPU_vTimerSetReloadMode_Unpriv:                       \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_vTimerSetReloadMode) : "memory"
  );
}

    #endif /* if ( configUSE_TIMERS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_TIMERS == 1)

BaseType_t MPU_xTimerGetReloadMode(TimerHandle_t xTimer) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

BaseType_t MPU_xTimerGetReloadMode(TimerHandle_t xTimer)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xTimerGetReloadModeImpl                   \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xTimerGetReloadMode_Unpriv                    \n"
    " MPU_xTimerGetReloadMode_Priv:                         \n"
    "     b MPU_xTimerGetReloadModeImpl                     \n"
    " MPU_xTimerGetReloadMode_Unpriv:                       \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xTimerGetReloadMode) : "memory"
  );
}

    #endif /* if ( configUSE_TIMERS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_TIMERS == 1)

UBaseType_t MPU_uxTimerGetReloadMode(TimerHandle_t xTimer) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

UBaseType_t MPU_uxTimerGetReloadMode(TimerHandle_t xTimer)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_uxTimerGetReloadModeImpl                  \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_uxTimerGetReloadMode_Unpriv                   \n"
    " MPU_uxTimerGetReloadMode_Priv:                        \n"
    "     b MPU_uxTimerGetReloadModeImpl                    \n"
    " MPU_uxTimerGetReloadMode_Unpriv:                      \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_uxTimerGetReloadMode) : "memory"
  );
}

    #endif /* if ( configUSE_TIMERS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_TIMERS == 1)

TickType_t MPU_xTimerGetPeriod(TimerHandle_t xTimer) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

TickType_t MPU_xTimerGetPeriod(TimerHandle_t xTimer)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xTimerGetPeriodImpl                       \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xTimerGetPeriod_Unpriv                        \n"
    " MPU_xTimerGetPeriod_Priv:                             \n"
    "     b MPU_xTimerGetPeriodImpl                         \n"
    " MPU_xTimerGetPeriod_Unpriv:                           \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xTimerGetPeriod) : "memory"
  );
}

    #endif /* if ( configUSE_TIMERS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_TIMERS == 1)

TickType_t MPU_xTimerGetExpiryTime(TimerHandle_t xTimer) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

TickType_t MPU_xTimerGetExpiryTime(TimerHandle_t xTimer)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xTimerGetExpiryTimeImpl                   \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xTimerGetExpiryTime_Unpriv                    \n"
    " MPU_xTimerGetExpiryTime_Priv:                         \n"
    "     b MPU_xTimerGetExpiryTimeImpl                     \n"
    " MPU_xTimerGetExpiryTime_Unpriv:                       \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xTimerGetExpiryTime) : "memory"
  );
}

    #endif /* if ( configUSE_TIMERS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_EVENT_GROUPS == 1)

EventBits_t MPU_xEventGroupWaitBitsEntry(const xEventGroupWaitBitsParams_t * pxParams) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

EventBits_t MPU_xEventGroupWaitBitsEntry(const xEventGroupWaitBitsParams_t * pxParams)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xEventGroupWaitBitsImpl                   \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xEventGroupWaitBits_Unpriv                    \n"
    " MPU_xEventGroupWaitBits_Priv:                         \n"
    "     b MPU_xEventGroupWaitBitsImpl                     \n"
    " MPU_xEventGroupWaitBits_Unpriv:                       \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xEventGroupWaitBits) : "memory"
  );
}

    #endif /* #if ( configUSE_EVENT_GROUPS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_EVENT_GROUPS == 1)

EventBits_t MPU_xEventGroupClearBits(EventGroupHandle_t xEventGroup,
                                     const EventBits_t uxBitsToClear) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

EventBits_t MPU_xEventGroupClearBits(EventGroupHandle_t xEventGroup,
                                     const EventBits_t uxBitsToClear)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xEventGroupClearBitsImpl                  \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xEventGroupClearBits_Unpriv                   \n"
    " MPU_xEventGroupClearBits_Priv:                        \n"
    "     b MPU_xEventGroupClearBitsImpl                    \n"
    " MPU_xEventGroupClearBits_Unpriv:                      \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xEventGroupClearBits) : "memory"
  );
}

    #endif /* #if ( configUSE_EVENT_GROUPS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_EVENT_GROUPS == 1)

EventBits_t MPU_xEventGroupSetBits(EventGroupHandle_t xEventGroup,
                                   const EventBits_t uxBitsToSet) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

EventBits_t MPU_xEventGroupSetBits(EventGroupHandle_t xEventGroup,
                                   const EventBits_t uxBitsToSet)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xEventGroupSetBitsImpl                    \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xEventGroupSetBits_Unpriv                     \n"
    " MPU_xEventGroupSetBits_Priv:                          \n"
    "     b MPU_xEventGroupSetBitsImpl                      \n"
    " MPU_xEventGroupSetBits_Unpriv:                        \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xEventGroupSetBits) : "memory"
  );
}

    #endif /* #if ( configUSE_EVENT_GROUPS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_EVENT_GROUPS == 1)

EventBits_t MPU_xEventGroupSync(EventGroupHandle_t xEventGroup,
                                const EventBits_t uxBitsToSet,
                                const EventBits_t uxBitsToWaitFor,
                                TickType_t xTicksToWait) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

EventBits_t MPU_xEventGroupSync(EventGroupHandle_t xEventGroup,
                                const EventBits_t uxBitsToSet,
                                const EventBits_t uxBitsToWaitFor,
                                TickType_t xTicksToWait)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xEventGroupSyncImpl                       \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xEventGroupSync_Unpriv                        \n"
    " MPU_xEventGroupSync_Priv:                             \n"
    "     b MPU_xEventGroupSyncImpl                         \n"
    " MPU_xEventGroupSync_Unpriv:                           \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xEventGroupSync) : "memory"
  );
}

    #endif /* #if ( configUSE_EVENT_GROUPS == 1 ) */
/*-----------------------------------------------------------*/

    #if ( (configUSE_EVENT_GROUPS == 1) && (configUSE_TRACE_FACILITY == 1) )

UBaseType_t MPU_uxEventGroupGetNumber(void * xEventGroup) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

UBaseType_t MPU_uxEventGroupGetNumber(void * xEventGroup)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_uxEventGroupGetNumberImpl                 \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_uxEventGroupGetNumber_Unpriv                  \n"
    " MPU_uxEventGroupGetNumber_Priv:                       \n"
    "     b MPU_uxEventGroupGetNumberImpl                   \n"
    " MPU_uxEventGroupGetNumber_Unpriv:                     \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_uxEventGroupGetNumber) : "memory"
  );
}

    #endif /* #if ( ( configUSE_EVENT_GROUPS == 1 ) && ( configUSE_TRACE_FACILITY == 1 ) ) */
/*-----------------------------------------------------------*/

    #if ( (configUSE_EVENT_GROUPS == 1) && (configUSE_TRACE_FACILITY == 1) )

void MPU_vEventGroupSetNumber(void * xEventGroup,
                              UBaseType_t uxEventGroupNumber) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

void MPU_vEventGroupSetNumber(void * xEventGroup,
                              UBaseType_t uxEventGroupNumber)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_vEventGroupSetNumberImpl                  \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_vEventGroupSetNumber_Unpriv                   \n"
    " MPU_vEventGroupSetNumber_Priv:                        \n"
    "     b MPU_vEventGroupSetNumberImpl                    \n"
    " MPU_vEventGroupSetNumber_Unpriv:                      \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_vEventGroupSetNumber) : "memory"
  );
}

    #endif /* #if ( ( configUSE_EVENT_GROUPS == 1 ) && ( configUSE_TRACE_FACILITY == 1 ) ) */
/*-----------------------------------------------------------*/

    #if (configUSE_STREAM_BUFFERS == 1)

size_t MPU_xStreamBufferSend(StreamBufferHandle_t xStreamBuffer,
                             const void * pvTxData,
                             size_t xDataLengthBytes,
                             TickType_t xTicksToWait) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

size_t MPU_xStreamBufferSend(StreamBufferHandle_t xStreamBuffer,
                             const void * pvTxData,
                             size_t xDataLengthBytes,
                             TickType_t xTicksToWait)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xStreamBufferSendImpl                     \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xStreamBufferSend_Unpriv                      \n"
    " MPU_xStreamBufferSend_Priv:                           \n"
    "     b MPU_xStreamBufferSendImpl                       \n"
    " MPU_xStreamBufferSend_Unpriv:                         \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xStreamBufferSend) : "memory"
  );
}

    #endif /* #if ( configUSE_STREAM_BUFFERS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_STREAM_BUFFERS == 1)

size_t MPU_xStreamBufferReceive(StreamBufferHandle_t xStreamBuffer,
                                void * pvRxData,
                                size_t xBufferLengthBytes,
                                TickType_t xTicksToWait) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

size_t MPU_xStreamBufferReceive(StreamBufferHandle_t xStreamBuffer,
                                void * pvRxData,
                                size_t xBufferLengthBytes,
                                TickType_t xTicksToWait)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xStreamBufferReceiveImpl                  \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xStreamBufferReceive_Unpriv                   \n"
    " MPU_xStreamBufferReceive_Priv:                        \n"
    "     b MPU_xStreamBufferReceiveImpl                    \n"
    " MPU_xStreamBufferReceive_Unpriv:                      \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xStreamBufferReceive) : "memory"
  );
}

    #endif /* #if ( configUSE_STREAM_BUFFERS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_STREAM_BUFFERS == 1)

BaseType_t MPU_xStreamBufferIsFull(StreamBufferHandle_t xStreamBuffer) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

BaseType_t MPU_xStreamBufferIsFull(StreamBufferHandle_t xStreamBuffer)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xStreamBufferIsFullImpl                   \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xStreamBufferIsFull_Unpriv                    \n"
    " MPU_xStreamBufferIsFull_Priv:                         \n"
    "     b MPU_xStreamBufferIsFullImpl                     \n"
    " MPU_xStreamBufferIsFull_Unpriv:                       \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xStreamBufferIsFull) : "memory"
  );
}

    #endif /* #if ( configUSE_STREAM_BUFFERS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_STREAM_BUFFERS == 1)

BaseType_t MPU_xStreamBufferIsEmpty(StreamBufferHandle_t xStreamBuffer) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

BaseType_t MPU_xStreamBufferIsEmpty(StreamBufferHandle_t xStreamBuffer)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xStreamBufferIsEmptyImpl                  \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xStreamBufferIsEmpty_Unpriv                   \n"
    " MPU_xStreamBufferIsEmpty_Priv:                        \n"
    "     b MPU_xStreamBufferIsEmptyImpl                    \n"
    " MPU_xStreamBufferIsEmpty_Unpriv:                      \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xStreamBufferIsEmpty) : "memory"
  );
}

    #endif /* #if ( configUSE_STREAM_BUFFERS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_STREAM_BUFFERS == 1)

size_t MPU_xStreamBufferSpacesAvailable(StreamBufferHandle_t xStreamBuffer) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

size_t MPU_xStreamBufferSpacesAvailable(StreamBufferHandle_t xStreamBuffer)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xStreamBufferSpacesAvailableImpl          \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xStreamBufferSpacesAvailable_Unpriv           \n"
    " MPU_xStreamBufferSpacesAvailable_Priv:                \n"
    "     b MPU_xStreamBufferSpacesAvailableImpl            \n"
    " MPU_xStreamBufferSpacesAvailable_Unpriv:              \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xStreamBufferSpacesAvailable) : "memory"
  );
}

    #endif /* #if ( configUSE_STREAM_BUFFERS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_STREAM_BUFFERS == 1)

size_t MPU_xStreamBufferBytesAvailable(StreamBufferHandle_t xStreamBuffer) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

size_t MPU_xStreamBufferBytesAvailable(StreamBufferHandle_t xStreamBuffer)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xStreamBufferBytesAvailableImpl           \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xStreamBufferBytesAvailable_Unpriv            \n"
    " MPU_xStreamBufferBytesAvailable_Priv:                 \n"
    "     b MPU_xStreamBufferBytesAvailableImpl             \n"
    " MPU_xStreamBufferBytesAvailable_Unpriv:               \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xStreamBufferBytesAvailable) : "memory"
  );
}

    #endif /* #if ( configUSE_STREAM_BUFFERS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_STREAM_BUFFERS == 1)

BaseType_t MPU_xStreamBufferSetTriggerLevel(StreamBufferHandle_t xStreamBuffer,
                                            size_t xTriggerLevel) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

BaseType_t MPU_xStreamBufferSetTriggerLevel(StreamBufferHandle_t xStreamBuffer,
                                            size_t xTriggerLevel)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xStreamBufferSetTriggerLevelImpl          \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xStreamBufferSetTriggerLevel_Unpriv           \n"
    " MPU_xStreamBufferSetTriggerLevel_Priv:                \n"
    "     b MPU_xStreamBufferSetTriggerLevelImpl            \n"
    " MPU_xStreamBufferSetTriggerLevel_Unpriv:              \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xStreamBufferSetTriggerLevel) : "memory"
  );
}

    #endif /* #if ( configUSE_STREAM_BUFFERS == 1 ) */
/*-----------------------------------------------------------*/

    #if (configUSE_STREAM_BUFFERS == 1)

size_t MPU_xStreamBufferNextMessageLengthBytes(StreamBufferHandle_t xStreamBuffer) __attribute__( (naked) ) FREERTOS_SYSTEM_CALL;

size_t MPU_xStreamBufferNextMessageLengthBytes(StreamBufferHandle_t xStreamBuffer)           /* __attribute__ (( naked )) FREERTOS_SYSTEM_CALL */
{
  __asm volatile
  (
    " .syntax unified                                       \n"
    " .extern MPU_xStreamBufferNextMessageLengthBytesImpl   \n"
    "                                                       \n"
    " push {r0}                                             \n"
    " mrs r0, control                                       \n"
    " tst r0, #1                                            \n"
    " pop {r0}                                              \n"
    " bne MPU_xStreamBufferNextMessageLengthBytes_Unpriv    \n"
    " MPU_xStreamBufferNextMessageLengthBytes_Priv:         \n"
    "     b MPU_xStreamBufferNextMessageLengthBytesImpl     \n"
    " MPU_xStreamBufferNextMessageLengthBytes_Unpriv:       \n"
    "     svc %0                                            \n"
    "                                                       \n"
    : : "i" (SYSTEM_CALL_xStreamBufferNextMessageLengthBytes) : "memory"
  );
}

    #endif /* #if ( configUSE_STREAM_BUFFERS == 1 ) */
/*-----------------------------------------------------------*/

#endif /* ( configENABLE_MPU == 1 ) && ( configUSE_MPU_WRAPPERS_V1 == 0 ) */
